!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    amr_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir amr
!! @brief Contains modules for managing amr algorithm

!> @file amr_control.f90
!! @brief Main file for module AmrControl

!> @defgroup AMR Adaptive Mesh Refinement
!! @brief Group of modules for managing AMR algorithm

!> @defgroup AmrControl AmrControl
!> @brief Main module for managing AMR algorithm
!! @ingroup AMR

!> Main module for managing AMR algorithm
!! @ingroup AmrControl
MODULE AmrControl
  USE TreeLevelOps, ONLY : InheritNeighborsChildren, InheritOldNodeOverlapsChildren, InheritNewNodeOverlapsChildren, &
       InheritOverlapsOldChildren, InheritOverlapsNewChildren, CreateChildrens, UpdateOverlaps, NullifyNeighbors, &
       AgeNodesChildren, AgeNodes, DestroyNodes, DestroyOldNodes

  USE DataLevelOps, ONLY : ProlongateParentsData, ApplyOverlaps, ApplyChildrenData, ApplyInitialChildrenData, SyncFluxes, &
       InitInfos, InitialInitInfos, AfterOverlaps, UpdateChildMasks, SetErrFlags, RestrictionFixups, &
       AccumulateFluxes, CoarsenDataForParents, CoarsenInitialDataForParents, InitGrids, AdvanceGrids, &
       ApplyPhysicalBCs, AfterFixups, ChildMaskOverlaps, ScheduledAdvanceGrids, CompleteAdvanceGrids, WaitingAdvances, InitRestartGrids, UpdateTimeDerivs, ClearFixupFluxes, UpdateMeanDensity, ClearParentFixups, ClearChildFixups, TransferTimeDerivs

  USE DataDeclarations
  USE DistributionControl
  USE ModuleControl, ONLY : BeforeGlobalStep
  USE TimeStep
  USE TreeDeclarations
  USE TreeNodeOps
  USE CommunicationControl
  USE ExplicitControl
  USE IOControl 
  USE Timing
  USE ParticleControl
  USE ProcessingControl
  USE ProcessingDeclarations
# if defined HYPRE
  USE EllipticControl, ONLY : Elliptic, InitialElliptic, EllipticInit, ApplyEllipticBC
# endif
# if defined PTHREADS
  USE ThreadControl
# endif
# if defined PTH
  USE PthControl
# endif

  IMPLICIT NONE
  PRIVATE
  PUBLIC AMRInit, AMRAdvance, AMRStart, DomainInit, ClearAllNodeLists, DestroyAllnodes, LevelsInit, AMR, BackupData, RestoreData, BoundaryZoneInit, PrintAllocations
  INTEGER :: framecounter=0
  REAL(KIND=qPREC) :: temp
  SAVE
  PUBLIC MpiTest



CONTAINS
  !> @name Initialization routines 
  !! @{

  !> Reads in global data and initializes tree level pointers
   SUBROUTINE AmrInit
      INTEGER :: i
      CALL ReadInGlobalData()
!      ALLOCATE(leveldX(-2:MaxLevel), leveldt(0:MaxLevel), leveltnow(0:MaxLevel), levelgmbc(-2:MaxLevel,2), levelMX(-2:MaxLevel,3))
      ALLOCATE(Nodes(-2:MaxLevel), OldNodes(-2:MaxLevel), ExternalNodes(-2:MaxLevel), &
               OldExternalNodes(-2:MaxLevel), LastLocalNode(-2:MaxLevel), LastExternalNode(-2:MaxLevel), &
               LastOldLocalNode(-2:MaxLevel), LastOldExternalNode(-2:MaxLevel), BackupNodes(-2:MaxLevel), BackupExternalNodes(-2:MaxLevel))
      DO i=-2,MaxLevel
         NULLIFY(Nodes(i)%p)
         NULLIFY(OldNodes(i)%p)
         NULLIFY(ExternalNodes(i)%p)
         NULLIFY(OldExternalNodes(i)%p)
         NULLIFY(LastLocalNode(i)%p)
         NULLIFY(LastExternalNode(i)%p)
         NULLIFY(LastOldLocalNode(i)%p)
         NULLIFY(LastOldExternalNode(i)%p)
         NULLIFY(BackupNodes(i)%p)
         NULLIFY(BackupExternalNodes(i)%p)
      END DO
      GmGlobal(:,1)=1
      GmGlobal(:,2)=GmX
      IF (ANY(ANY(Gmthbc(1:nDim,:)==2,2) .AND. .NOT. ALL(Gmthbc(1:nDim,:)==2,2))) THEN
         PRINT*, 'Bad Periodic Boundary conditions.  If one side is periodic, the other side must be as well.  Stopping...'
         STOP
      END IF
      lHydroPeriodic(nDim+1:)=.false.
      lHydroPeriodic(1:nDim)=(Gmthbc(1:nDim,1)==2)
      lAnyPeriodic=lAnyPeriodic .OR. lHydroPeriodic
# if defined PTH
      CALL PthInit()
# else
      IF (iThreaded == THREADED) THEN
         IF (MPI_ID == 0) PRINT*, 'Error - You must compile with PTHREADFLAG=2 to use iThreaded == ', THREADED
         STOP
      END IF
# endif
      lRestart=lRestart .OR. lPostProcess
   END SUBROUTINE AmrInit

!> Initializes level variables
   SUBROUTINE LevelsInit
      USE HyperbolicDeclarations
      INTEGER :: i, step
      levels(-2:MaxLevel)%id=(/(i,i=-2,MaxLevel)/)
      levels(-2)%dx=(GxBounds(1,2)-GxBounds(1,1))/GmX(1)
      levels(-2)%mx=Gmx
      levels(-2)%steps=1      
      levels(-2)%step=1
      DO i=-1,MaxLevel
         levels(i)%dx = levels(i-1)%dx / levels(i-1)%CoarsenRatio
         levels(i)%mx(1:nDim)=levels(i-1)%mX(1:nDim)*levels(i-1)%CoarsenRatio
         levels(i)%mx(nDim+1:3)=1
         levels(i)%steps=levels(i-1)%CoarsenRatio
         levels(i)%step=1
      END DO
      CALL SinkParticleInit

    ! Now calculate mbc's staring with the finest level and working backwards.
!                              (lost)           (needed)

!      levels(MaxLevel)%ombc(2)=max(levels(MaxLevel-1)%steps*hyperbolic_mbc+source_mbc,particle_mbc) !ghost cells in overlap 2

# if defined PTHREADS
   CALL ThreadsInit()
# endif
# if defined HYPRE
   CALL EllipticInit()
# else
   IF(lElliptic)THEN
     IF(MPI_ID==0) PRINT*, "Elliptic solver requested, but AstroBEAR is not compiled with hypre. Stopping... "
     STOP
   END IF
# endif
   CALL ExplicitInit()
   END SUBROUTINE LevelsInit
   

   !> @brief sets information on boundary zone sizes
   SUBROUTINE BoundaryZoneInit()
      INTEGER :: i, step

      IF (lPostProcess) THEN
         hyperbolic_mbc=0d0
         particle_mbc=0d0
         elliptic_mbc=0d0
         afterstep_mbc=0d0
      END IF
      levels(MaxLevel)%ambc(1)=(levels(MaxLevel)%steps-1)*(hyperbolic_mbc+afterstep_mbc)+afterstep_mbc !Size of ghost zone region to update in first advance
      levels(MaxLevel)%ambc(2)=afterstep_mbc

      levels(MaxLevel)%egmbc(1)=max(levels(MaxLevel)%ambc(1)+hyperbolic_mbc+elliptic_mbc, particle_mbc, Processing_mbc)  ! Initial elliptic prolongate and ghost
      levels(MaxLevel)%gmbc(1)=max(levels(MaxLevel)%ambc(1)+hyperbolic_mbc,particle_mbc, processing_mbc)               ! Initial q prolongate and ghost
      
      levels(MaxLevel)%egmbc(2)=max(levels(MaxLevel)%ambc(2)+hyperbolic_mbc+elliptic_mbc, particle_mbc, processing_mbc)  ! Second elliptic ghost
      levels(MaxLevel)%gmbc(2)=max(levels(MaxLevel)%ambc(2)+hyperbolic_mbc, particle_mbc, processing_mbc)              ! Second q ghost


      levels(MaxLevel)%ombc(1)=levels(MaxLevel)%egmbc(1) !Initial Overlap range
      levels(MaxLevel)%nmbc=levels(MaxLevel)%egmbc(2)
      levels(MaxLevel)%ombc(2)=levels(MaxLevel)%nmbc !Overlaps become neighbors on second step

      levels(MaxLevel)%pmbc=levels(MaxLevel)%ombc(1)+modulo(levels(MaxLevel)%ombc(1),levels(MaxLevel-1)%CoarsenRatio) !ensures it is divisible by the coarsen ratio

      DO i=MaxLevel-1,-2,-1
         step=levels(i)%steps

         IF (i > -2) levels(i)%pmbc=ceiling(real(levels(i+1)%ombc(1))/real(levels(i)%CoarsenRatio))

         levels(i)%ambc(step)=afterstep_mbc
         levels(i)%egmbc(step)=max(levels(i)%ambc(step)+hyperbolic_mbc+elliptic_mbc, ceiling(real(particle_mbc)/real(product(Levels(i:MaxLevel-1)%CoarsenRatio))), levels(i)%pmbc, Processing_mbc)  ! Second elliptic ghost
         levels(i)%gmbc(step)=max(levels(i)%ambc(step)+hyperbolic_mbc, ceiling(real(particle_mbc)/real(product(Levels(i:MaxLevel-1)%CoarsenRatio))), levels(i)%pmbc, Processing_mbc)              ! Second q ghost         
         levels(i)%ombc(step)=max(levels(i)%egmbc(step), levels(i)%pmbc) !Initial Neighbor range.  Ensures that no missed nieghbors children could be within ombc(1) on the next level
         levels(i)%nmbc=levels(i)%ombc(step) !Overlaps become neighbors on second step

         IF (levels(i)%steps == 1) THEN
            levels(i)%ambc(2)=0
            levels(i)%egmbc(2)=0
            levels(i)%gmbc(2)=0
            levels(i)%ombc(2)=0
         ELSE
            DO step=levels(i)%steps-1, 1, -1
               levels(i)%ambc(step)=levels(i)%ombc(step+1)+afterstep_mbc
               levels(i)%egmbc(step)=max(levels(i)%ambc(step)+hyperbolic_mbc+elliptic_mbc, ceiling(real(particle_mbc)/real(product(Levels(i:MaxLevel-1)%CoarsenRatio))), Processing_mbc)  ! Initial elliptic prolongate and ghost
               levels(i)%gmbc(step)=max(levels(i)%ambc(step)+hyperbolic_mbc,ceiling(real(particle_mbc)/real(product(Levels(i:MaxLevel-1)%CoarsenRatio))), Processing_mbc)               ! Initial q prolongate and ghost     
               levels(i)%ombc(step)=max(levels(i)%egmbc(step), ceiling(real(levels(i+1)%ombc(1))/real(levels(i)%CoarsenRatio))) !Initial Overlap range
            END DO
         END IF
      END DO

!      IF (MPI_ID == 0) THEN
!         DO i=-2,MaxLevel
!            write(*,'(A8,I2,A8,2I2,A8,2I2,A8,2I2,A8,2I2,A8,1I2,A8,1I2)') "level", i, "ambc=", levels(i)%ambc, "gmbc=",levels(i)%gmbc(1:2),"egmbc=",levels(i)%egmbc,"ombc=",levels(i)%ombc, "nmbc=", levels(i)%nmbc, "pmbc=",levels(i)%pmbc
!         END DO
!      END IF

      nPeriodic_overlaps(1:nDim)=levels(0)%ombc(1)/Gmx(1:nDim) + 1
!      write(*,*) 'nPeriodic_overlaps = ', nPeriodic_overlaps, levels(0)%ombc(1)
   END SUBROUTINE BoundaryZoneInit

!> Initializes level -2 master node and level -1 domains
   SUBROUTINE DomainInit
      INTEGER :: i, iErr
      TYPE(NodeBox), POINTER :: root_box
      TYPE(NodeDef), POINTER :: root_node
      TYPE(DomainDef) :: Domain
      NAMELIST /DomainData/ Domain


      NULLIFY(root_box)
      NULLIFY(root_node)

      IF (MPI_ID == 0) THEN         

         CALL CreateNodeBox(GmGlobal, root_box, 0)

         CALL AddNode(-2, root_box, root_node)
         !CALL InitialInitInfo(root_node%Info,-2, root_box%mGlobal)
      END IF

      ALLOCATE(Domains(nDomains))
      Domains(1)%mthbc=Gmthbc
      Domains(1)%mGlobal=1
      Domains(1)%mGlobal(1:nDim,2)=Gmx(1:nDim)

!      DO i=1,nDomains           
!            ! Read in level data values.
!         READ(GLOBAL_DATA_HANDLE,NML=DomainData,IOStat=iErr)
!         IF(iErr/=0) THEN
!            PRINT *, "AmrInit() error:  unable to read DomainData namelist from ", GLOBAL_DATA_FILE, "."
!            STOP
!         END IF
!         Domains(i)=Domain
!      END DO
!         ALLOCATE(root_node%proclist(MPI_NP), root_node%proctime(MPI_NP))
!         root_node%proclist=(/(i-1,i=1,MPI_NP)/)
!         root_node%proctime=1d0

   END SUBROUTINE DomainInit

!> Reads in GlobalData and level data
   SUBROUTINE ReadInGlobalData()
      INTEGER :: iErr,i
      INTEGER, DIMENSION(0:MaxDepth) :: InterpOpt
      INTEGER, DIMENSION(0:MaxDepth) :: CoarsenRatio=2
      REAL(KIND=qPREC), DIMENSION(0:MaxDepth) :: qTolerance
      REAL(KIND=qPrec), DIMENSION(0:MaxDepth) :: DesiredFillRatios
      LOGICAL :: temp
      NAMELIST /LevelData/ qTolerance, DesiredFillRatios
      
      MaintainAuxArrays=.true.
      ! Open data file (quit if not successful).
      OPEN(UNIT=GLOBAL_DATA_HANDLE,FILE=GLOBAL_DATA_FILE)
     
      ! Read in global data values.
      !READ(GLOBAL_DATA_HANDLE,NML=GlobalData,IOStat=iErr)
      READ(GLOBAL_DATA_HANDLE,NML=GlobalData)

      ! If regridding then also restarting
      lRestart=lRestart .OR. lRegrid

      IF (MaintainAuxArrays) THEN
         MaintainAuxArrays = .false.
         CLOSE(UNIT=GLOBAL_DATA_HANDLE)

         OPEN(UNIT=GLOBAL_DATA_HANDLE,FILE=GLOBAL_DATA_FILE)

         ! Read in global data values.
         !READ(GLOBAL_DATA_HANDLE,NML=GlobalData,IOStat=iErr)
         READ(GLOBAL_DATA_HANDLE,NML=GlobalData)

         IF (MaintainAuxArrays) THEN
            IF (MPI_ID == 0) THEN
               write(*,*) 'MaintainAuxArrays is no longer set in global.data and should be removed'
               write(*,*) 'It is instead automatically turned on if lMHD is true in physics.data'
            END IF
         END IF
      ELSE
         IF (MPI_ID == 0) THEN
            write(*,*) 'MaintainAuxArrays is no longer set in global.data and should be removed'
            write(*,*) 'It is instead automatically turned on if lMHD is true in physics.data'
         END IF         
      END IF

      ! Read in level data values.
      ALLOCATE(levels(-2:MaxLevel))
      InterpOpt=-1
      qTolerance=-1
      DesiredFillRatios=-1
      READ(GLOBAL_DATA_HANDLE,NML=LevelData,IOStat=iErr)
      
      IF(iErr/=0) THEN
         PRINT *, "ReadInGlobalData() error:  unable to read LevelData namelist from ", GLOBAL_DATA_FILE, "."
         STOP
      END IF
      IF (ANY(InterpOpt > -1)) THEN
         IF (MPI_ID == 0) THEN
            write(*,*) 'InterpOpt in global.data is now depracated'
            write(*,*) 'Use InterpOpts in physics.data instead'
         END IF
      END IF
      DO i=1,MaxLevel
         IF (qTolerance(i) == -1) qTolerance(i)=qTolerance(i-1)
         IF (DesiredFillRatios(i) == -1) DesiredFillRatios(i)=DesiredFillRatios(i-1)
      END DO
         

      IF (iThreaded == NON_THREADED) LevelBalance=(/0d0,0d0/)
      DO i=-2,MIN(ROOT_LEVEL-1, MaxLevel)
         levels(i)%CoarsenRatio=1
         levels(i)%qTolerance=0
         levels(i)%DesiredFillRatios=1
         levels(i)%steps=1
         levels(i)%step=1
      END DO
      DO i=ROOT_LEVEL,MaxLevel
         levels(i)%CoarsenRatio=CoarsenRatio(i)
         levels(i)%qTolerance=qTolerance(i)
         levels(i)%DesiredFillRatios=DesiredFillRatios(i)
         levels(i)%steps=levels(i-1)%CoarsenRatio
         levels(i)%step=1
      END DO
   END SUBROUTINE ReadInGlobalData

   !> @}

   !> @name Advance Routine
   !! @{

   !> Advances grids until time tnext
   !! @details Takes the necessary number of time steps based on the cfl limitations
   !! @param tnext - time to advance to
   SUBROUTINE AMRAdvance(tnext)
      REAL(KIND=qPREC) :: tnext,tback
      LOGICAL :: lasttimestep, lSuccess
      lasttimestep=.false.
      DO WHILE (.NOT. lasttimestep) !levels(ROOT_LEVEL)%tnow < tnext)
         CALL BackupData()
         tback=levels(ROOT_LEVEL)%tnow
         lSuccess=.false.
         DO WHILE (.not. lSuccess)
            CALL GetNextTimeStep(tnext-levels(ROOT_LEVEL)%tnow, lasttimestep)
            CALL AMR(BaseLevel)
            CALL PrintAllocations
            IF (lPrintDebugFrame) THEN
               CALL ProcessData()
               CALL WriteDataFrame(1000+current_frame)
               lPrintDebugFrame=.false.
            END IF
            CALL TestBadCFL(lSuccess)
            IF (.NOT. lSuccess) THEN
               levels(ROOT_LEVEL:MaxLevel)%tnow=tback
               CALL RestoreData()
               lasttimestep=.false.
               levels(ROOT_LEVEL)%CurrentLevelStep=levels(ROOT_LEVEL)%CurrentLevelStep-1
            END IF
         END DO
      END DO
   END SUBROUTINE AMRAdvance
   
   !> @}



   !> @name Recursive AMR subroutines
   !! @{

   !> Creates the next level of grids, advances the current level one timestep - and calls AMR on the next higher level
   !! @param n current level of node tree
   RECURSIVE SUBROUTINE AMR(n,ladvance_opt)
     USE TreeLevelComms
     USE DataLevelComms
     INTEGER :: n, nSteps, step
     INTEGER :: iErr
     LOGICAL, OPTIONAL :: ladvance_opt
     LOGICAL :: lHasData, lHasChildren, lHasParents, lHasParentswData, lHasChildrenwData, lOverlapsAreOld
     LOGICAL :: lAdvance
     CALL StartTimer(iAMR, n)
     lHasData = (n >= 0)
     lHasParentswData = (n >= 1)
     lHasChildren = (n < MaxLevel)
     lHasParents = lRegridLevel(n) !(n > BaseLevel)
     nsteps=levels(n)%steps      
     lAdvance=.true.
     IF (PRESENT(lAdvance_opt)) lAdvance=lAdvance_opt
     IF (.NOT. lAdvance) nSteps=1

     ! ========================================================!
     !                        Stage 1                          !
     ! Complete communication with parents and get level load  !
     ! ========================================================!
     IF (lRegridLevel(n)) THEN
        CALL CompRecvGridsFromParents(n)
        CALL CompSendGridsToChildren(n-1)
        CALL PostRecvOverlapsNeighbors(n)
        IF (lHasParentswData) CALL PostRecvParentsData(n)
        CALL SortNodes(n)

        IF (lHasData) CALL InitInfos(n)

        CALL CompRecvOverlapsNeighbors(n)
        CALL CompSendOverlapsNeighbors(n-1)

        IF (lHasParentswData) THEN
           CALL CompRecvParentsData(n)         
           CALL CompSendChildrenData(n-1)         
           CALL ProlongateParentsData(n)     
        END IF

        IF (lHasData) CALL ChildMaskOverlaps(n) !need to know about neighbors for elliptic calls
!        CALL CompRecvOldNodeOverlaps(n)        
     ELSE
        IF (lHasParentswData) THEN
           CALL PostRecvParentsData(n)
           CALL CompRecvParentsData(n)         
           CALL CompSendChildrenData(n-1)         
           CALL ProlongateParentsData(n)     
        END IF
     END IF
     IF (lHasData) CALL GetLevelLoad(n)         
     IF (lHasParentswData) CALL ClearParentFixups(n)

     ! ========================================================!
     !                        Stage 2                          !
     ! Begin AMR steps                                         !
     ! ========================================================!

     DO step=1,nsteps
        IF (lHasData) CALL ClearFixupFluxes(n)
        CALL UpdateLevelStep(n, step, lAdvance)
        lOverlapsAreOld = (step == 1 .AND. lRegridLevel(n)) ! On first step of non-persistent level, overlaps come from old grids

        ! ========================================================!
        !                        Stage 2a                         !
        ! Fill Ghost zones with overlap data                      !
        ! ========================================================!

        IF (.NOT. lRegridLevel(n) .OR. step == 2) CALL UpdateOverlaps(n)  ! On second and subsequent steps - ghost zones are filled from neighbors - so overlaps=>neighbors

        IF (lHasData) THEN
           CALL PostRecvOverlaps(n)
           CALL BeforeGlobalStep(n)
           CALL PostSendOverlaps(n)
           CALL ClearCompletedWorkLoads(n)
           CALL ApplyOverlaps(n,step)
           CALL CompSendOverlaps(n)
           CALL CompRecvOverlaps(n)
           IF (lHasParentswData .AND. step == 1) CALL AfterOverlaps(n)
           CALL ParticlePreUpdate(n)
           CALL ApplyPhysicalBCs(n)
# if defined HYPRE
           IF (lElliptic) CALL ApplyEllipticBC(n)
           IF (lElliptic .AND. (.NOT. lAdvance .OR. lRegridLevel(n))) THEN  !Don't need to solve again if we have not regridded.
              IF (lNeedMeanDensity) CALL UpdateMeanDensity(n)
              IF (n == 0 .AND. lRegridLevel(0)) CALL TransferTimeDerivs(0)
              CALL InitialElliptic(n)
           END IF
# endif
           IF(lExplicit) CALL InitialExplicit(n)
        END IF

        ! ========================================================!
        !                        Stage 2b                         !
        ! Age children and create new level n+1 nodes             !
        ! ========================================================!

        IF (lHasChildren) THEN
           IF (lRegridLevel(n+1)) THEN
              IF (lHasData) CALL SetErrFlags(n)
              CALL AgeNodesChildren(n)
              CALL AgeNodes(n+1)
              CALL CreateChildrens(n) 
#if defined PTH
              IF (iThreaded == THREADED) CALL pth_MPI_BARRIER2(MPI_COMM_WORLD, ierr)
#endif
              CALL DistributeChildrens(n)


           ! ========================================================!
           !                        Stage 2c                         !
           ! Communicate child information with neighbors            !
           ! If iThreaded == NON_THREADED then we can do level -1    !
           ! advances to give everyone a chance to catch up          !
           ! Also post send/recv of child information to child procs !
           ! ========================================================!

              CALL PostRecvNeighboringChildren(n)
              CALL PostSendNeighboringChildren(n)      
              IF (lOverlapsAreOld) THEN
                 CALL PostRecvOverlappingChildrenFromOldNodes(n)  
                 CALL PostRecvOverlappingChildrenFromNewNodes(n)
                 CALL PostSendOverlappingChildrenToOldNodes(n)    
                 CALL PostSendOverlappingChildrenToNewNodes(n)    
                 CALL InheritOldNodeOverlapsChildren(n)          
                 CALL InheritNewNodeOverlapsChildren(n) 
                 CALL InheritNeighborsChildren(n)
              ELSE
                 CALL InheritOverlapsOldChildren(n) 
                 CALL InheritNeighborsChildren(n)
              END IF
           END IF
           IF (lHasData) CALL ClearChildFixups(n)
           ! If we need to pre-advance coarse grids then do so now.
           IF (iThreaded == NON_THREADED .AND. lHasData .AND. lAdvance) THEN
              !advance and call elliptic and update qchild(iPhiDot) before sending children data
              CALL AdvanceGrids(n)
#if defined HYPRE                                            
              IF (lElliptic) THEN
                 IF (lNeedMeanDensity) CALL UpdateMeanDensity(n)
                 CALL Elliptic(n)
              END IF
#endif
              IF(lExplicit) THEN
                 CALL Explicit(n)
              END IF
              CALL ParticlePostUpdate(n)
              CALL PrintAdvance(n)            
              CALL UpdateTimeDerivs(n)
           END IF
           
           IF (lRegridLevel(n+1)) THEN
           
              IF (lOverlapsAreOld) THEN
                 CALL CompRecvOverlappingChildrenFromOldNodes(n)
                 CALL CompRecvOverlappingChildrenFromNewNodes(n)
                 CALL PostSendOverlapsToOldNodesChildren(n)
                 CALL PostRecvOldNodeOverlaps(n+1)
                 CALL CompRecvNeighboringChildren(n)
                 CALL CompSendOverlappingChildrenToNewNodes(n)
                 CALL CompSendOverlappingChildrenToOldNodes(n)
                 CALL CompSendOverlapsToOldNodesChildren(n)        
                 CALL CompRecvOldNodeOverlaps(n+1)        
                 CALL CompSendNeighboringChildren(n)
              ELSE
                 CALL CompRecvNeighboringChildren(n)
                 CALL InheritOverlapsNewChildren(n) 
                 CALL PostRecvOldNodeOverlaps(n+1)
                 CALL PostSendOverlapsToNodesOldChildren(n)        
                 CALL CompSendNeighboringChildren(n)
                 CALL CompSendOverlapsToNodesOldChildren(n)
                 CALL CompRecvOldNodeOverlaps(n+1)                
              END IF

              CALL PostSendGridsToChildren(n)         
              CALL PostRecvGridsFromParents(n+1)

              CALL PostSendOverlapsNeighbors(n)         
           END IF
           
           IF (lHasData) CALL PostSendChildrenData(n)         
           IF (lAdvance .AND. lHasData) CALL PostRecvChildrenData(n)         

           IF (.NOT. lAdvance) THEN
              CALL ParticlePostUpdate(n)
              CALL AMR(n+1, lAdvance_opt=.false.)
           END IF
        END IF


        IF (lAdvance) THEN

           ! ========================================================!
           !                        Stage 2d                         !
           ! CALL AMR and/or Advance Grids depending on threading    !
           ! choice                                                  !
           ! ========================================================!

           IF (iThreaded == NON_THREADED) THEN
              IF (lHasChildren) THEN
                 CALL AMR(n+1) 
              ELSE
                 CALL AdvanceGrids(n)
#if defined HYPRE                                            
                 IF (lNeedMeanDensity) CALL UpdateMeanDensity(n)
                 IF (lElliptic) CALL Elliptic(n)
#endif
                 IF(lExplicit) CALL Explicit(n)
                 CALL ParticlePostUpdate(n)
                 CALL PrintAdvance(n)            
              END IF
           ELSE IF (iThreaded == PSEUDO_THREADED) THEN 
              IF (lHasData)  CALL ScheduledAdvanceGrids(n)
              IF (lHasChildren) CALL AMR(n+1)            
              IF (lHasData) CALL CompleteAdvanceGrids(n)
#if defined PTHREADS || defined PTH
           ELSEIF (iThreaded == THREADED) THEN            
              IF (lHasData) CALL LaunchAdvanceThread(n)
              IF (lHasChildren) CALL AMR(n+1)
              IF (lHasData) CALL JoinAdvanceThread(n)
# endif
           END IF


           ! ========================================================!
           !                        Stage 2e                         !
           ! Synchronize Data with children                          !
           ! ========================================================!
           IF (lHasChildren) THEN
              IF (lHasData) THEN
                 CALL ApplyChildrenData(n)
                 CALL CompSendParentsData(n+1)         
                 CALL CompRecvChildrenData(n)         
              END IF
           END IF


           ! ========================================================!
           !                        Stage 2e                         !
           ! Synchronize Data with neighbors                         !
           ! and if threaded do some advances to kill time           !
           ! ========================================================!

           IF (lHasData) THEN
              CALL RestrictionFixups(n) 
              CALL AfterFixups(n)
              CALL PostRecvFluxes(n)         
              CALL PostSendFluxes(n)      

              IF (iThreaded == PSEUDO_THREADED .OR. iThreaded == THREADED) THEN
                 IF (iThreaded == PSEUDO_THREADED .AND. n > 0) CALL WaitingAdvances(n)
#if defined PTH
                 IF (iThreaded == THREADED .AND. (lElliptic .OR. lParticles)) CALL pth_MPI_BARRIER(MPI_COMM_WORLD, ierr)
#endif
#if defined HYPRE                                            
                 IF (lElliptic) CALL Elliptic(n)
#endif
                 IF(lExplicit) CALL Explicit(n)
                 IF (lParticles) CALL ParticlePostUpdate(n)
                 CALL PrintAdvance(n)            
              END IF
              IF (lHasChildren .AND. lRegridLevel(n+1)) CALL UpdateChildMasks(n) !In order to synchronize fluxes we need to know about neighbors children
              CALL SyncFluxes(n) 
              CALL CompSendFluxes(n)          
              CALL CompRecvFluxes(n)  
              IF (lHasParentswData) CALL AccumulateFluxes(n)
           END IF
!           IF (step == 2) CALL NullifyNeighbors(n) !Since neighbors became overlaps in UpdateOverlaps - we want to avoid deallocating the list twice
           IF (RestartStep) EXIT
        END IF
     END DO


     ! ========================================================!
     !                        Stage 2f                         !
     ! Coarsen Data for parents                                !
     ! ========================================================!
     IF (lAdvance) THEN
        IF (lHasParentswData) THEN
           CALL CoarsenDataForParents(n) 
           CALL PostSendParentsData(n)                    
        END IF
     END IF
     CALL StopTimer(iAMR, n)
   END SUBROUTINE AMR




   !> An "AMR lite" routine that creates a tree with no ghost zones or communication.
   !! @details This routine creates a refined AMR tree, but without initializing any ghost zones or communication.  
   !! This tree is necessary in order to calculate CPU load and write an initial data file, but should not call any communication routines, since there is no communication required yet.
   !! @param n current level of node tree
   RECURSIVE SUBROUTINE AMRStart(n)
     USE TreeLevelComms
     USE DataLevelComms
     INTEGER :: n, nSteps, step, ierr
     LOGICAL :: lHasData, lHasChildren, lHasParents, lHasParentswData, lHasChildrenwData, lOverlapsAreOld

     lHasData = (n >= 0)
     lHasParentswData = (n >= 1 .AND. .NOT. lRestart)
     lHasChildren = ((.NOT. lRestart .AND. n < MaxLevel) .OR. (lRestart .AND. n < RestartLevel))
     lHasParents = (n > -2)
     nsteps=1
     step=1
     levels(n)%step=1
     levels(n+1:)%step=1
     IF (n > -1 .AND. MPI_ID == 0)  THEN
        IF (lRestart) THEN
           write(*,*) 'Reloading Grids on level', n
        ELSE
           write(*,*) 'Initializing Grids on level', n
        END IF
     END IF
     ! ========================================================!
     !                        Stage 1                          !
     ! Complete communication with parents and get level load  !
     ! ========================================================!

     IF (lHasParents) THEN
        CALL CompRecvGridsFromParents(n)
        CALL CompSendGridsToChildren(n-1)         
        CALL PostRecvOverlapsNeighbors(n)
        IF (lHasParentswData) CALL PostRecvParentsData(n) 
        CALL SortNodes(n)

        IF (lRestart) THEN
           IF (lHasData) CALL InitInfos(n)
        ELSE
           IF (lHasData) CALL InitialInitInfos(n)
        END IF

        CALL CompRecvOverlapsNeighbors(n)        
        CALL CompSendOverlapsNeighbors(n-1)

        IF (lHasParentswData) THEN
           CALL CompRecvParentsData(n)         
           CALL CompSendChildrenData(n-1)
           CALL ProlongateParentsData(n)     
        END IF

        IF (lHasData) CALL ChildMaskOverlaps(n) !need to know about neighbors for elliptic calls
     END IF
     IF (lHasData) CALL GetLevelLoad(n)


     ! ========================================================!
     !                        Stage 2a                         !
     ! Initialize grids and ghost data                         !
     ! ========================================================!

     IF (lRestart) THEN
        CALL IOReloadLevel(n)
     ELSE
        IF (lHasData) THEN
           IF (lHasParentswData) CALL AfterOverlaps(n)
           CALL InitGrids(n)
           CALL BeforeGlobalStep(n)
           IF (lNeedMeanDensity) CALL UpdateMeanDensity(n)
           levels(n)%step=levels(n)%steps
           CALL UpdateOverlaps(n)  ! On last step - ghost zones are filled from neighbors - so overlaps=>neighbors
           CALL PostRecvOverlaps(n)        
           CALL PostSendOverlaps(n,.true. )        
           CALL ApplyOverlaps(n,levels(n)%step) !Ghost zones are filled from neighbors
           CALL CompSendOverlaps(n)                    
           CALL CompRecvOverlaps(n)
           levels(n)%step=step
           CALL ApplyPhysicalBCs(n)
           CALL ParticlePreUpdate(n)
           CALL ParticlePostUpdate(n)
# if defined HYPRE
           IF (lElliptic) THEN
              CALL ApplyEllipticBC(n)
              CALL InitialElliptic(n)
           END IF
# endif
           IF (lExplicit) CALL InitialExplicit(n)
        END IF
     END IF

     ! ========================================================!
     !                        Stage 2b                         !
     ! Create children                                         !
     ! ========================================================!

     IF (lHasChildren) THEN 
        IF (lRestart) THEN
           CALL DistributeChildrens(n, .false.) !no splitting allowed
        ELSE
           IF (lHasData) CALL SetErrFlags(n)
           CALL CreateChildrens(n) 
           CALL DistributeChildrens(n)
        END IF
  
        ! ========================================================!
        !                        Stage 2c                         !
        ! Communicate child information with neighbors            !
        ! Also post send/recv of child information to child procs !
        ! ========================================================!

        CALL PostRecvNeighboringChildren(n)
        CALL PostSendNeighboringChildren(n)      
        CALL InheritNeighborsChildren(n)
        CALL CompRecvNeighboringChildren(n)   
        CALL CompSendNeighboringChildren(n)     
        
        CALL PostSendGridsToChildren(n)         
        CALL PostRecvGridsFromParents(n+1)

        CALL PostSendOverlapsNeighbors(n)         

        IF (.NOT. lRestart) THEN
           IF (lHasData) CALL ClearChildFixups(n)
           IF (lHasData) CALL PostSendChildrenData(n)         
           IF (lHasData) CALL PostRecvInitialChildrenData(n)         
        END IF

        ! ========================================================!
        !                        Stage 2d                         !
        ! CALL AMRStart                                           !
        ! ========================================================!
        
        IF (lHasChildren) CALL AMRStart(n+1)         


        ! ========================================================!
        !                        Stage 2e                         !
        ! Synchronize Data with children                          !
        ! ========================================================!

        IF (.NOT. lRestart) THEN
           IF (lHasChildren) THEN
              IF (lHasData) THEN
                 CALL ApplyInitialChildrenData(n)
                 CALL CompSendParentsInitialData(n+1)           
                 CALL CompRecvInitialChildrenData(n)
              END IF
           END IF
        END IF
        ! ========================================================!
        !                        Stage 2e                         !
        ! Synchronize Data with neighbors                         !
        ! ========================================================!
     END IF
     IF (lHasData) CALL AfterFixups(n)
!     IF (n > BaseLevel) CALL NullifyNeighbors(n)

     ! ========================================================!
     !                        Stage 2f                         !
     ! Coarsen Data for parents                                !
     ! ========================================================!
        
     IF (lHasParentswData) THEN
        CALL CoarsenInitialDataForParents(n) 
        CALL PostSendParentsInitialData(n)                    
     END IF
        
  END SUBROUTINE AMRStart


   !> @}


   !> @name Miscellaneous routines
   !! @{

   !> Deletes all nodes from a given level on up.
   !! @param first_layer_gone The first level to be completely stripped of nodes.   
   SUBROUTINE ClearAllNodeLists(first_layer_gone)

       INTEGER :: first_layer_gone
       INTEGER :: n

       DO n = first_layer_gone, MaxLevel
           CALL AgeNodes(n)
!           CALL NullifyNodes(n)
       END DO

   END SUBROUTINE ClearAllNodeLists

   !> Destroys all nodes
   SUBROUTINE DestroyAllNodes()
       INTEGER :: n
       DO n = MaxLevel, -2, -1
           CALL DestroyOldNodes(n)
       END DO
       DO n = MaxLevel, -2, -1
         CALL DestroyNodeList(BackupNodes(n)%p)
         CALL DestroyNodeList(BackupExternalNodes(n)%p)
       END DO

       DO n = MaxLevel, -2, -1
          !IF (n <= BaseLevel) CALL NullifyNeighbors(n)
           CALL DestroyNodes(n)
       END DO
   END SUBROUTINE DestroyAllNodes



   !> Routine for creating a restart point without dumping to disk
   !! Needs to copy the old nodes, their overlap data(internal values of q, aux, & costmap), and pointers to their parent and child nodes
   !! Also needs to call any user routines that have to do with restart data (sink particles for example)
   SUBROUTINE BackupData()     
      INTEGER :: n
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iBackupData, BaseLevel)
      ! First make backup copies of all nodes and their info structures...
      DO n=BaseLevel, MaxLevel
         !Destroy Previous Backups
         CALL DestroyNodeList(BackupNodes(n)%p)
         CALL DestroyNodeList(BackupExternalNodes(n)%p)
         !Make new Backups
         CALL BackupNodelist(Nodes(n)%p, BackupNodes(n)%p, .false.)
         CALL BackupNodelist(ExternalNodes(n)%p, BackupExternalNodes(n)%p, .false.)
      END DO
      ! Now make a backup of any necessary relationships.
      ! For static nodes - the parent and neighbor relationships don't need to be backed up
      ! However if the children are not static - these connections will be lost unless backed up.
      DO n=LastStaticLevel, MaxLevel
         nodelist=>BackupNodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            CALL BackupChildren(n, nodelist%self)
            IF (n > LastStaticLevel) CALL BackupParent(n, nodelist%self)
            nodelist=>nodelist%next
         END DO
         nodelist=>BackupExternalNodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            CALL BackupChildren(n, nodelist%self)
            nodelist=>nodelist%next
         END DO
      END DO
      CALL SinkParticleBackup()
      CALL StopTimer(iBackupData, BaseLevel)
   END SUBROUTINE BackupData



   !> Routine for creating a restart point without dumping to disk
   !! Needs to restore nodes, their overlap data(internal values of q, aux, & costmap), and pointers to their parent and child nodes
   !! Also needs to call any user routines that have to do with restart data (sink particles for example)
   SUBROUTINE RestoreData()     
      INTEGER :: n
      TYPE(NodeDefList), POINTER :: nodelist,backedupnodelist
      TYPE(NodeDef), POINTER :: node
      TYPE(InfoDef), POINTER :: tempinfo
      ! First restore info structures of permanent nodes
      DO n=0, min(LastStaticLevel, MaxLevel)
         nodelist=>Nodes(n)%p
         backedupnodelist=>BackupNodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            tempinfo=>nodelist%self%info
            NULLIFY(nodelist%self%info)
            CALL BackupInfo(backedupnodelist%Self%info, nodelist%self%info, .true.)
            IF (LastStaticLevel > n) THEN !Keep childfixups
               nodelist%self%info%childfixups=>tempinfo%childfixups
               nullify(tempinfo%childfixups)
            END IF
            CALL DestroyInfo(tempinfo)
            nodelist=>nodelist%next
            backedupnodelist=>backedupnodelist%next
         END DO
      END DO

      ! Now restore old nodes
      DO n=LastStaticLevel+1, MaxLevel
         CALL DestroyNodeList(Nodes(n)%p)
         NULLIFY(Nodes(n)%p, LastLocalNode(n)%p)
         CALL BackupNodelist(BackupNodes(n)%p,Nodes(n)%p, .true., LastLocalNode(n)%p)
         CALL DestroyNodeList(ExternalNodes(n)%p)
         NULLIFY(ExternalNodes(n)%p, LastExternalNode(n)%p)
         CALL BackupNodelist(BackupExternalNodes(n)%p,ExternalNodes(n)%p, .true.,LastExternalNode(n)%p)         
      END DO

      ! First clear current child connections for last static level
      ! Now re-establish parent and child connections between restored nodes
      DO n=LastStaticLevel, MaxLevel
         nodelist=>Nodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            node=>nodelist%self
            IF (n == LastStaticLevel) THEN
               CALL ClearNodeList(node%children)
               NULLIFY(node%lastchild)
            END IF
            IF (n > LastStaticLevel) CALL RestoreParent(n, node)
            CALL RestoreChildren(n, node)
            nodelist=>nodelist%next
         END DO

! Possible that an external node would have been added since backup because of a child being distributed locally
         nodelist=>ExternalNodes(n)%p
         DO WHILE (ASSOCIATED(nodelist))
            node=>nodelist%self
            IF (n == LastStaticLevel) THEN
               CALL ClearNodeList(node%children)
               NULLIFY(node%lastchild)
            END IF
            CALL RestoreChildren(n, node, n /= LastStaticLevel)
            nodelist=>nodelist%next
         END DO
      END DO
      CALL SinkParticleRestore()
    END SUBROUTINE RestoreData



    SUBROUTINE UpdateLevelStep(n, step, lAdvance)
      INTEGER :: n, step
      LOGICAL :: lAdvance
      IF (lAdvance) THEN
         IF (n > ROOT_LEVEL) THEN
            levels(n)%CurrentLevelStep=levels(n-1)%CoarsenRatio*(levels(n-1)%CurrentLevelStep-1)+step
         ELSE
            levels(n)%CurrentLevelStep=levels(n)%CurrentLevelStep+1
         END IF
      END IF
      levels(n)%step=step
      levels(n+1:)%step=1
    END SUBROUTINE UpdateLevelStep

   !> Tests MPI_Packing...
   SUBROUTINE MpiTest
      USE TreeLevelComms
      CALL PackTest
   END SUBROUTINE MpiTest



   SUBROUTINE PrintAllocations
      REAL(KIND=qPREC), DIMENSION(3) :: totalallocators, maxallocators
      REAL(KIND=qPREC) :: totalcells(0:MaxDepth), cupspcpu, walltimeleft, cuprs
      INTEGER :: iErr, i
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: TotalWorkDone
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: TotalEllipticWorkDone
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: TotalInternalCellUpdates
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: TotalExternalCellUpdates
      !     CALL MPI_ALLREDUCE(maxallocation, maxallocators, 3, MPI_DOUBLE_PRECISION, MPI_MAX, MPI_COMM_WORLD, iErr)
      !     CALL MPI_ALLREDUCE(allocator, totalallocators, 3, MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, iErr)
      !     CALL MPI_ALLREDUCE(NumCellsByLevel, totalcells, MaxDepth+1, MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, iErr)
      !     IF (totalallocators(InfoAllocator) > 25.0*1024.0**3) THEN
      !        lPrintDebugFrame=.true.
      !     END IF
      ALLOCATE(TotalExternalCellUpdates(0:MaxLevel))
      ALLOCATE(TotalInternalCellUpdates(0:MaxLevel))
      ALLOCATE(TotalWorkDone(0:MaxLevel))
      ALLOCATE(TotalEllipticWorkDone(0:MaxLevel))

      CALL MPI_REDUCE(maxallocation, maxallocators, 3, MPI_DOUBLE_PRECISION, MPI_MAX, 0, MPI_COMM_WORLD, iErr)
      CALL MPI_REDUCE(allocator, totalallocators, 3, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      CALL MPI_REDUCE(NumCellsByLevel, totalcells, MaxDepth+1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      CALL MPI_REDUCE(REAL(InternalCellUpdates, KIND=qPREC), TotalInternalCellUpdates, MaxLevel+1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      CALL MPI_REDUCE(REAL(CellUpdates, KIND=qPREC), TotalExternalCellUpdates, MaxLevel+1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      CALL MPI_REDUCE(Timers(iAdvanceGrids)%Accumulator(0:MaxLevel), TotalWorkDone, MaxLevel+1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      IF (lElliptic) CALL MPI_REDUCE(Timers(iElliptic)%Accumulator(0:MaxLevel), TotalEllipticWorkDone, MaxLevel+1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
      
      IF (MPI_ID == 0) THEN
         cupspcpu=REAL(SUM(TotalInternalCellUpdates))/(MPI_NP*(mpi_wtime()-StartTime))
         cuprs=REAL(SUM(totalcells(0:MaxLevel)*2**(/(i,i=0,MaxLevel)/))) !/cupspcpu/REAL(MPI_NP)
         walltimeleft=(final_time-levels(0)%tnow)/max(levels(0)%dt,olddt) * cuprs / (cupspcpu*MPI_NP)

         write(*,'(A23,2A10)')       'Info allocations    = ', printsize(totalallocators(1)), printsize(maxallocators(1))
         write(*,'(A23,2A10)')       'message allocations = ', printsize(totalallocators(2)), printsize(maxallocators(2))
         write(*,'(A23,2A10)')       'sweep allocations   = ', printsize(totalallocators(3)), printsize(maxallocators(3))
         write(*,'(A23,100F7.3)')    'filling fractions   = ', (/(merge(real(totalcells(i))/real(totalcells(i-1)*levels(i-1)%CoarsenRatio**nDim),0.0,totalcells(i-1)>0), i=1, MaxLevel)/)
         IF (lElliptic) THEN
            write(*,'(A23,I3,A2,I3,A2,I3,A2)')      'Current efficiency  = ', nint(SUM(TotalWorkDone)/(MPI_NP*(mpi_wtime()-StartTime))*100d0), '% ',nint(SUM(TotalEllipticWorkDone)/(MPI_NP*(mpi_wtime()-StartTime))*100d0), '% ',nint(SUM(TotalWorkDone+TotalEllipticWorkDone)/(MPI_NP*(mpi_wtime()-StartTime))*100d0), '% '
         ELSE
            write(*,'(A23,I3,A2)')      'Current efficiency  = ', nint(SUM(TotalWorkDone)/(MPI_NP*(mpi_wtime()-StartTime))*100d0), '% '
         END IF
         write(*,'(A23,2I10,I4,A1)') 'Cell updates/second = ',  nint(cupspcpu), nint(REAL(SUM(TotalExternalCellUpdates))/(MPI_NP*(mpi_wtime()-StartTime))), nint(100*SUM(TotalInternalCellUpdates)/SUM(TotalExternalCellUpdates)), '%'
         write(*,'(A23,A11,A10,F6.1,A4,I6)')        'Wall Time Remaining = ', printtime(walltimeleft), ' at frame ', (levels(0)%tnow-start_time)/(final_time-start_time)*REAL(final_frame-start_frame)+start_frame, ' of ',final_frame

         write(*,'(A23,E16.4)')      'AMR Speed-Up Factor = ', .95*AdvanceCost(max(1,nint(REAL(levels(MaxLevel)%mX)/(REAL(MPI_NP)**(1d0/REAL(nDim))))))*2d0**MaxLevel/(cuprs/(cupspcpu*MPI_NP))
      END IF
      
      
      DEALLOCATE(TotalWorkDone, TotalInternalCellUpdates, TotalEllipticWorkDone, TotalExternalCellUpdates)
      
   END SUBROUTINE PrintAllocations

   !> @}



END MODULE AmrControl

