!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    time_step.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file time_step.f90
!! @brief Main file for module TimeStep

!> @defgroup TimeStep TimeStep
!! @brief Provide a set of functions and parameters for timestep and CFL management.
!! @ingroup AMR



!============================================================================================
! Public Methods:    RelaxTimeStep(), TestBadCFL(), MIN_TIMESTEP
! Created:            20100629 by Brandon D. Shroyer
! Notes:            Not a strictly necessary module, but the functionality here is replicated
!                   across both MUSCL and sweep, so there's no reason to try and get it right
!                   twice.
!============================================================================================

!> @brief Provide a set of functions and parameters for timestep and CFL management.
!! @ingroup TimeStep
MODULE TimeStep

   USE DataDeclarations
   USE GlobalDeclarations
   USE EllipticDeclarations
   USE ExplicitDeclarations
   USE HyperbolicDeclarations
   USE ParticleDeclarations
   USE Timing
   IMPLICIT NONE
   SAVE
   PRIVATE

   PUBLIC RelaxTimeStep, TestBadCFL, TimeStepInit, GetNextTimeStep, GetFirstTimeStep, PrintAdvance

   REAL(KIND=qPREC), PUBLIC :: maxcfl
   REAL, PARAMETER :: BAD_CFL_THRESHOLD = 1d3				! Kill the program if the CFL goes above this.
   REAL(KIND=qPrec), PARAMETER :: MIN_TIMESTEP = 1.d-32
   ! CFL-management variables and their index parameters.
   REAL(KIND=qPREC), PUBLIC :: overall_maxspeed, olddt
   INTEGER, PARAMETER :: MAX_CFL = 1
   INTEGER, PARAMETER :: TARGET_CFL = 2
   INTEGER, PARAMETER :: CFL_RELAXATION = 3   
!   INCLUDE 'mpif.h'
CONTAINS

!> Reads in parameters related to time step control
   SUBROUTINE TimeStepInit()
      INTEGER :: iErr
      ! Open and read in data from the physics data file.
!      READ(GLOBAL_DATA_HANDLE, NML=TimeStepData, IOSTAT=ierr)

!      IF (iErr /= 0) THEN 
!         PRINT *, "TimeStepInit() error: unable to read TimeStepData namelist."
!         STOP
!      END IF
      CLOSE(GLOBAL_DATA_HANDLE, IOStat=iErr)

!      IF (iErr /= 0) THEN
!         PRINT *, "TimeStepInit() error:  unable to close ", GLOBAL_DATA_FILE, "."
!      END IF


   END SUBROUTINE TimeStepInit

!> Determines first time step from initial_maxspeed
   SUBROUTINE GetFirstTimeStep
      overall_maxspeed=initial_maxspeed
      olddt=cfl_vars(TARGET_CFL) * (levels(ROOT_LEVEL)%dX/overall_maxspeed)!levels(ROOT_LEVEL)%dt
      maxcfl=cfl_vars(TARGET_CFL)
   END SUBROUTINE GetFirstTimeStep


!> Gets the next time step up to maxdt
!! @param maxdt - maximum next time step
!! @param lasttimestep - logical flag set to true if next time step is maxdt
   SUBROUTINE GetNextTimeStep(maxdt, lasttimestep)
      INTEGER :: level    
      REAL(KIND=qPREC), OPTIONAL :: maxdt
      REAL(KIND=qPREC) :: newdt
      LOGICAL :: lasttimestep
      newdt=RelaxTimeStep()
      IF (PRESENT(maxdt)) THEN
         levels(ROOT_LEVEL)%dt = min(newdt, maxdt)
         if (newdt >= maxdt) THEN
            lasttimestep=.true.
         ELSEIF (newdt > .8*maxdt) THEN !should be able to guarantee two steps so shorten this one to avoid a tiny time step later.
            levels(ROOT_LEVEL)%dt = .5*maxdt
         ELSE
            olddt=newdt !create backup of full time step to use for future time step calculations
         END if
!         IF (levels(ROOT_LEVEL)%dt <= maxdt) lasttimestep=.true.
      ELSE
         levels(ROOT_LEVEL)%dt = newdt
         olddt=newdt !create backup of full time step
      END IF
      DO level=ROOT_LEVEL+1,MaxLevel
         levels(level)%dt = levels(level-1)%dt / levels(level-1)%CoarsenRatio
      END DO
      RestartStep=.false.
      lRequestRestart=.false.
   END SUBROUTINE GetNextTimeStep



   !> @brief  Use the CFL and the current timestep to calculate a new timestep.
   ! Inputs:    A CFL number (should be between 0 and 1) and a timestep value.
   ! Globals:    Relies on the global array cfl_vars for target and relaxation parameters.  The
   !           cfl_vars array and its index constants can be found in global_declarations.f90.
   ! Returns:    A timestep that has been "relaxed" towards the target CFL value and away from the 
   !           current CFL value.
   ! Created:    20100629 by Brandon D. Shroyer   

   FUNCTION RelaxTimeStep()
      REAL(KIND=qPrec) :: RelaxTimeStep
      REAL(KIND=qPrec) :: dt_wanted


      ! Establish a target timestep.
      dt_wanted = cfl_vars(TARGET_CFL) * levels(ROOT_LEVEL)%dX / overall_maxspeed

      ! Create a timestep adjusted in the direction of dt_wanted.  The relaxation parameter
      ! limits the degree to which the timestep can be shifted.
      IF (restartstep) THEN
         RelaxTimeStep=levels(ROOT_LEVEL)%dt*.5d0
      ELSE IF (cfl_vars(TARGET_CFL) > maxcfl) THEN
         RelaxTimeStep = (dt_wanted - olddt) * cfl_vars(CFL_RELAXATION) + olddt
      ELSE
         RelaxTimeStep = MAX((dt_wanted - olddt) / cfl_vars(CFL_RELAXATION) + olddt, levels(ROOT_LEVEL)%dt*cfl_vars(CFL_RELAXATION))
      END IF

      ! Make sure time step is above a minimum timestep
      RelaxTimeStep = MAX(RelaxTimeStep, MIN_TIMESTEP)
      maxspeed=0
      IF (lElliptic) elliptic_maxspeed=0
      IF (lExplicit) explicit_maxspeed=0
      maxcfl=0
   END FUNCTION RelaxTimeStep

   !> @brief Print an error message and kill the program if the CFL number is irreparably high.
   !! @author 20100629 by Brandon D. Shroyer
   ! Inputs:    A CFL number to be tested.
   SUBROUTINE TestBadCFL(lSuccess)
      LOGICAL :: lSuccess
      REAL(KIND=qPREC) :: LevelMaxSpeeds(4)
      INTEGER :: iErr
      CALL StartTimer(iTestBadCFL, -2)
      IF (lRequestRestart) THEN
         LevelMaxSpeeds(1)=1
      ELSE
         LevelMaxSpeeds(1)=0
      END IF

      LevelMaxSpeeds(2)=maxval(maxspeed(0:MaxLevel))
      LevelMaxSpeeds(3)=maxval(explicit_maxspeed(0:MaxLevel))

# if defined HYPRE
      LevelMaxSpeeds(4)=maxval(elliptic_maxspeed(0:MaxLevel))
      CALL StartTimer(iBarrier, 0)
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, LevelMaxSpeeds, 4, MPI_DOUBLE_PRECISION, MPI_MAX, levels(0)%MPI_COMM, iErr)
      CALL StopTimer(iBarrier, 0)
      overall_maxspeed=maxval(LevelMaxSpeeds(2:4))
# else
      CALL StartTimer(iBarrier, 0)
      CALL MPI_ALLREDUCE(MPI_IN_PLACE, LevelMaxSpeeds, 3, MPI_DOUBLE_PRECISION, MPI_MAX, levels(0)%MPI_COMM, iErr)    
      CALL StopTimer(iBarrier, 0)
      overall_maxspeed=maxval(LevelMaxSpeeds(2:3))
# endif


      IF (LevelMaxSpeeds(1) == 1) THEN !Someone requested a restart
         RestartStep=.true.
      END IF

      IF (lParticles) overall_maxspeed=max(overall_maxspeed,ParticleMaxSpeed)

      maxcfl=overall_maxspeed*levels(0)%dt/levels(0)%dx
      IF (maxcfl > cfl_vars(MAX_CFL) .OR. RestartStep) THEN
         IF (MPI_ID == 0) THEN
            IF (maxcfl > cfl_vars(MAX_CFL)) THEN
               write(*,'(A,E13.5,A,E13.5)') 'High CFL - restarting step. ', maxcfl, '  >  ', cfl_vars(MAX_CFL)
            ELSE
               write(*,*) 'Restart requested'
            END IF
         END IF
         lSuccess=.false. 
         olddt=levels(ROOT_LEVEL)%dt      !Make sure we can correct the old time step         
      ELSE
         lSuccess=.true.
      END IF
      CALL StopTimer(iTestBadCFL, -2)
   END SUBROUTINE TestBadCFL

   !> Outputs on the master when the advance has been completed.
   SUBROUTINE PrintAdvance(n)
      INTEGER :: n
      INTEGER :: i
      !      IF (lRequestRestart) THEN
      !         write(*,*) 'processor', mpi_id, 'would like to restart', restartstep
      !      END IF
      IF (MPI_ID == 0 .AND. n <= FinestLevel) THEN
         WRITE(*,ADVANCE="NO",FMT='(8A2)')('  ',i=1,n)
         IF (lElliptic) THEN
            IF (lParticles) THEN
               WRITE(*,ADVANCE="YES",FMT=1003) n,&
                    levels(n)%tnow, &
                    levels(n)%dt, &
                    max(maxspeed(n),elliptic_maxspeed(n))*levels(n)%dt/levels(n)%dx, &
                    maxspeed(n), &
                    elliptic_maxspeed(n), Particlemaxspeed

            ELSE

               WRITE(*,ADVANCE="YES",FMT=1001) n,&
                    levels(n)%tnow, &
                    levels(n)%dt, &
                    max(maxspeed(n),elliptic_maxspeed(n))*levels(n)%dt/levels(n)%dx, &
                    maxspeed(n), &
                    elliptic_maxspeed(n)
            END IF
         ELSE
            IF (lParticles) THEN
               WRITE(*,ADVANCE="YES",FMT=1004) n,&
                    levels(n)%tnow, &
                    levels(n)%dt, &
                    maxspeed(n)*levels(n)%dt/levels(n)%dx, &
                    maxspeed(n), particlemaxspeed

            ELSE
               WRITE(*,ADVANCE="YES",FMT=1002) n,&
                    levels(n)%tnow, &
                    levels(n)%dt, &
                    maxspeed(n)*levels(n)%dt/levels(n)%dx, &
                    maxspeed(n)
            END IF
         END IF
1001     FORMAT(' Advanced level ',i2,' to tnext= ',e10.4,' with dt= ',e10.4,' CFL= ',e10.4,' max speed= ',e10.4,' elliptic_speed= ',e10.4)
1002     FORMAT(' Advanced level ',i2,' to tnext= ',e10.4,' with dt= ',e10.4,' CFL= ',e10.4,' max speed= ',e10.4)
1003     FORMAT(' Advanced level ',i2,' to tnext= ',e10.4,' with dt= ',e10.4,' CFL= ',e10.4,' max speed= ',e10.4,' elliptic_speed= ',e10.4, ' particle_speed= ',e10.4)
1004     FORMAT(' Advanced level ',i2,' to tnext= ',e10.4,' with dt= ',e10.4,' CFL= ',e10.4,' max speed= ',e10.4, ' particle_speed= ',e10.4)
      END IF

   END SUBROUTINE PrintAdvance

END MODULE TimeStep

