!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    Testing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE Testing
   USE GlobalDeclarations
   IMPLICIT NONE

   PUBLIC GetiTest, TestNeedsRefFrame, DoTest
   INTEGER, PUBLIC :: iTest
   INTEGER, PARAMETER :: MaxFields=20
   REAL, PUBLIC, DIMENSION(MaxFields) :: L2GenericNormPass
   REAL, PUBLIC, DIMENSION(MaxFields) :: LInfGenericNormPass
   PRIVATE
   INTEGER, PARAMETER :: iFieldLoop2D=1
   INTEGER, PARAMETER :: iJetMHD3D=2
   INTEGER, PARAMETER :: iKHTest=3
   INTEGER, PARAMETER :: iRTTest=4
   INTEGER, PARAMETER :: iGenericTest=5
   INTEGER, PARAMETER :: nTests=5

CONTAINS

   SUBROUTINE GetiTest()
      iTest=0
      DO WHILE(.NOT. (iTest > 0 .AND. iTest <= nTests))
         WRITE(*, '(A)') "Select which test problem you wish to check:"
         WRITE(*, '(I4,A)') iFieldLoop2D,")  FieldLoop2D."
         WRITE(*, '(I4,A)') iJetMHD3D,")  JetMHD3D."
         WRITE(*, '(I4,A)') iKHTest,")  KHTest."
         WRITE(*, '(I4,A)') iRTTest,")  RTTest."
         WRITE(*, '(I4,A)') iGenericTest,")  Generic Test."
         WRITE(*,'(A)',ADVANCE='NO')'Please input your selection:  '
         READ*, iTest
         IF (iTest == iGenericTest) THEN
            WRITE(*, '(A,I2)') "Please input maximum mean variance allowable for each variable from 1 to ", NrVars
            READ*, L2GenericNormPass(1:NrVars)
            WRITE(*, '(A,I2)') "Please input maximum largest error allowable for each variable from 1 to ", NrVars    
            READ*, LInfGenericNormPass(1:NrVars)
         END IF
      END DO
   END SUBROUTINE GetiTest

   LOGICAL FUNCTION TestNeedsRefFrame()
      SELECT CASE(iTest)
      CASE(iFieldLoop2D)
         TestNeedsRefFrame=.true.
      CASE(iJetMHD3D)
         TestNeedsRefFrame=.true.
      CASE(iKHTest)
         TestNeedsRefFrame=.false.
      CASE(iRTTest)
         TestNeedsRefFrame=.false.
      CASE(iGenericTest)
         TestNeedsRefFrame=.true.
      CASE DEFAULT
         TestNeedsRefFrame=.false.
      END SELECT
   END FUNCTION TestNeedsRefFrame

   SUBROUTINE DoTest(info,qref)
      TYPE(GridInfo) :: Info
      REAL, POINTER, DIMENSION(:,:,:,:), OPTIONAL :: qref
      SELECT CASE(iTest)
      CASE(iFieldLoop2D)
         CALL FieldLoop2D(info,qref)
      CASE(iJetMHD3D)
         CALL JetMHD3D(info,qref)
      CASE(iKHTest)
         CALL KHTest(info)
      CASE(iRTTest)
         CALL RTTest(info)
      CASE(iGenericTest)
         CALL GenericTest(info, qref)
      END SELECT
   END SUBROUTINE DoTest


   SUBROUTINE FieldLoop2D(info, qref)
      TYPE(GridInfo) :: Info
      REAL, POINTER, DIMENSION(:,:,:,:) :: qref
      LOGICAL :: lPassedTest
      REAL :: L1Norm
      REAL, PARAMETER :: L1NormPass=.001
      INTEGER :: i,j
      L1Norm=SUM(ABS(info%q(:,:,:,:)-qref(:,:,:,:)))/product(gi_fixed%mX(1:nDim))/NrVars
      lPassedTest= (L1Norm <= L1NormPass)
      write(*,'(A,L4,A,E18.6,A,E18.6)') "Test Result: ", lPassedTest, " FieldLoop2D Test - L1Norm = ", L1Norm, " L1NormPass= ", L1NormPass
   END SUBROUTINE FieldLoop2D

   SUBROUTINE JetMHD3D(info, qref)
      TYPE(GridInfo) :: Info

      REAL, POINTER, DIMENSION(:,:,:,:) :: qref 
      LOGICAL :: lPassedTest
      REAL :: L1Norm
      REAL, PARAMETER :: L1NormPass=.001

      L1Norm=SUM(ABS(info%q(:,:,:,:)-qref(:,:,:,:)))/product(gi_fixed%mX(1:nDim))/NrVars
      lPassedTest= (L1Norm <= L1NormPass)
      write(*,'(A,L4,A,E18.6,A,E18.6)') "Test Result: ", lPassedTest, " JetMHD3D Test - L1Norm = ", L1Norm, " L1NormPass= ", L1NormPass
   END SUBROUTINE JetMHD3D

   SUBROUTINE KHTest(Info)
     TYPE(GridInfo) :: Info
     INTEGER, PARAMETER :: lastframe=4
     REAL(KIND=qPREC), DIMENSION(0:lastframe) :: KE_y, times, KE_y_Analytic
     REAL(KIND=qPREC), PARAMETER :: KHtolPass=1e-3
     REAL(KIND=qPREC) :: KHTol
     INTEGER :: i
     LOGICAL :: lPassedTest
     KE_y(Info%frame)=half*sum(Info%q(:,:,:,ivy)**2/Info%q(:,:,:,1))*PRODUCT(Info%dx(1:ndim))
     times(Info%frame) = Info%time
     KE_y_Analytic(Info%frame)=KH_Analytic(Info%time)
     write(*,*) "frame", Info%frame, KE_y(Info%frame), times(Info%frame)
     IF (Info%frame==lastframe) THEN
        DO i=0,lastframe
           write(*,*) times(i), KE_y(i), KE_y_Analytic(i)
        END DO
        KHtol=maxval(abs(KE_y-KE_y_analytic))
        lPassedTest=KHtol <= KHtolPass
        write(*,'(A,L4,A,E18.6,A,E18.6)') "Test Result: ", lPassedTest, " KH Test - KHtol = ", KHtol, " KHtolPass= ", KHTolPass
     END IF
   CONTAINS
     FUNCTION KH_Analytic(t)
       REAL(KIND=qPREC) :: KH_Analytic, t
       KH_Analytic=t**2
     END FUNCTION KH_Analytic

   END SUBROUTINE KHTest


   SUBROUTINE RTTest(info)
      TYPE(GridInfo) :: Info
      INTEGER, PARAMETER :: lastframe=40
      REAL(KIND=qPREC), DIMENSION(0:lastframe) :: times, pos
      REAL(KIND=qPREC) :: lambda_simulation, lambda_analytic, rel_error
      REAL(KIND=qPREC) :: sum_pos, sum_t=0, sum_logpos=0, sum_t_logpos=0, sum_t2=0, N=0
      INTEGER :: i, j, counter
      OPEN(UNIT=82, FILE="GrowthRate.data", STATUS="OLD")
      READ(82,*) lambda_analytic
      CLOSE(82)

      counter = 0
      sum_pos = 0
      times(Info%frame) = Info%time
      DO i=100, 101
         DO j=250, 350
            IF ((Info%q(i,j+1,1,1) - Info%q(i,j,1,1)) / ((Info%q(i,j+1,1,1)+Info%q(i,j,1,1))/2) > 0.005) THEN
               sum_pos = sum_pos + REAL(j+1)*Info%dx(2) - Info%Xupper(2) 
               counter = counter + 1
            END IF
         END DO
      END DO
      pos(Info%frame) = sum_pos / counter
      PRINT *, pos(Info%frame)

      IF (Info%frame==lastframe) THEN
         DO i=26, lastframe
            sum_t = sum_t + times(i)
            sum_logpos = sum_logpos + LOG(pos(i))
            sum_t_logpos = sum_t_logpos + times(i)*LOG(pos(i))
            sum_t2 = sum_t2 + times(i)**2
            N = N + 1
         END DO
         lambda_simulation = (sum_t_logpos - (1/N) * sum_t * sum_logpos) / (sum_t2 - (1/N) * (sum_t)**2)
         rel_error = (ABS(lambda_simulation - lambda_analytic) / lambda_analytic) * 100
         PRINT *, "Simulation Growth Rate: ", lambda_simulation
         PRINT *, "Analytic Growth Rate: ", lambda_analytic
         PRINT *, "Relative Error: ", rel_error, "%"
      END IF
   END SUBROUTINE RTTest

   SUBROUTINE GenericTest(info, qref)
      TYPE(GridInfo) :: Info
      REAL, POINTER, DIMENSION(:,:,:,:) :: qref
      LOGICAL :: lPassedTest
      REAL :: L1Norm
      REAL, DIMENSION(MaxFields) :: L2Norms, LInfNorms
      INTEGER :: i,j      
      DO i=1, NrVars
         L2Norms(i)=sqrt(SUM((info%q(:,:,:,i)-qref(:,:,:,i))**2))/product(gi_fixed%mX(1:nDim))
         LInfNorms(i)=maxval(abs(info%q(:,:,:,i)-qref(:,:,:,i)))
      END DO
      lPassedTest= ALL(L2Norms <= L2GenericNormPass) .AND. ALL(LInfNorms <= LInfGenericNormPass)
      write(*,'(A,L4,A)') "Test Result: ", lPassedTest
      write(*,'(A50,100E13.4)') "mean variances for each variable = ", L2Norms(1:NrVars)
      write(*,'(A50,100E13.4)') "allowed variances for each variable = ", L2GenericNormPass(1:NrVars)
      write(*,*)
      write(*,'(A50,100E13.4)') "maxium error for each variable = ", LInfNorms(1:NrVars)
      write(*,'(A50,100E13.4)') "allowed maxium error for each variable = ", LInfGenericNormPass(1:NrVars)
      write(*,*)
      write(*,'(A)') 'If errors are acceptible consider resetting errors 5% higher then current in bear2fix.data as follows:'
      write(*,'(A)') "new mean variances for each variable: "
      write(*,'(A50,100E13.4)') "L2GENERICNORMPASS = ", L2Norms(1:NrVars)*1.05
      write(*,'(A)') "new maxium error for each variable: "
      write(*,'(A50,100E13.4)') "LINFGENERICNORMPASS = ", LInfNorms(1:NrVars)*1.05        
   END SUBROUTINE GenericTest
END MODULE Testing
