!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    analyze.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE analyze
  USE GlobalDeclarations
  IMPLICIT NONE
  INCLUDE 'fftw3.f'  
  SAVE

  REAL, ALLOCATABLE, DIMENSION(:) :: Spectrum, Spectrum_acc
  LOGICAL :: lSpectrum_split, lSEnergy_Spectrum, lEnergy_Spectrum, lMomentum_Spectrum, lFleck_Spectrum, lDensity_Spectrum,lBField_Spectrum
  REAL :: old_dx(MaxDims)
  INTEGER, ALLOCATABLE, DIMENSION(:,:,:) :: mapping
  REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:,:) :: weights
  INTEGER :: k_min,k_max
  !Stuff for totals
  REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: totals, my_totals 
  LOGICAL :: lTotal_KE, lTotal_v2, lTotal_vrms, lTotal_P, lTotal_Pmag, lTotal_Mass, lTotal_BE,lTotal_iE
  INTEGER :: iTotal_KEx, iTotal_KEy, iTotal_KEz, iTotal_v2x, iTotal_v2y, iTotal_v2z, iTotal_vrms, iTotal_iE, iTotal_BEx, iTotal_BEy, iTotal_BEz, iTotal_KE, iTotal_v2, iTotal_Px, iTotal_Py, iTotal_Pz, iTotal_Pmag, iTotal_Pmagx, iTotal_Pmagy, iTotal_Pmagz, iTotal_Mass, iTotal_BE
  INTEGER :: total_count=0
  INTEGER :: collapse_vars(20)=-1
  INTEGER :: collapse_dirs(20)=-1

  REAL :: data_cube_vbounds(2)
  INTEGER ::  mYmX(3), total_filehandle=17
  LOGICAL :: lPhi, lColumnDensity_Spectrum, lVelocityCentroid_Spectrum, lVortensity_Spectrum, &
       l_linewidths, lChombo, lFirstTime=.true.,lVCA

  TYPE string
    CHARACTER(LEN=10) :: s
  end TYPE string
  type(string) :: totals_name(40)

  NAMELIST /ProcessData/ lSpectrum_split, lSEnergy_Spectrum, lEnergy_Spectrum, lMomentum_Spectrum, lFleck_Spectrum, lDensity_Spectrum, lBField_Spectrum, lTotal_KE, lTotal_v2, lTotal_vrms, lTotal_P, lTotal_Pmag, lTotal_Mass, lTotal_iE, lTotal_BE, &
       collapse_vars,collapse_dirs, lPhi, lColumnDensity_Spectrum, lVortensity_Spectrum, lVelocityCentroid_Spectrum, lVCA, data_cube_vbounds

contains

  SUBROUTINE init_diagnostics(frames)
    INTEGER :: frames(2)
    OPEN(UNIT=25, FILE='process.data')
    READ(25,NML=ProcessData)
    CLOSE(25)
    CALL set_diagnostics()
    ALLOCATE(totals(total_count))
    old_dx=0
  END SUBROUTINE init_diagnostics

  SUBROUTINE run_diagnostics(frame)
    INTEGER :: frame
    CHARACTER(LEN=40) :: FileName
    write(*,*) "processing frame", frame
    write(FileName,'(A6,I6.6,A5)')  'out/P_',100*frame,'.data'
    OPEN(UNIT=11,status="unknown",FILE=FileName)
    CALL calc_totals(frame)
    IF (ANY(gi_fixed%dX(1:nDim) /= old_dx(1:nDim))) CALL set_spectral_weights()
    CALL calc_spectrum(frame)
    CALL output_collapsed(frame)
    IF (lVCA) CALL VCA(frame)
  END SUBROUTINE run_diagnostics

  FUNCTION cyc(n,p)
    INTEGER :: n,p,cyc
    cyc=modulo(n-1,p)+1
  END FUNCTION cyc

  FUNCTION norm(x,p)
    REAL(KIND=qPREC) :: x(:), l2, norm
    INTEGER :: p
    l2=DOT_PRODUCT(x,x)
    IF (p==2) THEN 
       norm=l2
    ELSE IF (p==1) THEN
       norm=sqrt(l2)
    ELSE
       norm=sqrt(l2)**p
    END IF
  END FUNCTION norm

  SUBROUTINE calc_totals(frame)
    INTEGER ::  p,frame
    IF (total_count == 0) RETURN
    IF (lTotal_KE) THEN
       totals(iTotal_KEx) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * (gi_fixed%q(:,:,:,ivx)**2),3),2),1)
       totals(iTotal_KEy) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * (gi_fixed%q(:,:,:,ivy)**2),3),2),1)
       totals(iTotal_KEz) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * (gi_fixed%q(:,:,:,ivz)**2),3),2),1)
       totals(iTotal_KE) = SUM(totals( iTotal_KEx:iTotal_KEz))
    END IF
    IF (lTotal_v2) THEN
       totals(iTotal_v2x) = SUM(SUM(SUM(gi_fixed%q(:,:,:,ivx)**2,3),2),1)
       totals(iTotal_v2y) = SUM(SUM(SUM(gi_fixed%q(:,:,:,ivy)**2,3),2),1)
       totals(iTotal_v2z) = SUM(SUM(SUM(gi_fixed%q(:,:,:,ivz)**2,3),2),1)
       totals(iTotal_v2) = SUM(totals( iTotal_v2x:iTotal_v2z))
    END IF
    IF (lTotal_vrms) totals(iTotal_vrms) = SUM(SUM(SUM(sqrt(sum(gi_fixed%q(:,:,:,ivx:ivx+nDim-1)**2,4)),3),2),1)

    IF (lTotal_BE) THEN
!       gi_fixed%q(:,:,:,iBx)=gi_fixed%q(:,:,:,iBx)-3.7262095934d-1
       totals(iTotal_BEx) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,iBx)**2,3),2),1)
       totals(iTotal_BEy) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,iBy)**2,3),2),1)
       totals(iTotal_BEz) = .5*SUM(SUM(SUM(gi_fixed%q(:,:,:,iBz)**2,3),2),1)
       totals(iTotal_BE) = SUM(totals( iTotal_BEx:iTotal_BEz))
    END IF

    IF (lTotal_P) THEN
       totals( iTotal_Px) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * gi_fixed%q(:,:,:,ivx),3),2),1)
       totals( iTotal_Py) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * gi_fixed%q(:,:,:,ivy),3),2),1)
       totals( iTotal_Pz) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * gi_fixed%q(:,:,:,ivz),3),2),1)
    END IF
    IF (lTotal_Pmag) THEN
       totals( iTotal_Pmag) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * sqrt(sum(gi_fixed%q(:,:,:,ivx:ivx+nDim-1)**2,4)),3),2),1)
       totals( iTotal_Pmagx) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * abs(gi_fixed%q(:,:,:,ivx)),3),2),1)
       totals( iTotal_Pmagy) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * abs(gi_fixed%q(:,:,:,ivy)),3),2),1)
       totals( iTotal_Pmagz) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1) * abs(gi_fixed%q(:,:,:,ivz)),3),2),1)
    END IF
    IF (lTotal_Mass) THEN
       totals( iTotal_Mass) = SUM(SUM(SUM(gi_fixed%q(:,:,:,1),3),2),1)
    END IF
    IF (lTotal_iE .AND. .NOT. lIsothermal) THEN
       totals(iTotal_iE)=gamma7*SUM(SUM(SUM(gi_fixed%q(:,:,:,iE),3),2),1)
    END IF
    call output_totals(frame)
  END SUBROUTINE calc_totals

  SUBROUTINE calc_spectrum(frame)
    INTEGER ::  p2,p3,frame,i,j
    COMPLEX(8),DIMENSION(:,:,:), ALLOCATABLE :: invar,outvar
    COMPLEX(8),DIMENSION(:,:), ALLOCATABLE :: invar2D,outvar2D
    COMPLEX, DIMENSION(:,:,:,:), ALLOCATABLE :: vd
    IF (nDim == 3) THEN
       ALLOCATE(invar(gi_fixed%mx(1), gi_fixed%mx(2), gi_fixed%mx(3)), outvar(gi_fixed%mx(1), gi_fixed%mx(2), gi_fixed%mx(3)))
       CALL dfftw_plan_dft_3d(p3, gi_fixed%mx(1), gi_fixed%mx(2), gi_fixed%mx(3), invar, outvar, FFTW_FORWARD, FFTW_ESTIMATE) 
       IF (lSpectrum_split) ALLOCATE(vd(gi_fixed%mx(1), gi_fixed%mx(2), gi_fixed%mx(3),3))
    END IF
    IF (nDim == 2 .OR. lColumnDensity_Spectrum .OR. lVelocityCentroid_Spectrum) THEN
       ALLOCATE(invar2D(gi_fixed%mx(1), gi_fixed%mx(2)), outvar2D(gi_fixed%mx(1), gi_fixed%mx(2)))
       CALL dfftw_plan_dft_2d(p2, gi_fixed%mx(1), gi_fixed%mx(2), invar2D, outvar2D, FFTW_FORWARD, FFTW_ESTIMATE) 
    END IF

    IF (nDim == 3) THEN
       IF (lDensity_Spectrum .OR. lPhi) THEN
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)
          IF (lDensity_Spectrum) THEN 
             Spectrum=0
             CALL dfftw_execute(p3)
             CALL bin3D(REAL(ABS(outvar)**2))
             CALL output_spectra('Density')
          END IF
          IF (lPhi)  CALL calc_phi(outvar,invar,frame)
       END IF
       IF (lColumnDensity_Spectrum) THEN
          Spectrum=0
          invar2D=sum(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1),3)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(REAL(ABS(outvar2D)**2)))
          CALL output_spectra('ColumnDensity')
       END IF
       IF (lVelocityCentroid_Spectrum) THEN
          Spectrum=0
          CALL get_velocity_centroids(invar2D,1)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(abs(outvar2D)**2))
          CALL output_spectra('VelocityCentroid')
          CALL dump(invar2D,frame,"VC")
       END IF
       IF (lSEnergy_Spectrum) THEN
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivx)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('Velocity_x')
          IF (lSpectrum_split) vd(:,:,:,1)=outvar
          Spectrum_acc=Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivy)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('Velocity_y')
          IF (lSpectrum_split) vd(:,:,:,2)=outvar 
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivz)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))
          IF (lSpectrum_split) vd(:,:,:,3)=outvar
          CALL output_spectra('Velocity_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('Velocity')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin3D(abs(vd(:,:,:,1))**2)
             CALL bin3D(abs(vd(:,:,:,2))**2)
             CALL bin3D(abs(vd(:,:,:,3))**2)
             CALL output_spectra('Velocity_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('Velocity_Sol')
          END IF
       END IF
       IF (lEnergy_Spectrum) THEN
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivx)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1))
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('KE_x')
          IF (lSpectrum_split) vd(:,:,:,1)=outvar
          Spectrum_acc=Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivy)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1))
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('KE_y')
          IF (lSpectrum_split) vd(:,:,:,2)=outvar       
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivz)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1))
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))
          IF (lSpectrum_split) vd(:,:,:,3)=outvar
          CALL output_spectra('KE_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('KE')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin3D(abs(vd(:,:,:,1))**2)
             CALL bin3D(abs(vd(:,:,:,2))**2)
             CALL bin3D(abs(vd(:,:,:,3))**2)
             CALL output_spectra('KE_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('KE_Sol')
          END IF
       END IF

       IF (lMomentum_Spectrum) THEN
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivx)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('P_x')
          IF (lSpectrum_split) vd(:,:,:,1)=outvar
          Spectrum_acc=Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivy)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('P_y')
          IF (lSpectrum_split) vd(:,:,:,2)=outvar       
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivz)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))
          IF (lSpectrum_split) vd(:,:,:,3)=outvar
          CALL output_spectra('P_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('P')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin3D(abs(vd(:,:,:,1))**2)
             CALL bin3D(abs(vd(:,:,:,2))**2)
             CALL bin3D(abs(vd(:,:,:,3))**2)
             CALL output_spectra('P_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('P_Sol')
          END IF
       END IF
       IF (lFleck_Spectrum) THEN
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivx)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)**(1d0/3d0)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('Fleck_x')
          IF (lSpectrum_split) vd(:,:,:,1)=outvar
          Spectrum_acc=Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivy)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)**(1d0/3d0)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('Fleck_y')
          IF (lSpectrum_split) vd(:,:,:,2)=outvar       
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),ivz)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),1)**(1d0/3d0)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))
          IF (lSpectrum_split) vd(:,:,:,3)=outvar
          CALL output_spectra('Fleck_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('Fleck')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin3D(abs(vd(:,:,:,1))**2)
             CALL bin3D(abs(vd(:,:,:,2))**2)
             CALL bin3D(abs(vd(:,:,:,3))**2)
             CALL output_spectra('Fleck_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('Fleck_Sol')
          END IF
       END IF
       IF (lBField_Spectrum) THEN
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),iBx)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('B_x')
          IF (lSpectrum_split) vd(:,:,:,1)=outvar
          Spectrum_acc=Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),iBy)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))       
          CALL output_spectra('B_y')
          IF (lSpectrum_split) vd(:,:,:,2)=outvar       
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1:gi_fixed%mx(3),iBz)
          CALL dfftw_execute(p3)
          CALL bin3D(REAL(ABS(outvar)**2))
          IF (lSpectrum_split) vd(:,:,:,3)=outvar
          CALL output_spectra('B_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('B')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin3D(abs(vd(:,:,:,1))**2)
             CALL bin3D(abs(vd(:,:,:,2))**2)
             CALL bin3D(abs(vd(:,:,:,3))**2)
             CALL output_spectra('B_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('B_Sol')
          END IF
       END IF
    ELSE IF (nDim == 2) THEN
       IF (lDensity_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))
          CALL output_spectra('Density')
       END IF
       IF (lSEnergy_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivx)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('Velocity_x')
          IF (lSpectrum_split) vd(:,:,1,1)=outvar2D
          Spectrum_acc=Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivy)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('Velocity_y')
          IF (lSpectrum_split) vd(:,:,1,2)=outvar2D 
          Spectrum_acc=Spectrum_acc+Spectrum
          IF (ivz .ne. 0) THEN
             Spectrum=0
             invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivz)
             CALL dfftw_execute(p2)
             CALL bin2D(REAL(ABS(outvar2D)**2))
             IF (lSpectrum_split) vd(:,:,1,3)=outvar2D
             CALL output_spectra('Velocity_z')
             Spectrum_acc=Spectrum_acc+Spectrum
          END IF
          Spectrum=Spectrum_acc
          CALL output_spectra('Velocity')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin2D(abs(vd(:,:,1,1))**2)
             CALL bin2D(abs(vd(:,:,1,2))**2)
             !CALL bin2D(abs(vd(:,:,1,3))**2)
             CALL output_spectra('Velocity_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('Velocity_Sol')
          END IF
       END IF
       IF (lVortensity_Spectrum) THEN
          Spectrum=0
          invar2D=0
          DO i = 2, gi_fixed%mX(1) - 1
             DO j = 2, gi_fixed%mX(2) - 1
                invar2D(i,j)= &
                     ABS((gi_fixed%q(i-1,j,1,3)/gi_fixed%q(i-1,j,1,1) - gi_fixed%q(i+1,j,1,3)/gi_fixed%q(i+1,j,1,1))/gi_fixed%dX(1) - &
                     (gi_fixed%q(i,j-1,1,2)/gi_fixed%q(i,j-1,1,1) - gi_fixed%q(i,j+1,1,2)/gi_fixed%q(i,j+1,1,1))/gi_fixed%dX(2))
             END DO
          END DO
          CALL dfftw_execute(p2)
          CALL bin2D(real(abs(outvar2D)**2))
          CALL output_spectra('Vortensity')
       END IF

       IF (lEnergy_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivx)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1))
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('KE_x')
          IF (lSpectrum_split) vd(:,:,1,1)=outvar2D
          Spectrum_acc=Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivy)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1))
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('KE_y')
          IF (lSpectrum_split) vd(:,:,1,2)=outvar2D       
          Spectrum_acc=Spectrum_acc+Spectrum
          IF (ivz .ne. 0) THEN
             Spectrum=0
             invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivz)*sqrt(gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1))
             CALL dfftw_execute(p2)
             CALL bin2D(REAL(ABS(outvar2D)**2))
             IF (lSpectrum_split) vd(:,:,1,3)=outvar2D
             CALL output_spectra('KE_z')
             Spectrum_acc=Spectrum_acc+Spectrum
          END IF
          Spectrum=Spectrum_acc
          CALL output_spectra('KE')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin2D(abs(vd(:,:,1,1))**2)
             CALL bin2D(abs(vd(:,:,1,2))**2)
             !CALL bin2D(abs(vd(:,:,1,3))**2)
             CALL output_spectra('KE_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('KE_Sol')
          END IF
       END IF

       IF (lMomentum_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivx)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('P_x')
          IF (lSpectrum_split) vd(:,:,1,1)=outvar2D
          Spectrum_acc=Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivy)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('P_y')
          IF (lSpectrum_split) vd(:,:,1,2)=outvar2D       
          Spectrum_acc=Spectrum_acc+Spectrum
          IF (ivz .ne. 0) THEN
             Spectrum=0
             invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivz)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)
             CALL dfftw_execute(p2)
             CALL bin2D(REAL(ABS(outvar2D)**2))
             IF (lSpectrum_split) vd(:,:,1,3)=outvar2D
             CALL output_spectra('P_z')
             Spectrum_acc=Spectrum_acc+Spectrum
          END IF
          Spectrum=Spectrum_acc
          CALL output_spectra('P')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin2D(abs(vd(:,:,1,1))**2)
             CALL bin2D(abs(vd(:,:,1,2))**2)
             !CALL bin2D(abs(vd(:,:,1,3))**2)
             CALL output_spectra('P_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('P_Sol')
          END IF
       END IF
       IF (lFleck_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivx)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)**(1d0/3d0)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('Fleck_x')
          IF (lSpectrum_split) vd(:,:,1,1)=outvar2D
          Spectrum_acc=Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivy)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)**(1d0/3d0)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('Fleck_y')
          IF (lSpectrum_split) vd(:,:,1,2)=outvar2D       
          Spectrum_acc=Spectrum_acc+Spectrum
          IF (ivz .ne. 0) THEN
             Spectrum=0
             invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,ivz)*gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,1)**(1d0/3d0)
             CALL dfftw_execute(p2)
             CALL bin2D(REAL(ABS(outvar2D)**2))
             IF (lSpectrum_split) vd(:,:,1,3)=outvar2D
             CALL output_spectra('Fleck_z')
             Spectrum_acc=Spectrum_acc+Spectrum
          END IF
          Spectrum=Spectrum_acc
          CALL output_spectra('Fleck')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin2D(abs(vd(:,:,1,1))**2)
             CALL bin2D(abs(vd(:,:,1,2))**2)
             !CALL bin2D(abs(vd(:,:,1,3))**2)
             CALL output_spectra('Fleck_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('Fleck_Sol')
          END IF
       END IF
       IF (lBField_Spectrum) THEN
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,iBx)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('B_x')
          IF (lSpectrum_split) vd(:,:,1,1)=outvar2D
          Spectrum_acc=Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,iBy)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))       
          CALL output_spectra('B_y')
          IF (lSpectrum_split) vd(:,:,1,2)=outvar2D       
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=0
          invar2D=gi_fixed%q(1:gi_fixed%mx(1),1:gi_fixed%mx(2),1,iBz)
          CALL dfftw_execute(p2)
          CALL bin2D(REAL(ABS(outvar2D)**2))
          IF (lSpectrum_split) vd(:,:,1,3)=outvar2D
          CALL output_spectra('B_z')
          Spectrum_acc=Spectrum_acc+Spectrum
          Spectrum=Spectrum_acc
          CALL output_spectra('B')     
          Spectrum=0
          IF (lSpectrum_split) THEN
             CALL decompose(vd)
             CALL bin2D(abs(vd(:,:,1,1))**2)
             CALL bin2D(abs(vd(:,:,1,2))**2)
             !CALL bin2D(abs(vd(:,:,1,3))**2)
             CALL output_spectra('B_Div')
             Spectrum=Spectrum_acc-Spectrum
             CALL output_spectra('B_Sol')
          END IF
       END IF
    END IF

    IF (nDim == 3) THEN
       CALL dfftw_destroy_plan(p3)
       DEALLOCATE(invar,outvar)
       IF (lSpectrum_split) DEALLOCATE(vd)
    END IF
    IF (nDim == 2 .OR. lColumnDensity_Spectrum .OR. lVelocityCentroid_Spectrum) THEN
       CALL dfftw_destroy_plan(p2)
       DEALLOCATE(invar2D,outvar2D)
    END IF
  END SUBROUTINE calc_spectrum

  SUBROUTINE output_Spectra(name)
    CHARACTER(LEN=*) :: name
    INTEGER :: my_kmax,my_kmin,i
    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: mySpectrum,recv_buffer
    Spectrum=Spectrum/product(real(gi_fixed%mX(1:nDim),8))**2
    write(11,'(A2,A)') "# ",name
    DO i=max(0,lbound(spectrum,1)), ubound(spectrum,1)
       write(11,'(I12,E24.16)')  i,spectrum(i)
    END DO
  END SUBROUTINE output_Spectra
    

  SUBROUTINE set_spectral_weights()
    REAL(KIND=qPREC), DIMENSION(3) ::  kkx
    INTEGER, DIMENSION(3) :: kx, sample_res,hmx,mx
    REAL(KIND=qpREC) :: fact,dk(MaxDims)
    INTEGER :: kr_base,kr,i,j,k,l,m,n
    write(*,*) "Info%dx, old_dx=", gi_fixed%dx, old_dx
    IF (ALLOCATED(weights)) DEALLOCATE(weights,mapping)
    ALLOCATE(weights(gi_fixed%mx(1),gi_fixed%mx(2),gi_fixed%mx(3),-1:1))
    ALLOCATE(mapping(gi_fixed%mx(1),gi_fixed%mx(2),gi_fixed%mx(3)))
    weights=0
    mx=gi_fixed%mx(1:3)
    hmx=gi_fixed%mX(1:3)/2
    sample_res=1
    sample_res(1:nDim)=10
    fact=1d0/REAL(product(sample_res))
    DO i=1,gi_fixed%mx(1)
       DO j=1,gi_fixed%mx(2)
          DO k=1,gi_fixed%mx(3)
             kx(1)=mod(i-1+hmx(1),mx(1))-hmx(1)
             kx(2)=mod(j-1+hmx(2),mx(2))-hmx(2)
             kx(3)=mod(k-1+hmx(3),mx(3))-hmx(3)
             kr_base=nint(sqrt(REAL(sum(kx(1:nDim)**2))))
             mapping(i,j,k)=kr_base
             DO l=1,sample_res(1)
                DO m=1,sample_res(2)
                   DO n=1,sample_res(3)
                      kkx(:)=kx-.5+((/REAL(l),REAL(m),REAL(n)/)-.5)/sample_res(:)
                      kr=nint(sqrt(sum(kkx(1:nDim)**2)))
                      weights(i,j,k,kr-kr_base)=weights(i,j,k,kr-kr_base)+fact
                   END DO
                END DO
             END DO
          END DO
       END DO
    END DO
    k_min=minval(mapping)-1
    k_max=(maxval(mapping)+1)
    IF (ALLOCATED(Spectrum)) DEALLOCATE(Spectrum,Spectrum_acc)
    ALLOCATE(Spectrum(k_min:k_max))
    ALLOCATE(Spectrum_acc(k_min:k_max))
    Spectrum=0
    Spectrum_acc=0
    old_dx=gi_fixed%dX
    write(*,*) "setting old_dx=", old_dx
  END SUBROUTINE set_spectral_weights

  SUBROUTINE dump(outvar,i,name)
    USE GlobalDeclarations
    COMPLEX(8), DIMENSION(:,:) :: outvar
    CHARACTER(LEN=40) :: FileName
!    INTEGER, DIMENSIOn(:,:) :: index_range
    INTEGER :: filehandle,proc,frame,i
    REAL(KIND=xPrec) :: mysize(3),myxlower(3)
    CHARACTER(LEN=*) :: name
    filehandle=11
    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i, ".dat"
    OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
    write(filehandle) real(outvar)
    CLOSE(filehandle)

    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i,".bov"

    OPEN(UNIT=filehandle, FILE=Filename)
    WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
    write(filehandle,'(A17,A,I3.3,A4)') "DATA_FILE: spect_",name,i,".dat"
    WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", size(outvar,1), size(outvar,2)
    WRITE(filehandle,*)  "DATA_FORMAT: FLOAT"
    WRITE(filehandle,*)  "VARIABLE: Q"
    WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
    WRITE(filehandle,*)  "CENTERING: zonal"

    WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", gi_fixed%xLower(1:2)
    WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", gi_fixed%xUpper(1:2)-gi_fixed%xLower(1:2)

    WRITE(filehandle,*)  "BYTE_OFFSET: 4"
    WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",1
    close(filehandle)
  end SUBROUTINE dump

  SUBROUTINE dump3D(outvar,frame,name)
    USE GlobalDeclarations
    COMPLEX(8), DIMENSION(:,:,:) :: outvar
    CHARACTER(LEN=40) :: FileName
!    INTEGER, DIMENSIOn(:,:) :: index_range
    INTEGER :: filehandle,proc,frame,i,nvars
    REAL(KIND=xPrec) :: mysize(3),myxlower(3)
    CHARACTER(LEN=*) :: name
    REAL, DIMENSION(:,:,:,:),allocatable :: tempq
    nvars=size(gi_fixed%q,4)
    allocate(tempq(gi_fixed%mX(1),gi_fixed%mX(2),gi_fixed%mX(3),nvars+1))
    write(*,*) gi_fixed%mX, shape(gi_fixed%q), nvars
    DO i=1,size(gi_fixed%q,4)
       tempq(:,:,:,i)=gi_fixed%q(:,:,:,i)
    END DO
    tempq(:,:,:,nvars+1)=real(outvar)
    filehandle=11
    write(*,*) minval(gi_fixed%q(:,:,:,1))
    write(*,*) minval(tempq(:,:,:,1))
    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,frame, ".dat"
    OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
    write(filehandle) transpose(reshape(tempq(:,:,:,:), (/product(gi_fixed%mX(1:3)), nvars+1/)))
    CLOSE(filehandle)

    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,frame,".bov"

    OPEN(UNIT=filehandle, FILE=Filename)
    WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
    write(filehandle,'(A17,A,I3.3,A4)') "DATA_FILE: spect_",name,frame,".dat"
    WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", gi_fixed%mX(1:3)
    WRITE(filehandle,*)  "DATA_FORMAT: FLOAT"
    WRITE(filehandle,*)  "VARIABLE: Q"
    WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
    WRITE(filehandle,*)  "CENTERING: zonal"

    WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", gi_fixed%xLower(1:3)
    WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", gi_fixed%xUpper(1:3)-gi_fixed%xLower(1:3)
    WRITE(filehandle,*)  "BYTE_OFFSET: 4"
    WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",size(tempq,4)
    close(filehandle)
    deallocate(tempq)
  end SUBROUTINE dump3D


  SUBROUTINE dump_real3D(outvar,i,name)
    USE GlobalDeclarations
    REAL, DIMENSION(:,:) :: outvar
    CHARACTER(LEN=40) :: FileName
!    INTEGER, DIMENSIOn(:,:) :: index_range
    INTEGER :: filehandle,proc,frame,i
    REAL(KIND=xPrec) :: mysize(3),myxlower(3)
    CHARACTER(LEN=*) :: name
    filehandle=11
    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i, ".dat"
    OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
    write(filehandle) outvar
    CLOSE(filehandle)

    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i,".bov"

    OPEN(UNIT=filehandle, FILE=Filename)
    WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
    write(filehandle,'(A17,A,I3.3,A4)') "DATA_FILE: spect_",name,i,".dat"
    WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", gi_fixed%mX(1:3)
    WRITE(filehandle,*)  "DATA_FORMAT: REAL"
    WRITE(filehandle,*)  "VARIABLE: Q"
    WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
    WRITE(filehandle,*)  "CENTERING: zonal"

    WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", gi_fixed%xLower(1:3)
    WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", gi_fixed%xUpper(1:3)-gi_fixed%xLower(1:3)

    WRITE(filehandle,*)  "BYTE_OFFSET: 4"
    WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",1
    close(filehandle)
  end SUBROUTINE dump_real3D

  SUBROUTINE dump_real(outvar,i,name)
    USE GlobalDeclarations
    REAL, DIMENSION(:,:) :: outvar
    CHARACTER(LEN=40) :: FileName
!    INTEGER, DIMENSIOn(:,:) :: index_range
    INTEGER :: filehandle,proc,frame,i
    REAL(KIND=xPrec) :: mysize(3),myxlower(3)
    CHARACTER(LEN=*) :: name
    filehandle=11
    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i, ".dat"
    OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
    write(filehandle) outvar
    CLOSE(filehandle)

    write(FileName,'(A10,A,I3.3,A4)') "out/spect_",name,i,".bov"

    OPEN(UNIT=filehandle, FILE=Filename)
    WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
    write(filehandle,'(A17,A,I3.3,A4)') "DATA_FILE: spect_",name,i,".dat"
    WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", size(outvar,1), size(outvar,2)
    WRITE(filehandle,*)  "DATA_FORMAT: REAL"
    WRITE(filehandle,*)  "VARIABLE: Q"
    WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
    WRITE(filehandle,*)  "CENTERING: zonal"

    WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", gi_fixed%xLower(1:2)
    WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", gi_fixed%xUpper(1:2)-gi_fixed%xLower(1:2)

    WRITE(filehandle,*)  "BYTE_OFFSET: 4"
    WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",1
    close(filehandle)
  end SUBROUTINE dump_real


  SUBROUTINE decompose(vd)
    INTEGER :: i,j,k,kx(3),hmx(maxdims),mx(maxdims)
    complex, DIMENSION(:,:,:,:) :: vd
    REAL :: dk(3)
    hmx=gi_fixed%mX/2
    mx=gi_fixed%mX
    DO i=1,gi_fixed%mX(1)
       DO j=1,gi_fixed%mX(2)
          DO k=1,gi_fixed%mX(3)
             kx(1)=mod(i-1+hmx(1),mx(1))-hmx(1)
             kx(2)=mod(j-1+hmx(2),mx(2))-hmx(2)
             kx(3)=mod(k-1+hmx(3),mx(3))-hmx(3)
             if (i+j+k > 3) then
                vd(i,j,k,:)=(DOT_PRODUCT(vd(i,j,k,:), kx(:)))/DOT_PRODUCT(kx,kx)*kx
             END if
          end DO
       end DO
    end DO
  END SUBROUTINE DECOMPOSE

  SUBROUTINE decompose2D(vd)
    INTEGER :: i,j,k,kx(2),hmx(maxdims),mx(maxdims)
    complex, DIMENSION(:,:,:,:) :: vd
    mx=gi_fixed%mX
    hmx=gi_fixed%mX/2
    DO i=1,gi_fixed%mX(1)
       DO j=1,gi_fixed%mX(2)
          kx(1)=mod(i-1+hmx(1),mx(1))-hmx(1)
          kx(2)=mod(j-1+hmx(2),mx(2))-hmx(2)
          if (i+j > 2) then
             vd(i,j,1,:)=(DOT_PRODUCT(vd(i,j,1,:), kx(:)))/DOT_PRODUCT(kx,kx)*kx
          END if
       end DO
    end DO
  END SUBROUTINE DECOMPOSE2D

  SUBROUTINE bin3D(data)
    real, DIMENSION(:,:,:) :: data
    INTEGER :: i,j,k,l
    DO i=1,size(data,1)
       DO j=1,size(data,2)
          DO k=1,size(data,3)
             DO l=-1,1
                Spectrum(mapping(i,j,k)+l)=Spectrum(mapping(i,j,k)+l)+data(i,j,k)*weights(i,j,k,l)
             END DO
          END DO
       END DO
    END DO
  END SUBROUTINE bin3D

  SUBROUTINE bin2D(data)
    real, DIMENSION(:,:) :: data
    INTEGER :: i,j,k,l
    DO i=1,size(data,1)
       DO j=1,size(data,2)
          DO l=-1,1
             Spectrum(mapping(i,j,1)+l)=Spectrum(mapping(i,j,1)+l)+data(i,j)*weights(i,j,1,l)
          END DO
       END DO
    END DO
  END SUBROUTINE bin2D

  SUBROUTINE set_diagnostics()
    INTEGER :: analysis=0, kx, ky, kz, i
    CHARACTER(len=60) :: FileName
    total_count=0
    analysis=0
    iTotal_KEx=-1
    iTotal_KEy=-1
    iTotal_KEz=-1
    iTotal_KE=-1
    iTotal_v2x=-1
    iTotal_v2y=-1
    iTotal_v2z=-1
    iTotal_v2=-1
    iTotal_vrms=-1
    iTotal_BEx=-1
    iTotal_BEy=-1
    iTotal_BEz=-1
    iTotal_BE=-1
    iTotal_Px=-1
    iTotal_Py=-1
    iTotal_Pz=-1
    iTotal_Px=-1
    iTotal_Py=-1
    iTotal_Pz=-1
    iTotal_Pmag=-1
    iTotal_Pmagx=-1
    iTotal_Pmagy=-1
    iTotal_Pmagz=-1
    iTotal_iE=-1
    total_count=total_count+1
    totals_name(total_count)%s="Time"
    IF (lTotal_KE) THEN     
       iTotal_KEx = total_count+1
       totals_name(total_count+1)%s="KE_x"
       iTotal_KEy = total_count+2
       totals_name(total_count+2)%s="KE_y"
       iTotal_KEz = total_count+3
       totals_name(total_count+3)%s="KE_z"
       iTotal_KE = total_count+4
       totals_name(total_count+4)%s="KE"
       total_count = total_count + 4
    END IF
    IF (lTotal_v2) THEN     
       iTotal_v2x = total_count+1
       totals_name(total_count+1)%s="v2_x"
       iTotal_v2y = total_count+2
       totals_name(total_count+2)%s="v2_y"
       iTotal_v2z = total_count+3
       totals_name(total_count+3)%s="v2_z"
       iTotal_v2 = total_count+4
       totals_name(total_count+4)%s="v2"
       total_count = total_count + 4
    END IF
    IF (lTotal_vrms) THEN 
       iTotal_vrms = total_count+1
       totals_name(total_count+1)%s="v_rms"
       total_count = total_count + 1      
    END IF
    IF (lTotal_BE) THEN     
       iTotal_BEx = total_count+1
       totals_name(total_count+1)%s="BE_x"
       iTotal_BEy = total_count+2
       totals_name(total_count+2)%s="BE_y"
       iTotal_BEz = total_count+3
       totals_name(total_count+3)%s="BE_z"
       iTotal_BE = total_count+4
       totals_name(total_count+4)%s="BE"
       total_count = total_count + 4
    END IF
    IF (lTotal_P) THEN     
       iTotal_Px = total_count+1
       totals_name(total_count+1)%s="P_x"
       iTotal_Py = total_count+2
       totals_name(total_count+2)%s="P_y"
       iTotal_Pz = total_count+3
       totals_name(total_count+3)%s="P_z"
       total_count = total_count + 3
    END IF
    IF (lTotal_Pmag) THEN     
       iTotal_Pmag = total_count+1
       totals_name(total_count+1)%s="P_mag"
       iTotal_Pmagx = total_count+2
       totals_name(total_count+2)%s="Pmag_x"
       iTotal_Pmagy = total_count+3
       totals_name(total_count+3)%s="Pmag_y"
       iTotal_Pmagz = total_count+4
       totals_name(total_count+4)%s="Pmag_z"
       total_count = total_count + 4
    END IF
    IF (lTotal_Mass) THEN     
       iTotal_Mass = total_count+1
       totals_name(total_count+1)%s="Mass"
       total_count = total_count + 1
    END IF
    IF (lTotal_iE) THEN     
       iTotal_iE = total_count+1
       totals_name(total_count+1)%s="iE"
       total_count = total_count + 1
    END IF
  END SUBROUTINE set_diagnostics

  SUBROUTINE output_totals(frame)
    INTEGER :: frame,i
    CHARACTER(len=60) :: FileName
    totals=totals/product(REAL(gi_fixed%mX(1:nDim),8))
    write(11,'(A2,A12,E24.16)') "# ",totals_name(1)%s, gi_Fixed%time
    DO i=2,total_count
       write(11,'(A2,A12)') "# ",totals_name(i)%s
       write(11,'(2E24.16)') gi_fixed%time, totals(i)
    END DO
  END SUBROUTINE output_totals

  SUBROUTINE output_collapsed(frame)
    INTEGER :: frame,i,mymx(3),myxlower(1:3),myxupper(1:3)
    INTEGER, PARAMETER :: iB2=21, iLineWidths=22, iDataCube=23,iDataCubeWeighted=24
    REAL, POINTER, DIMENSION(:,:,:,:) :: q_collapse
    CHARACTER(LEN=26) :: Filename
    REAL :: mean, sigma
    DO i=1,size(collapse_vars)
       mymx=gi_fixed%mX(1:3)
       myxlower(1:3)=gi_fixed%Xlower(1:3)
       myxupper(1:3)=gi_fixed%XUpper(1:3)
       IF (collapse_vars(i) <= 0 .OR. collapse_dirs(i) <= 0) RETURN
       IF (collapse_dirs(i) > nDim) THEN
          WRITE(*,*) "error during collapse", collapse_vars(i), NrVars, collapse_dirs(i), nDim
          STOP
       END IF
       mymx(collapse_dirs(i))=1
       myxupper(collapse_dirs(i))=myxlower(collapse_dirs(i))+gi_fixed%dx(collapse_dirs(i))
       ALLOCATE(q_collapse(1:mymx(1),1:mymx(2),1:mymx(3),1))
       IF (collapse_vars(i) <= NrVars) THEN
          q_collapse=SPREAD(SUM(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),collapse_vars(i):collapse_vars(i)),collapse_dirs(i)),collapse_dirs(i),1)*gi_fixed%dX(collapse_dirs(i))
       ELSEIF (collapse_vars(i) == iB2) THEN
          q_collapse=SPREAD(SUM(spread(sum(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),iBx:iBz)**2,4),4,1),collapse_dirs(i)),collapse_dirs(i),1)*gi_fixed%dX(collapse_dirs(i))
       ELSEIF (collapse_vars(i) == iLineWidths) THEN
          q_collapse=spread(maxval(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i):1+collapse_dirs(i)),collapse_dirs(i))- minval(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i):1+collapse_dirs(i)),collapse_dirs(i)), collapse_dirs(i),1)
       ELSEIF (collapse_vars(i) == iDataCube) THEN
          DEALLOCATE(q_collapse)
          mymx=gi_fixed%mX(1:3)
          mymx(collapse_dirs(i))=ceiling(.2*mymx(collapse_dirs(i)))
          ALLOCATE(q_collapse(1:mymx(1),1:mymx(2),1:mymx(3),1))
          call get_statistics(mean,sigma,gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i)))
          CALL bin_vel3D(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i)),q_collapse(:,:,:,1),collapse_dirs(i),data_cube_vbounds(1), data_cube_vbounds(2))  !mean+-4d0*sigma
        ELSEIF (collapse_vars(i) == iDataCubeWeighted) THEN
          DEALLOCATE(q_collapse)
          mymx=gi_fixed%mX(1:3)
          mymx(collapse_dirs(i))=ceiling(.2*mymx(collapse_dirs(i)))
          ALLOCATE(q_collapse(1:mymx(1),1:mymx(2),1:mymx(3),1))
          call get_statistics(mean,sigma,gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i)))
          CALL bin_vel3D(gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1+collapse_dirs(i)),q_collapse(:,:,:,1),collapse_dirs(i),data_cube_vbounds(1), data_cube_vbounds(2),gi_fixed%q(1:gi_fixed%mX(1),1:gi_fixed%mX(2),1:gi_fixed%mX(3),1))
       ELSE
          WRITE(*,*) "error during collapse", collapse_vars(i), NrVars, collapse_dirs(i), nDim
       END IF

       write(FileName,'(A13,I2.2,A1,I1.1,A1,I4.4,A4)') "out/collapse_",collapse_vars(i),"_",collapse_dirs(i),"_",frame,".dat"
       OPEN(UNIT=11, FILE=FileName, status="replace", FORM="unformatted")
       write(11) transpose(reshape(q_collapse(1:mymX(1),1:mymX(2),1:mymX(3),1), (/product(mymX(1:3)),1/)))
       CLOSE(11)

       write(FileName,'(A13,I2.2,A1,I1.1,A1,I4.4,A4)') "out/collapse_",collapse_vars(i),"_",collapse_dirs(i),"_",frame,".bov"
       OPEN(UNIT=11, FILE=FileName, status="replace")
       WRITE(11,*)  "TIME: ", gi_fixed%time
       write(11,'(A20,I2.2,A1,I1.1,A1,I4.4,A4)') "DATA_FILE: collapse_" , collapse_vars(i),"_",collapse_dirs(i),"_",frame, ".dat"
       WRITE(11,'(A,3I6)')  "DATA_SIZE: ", mymX(1), mymX(2), mymX(3)
       WRITE(11,*)  "DATA_FORMAT: FLOAT"
       WRITE(11,'(A15,I2.2,A5,I1.1)')  "VARIABLE: field" ,collapse_vars(i) , "along" , collapse_dirs(i)
       WRITE(11,*)  "DATA_ENDIAN: LITTLE"
       WRITE(11,*)  "CENTERING: zonal"
       WRITE(11,'(A,3F8.3)')  "BRICK_ORIGIN: ", gi_fixed%Xlower(1), gi_fixed%Xlower(2), gi_fixed%Xlower(3)
       WRITE(11,'(A,3F8.3)')  "BRICK_SIZE: ", gi_fixed%Xupper(1)-gi_fixed%Xlower(1), gi_fixed%Xupper(2)-gi_fixed%Xlower(2), &
            gi_fixed%Xupper(3)-gi_fixed%Xlower(3)

       WRITE(11,*)  "BYTE_OFFSET: 4"
       WRITE(11,'(A,I4)')  "DATA_COMPONENTS: ",1
       CLOSE(11)
       DEALLOCATE(q_collapse)
    END DO
  END SUBROUTINE output_collapsed

  SUBROUTINE bin_vel3D(data, bins, dir, min, max,weight)
    REAL, DIMENSION(:,:,:) :: data, bins
    REAL, OPTIONAL, DIMENSION(:,:,:) :: weight

    REAL :: min, max, df,bin_low,bin_high
    INTEGER :: dir,i
    df=(max-min)/size(bins,dir)

    IF (PRESENT(weight)) THEN
       SELECT CASE(dir)
       CASE(1)
          DO i=1,size(bins,1)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(i,:,:)=sum(weight(:,:,:),1, (data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high))
          END DO
       CASE(2)
          DO i=1,size(bins,2)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(:,i,:)=sum(weight(:,:,:),2, (data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high))
          END DO
       CASE(3)
          DO i=1,size(bins,3)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(:,:,i)=sum(weight(:,:,:),3, (data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high))
          END DO
       END SELECT
       bins=bins/sum(weight)
    ELSE
       SELECT CASE(dir)
       CASE(1)
          DO i=1,size(bins,1)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(i,:,:)=count(data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high, 1)
          END DO
       CASE(2)
          DO i=1,size(bins,2)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(:,i,:)=count(data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high, 2)
          END DO
       CASE(3)
          DO i=1,size(bins,3)
             bin_high=min+i*df
             bin_low=bin_high-df
             bins(:,:,i)=count(data(:,:,:) > bin_low .AND. data(:,:,:) <= bin_high, 3)
          END DO
       END SELECT
       bins=bins/size(bins,dir)

    ENd IF
  

  END SUBROUTINE bin_vel3D

  SUBROUTINE get_statistics(mean, sigma, data)
    REAL, DIMENSION(:,:,:) :: data
    REAL :: mean, sigma
    mean = SUM(SUM(SUM(data,3),2),1)/product(shape(data))
    sigma = sqrt(SUM(SUM(SUM((data-mean)**2,3),2),1))
  END SUBROUTINE get_statistics

  subroutine calc_phi(in,phi,frame)
    complex(8), dimension(:,:,:) :: in ! maps to transform of density
    complex(8), dimension(:,:,:) :: phi !
    INTEGER :: i,j,k,frame,ikx,iky,ikz
    INTEGER(8) :: p
    REAL, DIMENSION(:), ALLOCATABLE :: kx,ky,kz
    REAL :: Pi, C1,dk(3)
    ScaleGrav=1d0
    Pi=ACOS(-1d0)
    C1=4d0*Pi*ScaleGrav/(PRODUCT(gi_fixed%mX(:))*product(gi_fixed%dX(1:nDim))) !extra scaling factors (scale_grav is just G_ scaled to computational units)
    dk(:)=2d0*Pi/(gi_fixed%Xupper(1:3)-gi_fixed%Xlower(1:3))/(sqrt(C1))
    ALLOCATE (kx(gi_fixed%mX(1)),ky(gi_fixed%mX(2)),kz(gi_fixed%mX(3)))
    DO ikx = 0, gi_fixed%mX(1)-1
       kx(ikx+1)=dk(1)*REAL((mod((ikx+gi_fixed%mX(1)/2),gi_fixed%mX(1))-gi_fixed%mX(1)/2),8)
    END DO
    DO iky = 0, gi_fixed%mX(2)-1
       ky(iky+1)=dk(2)*REAL((mod((iky+gi_fixed%mX(2)/2),gi_fixed%mX(2))-gi_fixed%mX(2)/2),8)
    END DO
    DO ikz = 0, gi_fixed%mX(3)-1
       kz(ikz+1)=dk(3)*REAL((mod((ikz+gi_fixed%mX(3)/2),gi_fixed%mX(3))-gi_fixed%mX(3)/2),8)
    END DO
    write(*,*) maxval(abs(in))
    DO i=1,gi_fixed%mX(1)
       DO j=1,gi_fixed%mX(2)
          DO k=1,gi_fixed%mX(3)
!             if (abs(kx(i))+abs(ky(j))+abs(kz(k)) == 0) THEN
!                write(*,*) i,j,k,kx(i),ky(i),kz(i)
!                STOP
!             END if
             in(i,j,k)=-1d0*in(i,j,k)/REAL(kx(i)**2+ky(j)**2+kz(k)**2)
          END DO
       END DO
    END DO
    in(1,1,1)=0d0 !periodic bc's
    CALL dfftw_plan_dft_3d(p, gi_fixed%mx(1), gi_fixed%mx(2), gi_fixed%mx(3), in, phi, FFTW_BACKWARD, FFTW_ESTIMATE) 
    CALL dfftw_execute(p)
    call dump3D(phi,frame,"QPhi")
  end subroutine calc_phi

  SUBROUTINE get_velocity_centroids(centroids, dir)
    complex(8), DIMENSION(:,:) :: centroids
    INTEGER :: dir,i,j,k
    SELECT CASE(dir)
    CASE(1)
       FORALL(j=1:gi_fixed%mX(2),k=1:gi_fixed%mX(3))
          centroids(j,k)=get_centroid(gi_fixed%q(:,j,k,1),gi_fixed%q(:,j,k,ivx))
       END FORALL
    CASE(2)
       FORALL(i=1:gi_fixed%mX(1),k=1:gi_fixed%mX(3))
          centroids(k,i)=get_centroid(gi_fixed%q(i,:,k,1),gi_fixed%q(i,:,k,ivy))
       END FORALL
    CASE(3)
       FORALL(i=1:gi_fixed%mX(1),j=1:gi_fixed%mX(2))
          centroids(i,j)=get_centroid(gi_fixed%q(i,j,:,1),gi_fixed%q(i,j,:,ivz))
       END FORALL
    END SELECT
  END SUBROUTINE get_velocity_centroids

  PURE FUNCTION get_centroid(weights, values)
    REAL :: get_centroid,mysum, halfweight
    REAL, DIMENSION(:), INTENT(IN) :: weights, values
    REAL, DIMENSION(:,:), POINTER :: sortedvalues
    INTEGER :: i
    ALLOCATE (sortedvalues(2,size(values)))
    sortedvalues(1,:)=values
    sortedvalues(2,:)=weights
    CALL shellsort(sortedvalues)
    halfweight=.5d0*sum(sortedvalues(2,:))
    mysum=0
    DO i=1,size(values)
       mysum=mysum+sortedvalues(2,i)
       if (mysum >= halfweight) THEN
          get_centroid=sortedvalues(1,i)
          EXIT
       END if
    END DO
  END FUNCTION get_centroid

  PURE SUBROUTINE shellsort(a)
    INTEGER :: i, j, n,inc
    REAL :: temp(2)
    REAL, POINTER :: a(:,:)
    n=size(a,2)
    inc=nint(real(n)/2)
    do while (inc > 0)
       do i=inc+1, n
          temp=a(:,i)
          j=i
          do while (j > inc)
             IF (a(1,j-inc) <= temp(1)) EXIT
             a(:,j)=a(:,j-inc)
             j=j-inc
          END do
          a(:,j)=temp
       END do
       inc=nint(real(inc)/2.2)
    end do
  END SUBROUTINE shellsort


  SUBROUTINE VCA(frame)
    REAL,ALLOCATABLE :: mean(:,:)
    REAL, DIMENSION(:,:,:), POINTER :: data,weight
    REAL, DIMENSION(:,:,:,:), POINTER :: q
    INTEGER :: frame
    data=>gi_fixed%q(:,:,:,ivx)
    weight=>gi_fixed%q(:,:,:,1)
    q=>gi_fixed%q(:,:,:,:)
    ALLOCATE(mean(gi_fixed%mX(2),gi_fixed%mX(3)))
    CALL dump_real(minval(data,1),frame,'v_min')
    CALL dump_real(maxval(data,1),frame,'v_max')
    mean=sum(data,1)/gi_fixed%mX(1)
    CALL dump_real(mean,frame,'v_bar')
    CALL dump_real(sqrt(sum((data-spread(mean,1,gi_fixed%mX(1)))**2,1)/gi_fixed%mX(1)),frame, 'v_sig')
    mean=sum(data*weight,1)/sum(weight,1)
    CALL dump_real(mean,frame,'e_bar')
    CALL dump_real(sqrt(sum((data-spread(mean,1,gi_fixed%mX(1)))**2*weight,1)/sum(weight,1)), frame, 'e_sig')
    NULLIFY(data,weight)
    DEALLOCATE(mean)
    ALLOCATE(data(gi_fixed%mX(1)/2, gi_fixed%mX(2)/2, gi_fixed%mX(3)/2), weight(gi_fixed%mX(1)/2, gi_fixed%mX(2)/2,gi_fixed%mX(3)/2))
    data=.125d0*(q(1:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,ivx)+&
                 q(2:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,ivx)+&
                 q(1:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,ivx)+&
                 q(2:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,ivx)+&
                 q(1:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,ivx)+&
                 q(2:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,ivx)+&
                 q(1:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,ivx)+&
                 q(2:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,ivx))
    weight=.125d0*(q(1:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,1)+&
                   q(2:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,1)+&
                   q(1:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,1)+&
                   q(2:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,1:gi_fixed%mX(3):2,1)+&
                   q(1:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,1)+&
                   q(2:gi_fixed%mX(1):2,1:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,1)+&
                   q(1:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,1)+&
                   q(2:gi_fixed%mX(1):2,2:gi_fixed%mX(2):2,2:gi_fixed%mX(3):2,1))
    ALLOCATE(mean(gi_fixed%mX(2)/2,gi_fixed%mX(3)/2))
    CALL dump_real(minval(data,1),frame,'v_min_r')
    CALL dump_real(maxval(data,1),frame,'v_max_r')
    mean=sum(data,1)/(gi_fixed%mX(1)/2)
    CALL dump_real(mean,frame,'v_bar_r')
    CALL dump_real(sqrt(sum((data-spread(mean,1,gi_fixed%mX(1)/2))**2,1)/(gi_fixed%mX(1)/2)),frame, 'v_sig_r')
    mean=sum(data*weight,1)/sum(weight,1)
    CALL dump_real(mean,frame,'e_bar_r')
    CALL dump_real(sqrt(sum((data-spread(mean,1,gi_fixed%mX(1)/2))**2*weight,1)/sum(weight,1)), frame, 'e_sig_r')         
    DEALLOCATE(data,weight)
    NULLIFY(data,weight)
    DEALLOCATE(mean)
    

  END SUBROUTINE VCA

END MODULE analyze
