!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    angledemission.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
PROGRAM angledemission

  IMPLICIT NONE

  INTEGER :: framestart,frameend,framestride,scales,logscale,projection
  INTEGER :: step
  REAL :: scalemin,scalemax,lScale
  INTEGER :: anglestart,angleend,anglestride
  REAL :: minoverall=0,maxoverall=0
  LOGICAL :: lWriteFits
  LOGICAL,PARAMETER :: T=.TRUE.,F=.FALSE.

  ! mx, my, mz read from BOV file
  INTEGER :: mx,my,mz
  INTEGER :: iErr,i
  REAL :: units(3)

  NAMELIST/emissionlist/framestart,frameend,framestride,scales,LogScale,scalemin,scalemax,projection,anglestart,angleend,anglestride,lScale,lWriteFits,units

  REAL,DIMENSION(:,:,:),ALLOCATABLE :: Halpha,SII
  REAL,DIMENSION(:),ALLOCATABLE :: inputbuffer

  INTEGER :: angle

  CHARACTER(LEN=120) :: plotlabel

  ! Read in namelist
  OPEN(10,FILE='emission.dat',IOSTAT=iErr)
  IF(iErr==0) THEN
     READ(10,NML=emissionlist)
  ELSE
     framestart=0
     frameend=100
     framestride=1
     scales=0
     LogScale=1
     scalemin=0.06
     scalemax=602
     projection=0
     anglestart=0
     angleend=0
     anglestride=0
     lScale=1.5e15
     units= (/ 48, 12, 12 /)
     lWriteFits=F
  END IF
  CLOSE(10)

!!$  PRINT*,framestart,frameend,framestride
!!$  PRINT*,scales,scalemin,scalemax
!!$  PRINT*,logScale
!!$  PRINT*,projection
!!$  PRINT*,angle

  ! Get mx,my,mz
  CALL ReadSize(framestart)
  ALLOCATE(Halpha(mx,my,mz))
  ALLOCATE(Sii(mx,my,mz))
  ALLOCATE(inputbuffer(mx*my*mz))

  ! Loop over each frame and plot data at specified angle
  scalemin=0d0;scalemax=0d0
  IF(scales==1) THEN
     IF(frameend-framestart .gt. 10 .AND. framestride==1) THEN
        step = CEILING((REAL(frameend)-REAL(framestart))/10.)
     ELSE
        step = framestride
     END IF
     DO i=framestart,frameend,step
        CALL ReadData(i)
!!$        IF(scalemin==0) THEN
!!$           scalemin=MIN(MINVAL(Halpha),MINVAL(Sii))
!!$        ELSE
!!$           scalemin=MIN(scalemin,MINVAL(Halpha),MINVAL(Sii))
!!$        END IF
!!$        scalemax=MAX(scalemax,MAXVAL(Halpha),MAXVAL(Sii))!,Halpha<1d0/0d0),MAXVAL(Sii,Sii<1d0/0d0))
!!$        IF(scalemin==0) THEN
!!$           scalemin=MINVAL(Sii)!,Sii>1e-2)
!!$        ELSE
!!$           scalemin=MIN(scalemin,MINVAL(Sii))!,Sii>1e-6))
!!$        END IF
!!$        scalemax=MAX(scalemax,MAXVAL(Halpha))
        scalemax=MAX(scalemax,MAXVAL(Halpha),MAXVAL(Sii))
     END DO
        scalemin=1d-5*scalemax
        PRINT*,'Scalemin, scalemax:',scalemin,scalemax
  END IF

  DO i=framestart,frameend,framestride
     CALL ReadData(i)
     IF(anglestride>0) THEN
        DO angle=anglestart,angleend,anglestride
           PRINT*,'angle is',angle
           CALL PlotData(scalemin,scalemax,angle,i)
        END DO
     ELSE
        CALL PlotData(scalemin,scalemax,anglestart,i)
     END IF
  END DO

!!$  DO i=framestart,frameend,framestride
!!$     CALL ReadData(i)
!!$     IF(scales==1) THEN
!!$      IF(scalemin==0) THEN 
!!$         scalemin=MIN(MINVAL(Halpha,Halpha>0),MINVAL(Sii,Sii>0))
!!$      ELSE
!!$         scalemin=MIN(scalemin,MINVAL(Halpha),MINVAL(Sii))
!!$      END IF
!!$      scalemax=MAX(scalemax,MAXVAL(Halpha),MAXVAL(Sii))
!!$      IF(scalemin==0) THEN
!!$         scalemin=MINVAL(Sii,Sii>1e-2)
!!$      ELSE
!!$         scalemin=MIN(scalemin,MINVAL(Sii,Sii>1e-6))
!!$      END IF
!!$      scalemax=MAX(scalemax,MAXVAL(Halpha))
!!$      IF(i==frameend) &
!!$         PRINT*,'Scalemin, scalemax:',scalemin,scalemax
!!$     ELSE
!!$        CALL ReadData(i)
!!$        CALL PlotData(scalemin,scalemax,angle,i)
!!$     END IF
!!$  END DO


CONTAINS


  !
  ! Get array size from BOV file
  !
  SUBROUTINE ReadSize(frame)
    INTEGER :: frame
    CHARACTER(LEN=20) :: filename,datastring
    CHARACTER(LEN=60) :: dummy1,dummy2
    REAL :: junk

    WRITE(filename,'(A8,I3.3)') 'Hal_'//'3-2_',frame
    !PRINT*,TRIM(filename)//'.bov'
    OPEN(11,FILE=TRIM(filename)//'.bov')

    ! skip first two lines, read in third
    READ(11,'(A,E)')dummy1,junk
    !PRINT*,dummy1,junk
    READ(11,'(A11,A)')dummy1,dummy2
    !PRINT*,dummy1,dummy2
    READ(11,'(A11,3I12)')datastring, mx, my, mz
    !PRINT*,datastring,mx,my,mz
    CLOSE(11)
    !PRINT*,'read Halpha frame',frame
  END SUBROUTINE ReadSize

  !
  ! Read in Halpha & SII data for a given frame
  !
  SUBROUTINE ReadData(frame)
    INTEGER :: frame
    CHARACTER(LEN=20) :: filename,datastring
    CHARACTER(LEN=60) :: dummy1,dummy2
    REAL :: junk
    INTEGER :: i,iErr

    WRITE(*,'(A,I6.2)'),'Reading frame ',frame
    ! H alpha emission
    WRITE(filename,'(A8,I3.3)') 'Hal_'//'3-2_',frame
    OPEN(12,FILE=TRIM(filename)//'.dat',FORM='unformatted',IOSTAT=iErr)
    READ(12) inputbuffer
    Halpha = RESHAPE(inputbuffer, (/ mx, my, mz /))
    CLOSE(12)
    WHERE(Halpha<0d0 .OR. Halpha>HUGE(Halpha)) Halpha=0d0
    PRINT*,'Hal ',MINVAL(Halpha),MAXVAL(Halpha)
    !PRINT*,'Read Halpha'

    ! H alpha emission
    WRITE(filename,'(A8,I3.3)') 'SII_'//'2-1_',frame
    OPEN(13,FILE=TRIM(filename)//'.dat',FORM='unformatted')

    READ(13) inputbuffer
    Sii = RESHAPE(inputbuffer, (/ mx, my, mz /))
    CLOSE(13)
    WHERE(Sii<0d0 .OR. Sii>HUGE(Sii)) Sii=0d0
    PRINT*,'Sii ',MINVAL(Sii),MAXVAL(Sii)
!    WHERE(Sii==1d0/0d0) Sii=0d0
    !PRINT*,'Read Sii'
  END SUBROUTINE ReadData


  !
  ! Plot Halpha and SII with the same extents at angle ANG
  ! ANG is assumed to be in the XZ plane, with the jet propagating
  ! along X and the image plane in XY.
  !
  SUBROUTINE PlotData(minsc,maxsc,ang,frame)
    REAL,INTENT(IN) :: minsc,maxsc
    INTEGER :: ang,frame
    !
    INTEGER :: i,j,k,imgmxL,imgmxH,imgmx,imgmy,posx,posy
    REAL,DIMENSION(:,:),ALLOCATABLE :: imageHalpha,imageSii
    REAL,DIMENSION(3) :: camera_pos,camera_focus,camera_up,camera_norm,camera_right, &
         vec
    REAL :: vecx,vecy,vecz,fov,screenz
    !
    INTEGER :: iim,jim
    CHARACTER(LEN=20) :: filename
    CHARACTER(LEN=5)  :: framechar
    CHARACTER(LEN=3)  :: angchar
    REAL,DIMENSION(6) :: TR
    !
    REAL :: minscale,maxscale,angreal

    WRITE(framechar,'(I5.5)')frame
    minscale=minsc;maxscale=maxsc
    WRITE(angchar,'(I3.3)')ang

    angreal=REAL(ang)

    IF(ang==0) THEN
       imgmxL=1
       imgmxH=mx
       imgmx=imgmxH-imgmxL+1
       imgmy=my
       ALLOCATE(imageHalpha(imgmx,imgmy),imageSii(imgmx,imgmy))
       imageHalpha = SUM(Halpha,DIM=3)*units(3)*lScale
       imageSii = SUM(Sii,DIM=3)*units(3)*lScale
       minscale = minscale*lScale*units(3)
       maxscale = maxscale*lScale*units(3)
    ELSE
       IF(projection==0) THEN
          !PRINT*,'parallel projection'
!          imgmxL = SIN(3.1415926*(angreal/180.))+1
!          imgmxH = COS(3.1415926*(angreal/180.))*mx-SIN(3.1415926*(angreal/180.))
          imgmxL = 1
          imgmxH = mx
          imgmx = imgmxH-imgmxL+1
          imgmy = my
          
          PRINT*,imgmx,imgmy
          ALLOCATE(imageHalpha(imgmxL:imgmxH,imgmy),imageSii(imgmxL:imgmxH,imgmy))
          imageHalpha = 0d0;imageSii = 0d0

          DO i=1,mx
          DO j=1,my
          DO k=1,mz
             iim = NINT(COS(3.1415926*angreal/180.)*REAL(i)) + NINT(SIN(3.1415926*angreal/180.)*REAL(k))
             jim = j
             IF(iim/=iim) PRINT*,i,j,k
             IF(iim > imgmxL .AND. iim <= imgmxH) THEN
                imageHalpha(iim,jim) = imageHalpha(iim,jim) + Halpha(i,j,k)
                imageSii(iim,jim) = imageSii(iim,jim) + Sii(i,j,k)
               END IF
          END DO
          END DO
          END DO
          imageHalpha = imageHalpha*lScale*( COS(3.1415926*angreal/180.)*units(3)+SIN(3.1415926*angreal/180.)*units(1) )
          imageSii = imageSii*lScale*( COS(3.1415926*angreal/180.)*units(3)+SIN(3.1415926*angreal/180.)*units(1) )
          minscale = minscale*lScale*( COS(3.1415926*angreal/180.)*units(3)+SIN(3.1415926*angreal/180.)*units(1) )
          maxscale = maxscale*lScale*( COS(3.1415926*angreal/180.)*units(3)+SIN(3.1415926*angreal/180.)*units(1) )


       ELSE
          imgmxL = 1
          imgmxH = mx/2
          imgmx = imgmxH-imgmxL+1
          imgmy = my/2
          ALLOCATE(imageHalpha(imgmx,imgmy),imageSii(imgmx,imgmy))

          camera_pos=(/ 640,160,-640 /)
          camera_focus=(/ 640,160,160 /)
          camera_up=(/ 0,1,0 /)
          ! 0*
          !camera_norm = (/ 0, 0, 1 /)
          ! 20*
          camera_norm=(/.34202, 0, 0.93969 /)  !camera_pos-camera_focus
          ! 90*
          !camera_norm = (/ 1, 0, 0 /)
          camera_norm=camera_norm/SQRT(DOT_PRODUCT(camera_norm,camera_norm))
          camera_up=camera_up-DOT_PRODUCT(camera_up,camera_norm)*camera_norm
          camera_up=camera_up/SQRT(DOT_PRODUCT(camera_up,camera_up))
          camera_right=cross_prod(camera_up,camera_norm)

          fov=3.1415926*(90./180)
          screenz=imgmx/2*TAN(fov/2.)**-1

          PRINT*,'camera_up',camera_up
          PRINT*,'camera_norm',camera_norm
          PRINT*,'camera_right',camera_right
          PRINT*,'screenz',screenz

          DO i=1,mx
             DO j=1,my
                DO k=1,mz
                   vec = (/ i,j,k /) - camera_pos
                   vecz = DOT_PRODUCT(vec,camera_norm)
                   vecy = DOT_PRODUCT(vec,camera_up)
                   vecx = DOT_PRODUCT(vec,camera_right)
                   posx = NINT(imgmx/2. + vecx/vecz * screenz)
                   posy = NINT(imgmy/2. + vecy/vecz * screenz)

                   IF(posx >= 1 .AND. posx <= imgmx .AND. posy >= 1 .AND. posy <= imgmy) THEN
                      !                   PRINT*,'yes',posx,posy
                      imageHalpha(posx, posy) = imageHalpha(posx,posy) + Halpha(i,j,k)
                      imageSii(posx, posy) = imageSii(posx,posy) + Sii(i,j,k)
                   END IF
                END DO
             END DO
          END DO
       END IF
    END IF

!    WHERE(ABS(imageHalpha)==1d0/0d0) imageHalpha=0d0

!    PRINT*,minscale,max
!    PRINT*,MINVAL(imageHalpha),MAXVAL(imageHalpha)

    PRINT*,minscale,maxscale
    PRINT*,MINVAL(imageHalpha),MAXVAL(imageHalpha),MINVAL(imageSii),MAXVAL(imageSii)

    IF(Logscale) THEN
       WHERE(imageHalpha>0) imageHalpha = LOG10(imageHalpha)
       WHERE(imageSii>0) imageSii = LOG10(imageSii)
       IF(scales==0) THEN
          minscale=MAX(-12d0,MINVAL(imageHalpha,imageHalpha/=0))
          maxscale=MAXVAL(imageHalpha,imageHalpha/=0)
       ELSEIF(scales==1) THEN
          minscale=LOG10(minscale)
          maxscale=LOG10(maxscale)
       ENDIF
       WHERE(imageHalpha==0) imageHalpha=minscale
       WHERE(imageSii==0) imageSii=minscale
    ELSEIF(scales==0) THEN
       minscale=MAX(1e-12,MINVAL(imageHalpha))
       maxscale=MAXVAL(imageHalpha)
    ENDIF

    ! Final NaN check
    WHERE(imageHalpha/=imageHalpha) imageHalpha=minscale


    PRINT*,'min/max scale: ',minscale,maxscale



    ! Grayscale plot Halpha, Sii

    filename=''
    WRITE(filename,'(A11,I4.4)')'Halpha-'//angchar//'_',frame
    CALL PGOPEN(TRIM(filename)//'.ps/cps')
    CALL PGSetup(imgmxL,imgmxH,imgmy,TR)
!    PRINT*,SHAPE(imageHalpha)
!    PRINT*,MINVAL(imageHalpha),MAXVAL(imageHalpha)
!    PRINT*,imgmx,imgmy,imgmxL,imgmxH
!    PRINT*,minscale,maxscale
    CALL PGIMAG(imageHalpha,imgmx,imgmy,imgmxL,imgmxH,1,imgmy,minscale,maxscale,TR)
    ! set the color
    CALL PGSCI(1)
    !Annotate the plot.
    CALL PGSCH(1.)

    CALL PGMTXT('t',1.0,0.0,0.0,'Y')
    CALL PGMTXT('b',3.0,1.0,1.0,'X')

!    CALL PGMTXT('t',1.0,0.0,0.0,'R')
!    CALL PGMTXT('b',3.0,1.0,1.0,'Z')

    CALL PGBOX('bcntsi',0.0,0,'bcntsiv',0.0,0)
    ! draw color wedge
    WRITE(plotlabel,'(A)')''
!!$    IF(Logscale) THEN
!!$       WRITE(plotlabel,'(A)')'log\d10\u(H-alpha)'
!!$    ELSE
!!$       WRITE(plotlabel,'(A)')'H-alpha'
!!$    END IF
    CALL PGWEDG('RI', 1.0, 4.0, minscale, maxscale, TRIM(plotlabel))
    CALL PGEND
    !CALL SYSTEM('echo convert -rotate 90 -density 100x100 '//TRIM(filename)//'.ps '//TRIM(filename)//'.png')
    CALL SYSTEM('convert -rotate 90 -density 100x100 '//TRIM(filename)//'.ps '//TRIM(filename)//'.png')
    CALL SYSTEM('convert '//TRIM(filename)//'.png +level-colors black,lime green'//TRIM(framechar)//'.png')

    WRITE(filename,'(A8,I4.4)')'Sii-'//angchar//'_',frame
    CALL PGOPEN(TRIM(filename)//'.ps/cps')
    CALL PGSetup(imgmxL,imgmxH,imgmy,TR)
    CALL PGIMAG(imageSii,imgmx,imgmy,imgmxL,imgmxH,1,imgmy,minscale,maxscale,TR)
    ! set the color
    CALL PGSCI(1)
    !Annotate the plot.
    CALL PGSCH(1.)

    CALL PGMTXT('t',1.0,0.0,0.0,'Y')
    CALL PGMTXT('b',3.0,1.0,1.0,'X')

!    CALL PGMTXT('t',1.0,0.0,0.0,'R')
!    CALL PGMTXT('b',3.0,1.0,1.0,'Z')

    CALL PGBOX('bcntsi',0.0,0,'bcntsiv',0.0,0)
    ! draw color wedge
!!$    IF(Logscale) THEN
!!$       WRITE(plotlabel,'(A)')'log\d10\u(SII [2-1])'
!!$    ELSE
!!$       WRITE(plotlabel,'(A)')'SII [2-1]'
!!$    END IF
    CALL PGWEDG('RI', 1.0, 4.0, minscale, maxscale, TRIM(plotlabel))
    CALL PGEND
    !CALL SYSTEM('echo convert -rotate 90 -density 100x100 '//TRIM(filename)//'.ps '//TRIM(filename)//'.png')
    CALL SYSTEM('convert -rotate 90 -density 100x100 '//TRIM(filename)//'.ps '//TRIM(filename)//'.png')
    CALL SYSTEM('convert '//TRIM(filename)//'.png +level-colors black,red red'//TRIM(framechar)//'.png')

    CALL SYSTEM('composite -compose plus red'//TRIM(framechar)//'.png green'//TRIM(framechar)//'.png emission-'//angchar//'_'//TRIM(framechar)//'.png')


    IF(lWriteFits) THEN
       WRITE(filename,'(A6,I4.4)')'Halpha',frame
       CALL writfits_r4(TRIM(filename)//'.fits',imageHalpha,imgmx,imgmy)
       WRITE(filename,'(A3,I4.4)')'Sii',frame
       CALL writfits_r4(TRIM(filename)//'.fits',imageSii,imgmx,imgmy)
    END IF

    DEALLOCATE(imageSii,imageHalpha)

  END SUBROUTINE PlotData

! -----  Helper functions ----

  FUNCTION cross_prod(A,B)
    REAL,DIMENSION(3), INTENT (IN)    :: A
    REAL,DIMENSION(3), INTENT (IN)    :: B
    REAL,DIMENSION(3) :: cross_prod,C


    C(1) = A(2)*B(3) - A(3)*B(2)
    C(2) = A(3)*B(1) - A(1)*B(3)
    C(3) = A(1)*B(2) - A(2)*B(1)

    cross_prod = C
  END FUNCTION cross_prod

  SUBROUTINE PGSetup(imgmxL,imgmxH,imgmy,TR)
    INTEGER :: imgmxL,imgmxH,imgmy
    REAL,DIMENSION(6),INTENT(OUT) :: TR
    !
    REAL :: CONTRA, BRIGHT
    REAL :: VPX1,VPX2, VPY1,VPY2,D
    !
    REAL :: XL(2),XU(2)
    INTEGER :: mF(2)
    !

    !PRINT*,imgmx,imgmy

    XL = (/ 1.0, 1.0 /)
    XU = (/ units(1), units(2) /) * lScale /1.5e13
    mF = (/ (imgmxH-imgmxL+1), imgmy /)

    CONTRA = 1.0
    BRIGHT = 0.5
    ! Setup color table
    ! 1 - gray
    ! 2 - rainbow
    ! 3 - heat
    CALL PALETT(1, CONTRA, BRIGHT)
!    CALL PGCTAB( ColorTab(:,1), ColorTab(:,2), ColorTab(:,3), ColorTab(:,4), 2, CONTRA, BRIGHT)
    ! Setup page
    CALL PGPAGE
    CALL PGSVP( 0.0, 1.0, 0.0, 1.0 )
    CALL PGQVP( 1, VPX1, VPX2, VPY1, VPY2 )
    D = MIN(VPX2-VPX1, VPY2-VPY1)/40.0
    VPX1 = VPX1 + 5.0*D
    VPX2 = VPX2 - 8.0*D
    VPY1 = VPY1 + 5.0*D
    VPY2 = VPY2 - 2.0*D
    CALL PGVSIZ( VPX1, VPX2, VPY1, VPY2 )

    CALL PGWNAD( XL(1), XU(1), XL(2), XU(2) )
    ! Setup line width
    CALL PGSLW(2)
    ! Setup coordinate transformation
    TR(1) = XL(1)-0.5*(XU(1)-XL(1))/REAL(mF(1))
    TR(2) = (XU(1)-XL(1))/REAL(mF(1))
    TR(3) = 0.0
    TR(4) = XL(2)-0.5*(XU(2)-XL(2))/REAL(mF(2))
    TR(5) = 0.0
    TR(6) = (XU(2)-XL(2))/REAL(mF(2))
  END SUBROUTINE PGSetup

 SUBROUTINE PALETT(TYPE, CONTRA, BRIGHT)
    !-----------------------------------------------------------------------
    ! Set a "palette" of colors in the range of color indices used by
    ! PGIMAG.
    !-----------------------------------------------------------------------
    INTEGER TYPE
    REAL CONTRA, BRIGHT
    !
    REAL GL(2), GR(2), GG(2), GB(2)
    REAL RL(9), RR(9), RG(9), RB(9)
    REAL HL(5), HR(5), HG(5), HB(5)
    REAL WL(10), WR(10), WG(10), WB(10)
    REAL AL(20), AR(20), AG(20), AB(20)
    !
    DATA GL /0.0, 1.0/
    DATA GR /0.0, 1.0/
    DATA GG /0.0, 1.0/
    DATA GB /0.0, 1.0/
    !
    DATA RL /-0.5, 0.0, 0.17, 0.33, 0.50, 0.67, 0.83, 1.0, 1.7/
    DATA RR / 0.0, 0.0,  0.0,  0.0,  0.6,  1.0,  1.0, 1.0, 1.0/
    DATA RG / 0.0, 0.0,  0.0,  1.0,  1.0,  1.0,  0.6, 0.0, 1.0/
    DATA RB / 0.0, 0.3,  0.8,  1.0,  0.3,  0.0,  0.0, 0.0, 1.0/
    !
    DATA HL /0.0, 0.2, 0.4, 0.6, 1.0/
    DATA HR /0.0, 0.5, 1.0, 1.0, 1.0/
    DATA HG /0.0, 0.0, 0.5, 1.0, 1.0/
    DATA HB /0.0, 0.0, 0.0, 0.3, 1.0/
    !
    DATA WL /0.0, 0.5, 0.5, 0.7, 0.7, 0.85, 0.85, 0.95, 0.95, 1.0/
    DATA WR /0.0, 1.0, 0.0, 0.0, 0.3,  0.8,  0.3,  1.0,  1.0, 1.0/
    DATA WG /0.0, 0.5, 0.4, 1.0, 0.0,  0.0,  0.2,  0.7,  1.0, 1.0/
    DATA WB /0.0, 0.0, 0.0, 0.0, 0.4,  1.0,  0.0,  0.0, 0.95, 1.0/
    !
    DATA AL /0.0, 0.1, 0.1, 0.2, 0.2, 0.3, 0.3, 0.4, 0.4, 0.5, &
             0.5, 0.6, 0.6, 0.7, 0.7, 0.8, 0.8, 0.9, 0.9, 1.0/
    DATA AR /0.0, 0.0, 0.3, 0.3, 0.5, 0.5, 0.0, 0.0, 0.0, 0.0, &
             0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0/
    DATA AG /0.0, 0.0, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.8, 0.8, &
             0.6, 0.6, 1.0, 1.0, 1.0, 1.0, 0.8, 0.8, 0.0, 0.0/
    DATA AB /0.0, 0.0, 0.3, 0.3, 0.7, 0.7, 0.7, 0.7, 0.9, 0.9, &
             0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0/
    !
    IF (TYPE.EQ.1) THEN
       !-- gray scale
       CALL PGCTAB(GL, GR, GG, GB, 2, CONTRA, BRIGHT)
    ELSE IF (TYPE.EQ.2) THEN
       !-- rainbow
       CALL PGCTAB(RL, RR, RG, RB, 9, CONTRA, BRIGHT)
    ELSE IF (TYPE.EQ.3) THEN
       !-- heat
       CALL PGCTAB(HL, HR, HG, HB, 5, CONTRA, BRIGHT)
    ELSE IF (TYPE.EQ.4) THEN
       !-- weird IRAF
       CALL PGCTAB(WL, WR, WG, WB, 10, CONTRA, BRIGHT)
    ELSE IF (TYPE.EQ.5) THEN
       !-- AIPS
       CALL PGCTAB(AL, AR, AG, AB, 20, CONTRA, BRIGHT)
    END IF
  END SUBROUTINE PALETT



END PROGRAM angledemission
