!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    bear2fixIO.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE BearIO
  USE GlobalDeclarations
  USE HDF5
  IMPLICIT NONE
  PUBLIC
  ! fixed grid variables
  INTEGER(HID_T) :: hid_box_id
  INTEGER(HID_T) :: hid_bound_id
  REAL(KIND=xPrec) ::  Time,cr ! fixed grid coarsening ratio, relative to finest level
  REAL(KIND=xPrec), DIMENSION(4) :: dxfine,dxFix,Xlower,Xupper
  REAL(KIND=xPrec), DIMENSION(3) :: xBound 
  INTEGER,DIMENSION(4) :: mxFix
  INTEGER :: nvars,ngrids,nFrame
  INTEGER :: qnvars=-1,userfield=-1
  REAL, POINTER, DIMENSION(:,:,:,:) :: qFix
  REAL(KIND=qprec), POINTER, DIMENSION(:,:,:,:,:) :: auxFix
  REAL, ALLOCATABLE, DIMENSION(:,:) :: gridBounds
  INTEGER, ALLOCATABLE, DIMENSION(:) :: gridLevels

  ! fixed grid of primitive variables
  REAL, POINTER, DIMENSION(:,:,:,:) :: pFix
  INTEGER :: pnvars

  ! [BDS] [20071023]:  File input variables for use in the CreateFixedGrid() function.
  INTEGER :: i_input_type
  INTEGER, PARAMETER :: I_STANDARD_INPUT = 1
  INTEGER, PARAMETER :: I_CHOMBO_INPUT = 2

  ! field output flag
  INTEGER,PARAMETER :: CONS=1,PRIM=2
  ! quantity constants
  INTEGER,PARAMETER :: qn=0,qrho=1,qvx=2,qvy=3,qvz=4,qv=5,qEth=6,qP=7,qTemp=8,qcs=9,qemission=10, &
       mBtot=12, mEtot=13,mbx=14,mby=15,mbz=16,qvorticity=17,beta=18
  CHARACTER(LEN=9),DIMENSION(0:21) :: qlist=(/ 'n        ', 'rho      ', 'vx       ', 'vy       ', 'vz       ' &
       , 'v        ', 'Eth      ', 'P        ', 'Temp     ', 'cs       ' &
       , 'emission ', '         ', '         ', 'mEtot    ', 'mbx      ' &
       , 'mby      ', 'mbz      ', 'vorticity', 'beta     ', 'tracer1  ' &
       , 'tracer2  ', 'tracer3  '/)

  CHARACTER(LEN=3) :: outdir='out'

  ! Chombo Parameter Declarations	
  CHARACTER(LEN = 1), PARAMETER :: S_ROOT_GROUP = '/'
  REAL(KIND = qPrec), PARAMETER :: DBL_TEST_REAL_DAT = 1.0
  INTEGER(HID_T), PARAMETER :: HID_ATTRIBUTE_RANK = 0
  INTEGER(HSIZE_T), DIMENSION(1), PARAMETER :: IA_SCALAR_ATTRIB_DIMS = (/1/)
  INTEGER, PARAMETER ::   I_MAX_COMPONENTS = 20
  INTEGER, PARAMETER :: I_MAX_CNAME_LENGTH = 100
  INTEGER, PARAMETER :: I_MAX_NUM_LEVELS = 20
  INTEGER, PARAMETER :: I_DATASET_RANK = 1
  INTEGER, PARAMETER :: I_DEFLATE_LEVEL=6
  
  DOUBLE PRECISION :: dbl_tstart, dbl_tend, t_accumulator = 0.d0
  INTEGER :: t_counter = 0

  LOGICAL :: lConvert=.false.

CONTAINS

  ! Updated 20071023 by Brandon D. Shroyer.  Added code to select the file input type.
  !
  ! allocates and interpolates fixed grid qFix
  ! frame = dataset number
  ! coarseRatio =  fixed grid coarsening ratio
  SUBROUTINE CreateFixedGrid(frame)
    INTEGER :: frame
    !
    INTEGER :: dref ! derefinment ratio relative to fixed grid
    INTEGER :: iErr,maxlevel,i,j,k,nv,i0,j0,k0,iErr2
    !
    ![BDS][20070205]:  Deallocate qFix array.
    CHARACTER(LEN=23) :: TreeFile

    lDataFromBlueGene=.false.

    IF (ASSOCIATED(qFix)) DEALLOCATE(qFix)
    IF (ASSOCIATED(auxFix)) DEALLOCATE(auxFix)
    NULLIFY(qfix, auxfix)

    ! [BDS][20071023]:  Generate a fixed grid using the out/chombo[frame].hdf input file.
    WRITE(TreeFile,'(A2,A3,A7,I5.5,A4)')'./',outdir,'/chombo',frame,'.hdf'
    OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr)
    CLOSE(UNIT=77)

    IF (iErr == 0) THEN
    !   lConvert=lDataFromBlueGene
       CALL FixedGridFromChombo(frame)
    !   lConvert=.false.
    ELSE
       write(TreeFile,'(A3,A7,I3.3A4)') outdir, "/W_000_",  frame , ".bov"
       OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr)
       CLOSE(UNIT=77)
       write(TreeFile,'(A3,A3,I3.3A4)') outdir,"/W_" , frame , ".bov"
       OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr2)
       CLOSE(UNIT=77)
       IF (iErr == 0 .OR. iErr2==0) THEN
          CALL FixedGridFromBOV(frame,3)
       ELSE
          write(TreeFile,'(A3,A8,I4.4A4)') outdir,"/W_0000_",  frame , ".bov"
          OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr)
          CLOSE(UNIT=77)
          write(TreeFile,'(A3,A3,I4.4A4)') outdir,"/W_" , frame , ".bov"
          OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr2)
          CLOSE(UNIT=77)
          IF (iErr == 0 .OR. iErr2==0) THEN
             CALL FixedGridFromBOV(frame,4)
          ELSE
             PRINT*,'Unable to find data file(s) for frame',frame
          END IF
       END IF
    END IF

  END SUBROUTINE CreateFixedGrid

  ! writes data set given by field parameter out to hdf5 file
SUBROUTINE WriteHDF(i_frame)  ! fixed grid HDF5
	USE HDF5
	! Input parameter declarations.
	INTEGER, INTENT(IN) :: i_frame			! Frame number.
 
	! Variable Declarations
	CHARACTER(LEN=23) :: s_filename			! Partially-pathed filename.
        CHARACTER(LEN=7)::coarse_level_group
	INTEGER :: i_err

        !Type IDs
	INTEGER(HID_T) :: hid_file_id			! Chombo HDF file handle
        INTEGER(HID_T) :: hid_dspace_id
        INTEGER(HID_T) :: hid_dset_id
        INTEGER(HSIZE_T)::max_dims
        INTEGER(HSIZE_T), DIMENSION(4)::  hid_dimensions

	INTEGER :: i, j, k
	INTEGER, POINTER, DIMENSION(:,:) :: ia_box_data
	INTEGER, DIMENSION(6) :: ia_prob_domain
	REAL(KIND=xPrec), POINTER, DIMENSION(:) :: dbla_data
	INTEGER, POINTER, DIMENSION(:) :: ia_ref_ratios
	LOGICAL, POINTER, DIMENSION(:) :: la_ref_mask

	INTEGER (HSIZE_T) :: i_index
        INTEGER :: ierr
	INTEGER :: i_grid
	INTEGER :: i_level
	INTEGER :: i_maxlevel
	INTEGER :: i_variable
	INTEGER :: i_gridcount
	INTEGER :: i_varcount
	INTEGER :: i_space_dim
	INTEGER :: i_level_ref_ratio
        INTEGER :: ngrid, mx(3)
        REAL(KIND=xPrec),DIMENSION(3) :: dx

        
	! Initializes the HDF5 library.  This function is safe to call
	! even if it has already been called.
	CALL h5open_f(i_err)

	! Die if unable to initialize HDF5 library.	
	IF (i_err < 0) THEN 
	        PRINT *,'WriteHDF error ', i_err, ': call to h5open_f failed.'
        	STOP
	END IF

	! Create file name.
	WRITE(s_filename, '(A3,A6,I5.5,A3)') outdir, '/fixed', i_frame, '.h5'
!        PRINT*, s_filename

        CALL h5fcreate_f(TRIM(s_filename),H5F_ACC_TRUNC_F,hid_file_id,i_err)

	! Die if unable to open s_filename.
	IF (i_err < 0) THEN
		PRINT *, 'WriteHDF5 error ', i_err, ': unable to create file ', &
			s_filename, '.'
		STOP
	END IF

        DO i = 1,4
           hid_dimensions(i) = SIZE(qFix,i)
        END DO
        CALL h5screate_simple_f(4,hid_dimensions,hid_dspace_id,ierr)
        CALL h5dcreate_f(hid_file_id,"fixed grid data",H5T_NATIVE_REAL,hid_dspace_id,hid_dset_id,ierr)
        IF (ierr==0) PRINT*,"created dataspace"
        CALL h5dwrite_f(hid_dset_id,H5T_NATIVE_REAL,qFix,hid_dimensions,ierr)
        IF (ierr==0) THEN
           PRINT *, "wrote h5 file"
        ELSE
           PRINT *, "write failed"
           PRINT *, "hdf5 error=",ierr
           STOP
        END IF
        CALL h5dread_f(hid_dset_id,H5T_NATIVE_REAL,qFix,hid_dimensions,ierr)
        IF (ierr==0)  PRINT*,"Wrote h5 file"

        CALL h5dclose_f(hid_dset_id,ierr)
        CALL h5sclose_f(hid_dspace_id,ierr)
        CALL h5fclose_f(hid_file_id,ierr)
        CALL h5close_f(ierr)


END SUBROUTINE WriteHDF



  ! writes data set given by field parameter out to fits file
  SUBROUTINE WriteFits(field)
    INTEGER :: field ! field to write.  1) qFix - raw conserved fields
                     !                  2) pFix - any fields in primitive array
    CHARACTER(LEN=14) :: fileName

    WRITE(fileName,'(A3,A6,I5.5)') outdir,'/pfix',nFrame
    SELECT CASE(nDim)
    CASE(2)
       CALL WriteFits2D(filename,field)
    CASE(3)
       !CALL WriteFits3D(filename,field)
    END SELECT
    !CALL WriteHDFBounds
  END SUBROUTINE WriteFits

  ! writes data set given by field parameter out to hdf file
  SUBROUTINE WriteFits2D(filename,field)
    INTEGER :: field ! field to write.
    CHARACTER(LEN=*) :: fileName
!!$    CHARACTER(LEN=1) ::fieldname
!!$    INTEGER :: iErr,nv,valnvars
!!$    !
!!$    LOGICAL ::simple,extend
!!$    INTEGER :: status,unit,blocksize,bitpix,naxis,group,fpixel,nelements
!!$    INTEGER, DIMENSION(2) :: naxes
!!$    !
!!$    REAL,POINTER,DIMENSION(:,:) :: qbuf
!!$    REAL,POINTER,DIMENSION(:,:,:,:) :: valFix
!!$
!!$    ALLOCATE(qbuf(1:mxFix(1),1:mxFix(2)), STAT=iErr)
!!$    IF (iErr /= 0) THEN
!!$       PRINT *,'!!! ERROR: failed to allocate qbuf array'
!!$       STOP
!!$    END IF
!!$
!!$    print*,mxfix(1:2)
!!$
!!$    SELECT CASE(field)
!!$    CASE(CONS)
!!$       valFix=>qFix
!!$       valnvars=qnvars
!!$    CASE(PRIM)
!!$       valFix=>pFix
!!$       valnvars=pnvars
!!$    END SELECT
!!$    DO nv=1,valnvars
!!$       WRITE(fieldName,'(I1.1)') nv
!!$       qbuf=valFix(:,:,1,nv)
!!$       !
!!$       status=0
!!$       CALL ftgiou(unit,status)
!!$       blocksize=1
!!$       CALL ftinit(unit,filename//'.'//fieldname//'.fits',blocksize,status)
!!$       simple=.TRUE.
!!$       bitpix=-32
!!$       naxis=2
!!$       naxes(1)=mxFix(1)
!!$       naxes(2)=mxFix(2)
!!$       extend=.FALSE.
!!$       CALL ftphpr(unit,simple,bitpix,naxis,naxes,0,1,extend,status)
!!$       group=1
!!$       fpixel=1
!!$       nelements=naxes(1)*naxes(2)
!!$       CALL ftppre(unit,group,fpixel,nelements,qbuf,status)
!!$       CALL ftclos(unit,status)
!!$       CALL ftfiou(unit,status)
!!$    END DO
!!$    DEALLOCATE(qbuf)
  END SUBROUTINE WriteFits2D

  ! allocates fixed grid primitive array pFix
  SUBROUTINE PrimitiveInit(nfields)
    INTEGER :: nfields,iErr
   
   !write(*,*) "PrimitiveInit, mxFix=", mxFix

    IF(pnvars/=nfields .AND. ASSOCIATED(pFix)) DEALLOCATE(pfix)
    pnvars=nfields
    IF(.NOT. ASSOCIATED(pFix)) THEN
       ALLOCATE(pFix(1:mxFix(1),1:mxFix(2),1:mxFix(3),pnvars),STAT=iErr)
       IF (iErr /= 0) THEN
          PRINT *,'!!! ERROR: failed to allocate p array'
          STOP
       END IF
    END IF
  END SUBROUTINE PrimitiveInit

  ! computes primitive field quantity in p array
  SUBROUTINE Primitive(quantity,field)
    USE GlobalDeclarations
    INTEGER :: quantity,field
    INTEGER :: i,j,k

    SELECT CASE(quantity)
    CASE(qvorticity)
       SELECT CASE(iSpeedHI)
       CASE(3)
          k=1
          pFix(:,:,:,field) = 0
          DO i=2,mxFix(1)-1;DO j=2,mxFix(2)-1
             pFix(i,j,k,field) = &
                  (qFix(i-1,j,k,3)/qFix(i-1,j,k,1)-qFix(i+1,j,k,3)/qFix(i+1,j,k,1))/dxFix(1) - &
                  (qFix(i,j-1,k,2)/qFix(i,j-1,k,1)-qFix(i,j+1,k,2)/qFix(i,j+1,k,1))/dxFix(2)
          END DO;END DO
       CASE(4)
          pFix(:,:,:,field:field+2) = 0
          DO i=2,mxFix(1)-1;DO j=2,mxFix(2)-1;DO k=2,mxFix(3)-1
             pFix(i,j,k,field)   = &
                  (qFix(i,j-1,k,4)/qFix(i,j-1,k,1)-qFix(i,j+1,k,4)/qFix(i,j+1,k,1))/dxFix(2) - &
                  (qFix(i,j,k-1,3)/qFix(i,j,k-1,1)-qFix(i,j,k+1,3)/qFix(i,j,k+1,1))/dxFix(3)
             pFix(i,j,k,field+1) = &
                  (qFix(i,j,k-1,2)/qFix(i,j,k-1,1)-qFix(i,j,k+1,2)/qFix(i,j,k+1,1))/dxFix(3) - &
                  (qFix(i-1,j,k,4)/qFix(i-1,j,k,1)-qFix(i+1,j,k,4)/qFix(i+1,j,k,1))/dxFix(1)
             pFix(i,j,k,field+2) = &
                  (qFix(i-1,j,k,3)/qFix(i-1,j,k,1)-qFix(i+1,j,k,3)/qFix(i+1,j,k,1))/dxFix(1) - &
                  (qFix(i,j-1,k,2)/qFix(i,j-1,k,1)-qFix(i,j+1,k,2)/qFix(i,j+1,k,1))/dxFix(2)
          END DO;END DO;END DO
          pFix(:,:,:,field:field+2) = pFix(:,:,:,field:field+2)/RunTimeSc
       END SELECT
    CASE  DEFAULT
       DO i=1,mxFix(1);DO j=1,mxFix(2);DO k=1,mxFix(3)
          pFix(i,j,k,field) = PrimCell(quantity,qFix(i,j,k,:))
       END DO;END DO;END DO
    END SELECT
  END SUBROUTINE Primitive

  ! compute primitive value for one cell
  FUNCTION PrimCell(quantity,q)
    USE GlobalDeclarations
    USE Cool
    REAL :: PrimCell,gc,gis
    REAL,DIMENSION(:) :: q
    INTEGER :: quantity
    
    SELECT CASE(quantity)
    CASE(qn)
       PrimCell = q(1)*nScale
    CASE(qrho)
       PrimCell = q(1)*rScale
    CASE(qvx)
       PrimCell = q(2)/q(1)*velScale
    CASE(qvy)
       PrimCell = q(3)/q(1)*velScale
    CASE(qvz)
       PrimCell = q(4)/q(1)*velScale
    CASE(qv)
       PrimCell = SQRT(DOT_PRODUCT(q(2:iSpeedHI),q(2:iSpeedHI))/(q(1)**2))*velScale
    CASE(qEth)
       PrimCell = (q(iE)-0.5*DOT_PRODUCT(q(2:iSpeedHI),q(2:iSpeedHI))/q(1))*pScale
    CASE(qP)
       PrimCell = Press(q)*pScale
    CASE(qTemp)
       PrimCell = (Press(q)/q(1))*TempScale
    CASE(qemission)
       PrimCell = Press(q)*EOSConstants/q(1)
       PrimCell = (q(1)*nScale)**2*DMCoolingRate(REAL(PrimCell,KIND=qprec))
    CASE(qcs)
       gc=ratio_heat(q,gis)
       PrimCell=SQRT(gis*Press(q)/q(1)) * velscale
    ! MHD related fields -- require no globaldeclarations variables
    CASE(mBtot)
       PrimCell = .5*SUM(q(6:8)**2)*pScale!-0.5*(DOT_PRODUCT(q(2:4),q(2:4))/q(1)-DOT_PRODUCT(q(6:8),q(6:8)))
    CASE(mEtot)
       PrimCell = q(5)*pScale!-0.5*(DOT_PRODUCT(q(2:4),q(2:4))/q(1)-DOT_PRODUCT(q(6:8),q(6:8)))
    CASE(mbx)
       PrimCell = q(6)*SQRT(pScale)
    CASE(mby)
       PrimCell = q(7)*SQRT(pScale)
    CASE(mbz)
       PrimCell = q(8)*SQRT(pScale)
    CASE(beta)
       PrimCell = (2*NoMagPress(q))/(q(6)**2 + q(7)**2 + q(8)**2)
    END SELECT
  CONTAINS

    ! returns gamma 
    ! if present, gammais returns with value of isentropic gamma
    FUNCTION ratio_heat(q,gammais)
      USE cool
      !USE TF
      REAL, DIMENSION(:), INTENT(IN) :: q
      REAL :: gamma_neq,ratio_heat,gammaisin
      REAL, OPTIONAL, INTENT(OUT) :: gammais
      REAL(KIND=qprec) :: dgamma_neq,dgammaisin

      IF(icooling == 2) THEN
         IF(PRESENT(gammais)) THEN
            CALL EOS_vars(REAL(q,qprec),gamma=dgamma_neq,gammais=dgammaisin)
            gammaisin=dgammaisin
         ELSE
            CALL EOS_vars(REAL(q,qprec),gamma=dgamma_neq)
         END IF
         gamma_neq=dgamma_neq
         ratio_heat = gamma_neq
      ELSE
         SELECT CASE(iEOS)
         CASE(0)
            ratio_heat = gamma
            gammaisin = gamma
         CASE(1)
            gammaisin = gamma
            ratio_heat = gammac
         CASE DEFAULT
            !CALL TFEOS_SC(q(:),ratio_heat,gammaisin)
            PRINT*,'EOS option not yet implemented in bear2fix'
            STOP
         END SELECT
      END IF
      IF(PRESENT(gammais)) gammais=gammaisin
    END FUNCTION ratio_heat

    ! compute pressure, depending on what microphysics we have going on
    ! chi =  derivative of pressure wrt conserved variables, if present 
    !        -- sometimes needed geometric jacobian terms
    ! retuns magnetic+themal pressure in pseudo-MHD case
    FUNCTION Press(q,chi,gamma,gammais)
      REAL, DIMENSION(:), INTENT(IN) :: q
      REAL, DIMENSION(:), OPTIONAL, INTENT(OUT) :: chi
      REAL, OPTIONAL,INTENT(OUT) :: gamma,gammais
      REAL :: Press
      ! Internal variables
      REAL (KIND=qPrec) :: gamma1, ke, invq1, be

      IF(PRESENT(gammais)) gammais = ratio_heat(q,gammais=gammais)
      gamma1 = ratio_heat(q)-1.
      IF(PRESENT(gamma)) gamma=gamma1+1.

      invq1 = 1./q(1)
      ke = 0.5*DOT_PRODUCT(q(2:iSpeedHI),q(2:iSpeedHI))*invq1
      be = 0d0
      IF(lMHD) be = 0.5*DOT_PRODUCT(q(6:8),q(6:8))
      gamma1 = 2d0/3d0
      Press = gamma1*(q(iE) - ke - be)

      IF(PRESENT(chi)) THEN
         chi = 0.
         chi(1)        = gamma1*ke*invq1
         chi(2:iSpeedHI) = -gamma1*q(2:iSpeedHI)/q(1)
         chi(iE)       = gamma1
      END IF
    END FUNCTION Press
    FUNCTION NoMagPress(q,chi,gamma,gammais)
      REAL, DIMENSION(:), INTENT(IN) :: q
      REAL, DIMENSION(:), OPTIONAL, INTENT(OUT) :: chi
      REAL, OPTIONAL,INTENT(OUT) :: gamma,gammais
      REAL :: NoMagPress
      ! Internal variables
      REAL (KIND=qPrec) :: gamma1, ke, invq1, be

      IF(PRESENT(gammais)) gammais = ratio_heat(q,gammais=gammais)
      gamma1 = ratio_heat(q)-1.
      IF(PRESENT(gamma)) gamma=gamma1+1.

      invq1 = 1./q(1)
      ke = 0.5*DOT_PRODUCT(q(2:iSpeedHI),q(2:iSpeedHI))*invq1
      be = 0.5*DOT_PRODUCT(q(6:8),q(6:8))
      NoMagPress = gamma1*(q(iE) - ke - be)

      IF(PRESENT(chi)) THEN
         chi = 0.
         chi(1)        = gamma1*ke*invq1
         chi(2:iSpeedHI) = -gamma1*q(2:iSpeedHI)/q(1)
         chi(iE)       = gamma1
      END IF
    END FUNCTION NoMagPress
  END FUNCTION PrimCell


  SUBROUTINE FixedGridFromBOV(i_frame,fsize)
    CHARACTER(LEN=40) :: filename,datafile
    INTEGER :: filehandle
    INTEGER :: my_rank=0,frameend,i,myilower(3)
    INTEGER :: i_frame,mymx(3),n
    INTEGER :: procs,components,iErr,iErr2,proc_id,fsize
    CHARACTER (LEN=60) :: dummy
    REAL(KIND=xPrec) :: tend, myxlower(3),mysize(3)
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: inputbuffer
    ngrids=1
    filehandle=11
    procs=1
    IF (fsize==3) THEN
       write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir,"/W_" , i_frame , ".bov"
    ELSEIF (fsize == 4) THEN
       write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir,"/W_" , i_frame , ".bov"
    ELSE
       write(*,*) "can't find file in fixedgridfrombov for frame", i_frame
       STOP
    END IF
    OPEN(UNIT=77,FILE=FileName,STATUS='OLD',IOSTAT=iErr2)
    CLOSE(UNIT=77)
    IF (iErr2 .ne. 0) THEN
       iErr=0
       proc_id=0
       DO WHILE (iErr == 0)
          IF (fsize == 3) THEN
             write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir, "/W_" , proc_id, "_",  i_frame , ".bov"
          ELSEIF (fsize==4) THEN
             write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir, "/W_" , proc_id, "_",  i_frame , ".bov"
          ELSE
             write(*,*) "can't find file in fixedgridfrombov for frame", i_frame, "and processor", proc_id
             STOP
          END IF
          OPEN(UNIT=77,FILE=FileName,STATUS='OLD',IOSTAT=iErr)
          CLOSE(UNIT=77)
          proc_id=proc_id+1
       END DO
       IF (proc_id == 1) THEN
          write(*,*) "Unable to find data for frame", i_frame
          STOP
       END IF
       procs=proc_id-1
    END IF
!    write(*,*) "found", procs, "processors for frame", i_frame
    DO proc_id=0,procs-1
       IF (procs > 1) THEN
          IF (fsize == 3) write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir, "/W_" , proc_id, "_",  i_frame , ".bov"
          IF (fsize == 4) write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir, "/W_" , proc_id, "_",  i_frame , ".bov"
       ELSE
          IF (fsize == 3) write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir, "/W_" , i_frame , ".bov"
          IF (fsize == 4) write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir, "/W_" , i_frame , ".bov"
       END IF
       OPEN(UNIT=filehandle, FILE=Filename)
       read(filehandle,'(A6,E25.16)')  dummy, tend
       read(filehandle,'(A11,A)') dummy, datafile
       READ(filehandle,'(A11,3I12)')  dummy, mymX(1), mymX(2), mymX(3)

       READ(filehandle,*)  dummy
       READ(filehandle,* )  dummy
       READ(filehandle,*)  dummy
       READ(filehandle,*)  dummy
       READ(filehandle,'(A14,3F8.3)')  dummy, myXlower(1), myXlower(2), myXlower(3)
       READ(filehandle,'(A12,3F8.3)')  dummy, mysize(1),mysize(2),mysize(3)
       READ(filehandle,*)  dummy
       READ(filehandle,'(A17,I4)')  dummy,components
       CLOSE(filehandle)
       DO i=1,3
          Xlower(i)=min(gi_fixed%Xlower(i),myXlower(i))
          Xupper(i)=max(gi_fixed%Xupper(i),myXlower(i)+mysize(i))
       END DO
       IF (proc_id==0) THEN
          dXfix(1:3)=mysize/real(mymx)
          time=tend
          nrvars=components
          IF (dXfix(3)==0) THEN
             nDim=2
          ELSE
             nDim=3
          END IF
          dxfix(nDim+1:)=1
       END IF
    END DO
    mxfix=1
    mxfix(1:nDim)=floor((Xupper(1:nDim)-Xlower(1:nDIm))/dxfix(1:nDim)+.5d0)
    ALLOCATE(qfix(mxfix(1),mxfix(2),mxfix(3),nrvars))
    DO proc_id=0,procs-1
       IF (procs > 1) THEN
          IF (fsize == 3) write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir,"/W_" , proc_id, "_",  i_frame , ".bov"
          IF (fsize == 4) write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir,"/W_" , proc_id, "_",  i_frame , ".bov"
       ELSE
          IF (fsize == 3) write(FileName,'(A3,A3,I3.3A,I3.3A)') outdir,"/W_" , i_frame , ".bov"
          IF (fsize == 4) write(FileName,'(A3,A3,I4.4A,I4.4A)') outdir,"/W_" , i_frame , ".bov"
       END IF
       OPEN(UNIT=filehandle, FILE=Filename)
       read(filehandle,'(A6,E25.16)')  dummy, tend
       read(filehandle,'(A11,A)') dummy, datafile
       READ(filehandle,'(A11,3I12)')  dummy, mymX(1), mymX(2), mymX(3)

       READ(filehandle,*)  dummy
       READ(filehandle,* )  dummy
       READ(filehandle,*)  dummy
       READ(filehandle,*)  dummy
       READ(filehandle,'(A14,3F8.3)')  dummy, myXlower(1), myXlower(2), myXlower(3)
       READ(filehandle,'(A12,3F8.3)')  dummy, mysize(1),mysize(2),mysize(3)
       READ(filehandle,*)  dummy
       READ(filehandle,'(A17,I4)')  dummy,components
       CLOSE(filehandle)
       write(datafile,*) outdir,'/',TRIM(datafile)
       OPEN(UNIT=filehandle, FILE=datafile, status="OLD", FORM="unformatted",IOSTAT=iErr)
       IF (iErr .ne. 0) THEN
          PRINT*, 'unable to find data file', datafile,'needed by ', Filename
          STOP
       END IF
       !Read in data from data file
       ALLOCATE(inputbuffer(product(mymx(:))*components))
       read(filehandle) inputbuffer      
       close(filehandle)
       myilower=floor((myxlower-xlower(1:3))/dxfix(1:3)+.5d0)+1
!       write(*,*) "processor", proc_id, "reading in data for bounds", myilower,myilower+mymx-1
       qfix(myilower(1):myilower(1)+mymx(1)-1, myilower(2):myilower(2)+mymx(2)-1,myilower(3):myilower(3)+mymx(3)-1,1:nrvars) = &
            reshape(transpose(reshape(inputbuffer,(/nrvars,product(mymX(1:3))/))), (/mymX(1), mymX(2), mymX(3),nrvars/))
       DEALLOCATE(inputbuffer)
    END DO
    
    gi_fixed%Xlower=Xlower
    gi_fixed%Xupper=Xupper
    gi_fixed%dx=dxfix
    gi_fixed%Time=time
    gi_fixed%mX=mxFix
    gi_fixed%q=>qfix
    gi_fixed%nVars=NrVars
    gi_fixed%nDim=nDim

END SUBROUTINE FixedGridFromBOV
! Created 2007-10-23 by Brandon D. Shroyer
!
SUBROUTINE FixedGridFromChombo(i_frame)

  USE HDF5
  ! Input parameter declarations.
  INTEGER, INTENT(IN) :: i_frame			! Frame number.
  INTEGER, DIMENSION(3) :: RootmX
  ! Variable Declarations
  CHARACTER(LEN=23) :: s_filename			! Partially-pathed filename.
  CHARACTER(LEN=7)::coarse_level_group
  INTEGER :: i_err
  INTEGER :: InterpLevel
  INTEGER(HID_T) :: hid_file_id			! Chombo HDF file handle

  ! Chombo group handles.
  INTEGER(HID_T) :: hid_root_group_id
  INTEGER(HID_T) :: hid_chombo_global_group_id
  INTEGER(HID_T) :: hid_level_group_id
  INTEGER(HID_T) :: hid_max_level_group_id
  INTEGER(HID_T) :: hid_attribute_id

  CHARACTER(LEN=7) :: s_level_group

  INTEGER :: i, j, k
  INTEGER, POINTER, DIMENSION(:,:) :: ia_box_data
  INTEGER, DIMENSION(6) :: ia_prob_domain, ia_mhd_prob_domain
  REAL(KIND=xPrec), POINTER, DIMENSION(:) :: dbla_data
  INTEGER, POINTER, DIMENSION(:) :: ia_ref_ratios
  LOGICAL, POINTER, DIMENSION(:) :: la_ref_mask

  INTEGER (HSIZE_T) :: i_index
  INTEGER :: i_grid
  INTEGER :: i_level
  INTEGER :: i_maxlevel
  INTEGER :: i_variable
  INTEGER :: i_gridcount
  INTEGER :: i_varcount
  INTEGER :: i_space_dim
  INTEGER :: i_level_ref_ratio
  INTEGER :: ngrid, mx(3)
  REAL(KIND=xPrec),DIMENSION(3) :: dx

  ! Initializes the HDF5 library.  This function is safe to call
  ! even if it has already been called.
  CALL h5open_f(i_err)

  ! Die if unable to initialize HDF5 library.	
  IF (i_err < 0) THEN 
     PRINT *,'FixedGridFromChombo error ', i_err, ': call to h5open_f failed.'
     STOP
  END IF

  ! Create file name.
  WRITE(s_filename, '(A3,A7,I5.5,A4)') outdir,'/chombo', i_frame, '.hdf'

  ! Open file s_filename with read-only access.
  CALL h5fopen_f(TRIM(s_filename), H5F_ACC_RDONLY_F, hid_file_id, i_err)


  ! Die if unable to open s_filename.
  IF (i_err < 0) THEN
     PRINT *, 'FixedGridFromChombo error ', i_err, ': unable to open file ', &
          s_filename, '.'
     STOP
  END IF

  ! Open root group.
  CALL h5gopen_f(hid_file_id, "/", hid_root_group_id, i_err)

  ! Die if unable to open root group.
  IF (i_err < 0) THEN
     PRINT *, 'FixedGridFromChombo error ', i_err, ': unable to open root group.'
     STOP
  END IF

  ! Open Chombo_global group.
  CALL h5gopen_f(hid_root_group_id, "Chombo_global", hid_chombo_global_group_id, i_err)

  ! Die if unable to open root group.
  IF (i_err < 0) THEN
     PRINT *, 'FixedGridFromChombo error ', i_err, ': unable to open root group.'
     STOP
  END IF

  i_maxlevel = Read_Chombo_Attribute_Int("max_level", hid_root_group_id)
  i_varcount = Read_Chombo_Attribute_Int("num_components", hid_root_group_id)
  !write(*,*) 'i_maxlevel, i_varcount=', i_maxlevel, i_varcount
  i_gridcount = GetNumberOfGrids(hid_root_group_id)

  IF (MaxLevel == -1 .OR. MaxLevel == i_maxlevel) THEN 
     write(*,'(A,I3,A)') "FixedGridFromChombo loading all ", i_maxlevel+1, " levels"
     InterpLevel=i_maxlevel
  ELSE
     IF (MaxLevel < i_maxlevel) write(*,'(A,I3,A1,I3)') "FixedGridFromChombo ignoring levels ",Maxlevel+1,"-", i_maxlevel
     IF (MaxLevel > i_maxlevel) write(*,'(A,I3,A4,I3)') "FixedGridFromChombo Interpolating out from ",i_maxlevel, " to ",Maxlevel
     InterpLevel=MaxLevel
  END IF


  !If an MHD simulation, we need to get the Aux Arrays in order to make a proper restart file
  WRITE(coarse_level_group,'(A6,I1.1)') 'level_', 0

  CALL h5gopen_f(hid_root_group_id, coarse_level_group,hid_level_group_id,i_err)

  IF (i_err<0) THEN
     PRINT *, 'FixedGridFromChombo error ', i_err,': unable to open the first level group'
     STOP
  END IF

  !Read the problem domain for the coarsest level to allocate the auxfix array 
  ia_mhd_prob_domain = Read_Chombo_Attribute_Box("prob_domain", hid_level_group_id)
  !write (*,*) 'prob_domain=', ia_mhd_prob_domain
!  ia_mhd_prob_domain = Read_Chombo_Attribute_Box("prob_domain", hid_level_group_id)
!  write(*,*) ia_mhd_prob_domain, hid_level_group_id
!  i_space_dim = Read_Chombo_Attribute_Int("SpaceDim", hid_chombo_global_group_id)
  i_space_dim = nDim
  RootmX(1:3)=ia_mhd_prob_domain(4:6)-ia_mhd_prob_domain(1:3)+1
  mx=rootmx
  !write(*,*) ia_mhd_prob_domain
  CALL h5gclose_f(hid_level_group_id,i_err)

  ia_prob_domain=0

  ! Die if unable to open root group.
  IF (i_err < 0) THEN
     PRINT *, 'FixedGridFromChombo error ', i_err, ': unable to open root group.'
     STOP
  END IF

  ! Allocate refinement ratio arrays.
  ALLOCATE(ia_ref_ratios(0:InterpLevel), stat=i_err)
  ALLOCATE(la_ref_mask(0:InterpLevel), stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "FixedGridFromChombo error: unable to allocate refinement ratio arrays."
     STOP
  END IF

  ! Set the default refinement ratio array values.
  ia_ref_ratios = 1

  ngrids=0
  ! Fill the refinement ratio array. & find out how many grids
  DO i_level = 0, InterpLevel

     WRITE(s_level_group, '(A6,I1.1)') 'level_', i_level

     ! Open level i_level group.
     CALL h5gopen_f(hid_root_group_id, s_level_group, hid_level_group_id, i_err)
     ! Die if unable to open group.
     IF (i_err < 0) THEN
        PRINT *, "FixedGridFromChombo error:  unable to open ", TRIM(s_level_group), &
             " group to fill ratio array."
        STOP
     END IF

     ! Get the refinement ratio for the current level.
     ia_ref_ratios(i_level) = Read_Chombo_Attribute_Int("ref_ratio", hid_level_group_id)

     ! Read in the box data and the grid data.
     CALL Read_Chombo_Dataset_Box("boxes", hid_level_group_id, ia_box_data)
     ngrids=ngrids+SIZE(ia_box_data, 2)

     CALL h5gclose_f(hid_level_group_id, i_err)
     ! Die if unable to close group.
     IF (i_err < 0) THEN
        PRINT *, "FixedGridFromChombo error: unable to close ", TRIM(s_level_group), &
             " group after filling ratio array."
        STOP
     END IF

  END DO

  ! Open maximum level group.
  CALL h5gopen_f(hid_root_group_id, Coarse_level_group, hid_level_group_id, i_err)


  !Initialize global variables (other than qFix) that are used outside this function.
  !Xlower = Read_Chombo_Attribute_FloatVector("lower_bound", hid_root_group_id)
  !Xupper = Read_Chombo_Attribute_FloatVector("upper_bound", hid_root_group_id)
  !write(*,*) 'after read_chombo_attribute_floatvector, Xlower=', Xlower
  !write(*,*) 'after read_chombo_attribute_floatvector, Xupper=', Xupper

  xBound = Read_Chombo_Attribute_Bound("lower_bound", hid_root_group_id)
  Xlower(1:3) = XBound(:)
  xBound = Read_Chombo_Attribute_Bound("upper_bound", hid_root_group_id)
  Xupper(1:3) = XBound(:)
  !write(*,*) 'after read_chombo_attribute_Bound, Xlower=', Xlower
  !write(*,*) 'after read_chombo_attribute_Bound, Xupper=', Xupper

  !call exit(1)
 
!          dxFix=1.d0

  ! check for anisotropic attribute
  !        CALL h5aopen_name_f(hid_max_level_group_id, "anisotropic" , hid_attribute_id, i_err)
  !        IF(i_err==0) THEN          
  !           dxFix(1:3) = Read_Chombo_Attribute_FloatVector("anisotropic", hid_max_level_group_id)
  !        END IF
  !	dxFix(1:3) = dxFix(1:3)*Read_Chombo_Attribute_Double("dx", hid_max_level_group_id)
  !        CALL h5aclose_f(hid_attribute_id, i_err)


  time = Read_Chombo_Attribute_Double("time", hid_root_group_id)
  write(*,*) 'after read_chombo_attribute_double, time=', time 
 
  maxlevels = Read_Chombo_Attribute_Int("max_level", hid_root_group_id)
  nvars = i_varcount
  NrVars=nvars
  !        dxFix(1:nDim)=dxFix(1:nDim)*2d0**(i_maxlevel-InterpLevel)!product(ia_ref_ratios(InterpLevel-1):i_maxlevel)
  ! allocate array to store grid box / level information
  IF(ALLOCATED(gridLevels)) DEALLOCATE(gridLevels)
  IF(ALLOCATED(gridBounds)) DEALLOCATE(gridBounds)
  ALLOCATE(gridBounds(nGrids,6),gridLevels(nGrids),STAT=i_Err)
  IF (i_Err /= 0) THEN
     PRINT *,'!!! ERROR: failed to allocate gridBounds arrays'
     STOP
  END IF

  ! Initialize the mxFix array defaults.
  ngrid=0

  IF (MaintainAuxArrays) THEN

     IF (i_space_dim == 3) THEN
        ALLOCATE(auxFix(RootmX(1)+1,RootmX(2)+1,rootmx(3)+1,1,3), stat=i_err)
     ELSE
        ALLOCATE(auxFix(RootmX(1)+1,rootmx(2)+1,1,1,2), stat=i_err)
     END IF
     auxFix=0
  END IF

  ! Allocate the fixed-grid array using the problem domain bounds of the 
  ! highest refinement level.
  IF (i_space_dim == 3) THEN
     ALLOCATE(qFix(rootmx(1),rootmx(2),rootmx(3),i_varcount))
  ELSE
!     write(*,*) 'allocated qfix', rootmx, i_varcount
     ALLOCATE(qFix(rootmx(1),rootmx(2),1,i_varcount))

  END IF

  ! Loop over levels, writing each level's data to the fixed grid.
  !	DO i_level = 0, i_maxlevel

  DO i_level = 0, InterpLevel
     WRITE(s_level_group, '(A6,I1.1)') 'level_', i_level

     IF (i_level <= i_maxlevel) THEN
        ! Open level i_level group.
        CALL h5gopen_f(hid_root_group_id, s_level_group, hid_level_group_id, i_err)

        ! Die if unable to open group.
        IF (i_err < 0) THEN
           PRINT *, "FixedGridFromChombo error:  unable to open ", TRIM(s_level_group), " group."
           STOP
        END IF

        ! Multiply all elements in ia_ref_ratios together, except for the ones on coarser
        ! levels than this one.
        la_ref_mask(i_level) = .FALSE.
        i_level_ref_ratio = PRODUCT(ia_ref_ratios(i_level+1:InterpLevel))
        !		i_level_ref_ratio = PRODUCT(ia_ref_ratios, 1, la_ref_mask)

        ! Get the level's area (for statistical purposes).
        !		levelArea(i_level) = GetLevelArea(i_level, hid_root_group_id)

        ! Read in the box data and the grid data.
        dx(1:3)=1.d0

        ! check for anisotropic attribute
        !                   CALL h5aopen_name_f(hid_level_group_id, "anisotropic" , hid_attribute_id, i_err)
        !                   IF(i_err==0) THEN
        dx(1:3) = Read_Chombo_Attribute_FloatVector("anisotropic", hid_level_group_id)
        !                   END IF
!        call printbits(dx(1))
!        call printbits(1d0)
        dx(1:3) = dx(1:3)*Read_Chombo_Attribute_Double("dx", hid_level_group_id)
        !                   CALL h5aclose_f(hid_attribute_id, i_err)

        CALL Read_Chombo_Dataset_Box("boxes", hid_level_group_id, ia_box_data)
        CALL Read_Chombo_Dataset_Float("data:datatype=0", hid_level_group_id, dbla_data)

        IF(MaintainAuxArrays) CALL GET_MHD_DATA(i_frame,hid_level_group_id,i_level,i_space_dim,dx)
!        IF (lMHD .AND. i_space_dim>1 .AND. nDim>1)  CALL GET_MHD_DATA(i_frame,hid_level_group_id,i_level,i_space_dim,dx)

        i_index = 0

        ngrid=0
        DO i_grid = 1, SIZE(ia_box_data, 2)
           ! populate grid box information for drawing hotboxes
           ngrid=ngrid+1
           gridLevels(nGrid)=i_level
           gridBounds(nGrid,1:3)=(REAL(ia_box_data(1:3,i_grid)  ))*dx(1:3)+XLower(1:3)
           gridBounds(nGrid,4:6)=(REAL(ia_box_data(4:6,i_grid)+1))*dx(1:3)+XLower(1:3)


           DO i_variable = 1, i_varcount
              DO k = ia_box_data(3, i_grid) + 1, ia_box_data(6, i_grid) + 1
                 DO j = ia_box_data(2, i_grid) + 1, ia_box_data(5, i_grid) + 1
                    DO i = ia_box_data(1, i_grid) + 1, ia_box_data(4, i_grid) + 1


                       i_index = i_index + 1
                       
                       qFix(i,j,k, i_variable)=dbla_data(i_index)
                    END DO 				! END DO i
                 END DO 				! END DO j
              END DO 				! END DO k
           END DO 			! END DO i_variable
        END DO			! END DO i_grid

        DEALLOCATE(ia_box_data, stat=i_err)
        DEALLOCATE(dbla_data, stat=i_err)

        IF (i_err /= 0) THEN
           PRINT *, "FixedGridFromChombo error:  unable to deallocate level ", i_level, " arrays."
           STOP
        END IF

        ! Close the level group.
        CALL h5gclose_f(hid_level_group_id, i_err)

        IF (i_err < 0) THEN
           PRINT *, "FixedGridFromChombo error: unable to close group ", s_level_group, "."
           STOP
        END IF
     END IF
     IF (i_space_dim>1 .AND. i_level < InterpLevel) THEN
        CALL regrid(qfix,auxFix,i_space_dim,mx)
        mx(1:nDim)=mx(1:nDim)*2
        dx(1:nDim)=dx(1:nDim)/2

     END IF
  END DO	! End loop over i_level
  dxfix(1:3)=dx
  mxfix(1:3)=mx
  ! Free memory from refinement ratio arrays.
  DEALLOCATE(ia_ref_ratios, stat=i_err)
  DEALLOCATE(la_ref_mask, stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "FixedGridFromChombo error: unable to deallocate refinement ratio arrays."
     STOP
  END IF

  ! Close basic HDF5 groups.
  CALL h5gclose_f(hid_chombo_global_group_id, i_err)
  !	CALL h5gclose_f(hid_max_level_group_id, i_err)
  CALL h5gclose_f(hid_root_group_id, i_err)

  ! Die if unable to close groups.
  IF (i_err < 0) THEN
     PRINT *, "FixedGridFromChombo error: unable to close one or more HDF5 groups."
     STOP
  END IF

  ! Close data file.
  CALL h5fclose_f(hid_file_id, i_err)

  ! Die if unable to close file.
  IF (i_err < 0) THEN
     PRINT *, "FixedGridFromChombo error: unable to close HDF5 file."
     STOP
  END IF

  PRINT *, "FixedGridFromChombo(", i_frame, ") done."

  gi_fixed%Xlower=Xlower
  gi_fixed%Xupper=Xupper
  gi_fixed%dx=dxfix
  gi_fixed%Time=time
  gi_fixed%mX=mxFix
  gi_fixed%nVars=i_varcount
  gi_fixed%nDim=i_space_dim
  gi_fixed%q=>qfix
  gi_fixed%frame=i_frame
  nrVars=gi_fixed%nVars
  nDim=gi_fixed%nDim
  IF (MaintainAuxArrays) THEN
     qfix(:,:,:,iBx)=.5*(auxfix(1:mxFix(1),1:mxFix(2),1:mxFix(3),1,1)+auxfix(2:mxFix(1)+1,1:mxFix(2),1:mxFix(3),1,1))
     qfix(:,:,:,iBy)=.5*(auxfix(1:mxFix(1),1:mxFix(2),1:mxFix(3),1,2)+auxfix(1:mxFix(1),2:mxFix(2)+1,1:mxFix(3),1,2))
     If (i_space_dim > 2) qfix(:,:,:,iBz)=.5*(auxfix(1:mxFix(1),1:mxFix(2),1:mxFix(3),1,3)+auxfix(1:mxFix(1),1:mxFix(2),2:mxFix(3)+1,1,3))
  END IF
!  Print *, mxfix, dxfix, xlower, xupper, i_varcount, i_space_dim, nrVars, nDim, shape(qfix)
END SUBROUTINE FixedGridFromChombo

SUBROUTINE GET_MHD_DATA(i_frame, hid_level_group_id, i_level,i_space_dim,dx)
  USE HDF5
  ! Input parameter declarations.
  INTEGER, INTENT(IN) :: i_frame, i_level			! Frame number.

  ! Variable Declarations
  INTEGER :: i_err

  ! Chombo group handles.
  INTEGER(HID_T) :: hid_level_group_id
  INTEGER(HID_T) :: hid_attribute_id

  CHARACTER(LEN=7) :: s_level_group

  INTEGER :: i, j, k
  INTEGER, POINTER, DIMENSION(:,:) :: ia_box_data
  INTEGER, DIMENSION(6) :: ia_prob_domain
  REAL(KIND=xPrec), POINTER, DIMENSION(:) :: dbla_data
  INTEGER,DIMENSION(3)::aux_offset

  INTEGER (HSIZE_T) :: i_index
  INTEGER :: i_grid
  INTEGER :: i_variable
  INTEGER :: i_gridcount
  INTEGER :: i_varcount
  INTEGER :: i_space_dim
  INTEGER :: ngrid
  INTEGER(HSIZE_T)::i_offset
  REAL(KIND=xPrec)::dx(3)


  ! Read in the box data and the grid data.
  CALL Read_Chombo_Dataset_Box("boxes", hid_level_group_id, ia_box_data)

  CALL Read_Chombo_Dataset_Float("MHD_data", hid_level_group_id, dbla_data)

  i_index = 0                   
  DO i_grid = 1, SIZE(ia_box_data, 2)
     ! populate grid box information for drawing hotboxes

     DO i_variable = 1, i_space_dim
        aux_offset=0
        aux_offset(i_variable)=1
        DO k = ia_box_data(3, i_grid) + 1, ia_box_data(6, i_grid) + 1+aux_offset(3)
           DO j = ia_box_data(2, i_grid) + 1, ia_box_data(5, i_grid) + 1+aux_offset(2)
              DO i = ia_box_data(1, i_grid) + 1, ia_box_data(4, i_grid) + 1+aux_offset(1) 

                 i_index = i_index + 1

                 ! There will be no refinement ratio operations on the z-axis if the 
                 ! plot is two-dimensional.
                 IF (i_space_dim == 3) THEN		
                    auxFix(i,j,k,1,i_variable) = dbla_data(i_index)

                 ELSE
                    auxFix(i,j,k,1,i_variable) = dbla_data(i_index)
                 END IF

              END DO 				! END DO i
           END DO 				! END DO j
        END DO 				! END DO k
     END DO 			! END DO i_variable
  END DO			! END DO i_grid

  DEALLOCATE(ia_box_data, stat=i_err)
  DEALLOCATE(dbla_data, stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "GetMHDData Error:  unable to deallocate level ", i_level, " arrays."
     STOP
  END IF

END SUBROUTINE GET_MHD_DATA
!	Created: 2007-08-13 by Brandon D. Shroyer
!	
SUBROUTINE Read_Slab_From_Dataset_Float(s_name, hid_group_id, dbla_data, i_offset)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name								! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id							! Group identifier.
  REAL(KIND(0.d0)), POINTER, DIMENSION(:), INTENT(OUT) :: dbla_data	! Data input
  INTEGER(HSIZE_T), INTENT(IN) :: i_offset

  INTEGER(HID_T) :: hid_dataset_id				! Dataset handle.
  INTEGER(HID_T) :: hid_dataspace_id              ! Dataspace handle.
  INTEGER(HID_T) :: hid_memspace_id               ! Dataspace handle for hyperslab reference.
  INTEGER(HSIZE_T) :: i_dataset_size				! Size of dataset to be retrieved.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims		! Array to hold the size of the input.
  INTEGER :: i_err
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_data_offset
  INTEGER(HSIZE_T), DIMENSION(1), PARAMETER :: IA_NO_OFFSET = (/0/)
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_slab_size


  ! Set the size of the hyperslab.
  ia_slab_size(1) = SIZE(dbla_data)

  ! Store the size of the offset in a dimension array.
  ia_data_offset(1) = i_offset

  ! Open the dataset in the Chombo file.
  CALL h5dopen_f(hid_group_id, s_name, hid_dataset_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ": unable to open ", &
          "dataset ", s_name, "."	
     STOP
  END IF


  ! Retrieve the total dataspace for the dataset.
  CALL h5dget_space_f(hid_dataset_id, hid_dataspace_id, i_err)

  ! Die if unable to retrieve the total dataspace.
  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ": unabe to retrieve ", &
          "dataspace for dataset ", s_name, "."
     STOP
  END IF


  CALL h5sselect_hyperslab_f(hid_dataspace_id, H5S_SELECT_SET_F, &
       ia_data_offset, ia_slab_size, i_err) 

  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ": Unable to select dataspace ", &
          "hyperslab from dataset ", s_name, "."
     STOP
  END IF


  ! Create dataspace for the upcoming hyperslab.
  CALL h5screate_simple_f(I_DATASET_RANK, ia_slab_size, hid_memspace_id, i_err)

  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ":  Unable to create hyperslab ", & 
          "dataspace for dataset ", s_name, "."
     STOP
  END IF


  ! Read data into newly-allocated array.
  CALL h5dread_f(hid_dataset_id, H5T_NATIVE_DOUBLE, &
       dbla_data, ia_dims, i_err, hid_memspace_id, hid_dataspace_id)

  ! Die if unable to read data in from dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ": unable to read data ", &
          "from dataset ", s_name, "."
     STOP 
  END IF


  ! Close HDF5 handles.
  CALL h5sclose_f(hid_memspace_id, i_err)
  CALL h5sclose_f(hid_dataspace_id, i_err)
  CALL h5dclose_f(hid_dataset_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Slab_From_Dataset_Float error ", i_err, ": unable to close ", &
          "one or more dataspace/dataset handles ", s_name, "."
     STOP
  END IF

END SUBROUTINE Read_Slab_From_Dataset_Float

SUBROUTINE Read_Chombo_Dataset_Float(s_name, hid_group_id, dbla_data)

  !    IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name						! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id					! Group identifier.
  REAL(KIND(0.d0)), POINTER, DIMENSION(:) :: dbla_data		! Data input

  INTEGER(HID_T) :: hid_dataset_id				! Dataset handle.
  INTEGER(HSIZE_T) :: i_dataset_size				! Size of dataset to be retrieved.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims		! Array to hold the size of the input.
  INTEGER :: i_err

  ! Open the dataset in the Chombo file.
  CALL h5dopen_f(hid_group_id, s_name, hid_dataset_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Float error ", i_err, ": unable to open ", &
          "dataset ", s_name, "."	
     STOP
  END IF

  ! Obtain the dataset size.
  CALL h5dget_storage_size_f(hid_dataset_id, i_dataset_size, i_err)
  i_dataset_size=1280*320*320*9
  
  ! Die if unable to obtain dataset size.
  IF (i_err < 0) THEN

     PRINT *, 'Read_Chombo_Dataset_Float ', i_err, ': unable to get size of ', &
          'dataset ', s_name, ' .'
     STOP
  END IF

  ! Allocate a 1D array large enough to hold the s_name dataset.
  ALLOCATE(dbla_data(i_dataset_size), stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "Read_Chombo_Dataset_Float error", i_err, ": unable to create ", &
          "dataset array for dataset ", s_name, "."
     STOP
  END IF

  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = i_dataset_size

  ! Read data into newly-allocated array.
  CALL h5dread_f(hid_dataset_id, H5T_NATIVE_DOUBLE, &
       dbla_data, ia_dims, H5S_ALL_F, H5S_ALL_F, H5P_DEFAULT_F)

  ! Die if unable to read data in from dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Float error ", i_err, ": unable to read data ", &
          "from dataset ", s_name, "."
     STOP 
  END IF

  ! Close dataset.
  CALL h5dclose_f(hid_dataset_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Float error ", i_err, ": unable to close ", &
          "dataset ", s_name, "."
     STOP
  END IF

END SUBROUTINE Read_Chombo_Dataset_Float


!	Created: 2007-10-22 by Brandon D. Shroyer
!	
SUBROUTINE Read_Chombo_Dataset_Box(s_name, hid_group_id, ia_data)

  USE HDF5

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name						! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id					! Group identifier.
  INTEGER, POINTER, DIMENSION(:,:) :: ia_data					! Data input

  INTEGER(HID_T) :: hid_dataset_id				! Dataset handle.
  INTEGER(HID_T) :: hid_dataspace_id				! Dataspace handle.
  INTEGER(HID_T) :: hid_type_id					! Type ID handle.

  INTEGER(HSIZE_T) :: i_dims						! Number of grids (dimensions of dataspace).
  INTEGER(SIZE_T) :: i_datatype_size				! Size of Box data type.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims		! Array to hold the size of the input.
  INTEGER :: i_err
  INTEGER(HSIZE_T) :: i_storage_size
  INTEGER :: i, j,k

  ! Open the dataset in the Chombo file.
  CALL h5dopen_f(hid_group_id, s_name, hid_dataset_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box error ", i_err, ": unable to open ", &
          "dataset ", s_name, "."	
     STOP
  END IF

  ! Retrieve dataset's dataspace.
  CALL h5dget_space_f(hid_dataset_id, hid_dataspace_id, i_err)

  ! Die if unable to open dataspace.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box (", TRIM(s_name), "):  unable to open dataspace."
     STOP
  END IF


  ! Retrieve the dataspace's dimensions.  The boxes dataset is of the Chombo box datatype,
  ! so each element in the boxes dataspace corresponds to a grid.
  !	CALL h5sget_simple_extent_ndims_f(hid_dataspace_id, i_dims, i_err)
  !	CALL h5sget_select_npoints_f(hid_dataspace_id, i_dims, i_err)
  CALL h5dget_storage_size_f(hid_dataset_id, i_dims, i_err)

  ! Die if unable to retrieve dimensions.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box (", TRIM(s_name), "):  unable to retrieve dataset dimensions."
     STOP
  END IF


  ! Get the dataset's type (Box, in this case).
  CALL h5dget_type_f(hid_dataset_id, hid_type_id, i_err)

  ! Die if unable to retrieve type.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box (", TRIM(s_name), "):  unable to retrieve box data type."
     STOP
  END IF

  ! Get Box datatype size.
  CALL h5tget_size_f(hid_type_id, i_datatype_size, i_err)

  ! Die if unable to obtain datatype size.
  IF (i_err < 0) THEN
     PRINT *, 'Read_Chombo_Dataset_Box (", TRIM(s_name), "): unable to get size of ', &
          'Box datatype.'
     STOP
  END IF

  ! Allocate a 2D array large enough to hold the s_name box dataset.
  !	ALLOCATE(ia_data(i_datatype_size, i_dims), stat=i_err)
  ALLOCATE(ia_data(6, i_dims / i_datatype_size), stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box (", s_name, ") error", i_err, ": unable to create ", &
          "dataset array for dataset ", s_name, "."
     STOP
  END IF

  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = i_dims

  ! Read data into newly-allocated array.
  CALL h5dread_f(hid_dataset_id, hid_box_id, &
       ia_data, ia_dims, H5S_ALL_F, H5S_ALL_F, H5P_DEFAULT_F)


  !write(*,*) 'just read boxes', ia_data, s_name
  !IF (lConvert) THEN
     !may need to loop over ia_data
  !   do k=1,size(ia_data,2)
  !      do j=1,size(ia_data,1)
  !         ia_data(j,k)=convertfrombluegene(ia_data(j,k))
  !      END do
  !   end do
  !END IF
  ! Die if unable to read data in from dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box error ", i_err, ": unable to read data ", &
          "from dataset ", s_name, "."
     STOP 
  END IF

  ! Close dataset.
  CALL h5tclose_f(hid_type_id, i_err)
  CALL h5sclose_f(hid_dataspace_id, i_err)
  CALL h5dclose_f(hid_dataset_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Dataset_Box error ", i_err, ": unable to close ", &
          s_name, " HDF5 handles."
     STOP
  END IF

  !	PRINT *, "x_val(box data) = ", SIZE(ia_data, 1), ", y_val(box data) = ", SIZE(ia_data, 2), "."

END SUBROUTINE Read_Chombo_Dataset_Box

!	Created: 2007-08-27 by Brandon D. Shroyer
!	
INTEGER FUNCTION Read_Chombo_Attribute_Int(s_name, hid_group_id)

  !    IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name		! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id	! Group identifier.

  ! Variable declarations
  INTEGER(HID_T) :: hid_attribute_id		! Dataset handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims	! Array to hold the size of the input.
  INTEGER :: i_output				! Used to store output from HDF5 API.
  INTEGER :: i_err


  ! Open the attribute in the Chombo file.
  CALL h5aopen_name_f(hid_group_id, s_name, hid_attribute_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Int error ", i_err, ": unable to open ", &
          "attribute ", s_name, "."	
     STOP
  END IF


  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = 1

  ! Read data into newly-allocated array.
  CALL h5aread_f(hid_attribute_id, H5T_NATIVE_INTEGER, &
       i_output, ia_dims, i_err)

  ! Die if unable to read data in from dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Int error ", i_err, ": unable to read value ", &
          "from attribute ", s_name, "."
     STOP 
  END IF

  ! Close dataset.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Int error ", i_err, ": unable to close ", &
          "attribute ", s_name, "."
     STOP
  END IF

  Read_Chombo_Attribute_Int = i_output

END FUNCTION Read_Chombo_Attribute_Int


!	Created: 2007-10-16 by Brandon D. Shroyer
!	
REAL(KIND=qprec) FUNCTION Read_Chombo_Attribute_Double(s_name, hid_group_id)

  !    IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name		! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id	! Group identifier.

  ! Variable declarations
  INTEGER(HID_T) :: hid_attribute_id		! Dataset handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims	! Array to hold the size of the input.
  REAL(KIND=xPrec) :: dbl_output				! Used to store output from HDF5 API.
  INTEGER :: i_err


  ! Open the attribute in the Chombo file.
  CALL h5aopen_name_f(hid_group_id, s_name, hid_attribute_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Double error ", i_err, ": unable to open ", &
          "attribute ", s_name, "."	
     STOP
  END IF


  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = 1

  ! Read data into newly-allocated array.
  CALL h5aread_f(hid_attribute_id, H5T_NATIVE_DOUBLE, &
       dbl_output, ia_dims, i_err)

  ! Die if unable to read data in from dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Double error ", i_err, ": unable to read value ", &
          "from attribute ", s_name, "."
     STOP 
  END IF

  ! Close dataset.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Double error ", i_err, ": unable to close ", &
          "attribute ", s_name, "."
     STOP
  END IF

  Read_Chombo_Attribute_Double = dbl_output

END FUNCTION Read_Chombo_Attribute_Double



! Created 2007-10-19 by Brandon D. Shroyer
! Returns a FloatVector (3-element double array) from a Chombo file.
FUNCTION Read_Chombo_Attribute_FloatVector(s_name, hid_group_id) RESULT(dbla_floatvector)

  !	IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name		! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id	! Group identifier.

  ! Variable Declarations
  INTEGER(HID_T) :: hid_attribute_id		! Attribute handle.
  INTEGER(HID_T) :: hid_type_id			! Attribute type handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims	! Array to hold the size of the input.
  REAL(KIND=xPrec), DIMENSION(:), POINTER :: dbla_floatvector
  INTEGER :: i_err, j


  ! Open the attribute in the Chombo file.
  CALL h5aopen_name_f(hid_group_id, s_name, hid_attribute_id, i_err)

  ! Die if unable to open attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_FloatVector error ", i_err, ": unable to open ", &
          "attribute ", s_name, "."	
     STOP
  END IF


  ! Get the type of the attribute we just opened.
  CALL h5aget_type_f(hid_attribute_id, hid_type_id, i_err)

  ! Die if unable to retrieve attribute type.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_FloatVector error ", i_err, ": unable to open ", &
          "get type of attribute ", s_name, "."
     STOP
  END IF


  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = 1

  ! Allocate a 3-element array.
  ALLOCATE(dbla_floatvector(3), stat = i_err)

  ! Die if memory allocation fails.
  IF (i_err /= 0) THEN
     PRINT *, "Read_Chombo_Attribute_FloatVector(", s_name, &
          ") error: memory allocation failed."
     STOP
  END IF

  dbla_floatvector = 0.d0


  ! Read data into newly-allocated array.
  CALL h5aread_f(hid_attribute_id, hid_type_id, dbla_floatvector, ia_dims, i_err)

  ! Die if unable to read data in from attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_FloatVector error ", i_err, ": unable to read value ", &
          "from attribute ", s_name, "."
     STOP 
  END IF

!  IF (lConvert) THEN
!     do j=1,size(dbla_floatvector)
!        dbla_floatvector(j)=convertfrombluegenefloat(dbla_floatvector(j))
!     END do
!  END IF


  ! Close type and attribute handles.
  CALL h5tclose_f(hid_type_id, i_err)
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Die if unable to close handles.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_FloatVector error ", i_err, ": unable to close ", &
          "one or more HD5 objects."
     STOP
  END IF

END FUNCTION Read_Chombo_Attribute_FloatVector


! Created 2007-10-19 by Brandon D. Shroyer
! Returns a Box (6-element integer array) from a Chombo file.
FUNCTION Read_Chombo_Attribute_Box(s_name, hid_group_id) RESULT(ia_box)

  !	IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name		! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id	! Group identifier.

  ! Variable Declarations
  INTEGER(HID_T) :: hid_attribute_id		! Attribute handle.
  INTEGER(HID_T) :: hid_type_id			! Attribute type handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims	! Array to hold the size of the input.
  INTEGER, DIMENSION(6) :: ia_box, temp
  INTEGER :: i_err,j

  
  ! Open the attribute in the Chombo file.
  CALL h5aopen_name_f(hid_group_id, TRIM(s_name), hid_attribute_id, i_err)
  
 ! write(*,*) 'attribute box hid_group_id=', hid_group_id
 ! write(*,*) 'attribute box hid_attribute_id=', hid_attribute_id

  ! Die if unable to open attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box error ", i_err, ": unable to open ", &
          "attribute ", s_name, "."	
     STOP
  END IF

!  print *,' before hid type=', hid_type_id
  ! Get the type of the attribute we just opened.
  CALL h5aget_type_f(hid_attribute_id, hid_type_id, i_err)

!  print *, 'hid type=',hid_type_id
! print *, 'hid box=', hid_box_id
 ! hid_type_id=hid_box_id

  ! Die if unable to retrieve attribute type.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box error ", i_err, ": unable to open ", &
          "get type of attribute ", s_name, "."
     STOP
  END IF


  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = 1


  ! Die if memory allocation fails.
  IF (i_err /= 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box(", s_name, &
          ") error: memory allocation failed."
     STOP
  END IF
  
  ia_box = 0.d0
  
  !print *, "hid_box_id=", hid_box_id 

  ! Read data into newly-allocated array.
  CALL h5aread_f(hid_attribute_id, hid_box_id, ia_box, ia_dims, i_err)
 
  !write(*,*) 'just read attribute box', hid_attribute_id, hid_box_id,  s_name
  !write(*,*) 'ia_box', ia_box 

! IF (lConvert) THEN
!     do j=1,size(ia_box)
!        ia_box(j)=convertfrombluegene(ia_box(j))
!     END do
!  END IF
!  CALL h5aread_f(hid_attribute_id, hid_type_id, temp, ia_dims, i_err)

!  CALL native_4byte_real( temp, ia_box )
  

  ! Die if unable to read data in from attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box error ", i_err, ": unable to read value ", &
          "from attribute ", s_name, "."
     STOP 
  END IF


  ! Close type and attribute handles.
  CALL h5tclose_f(hid_type_id, i_err)
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Die if unable to close handles.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box error ", i_err, ": unable to close ", &
          "one or more HD5 objects."
     STOP
  END IF

END FUNCTION Read_Chombo_Attribute_Box


! Created 2012-10-29 by Baowei Liu
! Returns a Bound (3-element float array) from a Chombo file.
FUNCTION Read_Chombo_Attribute_Bound(s_name, hid_group_id) RESULT(ia_bound)

  !	IMPLICIT NONE

  USE HDF5


  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name		! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id	! Group identifier.

  ! Variable Declarations
  INTEGER(HID_T) :: hid_attribute_id		! Attribute handle.
  INTEGER(HID_T) :: hid_type_id			! Attribute type handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dims	! Array to hold the size of the input.
  REAL(KIND=xPrec), DIMENSION(3) :: ia_bound, temp
  INTEGER :: i_err,j

  
  ! Open the attribute in the Chombo file.
  CALL h5aopen_name_f(hid_group_id, TRIM(s_name), hid_attribute_id, i_err)
  
 ! write(*,*) 'attribute bound hid_group_id=', hid_group_id
 ! write(*,*) 'attribute bound hid_attribute_id=', hid_attribute_id

  ! Die if unable to open attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Bound error ", i_err, ": unable to open ", &
          "attribute ", s_name, "."	
     STOP
  END IF

!  print *,' before hid type=', hid_type_id
  ! Get the type of the attribute we just opened.
  CALL h5aget_type_f(hid_attribute_id, hid_type_id, i_err)

!  print *, 'hid type=',hid_type_id
! print *, 'hid bound=', hid_bound_id
 ! hid_type_id=hid_bound_id

  ! Die if unable to retrieve attribute type.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Box error ", i_err, ": unable to open ", &
          "get type of attribute ", s_name, "."
     STOP
  END IF


  ! Initializing dimension array with size of chombo array.
  ia_dims(1) = 1


  ! Die if memory allocation fails.
  IF (i_err /= 0) THEN
     PRINT *, "Read_Chombo_Attribute_Bound(", s_name, &
          ") error: memory allocation failed."
     STOP
  END IF
  
  ia_bound = 0.d0
  
  !print *, "hid_bound_id=", hid_bound_id 

  ! Read data into newly-allocated array.
  CALL h5aread_f(hid_attribute_id, hid_bound_id, ia_bound, ia_dims, i_err)
 
  !write(*,*) 'just read attribute bound', hid_attribute_id, hid_bound_id,  s_name
  !write(*,*) 'ia_bound', ia_bound 

  ! Die if unable to read data in from attribute.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Bound error ", i_err, ": unable to read value ", &
          "from attribute ", s_name, "."
     STOP 
  END IF


  ! Close type and attribute handles.
  CALL h5tclose_f(hid_type_id, i_err)
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Die if unable to close handles.
  IF (i_err < 0) THEN
     PRINT *, "Read_Chombo_Attribute_Bound error ", i_err, ": unable to close ", &
          "one or more HD5 objects."
     STOP
  END IF

END FUNCTION Read_Chombo_Attribute_Bound

!FUNCTION convertfrombluegene(a)
!  INTEGER :: a, convertfrombluegene,i,j
!  convertfrombluegene=0
!  DO i=0, 3
!     DO j=0,7
!        if (btest(a,8*i+j))  convertfrombluegene=ibset(convertfrombluegene, 8*(3-i)+j)
!     END DO
!  END DO
!END FUNCTION convertfrombluegene


!FUNCTION convertfrombluegenefloat(a)
!  REAL(8) :: a, convertfrombluegenefloat
!  INTEGER(8) :: ai, converti
!  INTEGER :: i,j
!  converti=0
!  convertfrombluegenefloat=0
!  ai=transfer(a,ai)
!  write(*,*) 'beginning conversion'
!  CALL printbits(a)
!  CALL printbitsint(ai)
!  DO i=0, 8
!     DO j=0,7
!        if (btest(ai,8*i+j))  converti=ibset(converti, 8*(7-i)+j)
!     END DO
!  END DO
!  convertfrombluegenefloat=transfer(converti,convertfrombluegenefloat)
!  CALL printbitsint(converti)
!  CALL printbits(convertfrombluegenefloat)

!END FUNCTION convertfrombluegenefloat


SUBROUTINE PrintBits(a)
  REAL(8) :: a
  INTEGER(8) :: ai
  INTEGER :: i
  ai=transfer(a,ai)
  write(*,'(E24.16,A3)', advance='no') a, ' = '
  do i=0,bit_size(ai)-1
     write(*, '(l1)', advance='no') btest(ai,i)
  END dO
  write(*,*) 
END SUBROUTINE PrintBits


SUBROUTINE PrintBitsInt(ai)
  INTEGER(8) :: ai
  INTEGER :: i
  write(*,'(I10,A3)', advance='no') ai, ' = '
  do i=0,bit_size(ai)-1
     write(*, '(l1)', advance='no') btest(ai,i)
  END dO
  write(*,*) 
END SUBROUTINE PrintBitsInt



! Created 2007-10-19 by Brandon D. Shroyer
! Input:  The handle of an HDF5 root group.
! Output: The number of grids in the HDF5 file's frame, expressed as an integer.
INTEGER FUNCTION GetNumberOfGrids(hid_root_group_id)

  !	IMPLICIT NONE

  USE HDF5


  ! Input declarations
  INTEGER(HID_T) :: hid_root_group_id

  ! Variable declarations
  INTEGER :: i_level
  INTEGER :: i_num_levels
  CHARACTER(LEN=7) :: s_level_group

  INTEGER(HID_T) :: hid_level_group_id
  INTEGER(HID_T) :: hid_dataset_id
  INTEGER(HID_T) :: hid_dataspace_id

  INTEGER :: i_dims
  INTEGER :: i_cumulative_grids
  INTEGER :: i_err


  i_cumulative_grids = 0

  ! Retrieve the number of levels in the file.
  i_num_levels = Read_Chombo_Attribute_Int("num_levels", hid_root_group_id)

  DO i_level = 0, i_num_levels - 1

     i_dims = 0	!  Clear out i_dims counter.

     WRITE(s_level_group, '(A6,I1.1)') 'level_', i_level

     ! Open level i_level group.
     CALL h5gopen_f(hid_root_group_id, s_level_group, hid_level_group_id, i_err)

     ! Die if unable to open group.
     IF (i_err < 0) THEN
        PRINT *, "GetNumberOfGrids error:  unable to open ", TRIM(s_level_group), " group."
        STOP
     END IF


     ! Open boxes dataset for this level.
     CALL h5dopen_f(hid_level_group_id, "boxes", hid_dataset_id, i_err)

     ! Die if unable to open dataset.
     IF (i_err < 0) THEN
        PRINT *, "GetNumberOfGrids error:  unable to open ", TRIM(s_level_group), " dataset."
        STOP
     END IF


     ! Retrieve dataset's dataspace.
     CALL h5dget_space_f(hid_dataset_id, hid_dataspace_id, i_err)

     ! Die if unable to open dataspace.
     IF (i_err < 0) THEN
        PRINT *, "GetNumberOfGrids error:  unable to open ", TRIM(s_level_group), " dataspace."
        STOP
     END IF

     ! Retrieve the dataspace's dimensions.  The boxes dataset is of the Chombo box datatype,
     ! so each element in the boxes dataspace corresponds to a grid.
     CALL h5sget_simple_extent_ndims_f(hid_dataspace_id, i_dims, i_err)

     ! Die if unable to retrieve dimensions.
     IF (i_err < 0) THEN
        PRINT *, "GetNumberOfGrids error:  unable to retrieve ", TRIM(s_level_group), &
             " dimensions."
        STOP
     END IF


     ! Add this level's grids to the cumulative total.
     i_cumulative_grids = i_cumulative_grids + i_dims


     ! Close handles associated with this level.
     CALL h5sclose_f(hid_dataspace_id, i_err)
     CALL h5dclose_f(hid_dataset_id, i_err)
     CALL h5gclose_f(hid_level_group_id, i_err)

     ! Die if unable to close handles.
     IF (i_err < 0) THEN
        PRINT *, "GetNumberOfGrids error: unable to close handles for ", &
             TRIM(s_level_group), "."
        STOP
     END IF

  END DO	! End loop over i_level

  GetNumberOfGrids = i_cumulative_grids

END FUNCTION GetNumberOfGrids


REAL(KIND=xPrec) FUNCTION GetLevelArea(i_level, hid_root_group_id)

  USE HDF5

  ! Input declarations
  INTEGER :: i_level
  INTEGER(HID_T) :: hid_root_group_id

  ! Variable declarations
  CHARACTER(LEN=7) :: s_level_group
  REAL(KIND=xPrec) :: dbl_dx
  INTEGER :: i_dims
  INTEGER :: i_grid
  INTEGER, DIMENSION(1) :: ia_dims
  REAL(KIND=xPrec) :: dbl_cumulative_area

  INTEGER(HID_T) :: hid_level_group_id
  INTEGER(HID_T) :: hid_dataset_id
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_type_id

  INTEGER, DIMENSION(:,:), POINTER :: ia_bounds

  INTEGER :: i_err


  WRITE(s_level_group, '(A6,I1.1)') 'level_', i_level

  ! Open level i_level group.
  CALL h5gopen_f(hid_root_group_id, s_level_group, hid_level_group_id, i_err)

  ! Die if unable to open group.
  IF (i_err < 0) THEN
     PRINT *, "GetLevelArea error:  unable to open ", TRIM(s_level_group), " group."
     STOP
  END IF

  ! Retrieve the dx value for this level.
  dbl_dx = Read_Chombo_Attribute_Double("dx", hid_level_group_id)

  ! Clear accumulator.
  dbl_cumulative_area = 0.d0

  ! Read in the box data.
  CALL Read_Chombo_Dataset_Box("boxes", hid_level_group_id, ia_bounds)	

  PRINT *,'index2 of ia_bounds',SIZE(ia_bounds, 2)
  i_dims=size(ia_bounds,2)
  ! Add up the areas/volumes of each grid on the level.
  DO i_grid = 1, i_dims

     ! If the z-axis has a width of 0, then assume the problem is two-dimensional.
     IF (ia_bounds(6, i_grid) - ia_bounds(3, i_grid) == 0) THEN
        dbl_cumulative_area = dbl_cumulative_area + &
             (ia_bounds(4, i_grid) - ia_bounds(1, i_grid) + 1) * dbl_dx * &
             (ia_bounds(5, i_grid) - ia_bounds(2, i_grid) + 1) * dbl_dx
     ELSE
        dbl_cumulative_area = dbl_cumulative_area + &
             (ia_bounds(4, i_grid) - ia_bounds(1, i_grid) + 1) * dbl_dx * &
             (ia_bounds(5, i_grid) - ia_bounds(2, i_grid) + 1) * dbl_dx * &
             (ia_bounds(6, i_grid) - ia_bounds(3, i_grid) + 1) * dbl_dx
     END IF

  END DO	!	End loop over dimensions.

  DEALLOCATE(ia_bounds, stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "GetLevelArea (", s_level_group, ") error: unable to deallocate ia_bounds array."
     STOP
  END IF

  ! Close handles associated with this level.
  CALL h5gclose_f(hid_level_group_id, i_err)

  ! Die if unable to close handles.
  IF (i_err < 0) THEN
     PRINT *, "GetLevelArea (", s_level_group, ") error: unable to close handles for ", &
          TRIM(s_level_group), "."
     STOP
  END IF

END FUNCTION GetLevelArea

![BDS][20080220]:  Returns true or false, depending upon whether a specific data element
!                  is found in the given group of a chombo file.
LOGICAL FUNCTION MemberExists(hid_group_id, s_groupname, s_searchname)

  USE HDF5

  INTEGER(HID_T) :: hid_group_id   ! File or group identifier 
  CHARACTER(LEN=*), INTENT(IN) :: s_groupname  ! Name of the group 
  CHARACTER(LEN=*), INTENT(IN) :: s_searchname ! Name of the target item

  INTEGER :: i_membercount                    ! Number of members in the group
  INTEGER :: i_err                            ! Error code 

  INTEGER :: i                                    ! index variable for looping.
  INTEGER :: i_objtype                            ! dummy for holding object type.
  CHARACTER(LEN=25) :: s_compname  ! used to retrieve object names.


  MemberExists = .FALSE.

  ! Retrieves an index list of all members in group hid_group_id.
  CALL h5gn_members_f(hid_group_id, s_groupname, i_membercount, i_err)


  ! Loop over the group member indices and return true if the target group is among them.
  IF (i_membercount > 0) THEN

     loop1: DO i = 1, i_membercount

        CALL h5gget_obj_info_idx_f(hid_group_id, s_groupname, i, s_compname, i_objtype, i_err)           

        IF (s_searchname == s_compname) THEN
           MemberExists = .TRUE.
           EXIT loop1
        END IF

     END DO loop1
  END IF

END FUNCTION MemberExists

SUBROUTINE PrintGenericJetsetFile(n_frame, dbl_gamma)

  ! Input declarations.
  INTEGER :: n_frame
  REAL(KIND=qPrec) :: dbl_gamma

  ! Variable declarations
  CHARACTER(LEN=21) :: s_filename
  INTEGER :: i_err
  INTEGER :: i, j
  REAL(KIND=qPrec) :: pressure

  ! Open the file.
  WRITE(s_filename,'(A2,A3,A7,I5.5,A4)')'./',outdir,'/jetset',n_frame,'.dat'
  OPEN(UNIT=313,FILE=s_filename,STATUS='REPLACE',IOSTAT=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "PrintGenericJetsetFile error: unable to open file ", s_filename, "."
     STOP
  END IF

  DO j = 1, SIZE(pFix, 2)
     DO i = 1, SIZE(pFix, 1)

        ! P = (gamma - 1)*(total energy - kinetic - energy due to magnetic field)
        pressure =  (dbl_gamma - 1d0) * (qFix(i,j,1,5) - &
             0.5 * DOT_PRODUCT(qFix(i,j,1,2:4), qFix(i,j,1,2:4)) / qFix(i,j,1,1) - &
             0.5 * DOT_PRODUCT(qFix(i,j,1,6:8), qFix(i,j,1,6:8)))

        WRITE(313, "(i8, ' ', i8, ' ', ' ', f15.6, ' ', f15.6, ' ', f15.6, ' ', f15.6, ' ', f15.6, ' ', f15.6, ' ', f15.6, ' ', f15.6)")   i, j, &
             qFix(i,j,1,1), &
             qFix(i,j,1,2)/qFix(i,j,1,1), &
             qFix(i,j,1,3)/qFix(i,j,1,1), &
             qFix(i,j,1,4)/qFix(i,j,1,1), &
             qFix(i,j,1,6), &
             qFix(i,j,1,7), &
             qFix(i,j,1,8), &
             pressure

     END DO
  END DO

  CLOSE(UNIT=313, IOSTAT = i_err)

  IF (i_err /= 0) THEN
     PRINT *, "PrintGenericJetsetFile error: unable to close file ", s_filename, "."
     STOP
  END IF

END SUBROUTINE PrintGenericJetsetFile

SUBROUTINE PrintASCIicolumnFILE(n_frame,Emiss)
  !Martin HE, 22 Sept. 2010
  !  Produces an ASCII column file with data from the simulations to be
  !read into Shape* in order to produce emission and pv diagrams. This
  !routine takes 2d data and revolves it around the symmetry axis using random
  !positions. Then, it prints 3d data into a .dat file which will next be 
  !read into the 3d module of Shape*.
  !
  !* http://bufadora.astrosen.unam.mx/shape/

  ! Input declarations.
  INTEGER :: n_frame
  REAL(KIND=qPrec),DIMENSION(:,:,:),OPTIONAL :: Emiss

  ! Variable declarations
  REAL(KIND=qPrec) :: harvest, theta, densUPPERlim
  CHARACTER(LEN=99) :: s_filename
  INTEGER :: i_err, i, j, k, sim, gridBIT
  !
  REAL(KIND=qPrec),DIMENSION(:),ALLOCATABLE :: rhobuf
  REAL(KIND=qPrec),DIMENSION(:,:),ALLOCATABLE :: velbuf,xyzbuf

  INTEGER :: n,npoints,samplingtype,ijk(3),samplingmethod
  INTEGER :: nbins,minrholoc(1),ind
  INTEGER,DIMENSION(:),ALLOCATABLE :: popbins,cumulpopbins
  REAL(KIND=qPrec),DIMENSION(:),ALLOCATABLE :: valbins
  REAL(KIND=qPrec) :: minrho,x,x0,x1,x2,y0,y1,y2

  !   The following are arbitrary parameters used to
  !produce a uniform-ish distribution of paticles from
  !a total of ~60,000 of them. More particles would make 
  !Shape slow.
  REAL, PARAMETER :: arb1=40., arb2=3., arb3=2., arb4=5.

  IF(PRODUCT(mxFix)==0d0) THEN
     mxFix(1)=SIZE(qFix,1)
     mxFix(2)=SIZE(qFix,2)
     mxFix(3)=SIZE(qFix,3)
  END IF

  ! *** 2D data ***
  IF(nDim==2) THEN
     print*,''
     print*,'This routine assumes that the chombo files contain 2D data.'
     print*,'Where is your data given at:'
     print*,''
     print*,'       y (or r in cylind. coords.)'
     print*,'       ^                                    1= x>0 and y>0 (top right quarter)'
     print*,'       |                                    2= y>0 (upper half)'
     print*,'       |                                    3= x>0 (right half)'
     print*,'-x ----|----> x (or z in cylind. coords.)   4= all space'
     print*,'       |'
     print*,'       |'
     read*,gridBIT
     print*,''
     print*,'nScale and lScale must be in cgs units. Otherwise, please '
     print*,'edit the unit factors in bear2fixIO.f90, PrintASCIicolumnFILE,'
     print*,'lines ~ 1960-2000. Note that Shape reads, and plots, distances'
     print*,'in parsecs, velocities in km/s and the emission, i.e. (number'
     print*,'desity)^2, in (m)^(-3*2).'

     ! Open the file.
     WRITE(s_filename,'(A,I5.5,A)')'./'//outdir//'/SHAPE',n_frame,'.dat'
     OPEN(UNIT=987,FILE=TRIM(s_filename),STATUS='REPLACE',IOSTAT=i_err)

     IF (i_err /= 0) THEN
        PRINT *, "PrintASCIIcolumnFILE error: unable to open file ", s_filename, "."
        STOP
     END IF

     !   The following is a general template that will produce data for good
     !quality shape maps from the chombo files. You may want to include IF
     !statements to print some relevant parts of the data only, e.g. shocks
     !or highly compressed regions.

     DO     j = 1, SIZE(qFix, 2),INT(arb2*SIZE(qFix, 2)/arb1)
        DO i = 1, SIZE(qFix, 1),INT(arb2*SIZE(qFix, 1)/arb1)

           !actual cells
           WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ', ES14.7)") &
                lScale/3.0856e16*(Xlower(1)+REAL(i-0.5)*dxFix(1)) ,& !x, parsecs
                lScale/3.0856e16*(Xlower(2)+REAL(j-0.5)*dxFix(2)) ,& !y, parsecs
                0.0000000                                         ,& !z, parsecs
                velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)          ,& !vx, km/s
                velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)          ,& !vy, km/s
                0.0000000                                         ,& !vz, km/s  
                (1.e6*nScale*qFix(i,j,1,1))**2                       !dens**2, (1/m/m/m)^2

           !-x reflection of cell
           IF (gridBIT==1   .or.   gridBIT==3) THEN
              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ', ES14.7)") &
                   -lScale/3.0856e16*(Xlower(1)+REAL(i-0.5)*dxFix(1)) ,&
                   lScale/3.0856e16*(Xlower(2)+REAL(j-0.5)*dxFix(2)) ,&
                   0.0000000                                          ,&
                   -velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)          ,&
                   velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)           ,&
                   0.0000000                                          ,&
                   (1.e6*nScale*qFix(i,j,1,1))**2
           END IF

           !-y reflection of cell
           IF (gridBIT==1   .or.   gridBIT==2) THEN
              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ', ES14.7)") &
                   lScale/3.0856e16*(Xlower(1)+REAL(i-0.5)*dxFix(1)) ,&
                   -lScale/3.0856e16*(Xlower(2)+REAL(j-0.5)*dxFix(2)) ,&
                   0.0000000                                          ,&
                   velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)          ,&
                   -velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)          ,&
                   0.0000000                                          ,&
                   (1.e6*nScale*qFix(i,j,1,1))**2
           END IF

           !-x,-y reflection of cell
           IF (gridBIT==1) THEN
              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ', ES14.7)") &
                   -lScale/3.0856e16*(Xlower(1)+REAL(i-0.5)*dxFix(1)) ,&
                   -lScale/3.0856e16*(Xlower(2)+REAL(j-0.5)*dxFix(2)) ,&
                   0.0000000                                          ,&
                   -velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)          ,&
                   -velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)          ,&
                   0.0000000                                          ,&
                   (1.e6*nScale*qFix(i,j,1,1))**2
           END IF

           !copies of cell along axisymmetric rings
           DO k = 1, abs(INT(   2.*3.141592*arb1/(arb3*SIZE(qFix, 2))*&
                (Xlower(2)+REAL(j-0.5)*dxFix(2))  ) )

              call random_number(harvest)
              theta=harvest*2.*3.1415926535

              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                   lScale/3.0856e16*(Xlower(1) + REAL(i - 0.5) * dxFix(1))            ,&
                   lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*cos(theta) ,&
                   lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*sin(theta) ,&
                   velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)                           ,&
                   velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*cos(theta)                ,&
                   velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*sin(theta)                ,&
                   (1.e6*nScale*qFix(i,j,1,1))**2

              !-x reflection  
              IF (gridBIT==1   .or.   gridBIT==3) THEN
                 WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                      -lScale/3.0856e16*(Xlower(1) + REAL(i -0.5) * dxFix(1))            ,&
                      lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*cos(theta) ,&
                      lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*sin(theta) ,&
                      -velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)                          ,&
                      velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*cos(theta)                ,&
                      velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*sin(theta)                ,&
                      (1.e6*nScale*qFix(i,j,1,1))**2
              END IF

              !-y reflection  
              IF (gridBIT==1   .or.   gridBIT==2) THEN
                 WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                      lScale/3.0856e16*(Xlower(1) + REAL(i -0.5) * dxFix(1))            ,&
                      -lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*cos(theta) ,&
                      lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*sin(theta) ,&
                      velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)                          ,&
                      -velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*cos(theta)                ,&
                      velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*sin(theta)                ,&
                      (1.e6*nScale*qFix(i,j,1,1))**2
              END IF

              !-x,-y reflection  
              IF (gridBIT==1) THEN
                 WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                      -lScale/3.0856e16*(Xlower(1) + REAL(i -0.5) * dxFix(1))            ,&
                      -lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*cos(theta) ,&
                      lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2))*sin(theta) ,&
                      -velScale/1e5*qFix(i,j,1,2)/qFix(i,j,1,1)                          ,&
                      -velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*cos(theta)                ,&
                      velScale/1e5*qFix(i,j,1,3)/qFix(i,j,1,1)*sin(theta)                ,&
                      (1.e6*nScale*qFix(i,j,1,1))**2
              END IF
           END DO!k
        END DO!i
     END DO!j
     CLOSE(UNIT=987, IOSTAT = i_err)

     IF (i_err /= 0) THEN
        PRINT *, " PrintASCIIcolumnFILE error: unable to close file ", s_filename, "."
        STOP
     END IF

  ! *** 3D data ***
  ELSE
     WRITE(*,'(A,I12.3,A)',ADVANCE='NO') ' Size of fixed grid is ',SIZE(qFix(:,:,:,1)),'. How many points do you want? '
     READ*,npoints
     IF(npoints > SIZE(qFix(:,:,:,1))) THEN
        PRINT*,' Error: I cant subsample the space. Choose a lower value. '
        STOP
     END IF
     PRINT*,' Select sampling method: '
     PRINT*,'    0:  arbitrary (not yet implemented) '
     PRINT*,'    1:  weighted by density '
!!$     IF(lEmissShape) &
!!$     PRINT*,'    2: weighted by brightest emission (assumed to be Halpha or Sii) '
     WRITE(*,'(A)',ADVANCE='NO') ' ? '
     READ*,samplingType


     ! Open the file(s)
     WRITE(s_filename,'(A,I5.5,A)')'./'//outdir//'/SHAPE',n_frame,'.dat'
     OPEN(UNIT=987,FILE=TRIM(s_filename),STATUS='REPLACE',IOSTAT=i_err)
!!$     IF(lEmissShape) THEN
!!$     ! NOTE: assumes Halpha and Sii 
!!$        WRITE(s_filename,'(A,I5.5,A)')'./'//outdir//'/SHAPE_Halpha',n_frame,'.dat'
!!$        OPEN(UNIT=988,FILE=TRIM(s_filename),STATUS='REPLACE',IOSTAT=i_err)
!!$        WRITE(s_filename,'(A,I5.5,A)')'./'//outdir//'/SHAPE_Sii',n_frame,'.dat'
!!$        OPEN(UNIT=989,FILE=TRIM(s_filename),STATUS='REPLACE',IOSTAT=i_err)
!!$     END IF


     SELECT CASE(samplingtype)
     CASE(0)
     ! arbitrary, similar to the above

        PRINT*,' I told you, not yet implemented.'
        STOP

!!$        IF(npoints /= PRODUCT(mxFix)) THEN
!!$           stride = ...
!!$        ELSE
!!$           stride(:)=1
!!$        END IF
!!$
!!$        DO k=1,mxFix(3),stride(3)
!!$        DO j=1,mxFix(2),stride(2)
!!$        DO i=1,mxFix(1),stride(1)
!!$           
!!$        END DO
!!$        END DO
!!$        END DO

     CASE(1)
     ! density-weighted
        
        samplingmethod=1

        SELECT CASE(samplingmethod)
        CASE(0)
        ! Simplest but slowest: iterate through qFix using MAXLOC
           ALLOCATE(rhobuf(npoints),velbuf(npoints,3),xyzbuf(npoints,3))

           DO n=1,npoints
              IF(MOD(n,10)==0) PRINT*,n
              ! loop over density, successively pulling out MAXLOC and then zeroing it after writing
              ijk(:)=MAXLOC(qFix(:,:,:,1),qFix(:,:,:,1)>0d0)
              i=ijk(1);j=ijk(2);k=ijk(3)

              xyzbuf(n,1)=REAL(i)
              xyzbuf(n,2)=REAL(j)
              xyzbuf(n,3)=REAL(k)
              !           xyzbuf(n,:)=(/ (Xlower(1) + REAL(i - 0.5) * dxFix(1)) &
              !                      , (Xlower(2) + REAL(j - 0.5) * dxFix(2)) &
              !                      , (Xlower(3) + REAL(k - 0.5) * dxFix(3)) /)
              velbuf(n,1:3)=qFix(i,j,k,2:4)
              rhobuf(n)=qFix(i,j,k,1)
              !           velbuf(n,:)=(/ qFix(i,j,k,2)/rhobuf(n) &
              !                        , qFix(i,j,k,3)/rhobuf(n) &
              !                        , qFix(i,j,k,4)/rhobuf(n) /)
              qFix(i,j,k,1)=0d0
           END DO
           FORALL(i=1:3)
              velbuf(:,i)=velbuf(:,i) * velScale/1e5 * rhobuf(:)**(-1)
              xyzbuf(:,i)=xlower(i) + (xyzbuf(:,i) -0.5)*dxFix(i)
           END FORALL

           xyzbuf=xyzbuf * 1e2*lScale/3.0856e16
           rhobuf=rhobuf**2*1e6*nScale
           PRINT*,'FInished constructing. Writing...'

           DO n=1,npoints
              IF(MOD(n,10)==0) PRINT*,n
              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                   xyzbuf(n,1),xyzbuf(n,2),xyzbuf(n,3) &
                   , velbuf(n,1),velbuf(n,2),velbuf(n,3) &
                   , rhobuf(n)
!!$                1e2*lScale/3.0856e16*(Xlower(1) + REAL(i - 0.5) * dxFix(1))  ,&
!!$                1e2*lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2)) ,&
!!$                1e2*lScale/3.0856e16*(Xlower(3) + REAL(k - 0.5) * dxFix(3)) ,&
!!$                velScale/1e5*qFix(i,j,k,2)/qFix(i,j,k,1)                ,&
!!$                velScale/1e5*qFix(i,j,k,3)/qFix(i,j,k,1)                ,&
!!$                velScale/1e5*qFix(i,j,k,4)/qFix(i,j,k,1)                ,&
!!$                (1.e6*nScale*qFix(i,j,k,1))**2
           END DO
           DEALLOCATE(rhobuf,velbuf,xyzbuf)

        CASE(1)
        ! Bin densities and populate file based on them
           PRINT*,'beginning binning'
           nbins=50
           ALLOCATE(popbins(nbins),cumulpopbins(nbins),valbins(nbins))

           valbins = (/( (REAL(n)-1d0)/(REAL(nbins)-1d0)*MAXVAL(qFix(:,:,:,1)) + (REAL(nbins)-REAL(n))/(REAL(nbins)-1d0)*MINVAL(qFix(:,:,:,1)), n=1,nbins )/)
!           valbins = LOG10(valbins)
           PRINT*,'valbins populated'
           PRINT*,valbins

           popbins(1)=0d0
           DO n=2,nbins
              IF(MOD(n,10)==0) PRINT*,n
              popbins(n) = COUNT( (qFix(:,:,:,1).gt.valbins(n-1) .AND. qFix(:,:,:,1).le.valbins(n)) )
           END DO

           PRINT*,' getting cumulative populations'
           cumulpopbins(nbins)=0d0
           DO n=nbins-1,1,-1
              cumulpopbins(n) = popbins(n+1) + cumulpopbins(n+1)
           END DO

           OPEN(34,FILE='densitybins.dat')
           DO n=1,nbins
              WRITE(34,'(I12,I12,F12.6)')popbins(n),cumulpopbins(n),valbins(n)
           END DO


           ! Based on npoints, find out what the density threshold will be
           ! Get lowest-density bin which cumulatively has more than npoints
           minrholoc(:)=MINLOC(cumulpopbins-npoints,cumulpopbins-npoints.gt.0)
           ind=minrholoc(1)

           PRINT*,'using bins, the number of points to get is',COUNT( qFix(:,:,:,1).ge.valbins(ind) )

           ! Parabolic extrapolate valbins to get minrho
           x=REAL(npoints)
           x0=REAL(cumulpopbins(ind+1)); x1=REAL(cumulpopbins(ind)); x2=REAL(cumulpopbins(ind-1))
           y0=valbins(ind+1); y1=valbins(ind); y2=valbins(ind-1)
           minrho = y0*((x-x1)*(x-x2)) / ((x0-x1)*(x0-x2)) &
                  + y1*((x-x0)*(x-x2)) / ((x1-x0)*(x1-x2)) &
                  + y2*((x-x0)*(x-x1)) / ((x2-x0)*(x2-x1))
           
           PRINT*,'x0,x1,x2,x',x0,x1,x2,x
           PRINT*,'y0,y1,y2',y0,y1,y2
           PRINT*,'bin is,has',valbins(ind),cumulpopbins(ind)
           PRINT*,'extrap value is',minrho
           WRITE(34,*)'x0,x1,x2,x',x0,x1,x2,x
           WRITE(34,*)'y0,y1,y2',y0,y1,y2
           WRITE(34,*)'bin is,has',valbins(ind),cumulpopbins(ind)
           WRITE(34,*)'extrap value is',minrho

!           minrho = 1d1**minrho

           PRINT*,'number of points to get is',COUNT( qFix(:,:,:,1).ge.minrho )
           PRINT*,'mxfix is',mxFix
           PRINT*,'shape of qfix is',SHAPE(qFix(:,:,:,1))
           
           ! Writing to SHAPE file
           n=0
           DO k=1,mxFix(3)
           DO j=1,mxFix(2)
           DO i=1,mxFix(1)
              IF(qFix(i,j,k,1).lt.minrho) CYCLE

              n=n+1
!              IF( MOD( NINT( REAL(n)/REAL(npoints)*1d2), 10) ==0) PRINT*,n
              WRITE(987, "(ES14.7, ' ', ES14.7, ' ', ES14.7, ' ', f25.16, ' ', f25.16, ' ', f25.16, ' ',ES14.7)") &
                   1e2*lScale/3.0856e16*(Xlower(1) + REAL(i - 0.5) * dxFix(1))  ,&
                   1e2*lScale/3.0856e16*(Xlower(2) + REAL(j - 0.5) * dxFix(2)) ,&
                   1e2*lScale/3.0856e16*(Xlower(3) + REAL(k - 0.5) * dxFix(3)) ,&
                   velScale/1e5*qFix(i,j,k,2)/qFix(i,j,k,1)                ,&
                   velScale/1e5*qFix(i,j,k,3)/qFix(i,j,k,1)                ,&
                   velScale/1e5*qFix(i,j,k,4)/qFix(i,j,k,1)                ,&
                   (1.e6*nScale*qFix(i,j,k,1))**2
           END DO
           END DO
           END DO
           PRINT*,'output ',n,' points'
           WRITE(34,*)'output ',n,' points'
           CLOSE(34)
        END SELECT

!           IF(lEmissShape) THEN
!           END IF

!           ! zero out value for next iteration
!           qFix(i,j,k,1) = 0d0
!        END DO
!!$
!!$     CASE(2)
!!$     ! emission-weighted
     CASE DEFAULT
     END SELECT

     CLOSE(987)

  END IF
END SUBROUTINE PrintASCIIcolumnFILE


SUBROUTINE WriteToKHEnergyFile

  ! Variable declarations
  CHARACTER(LEN=21) :: s_filename
  INTEGER :: i_err
  INTEGER :: i, j
  REAL(KIND=qPrec) :: Ey


  ! Open the file.
  WRITE(s_filename,'(A2,A3,A14)')'./',outdir,'/kh_energy.dat'
  OPEN(UNIT=314,FILE=s_filename,STATUS='OLD',POSITION='append',IOSTAT=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "WriteToKHEnergyFile error: unable to open file ", s_filename, "."
     STOP
  END IF

  Ey = 0

  DO j = 1, SIZE(qFix, 2)
     DO i = 1, SIZE(qFix, 1)

        Ey = Ey + 0.5 * (qFix(i,j,1,3)**2)/qFix(i,j,1,1)

     END DO
  END DO

  WRITE(314, "(f25.16, ' ', f25.16)")  Time , Ey


  CLOSE(UNIT=314, IOSTAT = i_err)

  IF (i_err /= 0) THEN
     PRINT *, "WriteToKHEnergyFile error: unable to close file ", s_filename, "."
     STOP
  END IF

END SUBROUTINE WriteToKHEnergyFile


SUBROUTINE PrintRadShockFlowFile(n_frame, dbl_gamma)

  ! Input declarations.
  INTEGER :: n_frame
  REAL(KIND=qPrec) :: dbl_gamma

  ! Variable declarations
  CHARACTER(LEN=21) :: s_filename
  INTEGER :: i_err
  INTEGER :: i, j
  REAL(KIND=qPrec) :: pressure


  ! Open the file.
  WRITE(s_filename,'(A2,A3,A9,I5.5,A4)')'./',outdir,'/radshock',n_frame,'.dat'
  OPEN(UNIT=313,FILE=s_filename,STATUS='REPLACE',IOSTAT=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "PrintRadShockFlowFile error: unable to open file ", s_filename, "."
     STOP
  END IF

  DO i = 1, SIZE(pFix, 1)

     ! P = (gamma - 1)*(total energy - kinetic energy)
     pressure =  (dbl_gamma - 1) * (qFix(i,j,1,5) - &
          0.5 * (qFix(i,j,1,2)**2) / qFix(i,j,1,1))

     WRITE(313, "(i6, ' ', ' ', f25.16, ' ', f25.16, ' ', f25.16)")   i, &
          qFix(i,j,1,1), &
          qFix(i,j,1,2)/qFix(i,j,1,1), &
          pressure

  END DO

  CLOSE(UNIT=313, IOSTAT = i_err)

  IF (i_err /= 0) THEN
     PRINT *, "PrintRadShockFlowFile error: unable to close file ", s_filename, "."
     STOP
  END IF

END SUBROUTINE PrintRadShockFlowFile


SUBROUTINE WriteToRadShockPositionFile(dbl_gamma)

  ! Input declarations
  REAL(KIND=qPrec) :: dbl_gamma

  ! Variable declarations
  CHARACTER(LEN=21) :: s_filename
  INTEGER :: i_err
  INTEGER :: i, j
  REAL(KIND=qPrec) :: x_shock

  REAL(KIND=qPrec), PARAMETER :: DBL_SHOCK_THRESHOLD = 1.0E-6


  ! Open the file.
  WRITE(s_filename,'(A2,A3,A17)')'./',outdir,'/radshock_pos.dat'
  OPEN(UNIT=314,FILE=s_filename,STATUS='OLD',POSITION='append',IOSTAT=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "WriteToRadShockPositionFile error: unable to open file ", s_filename, "."
     STOP
  END IF

  x_shock = 0

  shock_loop: DO i = SIZE(qFix, 1), 2, -1

     IF (ABS(GetPressure(i, dbl_gamma) - GetPressure(i-1, dbl_gamma)) > DBL_SHOCK_THRESHOLD) THEN

        x_shock = Xlower(1) + REAL(i - 0.5) * dxFix(1)
        EXIT shock_loop

     END IF

  END DO shock_loop

  WRITE(314, "(f25.16, ' ', f25.16)")  Time , x_shock


  CLOSE(UNIT=314, IOSTAT = i_err)

  IF (i_err /= 0) THEN
     PRINT *, "WriteToRadShockPositionFile error: unable to close file ", s_filename, "."
     STOP
  END IF

END SUBROUTINE WriteToRadShockPositionFile

!!!!! WARNING:  Only works for JetSet Radiative Shock and other 1D problems.
REAL(KIND=qPrec) FUNCTION GetPressure(i, dbl_gamma)

  INTEGER :: i
  REAL(KIND=qPrec) :: dbl_gamma

  GetPressure = (dbl_gamma - 1) * (qFix(i,1,1,5) - 0.5 * (qFix(i,1,1,2)**2) / qFix(i,1,1,1))

END FUNCTION GetPressure

SUBROUTINE CreateBoxType
  INTEGER(size_t) :: hid_boxid_type_size
  INTEGER(size_t) :: i_h5int_type_size
  INTEGER :: i_err
  INTEGER(size_t) :: i_offset


  CALL h5open_f(i_err)
  !write(*,*) H5T_NATIVE_INTEGER
  !write(*,*) H5T_NATIVE_DOUBLE
  !write(*,*) H5T_COMPOUND_F
  ! Get size of box type.
  CALL h5tget_size_f(H5T_NATIVE_INTEGER, i_h5int_type_size, i_err)
  ! Set box data size
  hid_boxid_type_size = 6 * i_h5int_type_size

  ! Die if unable to get type size of new box data type.
  IF (i_err < 0) THEN 
     PRINT *,'CreateBoxType error ', i_err, ': call to h5tget_size_f(H5T_NATIVE_INTEGER)', &
          'failed.'
     STOP
  END IF
  ! create hid_box_id
  CALL h5tcreate_f(H5T_COMPOUND_F, hid_boxid_type_size, hid_box_id, i_err)

  print *, "createBox", hid_box_id

  ! Die if unable to create new box data id.
  IF (i_err < 0) THEN 
     PRINT *,'CreateBoxType error ', i_err, ': call to h5tcreate_f(hid_box_id)', &
          'failed.'
     STOP
  END IF

  ! Clear i_offset
  i_offset = 0

  ! Insert the six boundary-descriptors into the box data type.
  CALL h5tinsert_f(hid_box_id, "lo_i", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_box_id, "lo_j", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_box_id, "lo_k", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_box_id, "hi_i", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_box_id, "hi_j", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_box_id, "hi_k", i_offset, H5T_NATIVE_INTEGER, i_err)

  ! Die if unable to insert all three directional fields into hid_box_id.
  IF (i_err < 0) THEN 
     PRINT *,'CreateBoxType error ', i_err, &
          ': one or more calls to h5tinsert_f(hid_box_id) failed.'
     STOP
  END IF
END SUBROUTINE CreateBoxType

!Create Bound Type to read the XLower, Xupper correctly
SUBROUTINE CreateBoundType
  INTEGER(size_t) :: hid_boundid_type_size
  INTEGER(size_t) :: i_h5float_type_size
  INTEGER :: i_err
  INTEGER(size_t) :: i_offset


  CALL h5open_f(i_err)
  !write(*,*) H5T_NATIVE_INTEGER
  !write(*,*) H5T_NATIVE_DOUBLE
  !write(*,*) H5T_COMPOUND_F
  ! Get size of box type.
  CALL h5tget_size_f(H5T_NATIVE_DOUBLE, i_h5float_type_size, i_err)
  ! Set box data size
  hid_boundid_type_size = 3 * i_h5float_type_size

  ! Die if unable to get type size of new box data type.
  IF (i_err < 0) THEN 
     PRINT *,'CreateBoundType error ', i_err, ': call to h5tget_size_f(H5T_NATIVE_DOUBLE)', &
          'failed.'
     STOP
  END IF
  ! create hid_box_id
  CALL h5tcreate_f(H5T_COMPOUND_F, hid_boundid_type_size, hid_bound_id, i_err)

  print *, "createBound", hid_bound_id

  ! Die if unable to create new box data id.
  IF (i_err < 0) THEN 
     PRINT *,'CreateBoundType error ', i_err, ': call to h5tcreate_f(hid_bound_id)', &
          'failed.'
     STOP
  END IF

  ! Clear i_offset
  i_offset = 0

  ! Insert the six boundary-descriptors into the box data type.
  CALL h5tinsert_f(hid_bound_id, "x", i_offset, H5T_NATIVE_DOUBLE, i_err)
  i_offset = i_offset + i_h5float_type_size

  CALL h5tinsert_f(hid_bound_id, "y", i_offset, H5T_NATIVE_DOUBLE, i_err)
  i_offset = i_offset + i_h5float_type_size

  CALL h5tinsert_f(hid_bound_id, "z", i_offset, H5T_NATIVE_DOUBLE, i_err)
  i_offset = i_offset + i_h5float_type_size


  ! Die if unable to insert all three directional fields into hid_box_id.
  IF (i_err < 0) THEN 
     PRINT *,'BreateBoundType error ', i_err, &
          ': one or more calls to h5tinsert_f(hid_bound_id) failed.'
     STOP
  END IF

END SUBROUTINE CreateBoundType


SUBROUTINE MakeChomboFile(frame)
  IMPLICIT NONE
  ! input declarations
  LOGICAL :: lConservative
  ! Variable declarations
  CHARACTER(LEN=30) :: s_filename
  INTEGER :: frame
  INTEGER(HID_T) :: hid_property_id
  INTEGER(HID_T) :: hid_file_id

  INTEGER(HID_T) :: hid_group_id
  INTEGER(HID_T) :: hid_chombo_global_group_id
  INTEGER(HID_T) :: hid_level_group_id

  INTEGER :: i_err

  INTEGER(size_t) :: i_h5int_type_size
  INTEGER(size_t) :: i_h5dbl_type_size
  INTEGER(size_t) :: i_offset

  INTEGER(size_t) :: hid_intvectid_type_size
  INTEGER(size_t) :: hid_floatvectid_type_size

  INTEGER(HID_T) :: hid_intvect_id
  INTEGER(HID_T) :: hid_floatvect_id
!  INTEGER(HID_T) :: hid_box_id

  INTEGER(HID_T) :: hid_boxes_property_id
  INTEGER(HID_T) :: hid_boxes_dataspace_id
  INTEGER(HID_T) :: hid_boxes_dataset_id

  INTEGER(HID_T) :: hid_data_attributes_group_id

  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dataset_dims 	! Holds the size of the dataset.


  INTEGER :: i, j, k		! indices to use when looping along spatial dimensions.

  REAL(KIND = qPrec), ALLOCATABLE, DIMENSION(:) :: dbla_chombo_qvar

  INTEGER(HSIZE_T) :: i_index		! Used for any incidental loops along the way.
  INTEGER(HSIZE_T) :: i_mhd_index	! Used when looping through aux data.
  INTEGER :: i_level		! Tracks the level in several loops.
  INTEGER :: i_dimension	! Used in the loop that assembles the boxes data.
  INTEGER :: i_variable	! Used in the loop that assembles chombo data.

  INTEGER(SIZE_T) :: i_qvar_offset    ! Stores the cumulative offset of the chombo dataset
  ! hyperslab that we are writing to.

  INTEGER(SIZE_T) :: i_aux_offset    ! Stores the cumulative offset of the chombo dataset
  ! hyperslab that we are writing to.

  CHARACTER(LEN = I_MAX_CNAME_LENGTH), DIMENSION(0:I_MAX_COMPONENTS) :: sa_component_names
  CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_tracer
  CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_component_tag

  CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_level_name

  INTEGER, DIMENSION(6) :: ia_box_global		! array for holding box data.

  INTEGER :: i_level_node_index

  ! Variable declarations.

  REAL(KIND = qPrec) :: dbl_domain_offset

  INTEGER :: i_grid_offset
  INTEGER, ALLOCATABLE, DIMENSION(:,:) :: ia_box_data
  !    INTEGER, DIMENSION(6) :: ia_box_data
  INTEGER(SIZE_T) :: i_box_offset
  INTEGER(HSIZE_T) :: i_grid_data_size

  REAL(KIND = xPrec), DIMENSION(MaxDims) :: dbla_minLower
  REAL(KIND = xPrec), DIMENSION(MaxDims) :: dbla_maxUpper

  ! Statically-defined arrays of predetermined size that can be used
  ! to cache the lower bounds.
  REAL(KIND = xPrec), DIMENSION(MaxDims) :: dbla_lowerbound
  REAL(KIND = xPrec), DIMENSION(MaxDims) :: dbla_upperbound


  ! MHD (auxiliary) variable handling.
  INTEGER(HSIZE_T) :: i_aux_data_size
  INTEGER, DIMENSION(3) :: ia_aux_extension
  REAL(KIND = qPrec), ALLOCATABLE, DIMENSION(:) :: dbla_chombo_aux




!!!!!! START CLOCK !!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!

  i_level=0
  ! Initializes the HDF5 library.  This function is safe to call
  ! even if it has already been called.
  CALL h5open_f(i_err)

  ! Die if unable to initialize HDF5 library.	
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5open_f failed.'
     STOP
  END IF


  CALL h5pcreate_f(H5P_FILE_ACCESS_F, hid_property_id, i_err)

  ! Die if unable to create a new property list object for parallel file access.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5pcreate_f failed.'
     STOP
  END IF


  ! Create file.
  PRINT *, "Frame number: ", frame, "."

  WRITE(s_filename, '(A3,A14,I5.5,A4)') outdir,'/chombo_fixed_', frame, '.hdf'
  !	WRITE(s_filename, '(A10,I5.5,A4)') 'out/auxtst', frame, '.hdf'

  CALL h5fcreate_f(s_filename, H5F_ACC_TRUNC_F, hid_file_id, i_err, &
       access_prp = hid_property_id)

  ! Die if unable to create a new HDF5 file.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5fcreate_f failed.'
     STOP
  ELSE
     PRINT *, 'Chombo file ', TRIM(s_filename), ' created.'
  END IF


  ! Close property list, now that it's purpose (file creation) has been fulfilled.
  CALL h5pclose_f(hid_property_id, i_err)

  ! Die if unable to close property list.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': call to h5pclose_f(hid_property_id) failed.'
     STOP
  END IF


  ! Open the root group (this group is automatically created when the 
  ! HDF5 file is created).
  CALL h5gopen_f(hid_file_id, S_ROOT_GROUP, hid_group_id, i_err)

  ! Die if unable to close property list.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5gopen_f failed.'
     STOP
  END IF

  ! Get size of H5T_NATIVE_INTEGER data type.
  CALL h5tget_size_f(H5T_NATIVE_INTEGER, i_h5int_type_size, i_err)
  hid_intvectid_type_size = 3 * i_h5int_type_size

  ! Die if unable to get type size of H5T_NATIVE_INTEGER.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': call to h5tgetsize_f(H5T_NATIVE_INTEGER) failed.'
     STOP
  END IF

  ! Create the intvectid data type.
  CALL h5tcreate_f(H5T_COMPOUND_F, hid_intvectid_type_size, hid_intvect_id, i_err)

  ! Die if unable to create the data type.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5tcreate_f(intvectid)', &
          'failed.'
     STOP
  END IF

  ! Clear i_offset.
  i_offset = 0

  ! Insert the three directional fields <i, j, k> into the integer vector data type.

  CALL h5tinsert_f(hid_intvect_id, "intvecti", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_intvect_id, "intvectj", i_offset, H5T_NATIVE_INTEGER, i_err)
  i_offset = i_offset + i_h5int_type_size

  CALL h5tinsert_f(hid_intvect_id, "intvectk", i_offset, H5T_NATIVE_INTEGER, i_err)

  ! Die if unable to insert all three directional fields into intvectid.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': one or more calls to h5tinsert_f(intvectid) failed.'
     STOP
  END IF



  ! Get size of H5T_NATIVE_DOUBLE data type.
  CALL h5tget_size_f(H5T_NATIVE_DOUBLE, i_h5dbl_type_size, i_err)
  hid_floatvectid_type_size = 3 * i_h5dbl_type_size

  ! Die if unable to get type size of H5T_NATIVE_DOUBLE.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5tgetsize_f(H5T_NATIVE_DOUBLE) failed.'
     STOP
  END IF


  ! Create the floatvectid data type.
  CALL h5tcreate_f(H5T_COMPOUND_F, hid_floatvectid_type_size, hid_floatvect_id, i_err)
  i_offset = 0

  ! Die if unable to create the data type.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, ': call to h5tcreate_f(floatvectid)', &
          'failed.'
     STOP
  END IF

  ! Clear i_offset
  i_offset = 0


  ! Insert the three directional fields <i, j, k> into the floating-point vector
  ! data type.
  CALL h5tinsert_f(hid_floatvect_id, "x", i_offset, H5T_NATIVE_DOUBLE, i_err)
  i_offset = i_offset + i_h5dbl_type_size

  CALL h5tinsert_f(hid_floatvect_id, "y", i_offset, H5T_NATIVE_DOUBLE, i_err)
  i_offset = i_offset + i_h5dbl_type_size

  CALL h5tinsert_f(hid_floatvect_id, "z", i_offset, H5T_NATIVE_DOUBLE, i_err)

  ! Die if unable to insert all three directional fields into floatvectid.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': one or more calls to h5tinsert_f(floatvectid) failed.'
     STOP
  END IF


!!!!! Add required attributes for Chombo HDF specification.  New attributes are
!!!!! added to the root group usingAdd_ChomboHDF_Attribute() subroutine, which
!!!!! is described below.

  ! time attribute (uses Info's current time).
  CALL Add_Chombo_Attribute_Float("time", hid_group_id, time)
  !	CALL Add_Chombo_Attribute_Float("time", hid_group_id, 0.d54)

  ! iteration attribute (uses Info's current frame number).
  CALL Add_Chombo_Attribute_Int("iteration", hid_group_id, frame)

  ! num_levels attribute (Chombo format requires that this be 1 greater the maximum
  ! level--essentially so that it acts as a level-index.
  CALL Add_Chombo_Attribute_Int("num_levels", hid_group_id, 1)

  ! max_level attribute (use Info's current MaxLevel value).
  CALL Add_Chombo_Attribute_Int("max_level", hid_group_id, 0)

  ! num_components attribute (use Info's number of output variables).
  CALL Add_Chombo_Attribute_Int("num_components", hid_group_id, NrVars)


!!!!!	The scaling factors below are not part of the Chombo HDF specification, but they're
!!!!!	useful enough that we're including them anyway.

  ! Number density.
  CALL Add_Chombo_Attribute_Float("n_density_scale", hid_group_id, nScale)

  ! Mass density.
  CALL Add_Chombo_Attribute_Float("rho_scale", hid_group_id, rScale)

  CALL Add_Chombo_Attribute_Float("length_scale", hid_group_id, lScale)

  CALL Add_Chombo_Attribute_Float("velocity_scale", hid_group_id, velScale)

  CALL Add_Chombo_Attribute_Float("pressure_scale", hid_group_id, pScale)

  CALL Add_Chombo_Attribute_Float("temperature_scale", hid_group_id, TempScale)

  ! If lMHD is true, then the current problem uses MHD.
  DO i=1, NrCons
     IF (var_index(1) == i) sa_component_names(i-1)="rho"
     IF (var_index(2) == i) sa_component_names(i-1)="E"
     IF (var_index(3) == i) sa_component_names(i-1)="px"
     IF (var_index(4) == i) sa_component_names(i-1)="py"
     IF (var_index(5) == i) sa_component_names(i-1)="pz"
     IF (var_index(6) == i) sa_component_names(i-1)="Bx"
     IF (var_index(7) == i) sa_component_names(i-1)="By"
     IF (var_index(8) == i) sa_component_names(i-1)="Bz"
  END DO
  IF (iGravity==3) sa_component_names(NrCons)="iPhi"

  DO i_index = nTracerLo, nTracerHi
     WRITE(s_tracer, '(A6,I2.2)' ) 'Tracer', (i_index - nTracerLo)
     sa_component_names(i_index-1) = s_tracer
  END DO

  ! Add derived scaling factors.
  CALL Add_Chombo_Attribute_Float("px_scale", hid_group_id, rScale * velScale)
  CALL Add_Chombo_Attribute_Float("py_scale", hid_group_id, rScale * velScale)
  CALL Add_Chombo_Attribute_Float("pz_scale", hid_group_id, rScale * velScale)

  CALL Add_Chombo_Attribute_Float("E_scale", hid_group_id, pScale)

  CALL Add_Chombo_Attribute_Float("Bx_scale", hid_group_id, pScale)
  CALL Add_Chombo_Attribute_Float("By_scale", hid_group_id, pScale)
  CALL Add_Chombo_Attribute_Float("Bz_scale", hid_group_id, pScale)


  ! This part is separated out so that changing the assembly of the components
  ! is easy, but the addition of attributes may eventually have to be done
  ! inline in order to reduce overhead.
  !
  ! Construct new component_n attributes for each sa_component_name(n).
  DO i_index = 0, NrVars - 1

     ! Create component tag (attribute name).
     IF (i_index <= 9) THEN
        WRITE(s_component_tag,'(A10,I1.1)') 'component_', i_index
     ELSE
        WRITE(s_component_tag,'(A10,I2.2)') 'component_', i_index
     END IF

     CALL Add_Chombo_Attribute_String(s_component_tag, &
          hid_group_id, &
          TRIM(sa_component_names(i_index)))
  END DO

!!! End Assembling Component Array.

!!! Create Group Chombo_global And Its Attributes

  ! Create Chombo_global group.
  CALL h5gcreate_f(hid_file_id, "Chombo_global", hid_chombo_global_group_id, i_err)

  ! Die if unable to create this group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to create group "Chombo_global".'
     STOP
  END IF

  ! Add testReal attribute to Chombo_global group (value is a literal).
  CALL Add_Chombo_Attribute_Float("testReal", hid_chombo_global_group_id, &
       DBL_TEST_REAL_DAT) 

  ! Add SpaceDim attribute to Chombo_global group.  We can just use Info%nDim
  ! because we don't generally do 4D problems.
  CALL Add_Chombo_Attribute_Int("SpaceDim", hid_chombo_global_group_id, nDim)


  ! Close the Chombo_global group.
  CALL h5gclose_f(hid_chombo_global_group_id, i_err)

  ! Die if unable to create this group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to close group "Chombo_global".'
     STOP
  END IF

!!! End Create Chombo_global group.

  ! If the root level has only one node, then save some CPU time by just copying
  ! the upper and lower boundary arrays straight over to minLower and maxUpper.

  ! [BDS] 2007-10-18:  dbla_minLower and dbla_maxUpper are used later on for 
  !					 calculating the computational boundaries of the domain.
  !					 This is in contrast to dbla_lowerbound and dbla_upperbound,
  !					 which cache the same data in a format more easily stored
  !					 as a FloatVector in the Chombo file.
  dbla_minLower=0.d0
  dbla_maxUpper=0.d0
  dbla_minLower = Xlower
  dbla_maxUpper = Xupper

!!! Storing the domain boundaries in the Chombo file.

  ! Initialize the caching boundary arrays to 0.
  dbla_lowerbound = 0
  dbla_upperbound = 0

  ! Cache the domain boundaries in the FloatVector-able arrays.
  IF (nDim < 4) THEN
     dbla_lowerbound(1:nDim) = dbla_minLower(1:nDim)
     dbla_upperbound(1:nDim) = dbla_maxUpper(1:nDim)
  ELSE
     dbla_lowerbound(1:3) = dbla_minLower(1:3)
     dbla_upperbound(1:3) = dbla_maxUpper(1:3)
  END IF

  CALL Add_Chombo_Attribute_FloatVector("lower_bound", hid_floatvect_id, &
       hid_group_id, dbla_lowerbound)

  CALL Add_Chombo_Attribute_FloatVector("upper_bound", hid_floatvect_id, &
       hid_group_id, dbla_upperbound)



!!!	Assembling the group_level(n) groups.

  WRITE(s_level_name, '(A7)') 'level_0'

  ! Create Chombo_global group.
  CALL h5gcreate_f(hid_group_id, trim(s_level_name), hid_level_group_id, i_err)

  ! Die if unable to create this group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to create group "level_0".'
     STOP
  END IF

  ! Create the dx attribute
  CALL Add_Chombo_Attribute_Float("dx", hid_level_group_id, dXfix(1))

  ! Cretate the anisotropic attribute (optional chomobo attrib for anisotropic spacing)
  CALL Add_Chombo_Attribute_FloatVector("anisotropic", hid_floatvect_id, hid_level_group_id, dXfix(:)/dXfix(1))

  ! Create the ref_ratio attribute.
  CALL Add_Chombo_Attribute_Int("ref_ratio", hid_level_group_id, 2)


  ! loop over the three spatial dimensions to construct the box object..
  DO i_index = 1,3

     ! Set the lower-bounds of the box object.  This will be the 
     ! lower-bound value in mGlobal, minus one as an offset for 
     ! Chombo's field-space.

     ia_box_global(i_index) = 0

     ! ia_box_global = #cells along each dimension, adjusted by the 
     ! 		  refinement level.  This can be found by taking
     !		  the difference between the max and min values
     !		  for each dimension and dividing by dX(dim).

     IF (dXfix(i_index) /= zero) THEN
        dbl_domain_offset = (dbla_maxUpper(i_index) - &
             dbla_minLower(i_index)) / dxfix(i_index)
     ELSE
        dbl_domain_offset = 0
     END IF

     ! If there is a fractional part to the domain offset, then 
     ! add another cell to the global box to accomodate it.
     ia_box_global(i_index + 3) = MAX(FLOOR(dbl_domain_offset) - 1, 0)

  END DO


  ! Create prob_domain attribute using a box object.
  CALL Add_Chombo_Attribute_Box("prob_domain", hid_box_id, hid_level_group_id, &
       ia_box_global)


!!! CREATE BOXES DATA OBJECT

  ia_dataset_dims(1) = 1

  ! Create transfer property list for boxes dataset.
  CALL h5pcreate_f(H5P_DATASET_XFER_F, hid_boxes_property_id, i_err)

  CALL h5pset_preserve_f(hid_boxes_property_id, .TRUE., i_err)

  CALL h5screate_simple_f(I_DATASET_RANK, ia_dataset_dims, &
       hid_boxes_dataspace_id, i_err)

  CALL h5dcreate_f(hid_level_group_id, "boxes", hid_box_id, &
       hid_boxes_dataspace_id, hid_boxes_dataset_id, i_err)

  ! Die if unable to create this dataset.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to set create dataset for boxes.'
     STOP
  END IF

!!! END CREATE BOXES DATA OBJECT


!!!!!!!! BOXES DATA LEVEL TRAVERSAL

  ! Clears the datasize index so that it can be used to determine the
  ! amount of data space required for this level.
  i_grid_data_size = 0

  !        ! Allocates a box dataset within the file, but does not populate it.
  !        CALL Initialize_Chombo_Dataset_Box("boxes", hid_level_group_id, hid_box_id, ia_dataset_dims(1))

  ! Initialize the box dataset's offset counter.
  i_box_offset = 0


  !  Allocate box data array (6 * number of nodes on this level).
  ALLOCATE(ia_box_data(SIZE(ia_box_global), 1), stat=i_err)

  IF (i_err /= 0) THEN
     PRINT *, "MakeChomboFile error ", i_err, ": failed to create ", &
          "a box data array on level 0."
     STOP
  END IF

  !  Initialize node level index (one of the indices used to track position
  !  in the box data array.
  i_level_node_index = 1

  ! Clear box data in between nodes.
  dbl_domain_offset = 0.d0
  i_grid_offset = 0			


  ! Loop over three spatial dimensions to create box data.
  DO i_dimension = 1, 3

     ! Clear the grid_offset variable.
     dbl_domain_offset = 0
     i_grid_offset = 0

     ! offset = (diff.(local lower bound, min. lower bound)) / dxfix
     IF (dxfix(i_dimension) /= zero) THEN
        dbl_domain_offset = (Xlower(i_dimension) - &
             dbla_minLower(i_dimension)) / &
             dxfix(i_dimension)
     ELSE
        dbl_domain_offset = zero
     END IF


     ! Truncate the domain offset to get the grid 
     ! offset in cells.
     i_grid_offset = INT(dbl_domain_offset)

     ! If the fractional part of the domain offset is close 
     ! to 1, then round the grid offset up.			

     IF (dbl_domain_offset - REAL(i_grid_offset) > 0.5) THEN
        i_grid_offset = i_grid_offset + 1

        ! Because the grid offset is the truncated version of the
        ! domain offset, it should never be larger.

     ELSEIF (dbl_domain_offset < REAL(i_grid_offset)) THEN
        PRINT *, "MakeChomboFile error on level ", &
             i_level, ": Domain offset shouldn't be", &
             " smaller than grid offset."
        STOP
     END IF

     ! Set the lower-bound of the ia_box_data dimensions using the 
     ! global boundary and the grid offset.  If there is no difference
     ! between the local lower-bounds and the global lower-bounds, 
     ! then i_grid_offset will be 0.

     ia_box_data(i_dimension, i_level_node_index) = i_grid_offset
     !			ia_box_data(i_dimension) = i_grid_offset

     ! Set the upper bound along i_dimension using the lower bound and the
     ! number of cells along i_dimension.

     ia_box_data(i_dimension + 3, i_level_node_index) = ia_box_data(i_dimension, i_level_node_index) + &
          mxfix(i_dimension) - 1
     !           ia_box_data(i_dimension + 3) = ia_box_data(i_dimension) + &
     !               mxfix(i_dimension) - 1

  END DO	! i_dimension = 1 to 3

  SELECT CASE (nDim)    

     ! Generate i_grid_data_size value to use when allocating memory for the 
     ! chombo data array.

     ! [BDS] [20070813]:  Removed the NrOfOutVars multiplier so that this value 
     ! 					 just represents the number of cells in the grid
     !					 (thereby making it useful for both q and aux arrays).
  CASE(2)
     i_grid_data_size = i_grid_data_size + (mxfix(1) * &
          mxfix(2) * NrVars)

  CASE(3)
     i_grid_data_size = i_grid_data_size + (mxfix(1) * &
          mxfix(2) * mxfix(3) * NrVars)

  CASE DEFAULT
     PRINT *,'MakeChomboFile error: Invalid nDim ', nDim, &
          '.  Data not written.'
     STOP

  END SELECT


  ! Write the collected box data to the HDF file.
  CALL h5dwrite_f(hid_boxes_dataset_id, hid_box_id, ia_box_data, &
       ia_dataset_dims, i_err, xfer_prp = hid_boxes_property_id)


  ! Free up space from box data array.
  DEALLOCATE(ia_box_data, stat=i_err)

  IF (i_err /= 0) THEN 
     PRINT *, "MakeChomboFile error ", i_err, ": could not ", &
          "deallocate a box data array on level 0."
     STOP
  END IF


  ! Close the HDF5 handles created to construct the boxes dataset.
  CALL h5dclose_f(hid_boxes_dataset_id, i_err)
  CALL h5sclose_f(hid_boxes_dataspace_id, i_err)
  CALL h5pclose_f(hid_boxes_property_id, i_err)

  ! Die if unable to close boxes handles.
  IF (i_err < 0) THEN 
     PRINT *,"MakeChomboFile error ", i_err, &
          ": one or more HDF5 handles used by boxes couldn't close."
     STOP
  END IF


!!!!!!!! END BOXES DATA LEVEL TRAVERSAL


  PRINT *, "Constructing dataset."

!!!!!!!! DATAZERO LEVEL TRAVERSAL

  ! Clear offset; it will be used to reference chombo data array in the loop below.
  i_qvar_offset = 0
  ! Initialize a dataset for the q-variable data, but don't populate it.
  CALL Initialize_Chombo_Dataset_Float("data:datatype=0", hid_level_group_id, i_grid_data_size)

  ! Allocate memory for the current node's grid.
  ALLOCATE(dbla_chombo_qvar(mxfix(1) * mxfix(2) * &
       mxfix(3) * NrVars), stat=i_err)

  ! Stops execution on a memory allocation error.
  IF (i_err /= 0) THEN
     PRINT *,"MakeChomboFile error: unable to allocate ", &
          "memory for q-var array."
     STOP
  END IF

  ! Clear index; it will be used to reference chombo data array in the loop below.
  i_index = 0

  ! Loop over each variable tracked by the program.	
  DO i_variable = 1, NrVars

     ! The nested loops below loop over each spatial dimension and reference the
     ! q value for each variable within a specific 3D coordinate; this value is
     ! then stored in the chombo data array.  In effect, what this does is flatten
     ! the multi-dimensional data in q down to a 1D array for HDF5 storage.

     DO k = 1, mxfix(3)
        DO j = 1, mxfix(2)
           DO i = 1, mxfix(1)

              i_index = i_index + 1

              dbla_chombo_qvar(i_index) = qfix(i, j, k, i_variable)

           END DO	! End i
        END DO		! End j
     END DO			! End k

  END DO		! End do i_variable =  to NrVars

  ! Write the data array to the Chombo file using the supplied offset.
  CALL Write_Slab_To_Dataset_Float("data:datatype=0", hid_level_group_id, &
       dbla_chombo_qvar, i_qvar_offset)

  ! Release array memory so that it can be used by the next grid.
  DEALLOCATE(dbla_chombo_qvar)

!!!!!!!! END DATAZERO LEVEL TRAVERSAL


  ! Create the data attributes group.
  CALL h5gcreate_f(hid_level_group_id, "data_attributes", &
       hid_data_attributes_group_id, i_err)

  ! Die if unable to create new group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to create data_attributes group.'
     STOP
  END IF

  ! Adds the output_ghost attribute using the literal intvector [0, 0, 0].
  CALL Add_Chombo_Attribute_IntVector("outputGhost", hid_intvect_id, &
       hid_data_attributes_group_id, (/0,0,0/))

  ! Adds the comps attribute (the same as num_components)
  CALL Add_Chombo_Attribute_Int("comps", hid_data_attributes_group_id, &
       NrVars)

  ! Adds the objectType attribute.
  CALL Add_Chombo_Attribute_String("objectType", &
       hid_data_attributes_group_id, "FArrayBox")

  ! Close data attributes group.
  CALL h5gclose_f(hid_data_attributes_group_id, i_err)

  ! Die if unable to close data_attributes group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to close data_attributes group.'
     STOP
  END IF


  ! Close level_n group.
  CALL h5gclose_f(hid_level_group_id, i_err)

  ! Die if unable to close level_n group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to close level', i_level, ' group.'
     STOP
  END IF

  PRINT *, 'Level ', i_level, ' data written.'
  PRINT *

  ! Closing type handles.
  CALL h5tclose_f(hid_intvect_id, i_err)
  CALL h5tclose_f(hid_floatvect_id, i_err)
  CALL h5tclose_f(hid_box_id, i_err)

  ! Die if unable to close all type handles.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': one or more type handles couldn\t be closed.'
     STOP
  END IF


  ! Close root group.
  CALL h5gclose_f(hid_group_id, i_err)

  ! Die if unable to close root group.
  IF (i_err < 0) THEN 
     PRINT *,'MakeChomboFile error ', i_err, &
          ': unable to close root group.'
     STOP
  END IF


  ! Close hdf5 file.
  CALL h5fclose_f(hid_file_id, i_err)

  ! Die if unable to close file handle.
  IF (i_err < 0) THEN 
     PRINT *,"MakeChomboFile error ", i_err, ": HDF5 file handle couldn't be closed."
     STOP
  END IF


!!!!! Stop Clock !!!!!!
  CALL CPU_TIME(dbl_tend)
  t_counter = t_counter + 1
  t_accumulator = t_accumulator + dbl_tend - dbl_tstart
  PRINT *, "Time to make this chombo file: ", dbl_tend - dbl_tstart, " seconds."
  PRINT *, "Total chombo file time:  ", t_accumulator, " seconds."
!!!!!!!!!!!!!!!!!!!!!!!

END SUBROUTINE MakeChomboFile


! Created 2007-07-13 by Brandon D. Shroyer.  Adds a new attribute to the Chombo HDF
!!!  file.  Note that this function assumes that the dataspace is scalar.

! Modified 2007-07-16 by Brandon D. Shroyer.  Added group_id to parameter list.

! Modified 2007-08-14 by Brandon D. Shroyer.  Removed type parameter.
SUBROUTINE Add_Chombo_Attribute_Int(s_name, hid_group_id, i_value)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  INTEGER :: i_value								! Data input

  ! Variable declarations
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err


  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, H5T_NATIVE_INTEGER, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, H5T_NATIVE_INTEGER, i_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_Int error ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF

END SUBROUTINE Add_Chombo_Attribute_Int



! Created 2007-11-13 by Brandon D. Shroyer.
SUBROUTINE Add_Chombo_Attribute_Float(s_name, hid_group_id, dbl_value)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  REAL(KIND(0.d0)) :: dbl_value					! Data input

  ! Variable declarations
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err



  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, H5T_NATIVE_DOUBLE, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, H5T_NATIVE_DOUBLE, dbl_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute error_Float ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF

END SUBROUTINE Add_Chombo_Attribute_Float


! Modified 2007-07-16 by Brandon D. Shroyer.  Added group_id to parameter list.
SUBROUTINE Add_Chombo_Attribute_IntVector(s_name, hid_type_id, hid_group_id, ia_value, i_size)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_type_id		! Describes the type of v_value.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  INTEGER, DIMENSION(3) :: ia_value				! Data input
  INTEGER(size_t), INTENT(IN), OPTIONAL :: i_size	! Size of input type.

  ! Variable declarations
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err

  ! If a size value was passed in, then set the type's size value to that.
  IF (PRESENT(i_size)) THEN
     CALL h5tset_size_f(hid_type_id, i_size ,i_err)
  END IF

  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, hid_type_id, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, hid_type_id, ia_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_IntVector error ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF

END SUBROUTINE Add_Chombo_Attribute_IntVector



! Modified 2007-07-16 by Brandon D. Shroyer.  Added group_id to parameter list.
SUBROUTINE Add_Chombo_Attribute_FloatVector(s_name, hid_type_id, hid_group_id, dbla_value, i_size)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_type_id		! Describes the type of v_value.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  REAL(KIND(0.d0)), DIMENSION(3) :: dbla_value	! Data input
  INTEGER(size_t), INTENT(IN), OPTIONAL :: i_size	! Size of input type.

  ! Variable declarations
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err


  ! If a size value was passed in, then set the type's size value to that.
  IF (PRESENT(i_size)) THEN
     CALL h5tset_size_f(hid_type_id, i_size ,i_err)
  END IF

  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, hid_type_id, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, hid_type_id, dbla_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_FloatVector error ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF

END SUBROUTINE Add_Chombo_Attribute_FloatVector

! Modified 2007-07-16 by Brandon D. Shroyer.  Added group_id to parameter list.
SUBROUTINE Add_Chombo_Attribute_String(s_name, hid_group_id, s_value)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  CHARACTER(LEN=*) :: s_value						! Data input

  ! Variable declarations
  INTEGER(HID_T) :: hid_type_id
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err


  ! Copy the H5T_NATIVE_CHARACTER data type to hid_h5string_type_id.
  CALL h5tcopy_f(H5T_NATIVE_CHARACTER, hid_type_id, i_err)

  ! Die if unable to copy the H5T_NATIVE_INTEGER type.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_String error ', i_err, &
          ': one or more calls to h5tcopy_f failed for ', TRIM(s_name), '.'
     STOP
  END IF

  ! Set the type's size value.
  CALL h5tset_size_f(hid_type_id, INT(LEN(TRIM(s_value)), size_t), i_err)

  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, hid_type_id, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, hid_type_id, s_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the type object.
  CALL h5tclose_f(hid_type_id, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_String error ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF

END SUBROUTINE Add_Chombo_Attribute_String


! Modified 2007-07-16 by Brandon D. Shroyer.  Added group_id to parameter list.
SUBROUTINE Add_Chombo_Attribute_Box(s_name, hid_type_id, hid_group_id, ia_value, i_size)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new attribute.
  INTEGER(HID_T), INTENT(IN) :: hid_type_id		! Used to set the string size.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  INTEGER, DIMENSION(6) :: ia_value				! Data input
  INTEGER(size_t), INTENT(IN), OPTIONAL :: i_size	! Size of input type.

  ! Variable declarations
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_attribute_id

  INTEGER :: i_err


  ! If a size value was passed in, then set the type's size value to that.
  IF (PRESENT(i_size)) THEN
     CALL h5tset_size_f(hid_type_id, i_size ,i_err)
  END IF

  ! Create a new dataspace.
  CALL h5screate_simple_f(HID_ATTRIBUTE_RANK, IA_SCALAR_ATTRIB_DIMS, &
       hid_dataspace_id, i_err)

  ! Create new attribute within dataspace.
  CALL h5acreate_f(hid_group_id, s_name, hid_type_id, hid_dataspace_id, &
       hid_attribute_id, i_err)

  ! Write the datavalue to the dataset.
  CALL h5awrite_f(hid_attribute_id, hid_type_id, ia_value, &
       IA_SCALAR_ATTRIB_DIMS, i_err)

  ! Close the attribute object.
  CALL h5aclose_f(hid_attribute_id, i_err)

  ! Close the dataspace object.
  CALL h5sclose_f(hid_dataspace_id, i_err)


  ! Die if unable to add this attribute to Chombo.
  IF (i_err < 0) THEN 
     PRINT *,'Add_Chombo_Attribute_Box error ', i_err, &
          ': unable to add attribute "', s_name, '".'
     STOP
  END IF


END SUBROUTINE Add_Chombo_Attribute_Box


! Created 2007-07-24 by Brandon D. Shroyer
! Returns an nDim-element array containing the minimum lower-bounds for each
! dimension on level i_level.

! Created: 2007-11-27 by Brandon D. Shroyer
SUBROUTINE Initialize_Chombo_Dataset_Float(s_name, hid_group_id, i_dataset_size)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  INTEGER(HSIZE_T) :: i_dataset_size               ! Size of forthcoming data array, used to
  !     initialize the dataspace.


  INTEGER(HID_T) :: hid_dataspace_id				! Dataspace handle.
  INTEGER(HID_T) :: hid_dataset_id				! Dataset handle.
  INTEGER(HID_T) :: hid_property_list_id				! Property list handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dataset_dims	! Array to hold the size of the input.
  INTEGER :: i_err

  PRINT *, "Initializing dataset ", TRIM(s_name), "..."


  ia_dataset_dims(1) = i_dataset_size

  ! Create the new dataspace.
  CALL h5screate_simple_f(I_DATASET_RANK, ia_dataset_dims, hid_dataspace_id, i_err)

  ! Die if unable to create the new dataset.
  IF (i_err < 0) THEN 
     PRINT *,'Initialize_Chombo_Dataset_Float error ', i_err, &
          ': Could not create dataspace for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  ! Create the property list for this dataset.
  CALL h5pcreate_f(H5P_DATASET_CREATE_F, hid_property_list_id, i_err)

  ! Chunk the dataset for compression.
  CALL h5pset_chunk_f(hid_property_list_id, I_DATASET_RANK, ia_dataset_dims, i_err)

  ! Forces the dataset to allocate the space at creation time.
  CALL h5pset_alloc_time_f(hid_property_list_id, H5D_ALLOC_TIME_EARLY_F, i_err)

  ! Set the dataset to use gzip compression (default for deflate option).
  !	CALL h5pset_deflate_f(hid_property_list_id,I_DEFLATE_LEVEL,i_err)


  ! Die if unable to set up property list.
  IF (i_err < 0) THEN
     PRINT *, "Initialize_Chombo_Dataset_Float error ", i_err, &
          " unable to set up property list for dataset ", TRIM(s_name), "."
     STOP
  END IF


  ! Create the new dataset.
  CALL h5dcreate_f(hid_group_id, s_name, H5T_NATIVE_DOUBLE, &
       hid_dataspace_id, hid_dataset_id, i_err, hid_property_list_id)

  ! Die if unable to create the new dataset.
  IF (i_err < 0) THEN 
     PRINT *,'Initialize_Chombo_Dataset_Float error ', i_err, &
          ': Could not create dataset ', TRIM(s_name), '.'
     STOP
  END IF


  CALL h5dclose_f(hid_dataset_id, i_err)		! Close the dataset.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Float error ', i_err, & 
          ': Could not close dataset ', TRIM(s_name), '.'
     STOP
  END IF

  CALL h5pclose_f(hid_property_list_id, i_err)	! Close the property list.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Float error ', i_err, & 
          ': Could not close property list for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  CALL h5sclose_f(hid_dataspace_id, i_err)	! Close the dataspace.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Float error ', i_err, & 
          ': Could not close dataspace for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  PRINT *, "Dataset ", TRIM(s_name), " initialized."

END SUBROUTINE Initialize_Chombo_Dataset_Float


! Created: 2007-11-27 by Brandon D. Shroyer
SUBROUTINE Write_Slab_To_Dataset_Float(s_dataset_name, hid_group_id, dbla_data, i_offset)

  CHARACTER(LEN=*), INTENT(IN) :: s_dataset_name
  INTEGER(HID_T) :: hid_group_id
  REAL(KIND=qPrec), DIMENSION(:) :: dbla_data		! Data input
  INTEGER(SIZE_T) :: i_offset

  INTEGER(HID_T) :: hid_dataset_id
  INTEGER(HID_T) :: hid_dataspace_id
  INTEGER(HID_T) :: hid_memspace_id
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_data_offset
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_slab_size
  INTEGER(HSIZE_T) :: i_dataset_size
  INTEGER :: i_err


  ! Set the size of the hyperslab.
  ia_slab_size(1) = SIZE(dbla_data)

  ! Store the size of the offset in a dimension array.
  ia_data_offset(1) = i_offset

  ! Open the dataset in the Chombo file.
  CALL h5dopen_f(hid_group_id, s_dataset_name, hid_dataset_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset_Float error ", i_err, ": unable to open ", &
          "dataset ", s_dataset_name, "."	
     STOP
  END IF


  CALL h5dget_space_f(hid_dataset_id, hid_dataspace_id, i_err)

  ! Die if unable to open dataset.
  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset_Float error ", i_err, ": unable to get ", &
          "dataspace for dataset ", s_dataset_name, "."
     STOP
  END IF


  ! Select hyperslab in the dataset (specifically, the dimensions of the input array).
  CALL h5sselect_hyperslab_f(hid_dataspace_id, H5S_SELECT_SET_F, &
       ia_data_offset, ia_slab_size, i_err) 

  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset error ", i_err, ": Unable to select hyperslab ", &
          "from dataset ", s_dataset_name, "."
     STOP
  END IF


  ! Create dataspace for the upcoming hyperslab.
  CALL h5screate_simple_f(I_DATASET_RANK, ia_slab_size, hid_memspace_id, i_err)

  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset error ", i_err, ":  Unable to create hyperslab ", & 
          "dataspace for dataset ", s_dataset_name, "."
     STOP
  END IF


  ! Write the chombo data to the dataset.
  CALL h5dwrite_f(hid_dataset_id, H5T_NATIVE_DOUBLE, dbla_data, &
       ia_slab_size, i_err, hid_memspace_id, hid_dataspace_id)

  ! Close hyperslab dataspace.
  CALL h5sclose_f(hid_memspace_id, i_err)

  ! Die if unable to close hyperslab dataspace.
  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset_Float error ", i_err, ": unable to close ", &
          "memory dataspace for dataset ", s_dataset_name, "."
     STOP
  END IF


  ! Close main file memory dataspace.
  CALL h5sclose_f(hid_dataspace_id, i_err)

  ! Die if unable to close dataspace.
  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset_Float error ", i_err, ": unable to close ", &
          "file dataspace for dataset ", s_dataset_name, "."
     STOP
  END IF


  CALL h5dclose_f(hid_dataset_id, i_err)

  ! Die if unable to close dataset.
  IF (i_err < 0) THEN
     PRINT *, "Write_Slab_To_Dataset_Float error ", i_err, ": unable to close ", &
          "dataset ", s_dataset_name, "."
     STOP
  END IF

END SUBROUTINE Write_Slab_To_Dataset_Float



SUBROUTINE Initialize_Chombo_Dataset_Box(s_name, hid_group_id, hid_type_id, i_dataset_size)

  IMPLICIT NONE

  ! Input parameter declarations
  CHARACTER(LEN=*), INTENT(IN) :: s_name			! Name of new dataset.
  INTEGER(HID_T), INTENT(IN) :: hid_group_id		! Group identifier.
  INTEGER(HID_T), INTENT(IN) :: hid_type_id        ! Box type identifier.
  INTEGER(HSIZE_T) :: i_dataset_size               ! Size of forthcoming box array, used to
  !     initialize the dataspace.


  INTEGER(HID_T) :: hid_dataspace_id				! Dataspace handle.
  INTEGER(HID_T) :: hid_dataset_id				! Dataset handle.
  INTEGER(HID_T) :: hid_property_list_id				! Property list handle.
  INTEGER(HSIZE_T), DIMENSION(1) :: ia_dataset_dims	! Array to hold the size of the input.
  INTEGER :: i_err

  PRINT *, "Initializing box dataset ", TRIM(s_name), "..."


  ia_dataset_dims(1) = i_dataset_size

  ! Create the new dataspace.
  CALL h5screate_simple_f(I_DATASET_RANK, ia_dataset_dims, hid_dataspace_id, i_err)

  ! Die if unable to create the new dataset.
  IF (i_err < 0) THEN 
     PRINT *,'Initialize_Chombo_Dataset_Box error ', i_err, &
          ': Could not create dataspace for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  ! Create the property list for this dataset.
  CALL h5pcreate_f(H5P_DATASET_CREATE_F, hid_property_list_id, i_err)

  ! Chunk the dataset for compression.
  CALL h5pset_chunk_f(hid_property_list_id, I_DATASET_RANK, ia_dataset_dims, i_err)

  ! Forces the dataset to allocate the space at creation time.
  CALL h5pset_alloc_time_f(hid_property_list_id, H5D_ALLOC_TIME_EARLY_F, i_err)

  ! Set the dataset to use gzip compression (default for deflate option).
  !	CALL h5pset_deflate_f(hid_property_list_id,I_DEFLATE_LEVEL,i_err)


  ! Die if unable to set up property list.
  IF (i_err < 0) THEN
     PRINT *, "Initialize_Chombo_Dataset_Box error ", i_err, &
          " unable to set up property list for dataset ", TRIM(s_name), "."
     STOP
  END IF


  ! Create the new dataset.
  CALL h5dcreate_f(hid_group_id, s_name, hid_type_id, &
       hid_dataspace_id, hid_dataset_id, i_err, hid_property_list_id)

  ! Die if unable to create the new dataset.
  IF (i_err < 0) THEN 
     PRINT *,'Initialize_Chombo_Dataset_Box error ', i_err, &
          ': Could not create dataset ', TRIM(s_name), '.'
     STOP
  END IF


  CALL h5dclose_f(hid_dataset_id, i_err)		! Close the dataset.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Box error ', i_err, & 
          ': Could not close dataset ', TRIM(s_name), '.'
     STOP
  END IF

  CALL h5pclose_f(hid_property_list_id, i_err)	! Close the property list.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Box error ', i_err, & 
          ': Could not close property list for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  CALL h5sclose_f(hid_dataspace_id, i_err)	! Close the dataspace.

  IF (i_err < 0) THEN
     PRINT *, 'Initialize_Chombo_Dataset_Box error ', i_err, & 
          ': Could not close dataspace for dataset ', TRIM(s_name), '.'
     STOP
  END IF

  PRINT *, "Dataset ", TRIM(s_name), " initialized."

END SUBROUTINE Initialize_Chombo_Dataset_Box


SUBROUTINE PrintBOVOfPotential(frame)
!   REAL,DIMENSION(mxfix(1),mxfix(2)) :: Bx,By
   REAL,DIMENSION(mxfix(1)+1,mxfix(2)+1) :: A
   CHARACTER(LEN=25) :: FileName
   INTEGER :: filehandle,proc,frame, i, j
   REAL(KIND=xPrec) :: mysize(3),myxlower(3)

   !Starts at lower left corner and integrates to the right and integrates up
!   A(1,1)=0
!   DO i=2, mxfix(1)+1
!      A(i,1)=A(i-1,1)-auxFix(i-1, 1,1,1,2)*dixfix(1)
!   END DO
!   DO j=2,mxfix(2)+1
!      A(1,j)=A(1,j-1)+auxFix(1,j-1,1,1,1)*dxfix(2)
!   END DO
!   DO i=2, mxfix(1)+1
!      DO j=2, mxfix(2)+1
!         A(i,j)=half*(A(i-1,j)+A(i,j-1)-auxfix(i,j,1,1,2)*dxfix(1)+ & !Averages interpolated potential from both directions
!              auxfix(i,j,1,1,1)*dixfix(2))
!      END DO
!   END DO!


!! Alternative is to start along one boundary and integrate in one direction...
   A(1,1)=0
   DO i=2, mxfix(1)+1
      A(i,1)=A(i-1,1)-auxFix(i-1, 1,1,1,2)*dxfix(1)
   END DO
   DO i=1, mxfix(1)+1
      DO j=2, mxfix(2)+1
         A(i,j)=A(i,j-1)+auxfix(i,j,1,1,1)*dxfix(2)
      END DO
   END DO

   ! A is returned on grid corners
!   CALL stream(A,Bx,By,mxfix(1),mxfix(2),dxfix(1),dxfix(2))

   filehandle=11

   write(FileName,'(A17,I4.4,A4)') "out/MagPotential_" ,frame,".bov"

   OPEN(UNIT=filehandle, FILE=Filename)
   WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
   write(filehandle,'(A24,I4.4,A4)') "DATA_FILE: MagPotential_" ,frame,".dat"
   WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", shape(A), 1
   WRITE(filehandle,*)  "DATA_FORMAT: REAL"
   WRITE(filehandle,*)  "VARIABLE: MagPotential"
   WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
   WRITE(filehandle,*)  "CENTERING: zonal"
!   myXlower=Xlower(1:3)+(index_range(1:3,1)-1)*dxfix(1:3)
!   mysize=(index_range(1:3,2)-index_range(1:3,1)+1)*dxfix(1:3)
   WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", Xlower(1:2)-half*dxfix(1:2), 0
   WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", dxfix(1:2)*(mxfix(1:2)+1), 0
   WRITE(filehandle,*)  "BYTE_OFFSET: 4"
   WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",1
   CLOSE(filehandle)
   write(FileName,'(A17,I4.4,A4)') "out/MagPotential_" ,frame,".dat"
   OPEN(UNIT=filehandle, FILE=Filename, FORM="unformatted", status="replace")
   write(filehandle) A
   CLOSE(filehandle)
END SUBROUTINE PrintBOVOfPotential

SUBROUTINE output_section(proc, index_range,frame)
   USE GlobalDeclarations
   INTEGER, DIMENSION(:,:) :: index_range
   INTEGER :: filehandle,filehandle2,proc,frame
   CHARACTER(LEN=40) :: FileName, FileName2
   filehandle = 11
   filehandle2= 12
   write(FileName,'(A3,A3,I3.3,A1,I3.3,A4)') outdir,"/Q_" , proc-1, "_",frame,".dat"
   OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
   write(filehandle) transpose(reshape(qFix(index_range(1,1):index_range(1,2), index_range(2,1):index_range(2,2), index_range(3,1):index_range(3,2),1:NrVars), (/product(index_range(:,2)-index_range(:,1)+1),nrVars/)))
   CLOSE(filehandle)
   CALL write_BOV(proc,index_range,frame)
   IF (lMHD) THEN
      write(FileName2,'(A3,A5,I3.3,A1,I3.3,A4)') outdir,"/aux_" , proc-1, "_",frame,".dat"
      OPEN(UNIT=filehandle2, FILE=FileName2, status="replace", FORM="unformatted")
      write(filehandle2) auxFix(index_range(1,1):index_range(1,2)+1, index_range(2,1):index_range(2,2), index_range(3,1):index_range(3,2),1,1)
      write(filehandle2) auxFix(index_range(1,1):index_range(1,2),index_range(2,1):index_range(2,2)+1, index_range(3,1):index_range(3,2),1,2)
      IF (nDim .eq. 3) THEN
         write(filehandle2) auxFix(index_range(1,1):index_range(1,2),index_range(2,1):index_range(2,2), index_range(3,1):index_range(3,2)+1,1,3)
      END IF
      CLOSE(filehandle2)
      CALL write_aux_to_BOV(proc,index_range,frame)
   END IF
END SUBROUTINE output_section

SUBROUTINE write_BOV(proc,index_range,frame)
   USE GlobalDeclarations
   CHARACTER(LEN=40) :: FileName
   INTEGER, DIMENSIOn(:,:) :: index_range
   INTEGER :: filehandle,proc,frame
   REAL(KIND=xPrec) :: mysize(3),myxlower(3)
   filehandle=11

   write(FileName,'(A3,A3,I3.3,A1,I3.3,A4)') outdir,"/Q_" , proc-1, "_",frame,".bov"

   OPEN(UNIT=filehandle, FILE=Filename)
   WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
   write(filehandle,'(A13,I3.3,A1,I3.3,A4)') "DATA_FILE: Q_" , proc-1, "_",frame,".dat"
   WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", index_range(1:3,2)-index_range(1:3,1)+1
   WRITE(filehandle,*)  "DATA_FORMAT: REAL"
   WRITE(filehandle,*)  "VARIABLE: Q"
   WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
   WRITE(filehandle,*)  "CENTERING: zonal"

   myXlower=Xlower(1:3)+(index_range(1:3,1)-1)*dxfix(1:3)
   mysize=(index_range(1:3,2)-index_range(1:3,1)+1)*dxfix(1:3)
   WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", myXlower(:)
   WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", mysize(:)

   WRITE(filehandle,*)  "BYTE_OFFSET: 4"
   WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",nrvars
   WRITE(11,'(A9,8I8)')  "MGLOBAL: ", index_range(1:3,1), 1, index_range(1:3,2), 1
   CLOSE(filehandle)
END SUBROUTINE write_BOV

SUBROUTINE write_aux_to_BOV(proc, index_range,frame)
   USE GlobalDeclarations
   CHARACTER(LEN=40)::FileName
   INTEGER, DIMENSION(:,:)::index_range
   INTEGER::filehandle,proc,frame
   REAL(KIND=xPrec)::mysize(3),myxlower(3)
   filehandle=12

   WRITE(FileName, '(A3,A5,I3.3,A1,I3.3,A4)') outdir,'/aux_',proc-1,'_',frame,'.bov'

   OPEN(UNIT=filehandle, FILE=Filename)
   WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
   write(filehandle,'(A15,I3.3,A1,I3.3,A4)') "DATA_FILE: aux_" , proc-1, "_",frame,".dat"
   WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", index_range(1:3,2)-index_range(1:3,1)+1
   WRITE(filehandle,*)  "DATA_FORMAT: REAL"
   WRITE(filehandle,*)  "VARIABLE: aux"
   WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
   WRITE(filehandle,*)  "CENTERING: zonal"

   myXlower=Xlower(1:3)+(index_range(1:3,1)-1)*dxfix(1:3)
   mysize=(index_range(1:3,2)-index_range(1:3,1)+1)*dxfix(1:3)
   WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", myXlower(:)
   WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", mysize(:)

   WRITE(filehandle,*)  "BYTE_OFFSET: 4"
   WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",3
   CLOSE(filehandle)

END SUBROUTINE write_aux_to_BOV

SUBROUTINE SplitGrid(np,frame)
   INTEGER :: np,frame
   integer :: procs(2),total_cells,mglobal(3,2),i
   REAL,DIMENSION(:,:,:),ALLOCATABLE :: comp_map
   INTEGER,DIMENSION(:,:,:),ALLOCATABLE :: grid_layout
   ! 
   REAL :: total_cost
   ALLOCATE(comp_map(mxFix(1),mxFix(2),mxFix(3)))
   ALLOCATE(grid_layout(np,3,2))
   comp_map=1
   mGlobal=1
   mglobal(1:nDim,2)=mxFix(1:nDim)
   procs=(/1,np/)
   !    write(*,*) mglobal, procs, nDim
   CALL split(mGlobal,comp_map,procs,grid_layout,nDim)

   !   total_cost=sum(comp_map)
   !   total_cells=product(mglobalupper-mGloballower+1)
   !   write(*,*) "grid sizes are :"
   !   write(*,'(5A10)') "mx", "my", "mz", "cells %","cost %"
   !   DO i=1,np_
   !      write(*,'(3I10,2F10.4)') grid_layout(i,1,2)-grid_layout(i,1,1)+1,grid_layout(i,2,2)-grid_layout(i,2,1)+1,grid_layout(i,3,2)-grid_layout(i,3,1)+1,&
   !           100*product(grid_layout(i,:,2)-grid_layout(i,:,1)+1)/REAL(total_cells), 100*sum(comp_map(grid_layout(i,1,1):grid_layout(i,1,2), grid_layout(i,2,1):grid_layout(i,2,2), grid_layout(i,3,1):grid_layout(i,3,2)))/total_cost
   !   END DO

   DO i=1,np
      CALL output_section(i,grid_layout(i,:,:),frame)
   END DO
END SUBROUTINE SplitGrid

recursive subroutine split(mymglobal, mymap, myprocs,grid_layout,ndim)
   INTEGER, PARAMETER :: MinimumGridPoints(0:0)=4
   INTEGER, DIMENSION(:,:) :: mymglobal
   INTEGER, DIMENSION(:,:,:), INTENT(INOUT) :: grid_layout
   INTEGER :: myprocs(2),nprocs,p(2),i,j,split_point(3),min_split(3), dir, newmglobal(3,2),newprocs(2),slice_mx,  mymx(3),ndim,mymingridpointsL, mymingridpointsR,numbergrids_perp,numbergrids_parallel
   REAL, DIMENSION(:,:,:) :: mymap
   REAL :: frac_sum,my_sum,total_sum
   REAL, DIMENSION(:), ALLOCATABLE :: collapse
   REAL, DIMENSION(:,:,:), ALLOCATABLE :: newmap
   !    write(*,'(A,6I4,2I4)') "splitting", mymglobal, myprocs
   nprocs=myprocs(2)-myprocs(1)+1
   IF (nprocs==1) THEN !Can't split
      !       write(*,'(A,I4,A,6I4)') "processor", myprocs(1), "has bounds", mymglobal
      grid_layout(myprocs(1),:,:)=mymglobal
      RETURN
   END IF
   p(1)=nprocs/2
   p(2)=nprocs-p(1)
   total_sum=sum(mymap)
   frac_sum=total_sum*p(1)/real(nprocs)
   mymx=shape(mymap)
   IF (maxval(mymx(1:nDim)) < 2d0*MinimumGridPoints(0)) THEN
      write(*,*) "can no longer split box of size", mymx(1:nDim)
      write(*,*) "try regridding to fewer processors"
      STOP
   END IF
   do i=1,ndim
      slice_mx=mymx(i)
      ALLOCATE(collapse(slice_mx))
      collapse(1:slice_mx)=sum(sum(mymap,max(modulo(i,3)+1,modulo(i+1,3)+1)),min(modulo(i,3)+1,modulo(i+1,3)+1))
      IF (nDim == 3) THEN
         numbergrids_perp=(mymx(modulo(i,3)+1)/MinimumGridPoints(0)) * (mymx(modulo(i+1,3)+1)/MinimumGridPoints(0))
      ELSE
         numbergrids_perp=(mymx(3-i)/MinimumGridPoints(0))
      END IF
      numbergrids_parallel=ceiling(real(nprocs)/real(numbergrids_perp)) !number of grids needed in direction of splitting
      mymingridpointsL=MinimumGridPoints(0)*max(1, p(1)/numbergrids_perp)
      mymingridpointsR=MinimumGridPoints(0)*max(1, p(2)/numbergrids_perp)
      my_sum=sum(collapse(1:mymingridpointsL-1))
      !       write(*,*) numbergrids_perp, p(2), p(2)/numbergrids_perp, mymingridpointsL,mymingridpointsR,slice_mx
      split_point(i)=slice_mx-mymingridpointsR
      min_split(i)=0!min(mymingridpointsR,mymingridpointsL)
      IF (mymingridpointsL+mymingridpointsR<=slice_mx) THEN
         min_split(i)=mymingridpointsL
      END IF
      do j=mymingridpointsL,slice_mx-mymingridpointsR
         my_sum=my_sum+collapse(j)
         if (my_sum >= frac_sum) THEN 
            split_point(i)=j
            min_split(i)=min(j,slice_mx-j)
            exit
         end if
      end do
      DEALLOCATE(collapse)
   end do
   dir=minval(maxloc(min_split(1:nDim)))
   !    write(*,'(3I4,2F8.2,6I4)') nprocs,p(1),p(2),frac_sum,sum(mymap),split_point(1:nDim), min_split(1:nDim),dir
   slice_mx=mymx(dir)
   ALLOCATE(collapse(slice_mx))
   collapse(1:slice_mx)=sum(sum(mymap,max(modulo(dir,3)+1,modulo(dir+1,3)+1)),min(modulo(dir,3)+1,modulo(dir+1,3)+1))
   frac_sum=sum(collapse(1:split_point(dir)))
   IF (p(1) < ceiling(nprocs*frac_sum/total_sum)) THEN
      !       write(*,*) frac_sum/total_sum, nprocs, p(1)
      !       write(*,*) "increasing p(1)"
      p(1)=min(ceiling(nprocs*frac_sum/total_sum),split_point(dir)/MinimumGridPoints(0) * numbergrids_perp,nprocs-1)
      p(2)=nprocs-p(1)
   END IF
   IF (p(2) < ceiling(nprocs*(1d0-frac_sum/total_sum))) THEN
      !       write(*,*) "increasing p(2)", slice_mx, split_point(dir), numbergrids_perp
      p(2)=min(ceiling(nprocs*(1d0-frac_sum/total_sum)),(slice_mx-split_point(dir))/MinimumGridPoints(0)*numbergrids_perp,nprocs-1)
      p(1)=nprocs-p(2)
   END IF
   !    write(*,*) p(1),p(2)
   IF (p(1)*p(2) == 0) STOP
   do i=1,2
      newmglobal=mymglobal
      IF (i==1) THEN
         newmglobal(dir,2)=mymglobal(dir,1)+split_point(dir)-1
         newprocs=(/myprocs(1),myprocs(1)+p(1)-1/)
         !          write(*,*) "newmglobal=", newmglobal
      ELSE
         newmglobal(dir,1)=mymglobal(dir,1)+split_point(dir)
         newprocs=(/myprocs(1)+p(1),myprocs(2)/)
         !          write(*,*) "newmglobal=", newmglobal
      END IF
      ALLOCATE(newmap(newmglobal(1,1):newmglobal(1,2), newmglobal(2,1):newmglobal(2,2),newmglobal(3,1):newmglobal(3,2)))
      newmap=mymap(newmglobal(1,1)-mymglobal(1,1)+1:newmglobal(1,2)-mymglobal(1,1)+1, newmglobal(2,1)-mymglobal(2,1)+1:newmglobal(2,2)-mymglobal(2,1)+1,newmglobal(3,1)-mymglobal(3,1)+1:newmglobal(3,2)-mymglobal(3,1)+1)

      CALL split(newmglobal, newmap,newprocs,grid_layout,ndim)
      DEALLOCATE(newmap)
   END do
end subroutine split

SUBROUTINE regrid(q,aux,ndim,mx)
   INTEGER :: r(3), ndim
   REAL(KIND=qprec),DIMENSION(:,:,:,:,:),POINTER::AUX, AUXOLD
   REAL, DIMENSION(:,:,:,:),POINTER :: q, qold
   INTEGER,DIMENSION(3),INTENT(INOUT)::mX

   r=1
   r(1:nDim)=2
   qold=>q
   ! ALLOCATE(qold(1:mX(1),1:mX(2),1:mX(3)+1,1,1:nrVars))
   ! nfo%qold=q
   ! DEALLOCATE(q)
   NULLIFY(q)
   ALLOCATE(q(1:r(1)*mX(1),1:r(2)*mX(2),1:r(3)*mX(3),1:NrVars))         
   !  SELECT CASE(nDim)
   !  CASE(2)
   !     ALLOCATE(auxold(1:mX(1)+1,1:mX(2)+1,2,1,1:nDim))
   !  CASE(3)
   !     ALLOCATE(auxold(1:mX(1)+1,1:mX(2)+1,1:mX(3)+1,1,1:nDim))
   !  END SELECT

   IF(MaintainAuxArrays) THEN
      auxold=>aux
      NULLIFY(aux)

      SELECT CASE(nDim)
      CASE(2)
         ALLOCATE(aux(1:r(1)*mX(1)+1,1:r(2)*mX(2)+1,2,1,1:nDim))
      CASE(3)
         ALLOCATE(aux(1:r(1)*mX(1)+1,1:r(2)*mX(2)+1,1:r(3)*mX(3)+1,1,1:nDim))
      END SELECT
   END IF

   CALL interpolate(q,qold,aux,auxold,mx)

   DEALLOCATE(qold)
   NULLIFY(qold)
   IF(MaintainAuxArrays) THEN
      DEALLOCATE(auxold)
      NULLIFY(auxold)
   END IF

   !  mX(1:nDim)=2d0*mX(1:nDim)

END SUBROUTINE regrid

SUBROUTINE interpolate(q,qold,aux,auxold,mx)
   REAL, DIMENSION(:,:,:,:), POINTER :: q,qold
   REAL, DIMENSION(:,:,:,:,:), POINTER :: slope
   REAL(KIND=qPREC), DIMENSION(:,:,:,:,:), POINTER :: aux, slope_aux,auxold
   INTEGER :: i,j,k,m,mx(3),newmx(3)
   REAL, PARAMETER :: C_1=.125d0
   REAL(8), DIMENSION(8) :: Bext2D
   REAL(8), PARAMETER, DIMENSION(4,8) :: A2D = &
        0.25d0*RESHAPE((/ &
        2.d0, 0.d0, 1.d0, 1.d0, & 
        0.d0, 2.d0, -1.d0, -1.d0, &
        2.d0, 0.d0, -1.d0, -1.d0, &
        0.d0, 2.d0, 1.d0, 1.d0, &
        1.d0, 1.d0, 2.d0, 0.d0, &
        -1.d0, -1.d0, 2.d0, 0.d0, &
        -1.d0, -1.d0, 0.d0, 2.d0, &
        1.d0, 1.d0, 0.d0, 2.d0 &
        /),(/4,8/) )
   REAL(8), DIMENSION(24) :: Bext3D
   REAL(8), PARAMETER, DIMENSION(12,24) :: A3D = &
        0.0625d0*RESHAPE((/ &
        8.d0, 0.d0, 0.d0, 0.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, &
        0.d0, 8.d0, 0.d0, 0.d0, 1.d0, 3.d0, 1.d0, 3.d0, -3.d0, -1.d0, -3.d0, -1.d0, &
        0.d0, 0.d0, 8.d0, 0.d0, -3.d0, -1.d0, -3.d0, -1.d0, 1.d0, 3.d0, 1.d0, 3.d0, &
        0.d0, 0.d0, 0.d0, 8.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, &
        8.d0, 0.d0, 0.d0, 0.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, &
        0.d0, 8.d0, 0.d0, 0.d0, -1.d0, -3.d0, -1.d0, -3.d0, 3.d0, 1.d0, 3.d0, 1.d0, &
        0.d0, 0.d0, 8.d0, 0.d0, 3.d0, 1.d0, 3.d0, 1.d0, -1.d0, -3.d0, -1.d0, -3.d0, &
        0.d0, 0.d0, 0.d0, 8.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, &
        3.d0, 1.d0, 3.d0, 1.d0, 8.d0, 0.d0, 0.d0, 0.d0, 3.d0, 3.d0, 1.d0, 1.d0, &
        1.d0, 3.d0, 1.d0, 3.d0, 0.d0, 8.d0, 0.d0, 0.d0, -3.d0, -3.d0, -1.d0, -1.d0, &
        -3.d0, -1.d0, -3.d0, -1.d0, 8.d0, 0.d0, 0.d0, 0.d0, -3.d0, -3.d0, -1.d0, -1.d0, &
        -1.d0, -3.d0, -1.d0, -3.d0, 0.d0, 8.d0, 0.d0, 0.d0, 3.d0, 3.d0, 1.d0, 1.d0, &
        -3.d0, -1.d0, -3.d0, -1.d0, 0.d0, 0.d0, 8.d0, 0.d0, 1.d0, 1.d0, 3.d0, 3.d0, &
        -1.d0, -3.d0, -1.d0, -3.d0, 0.d0, 0.d0, 0.d0, 8.d0, -1.d0, -1.d0, -3.d0, -3.d0, &
        3.d0, 1.d0, 3.d0, 1.d0, 0.d0, 0.d0, 8.d0, 0.d0, -1.d0, -1.d0, -3.d0, -3.d0, &
        1.d0, 3.d0, 1.d0, 3.d0, 0.d0, 0.d0, 0.d0, 8.d0, 1.d0, 1.d0, 3.d0, 3.d0, &
        3.d0, 3.d0, 1.d0, 1.d0, 3.d0, 3.d0, 1.d0, 1.d0, 8.d0, 0.d0, 0.d0, 0.d0, &
        -3.d0, -3.d0, -1.d0, -1.d0, -3.d0, -3.d0, -1.d0, -1.d0, 8.d0, 0.d0, 0.d0, 0.d0, &
        1.d0, 1.d0, 3.d0, 3.d0, -3.d0, -3.d0, -1.d0, -1.d0, 0.d0, 8.d0, 0.d0, 0.d0, &
        -1.d0, -1.d0, -3.d0, -3.d0, 3.d0, 3.d0, 1.d0, 1.d0, 0.d0, 8.d0, 0.d0, 0.d0, &
        -3.d0, -3.d0, -1.d0, -1.d0, 1.d0, 1.d0, 3.d0, 3.d0, 0.d0, 0.d0, 8.d0, 0.d0, &
        3.d0, 3.d0, 1.d0, 1.d0, -1.d0, -1.d0, -3.d0, -3.d0, 0.d0, 0.d0, 8.d0, 0.d0, &
        -1.d0, -1.d0, -3.d0, -3.d0, -1.d0, -1.d0, -3.d0, -3.d0, 0.d0, 0.d0, 0.d0, 8.d0, &
        1.d0, 1.d0, 3.d0, 3.d0, 1.d0, 1.d0, 3.d0, 3.d0, 0.d0, 0.d0, 0.d0, 8.d0 &
        /),(/12,24/) )

   newmX=1
   newmX(1:nDim)=2*mX(1:nDim)
   CALL cons_to_prim(qold(:,:,:,:))
   IF (nDim == 1) THEN
      ALLOCATE(slope(mx(1),1,1,1,NrVars))
      IF (lInterpolate) THEN
         FORALL(i=2:mx(1)-1, j=1:mx(2), k=1:mx(3), m=1:NrVars)
            slope(i,j,k,1,m)=C_1*(qold(i+1,j,k,m)-qold(i-1,j,k,m))
            slope(i,j,k,1,m)=sign(min(abs(qold(i+1,j,k,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i-1,j,k,m)),abs(slope(i,j,k,1,m))),slope(i,j,k,1,m))
         END FORALL
      ELSE
         slope=0d0
      END IF
      slope(1,:,:,1,:)=slope(2,:,:,1,:)
      slope(mx,:,:,1,:)=slope(mx-1,:,:,1,:)
      q(1:newmX(1):2, 1, 1,  :)=qold(1:mX(1),1,1,:)-slope(1:mX(1),1,1,1,:)
      q(2:newmX(1):2, 1, 1,  :)=qold(1:mX(1),1,1,:)+slope(1:mX(1),1,1,1,:)
   ELSE IF (nDim == 2) THEN
      ALLOCATE(slope(mx(1),mx(2),1,2,NrVars)) 

      IF (lInterpolate) THEN
         FORALL(i=2:mx(1)-1, j=1:mx(2), k=1:mx(3), m=1:NrVars)
            slope(i,j,k,1,m)=C_1*(qold(i+1,j,k,m)-qold(i-1,j,k,m))
            slope(i,j,k,1,m)=sign(min(abs(qold(i+1,j,k,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i-1,j,k,m)),abs(slope(i,j,k,1,m))),slope(i,j,k,1,m))
         END FORALL
         slope(1,:,:,1,:)=slope(2,:,:,1,:)
         slope(mx(1),:,:,1,:)=slope(mx(1)-1,:,:,1,:)
         FORALL(i=1:mx(1), j=2:mx(2)-1, k=1:mx(3), m=1:NrVars)
            slope(i,j,k,2,m)=C_1*(qold(i,j+1,k,m)-qold(i,j-1,k,m))
            slope(i,j,k,2,m)=sign(min(abs(qold(i,j+1,k,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i,j-1,k,m)),abs(slope(i,j,k,2,m))),slope(i,j,k,2,m))
         END FORALL
         slope(:,1,:,2,:)=slope(:,2,:,2,:)
         slope(:,mx(2),:,2,:)=slope(:, mx(2)-1,:,2,:)
      ELSE
         slope=0d0
      END IF
      q(1:newmX(1):2, 1:newmX(2):2, 1, :)=qold(1:mX(1),1:mX(2),1,:)-slope(1:mX(1),1:mX(2),1,1,:)-slope(1:mX(1),1:mX(2),1,2,:)
      q(1:newmX(1):2, 2:newmX(2):2, 1, :)=qold(1:mX(1),1:mX(2),1,:)-slope(1:mX(1),1:mX(2),1,1,:)+slope(1:mX(1),1:mX(2),1,2,:)
      q(2:newmX(1):2, 1:newmX(2):2, 1, :)=qold(1:mX(1),1:mX(2),1,:)+slope(1:mX(1),1:mX(2),1,1,:)-slope(1:mX(1),1:mX(2),1,2,:)
      q(2:newmX(1):2, 2:newmX(2):2, 1, :)=qold(1:mX(1),1:mX(2),1,:)+slope(1:mX(1),1:mX(2),1,1,:)+slope(1:mX(1),1:mX(2),1,2,:)

      IF (MaintainAuxArrays) THEN
         ALLOCATE(slope_aux(1:mx(1)+1, 1:mx(2)+1,1,2,2))
         IF (lInterpolate) THEN
            FORALL(i=1:mx(1)+1,j=2:mx(2)-1,k=1:mx(3))
               slope_aux(i,j,k,2,1)=C_1*(auxold(i,j+1,k,1,1)-auxold(i,j-1,k,1,1))
               slope_aux(i,j,k,2,1)=sign(min(abs(auxold(i,j+1,k,1,1)-auxold(i,j,k,1,1)),abs(auxold(i,j,k,1,1)-auxold(i,j-1,k,1,1)),abs(slope_aux(i,j,k,2,1))),slope_aux(i,j,k,2,1))
            END FORALL
            slope_aux(:,1,:,2,1)=slope_aux(:,2,:,2,1)
            slope_aux(:,mx(2),:,2,1)=slope_aux(:,mx(2)-1,:,2,1)
            
            FORALL(i=2:mx(1)-1,j=1:mx(2)+1,k=1:mx(3))
               slope_aux(i,j,k,1,2)=C_1*(auxold(i+1,j,k,1,2)-auxold(i-1,j,k,1,2))
               slope_aux(i,j,k,1,2)=sign(min(abs(auxold(i+1,j,k,1,2)-auxold(i,j,k,1,2)),abs(auxold(i,j,k,1,2)-auxold(i-1,j,k,1,2)),abs(slope_aux(i,j,k,1,2))),slope_aux(i,j,k,1,2))
            END FORALL
            slope_aux(1,:,:,1,2)=slope_aux(2,:,:,1,2)
            slope_aux(mx(1),:,:,1,2)=slope_aux(mx(1)-1,:,:,1,2)
         ELSE
            slope_aux=0d0
         END IF
         aux(1:newmX(1)+1:2,1:newmX(2):2,1, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1,1,1)-slope_aux(1:mX(1)+1,1:mX(2),1,2,1)
         aux(1:newmX(1)+1:2,2:newmX(2):2,1, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1,1,1)+slope_aux(1:mX(1)+1,1:mX(2),1,2,1)
         aux(1:newmX(1)+1:2,1:newmX(2)+1:2,1, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1,1,2)-slope_aux(1:mX(1),1:mX(2)+1,1,1,2)
         aux(2:newmX(1)+1:2,1:newmX(2)+1:2,1, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1,1,2)+slope_aux(1:mX(1),1:mX(2)+1,1,1,2)
         DO i=1,newmX(1),2
            DO j=1,newmX(2),2
               Bext2D=(/ aux(i,j,1,1,1),aux(i,j+1,1,1,1),aux(i+2,j,1,1,1),aux(i+2,j+1,1,1,1),&
                    aux(i,j,1,1,2),aux(i,j+2,1,1,2),aux(i+1,j,1,1,2),aux(i+1,j+2,1,1,2) /)
               aux(i+1,j,1,1,1) = DOT_PRODUCT(A2D(1,1:8),Bext2D(1:8))
               aux(i+1,j+1,1,1,1) = DOT_PRODUCT(A2D(2,1:8),Bext2D(1:8))
               aux(i,j+1,1,1,2) = DOT_PRODUCT(A2D(3,1:8),Bext2D(1:8))
               aux(i+1,j+1,1,1,2) = DOT_PRODUCT(A2D(4,1:8),Bext2D(1:8))
            END DO
         END DO
      END IF
   ELSE IF (nDim == 3) THEN
      ALLOCATE(slope(mx(1),mx(2),mx(3),3,NrVars))
      IF (lInterpolate) THEN
         FORALL(i=2:mx(1)-1, j=1:mx(2), k=1:mx(3), m=1:NrVars)
            slope(i,j,k,1,m)=C_1*(qold(i+1,j,k,m)-qold(i-1,j,k,m))
            slope(i,j,k,1,m)=sign(min(abs(qold(i+1,j,k,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i-1,j,k,m)),abs(slope(i,j,k,1,m))),slope(i,j,k,1,m))
         END FORALL
         slope(1,:,:,1,:)=slope(2,:,:,1,:)
         slope(mx(1),:,:,1,:)=slope(mx(1)-1,:,:,1,:)
         FORALL(i=1:mx(1), j=2:mx(2)-1, k=1:mx(3), m=1:NrVars)
            slope(i,j,k,2,m)=C_1*(qold(i,j+1,k,m)-qold(i,j-1,k,m))
            slope(i,j,k,2,m)=sign(min(abs(qold(i,j+1,k,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i,j-1,k,m)),abs(slope(i,j,k,2,m))),slope(i,j,k,2,m))
         END FORALL
         slope(:,1,:,2,:)=slope(:,2,:,2,:)
         slope(:,mx(2),:,2,:)=slope(:,mx(2)-1,:,2,:)
         FORALL(i=1:mx(1), j=1:mx(2), k=2:mx(3)-1, m=1:NrVars)
            slope(i,j,k,3,m)=C_1*(qold(i,j,k+1,m)-qold(i,j,k-1,m))
            slope(i,j,k,3,m)=sign(min(abs(qold(i,j,k+1,m)-qold(i,j,k,m)),abs(qold(i,j,k,m)-qold(i,j,k-1,m)),abs(slope(i,j,k,3,m))),slope(i,j,k,3,m))
         END FORALL
         slope(:,:,1,3,:)=slope(:,:,2,3,:)
         slope(:,:,mx(3),3,:)=slope(:,:,mx(3)-1,3,:)
      ELSE
         slope=0d0
      END IF
      !     slope=0d0
      q(1:newmX(1):2, 1:newmX(2):2, 1:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)-slope(1:mX(1),1:mX(2),1:mX(3),3,:)-slope(1:mX(1),1:mX(2),1:mX(3),1,:)-slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(1:newmX(1):2, 2:newmX(2):2, 1:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)-slope(1:mX(1),1:mX(2),1:mX(3),3,:)-slope(1:mX(1),1:mX(2),1:mX(3),1,:)+slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(2:newmX(1):2, 1:newmX(2):2, 1:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)-slope(1:mX(1),1:mX(2),1:mX(3),3,:)+slope(1:mX(1),1:mX(2),1:mX(3),1,:)-slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(2:newmX(1):2, 2:newmX(2):2, 1:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)-slope(1:mX(1),1:mX(2),1:mX(3),3,:)+slope(1:mX(1),1:mX(2),1:mX(3),1,:)+slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(1:newmX(1):2, 1:newmX(2):2, 2:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)+slope(1:mX(1),1:mX(2),1:mX(3),3,:)-slope(1:mX(1),1:mX(2),1:mX(3),1,:)-slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(1:newmX(1):2, 2:newmX(2):2, 2:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)+slope(1:mX(1),1:mX(2),1:mX(3),3,:)-slope(1:mX(1),1:mX(2),1:mX(3),1,:)+slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(2:newmX(1):2, 1:newmX(2):2, 2:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)+slope(1:mX(1),1:mX(2),1:mX(3),3,:)+slope(1:mX(1),1:mX(2),1:mX(3),1,:)-slope(1:mX(1),1:mX(2),1:mX(3),2,:)
      q(2:newmX(1):2, 2:newmX(2):2, 2:newmX(3):2, :)=qold(1:mX(1),1:mX(2),1:mX(3),:)+slope(1:mX(1),1:mX(2),1:mX(3),3,:)+slope(1:mX(1),1:mX(2),1:mX(3),1,:)+slope(1:mX(1),1:mX(2),1:mX(3),2,:)

      IF (MaintainAuxArrays) THEN
         ALLOCATE(slope_aux(1:mx(1)+1, 1:mx(2)+1,1:mx(3)+1,3,3))
         IF (lInterpolate) THEN
            FORALL(i=1:mx(1)+1,j=2:mx(2)-1,k=1:mx(3))
               slope_aux(i,j,k,2,1)=C_1*(auxold(i,j+1,k,1,1)-auxold(i,j-1,k,1,1))
               slope_aux(i,j,k,2,1)=sign(min(abs(auxold(i,j+1,k,1,1)-auxold(i,j,k,1,1)),abs(auxold(i,j,k,1,1)-auxold(i,j-1,k,1,1)),abs(slope_aux(i,j,k,2,1))),slope_aux(i,j,k,2,1))
            END FORALL
            slope_aux(:,1,:,2,1)=slope_aux(:,2,:,2,1)
            slope_aux(:,mx(2),:,2,1)=slope_aux(:,mx(2)-1,:,2,1)
            
            FORALL(i=1:mx(1)+1,j=1:mx(2),k=2:mx(3)-1)
               slope_aux(i,j,k,3,1)=C_1*(auxold(i,j,k+1,1,1)-auxold(i,j,k-1,1,1))
               slope_aux(i,j,k,3,1)=sign(min(abs(auxold(i,j,k+1,1,1)-auxold(i,j,k,1,1)),abs(auxold(i,j,k,1,1)-auxold(i,j,k-1,1,1)),abs(slope_aux(i,j,k,3,1))),slope_aux(i,j,k,3,1))
            END FORALL
            slope_aux(:,:,1,3,1)=slope_aux(:,:,2,3,1)
            slope_aux(:,:,mx(3),3,1)=slope_aux(:,:,mx(3)-1,3,1)
            
            FORALL(i=1:mx(1),j=1:mx(2)+1,k=2:mx(3)-1)
               slope_aux(i,j,k,3,2)=C_1*(auxold(i,j,k+1,1,2)-auxold(i,j,k-1,1,2))
               slope_aux(i,j,k,3,2)=sign(min(abs(auxold(i,j,k+1,1,2)-auxold(i,j,k,1,2)),abs(auxold(i,j,k,1,2)-auxold(i,j,k-1,1,2)),abs(slope_aux(i,j,k,3,2))),slope_aux(i,j,k,3,2))
            END FORALL
            slope_aux(:,:,1,3,2)=slope_aux(:,:,2, 3, 2)
            slope_aux(:,:,mx(3),3,2)=slope_aux(:,:,mx(3)-1, 3, 2)
            
            FORALL(i=2:mx(1)-1,j=1:mx(2)+1,k=1:mx(3))
               slope_aux(i,j,k,1,2)=C_1*(auxold(i+1,j,k,1,2)-auxold(i-1,j,k,1,2))
               slope_aux(i,j,k,1,2)=sign(min(abs(auxold(i+1,j,k,1,2)-auxold(i,j,k,1,2)),abs(auxold(i,j,k,1,2)-auxold(i-1,j,k,1,2)),abs(slope_aux(i,j,k,1,2))),slope_aux(i,j,k,1,2))
            END FORALL
            slope_aux(1,:,:,1,2)=slope_aux(2,:,:, 1,2)
            slope_aux(mx(1),:,:,1,2)=slope_aux(mx(1)-1,:,:, 1,2)
            
            FORALL(i=2:mx(1)-1,j=1:mx(2),k=1:mx(3)+1)
               slope_aux(i,j,k,1,3)=C_1*(auxold(i+1,j,k,1,3)-auxold(i-1,j,k,1,3))
               slope_aux(i,j,k,1,3)=sign(min(abs(auxold(i+1,j,k,1,3)-auxold(i,j,k,1,3)),abs(auxold(i,j,k,1,3)-auxold(i-1,j,k,1,3)),abs(slope_aux(i,j,k,1,3))),slope_aux(i,j,k,1,3))
            END FORALL
            slope_aux(1,:,:,1,3)=slope_aux(2,:,:,1,3)
            slope_aux(mx(1),:,:,1,3)=slope_aux(mx(1)-1,:,:,1,3)
            
            FORALL(i=1:mx(1),j=2:mx(2)-1, k=1:mx(3)+1)
               slope_aux(i,j,k,2,3)=C_1*(auxold(i,j+1,k,1,3)-auxold(i,j-1,k,1,3))
               slope_aux(i,j,k,2,3)=sign(min(abs(auxold(i,j+1,k,1,3)-auxold(i,j,k,1,3)),abs(auxold(i,j,k,1,3)-auxold(i,j-1,k,1,3)),abs(slope_aux(i,j,k,2,3))),slope_aux(i,j,k,2,3))
            END FORALL
            slope_aux(:,1,:,2,3)=slope_aux(:,1,:,2,3)
            slope_aux(:,mx(2),:,2,3)=slope_aux(:,mx(2)-1,:,2,3)
         ELSE
            slope_aux=0d0
         END IF

         aux(1:newmX(1)+1:2,1:newmX(2):2,1:newmX(3):2, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1:mX(3),1,1)-slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),3,1)-slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),2,1)
         aux(1:newmX(1)+1:2,2:newmX(2):2,1:newmX(3):2, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1:mX(3),1,1)-slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),3,1)+slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),2,1)
         aux(1:newmX(1)+1:2,1:newmX(2):2,2:newmX(3):2, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1:mX(3),1,1)+slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),3,1)-slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),2,1)
         aux(1:newmX(1)+1:2,2:newmX(2):2,2:newmX(3):2, 1, 1)=auxold(1:mX(1)+1,1:mX(2),1:mX(3),1,1)+slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),3,1)+slope_aux(1:mX(1)+1,1:mX(2),1:mX(3),2,1)

         aux(1:newmX(1):2,1:newmX(2)+1:2,1:newmX(3):2, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1:mX(3),1,2)-slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),3,2)-slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),1,2)
         aux(2:newmX(1):2,1:newmX(2)+1:2,1:newmX(3):2, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1:mX(3),1,2)-slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),3,2)+slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),1,2)
         aux(1:newmX(1):2,1:newmX(2)+1:2,2:newmX(3):2, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1:mX(3),1,2)+slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),3,2)-slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),1,2)
         aux(2:newmX(1):2,1:newmX(2)+1:2,2:newmX(3):2, 1, 2)=auxold(1:mX(1),1:mX(2)+1,1:mX(3),1,2)+slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),3,2)+slope_aux(1:mX(1),1:mX(2)+1,1:mX(3),1,2)

         aux(1:newmX(1):2,1:newmX(2):2,1:newmX(3)+1:2, 1, 3)=auxold(1:mX(1),1:mX(2),1:mX(3)+1,1,3)-slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,2,3)-slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,1,3)
         aux(2:newmX(1):2,1:newmX(2):2,1:newmX(3)+1:2, 1, 3)=auxold(1:mX(1),1:mX(2),1:mX(3)+1,1,3)-slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,2,3)+slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,1,3)
         aux(1:newmX(1):2,2:newmX(2):2,1:newmX(3)+1:2, 1, 3)=auxold(1:mX(1),1:mX(2),1:mX(3)+1,1,3)+slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,2,3)-slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,1,3)
         aux(2:newmX(1):2,2:newmX(2):2,1:newmX(3)+1:2, 1, 3)=auxold(1:mX(1),1:mX(2),1:mX(3)+1,1,3)+slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,2,3)+slope_aux(1:mX(1),1:mX(2),1:mX(3)+1,1,3)
         DO i=1,newmX(1),2
            DO j=1,newmX(2),2
               DO k=1,newmX(3),2
                  Bext3D=(/ aux(i,j,k,1,1),aux(i,j,k+1,1,1),aux(i,j+1,k,1,1),aux(i,j+1,k+1,1,1),&
                       aux(i+2,j,k,1,1),aux(i+2,j,k+1,1,1),aux(i+2,j+1,k,1,1),aux(i+2,j+1,k+1,1,1),&
                       aux(i,j,k,1,2),aux(i,j,k+1,1,2),aux(i,j+2,k,1,2),aux(i,j+2,k+1,1,2),&
                       aux(i+1,j,k,1,2),aux(i+1,j,k+1,1,2),aux(i+1,j+2,k,1,2) ,aux(i+1,j+2,k+1,1,2),&
                       aux(i,j,k,1,3),aux(i,j,k+2,1,3),aux(i,j+1,k,1,3),aux(i,j+1,k+2,1,3),&
                       aux(i+1,j,k,1,3),aux(i+1,j,k+2,1,3),aux(i+1,j+1,k,1,3) ,aux(i+1,j+1,k+2,1,3) /)

                  aux(i+1,j  ,k  ,1,1) = DOT_PRODUCT(A3D(1, 1:24),Bext3D(1:24))
                  aux(i+1,j  ,k+1,1,1) = DOT_PRODUCT(A3D(2, 1:24),Bext3D(1:24))
                  aux(i+1,j+1,k  ,1,1) = DOT_PRODUCT(A3D(3, 1:24),Bext3D(1:24))
                  aux(i+1,j+1,k+1,1,1) = DOT_PRODUCT(A3D(4, 1:24),Bext3D(1:24))
                  aux(i  ,j+1,k  ,1,2) = DOT_PRODUCT(A3D(5, 1:24),Bext3D(1:24))
                  aux(i  ,j+1,k+1,1,2) = DOT_PRODUCT(A3D(6, 1:24),Bext3D(1:24))
                  aux(i+1,j+1,k  ,1,2) = DOT_PRODUCT(A3D(7, 1:24),Bext3D(1:24))
                  aux(i+1,j+1,k+1,1,2) = DOT_PRODUCT(A3D(8, 1:24),Bext3D(1:24))
                  aux(i  ,j  ,k+1,1,3) = DOT_PRODUCT(A3D(9, 1:24),Bext3D(1:24))
                  aux(i  ,j+1,k+1,1,3) = DOT_PRODUCT(A3D(10,1:24),Bext3D(1:24))
                  aux(i+1,j  ,k+1,1,3) = DOT_PRODUCT(A3D(11,1:24),Bext3D(1:24))
                  aux(i+1,j+1,k+1,1,3) = DOT_PRODUCT(A3D(12,1:24),Bext3D(1:24))
               END DO
            END DO
         END DO
      END IF
   END IF
   DEALLOCATE(slope)
   IF (MaintainAuxArrays) DEALLOCATE(slope_aux)

   IF (MaintainAuxArrays) THEN
      IF (nDim >= 2) THEN
         FORALL(i=1:newmX(1),j=1:newmX(2),k=1:newmX(3))
            q(i,j,k,iBx)=half*(aux(i,j,k,1,1)+aux(i+1,j,k,1,1))
            q(i,j,k,iBy)=half*(aux(i,j,k,1,2)+aux(i,j+1,k,1,2))
         END FORALL
         IF (nDim >= 3) THEN
            FORALL(i=1:newmX(1),j=1:newmX(2),k=1:newmX(3))
               q(i,j,k,iBz)=half*(aux(i,j,k,1,3)+aux(i,j,k+1,1,3))
            END FORALL
         END IF
      END IF
   END IF
   CALL prim_to_cons(q(1:newmX(1),1:newmX(2),1:newmX(3),:))
END SUBROUTINE interpolate


END MODULE BEARIO
