!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    cool.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!============================================================================
!                  BEARCLAW Astrophysical Applications
!============================================================================
! Andrew Cunningham, Alexei Poludnenko, Adam Frank
! Department of Physics and Astronomy
! University of Rochester
! Rochester, NY
!----------------------------------------------------------------------------
! File:             cool.f90
! Type:             module
! Purpose:          molecular cooling and ionization (below T=10^4)
!                   inhomogeneous terms plus integration scheme
! Revision History: Ver. 1.0 July 2003 A. Cunningham
!                   Ver. 2.0 July 2003 A. Cunningham
!                      Added H2 cooling for high HI, density, critical density ~ 800/cc 
! -----------------------------------------------------------------------
! Features:
! H ionization and recombination for T < 10^5 (may be used to take over for bbc below 10^4 ...)
! H2 cooling for T < 10^6
! H2 dissociation recombination T < 10^5
! OI cooling using H2 as tracer for OI T < 1000
! dust cooling
! He+ ion recombination for T < 10^5 (may be used to take over for bbc below 10^4 ...)
!
MODULE cool  
  USE GlobalDeclarations,ONLY:cellinfo,&
       srcPrecision,& !real array    (-1:32)
       muH,muH2,muHe,mue,Xmu,boltzmann,ev,amu,&  !real constants
       BindH2,IonH,IonHe,gamma,gammac,gammaH,gammaH2,&!real
       rScale,nScale,TempScale,pScale,lScale,VelScale,&!real
       Lumin,mCentral,alpha,MinTemp,RunTimesc,ViscCD,ScaleGrav,ScaleCool,EOSConstants,&!real
       verbosity,&    !integer array (3)
       NrVars,nDim,iCooling,iEOS,iCylindrical,&  !integer
       iH2,iH,iHII,iHaux,iHe,iHeII,iHeIII,iE,iEntropy,nSpecies,nSpeciesLO,nSpeciesHI,iSpeedHI,&!interger indecies
       lH,lH2,lHII,lHe,lHeII,lHeIII!logical

  IMPLICIT NONE
  PRIVATE
  INTEGER, PARAMETER :: qPrec = SELECTED_REAL_KIND(14,32)
  INTEGER, PARAMETER :: xPrec = SELECTED_REAL_KIND(14,32)
  REAL(KIND=qprec), PUBLIC ::  nmine, &       ! minimum allowed electron density
                               Tdust, &       ! dust temp
                               Tfloor, &
                               OIFrac, &      ! nOatom/nHneucei
                               metalFrac      ! nMetals/nHneucei

  LOGICAL,PUBLIC :: lcooling, & ! for debugging, turn cooling off and alow state to evolove
                    lBBC, &     ! use BBC?
                    lProtectNeq, &
                    lchemion

  NAMELIST /CoolingData/ nmine, Tdust, Tfloor, OIFrac, metalFrac, &
                         lBBC, lcooling, lProtectNeq,lchemion
  NAMELIST /SpeciesData/  lH,lH2,lHII,lHe,lHeII,lHeIII
  REAL(KIND=qprec), PARAMETER :: PI = 3.141592654d0
  REAL(KIND=qprec),PUBLIC, PARAMETER :: zero=0.d0,one=1.d0,half=0.5d0
  ! delgarno mccray cooling table
  REAL (KIND=qPrec), DIMENSION(:), ALLOCATABLE :: DMCoolingTab

  REAL(KIND=qprec), PUBLIC, PARAMETER :: small=1.d-6

  ! Tables
  ! tables for analytic rate functions that can be split into temperature dependent part only
  ! table look ups are faster than raising to non-integer powers and divisions
  ! min_TAB and max_TAB are in units of log10(temp K)
  INTEGER,PARAMETER :: n_TAB=2048,min_TAB=0,max_TAB=6
  REAL(KIND=qprec),PARAMETER :: rn_TAB=n_TAB, delta_TAB=(max_TAB-min_TAB)/rn_TAB
  REAL(KIND=qprec), DIMENSION(n_TAB) :: H2_dust_Recomb_TAB,&
                                        H2_critical_cool_TAB,&
                                        dust_cool_TAB,&
                                        H_recomb_TAB,&
                                        H_ioniz_TAB,&
                                        He_ioniz_TAB,&
                                        He_recomb_TAB
  PUBLIC :: OI_cool,H2_cool,H2_critical_cool,dust_cool,H2_diss,H2_dust_Recomb,H_recomb,H_ioniz,He_ioniz,He_recomb,&
       get_mu,get_gamma,initneqCool,DMCoolingRate,EOS_vars

CONTAINS
  ! Parameter LBBC tells if this module will be used with BCC ionization / cooling
  SUBROUTINE initneqCool
    INTEGER :: iErr,i
    
    IF(Xmu/=1.0d0) THEN
       PRINT*, "cool.f90 initneqcool WARNING: Xmu must equal 1.0 for non equillibrium cooling routines",Xmu
       STOP
    END IF
    ! Read in cooling parameters
     OPEN(UNIT=10,IOSTAT=iErr,ERR=50,FILE='neqcool.data',STATUS='old',      &
             FORM='formatted')
50   IF (iErr /= 0) THEN
        PRINT *,'!!! cool.f90 ERROR: Error opening file "neqcool.data"'
        STOP
     END IF
     READ(10,NML=CoolingData,IOSTAT=iErr,ERR=100)
     READ(10,NML=SpeciesData,IOSTAT=iErr,ERR=101)
100   IF (iErr /= 0) THEN
        PRINT *,'!!! cool.f90 ERROR: Error reading Data from file ',       &
                '"neqcool.data", CoolingData'
        STOP
     END IF
101   IF (iErr /= 0) THEN
        PRINT *,'!!! cool.f90 ERROR: Error reading Data from file ',       &
                '"neqcool.data", SpeciesData'
        STOP
     END IF
     CLOSE(10)

     IF(.NOT. LBBC) CALL  InitDMCool

     nSpecieslo=MAX(iEntropy,iE)+1

     iH2=0.;iH=0.;iHII=0.;iHe=0.;iHeII=0.;iHeIII=0.
     i = nSpecieslo
     iHaux = nSpecieslo
     IF(lH) THEN
        iH = i
        i = i+1
        PRINT*,'iH=',iH
     ELSE
        iHaux = iHaux - 1
     END IF
     IF(lH2) THEN
        iH2 = i
        i = i+1
        PRINT*,'iH2=',iH2
     END IF
     IF(lHII) THEN
        iHII = i
        i = i+1
        PRINT*,'iHII=',iHII
     END IF
     IF(lHe) THEN
        iHe = i
        i = i+1
        PRINT*,'iHe=',iHe
     END IF
     IF(lHeII) THEN
        iHeII = i
        i = i+1
        PRINT*,'iHeII=',iHeII
     END IF
     IF(lHeIII) THEN
        iHeIII = i
        i = i+1
        PRINT*,'iHeIII=',iHeIII
     END IF
     nSpecieshi = i-1
     nSpecies = nSpecieshi-nSpecieslo+1

     ! build the tabulated forms of the cooling / ionization rates
     FORALL(i=1:n_TAB) H2_dust_Recomb_TAB(i) = H2_dust_Recomb(1.d0,1.d0,10**((i-1)*delta_TAB+min_TAB),Tdust)
     FORALL(i=1:n_TAB) H_recomb_TAB(i) = H_recomb(1.d0,1.d0,10**((i-1)*delta_TAB+min_TAB))
     DO i=1,n_TAB; H_ioniz_TAB(i) = H_ioniz(1.d0,1.d0,10**((i-1)*delta_TAB+min_TAB)); END DO
     DO i=1,n_TAB; He_ioniz_TAB(i) = He_ioniz(1.d0,1.d0,10**((i-1)*delta_TAB+min_TAB)); END DO
     FORALL(i=1:n_TAB) He_recomb_TAB(i) = He_recomb(1.d0,1.d0,10**((i-1)*delta_TAB+min_TAB))
     FORALL(i=1:n_TAB) H2_critical_cool_TAB(i) = H2_critical_cool(1.d0,10**((i-1)*delta_TAB+min_TAB))
     FORALL(i=1:n_TAB) dust_cool_TAB(i) = dust_cool(1.d0,10**((i-1)*delta_TAB+min_TAB))
  END SUBROUTINE initneqCool

  ! returns number density vector for each species
  ! if present, returns temperature in K, everything else in computational units
  ! NOTE : -emicro is the thermal + magnetic energy density
  !        -eth is thermal energy only
  !        -gammais = (dP/drho)*(rho/P) 
  !        -gamma = 1+P/(rho e)
  !        The gammas are not the same for the case of a Real Gas (Colella & Glaz 84).
  !        gammais =  gamma for a polytropic gas
  !        I have invoked a Real Gas EOS to provide a component of magnetic pressure to 
  !        support the collapse of strongly cooled shocks in pure hydro sumulations
  PURE SUBROUTINE EOS_vars(q,nvec,T,mu,gamma,gammais,eth,emicro,ke)
    IMPLICIT NONE
    REAL(KIND=qprec), DIMENSION(1:NrVars), INTENT(IN) :: q
    REAL(KIND=qprec), DIMENSION(1:NrVars) :: qin
    REAL(KIND=qprec), DIMENSION(iHaux:nSpeciesHi), INTENT(out), OPTIONAL :: nvec
    REAL(KIND=qprec), INTENT(OUT), OPTIONAL :: T,mu,gamma,gammais,eth,emicro,ke
    ! Internal variables
    REAL(KIND=qPrec) :: kein,ethin,emicroin
    REAL(KIND=qprec), DIMENSION(iHaux:nSpeciesHi) :: nvecin
    REAL(KIND=qprec) ::gammain,muin
    INTEGER :: ndim_in

    nvecin = 0.d0
    qin = q
    IF(lH2) nvecin(iH2) = qin(iH2)/muH2
    IF(lHII) nvecin(iHII) = qin(iHII)/muH
    IF(lHe) nvecin(iHe) = qin(iHe)/muHe
    IF(lHeII) nvecin(iHeII) = qin(iHeII)/muHe
    IF(lHeIII) nvecin(iHeIII) = qin(iHeIII)/muHe
    IF(lH) THEN
       nvecin(iH) = qin(iH)/muH
    ELSE 
       ! if we are not tracking H expiclitly, then it must be what is left over
       nvecin(iHaux) = (qin(1)-SUM(qin(nSpecieslo:nspecieshi)))/muH
    END IF
    nvecin = nvecin*nScale
    nvecin = MAX(nvecin,zero)
    IF(PRESENT(nvec)) nvec = nvecin(iHaux:nSpeciesHi)

    IF(PRESENT(T) .OR. PRESENT(mu) .OR. PRESENT(gamma) .OR. PRESENT(gammais) .OR. PRESENT(eth) &
         .OR. PRESENT(emicro) .OR. PRESENT(ke)) THEN
       muin =  get_mu(nvecin(iHaux:nSpeciesHi))
       gammain = get_gamma(nvecin(iHaux:nSpeciesHi))
       IF(PRESENT(gamma)) gamma=gammain
       IF(PRESENT(mu)) mu=muin
       ! cooling via EOS
       kein = half*DOT_PRODUCT(qin(2:iSpeedHI),qin(2:iSpeedHI))/(qin(1))
       IF(PRESENT(ke)) ke = kein
       ethin = qin(iE) - kein
       ! effective isentropic gamma
       IF(PRESENT(gammais)) gammais=gammain
       IF(PRESENT(T) .OR. PRESENT(eth) .OR. PRESENT(emicro)) THEN
          emicroin = ethin
          IF(PRESENT(emicro)) emicro = ethin
          IF(PRESENT(eth)) eth = ethin
          IF(PRESENT(T)) THEN
             T = EOSConstants*muin*(gammain-one)*(ethin)/qin(1)
          END IF
       END IF
    END IF
  END SUBROUTINE EOS_vars

  PURE FUNCTION get_gamma(nvec)
    REAL(KIND=qprec),INTENT(IN), DIMENSION(iHaux:nSpeciesHi) :: nvec
    REAL(KIND=qprec) get_gamma,n,ne,nH2

    CALL nparticle(nvec,npart=n)
    IF(lH2) THEN
       nH2 = nvec(iH2) 
    ELSE 
       nH2=0.d0
    END IF
    IF(nH2 < small) THEN
       get_gamma = gammaH
    ELSE 
       get_gamma = ((n-nH2)/n*gammaH/(gammaH-1.d0) + nH2/n*gammaH2/(gammaH2-1.d0)) / &
            ((n-nH2)/n*1.d0/(gammaH-1.d0) + nH2/n*1.d0/(gammaH2-1.d0))
    END IF
  END FUNCTION get_gamma

  PURE FUNCTION get_mu(nvec)
    REAL(KIND=qprec),INTENT(IN), DIMENSION(iHaux:nSpeciesHi) :: nvec
    REAL(KIND=qprec), DIMENSION(0:nSpeciesHi) :: nvecin
    REAL(KIND=qprec) get_mu,n,ne

    nvecin=0.
    nvecin(iHaux:nSpeciesHi)=nvec(iHaux:nSpeciesHi)

    CALL nparticle(nvec,npart=n) 
    
    IF(n .LT. small) THEN
       get_mu=1.0
       RETURN
    END IF
    ! note: we assume zero electron mass in this calculation
    !       ionization processes affect the mean molecular weight of the
    !       gas dramaticly, since energy is conserved, this reduces the
    !       temperature of the gas 
    ! ie) mu ~ 0.5 for a HI gas
    get_mu = (nvecin(iH2)*muH2 + (nvecin(iH) + nvecin(iHII))*muH + &
         (nvecin(iHe) + nvecin(iHeII) + nvecin(iHeIII))*muHe)/n !AMU's
  END FUNCTION get_mu

  PURE SUBROUTINE nparticle(nvec,npart,nneuc,ne)
    REAL(KIND=qprec), DIMENSION(iHaux:nSpeciesHi),INTENT(IN) :: nvec
    REAL(KIND=qprec), DIMENSION(0:nSpeciesHi) :: nvecin
    REAL(KIND=qprec), INTENT(OUT),OPTIONAL :: npart,nneuc,ne
    ! internal veriables
    REAL(KIND=qprec) :: npartin,nneucin,nein

    nvecin=0.
    nvecin(iHaux:nSpeciesHi)=nvec

    nneucin = nvecin(iH2)+nvecin(iH)+nvecin(iHII)+nvecin(iHe)+nvecin(iHeII)+nvecin(iHeIII)
     
    ! # of electrons
    nein = nvecin(iHII) + nvecin(iHeII) + 2.d0*nvecin(iHeIII)+nmine*nneucin
    ! # of particles in gas, including electrons
    npartin = nneucin + nein
    ! # of neuclei (either molecular, neutral, or ionized
    nneucin = nneucin + nvecin(iH2)

    IF(PRESENT(ne)) ne = nein
    IF(PRESENT(nneuc)) nneuc = nneucin
    IF(PRESENT(npart)) npart = npartin
  END SUBROUTINE nparticle

  ! computes energy source term from H2 radiative cooling
  ! See : Lepp & Schull 1983 (1983ApJ...270..578L), 
  !       O'Sullivan & Ray 2000 (2000A&A...363..355O)
  PURE FUNCTION H2_cool(nH2, nH, T) !erg*cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nH, nH2, T
    REAL(KIND=qprec) H2_cool,Tin,heat_H2_cool
    
    IF(nH2 < small) THEN
       H2_cool=0.
    ELSE
       ! this function is only good to 10^6
       Tin = min(1.d6,T)
       heat_H2_cool = (Lvh(MinTemp)/(1.+Lvh(MinTemp)/Lvl(nH2,nH,MinTemp)) + Lrh(MinTemp)/(1.+Lrh(MinTemp)/Lrl(nH2,nH,MinTemp)))
       H2_cool = nH2*(Lvh(Tin)/(1.+Lvh(Tin)/Lvl(nH2,nH,Tin)) + Lrh(Tin)/(1.+Lrh(Tin)/Lrl(nH2,nH,Tin)) - heat_H2_cool)
    END IF
  CONTAINS
    PURE FUNCTION Lvh(T) ! ergs/s
      REAL(KIND=qprec),INTENT(IN)::T
      REAL(KIND=qprec) Lvh
      ! Smith & Rosen 2003 correct error of Lepp & Schull 1.1d-18, NOT 1.1d-13
      Lvh = 1.10d-18*exp(-6744./T)
    END FUNCTION Lvh
    PURE FUNCTION Lvl(nH2,nh,T) ! ergs/s
      REAL(KIND=qprec),INTENT(IN)::nh,nH2,T
      REAL(KIND=qprec) Lvl
      Lvl = 8.18d-13*(nH*kH(T)+nH2*kH2(T))
    END FUNCTION Lvl
    PURE FUNCTION Qfunc(nH2,nH)
      REAL(KIND=qprec),INTENT(IN)::nH2,nH
      REAL(KIND=qprec) Qfunc!,nH2in,nHin
      !nH2in = max(0.d0,nH2)
      !nHin = max(0.d0,nH)
      Qfunc = nH2**.77 + 1.2*nH**.77
    END FUNCTION Qfunc
    PURE FUNCTION kh2(T) ! cm^3/s
      REAL(KIND=qprec),INTENT(IN)::T
      REAL(KIND=qprec) kh2
      ! deltaE =  v=0->1 energy gap of H2 in eV
      !        nH is not zero. 
      REAL(KIND=qprec),PARAMETER :: deltaE = 0.785
      kh2 = 1.54d-12*sqrt(T)*exp(-4.2*deltaE*ev/(boltzmann*(T+1190.)))
    END FUNCTION kh2
    ! piecewise functions:
    PURE FUNCTION kh(T) ! cm^3/s
      REAL(KIND=qprec),INTENT(IN)::T
      REAL(KIND=qprec) kh
      IF(T > 1635.) THEN
         kh = 1.0d-12*sqrt(T)*exp(-1000./T)
      ELSE
         kh = 1.4e-13*exp(T/125. - (T/577.)**2)
      END IF
    END FUNCTION kh

    PURE FUNCTION Lrh(T) ! ergs/s
      REAL(KIND=qprec),INTENT(IN)::T
      REAL(KIND=qprec) Lrh,x
      IF(T > 1087.) THEN
         Lrh = 3.90d-19*exp(-6118.d0/T) 
      ELSE
         x = log10(T/1.d4)
         Lrh = 10**(-19.24d0 + .0474d0*x - 1.247d0*x**2)
      END IF
    END FUNCTION Lrh

    PURE FUNCTION Lrl(nH2,nH,T) ! ergs/s
      REAL(KIND=qprec),INTENT(IN):: nH2, nH, T
      REAL(KIND=qprec) Lrl,x
      IF(T > 4031.) THEN
         Lrl = 1.38d-22*exp(-9243.d0/T)
      ELSE
         x = log10(T/1.d4)
         Lrl = 10**(-22.90d0 - .553d0*x - 1.148d0*x**2)
      END IF
      Lrl = LrL*Qfunc(nH2,nH)
    END FUNCTION Lrl
  END FUNCTION H2_cool

  PURE FUNCTION H2_critical_cool(nH2, T) !erg*cm^-3*s^-1
    REAL(KIND=qprec) :: H2_critical_cool, beta, exp_beta
    REAL(KIND=qprec),INTENT(IN)::nH2, T
    INTEGER :: j,v
    REAL(KIND=qprec), PARAMETER :: hc=6.6260755d-27*2.99792458d10
    !Rotational wavelengths & einstien coefficients for pure rotational v=0, S branch as function of J (rotation number)
    REAL(KIND=qprec), PARAMETER, DIMENSION(0:15) :: &
         A=(/2.94d-11,4.76d-10,2.76d-9,9.84d-9,2.64d-8,5.88d-8,1.14d-7,2.00d-7,&
             3.24d-7,4.90d-7,7.03d-7,9.64d-7,1.27d-6,1.62d-6,2.00d-6,2.41d-6 /), &
         lam=(/28.2240d-4,17.0378d-4,12.2806d-4,9.6662d-4,8.0255d-4,6.9087d-4,6.1056d-4,5.5045d-4, &
               5.0403d-4,4.6720d-4,4.3725d-4,4.1223d-4,3.9069d-4,3.7150d-4,3.5374d-4,3.3664d-4 /)
    ! I only include three brightest rotation-vibration lines
    ! rotation-vibration lines for v=1->0 as function of J (rotation number)
    ! S-branch
    REAL(KIND=qprec), PARAMETER, DIMENSION(0:1) :: &
         Av1_0S = (/2.53d-7,3.47d-7 /),&
         lamv1_0S = (/2.2232d-4,2.1217d-4 /),&
    ! Q-branch
         Av1_0Q =   (/zero,4.29d-7 /),&
         lamv1_0Q =  (/one,2.4066d-4 /)
    !
    beta=hc/(boltzmann*T)

    H2_critical_cool = 0
    ! add up v=0 lines
    v=0
    DO j=0,15
       H2_critical_cool = H2_critical_cool + hc/(4.d0*PI*lam(j))*A(j)*fv(v)*fj(j+2)
    END DO
    ! add up v=1 lines
    DO j=0,1
       v=1
       H2_critical_cool = H2_critical_cool + hc/(4.d0*PI*lamv1_0S(j))*Av1_0S(j)*fv(v)*fj(j+2)
       H2_critical_cool = H2_critical_cool + hc/(4.d0*PI*lamv1_0Q(j))*Av1_0Q(j)*fv(v)*fj(j)
    END DO
    H2_critical_cool = nH2*H2_critical_cool

    CONTAINS
    ! boltzmann population faction for rotational state j
    ! valid fo T > ~ 1000 K
    PURE FUNCTION fj(j)
      REAL(KIND=qprec) :: fj,x
      INTEGER,INTENT(IN) :: j
      
      x=beta*Bv(v)
      ! limit fj to one.  if T < 1000 K or so, apoximating patition function sum 
      ! as integal beaks down and can lead to unphysical fractions
      fj = MIN(gs(j)*(2*j+1)*half*x*exp(-x*j*(j+1)),one)
    END FUNCTION fj

    ! boltzmann population faction for vibrational state v
    PURE FUNCTION fv(v)
      REAL(KIND=qprec) :: fv,x
      REAL(KIND=qprec),PARAMETER :: we=4396.5011d0
      INTEGER,INTENT(IN) :: v
      
      x=beta*we
      fv = (one-exp(-x))*exp(-x*v)
    END FUNCTION fv

    ! degeneracy of rotational state j
    PURE FUNCTION gs(j)
      INTEGER,INTENT(IN) :: j
      INTEGER :: gs
      ! odd -> 3, even->one
      gs = MAX(MOD(j,2)*3,1)
    END FUNCTION gs

    PURE FUNCTION Bv(v)
      INTEGER,INTENT(IN) :: v
      REAL(KIND=qprec) Bv
      REAL(KIND=qprec),PARAMETER ::alphae=2.9263d0,Be=60.7864d0
      Bv = Be-alphae*(REAL(v,qprec)+half)
    END FUNCTION Bv
  END FUNCTION H2_critical_cool

  PURE FUNCTION H2_critical_cool_table(nH2, T) !erg*cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nH2, T
    REAL(KIND=qprec) :: H2_critical_cool_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    H2_critical_cool_table = nH2*(H2_critical_cool_TAB(lower)+(H2_critical_cool_TAB(upper)-H2_critical_cool_TAB(lower))*inc)
  END FUNCTION H2_critical_cool_table

  ! reutuns nH/nH_critical, where ncritical is the critical density for H2 cooling
  ! to reach an equillibrium level population & therefore different cooling funciton
  ! For citical density of H2 lines see:  Mandy & Martin (1993 ApJS...86..199M)
  ! Note that critical density (nCritical) is defined here at temperature TCritical.
  PURE FUNCTION CriticalDenH2(nH,T)
    REAL (KIND=qPrec), INTENT(IN)  :: nH,T
    REAL (KIND=qprec) :: CriticalDenH2,x,ncr

    ! x = n/nCritical, density as fraction of critical density
    x = log10(T/1.d4)
    ! HI citical density from Lepp & Schull
    ncr = 10**(4.d0-0.416d0*x-0.327d0*x**2)
    ! limit critical density to one particle per cc
    CriticalDenH2 = nH/MAX(ncr,one)
  END FUNCTION CriticalDenH2


  ! cooling contribution from H atoms "sticking" to dust
  ! see: Smith & Rosen 2003  (2003MNRAS.339..133S)
  !      Hollembach & Mckee 1989
  PURE FUNCTION dust_cool(n,T) !erg*cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::n, T
    REAL(KIND=qprec) dust_cool, Tin
    ! NOTE:  C=1.66044d-24**1.5*4.9d-33
    REAL(KIND=qprec), PARAMETER :: C=4.9d-33

    IF(T < 10.d0) THEN
       dust_cool = 0.d0
    ELSE
       dust_cool = n**2*3.8d-33*sqrt(T)*(T-Tdust)*(one-0.8d0*exp(-75.d0/T))
    END IF
  END FUNCTION dust_cool

  PURE FUNCTION dust_cool_table(n,T) !erg*cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::n, T
    REAL(KIND=qprec) :: dust_cool_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    dust_cool_table = n**2*(dust_cool_TAB(lower)+(dust_cool_TAB(upper)-dust_cool_TAB(lower))*inc)
  END FUNCTION dust_cool_table


  ! Delgarno & McCray cooling interoplation
  FUNCTION DMCoolingRate(Temp) ! erg cm^3 sec^-1
    ! Interface declarations
    REAL (KIND=qPrec) :: DMCoolingRate
    REAL (KIND=qPrec), INTENT(IN)  :: Temp
    ! Provide the value of the cooling rate (and heating rate) at a given
    ! temperature
    REAL (KIND=qPrec) :: tpos, dtpos
    INTEGER itpos1,itpos2

    tpos=(LOG10(Temp)-2.0)*10.0+1.0  ! Position on the cooling curve
    itpos1=MIN(60,MAX(1,INT(tpos)))  ! Ensure that the lower bounding value is
                                     ! between 1 and 60
    dtpos=tpos-REAL(itpos1,qPrec)    ! Interpolation increment
    itpos2=MIN(61,itpos1+1)          ! Upper bounding value
    
    IF(.NOT. ALLOCATED(DMCoolingTab)) CALL InitDMCool
    ! Perform the interpolation on the cooling curve and find the cooling rate
    IF(Temp .GT. 1.d2) THEN
       DMCoolingRate = (DMCoolingTab(itpos1)+(DMCoolingTab(itpos2)-DMCoolingTab(itpos1))*dtpos)
    ELSE
       DMCoolingRate = 1d-31
!       DMCoolingRate = DMCoolingTab(1)
    END IF
  END FUNCTION DMCoolingRate

  SUBROUTINE InitDMCool
    ! Initialize the cooling routine and load the cooling table from a file
    REAL (KIND=qPrec) :: Temp
    INTEGER i,nInput,iErr
    !
    OPEN(UNIT=85,IOSTAT=iErr,ERR=10,FILE='TABLES/cooling.tab',                &
         STATUS='old',FORM='formatted')
10  IF (iErr /= 0) THEN
       PRINT *,'!!! src FATAL: Error opening file "TABLES/cooling.tab"'
       STOP
    END IF
    !
    READ(85,*) nInput
    ALLOCATE(DMCoolingTab(nInput),STAT=iErr)
    IF (iErr/=0) THEN
       PRINT *,'!!! src FATAL: Cannot allocate DMCoolingTab array'
       STOP
    END IF
    DO i=1,nInput
       READ(UNIT=85,FMT=*,IOSTAT=iErr,END=20) Temp,DMCoolingTab(i)
20     IF (iErr /= 0) THEN
          PRINT *,'!!! src FATAL: EOF encountered while reading file ',       &
               '"TABLES/cooling.tab"'
          STOP
       END IF
    END DO
    CLOSE(85)                                                                
    ! Convert cooling from log to linear
    DMCoolingTab(1:nInput)=10.0d0**DMCoolingTab(1:nInput)
  END SUBROUTINE InitDMCool

  ! Oxygen cooling from OI excitation due to O-H collisions
  ! see: Launay & Roueff (1977A&A....56..289L)
  !
  ! This includes 63 and 44 micrometer lines due to collision with hydrogens. The
  ! dominant term is from the 63 micrometer line.
  !
  ! OI electron collisions are only important for electron fraction > ~1e-4
  ! Federman & Shipsey (ApJ...269..791F).   At temperatures high enough to attain
  ! significant electron fraction, H2 will have dissociated, so i think we can neglect
  ! OI interaction w/ e-.
  !
  ! Federman & Shipsey (ApJ...269..791F) also say that H2 - OI collisions results
  ! in no line cooling but rater an OH formation.
  !
  PURE FUNCTION OI_cool(nH2,nH,T)
    REAL(KIND=qprec),INTENT(IN)::nH2,nH,T
    REAL(KIND=qprec) OI_cool,Ttab
    REAL(KIND=qprec), DIMENSION(0:9), PARAMETER :: &
         CoolTab = (/1.91,10.7,21.3,31.5,41.1,50.0,58.2,65.8,72.9,79.7/)*1.d-25
    REAL(KIND=qprec), PARAMETER :: deltaT = 100, &
                                   minT = 100., &
                                   maxT = 1000. 
    INTEGER(KIND=qprec) :: lo,hi,Tlo,Thi

    IF(T <= minT) THEN
       OI_cool=0.
       RETURN
    ELSE IF(T < maxT) THEN
       Ttab = min(T,maxT)
       lo = INT((T-minT)/deltaT)
       Tlo = minT + deltaT*lo
       hi = lo+1
       Thi = minT + deltaT*hi
       OI_cool = CoolTab(lo) + (CoolTab(hi)-CoolTab(lo))/(Thi-Tlo)*(T-Tlo)
    ! if we are above 1000k, extrapolate out as sqrt(T)
    ELSE
       OI_cool = CoolTab(9)*sqrt(T/maxT)
    END IF
    ! number of O atom = OIFrac*nHneucei
    ! Ionization potential of OI is about the same as that of H.
    ! fraction of H atoms (either atomic or molecular) traces OI abundance
    ! OI_cool = (OIFrac*nneuclei)*(nH+2.*nH2)/nHneuclei*nH*OI_cool
    OI_cool = OIFrac*(nH+2.*nH2)*nH*OI_cool
  END FUNCTION OI_cool


  ! computes H2 dissociation / resociation source term
  ! recombination occurs only on dust grain surfaces
  ! This considers collisions with H, HE, H2, and e-, no H+ collision
  ! is considered.  H+ fractions (ane e- for that matter) are important
  ! only after all of the H2 has been dissociated anyway, I think ....
  ! See : O'Sullivan & Ray 2000 (2000A&A...363..355O) (implementation info)
  !       Lim et al. (2002 RAS,MNRAS 335,817-824) (another implementation)
  !       + references listed in sub-functions
  PURE FUNCTION H2_diss(nH2, nH, nHe, ne, T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nH2, nH, nHe, ne, T
    REAL(KIND=qprec) H2_diss,n,Tin
    ! collisional partner reference indecies
    INTEGER H,H2,He,e
    PARAMETER(H=1,H2=2,He=3,e=4)

    n = nH2 + nH + nHe + ne
    ! this function can cause nasty divergence at high temperatures,limit to 10k
    Tin = min(1.d5,T)
    ! NOTE:: low temp (T<~300) rates can cause divide by zero in kd function
    IF(nH2 <= small .OR. Tin < 1000.d0) THEN
       H2_diss=zero
    ELSE
       H2_diss = nH2*kd(H2,Tin)
       IF(nH>small) H2_diss = H2_diss + nH*kd(H,Tin)
       IF(nHe>small) H2_diss = H2_diss + nHe*kd(He,Tin)
       IF(ne>small) H2_diss = H2_diss + ne*kd(e,Tin)
       H2_diss = nH2*H2_diss
    END IF
    CONTAINS

    ! interpolates between high and low density limites of
    ! collisional dissociation rate for H2 + partner collision
    ! i = index for a particular collision partner
    ! H = 1
    ! H2 = 2
    ! if both limits are avalible -- otherwise it uses low density
    ! limit (an underestimate of the dissociation rate
    !
    ! For high density H2 & H collisions I use the rates of Lepp & Schull
    ! For low densuty H collisions I use the rate of Dove & Mandy
    ! For low density H2 collisions I use the rate of Martin & Keogh
    ! For e- and He collisions I use the low density limit only (no high linit avalible)
    ! from Martin & Keogh.  These are lower limits on the actual rate.
    ! The interpolation scheme between the limits is from Lepp & Schull
    !
    ! see: Lim et al. (2002 RAS,MNRAS 335,817-824), Appendix
    PURE FUNCTION kd(i,T)
      REAL(KIND=qprec),INTENT(IN)::T
      INTEGER,INTENT(IN):: i
      REAL(KIND=qprec):: kd,kdL,kdH,ncr,x
      
      kdL=0;kdH=0;kd=0;ncr=0

      SELECT CASE(i)
      CASE(H)
         kdH = kdH_LS(i,T)
         kdL = kdL_DM(i,T)
         !kdL = kdL_LS(i,T)
         x = log10(T/1.d4)
         ncr = 10**(4.d0-0.416d0*x-0.327d0*x**2)
      CASE(H2)
         kdH = kdH_LS(i,T)
         kdL = kdL_MK(i,T)
         !kdL = kdL_LS(i,T)
         x = log10(T/1.d4)
         ncr = 10**(4.13d0-0.986d0*x+0.119d0*x**2)
      CASE(He,e)
         kd = kdL_MK(i,T)
         RETURN
      END SELECT
      ! interpolation function
      kd = 10**(log10(kdH) - log10(kdH/kdL)/(one+n/MAX(ncr,small)))
    END FUNCTION kd

    ! collisional dissociation rate for H2 + partner collision
    ! i = index for a particular collision partner
    ! H = 1
    ! H2 = 2
    ! see:  Lepp & Schull (1983 ApJ)
    ! NOTE: these parameters are only good in high density limit n >~ 1e6
    !       for lower densities, these are upper bounds on the dissociation rate
    PURE FUNCTION kdH_LS(i,T)
      REAL(KIND=qprec),INTENT(IN)::T
      INTEGER,INTENT(IN):: i
      REAL(KIND=qprec)::kdH_LS

      kdH_LS = 0      
      SELECT CASE(i)
      CASE(H)
         kdH_LS = 3.52e-9*exp(-4.39d4/T) 
      CASE(H2)
         kdH_LS = 5.48e-9*exp(-5.30d4/T)
      END SELECT
    END FUNCTION kdH_LS

    ! collisional dissociation rate for H2 + partner collision
    ! i = index for a particular collision partner
    ! H = 1
    ! H2 = 2
    ! He = 3
    ! e- = 4
    ! see:  Martin & Keogh 1998 (1998ApJ...499..793M)
    ! NOTE: these parameters are only good in low density limit n <~ 1e3
    !       for larger densities, these are lower bounds on the dissociation rate
    !
    ! For low density H2-H2 collisions these are more accurate than those given by Lepp & Schull
    ! according to Lim et al. (2002 RAS,MNRAS 335,817-824), Appendix
    PURE FUNCTION kdL_MK(i,T)
      REAL(KIND=qprec),INTENT(IN)::T
      INTEGER,INTENT(IN):: i
      REAL(KIND=qprec)::kdL_MK,Tin
      REAL(KIND=qprec), DIMENSION(4) :: a,b,c,Eo,g,mu
      ! d = constant to give units in cm^3 S^-1
      ! k = boltzman constant in Martin & Keogh's screwed up unit scale
      REAL(KIND=qprec) d, k
      ! NOTE: g = Gamma(b+1)
      PARAMETER(g = (/3.60358,71.1182,1.70622,1.04386/))
      PARAMETER(a = (/54.1263,40.1008,4.8152,11.2474/))
      PARAMETER(b = (/2.5726,4.6881,1.8208,1.0948/))
      PARAMETER(c = (/3.45,2.1347,-0.9459,2.3182/))
      PARAMETER(Eo = (/0.168,0.1731,0.4146,0.3237/))
      PARAMETER(d = 1.849d-22)
      PARAMETER(k = 3.167d-6)
      mu = (/muH*muH2/(muH+muH2),muH2*muH2/(muH2+muH2),muHe*muH2/(muHe+muH2),mue*muH2/(mue+muH2)/)
      ! this function is only good to 10^5 according to Martin & Keogh
      Tin = min(1.d5,T)
      ! collisions with He are very strong above 10^5 and this number will overflow, limit to ~10^5
      if(i == 3) then
         Tin = min(1.2d5,T)
      else
         ! for species other than He, I find this function is similar to others
         ! below 10^6
         Tin = min(1.d6,T)
      end if
      kdL_MK = d*sqrt(8.d0*k*Tin/(PI*mu(i)*AMU))*a(i)*(k*Tin)**(b(i)-1.d0)*g(i)*exp(-Eo(i)/(k*Tin))/((1.+c(i)*k*Tin)**(b(i)+1.d0))
    END FUNCTION kdL_MK

    ! collisional dissociation rate for H2 + partner collision
    ! i = index for a particular collision partner
    ! H = 1
    ! see:  Dove & Mandy (1986)
    ! NOTE: these parameters are only good in low density limit n <~ 1e3
    !       for larger densities, these are lower bounds on the dissociation rate
    ! For low density H-H2 collisions these are more accurate than those given by Lepp & Schull
    ! according to Lim et al. (2002 RAS,MNRAS 335,817-824), Appendix
    PURE FUNCTION kdL_DM(i,T)
      REAL(KIND=qprec),INTENT(IN)::T
      INTEGER,INTENT(IN):: i
      REAL(KIND=qprec)::kdL_DM

      kdL_DM = 0      
      SELECT CASE(i)
      CASE(H)
         kdL_DM = 4.69d-14*T**0.746d0*exp(-55065d0/T)
      END SELECT
    END FUNCTION kdL_DM

    ! collisional dissociation rate for H2 + partner collision
    ! i = index for a particular collision partner
    ! H = 1
    ! H2 = 2
    ! see:  Lepp & Schull (1983 ApJ)
    ! NOTE: these parameters are only good in low density limit n <~ 1e3
    !       for larger densities, these are lower bounds on the dissociation rate
    !       These lower density limit rates have been superceeded by newer work.
    !       This will probably not be used -- I have coded them here just so I can
    !       test or make comparisions with previous simulations done using these rates.
    PURE FUNCTION kdL_LS(i,T)
      REAL(KIND=qprec),INTENT(IN)::T
      INTEGER,INTENT(IN):: i
      REAL(KIND=qprec)::kdL_LS

      kdL_LS = 0
      SELECT CASE(i)
      CASE(H)
         IF(T > 7390) THEN
            kdL_LS = 6.11d-14*exp(-2.93d4/T) 
         ELSE
            kdL_LS = 2.67d-15*exp(-(6750d0/T)**2)
         END IF
      CASE(H2)
         IF(T > 7291) THEN
            kdL_LS = 5.22d-14*exp(-3.22d4/T)
         ELSE
            kdL_LS = 3.17d-15*exp(-4060.d0/T-(7500d0/T)**2)
         END IF
      END SELECT
    END FUNCTION kdL_LS
  END FUNCTION H2_diss

  ! Recombination of H2 due to condensation on dust grains
  ! see: Hollenbach & McKee 1979 (1979ApJS...41..555H), page 564
  PURE FUNCTION H2_dust_Recomb(nHneuclei,nH,T,Tdust) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHneuclei,nH,T,Tdust
    REAL(KIND=qprec) H2_dust_Recomb
    REAL(KIND=qprec) T2, Tdust2, fa

    ! use fa = 0.5 as suggested in paper --
    fa = 0.5
    T2 = T/100.
    Tdust2 = Tdust/100.

    ! this function is only good to 10^5
    IF(T < 1.d5) THEN
       H2_dust_Recomb = nH*nHneuclei*(3.d-17*sqrt(T2)*fa/(1+0.4*sqrt(T2+Tdust2)+0.2*T2+0.08*T2**2))
    ELSE
       H2_dust_recomb = 0.
    END IF

  END FUNCTION H2_dust_Recomb
  
  !Tablular form of function
  PURE FUNCTION H2_dust_Recomb_table(nHneuclei,nH,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHneuclei,nH,T
    REAL(KIND=qprec) H2_dust_Recomb_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    H2_dust_Recomb_table = zero
    IF(nHneuclei <=small .OR. nH <= small) RETURN
    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    H2_dust_Recomb_table = nH*nHneuclei*(H2_dust_Recomb_TAB(lower)+(H2_dust_Recomb_TAB(upper)-H2_dust_Recomb_TAB(lower))*inc)
  END FUNCTION H2_dust_Recomb_table

  ! Inonization / recombination rate calculations:
  ! General reference see: Martin & Szczerba 1997  (1997A&A...325.1132M)

  ! radiative H+ + e- -> H recombination rate
  ! see: Mazzotta et. al. (1998A&AS..133..403M)
  ! see: Verner & Ferland (1996ApJS..103..467V)
  PURE FUNCTION H_recomb(nHII,ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHII,ne,T
    REAL(KIND=qprec) H_recomb
    REAL(KIND=qprec),PARAMETER :: a=7.982d-11,&
                                  b=0.7480d0,&
                                  To=3.148d0,&
                                  T1=7.036d5
      REAL(KIND=qprec) :: c

      c = sqrt(T/To)
      H_recomb = nHII*ne*(a*1.d0/(c*(1+c)**(1-b)*(1+sqrt(T/T1))**(1+b)))
  END FUNCTION H_recomb

  !Tablular form of function
  PURE FUNCTION H_recomb_table(nHII,ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHII,ne,T
    REAL(KIND=qprec) H_recomb_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    H_recomb_table=zero
    IF(ne <= zero .OR. nHII <= zero) RETURN

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    H_recomb_table = ne*nHII*(H_recomb_TAB(lower)+(H_recomb_TAB(upper)-H_recomb_TAB(lower))*inc)
  END FUNCTION H_recomb_table

  ! collisional ionization
  ! see: Mazzotta et. al. (1998A&AS..133..403M)
  ! see:Arnaud & Rothenflug (1985A&AS...60..425A)
  ! uses exponential integral function de1 from netlib.f, & function f2 from this file
  FUNCTION H_ioniz(nH,ne,T)
    REAL(KIND=qprec),INTENT(IN)::nH,ne,T
    REAL(KIND=qprec) H_ioniz
    REAL(KIND=qprec),PARAMETER:: I=13.6, A=22.8, B=-12.0, C=1.9, D=-22.6, const=6.69d-7
    REAL(KIND=qprec),PARAMETER:: k_ev = 8.617342d-5  !boltzmann in ev/K
    REAL(KIND=qprec) x,f1,f2,F
    DOUBLE PRECISION de1 !exponential integral function courtesy of slatec library

    H_ioniz=zero
    IF(T<500.) RETURN

    x = I/(k_ev*T)
    f1=exp(x)*de1(x)
    !f1=df1(x)
    f2=dF2(x)
    
    F=A*(1.-x*f1)+B*(1.+x-x*(2.+x)*f1)+C*f1+D*x*f2
    H_ioniz=nH*ne*(const*exp(-x)*F/((k_ev*T)**1.5d0*x))
  END FUNCTION H_ioniz

  !Tablular form of function
  PURE FUNCTION H_ioniz_table(nH,ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nH,ne,T
    REAL(KIND=qprec) H_ioniz_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    H_ioniz_table=zero
    IF(ne <= zero .OR. nH <= zero .OR. T<500.) RETURN

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    H_ioniz_table = nH*ne*(H_ioniz_TAB(lower)+(H_ioniz_TAB(upper)-H_ioniz_TAB(lower))*inc)
  END FUNCTION H_ioniz_table

  ! collisional ionization
  ! see: Mazzotta et. al. (1998A&AS..133..403M)
  ! see:Arnaud & Rothenflug (1985A&AS...60..425A)
  ! uses exponential integral function de1 from netlib.f, & function f2 from this file
  FUNCTION He_ioniz(nHe,ne,T)
    REAL(KIND=qprec),INTENT(IN)::nHe,ne,T
    REAL(KIND=qprec) He_ioniz
    REAL(KIND=qprec),PARAMETER::I=24.6, A=17.8,B=-11.0,C=7.0,D=-23.2, const=6.69d-7
    REAL(KIND=qprec),PARAMETER:: k_ev = 8.617342d-5  !boltzmann in ev/K
    REAL(KIND=qprec) x,f1,f2,F
    DOUBLE PRECISION de1 !exponential integral function

    IF(T>1000.) THEN
       x = I/(k_ev*T)
       f1=exp(x)*de1(x)
       !f1=df1(x)
       f2=dF2(x)
       
       F=A*(1.-x*f1)+B*(1.+x-x*(2.+x)*f1)+C*f1+D*x*f2
       He_ioniz=nHe*ne*(const*exp(-x)*F/((k_ev*T)**1.5d0*x))
    ELSE
       He_ioniz=0.
    END IF
  END FUNCTION He_ioniz

  !Tablular form of function
  PURE FUNCTION He_ioniz_table(nHeII,ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHeII,ne,T
    REAL(KIND=qprec) He_ioniz_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    He_ioniz_table = ne*nHeII*(He_ioniz_TAB(lower)+(He_ioniz_TAB(upper)-He_ioniz_TAB(lower))*inc)
  END FUNCTION He_ioniz_table

  ! radiative He+ + e- -> He recombination rate
  ! do not forget ionization energy source term
  PURE FUNCTION He_recomb(nHeII, ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN):: nHeII, ne, T
    REAL(KIND=qprec) He_recomb

    He_recomb = nHeII*ne*(He_rad_recomb(T) + He_dielec_recomb(T))
  CONTAINS

    ! radiative recombination
    ! see: Mazzotta et. al. (1998A&AS..133..403M)
    ! see: Verner & Ferland (1996ApJS..103..467V)
    PURE FUNCTION He_rad_recomb(T)
      REAL(KIND=qprec),INTENT(IN):: T
      REAL(KIND=qprec)  He_rad_recomb
      REAL(KIND=qprec),DIMENSION(2),PARAMETER :: a=(/3.294d-11,9.356d-11/),&
                                                 b=(/0.6910d0,0.7892d0/),&
                                                 To=(/1.554d1,4.266d-2/),&
                                                 T1=(/3.676d7,4.677d6/)
      REAL(KIND=qprec) :: c

      IF(T < 1.d6) THEN
         c = sqrt(T/To(1))
         He_rad_recomb = a(1)*1.d0/(c*(1+c)**(1-b(1))*(1+sqrt(T/T1(1)))**(1+b(1)))
      ELSE
         c = sqrt(T/To(2))
         He_rad_recomb = a(2)*1.d0/(c*(1+c)**(1-b(2))*(1+sqrt(T/T1(2)))**(1+b(2)))
      END IF
    END FUNCTION He_rad_recomb
  
    ! dielectric recombination rate
    ! see: Mazzotta et. al. 1998A&AS..133..403M
    PURE FUNCTION He_dielec_recomb(T)
      REAL(KIND=qprec),INTENT(IN) :: T
      REAL(KIND=qprec) He_dielec_recomb
      REAL(KIND=qprec),PARAMETER :: c=0.11200E-08 ,E=39.70
      
      He_dielec_recomb = T**(-3./2.) * c * exp(-E/T)
    END FUNCTION He_dielec_recomb
  END FUNCTION He_recomb

  !Tablular form of function
  PURE FUNCTION He_recomb_table(nHeII,ne,T) !cm^-3*s^-1
    REAL(KIND=qprec),INTENT(IN)::nHeII,ne,T
    REAL(KIND=qprec) He_recomb_table,Tin,inc
    INTEGER :: lower,upper  ! the lower,upper interpolation point

    Tin = log10(T)
    inc   = (Tin-min_TAB)/delta_TAB
    lower = MAX(MIN(INT(inc),n_TAB),1)
    inc   = inc-REAL(lower,qprec)
    upper = MAX(MIN(lower + 1,n_TAB),1)
    He_recomb_table = ne*nHeII*(He_recomb_TAB(lower)+(He_recomb_TAB(upper)-He_recomb_TAB(lower))*inc)
  END FUNCTION He_recomb_table

  !analytic approximation to f2 integral for inonization rates in Arnaud et. al.
  PURE FUNCTION df2(x)
    REAL(KIND=qprec),INTENT(IN)::x
    REAL(KIND=qprec) df2
    REAL(KIND=qprec),DIMENSION(0:13),PARAMETER::p=(/1.d0,2.1658d2,2.0336d4,1.0911d6,3.7114d7,8.3963d8,1.2889d10,&
         1.3449d11,9.4002d11,4.2571d12,1.1743d13,1.7549d13,1.0806d13,4.9776d11/)
    REAL(KIND=qprec),DIMENSION(0:14),PARAMETER::q=(/1.d0,2.1958d2,2.0984d4,1.1517d6,4.0349d7,9.49d8,1.5345d10,&
         1.7182d11,1.3249d12,6.9071d12,2.3531d13,4.9432d13,5.7760d13,3.0225d13,3.3641d12/)
    REAL(KIND=qprec) ptot,qtot
    REAL(KIND=qprec),DIMENSION(0:14)::xin
    INTEGER j

    FORALL(j=0:14)
       xin(j)=x**(-j)
    END FORALL

    ptot=DOT_PRODUCT(xin(0:13),p(0:13))
    qtot=DOT_PRODUCT(xin(0:14),q(0:14)) 

    df2=ptot/(x**2*qtot)
  END FUNCTION df2

  ! approximation to exponential integral funciton, should be faster than 
  ! exact calculation from netlib.f
  ! doesnt seem to work too well, not used 
  PURE FUNCTION df1(x)
    REAL(KIND=qprec),INTENT(IN)::x
    REAL(KIND=qprec) df1
    INTEGER,PARAMETER::nterms=15
    REAL(KIND=qprec),DIMENSION(nterms)::term
    INTEGER j

    IF(x < 0.02) THEN
       df1=exp(x)*(-log(x)-0.5772+x)
    ELSE IF(x < 1.5) THEN
       df1=log((x+1.)/x)-(0.36+0.03*(x+0.01)**(-0.5))/(x-1)**2
    ELSE IF(x < 10.) THEN
       df1=log((x+1.)/x)-(0.36+0.03*(x+0.01)**(0.5))/(x-1)**2
    ELSE
       term(1)=1./x
       DO j=2,nterms
          term(j)=-j*term(j-1)/x
       END DO      
       !df1=1/x*(1. - 1./x + 2./(x**2) - 6./(x**3) + 24./(x**4) - 5.*24./(x**5))
       df1=SUM(term)
    END IF
  END FUNCTION df1

INCLUDE 'i_evolve.f90'
END MODULE cool
