!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    emiss_calcs.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
SUBROUTINE emissivity(Nprobs1, nx1, ny1, nz1, NTXvals1, emissivities7, exlevels1, species1) 

! This subroutine calculates the emissivity from the formula (n*A*hv/(4Pi))
! The units are erg/cm^3/s/str


IMPLICIT NONE

REAL, PARAMETER :: Pi = 3.14159, k = 8.617E-5, & ! k is in eV/K

                   !Ionization Energies of neutral atoms
                   EionOI = 13.62, &
                   EionNI = 14.5341, &
                   EionHI = 13.60, &

                   ! Abundances, from Hartigan & Morse, APJ, 660, 2007 (solar abundances)
                   AbundO = 8.82, &
                   AbundN = 7.96, &
                   AbundS = 7.30

INTEGER :: Lx, Ly, Lz, i, & ! loop counters
           species1,      & ! chosen species for emission line
           nx1, ny1, nz1    ! 

INTEGER, DIMENSION(2) :: exlevels1 ! initial and final energy levels

REAL :: gOI, gOII, gNI, gNII, gHI, gHII, & ! Statistical weights
        Abund_rat                          ! Abundance of species, relative to Hydrogen

REAL, DIMENSION(5,5,5) :: A_vals,  &  ! Einstein A values
                          hv_ergs     ! Energies of level transitions, in ergs.

REAL, ALLOCATABLE, DIMENSION(:,:,:)  :: nO, nN, nS, &  ! Total number densities
                                        nIon_Neut, &   ! fraction of ion/neutral for given species
                                        nHII_nHI, &    ! fraction of ion/neutral for H
                                        kT, &          ! temperature in eV
                                        nSpec, &       ! includes: nOI, nOII, nNI, nNII, nSII
                                        Nprobs1,  &    ! fraction of a species in a given level
                                        NLevels1, &    ! number density of species in level
                                        emissivities7  ! calculated emissivity
 
REAL, DIMENSION(:,:,:,:) :: NTXvals1      ! array of n_e, T, Xh, nH_total


ALLOCATE( nO(nx1,ny1,nz1), nN(nx1,ny1,nz1), nS(nx1,ny1,nz1), & 
          nIon_Neut(nx1,ny1,nz1), nHII_nHI(nx1,ny1,nz1), kT(nx1,ny1,nz1), &
          nSpec(nx1,ny1,nz1), NLevels1(nx1,ny1,nz1) )


! Call subroutines to get the A-values and excitement energies that will be needed later
CALL A_Values(A_vals)
CALL E_Values(hv_ergs)


! kT values, in eV:
kT = k*NTXvals1(2,:,:,:)


! get abundance of species relative to Hydrogen
IF ( (species1 == 1) .OR.(species1 == 2) ) THEN
  Abund_rat = (AbundO - 12.)
ELSEIF ( (species1 == 3) .OR.(species1 == 4) ) THEN
  Abund_rat = (AbundN - 12.)
ELSEIF ( (species1 == 5) ) THEN
  Abund_rat = (AbundS - 12.)
ELSE
  PRINT *,'WTF?'
  STOP
END IF


! Statistical weights (ground states), not needed for SII, since all S is in SII
gHI = 2
gHII = 1

gOI = 9     ! (3)P(2,1,0) => gOI  = (2*2 + 1) + (2*1 + 1) + (2*0 + 1) = 9
gOII = 4    ! (4)S(3/2)   => gOII = (2*(3/2) + 1) = 4

gNI = 4     ! (4)S(3/2)   => gNI  = (2*(3/2) + 1) = 4
gNII = 9    ! (3)P(2,1,0) => gNII = (2*2 + 1) + (2*1 + 1) + (2*0 + 1) = 9


!*******************************************************************************************
!*******************************************************************************************
!*******************************************************************************************


! Calculate the densities of each excitation level for each species:
! Dr.Hartigan's program gives the ratio of the density of a species in a given
! level to the total density of that species.

! 1. Therefore, we need to multiply that ratio by the total density of the species.
! 2. To get total density of species, need total density of atom (neutral + ionized).
! 3. Get ratio of neutral to ionized from charge exchange.



! Beginning with # 3.
! [HI][OII] <-> [HII][OI]
! gHI*n[HI]*gOII*n[OII]*exp(-EionOI/kT) = 
! (n[HI]/n[HII])(n[OII]/n{OI]) = [(gHII*gOI)/(gHI*gOII)]*exp(-[EionHI-EionOI]/kT)

! # 2.
! nO = nOI + nOII = nOI(1 + nOII/nOI)    and also
! nH/nO = 10^12/10^8.82 -> nO = nH*10^8.82/10^12
! 	Combine:
! nOI = nH*10^(8.82-12) / (1+nOII/nOI)

! # 1. (e.g., for level 2)
! n(2,DrH,OI) = (n2,OI)/nOI -> n2,OI = nOI * n(2,DrH,OI)


!*******************************************************************************************
!*******************************************************************************************
!*******************************************************************************************


! Begin steps to calculate emissivity

! Total densities of O, N, & S, for each value of ne
! N_species = nHtot * 10^(Abund-12)
IF ( (species1 == 1) .OR. (species1 == 2) ) THEN
  nO = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )

ELSEIF ( (species1 == 3) .OR. (species1 == 4) ) THEN
  nN = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )

ELSEIF ( (species1 == 5) ) THEN
  nS = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )  ! = nSII, since all S is singly ionized.

END IF



! Next:
! Calculate ratio of nHII/nHI using:
! Xh = nHII/(nHI + nHII) = (nHII/nHI)[(1 + nHII/nHI)^-1]
! -> nHII/nHI = Xh/(1 - Xh)

WHERE ( (1. - NTXvals1(3,:,:,:)) == 0.0 ) &
      nHII_nHI(:,:,:) =  NTXvals1(3,:,:,:)/(1. - NTXvals1(3,:,:,:) + 1.e-10)

WHERE ( (1. - NTXvals1(3,:,:,:)) /= 0.0 ) &
      nHII_nHI(:,:,:) =  NTXvals1(3,:,:,:)/(1. - NTXvals1(3,:,:,:))



! Use Charge exchange and level populations to get number density of
! chosen species & level.

IF (species1 == 1) THEN ! OI
! nIon_Neut = nOII/nOI

    nIon_Neut(:,:,:) = nHII_nHI(:,:,:) * (gHI*gOII/(gHII*gOI)) * &
      EXP((EionHI - EionOI)/kT(:,:,:))
    nSpec(:,:,:) = nO(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 2) THEN  ! OII
! nIon_Neut = nOI/nOII

  WHERE (nHII_nHI == 0.0) nHII_nHI = 1.e-10

    nIon_Neut(:,:,:) = (1./nHII_nHI(:,:,:)) * (gHII*gOI/(gHI*gOII)) * &
      EXP(-(EionHI - EionOI)/kT(:,:,:))
    nSpec(:,:,:) = nO(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 3) THEN  ! NI
! nIon_Neut = nNII/nNI

    nIon_Neut(:,:,:) = nHII_nHI(:,:,:) * (gHI*gNII/(gHII*gNI)) * &
      EXP((EionHI - EionNI)/kT(:,:,:))
    nSpec(:,:,:) = nN(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 4) THEN  ! NII
! nIon_Neut = nNI/nNII

  WHERE (nHII_nHI == 0.0) nHII_nHI = 1.e-10

    nIon_Neut(:,:,:) = (1./nHII_nHI(:,:,:)) * (gHII*gNI/(gHI*gNII)) * &
       EXP(-(EionHI - EionNI)/kT(:,:,:))
    nSpec(:,:,:) = nN(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 5) THEN  ! SII
  NLevels1(:,:,:) = Nprobs1(:,:,:)*nS(:,:,:) ! nSpec (ie nSII) = nS

ELSE
   PRINT *, 'Something evil this way came'
   STOP

END IF    


!  Calculate the emissivity for each point, for the given species and levels using:
!  emissivity = NLevels1 * A(final, init) * hv(final, init) 

emissivities7(:,:,:) = ( NLevels1(:,:,:) * A_vals(exlevels1(2),exlevels1(1),species1) * &
                       hv_ergs(exlevels1(2),exlevels1(1),species1) ) / (4.*3.14159)



END SUBROUTINE emissivity




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


SUBROUTINE alpha_emiss(NTXvalsA, nx3,ny3,nz3, emissAlpha)

! This subroutine calculates the emissivity of H-alpha from recombination and collisional
! excitation from levels 1 -> 3,4,5

IMPLICIT NONE

REAL, PARAMETER :: Halpha_hv = 3.0263E-12, & ! ergs, = 1.88889 eV, for H n= 3->2 transition
                   E_thr_n3 = 12.09, &       ! in eV, for n=1->3
                   E_thr_n4 = 12.75, &       ! in eV, for n=1->4
                   E_thr_n5 = 13.056, &      ! in eV, for n=1->5
                   k_B = 8.617E-5, &         ! in eV/K
                   vern_a = 7.982d-11, &     ! cm^3/s
                   vern_b = 0.7480d0, &
                   vern_T0 = 3.148d0, &      ! K
                   vern_T1 = 7.036d5         ! K

CHARACTER(LEN = 80) :: filename

INTEGER :: i,j,k                             ! counters
INTEGER, INTENT(IN) :: nx3, ny3, nz3         ! array dimensions

REAL (KIND=8), DIMENSION(4,25) :: coll_str   ! table of effective collision strengths (e-H)

REAL, ALLOCATABLE, DIMENSION(:,:,:) :: &
                   recombcoef, &      ! recombination rate coefficient
                   collexcite,  &     ! collisional excitation rate coefficient
                   emissAlpha_rec, &  ! emissivity due to H-alpha recombination
                   emissAlpha_coll, & ! emissivity due to collisional excitation
                   TeLog              ! Log of Temperature (in K)

REAL, ALLOCATABLE, INTENT(IN OUT), DIMENSION(:,:,:) :: emissAlpha ! Total H-alp emissivity

REAL, DIMENSION(:,:,:,:) :: NTXvalsA ! array of ne, T, Xh, nHtotal
REAL,ALLOCATABLE,DIMENSION(:,:,:,:) :: upsilon ! effective collision strengths at each cell


ALLOCATE( recombcoef(nx3,ny3,nz3), collexcite(nx3,ny3,nz3),  &
          emissAlpha_rec(nx3,ny3,nz3), emissAlpha_coll(nx3,ny3,nz3), &
          upsilon(3,nx3,ny3,nz3), TeLog(nx3,ny3,nz3) )  


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Recombination for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


! Recombination coeff's from Verner & Ferland (1996 ApJS..103..467V)
! Range valid from 3K to 10^9 K
! recombination coefficient is in cm^3/s

recombcoef(:,:,:) = vern_a / ( SQRT(NTXvalsA(2,:,:,:)/vern_T0) * &
                    ((1.d0 + SQRT(NTXvalsA(2,:,:,:)/vern_T0))**(1.d0-vern_b)) * &
                    ((1.d0 + SQRT(NTXvalsA(2,:,:,:)/vern_T1))**(1.d0+vern_b)) )

                        
! emiss_rec = hv * Ne *NHii * recombcoef / (4Pi)      where NHii = Xh*NHtot
emissAlpha_rec(:,:,:) = Halpha_hv * NTXvalsA(1,:,:,:) * (NTXvalsA(3,:,:,:)* &
                        NTXvalsA(4,:,:,:)) * recombcoef(:,:,:) / (4.*3.14159) 




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Collisional Excitation for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


! First calculate the rate coefficient (in cm^3/s), based on effective
! collision strengths from:
! Anderson (2000):  JPhB..2000..33..1255, JPhB..2002..35..1613


  OPEN(49,file='Anderson_Coll_Str.dat',ACTION='READ')
  READ(49,'(4(ES12.4))'), coll_str
  ! First column is T
  ! 2nd, 3rd & 4th are collision strengths (logged) from n = 1 to n=3,4,5 respectively
  CLOSE(49)

  TeLog = LOG10(NTXvalsA(2,:,:,:))
  upsilon = 0.0

  DO i = 1,24

! I don't know why I couldn't combine these all into one WHERE construct,
! since they all have the same condition...

    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(1,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(2,i+1)-coll_str(2,i)) + coll_str(2,i) )
    END WHERE

    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(2,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(3,i+1)-coll_str(3,i)) + coll_str(3,i) )
    END WHERE

    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(3,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(4,i+1)-coll_str(4,i)) + coll_str(4,i) )
    END WHERE

  END DO

! collision strengths are tiny at low T, so below T in coll_str table, just set them to
! the minimum table value.
  WHERE (TeLog < coll_str(1,1)) 
    upsilon(1,:,:,:) = 10.**coll_str(2,1)
    upsilon(2,:,:,:) = 10.**coll_str(3,1)
    upsilon(3,:,:,:) = 10.**coll_str(4,1)
  END WHERE


! Calculate the rate coefficient (in cm^3/s)
  collexcite = ((8.63E-6)/(2*SQRT(NTXvalsA(2,:,:,:))))* &
              ( (EXP(-E_thr_n3/(k_B*NTXvalsA(2,:,:,:)))*upsilon(1,:,:,:)) + &
                (EXP(-E_thr_n4/(k_B*NTXvalsA(2,:,:,:)))*upsilon(2,:,:,:)) + &
                (EXP(-E_thr_n5/(k_B*NTXvalsA(2,:,:,:)))*upsilon(3,:,:,:)) )



! emiss_coll = hv * ne * nHi * collexcite / (4Pi)   where nHi = nHtot-(Xh*nHtot)
emissAlpha_coll(:,:,:) = Halpha_hv * NTXvalsA(1,:,:,:) * &
                  (NTXvalsA(4,:,:,:)-(NTXvalsA(3,:,:,:)*NTXvalsA(4,:,:,:))) * &
                  collexcite(:,:,:) / (4.*3.14159)




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Total Emission for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



! Total emission, combining recombination & collisional excitation
! in ergs/cm^3/s/str

emissAlpha = emissAlpha_rec + emissAlpha_coll


END SUBROUTINE alpha_emiss



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!




!!$SUBROUTINE sum_emiss(emiss2, nx2, ny2, nz2, partname, gridunits2) 
!!$
!!$IMPLICIT NONE
!!$
!!$! This subroutine will sum the emissivities along the z-axis
!!$
!!$CHARACTER(len=80) :: filename, partname
!!$
!!$INTEGER :: i, j, &        ! loop counters
!!$           nx2, ny2, nz2  ! array dimensions
!!$
!!$REAL :: scaler, & ! Physical length of each cell in z-dir
!!$        lscale    ! lscale set in physics.data, entered manually here...
!!$
!!$REAL, DIMENSION(3), INTENT(IN) :: gridunits2     ! mx,my,mz from bear.data
!!$REAL, ALLOCATABLE, DIMENSION(:, :, :) :: emiss2  ! calculated emissivities
!!$REAL, ALLOCATABLE, DIMENSION(:, :) :: emissSum   ! intensity:  emission summed along line of sight
!!$
!!$ALLOCATE( emissSum(nx2,ny2) )
!!$
!!$
!!$lscale = 1.0e17
!!$scaler = gridunits2(3)*lscale/REAL(nz2)
!!$
!!$
!!$DO i = 1, ny2
!!$  DO j = 1, nx2
!!$    emissSum(j,i) = SUM(emiss2(j,i,:))
!!$  END DO
!!$END DO
!!$
!!$! multiply by cell length to go from cm^-3 to cm^-2
!!$emissSum = emissSum*scaler 
!!$
!!$
!!$!   Write the data array to a file in case you want to add or divide emission lines
!!$filename = 'emiss_'//TRIM(partname)//'.dat'
!!$OPEN(20, file=filename, STATUS='REPLACE')
!!$WRITE(20,*) emissSum 
!!$CLOSE(20)
!!$
!!$
!!$!   Write the data to a FITS file
!!$filename = 'emiss_'//TRIM(partname)//'.fits'
!!$CALL writfits_r4(filename,emissSum(:,:),nx2,ny2)
!!$
!!$
!!$
!!$END SUBROUTINE sum_emiss




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


SUBROUTINE A_Values(a)

! This subroutine just returns a matrix with the A-values for the five lines of interest.
! The A-values are yanked from Dr. P.Hartigan's code fiveln.f, including sources.

! a(final level, initial level, species)

IMPLICIT NONE

REAL, DIMENSION(5,5,5) :: a   ! Einstein A transition values, in s^-1

! Einstein A coefficients for OI:
! A-values for OI: Baluja, K., and Zeippen, C. 1988, J. Phys B. Atom. Mol. Opt. Phys. 21, 1455.

a(4,5,1)=1.215
a(3,5,1)=0.0
a(2,5,1)=7.60e-2
a(1,5,1)=2.73e-4
a(5,5,1)=a(4,5,1)+a(3,5,1)+a(2,5,1)+a(1,5,1)
a(3,4,1)=8.93e-7
a(2,4,1)=1.82e-3
a(1,4,1)=5.63e-3
a(4,4,1)=a(3,4,1)+a(2,4,1)+a(1,4,1)
a(2,3,1)=1.74e-5
a(1,3,1)=1.20e-10
a(3,3,1)=a(2,3,1)+a(1,3,1)
a(1,2,1)=8.96e-5


! Einstein A coefficients for OII:
! A-values for OII: Zeippen, C. 1982, MNRAS 198, 111
!   (better than Zeippen, C. 1987, A&A 173, 410 according to
!   planetary nebula observations of Wang et al 2004 A&A 427, 873)

a(4,5,2)=2.08e-11
a(3,5,2)=1.02e-1
a(2,5,2)=6.15e-2
a(1,5,2)=2.32e-2
a(5,5,2)=a(4,5,2)+a(3,5,2)+a(2,5,2)+a(1,5,2)
a(3,4,2)=6.14e-2
a(2,4,2)=1.17e-1
a(1,4,2)=5.64e-2
a(4,4,2)=a(3,4,2)+a(2,4,2)+a(1,4,2)
a(2,3,2)=1.2e-7
a(1,3,2)=1.65e-4
a(3,3,2)=a(2,3,2)+a(1,3,2)
a(1,2,2)=3.82e-5


! Einstein A coefficients for NI:
! A-values for NI: Butler, K., and Zeippen, C. 1984, A&A 141, 274

a(4,5,3)=0.0
a(3,5,3)=2.523e-2
a(2,5,3)=5.589e-2
a(1,5,3)=6.61e-3
a(5,5,3)=a(4,5,3)+a(3,5,3)+a(2,5,3)+a(1,5,3)
a(3,4,3)=4.80e-2
a(2,4,3)=3.14e-2
a(1,4,3)=2.72e-3
a(4,4,3)=a(3,4,3)+a(2,4,3)+a(1,4,3)
a(2,3,3)=1.24e-8
a(1,3,3)=2.28e-5
a(3,3,3)=a(2,3,3)+a(1,3,3)
a(1,2,3)=6.13e-6


! Einstein A coefficients for NII:
! A-values for NII:
!   Bell, Hibbert, Stafford 1995, Phys. Scr. 52, 240
!   Storey and Zeippen 2000, MNRAS 312, 813.
!   Froese Fischer, C. and Saha, H. 1985, Phys Scr. 32, 181.

a(4,5,4)=0.923
a(3,5,4)=1.435e-4
a(2,5,4)=4.588e-2
a(1,5,4)=0.0
a(5,5,4)=a(4,5,4)+a(3,5,4)+a(2,5,4)+a(1,5,4)
a(3,4,4)=3.015e-3
a(2,4,4)=9.819e-4
a(1,4,4)=1.928e-6
a(4,4,4)=a(3,4,4)+a(2,4,4)+a(1,4,4)
a(2,3,4)=7.46e-6 
a(1,3,4)=1.062e-12
a(3,3,4)=a(2,3,4)+a(1,3,4)
a(1,2,4)=2.08e-6


! Einstein A coefficients for SII:
! A-values for SII, Mendoza & Zeippen 1982 MNRAS 198, 111.

a(4,5,5)=1.03e-6
a(3,5,5)=0.179
a(2,5,5)=0.133
a(1,5,5)=0.225
a(5,5,5)=a(4,5,5)+a(3,5,5)+a(2,5,5)+a(1,5,5)
a(3,4,5)=7.79e-2
a(2,4,5)=1.63e-1
a(1,4,5)=9.06e-2
a(4,4,5)=a(3,4,5)+a(2,4,5)+a(1,4,5)
a(2,3,5)=3.35e-7
a(1,3,5)=2.60e-4
a(3,3,5)=a(2,3,5)+a(1,3,5)
a(1,2,5)=8.82e-4

a(2:5,2,:) = 0.0
a(4:5,3,:) = 0.0
a(5,4,:) = 0.0


! Value for transitions out of level 1 = 0.0
a(:,1,:) = 0.0


END SUBROUTINE A_Values




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!




SUBROUTINE E_Values(erg)

! This subroutine generates a table with the energies for OI, OII, NI, NII, & SII
! These values were yanked from Dr. Hartigan's code, fiveln.f


IMPLICIT NONE

REAL, DIMENSION(5,5) :: e      ! energy of a given level in eV
REAL, DIMENSION(5,5,5) :: erg  ! erg is energy of transition converted from eV to ergs
INTEGER :: i,j,k               ! counters for loops


! Energy levels of OI (eV)
e(1,1)=0.0
e(1,2)=0.0196
e(1,3)=0.0282
e(1,4)=1.9689
e(1,5)=4.1931

! Energy levels of OII (eV)
e(2,1)=0.0
e(2,2)=3.3267
e(2,3)=3.3292
e(2,4)=5.0214
e(2,5)=5.0216

! Energy levels of NI (eV)
e(3,1)=0.0
e(3,2)=2.3854
e(3,3)=2.3865
e(3,4)=3.57840
e(3,5)=3.57845

! Energy levels of NII (eV)
e(4,1)=0.0
e(4,2)=0.00604
e(4,3)=0.0162
e(4,4)=1.900
e(4,5)=4.056

! Energy levels of SII (eV)
e(5,1)=0.0
e(5,2)=1.843  
e(5,3)=1.847 
e(5,4)=3.043
e(5,5)=3.049



DO i = 1,5 ! species
	DO j = 2,5  ! initial
		DO k = 1,5 ! final
		  IF (k < j) THEN
		    erg(k,j,i) = (e(i,j) - e(i,k))*(1.602E-12)
		  ELSE
		    erg(k,j,i) = 0.0
		  END IF
		END DO
	END DO
END DO

! where it starts in level 1, it can't transition down, so E=0
erg(:,1,:) = 0.0



END SUBROUTINE E_Values




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



SUBROUTINE linenames(species, exlevels, lambda)

! This subroutine assigns the proper wavelength to the chosen line transition, for
! use in file names.


IMPLICIT NONE

CHARACTER(len=5) :: lambda
INTEGER :: species
INTEGER, DIMENSION(2) :: exlevels

IF (species == 1) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '5577'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '2980'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '2974'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2960'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '6392'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '6363'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6300'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '144E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '439E3'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '633E3'
    END IF
  END IF

ELSEIF (species == 2) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '620E5'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '7333'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '7322'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2471'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '7332'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '7321'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2471'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '496E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3727'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '3730'
    END IF
  END IF

ELSEIF (species == 3) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = 'dunno'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '10401'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10410'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3467'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '10401'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10410'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3468'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '113E5'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '5202'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '5199'
    END IF
  END IF

ELSEIF (species == 4) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '5756'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '3069'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '3064'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3057'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '6585'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '6550'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6525'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '122E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '765E3'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '205E4'
    END IF
  END IF

ELSEIF (species == 5) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '207E4'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '10323'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10290'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '4070'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '10373'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10339'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '4078'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '310E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6718'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '6733'
    END IF
  END IF

END IF

lambda = TRIM(lambda)

END SUBROUTINE linenames



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



END MODULE emiss_calcs

