!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    emission_main.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
SUBROUTINE CalcEmiss(NTXvals,nx,ny,nz,species,spec_char,exlevels,lev1_char,lev2_char)

! 2009, 04/28: input is N_e, T, Xh, N_Htot.  This accounts for
!              potential He ionization, which means N_e /= N_Hii
!              (unless it's a purely H run)

! 2008, 08/27: input is N_e, T, N_Htot (with N_e assumed equal to N_Hii) 

! This program takes in an array of density, energy & ionization
! fraction values for a 3-D volume and then calculates the emissivity
! at each point and sums it up along the line of sight.  It does this
! for the user's choice of OI, OII, NI, NII, or SII, and for the
! desired energy level transitions (first 5 energy levels).

! It uses a modified version of Dr. Hartigan's fiveln.f program 
! which calculates the level populations of the chosen species.

! NOTE:  Compile this program with the module:
!         emiss_calcs.f90 - contains the subroutines for calculating 
!                           the emissivities and summing them up,
! and the external subroutine:
!         writfits_r4.f - for outputing *.fits images


USE emiss_calcs


IMPLICIT NONE


INTEGER, PARAMETER :: nl=5, &   ! number of energy levels to choose from
                      ns=5      ! number of species to choose from : OI,OII,NI,NII,SII
INTEGER :: listx,listy,listz, m, i, j, k, counter, &   ! various counters for do loops, etc. 
           nx, ny, nz, &        ! array size
           species             ! chosen species
INTEGER, DIMENSION(2) :: exlevels ! initial & final energy levels for emission

REAL :: zne, t, temp, ne, s
REAL*8 :: b(nl,nl), indx(nl), vv(nl), den(nl)

REAL, DIMENSION(3) :: gridunits ! 
REAL, DIMENSION(:,:,:,:) :: NTXvals
REAL, ALLOCATABLE, DIMENSION(:,:,:) :: emiss1, Nprobs

CHARACTER(LEN=1) :: lev1_char, lev2_char
CHARACTER(LEN=3) :: spec_char
CHARACTER(LEN=5) :: lambda
CHARACTER(LEN=6) :: atom(ns)
CHARACTER(LEN=85) :: filename, partname


!!$! Get the filename
!!$WRITE (*,'(A)',ADVANCE='NO') ' Enter the filename: '
!!$READ (*,'(A)') filename
!!$!filename ='/projects/hartigan/jnp1/chombofiles/thesis_4cell_200ypulse_2500y_NeXhTNhtot.emiss'
!!$
!!$OPEN(20, file = filename, ACTION='READ')
!!$PRINT *
!!$PRINT *,'Opening: ',filename
!!$
!!$
!!$READ(20,*) gridunits            !  x, y, z, defined in bear.data
!!$READ(20,*) counter, nx, ny, nz  !  ignore counter.  nx... = # of cells in fixed grid


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
! Currently taking, from Chombo, Ne, T, Xh (H ionization fraction), and N_Htot
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PRINT*,nx,ny,nz
PRINT*,species,spec_char
PRINT*,exlevels,lev1_char,lev2_char
PRINT*,SHAPE(NTXvals)
PRINT*,SIZE(NTXvals)


! Set the dimensions for the allocatable arrays
!ALLOCATE( Nprobs(nx,ny,nz), NTXvals(4,nx,ny,nz), emiss1(nx,ny,nz) )
ALLOCATE( Nprobs(nx,ny,nz), emiss1(nx,ny,nz) )



!!$! Re-read the input file  to get the Ne, Te, and Xh values
!!$READ(20,*) NTXvals
!!$CLOSE(20)
!!$
!!$
!!$! Make sure ionization fraction is between 0.0 and 1.0
!!$WHERE (NTXvals(3,:,:,:) <= 0.0) NTXvals(3,:,:,:) = 1.e-10
!!$WHERE (NTXvals(3,:,:,:) > 1.0) NTXvals(3,:,:,:) = 1.0




!!$! Get the element & ionization state desired
!!$WRITE (*,'(A)',ADVANCE='NO') ' Enter the element desired (OI = 1, OII = 2, NI = 3, NII = 4, SII = 5, H_alpha = 6): '
!!$READ (*,'(I1)') species
!!$!species = 6
!!$
!!$IF (species == 1) THEN
!!$        spec_char = 'OI'
!!$ELSEIF (species == 2) THEN
!!$        spec_char = 'OII'
!!$ELSEIF (species == 3) THEN
!!$        spec_char = 'NI'
!!$ELSEIF (species == 4) THEN
!!$        spec_char = 'NII'
!!$ELSEIF (species == 5) THEN
!!$        spec_char = 'SII'
!!$ELSEIF (species == 6) THEN
!!$        spec_char = 'Hal'
!!$END IF
!!$
!!$
!!$
!!$! Get the initial and final excitation levels
!!$WRITE (*,'(A)',ADVANCE='NO') ' Enter the initial and final excitation levels init, fin: '
!!$READ (*,*) exlevels(:)
!!$!exlevels = [4,3]
!!$IF (species == 6) exlevels = [3,2]
!!$lev1_char = ACHAR(exlevels(1)+48)
!!$lev2_char = ACHAR(exlevels(2)+48)
!!$


! Call separate subroutine to calculate H-alpha emission from recombination
IF (species == 6) THEN

  ! create a CHARACTER field to be tagged onto file names, specifying the species & levels
  partname = TRIM(spec_char)//'_'//lev1_char//'_'//lev2_char

  CALL alpha_emiss(NTXvals, nx,ny,nz, emiss1)     ! in emiss_calcs.f90

ELSE


  ! Call subroutine to get the wavelength for the given transition
  CALL linenames(species, exlevels, lambda)


  ! create a CHARACTER field to be tagged onto file names, specifying the species & levels
  partname = TRIM(spec_char)//'_'//lev1_char//'_'//lev2_char//'_'//TRIM(lambda)


!*********************************************************************************************
!*********************************************************************************************
!*********************************************************************************************



  ! BEGIN Dr.Hartigan's program now

  !        Complete 5-level atom calculation for OI,OII,NI,NII,SII
  !        Atomic data from CHIANTI project and Pradhan & Peng 1995 STScI Symp. #8
  !        and various other references as noted.


  ! Go through Dr.Hartigan's program for each coordinate.
  ! `list' is the counter for the coordinates.

  ! Calculate Nprobs (the level populations)

  counter = 0
  DO listx = 1, nx
    DO listy = 1, ny
      DO listz = 1, nz

        ! Put T values in 10^4 K
        t = NTXvals(2,listx,listy,listz)/(10.**4)

        ! Really begin Dr.Hartigan's program now.


        IF (NTXvals(1,listx,listy,listz) /= 0.) THEN
          ne = NTXvals(1,listx,listy,listz)
          zne = ne

          ! Matrix b gets overwritten each time when inverted, so need
          ! to initialize DO loop for each atomic ionization state
        
          if(species == 1) then
                call oi(b,nl,t,ne)
          elseif(species == 2) then
                call oii(b,nl,t,ne)
          elseif(species == 3) then
                call ni(b,nl,t,ne)
          elseif(species == 4) then
                call nii(b,nl,t,ne)
          elseif(species == 5) then
                call sii(b,nl,t,ne)
          END IF

          ! Do Matrix Inversion - note values in b and den are overwritten
          ! input: matrix b(i,j) and r.h.s. is (0,0,0,0,1) stored in den(i)
          ! output: LU decomposed matrix in b(i,j) and density of levels in den(i)
          ! From numerical recipes in Fortran

          do m=1,nl-1
                den(m)=0.
          END DO
          den(nl)=1.
          call ludcmp(b,nl,indx,s)
          call lubksb(b,nl,indx,den)

        ELSE
          den(1) = 1.
          do m=2,nl
            den(m)=0.
          end do

        END IF

      Nprobs(listx,listy,listz) = den(exlevels(1))
      IF (Nprobs(listx, listy, listz) == -0.) Nprobs(listx, listy, listz) = 0.0


      END DO  ! end listz
    END DO   ! end listy
  END DO   ! end listx


  ! This ends the main part of Dr.Hartigan's program.



!****************************************************************************
!****************************************************************************
!****************************************************************************



  ! The subroutine `emissivity' calculates the emissivities from the
  ! `n' values determined above along with Einstein A coefficients and
  !  Energies of level transitions.

  CALL emissivity(Nprobs, nx, ny, nz, NTXvals, emiss1, exlevels, species)

  END IF

  NTXvals(5,:,:,:) = emiss1

!!$! Call subroutine that sums up the the emissivities along the line of sight
!!$CALL sum_emiss(emiss1, nx, ny, nz, partname, gridunits)

!!$DEALLOCATE(Nprobs, NTXvals, emiss1)
DEALLOCATE(Nprobs, emiss1)

END SUBROUTINE CalcEmiss



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



! The following are Dr.Hartigan's subroutines called in the main
! part of his program above.


!       This subroutine loads up the OI matrix for inversion
        subroutine oi(b,nl,t,ne)
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl), t, ne

!        Statistical weights for OI
        g(1)=5.
        g(2)=3.
        g(3)=1.
        g(4)=5.
        g(5)=1.

!        Energy levels of OI (eV)
        e(1)=0
        e(2)=0.0196
        e(3)=0.0282
        e(4)=1.9689
        e(5)=4.1931

!        A-values for OI, Baluja, K., and Zeippen, C. 1988,
!        J. Phys B. Atom. Mol. Opt. Phys. 21, 1455.
        a(5,4)=1.215
        a(5,3)=0.
        a(5,2)=7.60e-2
        a(5,1)=2.73e-4
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=8.93e-7
        a(4,2)=1.82e-3
        a(4,1)=5.63e-3
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.74e-5
        a(3,1)=1.20e-10
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=8.96e-5

!         Collision strengths for OI; t=T/10^4, From Berrington and Burke 1981 Plan. Sp. Sci. 29, 377
!         except for the ^3P-^3P, which are from Le Dourneuf, M. and Nesbet, R. 1976, J Phys B.
!         Atom. Molec. Phys. 9, L241. Fits are mine, and work very well.
        o(1,4)=(5./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(2,4)=(3./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(3,4)=(1./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(1,5)=(5./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(2,5)=(3./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(3,5)=(1./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(1,2)=0.0987*t**1.11
        o(2,3)=0.0265*t**1.17
        o(1,3)=0.0292*t*(t**(log10(t**(-0.13))))
        o(4,5)=0.105*t**0.54

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine oi

!       This subroutine loads up the OII matrix for inversion
        subroutine oii(b,nl,t,ne)
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for OII
        g(1)=4.
        g(2)=6.
        g(3)=4.
        g(4)=4.
        g(5)=2.

!       Energy levels of OII (eV)
        e(1)=0
        e(2)=3.3267
        e(3)=3.3292
        e(4)=5.0214
        e(5)=5.0216

!       A-values for OII, Zeippen, C. 1982, MNRAS 198, 111
!       (better than Zeippen, C. 1987, A&A 173, 410 according to
!       planetary nebula observations of Wang et al 2004 A&A 427, 873)
        a(5,4)=2.08e-11
        a(5,3)=1.02e-1
        a(5,2)=6.15e-2
        a(5,1)=2.32e-2
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=6.14e-2
        a(4,2)=1.17e-1
        a(4,1)=5.64e-2
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.2e-7
        a(3,1)=1.65e-4
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=3.82e-5

!       Collision strengths for OII; t=T/10^4
!       McLaughlin, B., and Bell, K. results 1993 ApJ 408, 753 are spurious.
!       (see Wang et al 2004 A&A 427, 873). 
!       Instead, use Pradhan 1976 MNRAS 177, 31P 
        o(1,2)=0.6*1.335*(t**0.015)
        o(1,3)=0.4*1.335*(t**0.015)
        o(1,4)=(2./3.)*0.406*(t**0.02)
        o(1,5)=(1./3.)*0.406*(t**0.03)
        o(2,3)=1.168*(t**(-0.06))
        o(2,4)=1.706*0.428*(t**0.05)
        o(2,5)=1.706*0.173*(t**0.05)
        o(3,4)=1.706*0.239*(t**0.05)
        o(3,5)=1.706*0.161*(t**0.05)
        o(4,5)=0.287*(t**0.05)

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine oii

!       This subroutine loads up the NI matrix for inversion
        subroutine ni(b,nl,t,ne)
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for NI
        g(1)=4.
        g(2)=6.
        g(3)=4.
        g(4)=2.
        g(5)=4.

!       Energy levels of NI (eV)
        e(1)=0
        e(2)=2.3854
        e(3)=2.3865
        e(4)=3.57840
        e(5)=3.57845

!       A-values for NI, Butler, K., and Zeippen, C. 1984, A&A 141, 274
        a(5,4)=0
        a(5,3)=2.523e-2
        a(5,2)=5.589e-2
        a(5,1)=6.61e-3
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=4.80e-2
        a(4,2)=3.14e-2
        a(4,1)=2.72e-3
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.24e-8
        a(3,1)=2.28e-5
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=6.13e-6

!       Collision strengths for NI; t=T/10^4, Berrington and Burke 1981, Pl. Sp.Sci. 29, 377
!       ^4S terms fit REALly well to log10(Omega) = a + b*log10(T/10^4) + c*(log10(T/10^4))^2
!       ^2D-^2P are from BB81, fit with power law, with internal fine structure ratios (not ~ g!!)
!       as described in Table II of Pradhan 1976, MNRAS 177, 31P, which agree with 
!       ratios of Dopita Mason and Robb 1976 ApJ 207, 102.  ^2D-^2D and ^2P-^2P use DMR76 power
!       law fits.
        o(1,2)=0.6*0.484*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,3)=0.4*0.484*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,4)=(1./3.)*0.170*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,5)=(2./3.)*0.170*(t**0.82)*(t**(log10(t**(-0.30))))
        o(2,3)=0.773-0.825*(exp(-t/2.03))
        o(2,4)=0.173*0.819*(t**0.59)
        o(2,5)=0.428*0.819*(t**0.59)
        o(3,4)=0.161*0.819*(t**0.59)
        o(3,5)=0.239*0.819*(t**0.59)
        o(4,5)=0.071*(t**1.11)

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine ni

!       This subroutine loads up the NII matrix for inversion
        subroutine nii(b,nl,t,ne)
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for NII
        g(1)=1.
        g(2)=3.
        g(3)=5.
        g(4)=5.
        g(5)=1.

!       Energy levels of NII (eV)
        e(1)=0.
        e(2)=0.00604
        e(3)=0.0162
        e(4)=1.900
        e(5)=4.056

!       A-values for NII, Bell, Hibbert, Stafford 1995, Phys. Scr. 52, 240
!                   Storey and Zeippen 2000, MNRAS 312, 813.
!                   Froese Fischer, C. and Saha, H. 1985, Phys Scr. 32, 181.
        a(5,4)=0.923
        a(5,3)=1.435e-4
        a(5,2)=4.588e-2
        a(5,1)=0
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=3.015e-3
        a(4,2)=9.819e-4
        a(4,1)=1.928e-6
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=7.46e-6 
        a(3,1)=1.062e-12
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=2.08e-6

!       Collision strengths for NII; t=T/10^4 from Hudson and Bell A&A 430, 725, 2005
!       temperature dependence is small, not clearly a power law, and I am not sure I
!       believe it anyway. Makes the most difference for lowest 3 levels when T > 10^4K
        o(1,2)=0.43
        o(1,3)=0.27
        o(1,4)=0.30
        o(1,5)=0.035
        o(2,3)=1.15
        o(2,4)=0.91
        o(2,5)=0.11
        o(3,4)=1.51
        o(3,5)=0.18
        o(4,5)=0.81

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine nii

!       This subroutine loads up the SII matrix for inversion
        subroutine sii(b,nl,t,ne)
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for SII
        g(1)=4.
        g(2)=4.
        g(3)=6.
        g(4)=2.
        g(5)=4.

!       Energy levels of SII (eV)
        e(1)=0.
        e(2)=1.843  
        e(3)=1.847 
        e(4)=3.043
        e(5)=3.049

!       A-values for SII, Mendoza & Zeippen 1982 MNRAS 198, 111.
        a(5,4)=1.03e-6
        a(5,3)=0.179
        a(5,2)=0.133
        a(5,1)=0.225
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=7.79e-2
        a(4,2)=1.63e-1
        a(4,1)=9.06e-2
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=3.35e-7
        a(3,1)=2.60e-4
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=8.82e-4

!       Collision strengths for SII; t=T/10^4 from
!       Keenan et al 1996, MNRAS 281, 1073
        o(1,2)=2.76*(t**(-0.15))
        o(1,3)=4.14*(t**(-0.15))
        o(1,4)=1.17*(t**(-0.07))
        o(1,5)=2.35 *(t**(-0.07))
        o(2,3)=7.47*(t**(-0.10))
        o(2,4)=1.79*(t**(-0.17))
        o(2,5)=3.00*(t**(-0.12))
        o(3,4)=2.20*(t**(-0.11))
        o(3,5)=4.99*(t**(-0.15))
        o(4,5)=2.71*(t**(-0.14))

call loadb(b,nl,t,ne,g,e,a,o)
return
end subroutine sii



subroutine loadb(b,nl,t,ne,g,e,a,o)
REAL*8 :: b(nl,nl),q12,q13,q14,q15,q23,q24,q25,q34,q35,q45,q21,q31,q41,q51,q32,q42,q52,q43,q53,q54,q5,q4,q3,q2
REAL ::  g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

q12=(8.63e-8)*o(1,2)*(exp(-(e(2)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q13=(8.63e-8)*o(1,3)*(exp(-(e(3)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q14=(8.63e-8)*o(1,4)*(exp(-(e(4)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q15=(8.63e-8)*o(1,5)*(exp(-(e(5)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q23=(8.63e-8)*o(2,3)*(exp(-(e(3)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q24=(8.63e-8)*o(2,4)*(exp(-(e(4)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q25=(8.63e-8)*o(2,5)*(exp(-(e(5)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q34=(8.63e-8)*o(3,4)*(exp(-(e(4)-e(3))/(0.8614*t)))/(g(3)*(t**0.5))
q35=(8.63e-8)*o(3,5)*(exp(-(e(5)-e(3))/(0.8614*t)))/(g(3)*(t**0.5))
q45=(8.63e-8)*o(4,5)*(exp(-(e(5)-e(4))/(0.8614*t)))/(g(4)*(t**0.5))
q21=(8.63e-8)*o(1,2)/(g(2)*(t**0.5))
q31=(8.63e-8)*o(1,3)/(g(3)*(t**0.5))
q41=(8.63e-8)*o(1,4)/(g(4)*(t**0.5))
q51=(8.63e-8)*o(1,5)/(g(5)*(t**0.5))
q32=(8.63e-8)*o(2,3)/(g(3)*(t**0.5))
q42=(8.63e-8)*o(2,4)/(g(4)*(t**0.5))
q52=(8.63e-8)*o(2,5)/(g(5)*(t**0.5))
q43=(8.63e-8)*o(3,4)/(g(4)*(t**0.5))
q53=(8.63e-8)*o(3,5)/(g(5)*(t**0.5))
q54=(8.63e-8)*o(4,5)/(g(5)*(t**0.5))
q5=q51+q52+q53+q54
q4=q41+q42+q43+q45
q3=q31+q32+q34+q35
q2=q21+q23+q24+q25

! Matrix values before inversion (row,column)
b(1,1)=ne*q15
b(1,2)=ne*q25
b(1,3)=ne*q35
b(1,4)=ne*q45
b(1,5)=-a(5,5)-ne*q5
b(2,1)=ne*q14
b(2,2)=ne*q24
b(2,3)=ne*q34
b(2,4)=-a(4,4)-ne*q4
b(2,5)=a(5,4)+ne*q54
b(3,1)=ne*q13
b(3,2)=ne*q23
b(3,3)=-a(3,3)-ne*q3
b(3,4)=a(4,3)+ne*q43
b(3,5)=a(5,3)+ne*q53
b(4,1)=ne*q12
b(4,2)=-a(2,1)-ne*q2
b(4,3)=a(3,2)+ne*q32
b(4,4)=a(4,2)+ne*q42
b(4,5)=a(5,2)+ne*q52
b(5,1)=1
b(5,2)=1
b(5,3)=1
b(5,4)=1
b(5,5)=1
return
end subroutine loadb



subroutine ludcmp(b,nl,indx,s)
PARAMETER (tiny=1e-28)
REAL*8 :: b(nl,nl), indx(nl), vv(nl),aamax,dum, sum
REAL :: s
INTEGER :: i, j, k, nl, imax

s=1.
do 12 i=1, nl
  aamax=0.
  do 11 j=1, nl
    if(abs(b(i,j)) > aamax) aamax=abs(b(i,j))
11  continue
    if(aamax == 0.) pause 'singular matrix'
    vv(i)=1./aamax
12  continue
do 19 j=1,nl
  do 14 i=1,j-1
    sum=b(i,j)
      do 13 k=1,i-1
      sum=sum-b(i,k)*b(k,j)
13    continue
    b(i,j)=sum
14    continue
  aamax=0.
  do 16 i=j,nl
    sum=b(i,j)
  do 15 k=1,j-1
    sum=sum-b(i,k)*b(k,j)
15  continue
    b(i,j)=sum
    dum=vv(i)*abs(sum)
    if(dum >= aamax) then
      imax=i
      aamax=dum
    endif
16        continue
        if(j /= imax) then
                do 17 k=1,nl
                        dum=b(imax,k)
                        b(imax,k)=b(j,k)
                        b(j,k)=dum
17              continue
                s=-s
                vv(imax)=vv(j)
        endif
        indx(j)=imax
        if(b(j,j) == 0) b(j,j)=tiny
        if(j /= nl) then
                dum=1./b(j,j)
                do 18 i=j+1,nl
                        b(i,j)=b(i,j)*dum
18              continue
        endif
19  continue
return
end subroutine ludcmp



subroutine lubksb(b,nl,indx,den)
REAL*8 b(nl,nl), indx(nl), den(nl), sum
INTEGER :: i, ii, j, ll

ii=0
do 12 i=1,nl
  ll=indx(i)
  sum=den(ll)
  den(ll)=den(i)
  if(ii /= 0) then
    do 11 j=ii, i-1
      sum=sum-b(i,j)*den(j)
11    continue
  else if (sum /= 0.) then
    ii=i
  endif
  den(i)=sum
12  continue
do 14 i=nl,1,-1
        sum=den(i)
        do 13 j=i+1,nl
                sum=sum-b(i,j)*den(j)
13  continue
        den(i)=sum/b(i,i)
14  continue
return
end subroutine lubksb


