!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    i_emission.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
SUBROUTINE CalcEmiss(NTXvals,nx,ny,nz,species,spec_char,exlevels,lev1_char,lev2_char)

! 2009, 04/28: input is N_e, T, Xh, N_Htot.  This accounts for
!              potential He ionization, which means N_e /= N_Hii
!              (unless it's a purely H run)

! 2008, 08/27: input is N_e, T, N_Htot (with N_e assumed equal to N_Hii) 

! This program takes in an array of density, energy & ionization
! fraction values for a 3-D volume and then calculates the emissivity
! at each point and sums it up along the line of sight.  It does this
! for the user's choice of OI, OII, NI, NII, or SII, and for the
! desired energy level transitions (first 5 energy levels).

! It uses a modified version of Dr. Hartigan's fiveln.f program 
! which calculates the level populations of the chosen species.

! NOTE:  Compile this program with the module:
!         emiss_calcs.f90 - contains the subroutines for calculating 
!                           the emissivities and summing them up,
! and the external subroutine:
!         writfits_r4.f - for outputing *.fits images


INTEGER, PARAMETER :: nl=5, &   ! number of energy levels to choose from
                      ns=5      ! number of species to choose from : OI,OII,NI,NII,SII
INTEGER,PARAMETER :: filehandle=321
INTEGER :: listx,listy,listz, m, i, j, k, counter, &   ! various counters for do loops, etc. 
           nx, ny, nz, &        ! array size
           species(:)             ! chosen species
INTEGER, DIMENSION(:,:) :: exlevels ! initial & final energy levels for emission

REAL :: zne, t, temp, ne, s
REAL*8 :: b(nl,nl), indx(nl), vv(nl), den(nl)

REAL, DIMENSION(3) :: gridunits ! 
REAL, DIMENSION(:,:,:,:) :: NTXvals
REAL, ALLOCATABLE, DIMENSION(:,:,:) :: emiss1, Nprobs

CHARACTER(LEN=1),DIMENSION(:) :: lev1_char, lev2_char
CHARACTER(LEN=3),DIMENSION(:) :: spec_char
CHARACTER(LEN=5) :: lambda
CHARACTER(LEN=6) :: atom(ns)
CHARACTER(LEN=85) :: filename, partname

!!$! Get the filename
!!$WRITE (*,'(A)',ADVANCE='NO') ' Enter the filename: '
!!$READ (*,'(A)') filename
!!$!filename ='/projects/hartigan/jnp1/chombofiles/thesis_4cell_200ypulse_2500y_NeXhTNhtot.emiss'
!!$
!!$OPEN(20, file = filename, ACTION='READ')
!!$PRINT *
!!$PRINT *,'Opening: ',filename
!!$
!!$
!!$READ(20,*) gridunits            !  x, y, z, defined in bear.data
!!$READ(20,*) counter, nx, ny, nz  !  ignore counter.  nx... = # of cells in fixed grid


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
! Currently taking, from Chombo, Ne, T, Xh (H ionization fraction), and N_Htot
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

!PRINT*,species,spec_char,exlevels,lHSiiBOV

gridunits= (/ XUpper(1), XUpper(2), XUpper(3) /)


! Set the dimensions for the allocatable arrays
ALLOCATE( Nprobs(nx,ny,nz), emiss1(nx,ny,nz) )


IF(lEmissBOV) THEN
   DO i=1,SIZE(species)
      ! create a CHARACTER field to be tagged onto file names, specifying the species & levels
      partname = TRIM(spec_char(i))//'_'//lev1_char(i)//'-'//lev2_char(i)

      PRINT*,'Begin ',spec_char(i),' transition ',lev1_char(i),'-',lev2_char(i)

      IF(species(i)==6) THEN
         ! halpha
         CALL alpha_emiss(NTXvals, nx,ny,nz, emiss1)
         IF(ANY(ISNAN(emiss1))) THEN
            PRINT*,':('
         END IF
      ELSE
         counter = 0
         DO listx = 1, nx
         DO listy = 1, ny
         DO listz = 1, nz
            ! Put T values in 10^4 K
            t = NTXvals(2,listx,listy,listz)/(10.**4)

            IF (NTXvals(1,listx,listy,listz) /= 0.) THEN
               ne = NTXvals(1,listx,listy,listz)
               zne = ne

               ! Matrix b gets overwritten each time when inverted, so need
               ! to initialize DO loop for each atomic ionization state
               if(species(i)== 1) then
                  call oi(b,nl,t,ne)
               elseif(species(i)== 2) then
                  call oii(b,nl,t,ne)
               elseif(species(i)== 3) then
                  call ni(b,nl,t,ne)
               elseif(species(i)== 4) then
                  call nii(b,nl,t,ne)
               elseif(species(i)== 5) then
                  call sii(b,nl,t,ne)
               END IF
               
               ! Do Matrix Inversion - note values in b and den are overwritten
               ! input: matrix b(i,j) and r.h.s. is (0,0,0,0,1) stored in den(i)
               ! output: LU decomposed matrix in b(i,j) and density of levels in den(i)
               ! From numerical recipes in Fortran
               do m=1,nl-1
                  den(m)=0.
               END DO
               den(nl)=1.
               call ludcmp(b,nl,indx,s)
               call lubksb(b,nl,indx,den)
            ELSE
               den(1) = 1.
               do m=2,nl
                  den(m)=0.
               end do
            END IF

            Nprobs(listx,listy,listz) = den(exlevels(i,1))
            IF (Nprobs(listx, listy, listz) == -0.) Nprobs(listx, listy, listz) = 0.0

         END DO   ! end listz
         END DO   ! end listy
         END DO   ! end listx

         ! The subroutine `emissivity' calculates the emissivities from the
         ! `n' values determined above along with Einstein A coefficients and
         !  Energies of level transitions.
         CALL emissivity(Nprobs, nx, ny, nz, NTXvals, emiss1, exlevels(i,:), species(i))
      END IF

      ! Call subroutine that sums up the the emissivities along the line of sight
      CALL sum_emiss(emiss1, nx, ny, nz, partname, gridunits)


      PRINT*,'min max',MINVAL(emiss1),MAXVAL(emiss1)
      pFix(:,:,:,5) = emiss1

      write(FileName,'(A8,A4,I3.3,A4)') "out/"//spec_char(i)//"_",lev1_char(i)//'-'//lev2_char(i)//'_',n,".dat"
!      PRINT*,'file: ',filename
      OPEN(UNIT=filehandle, FILE=FileName, status="replace", FORM="unformatted")
      write(filehandle) RESHAPE( emiss1, (/ nx*ny*nz /)) !TRANSPOSE WAS HERE
      CLOSE(filehandle)

      write(FileName,'(A8,A4,I3.3,A4)') "out/"//spec_char(i)//"_",lev1_char(i)//'-'//lev2_char(i)//'_',n,".bov"
      OPEN(UNIT=filehandle, FILE=Filename)
      WRITE(filehandle,'(A6E15.3)')  "TIME: ", gi_fixed%time
      write(Filehandle,'(A15,A4,I3.3,A4)') "DATA_FILE: "//spec_char(i)//"_",lev1_char(i)//'-'//lev2_char(i)//'_',n,".dat"
!    write(filehandle,'(A18,I3.3,A4)') "DATA_FILE: Halpha_",n,".dat"
      WRITE(filehandle,'(A11,3I12)')  "DATA_SIZE: ", (/ nx, ny, nz /)
      WRITE(filehandle,*)  "DATA_FORMAT: REAL"
      WRITE(filehandle,*)  "VARIABLE: "//spec_char(i)
      WRITE(filehandle,*)  "DATA_ENDIAN: LITTLE"
      WRITE(filehandle,*)  "CENTERING: zonal"

      WRITE(filehandle,'(A14,3E26.16)')  "BRICK_ORIGIN: ", (/ 0d0,0d0,0d0 /)
      WRITE(filehandle,'(A12,3E26.16)')  "BRICK_SIZE: ", (/ REAL(nx),REAL(ny),REAL(nz) /)

      WRITE(filehandle,*)  "BYTE_OFFSET: 4"
      WRITE(filehandle,'(A17,I4)')  "DATA_COMPONENTS: ",1
      WRITE(filehandle,'(A9,8I8)')  "MGLOBAL: ", 1,1,1, nx,ny,nz
      CLOSE(filehandle)
   END DO


ELSE

ENDIF


!!$DEALLOCATE(Nprobs, NTXvals, emiss1)
DEALLOCATE(Nprobs, emiss1)

END SUBROUTINE CalcEmiss



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



! The following are Dr.Hartigan's subroutines called in the main
! part of his program above.


!       This subroutine loads up the OI matrix for inversion
        subroutine oi(b,nl,t,ne)
          INTEGER nl
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl), t, ne

!        Statistical weights for OI
        g(1)=5.
        g(2)=3.
        g(3)=1.
        g(4)=5.
        g(5)=1.

!        Energy levels of OI (eV)
        e(1)=0
        e(2)=0.0196
        e(3)=0.0282
        e(4)=1.9689
        e(5)=4.1931

!        A-values for OI, Baluja, K., and Zeippen, C. 1988,
!        J. Phys B. Atom. Mol. Opt. Phys. 21, 1455.
        a(5,4)=1.215
        a(5,3)=0.
        a(5,2)=7.60e-2
        a(5,1)=2.73e-4
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=8.93e-7
        a(4,2)=1.82e-3
        a(4,1)=5.63e-3
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.74e-5
        a(3,1)=1.20e-10
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=8.96e-5

!         Collision strengths for OI; t=T/10^4, From Berrington and Burke 1981 Plan. Sp. Sci. 29, 377
!         except for the ^3P-^3P, which are from Le Dourneuf, M. and Nesbet, R. 1976, J Phys B.
!         Atom. Molec. Phys. 9, L241. Fits are mine, and work very well.
        o(1,4)=(5./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(2,4)=(3./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(3,4)=(1./9.)*0.266*(t**0.97)*(t**(log10(t**(-0.27))))
        o(1,5)=(5./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(2,5)=(3./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(3,5)=(1./9.)*0.0324*t*(t**(log10(t**(-0.22))))
        o(1,2)=0.0987*t**1.11
        o(2,3)=0.0265*t**1.17
        o(1,3)=0.0292*t*(t**(log10(t**(-0.13))))
        o(4,5)=0.105*t**0.54

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine oi

!       This subroutine loads up the OII matrix for inversion
        subroutine oii(b,nl,t,ne)
          INTEGER nl
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for OII
        g(1)=4.
        g(2)=6.
        g(3)=4.
        g(4)=4.
        g(5)=2.

!       Energy levels of OII (eV)
        e(1)=0
        e(2)=3.3267
        e(3)=3.3292
        e(4)=5.0214
        e(5)=5.0216

!       A-values for OII, Zeippen, C. 1982, MNRAS 198, 111
!       (better than Zeippen, C. 1987, A&A 173, 410 according to
!       planetary nebula observations of Wang et al 2004 A&A 427, 873)
        a(5,4)=2.08e-11
        a(5,3)=1.02e-1
        a(5,2)=6.15e-2
        a(5,1)=2.32e-2
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=6.14e-2
        a(4,2)=1.17e-1
        a(4,1)=5.64e-2
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.2e-7
        a(3,1)=1.65e-4
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=3.82e-5

!       Collision strengths for OII; t=T/10^4
!       McLaughlin, B., and Bell, K. results 1993 ApJ 408, 753 are spurious.
!       (see Wang et al 2004 A&A 427, 873). 
!       Instead, use Pradhan 1976 MNRAS 177, 31P 
        o(1,2)=0.6*1.335*(t**0.015)
        o(1,3)=0.4*1.335*(t**0.015)
        o(1,4)=(2./3.)*0.406*(t**0.02)
        o(1,5)=(1./3.)*0.406*(t**0.03)
        o(2,3)=1.168*(t**(-0.06))
        o(2,4)=1.706*0.428*(t**0.05)
        o(2,5)=1.706*0.173*(t**0.05)
        o(3,4)=1.706*0.239*(t**0.05)
        o(3,5)=1.706*0.161*(t**0.05)
        o(4,5)=0.287*(t**0.05)

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine oii

!       This subroutine loads up the NI matrix for inversion
        subroutine ni(b,nl,t,ne)
          INTEGER nl
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for NI
        g(1)=4.
        g(2)=6.
        g(3)=4.
        g(4)=2.
        g(5)=4.

!       Energy levels of NI (eV)
        e(1)=0
        e(2)=2.3854
        e(3)=2.3865
        e(4)=3.57840
        e(5)=3.57845

!       A-values for NI, Butler, K., and Zeippen, C. 1984, A&A 141, 274
        a(5,4)=0
        a(5,3)=2.523e-2
        a(5,2)=5.589e-2
        a(5,1)=6.61e-3
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=4.80e-2
        a(4,2)=3.14e-2
        a(4,1)=2.72e-3
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=1.24e-8
        a(3,1)=2.28e-5
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=6.13e-6

!       Collision strengths for NI; t=T/10^4, Berrington and Burke 1981, Pl. Sp.Sci. 29, 377
!       ^4S terms fit REALly well to log10(Omega) = a + b*log10(T/10^4) + c*(log10(T/10^4))^2
!       ^2D-^2P are from BB81, fit with power law, with internal fine structure ratios (not ~ g!!)
!       as described in Table II of Pradhan 1976, MNRAS 177, 31P, which agree with 
!       ratios of Dopita Mason and Robb 1976 ApJ 207, 102.  ^2D-^2D and ^2P-^2P use DMR76 power
!       law fits.
        o(1,2)=0.6*0.484*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,3)=0.4*0.484*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,4)=(1./3.)*0.170*(t**0.82)*(t**(log10(t**(-0.30))))
        o(1,5)=(2./3.)*0.170*(t**0.82)*(t**(log10(t**(-0.30))))
        o(2,3)=0.773-0.825*(exp(-t/2.03))
        o(2,4)=0.173*0.819*(t**0.59)
        o(2,5)=0.428*0.819*(t**0.59)
        o(3,4)=0.161*0.819*(t**0.59)
        o(3,5)=0.239*0.819*(t**0.59)
        o(4,5)=0.071*(t**1.11)

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine ni

!       This subroutine loads up the NII matrix for inversion
        subroutine nii(b,nl,t,ne)
          INTEGER :: nl
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for NII
        g(1)=1.
        g(2)=3.
        g(3)=5.
        g(4)=5.
        g(5)=1.

!       Energy levels of NII (eV)
        e(1)=0.
        e(2)=0.00604
        e(3)=0.0162
        e(4)=1.900
        e(5)=4.056

!       A-values for NII, Bell, Hibbert, Stafford 1995, Phys. Scr. 52, 240
!                   Storey and Zeippen 2000, MNRAS 312, 813.
!                   Froese Fischer, C. and Saha, H. 1985, Phys Scr. 32, 181.
        a(5,4)=0.923
        a(5,3)=1.435e-4
        a(5,2)=4.588e-2
        a(5,1)=0
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=3.015e-3
        a(4,2)=9.819e-4
        a(4,1)=1.928e-6
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=7.46e-6 
        a(3,1)=1.062e-12
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=2.08e-6

!       Collision strengths for NII; t=T/10^4 from Hudson and Bell A&A 430, 725, 2005
!       temperature dependence is small, not clearly a power law, and I am not sure I
!       believe it anyway. Makes the most difference for lowest 3 levels when T > 10^4K
        o(1,2)=0.43
        o(1,3)=0.27
        o(1,4)=0.30
        o(1,5)=0.035
        o(2,3)=1.15
        o(2,4)=0.91
        o(2,5)=0.11
        o(3,4)=1.51
        o(3,5)=0.18
        o(4,5)=0.81

        call loadb(b,nl,t,ne,g,e,a,o)
        return
        end subroutine nii

!       This subroutine loads up the SII matrix for inversion
        subroutine sii(b,nl,t,ne)
          INTEGER nl
        REAL*8 :: b(nl,nl)
        REAL :: g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

!       Statistical weights for SII
        g(1)=4.
        g(2)=4.
        g(3)=6.
        g(4)=2.
        g(5)=4.

!       Energy levels of SII (eV)
        e(1)=0.
        e(2)=1.843  
        e(3)=1.847 
        e(4)=3.043
        e(5)=3.049

!       A-values for SII, Mendoza & Zeippen 1982 MNRAS 198, 111.
        a(5,4)=1.03e-6
        a(5,3)=0.179
        a(5,2)=0.133
        a(5,1)=0.225
        a(5,5)=a(5,4)+a(5,3)+a(5,2)+a(5,1)
        a(4,3)=7.79e-2
        a(4,2)=1.63e-1
        a(4,1)=9.06e-2
        a(4,4)=a(4,3)+a(4,2)+a(4,1)
        a(3,2)=3.35e-7
        a(3,1)=2.60e-4
        a(3,3)=a(3,2)+a(3,1)
        a(2,1)=8.82e-4

!       Collision strengths for SII; t=T/10^4 from
!       Keenan et al 1996, MNRAS 281, 1073
        o(1,2)=2.76*(t**(-0.15))
        o(1,3)=4.14*(t**(-0.15))
        o(1,4)=1.17*(t**(-0.07))
        o(1,5)=2.35 *(t**(-0.07))
        o(2,3)=7.47*(t**(-0.10))
        o(2,4)=1.79*(t**(-0.17))
        o(2,5)=3.00*(t**(-0.12))
        o(3,4)=2.20*(t**(-0.11))
        o(3,5)=4.99*(t**(-0.15))
        o(4,5)=2.71*(t**(-0.14))

call loadb(b,nl,t,ne,g,e,a,o)
return
end subroutine sii



subroutine loadb(b,nl,t,ne,g,e,a,o)
  INTEGER nl
REAL*8 :: b(nl,nl),q12,q13,q14,q15,q23,q24,q25,q34,q35,q45,q21,q31,q41,q51,q32,q42,q52,q43,q53,q54,q5,q4,q3,q2
REAL ::  g(nl),e(nl),a(nl,nl),o(nl,nl),t, ne

q12=(8.63e-8)*o(1,2)*(exp(-(e(2)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q13=(8.63e-8)*o(1,3)*(exp(-(e(3)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q14=(8.63e-8)*o(1,4)*(exp(-(e(4)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q15=(8.63e-8)*o(1,5)*(exp(-(e(5)-e(1))/(0.8614*t)))/(g(1)*(t**0.5))
q23=(8.63e-8)*o(2,3)*(exp(-(e(3)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q24=(8.63e-8)*o(2,4)*(exp(-(e(4)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q25=(8.63e-8)*o(2,5)*(exp(-(e(5)-e(2))/(0.8614*t)))/(g(2)*(t**0.5))
q34=(8.63e-8)*o(3,4)*(exp(-(e(4)-e(3))/(0.8614*t)))/(g(3)*(t**0.5))
q35=(8.63e-8)*o(3,5)*(exp(-(e(5)-e(3))/(0.8614*t)))/(g(3)*(t**0.5))
q45=(8.63e-8)*o(4,5)*(exp(-(e(5)-e(4))/(0.8614*t)))/(g(4)*(t**0.5))
q21=(8.63e-8)*o(1,2)/(g(2)*(t**0.5))
q31=(8.63e-8)*o(1,3)/(g(3)*(t**0.5))
q41=(8.63e-8)*o(1,4)/(g(4)*(t**0.5))
q51=(8.63e-8)*o(1,5)/(g(5)*(t**0.5))
q32=(8.63e-8)*o(2,3)/(g(3)*(t**0.5))
q42=(8.63e-8)*o(2,4)/(g(4)*(t**0.5))
q52=(8.63e-8)*o(2,5)/(g(5)*(t**0.5))
q43=(8.63e-8)*o(3,4)/(g(4)*(t**0.5))
q53=(8.63e-8)*o(3,5)/(g(5)*(t**0.5))
q54=(8.63e-8)*o(4,5)/(g(5)*(t**0.5))
q5=q51+q52+q53+q54
q4=q41+q42+q43+q45
q3=q31+q32+q34+q35
q2=q21+q23+q24+q25

! Matrix values before inversion (row,column)
b(1,1)=ne*q15
b(1,2)=ne*q25
b(1,3)=ne*q35
b(1,4)=ne*q45
b(1,5)=-a(5,5)-ne*q5
b(2,1)=ne*q14
b(2,2)=ne*q24
b(2,3)=ne*q34
b(2,4)=-a(4,4)-ne*q4
b(2,5)=a(5,4)+ne*q54
b(3,1)=ne*q13
b(3,2)=ne*q23
b(3,3)=-a(3,3)-ne*q3
b(3,4)=a(4,3)+ne*q43
b(3,5)=a(5,3)+ne*q53
b(4,1)=ne*q12
b(4,2)=-a(2,1)-ne*q2
b(4,3)=a(3,2)+ne*q32
b(4,4)=a(4,2)+ne*q42
b(4,5)=a(5,2)+ne*q52
b(5,1)=1
b(5,2)=1
b(5,3)=1
b(5,4)=1
b(5,5)=1
return
end subroutine loadb



subroutine ludcmp(b,nl,indx,s)
REAL*8, PARAMETER :: tiny=1e-28
REAL*8 :: b(nl,nl), indx(nl), vv(nl),aamax,dum, sum
REAL :: s
INTEGER :: i, j, k, nl, imax

s=1.
do 12 i=1, nl
  aamax=0.
  do 11 j=1, nl
    if(abs(b(i,j)) > aamax) aamax=abs(b(i,j))
11  continue
    if(aamax == 0.) print*, 'singular matrix'
    vv(i)=1./aamax
12  continue
do 19 j=1,nl
  do 14 i=1,j-1
    sum=b(i,j)
      do 13 k=1,i-1
      sum=sum-b(i,k)*b(k,j)
13    continue
    b(i,j)=sum
14    continue
  aamax=0.
  do 16 i=j,nl
    sum=b(i,j)
  do 15 k=1,j-1
    sum=sum-b(i,k)*b(k,j)
15  continue
    b(i,j)=sum
    dum=vv(i)*abs(sum)
    if(dum >= aamax) then
      imax=i
      aamax=dum
    endif
16        continue
        if(j /= imax) then
                do 17 k=1,nl
                        dum=b(imax,k)
                        b(imax,k)=b(j,k)
                        b(j,k)=dum
17              continue
                s=-s
                vv(imax)=vv(j)
        endif
        indx(j)=imax
        if(b(j,j) == 0) b(j,j)=tiny
        if(j /= nl) then
                dum=1./b(j,j)
                do 18 i=j+1,nl
                        b(i,j)=b(i,j)*dum
18              continue
        endif
19  continue
return
end subroutine ludcmp



subroutine lubksb(b,nl,indx,den)
REAL*8 b(nl,nl), indx(nl), den(nl), sum
INTEGER :: i, ii, j, ll, nl

ii=0
do 12 i=1,nl
  ll=indx(i)
  sum=den(ll)
  den(ll)=den(i)
  if(ii /= 0) then
    do 11 j=ii, i-1
      sum=sum-b(i,j)*den(j)
11    continue
  else if (sum /= 0.) then
    ii=i
  endif
  den(i)=sum
12  continue
do 14 i=nl,1,-1
        sum=den(i)
        do 13 j=i+1,nl
                sum=sum-b(i,j)*den(j)
13  continue
        den(i)=sum/b(i,i)
14  continue
return
end subroutine lubksb


SUBROUTINE emissivity(Nprobs1, nx1, ny1, nz1, NTXvals1, emissivities7, exlevels1, species1) 

! This subroutine calculates the emissivity from the formula (n*A*hv/(4Pi))
! The units are erg/cm^3/s/str


IMPLICIT NONE

REAL, PARAMETER :: Pi = 3.14159, k = 8.617E-5, & ! k is in eV/K

                   !Ionization Energies of neutral atoms
                   EionOI = 13.62, &
                   EionNI = 14.5341, &
                   EionHI = 13.60, &

                   ! Abundances, from Hartigan & Morse, APJ, 660, 2007 (solar abundances)
                   AbundO = 8.82, &
                   AbundN = 7.96, &
                   AbundS = 7.30

INTEGER :: Lx, Ly, Lz, i, & ! loop counters
           species1,      & ! chosen species for emission line
           nx1, ny1, nz1    ! 

INTEGER, DIMENSION(2) :: exlevels1 ! initial and final energy levels

REAL :: gOI, gOII, gNI, gNII, gHI, gHII, & ! Statistical weights
        Abund_rat                          ! Abundance of species, relative to Hydrogen

REAL, DIMENSION(5,5,5) :: A_vals,  &  ! Einstein A values
                          hv_ergs     ! Energies of level transitions, in ergs.

REAL, ALLOCATABLE, DIMENSION(:,:,:)  :: nO, nN, nS, &  ! Total number densities
                                        nIon_Neut, &   ! fraction of ion/neutral for given species
                                        nHII_nHI, &    ! fraction of ion/neutral for H
                                        kT, &          ! temperature in eV
                                        nSpec, &       ! includes: nOI, nOII, nNI, nNII, nSII
                                        Nprobs1,  &    ! fraction of a species in a given level
                                        NLevels1, &    ! number density of species in level
                                        emissivities7  ! calculated emissivity
 
REAL, DIMENSION(:,:,:,:) :: NTXvals1      ! array of n_e, T, Xh, nH_total


ALLOCATE( nO(nx1,ny1,nz1), nN(nx1,ny1,nz1), nS(nx1,ny1,nz1), & 
          nIon_Neut(nx1,ny1,nz1), nHII_nHI(nx1,ny1,nz1), kT(nx1,ny1,nz1), &
          nSpec(nx1,ny1,nz1), NLevels1(nx1,ny1,nz1) )


! Call subroutines to get the A-values and excitement energies that will be needed later
CALL A_Values(A_vals)
CALL E_Values(hv_ergs)


! kT values, in eV:
kT = k*NTXvals1(2,:,:,:)


! get abundance of species relative to Hydrogen
IF ( (species1 == 1) .OR.(species1 == 2) ) THEN
  Abund_rat = (AbundO - 12.)
ELSEIF ( (species1 == 3) .OR.(species1 == 4) ) THEN
  Abund_rat = (AbundN - 12.)
ELSEIF ( (species1 == 5) ) THEN
  Abund_rat = (AbundS - 12.)
ELSE
  PRINT *,'WTF?'
  STOP
END IF


! Statistical weights (ground states), not needed for SII, since all S is in SII
gHI = 2
gHII = 1

gOI = 9     ! (3)P(2,1,0) => gOI  = (2*2 + 1) + (2*1 + 1) + (2*0 + 1) = 9
gOII = 4    ! (4)S(3/2)   => gOII = (2*(3/2) + 1) = 4

gNI = 4     ! (4)S(3/2)   => gNI  = (2*(3/2) + 1) = 4
gNII = 9    ! (3)P(2,1,0) => gNII = (2*2 + 1) + (2*1 + 1) + (2*0 + 1) = 9


!*******************************************************************************************
!*******************************************************************************************
!*******************************************************************************************


! Calculate the densities of each excitation level for each species:
! Dr.Hartigan's program gives the ratio of the density of a species in a given
! level to the total density of that species.

! 1. Therefore, we need to multiply that ratio by the total density of the species.
! 2. To get total density of species, need total density of atom (neutral + ionized).
! 3. Get ratio of neutral to ionized from charge exchange.



! Beginning with # 3.
! [HI][OII] <-> [HII][OI]
! gHI*n[HI]*gOII*n[OII]*exp(-EionOI/kT) = 
! (n[HI]/n[HII])(n[OII]/n{OI]) = [(gHII*gOI)/(gHI*gOII)]*exp(-[EionHI-EionOI]/kT)

! # 2.
! nO = nOI + nOII = nOI(1 + nOII/nOI)    and also
! nH/nO = 10^12/10^8.82 -> nO = nH*10^8.82/10^12
! 	Combine:
! nOI = nH*10^(8.82-12) / (1+nOII/nOI)

! # 1. (e.g., for level 2)
! n(2,DrH,OI) = (n2,OI)/nOI -> n2,OI = nOI * n(2,DrH,OI)


!*******************************************************************************************
!*******************************************************************************************
!*******************************************************************************************


! Begin steps to calculate emissivity

! Total densities of O, N, & S, for each value of ne
! N_species = nHtot * 10^(Abund-12)
IF ( (species1 == 1) .OR. (species1 == 2) ) THEN
  nO = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )

ELSEIF ( (species1 == 3) .OR. (species1 == 4) ) THEN
  nN = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )

ELSEIF ( (species1 == 5) ) THEN
  nS = NTXvals1(4,:,:,:) * ( 10.**(Abund_rat) )  ! = nSII, since all S is singly ionized.

END IF



! Next:
! Calculate ratio of nHII/nHI using:
! Xh = nHII/(nHI + nHII) = (nHII/nHI)[(1 + nHII/nHI)^-1]
! -> nHII/nHI = Xh/(1 - Xh)

WHERE ( (1. - NTXvals1(3,:,:,:)) == 0.0 ) &
      nHII_nHI(:,:,:) =  NTXvals1(3,:,:,:)/(1. - NTXvals1(3,:,:,:) + 1.e-10)

WHERE ( (1. - NTXvals1(3,:,:,:)) /= 0.0 ) &
      nHII_nHI(:,:,:) =  NTXvals1(3,:,:,:)/(1. - NTXvals1(3,:,:,:))



! Use Charge exchange and level populations to get number density of
! chosen species & level.

IF (species1 == 1) THEN ! OI
! nIon_Neut = nOII/nOI

    nIon_Neut(:,:,:) = nHII_nHI(:,:,:) * (gHI*gOII/(gHII*gOI)) * &
      EXP((EionHI - EionOI)/kT(:,:,:))
    nSpec(:,:,:) = nO(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 2) THEN  ! OII
! nIon_Neut = nOI/nOII

  WHERE (nHII_nHI == 0.0) nHII_nHI = 1.e-10

    nIon_Neut(:,:,:) = (1./nHII_nHI(:,:,:)) * (gHII*gOI/(gHI*gOII)) * &
      EXP(-(EionHI - EionOI)/kT(:,:,:))
    nSpec(:,:,:) = nO(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 3) THEN  ! NI
! nIon_Neut = nNII/nNI

    nIon_Neut(:,:,:) = nHII_nHI(:,:,:) * (gHI*gNII/(gHII*gNI)) * &
      EXP((EionHI - EionNI)/kT(:,:,:))
    nSpec(:,:,:) = nN(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 4) THEN  ! NII
! nIon_Neut = nNI/nNII

  WHERE (nHII_nHI == 0.0) nHII_nHI = 1.e-10

    nIon_Neut(:,:,:) = (1./nHII_nHI(:,:,:)) * (gHII*gNI/(gHI*gNII)) * &
       EXP(-(EionHI - EionNI)/kT(:,:,:))
    nSpec(:,:,:) = nN(:,:,:) / (1. + nIon_Neut(:,:,:))
    NLevels1(:,:,:) = Nprobs1(:,:,:)*nSpec(:,:,:)



ELSEIF (species1 == 5) THEN  ! SII
  NLevels1(:,:,:) = Nprobs1(:,:,:)*nS(:,:,:) ! nSpec (ie nSII) = nS

ELSE
   PRINT *, 'Something evil this way came'
   STOP

END IF    


!  Calculate the emissivity for each point, for the given species and levels using:
!  emissivity = NLevels1 * A(final, init) * hv(final, init) 

emissivities7(:,:,:) = ( NLevels1(:,:,:) * A_vals(exlevels1(2),exlevels1(1),species1) * &
                       hv_ergs(exlevels1(2),exlevels1(1),species1) ) / (4.*3.14159)

END SUBROUTINE emissivity




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


SUBROUTINE alpha_emiss(NTXvalsA, nx3,ny3,nz3, emissAlpha)

! This subroutine calculates the emissivity of H-alpha from recombination and collisional
! excitation from levels 1 -> 3,4,5

IMPLICIT NONE

REAL, PARAMETER :: Halpha_hv = 3.0263E-12, & ! ergs, = 1.88889 eV, for H n= 3->2 transition
                   E_thr_n3 = 12.09, &       ! in eV, for n=1->3
                   E_thr_n4 = 12.75, &       ! in eV, for n=1->4
                   E_thr_n5 = 13.056, &      ! in eV, for n=1->5
                   k_B = 8.617E-5, &         ! in eV/K
                   vern_a = 7.982d-11, &     ! cm^3/s
                   vern_b = 0.7480d0, &
                   vern_T0 = 3.148d0, &      ! K
                   vern_T1 = 7.036d5         ! K

CHARACTER(LEN = 80) :: filename

INTEGER :: i,j,k                             ! counters
INTEGER, INTENT(IN) :: nx3, ny3, nz3         ! array dimensions

REAL (KIND=8), DIMENSION(4,25) :: coll_str   ! table of effective collision strengths (e-H)

REAL, ALLOCATABLE, DIMENSION(:,:,:) :: &
                   recombcoef, &      ! recombination rate coefficient
                   collexcite,  &     ! collisional excitation rate coefficient
                   emissAlpha_rec, &  ! emissivity due to H-alpha recombination
                   emissAlpha_coll, & ! emissivity due to collisional excitation
                   TeLog              ! Log of Temperature (in K)

REAL, ALLOCATABLE, INTENT(IN OUT), DIMENSION(:,:,:) :: emissAlpha ! Total H-alp emissivity

REAL, DIMENSION(:,:,:,:) :: NTXvalsA ! array of ne, T, Xh, nHtotal
REAL,ALLOCATABLE,DIMENSION(:,:,:,:) :: upsilon ! effective collision strengths at each cell


ALLOCATE( recombcoef(nx3,ny3,nz3), collexcite(nx3,ny3,nz3),  &
          emissAlpha_rec(nx3,ny3,nz3), emissAlpha_coll(nx3,ny3,nz3), &
          upsilon(3,nx3,ny3,nz3), TeLog(nx3,ny3,nz3) )  


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Recombination for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


! Recombination coeff's from Verner & Ferland (1996 ApJS..103..467V)
! Range valid from 3K to 10^9 K
! recombination coefficient is in cm^3/s

recombcoef(:,:,:) = vern_a / ( SQRT(NTXvalsA(2,:,:,:)/vern_T0) * &
                    ((1.d0 + SQRT(NTXvalsA(2,:,:,:)/vern_T0))**(1.d0-vern_b)) * &
                    ((1.d0 + SQRT(NTXvalsA(2,:,:,:)/vern_T1))**(1.d0+vern_b)) )

                        
! emiss_rec = hv * Ne *NHii * recombcoef / (4Pi)      where NHii = Xh*NHtot
emissAlpha_rec(:,:,:) = Halpha_hv * NTXvalsA(1,:,:,:) * (NTXvalsA(3,:,:,:)* &
                        NTXvalsA(4,:,:,:)) * recombcoef(:,:,:) / (4.*3.14159) 




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Collisional Excitation for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


! First calculate the rate coefficient (in cm^3/s), based on effective
! collision strengths from:
! Anderson (2000):  JPhB..2000..33..1255, JPhB..2002..35..1613


  OPEN(49,file='Anderson_Coll_Str.dat',ACTION='READ')
  READ(49,'(4(ES12.4))'), coll_str
  ! First column is T
  ! 2nd, 3rd & 4th are collision strengths (logged) from n = 1 to n=3,4,5 respectively
  CLOSE(49)

  TeLog = LOG10(NTXvalsA(2,:,:,:))
  upsilon = 0.0

  DO i = 1,24

! I don't know why I couldn't combine these all into one WHERE construct,
! since they all have the same condition...
    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(1,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(2,i+1)-coll_str(2,i)) + coll_str(2,i) )
    END WHERE

    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(2,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(3,i+1)-coll_str(3,i)) + coll_str(3,i) )
    END WHERE

    WHERE ( (TeLog >= coll_str(1,i)) .AND. (TeLog < coll_str(1,i+1)) )
      upsilon(3,:,:,:) = 10.**( ((TeLog-coll_str(1,i))/(coll_str(1,i+1)-coll_str(1,i))) &
                   *(coll_str(4,i+1)-coll_str(4,i)) + coll_str(4,i) )
    END WHERE

  END DO

! collision strengths are tiny at low T, so below T in coll_str table, just set them to
! the minimum table value.
  WHERE (TeLog < coll_str(1,1)) 
    upsilon(1,:,:,:) = 10.**coll_str(2,1)
    upsilon(2,:,:,:) = 10.**coll_str(3,1)
    upsilon(3,:,:,:) = 10.**coll_str(4,1)
  END WHERE


! Calculate the rate coefficient (in cm^3/s)
  collexcite = ((8.63E-6)/(2*SQRT(NTXvalsA(2,:,:,:))))* &
              ( (EXP(-E_thr_n3/(k_B*NTXvalsA(2,:,:,:)))*upsilon(1,:,:,:)) + &
                (EXP(-E_thr_n4/(k_B*NTXvalsA(2,:,:,:)))*upsilon(2,:,:,:)) + &
                (EXP(-E_thr_n5/(k_B*NTXvalsA(2,:,:,:)))*upsilon(3,:,:,:)) )



! emiss_coll = hv * ne * nHi * collexcite / (4Pi)   where nHi = nHtot-(Xh*nHtot)
emissAlpha_coll(:,:,:) = Halpha_hv * NTXvalsA(1,:,:,:) * &
                  (NTXvalsA(4,:,:,:)-(NTXvalsA(3,:,:,:)*NTXvalsA(4,:,:,:))) * &
                  collexcite(:,:,:) / (4.*3.14159)




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
!		Total Emission for H-alpha
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



! Total emission, combining recombination & collisional excitation
! in ergs/cm^3/s/str

emissAlpha = emissAlpha_rec + emissAlpha_coll


END SUBROUTINE alpha_emiss



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!




SUBROUTINE sum_emiss(emiss2, nx2, ny2, nz2, partname, gridunits2) 

IMPLICIT NONE

! This subroutine will sum the emissivities along the z-axis

CHARACTER(len=80) :: filename, partname

INTEGER :: i, j, k, &        ! loop counters
           nx2, ny2, nz2  ! array dimensions

REAL :: scaler !, & ! Physical length of each cell in z-dir
!        lscale    ! lscale set in physics.data, entered manually here...

REAL, DIMENSION(3), INTENT(IN) :: gridunits2     ! mx,my,mz from bear.data
REAL, ALLOCATABLE, DIMENSION(:, :, :) :: emiss2  ! calculated emissivities
REAL, ALLOCATABLE, DIMENSION(:, :) :: emissSum   ! intensity:  emission summed along line of sight

ALLOCATE( emissSum(nx2,ny2) )


!lscale = 1.0e17
scaler = gridunits2(3)*lscale/REAL(nz2)
PRINT*,scaler, nx2, ny2, nz2, MINVAL(emiss2), MAXVAL(emiss2),SIZE(emiss2)

DO i=1,nx2
DO j=1,ny2
DO k=1,nz2
   IF(emiss2(i,j,k)/=emiss2(i,j,k)) THEN
!      PRINT*,'NaN at ',i,j,k
   END IF
END DO
END DO
END DO


DO i = 1, ny2
  DO j = 1, nx2
    emissSum(j,i) = SUM(emiss2(j,i,:))
  END DO
END DO
!PRINT*,MINVAL(emissSum),MAXVAL(emissSum)

! multiply by cell length to go from cm^-3 to cm^-2
emissSum = emissSum*scaler 


!   Write the data array to a file in case you want to add or divide emission lines
filename = 'emiss_'//TRIM(partname)//'.dat'
OPEN(20, file=filename, STATUS='REPLACE')
DO i=1, nx2
DO j=1, ny2
   IF(emissSum(i,j)/=emissSum(i,j)) THEN
!      PRINT*,'NaN at ',i,j
   END IF
   WRITE(20,'(E22.12)') emissSum(i,j) 
END DO
END DO
CLOSE(20)


!   Write the data to a FITS file
!filename = 'emiss_'//TRIM(partname)//'.fits'
!CALL writfits_r4(filename,emissSum(:,:),nx2,ny2)



END SUBROUTINE sum_emiss




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


SUBROUTINE A_Values(a)

! This subroutine just returns a matrix with the A-values for the five lines of interest.
! The A-values are yanked from Dr. P.Hartigan's code fiveln.f, including sources.

! a(final level, initial level, species)

IMPLICIT NONE

REAL, DIMENSION(5,5,5) :: a   ! Einstein A transition values, in s^-1

! Einstein A coefficients for OI:
! A-values for OI: Baluja, K., and Zeippen, C. 1988, J. Phys B. Atom. Mol. Opt. Phys. 21, 1455.

a(4,5,1)=1.215
a(3,5,1)=0.0
a(2,5,1)=7.60e-2
a(1,5,1)=2.73e-4
a(5,5,1)=a(4,5,1)+a(3,5,1)+a(2,5,1)+a(1,5,1)
a(3,4,1)=8.93e-7
a(2,4,1)=1.82e-3
a(1,4,1)=5.63e-3
a(4,4,1)=a(3,4,1)+a(2,4,1)+a(1,4,1)
a(2,3,1)=1.74e-5
a(1,3,1)=1.20e-10
a(3,3,1)=a(2,3,1)+a(1,3,1)
a(1,2,1)=8.96e-5


! Einstein A coefficients for OII:
! A-values for OII: Zeippen, C. 1982, MNRAS 198, 111
!   (better than Zeippen, C. 1987, A&A 173, 410 according to
!   planetary nebula observations of Wang et al 2004 A&A 427, 873)

a(4,5,2)=2.08e-11
a(3,5,2)=1.02e-1
a(2,5,2)=6.15e-2
a(1,5,2)=2.32e-2
a(5,5,2)=a(4,5,2)+a(3,5,2)+a(2,5,2)+a(1,5,2)
a(3,4,2)=6.14e-2
a(2,4,2)=1.17e-1
a(1,4,2)=5.64e-2
a(4,4,2)=a(3,4,2)+a(2,4,2)+a(1,4,2)
a(2,3,2)=1.2e-7
a(1,3,2)=1.65e-4
a(3,3,2)=a(2,3,2)+a(1,3,2)
a(1,2,2)=3.82e-5


! Einstein A coefficients for NI:
! A-values for NI: Butler, K., and Zeippen, C. 1984, A&A 141, 274

a(4,5,3)=0.0
a(3,5,3)=2.523e-2
a(2,5,3)=5.589e-2
a(1,5,3)=6.61e-3
a(5,5,3)=a(4,5,3)+a(3,5,3)+a(2,5,3)+a(1,5,3)
a(3,4,3)=4.80e-2
a(2,4,3)=3.14e-2
a(1,4,3)=2.72e-3
a(4,4,3)=a(3,4,3)+a(2,4,3)+a(1,4,3)
a(2,3,3)=1.24e-8
a(1,3,3)=2.28e-5
a(3,3,3)=a(2,3,3)+a(1,3,3)
a(1,2,3)=6.13e-6


! Einstein A coefficients for NII:
! A-values for NII:
!   Bell, Hibbert, Stafford 1995, Phys. Scr. 52, 240
!   Storey and Zeippen 2000, MNRAS 312, 813.
!   Froese Fischer, C. and Saha, H. 1985, Phys Scr. 32, 181.

a(4,5,4)=0.923
a(3,5,4)=1.435e-4
a(2,5,4)=4.588e-2
a(1,5,4)=0.0
a(5,5,4)=a(4,5,4)+a(3,5,4)+a(2,5,4)+a(1,5,4)
a(3,4,4)=3.015e-3
a(2,4,4)=9.819e-4
a(1,4,4)=1.928e-6
a(4,4,4)=a(3,4,4)+a(2,4,4)+a(1,4,4)
a(2,3,4)=7.46e-6 
a(1,3,4)=1.062e-12
a(3,3,4)=a(2,3,4)+a(1,3,4)
a(1,2,4)=2.08e-6


! Einstein A coefficients for SII:
! A-values for SII, Mendoza & Zeippen 1982 MNRAS 198, 111.

a(4,5,5)=1.03e-6
a(3,5,5)=0.179
a(2,5,5)=0.133
a(1,5,5)=0.225
a(5,5,5)=a(4,5,5)+a(3,5,5)+a(2,5,5)+a(1,5,5)
a(3,4,5)=7.79e-2
a(2,4,5)=1.63e-1
a(1,4,5)=9.06e-2
a(4,4,5)=a(3,4,5)+a(2,4,5)+a(1,4,5)
a(2,3,5)=3.35e-7
a(1,3,5)=2.60e-4
a(3,3,5)=a(2,3,5)+a(1,3,5)
a(1,2,5)=8.82e-4

a(2:5,2,:) = 0.0
a(4:5,3,:) = 0.0
a(5,4,:) = 0.0


! Value for transitions out of level 1 = 0.0
a(:,1,:) = 0.0


END SUBROUTINE A_Values




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!




SUBROUTINE E_Values(erg)

! This subroutine generates a table with the energies for OI, OII, NI, NII, & SII
! These values were yanked from Dr. Hartigan's code, fiveln.f


IMPLICIT NONE

REAL, DIMENSION(5,5) :: e      ! energy of a given level in eV
REAL, DIMENSION(5,5,5) :: erg  ! erg is energy of transition converted from eV to ergs
INTEGER :: i,j,k               ! counters for loops


! Energy levels of OI (eV)
e(1,1)=0.0
e(1,2)=0.0196
e(1,3)=0.0282
e(1,4)=1.9689
e(1,5)=4.1931

! Energy levels of OII (eV)
e(2,1)=0.0
e(2,2)=3.3267
e(2,3)=3.3292
e(2,4)=5.0214
e(2,5)=5.0216

! Energy levels of NI (eV)
e(3,1)=0.0
e(3,2)=2.3854
e(3,3)=2.3865
e(3,4)=3.57840
e(3,5)=3.57845

! Energy levels of NII (eV)
e(4,1)=0.0
e(4,2)=0.00604
e(4,3)=0.0162
e(4,4)=1.900
e(4,5)=4.056

! Energy levels of SII (eV)
e(5,1)=0.0
e(5,2)=1.843  
e(5,3)=1.847 
e(5,4)=3.043
e(5,5)=3.049



DO i = 1,5 ! species
	DO j = 2,5  ! initial
		DO k = 1,5 ! final
		  IF (k < j) THEN
		    erg(k,j,i) = (e(i,j) - e(i,k))*(1.602E-12)
		  ELSE
		    erg(k,j,i) = 0.0
		  END IF
		END DO
	END DO
END DO

! where it starts in level 1, it can't transition down, so E=0
erg(:,1,:) = 0.0



END SUBROUTINE E_Values




!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



SUBROUTINE linenames(species, exlevels, lambda)

! This subroutine assigns the proper wavelength to the chosen line transition, for
! use in file names.


IMPLICIT NONE

CHARACTER(len=5) :: lambda
INTEGER :: species
INTEGER, DIMENSION(2) :: exlevels

IF (species == 1) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '5577'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '2980'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '2974'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2960'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '6392'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '6363'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6300'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '144E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '439E3'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '633E3'
    END IF
  END IF

ELSEIF (species == 2) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '620E5'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '7333'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '7322'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2471'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '7332'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '7321'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '2471'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '496E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3727'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '3730'
    END IF
  END IF

ELSEIF (species == 3) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = 'dunno'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '10401'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10410'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3467'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '10401'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10410'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3468'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '113E5'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '5202'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '5199'
    END IF
  END IF

ELSEIF (species == 4) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '5756'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '3069'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '3064'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '3057'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '6585'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '6550'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6525'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '122E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '765E3'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '205E4'
    END IF
  END IF

ELSEIF (species == 5) THEN
  IF (exlevels(1) == 5) THEN
    IF (exlevels(2) == 4) THEN
	lambda = '207E4'
    ELSEIF (exlevels(2) == 3) THEN
	lambda = '10323'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10290'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '4070'
    END IF
  ELSEIF (exlevels(1) == 4) THEN
    IF (exlevels(2) == 3) THEN
	lambda = '10373'
    ELSEIF (exlevels(2) == 2) THEN
	lambda = '10339'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '4078'
    END IF
  ELSEIF (exlevels(1) == 3) THEN
    IF (exlevels(2) == 2) THEN
	lambda = '310E4'
    ELSEIF (exlevels(2) == 1) THEN
	lambda = '6718'
    END IF
  ELSEIF (exlevels(1) == 2) THEN
    IF (exlevels(2) == 1) THEN
	lambda = '6733'
    END IF
  END IF

END IF

lambda = TRIM(lambda)

END SUBROUTINE linenames



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!





