!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    i_evolve.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!============================================================================
!                  BEARCLAW Astrophysical Applications
!============================================================================
! Andy Cunningham,Alexei Poludnenko, Adam Frank
! Department of Physics and Astronomy
! University of Rochester
! Rochester, NY
!----------------------------------------------------------------------------
! File:             i_evolve.f90
! Type:             module
! Purpose:          Modified source term integration from Alexei's code
!
!                   called from application as "evolve(qin,dtHydro)"
!                   will evolove the state vector qin by time dtHydro
!                   under influence of source terms.
!                   qin(1:NrVars) -- current state vector (computational units)
!                   dtHydro -- time step (comutational units)
!
! Revision History: Ver. 1.0 July 2003 A. Cunningham
! -----------------------------------------------------------------------

  SUBROUTINE Cool_Derivatives(q,dqdt,ci)
    ! Interface declaration
    TYPE(CELLINFO), INTENT(IN) :: ci
    REAL (KIND=qPrec), DIMENSION(1:NrVars),INTENT(IN) :: q
    REAL (KIND=qPrec), DIMENSION(1:NrVars), INTENT(OUT):: dqdt
    ! Internal variables
    REAL (KIND=qPrec) :: hion,ne,mu,nneuclei,nHneuc,H2diss,hrec,heion,herec,dm,Temp,OIcool, &
    dustcool,H2cool,H2heat,dustheat,H2term,qHeat,cooling,eth,coolfrac,gamma,CDH2,CD,dml
    REAL(KIND=qprec) :: minT=10**min_TAB, maxT=10**max_TAB
    REAL (KIND=qPrec), DIMENSION(0:nSpeciesHI) :: nvec

    dqdt = 0
    nvec = 0

    IF(.NOT. lchemion) RETURN
    CALL EOS_vars(q,nvec=nvec(iHaux:nSpeciesHI),T=Temp,mu=mu,gamma=gamma,eth=eth)
    Temp = MIN(MAX(Temp,minT),maxT)
    nvec = MAX(nvec,zero)

    call nparticle(nvec(iHaux:nSpeciesHI),nneuc=nneuclei,ne=ne)
    ! total number of neuclei NOT in molecular form (number of ions)
    nneuclei = nneuclei-2.d0*nvec(iH2)
    nHneuc = nvec(iH)+nvec(iHII)+nvec(iH2)

    H2diss=0;hion = 0; hrec = 0; heion = 0; herec = 0; dm = 0; OIcool=0; dustcool=0; H2cool=0; H2term=0

    IF(lH2) THEN
       IF(lcooling) THEN
          OIcool=OI_cool(nvec(iH2),nvec(iH),Temp)
          CDH2=CriticalDenH2(nvec(iH),Temp)
          ! function used to smothly conect high HI and low HI density cooling rates
          CD=one/(one+CDH2)
             IF(CD<0.01) THEN
                H2cool=H2_critical_cool_table(nvec(iH2),Temp)*(one-CD)
             ELSE IF(CD<0.99) THEN
                H2cool=H2_cool(nvec(iH2),nvec(iH),Temp)*CD+H2_critical_cool_table(nvec(iH2),Temp)*(one-CD)
             ELSE
                H2cool=H2_cool(nvec(iH2),nvec(iH),Temp)
             END IF
       END IF
       !H2diss = H2_diss_table(nvec(iH2), nvec(iH), nvec(iHe), ne, Temp)
       H2diss = H2_diss(nvec(iH2), nvec(iH), nvec(iHe), ne, Temp)
       !H2term=-H2diss + H2_dust_Recomb(nvec(iH)+nvec(iHII),nvec(iH),Temp,Tdust)
       H2term=-H2diss + H2_dust_Recomb_table(nvec(iH)+nvec(iHII),nvec(iH),Temp)
    END IF

    ! only turn on this physics if BBC is off or BCC is on and T < 1e4
    IF(Temp <= 1.1d4 .OR. .NOT. LBBC) THEN
       IF(lHII) THEN
          !hion = H_ioniz(nvec(iH),ne,Temp)
          !hrec = H_recomb(nvec(iHII),ne,Temp)
          hion = H_ioniz_table(nvec(iH),ne,Temp)
          hrec = H_recomb_table(nvec(iHII),ne,Temp)
       END IF
       IF(nvec(iHe)+nvec(iHeII)+nvec(iHeIII) > small .AND. lHeII) THEN
          !heion = He_ioniz(nvec(iHe),ne,Temp)
          !herec = He_recomb(nvec(iHeII),ne,Temp)
          heion = He_ioniz_table(nvec(iHe),ne,Temp)
          herec = He_recomb_table(nvec(iHeII),ne,Temp)
       END IF
       IF(lcooling) dm = DMCoolingRate(Temp)
       ! neglect dust cooling for now, un-comment to turn on
       ! dustcool=dust_cool_table(nHneuc-nvec(iH2),Temp) 
    ENDIF
    IF(Temp > Tfloor) THEN
       dml = one
       cooling = &
         (OIcool + &                  ! OI excitation cooling
         H2cool + &                         ! H2 vibrational rotational cooing
         !dustcool + &                               ! dust cooling due to H sticking
         dml*ne*nneuclei*dm + &  ! interstellar atomic cooling function of delgarno and mccray
         BindH2* H2diss + &           ! Energy taken from gas to dissociate H2's
         IonH*hion + &                ! Energy taken from gas to ionize the hydrogens
         IonHe*heion &                ! Energy taken from gas to ionize the heliums
         )
    ELSE
       cooling = zero
    END IF

    ! Provide the inhomogenious part of the conservation laws to be solved
    IF(lH2)    dqdt(iH2)    = H2term*muH2
    IF(lHII)   dqdt(iHII)   = (hion-hrec)*muH
    IF(lH)     dqdt(iH)     = (-(hion-hrec) - 2.d0*H2term)*muH
    IF(lHe)    dqdt(iHe)    = (herec-heion)*muHe
    IF(lHeII)  dqdt(iHeII)  = -dqdt(iHe)
    IF(lHeIII) dqdt(iHeIII) = 0.d0

    IF(lcooling) dqdt(iE) = -(cooling)/pscale ! do energy scaling

    ! do density scaling
    dqdt(nSpeciesLO:nSpeciesHI) = dqdt(nSpeciesLO:nSpeciesHI)/nScale
    ! do time scaling
    dqdt = dqdt*runtimesc
  END SUBROUTINE Cool_Derivatives

  SUBROUTINE Cool_Jacobian(q,dfdt,dfdq,ci)
    ! Interface declaration
    TYPE(CELLINFO), INTENT(IN) :: ci
    REAL (KIND=qPrec), DIMENSION(1:NrVars),INTENT(IN)   :: q
    REAL (KIND=qPrec), DIMENSION(1:NrVars), INTENT(OUT)  :: dfdt
    REAL (KIND=qPrec), DIMENSION(1:NrVars,1:NrVars), INTENT(OUT):: dfdq
    ! Internal variables
    REAL(KIND=qprec), PARAMETER :: small=1.d5*EPSILON(1.d0)
    REAL (KIND=qPrec), DIMENSION(1:NrVars) :: fplus, fminus, deltaq, qplus, qminus, qdumm, qdump
    INTEGER :: j

    ! Provide the Cool_Jacobian matrix for the system of equations being solved
    dfdt = 0
    dfdq = 0
    IF(lchemion) THEN
       ! Calculate rest of jacobian terms numerically
       ! I have choosen to do these terms numerically, for now -- so that it is easy
       ! to modify the rates in the future.
       ! This adds many calls to Derivatives. Otherwise there would be many
       ! analytic expressions to evaluate anyway.  The derivatives are calculated using
       ! tables and should not be THAT expensive ... I think.
       !
       ! The analytic route is complicated by the fact that some of these terms depend
       ! on the molecular weights --- which is dependent on the fraction of each spicies,
       ! especially in regimes of strong ionization.
       !
       ! Temperature depends on local gamma -- which depends on species fractions
       !
       ! species density depends on total density
       ! 
       ! temperature depends on momentum
       !
       ! so .. there are really few zero terms in the jacobian.

       ! determine some good scales for difference formula

       deltaq = MINVAL(srcPrecision)*ci%qScale

       qplus = q+MAX(deltaq,small)
       qminus = q-MAX(deltaq,small)
       ! make sure temperature cannot be negative
       qminus(iE) = q(iE)-deltaq(iE)

       ! second order difference formula for an easy estimate of jacobian terms
       DO j=1,nrVars
          qdumm = q
          qdump = q
          qdumm(j) = qminus(j)
          qdump(j) = qplus(j)
          ! Species fractions must remain consistent with total density
          ! small dq added to density, same fraction is added to each species
          ! small dq added to a species, same total density increase 
          IF(j==1) THEN
             qdumm(nSpeciesLO:nSpeciesHI)=q(nSpeciesLO:nSpeciesHI)*qminus(j)/q(j)
             qdump(nSpeciesLO:nSpeciesHI)=q(nSpeciesLO:nSpeciesHI)*qplus(j)/q(j)
          END IF
          IF(j>=nSpeciesLO .AND. j<=nSpeciesHI) THEN
             qdumm(1)=q(1)+(qdumm(j)-q(j))
             qdump(1)=q(1)+(qdump(j)-q(j))
          END IF
          CALL Cool_Derivatives(qdumm,fminus,ci)
          CALL Cool_Derivatives(qdump,fplus,ci)
          dfdq(:,j) = (fplus - fminus)/(qdump(j)-qdumm(j))
       END DO
    END IF
  END SUBROUTINE Cool_Jacobian

  SUBROUTINE evolve(qin)
    REAL (KIND=qPrec), DIMENSION(1:NrVars) :: qin
    ! Parameters specific to the Kaps-Rentrop method
    INTEGER, PARAMETER           :: MaxSteps = 100, StepsTol = 5
    REAL (KIND=qPrec), PARAMETER :: dtMin = 1.0d-30, Small = 1.0d-30
    !
    INTEGER :: nSteps,m
    LOGICAL :: Success

    REAL (KIND=qPrec) :: ke
    REAL (KIND=qPrec) :: dtHydro,tEnd,tCurr,dtCurr,dtDone,dtNext
    REAL (KIND=qPrec), DIMENSION(1:NrVars) :: q,dqdt
    REAL (KIND=qPrec), TARGET, DIMENSION(1:NrVars) :: qScale
    REAL (KIND=qPrec),PARAMETER:: r=1! r is not important for this driver routine
    TYPE(CELLINFO) :: ci
    q = qin
    ! Fetter out thermal energy only
    ke = DOT_PRODUCT(qin(2:iSpeedHI),qin(2:iSpeedHI))
    ke = ke/(2.d0*qin(1))
    q(iE) = qin(iE) - ke
    q(2:ndim+1) = 0.
    qin(2:nDim+2) = zero


    tEnd      = dtHydro
    tCurr = 0.0; dtCurr = dtHydro
    Success = .FALSE.

    ! initial protection, m is a dummy variable
    m=SrcProtect(q,ci,ci%dt,0)
    !
    ! get a rough estimate for starting timestep 
    ! sending a HUGE initial step results in singular matrix in stiff
    CALL Cool_Derivatives(q,dqdt,ci)
    ! Perform scaling used to monitor accuracy
    qScale = ABS(q)
    qScale(iE) = MAX(qScale(iE)-half*DOT_PRODUCT(qScale(2:iSpeedhi),qScale(2:iSpeedhi))/qScale(1),&
         MinTemp*qScale(1)/EOSConstants)
    qScale(nSpeciesLO:nSpeciesHI) = qScale(1)
    qScale(2:iSpeedhi)=qScale(1)*SQRT((gamma-1.d0)*qScale(iE)/qScale(1))
    qScale=ABS(qScale)
    ci%qScale=>qScale
    !
    dtcurr=MINVAL(qscale/MAX(ABS(dqdt),small))
    !
    Timestepping: DO nSteps=1,MaxSteps
       ci%iteration=nSteps
       IF(nSteps>1) THEN
          ! Perform scaling used to monitor accuracy
          qScale = ABS(q)
          qScale(iE) = MAX(qScale(iE)-half*DOT_PRODUCT(qScale(2:iSpeedhi),qScale(2:iSpeedhi))/qScale(1),&
               MinTemp*qScale(1)/EOSConstants)
          qScale(nSpeciesLO:nSpeciesHI) = qScale(1)
          qScale(2:iSpeedhi)=qScale(1)*SQRT((gamma-1.d0)*qScale(iE)/qScale(1))
          qScale=ABS(qScale)
          ci%qScale=>qScale
          CALL Cool_Derivatives(q,dqdt,ci)
       END IF
       ! If stepsize can overshoot, decrease the stepsize
       IF ((tCurr+dtCurr-tEnd) > 0.d0) dtCurr = tEnd-tCurr
       CALL Stiff(q,dqdt,qScale,MINVAL(srcPrecision),NrVars,3,tCurr,  &
            dtCurr,dtDone,dtNext,ci)
       ! Check if we are done
       IF ((tCurr-tEnd) >= 0.d0) THEN
          Success = .TRUE.
          Exit Timestepping
       END IF
       IF (dtNext < dtMin) THEN
          PRINT *,'!!! src ERROR: Stepsize became smaller than the ',&
               'allowed value'
          STOP
       END IF
       dtCurr = dtNext
    END DO Timestepping
    qin = q
    qin(iE) = q(iE) + ke
  END SUBROUTINE evolve

    ! ------------------
    ! Auxiliary routines
    ! ------------------
    ! returns weather or not to subcycle
    ! 1 = no protection, everything OK
    ! 0 = hard protection, timestep already too small for subcycle
    ! 2 = subcycle
    FUNCTION SrcProtect(q,ci,dt,i)
      INTEGER :: SrcProtect
      TYPE(CELLINFO) :: ci
      REAL (KIND=qPrec), DIMENSION(:), INTENT(INOUT) :: q
      REAL (KIND=qPrec), INTENT(INOUT) :: dt
      INTEGER :: i
      REAL (KIND=qPrec) :: Pressure,Temp,ke,eth,emicro,gamm,mu
      REAL (KIND=qPrec), PARAMETER :: Small = EPSILON(1.d0),Large=1.d40,dtmin=1.d-4,consistprec=1.d-2

      SrcProtect=1

      ! if q is not a number, or +/- Infinity then somthing awful happened, reduce the step size by a lot
!!$      IF(ANY(ISNAN(q)) .OR. ANY(ABS(q)>LARGE)) THEN
!!$         IF(i==0) THEN
!!$            PRINT*,'--- src: NAN on intitial step, stopping'
!!$            STOP
!!$         END IF
!!$         IF (Verbosity(2) == 2) THEN
!!$            WRITE(1,*) '--- src: Performing NAN protection #',i
!!$         END IF
!!$         dt = .1d0*dt
!!$         SrcProtect=2
!!$         RETURN
!!$      END IF
      RETURN
      IF(iCooling /= 2) THEN
         Pressure = Press(q)
         IF ((Pressure <= zero) .OR. (q(1) <= zero)) THEN
            IF (Verbosity(2) == 2) THEN
               WRITE(1,*) '--- src: Performing pressure protection #',i
            END IF
            dt = 0.25*dt
            SrcProtect=2
            RETURN
         END IF
      ELSE ! iCooling == 2
         ! species total consistent with total den and den protection
         IF((lprotectneq .AND. i>0 .AND.&
              (SUM(q(nSpeciesLO:nSpeciesHI))>(one+consistprec)*q(1) .OR. SUM(q(nSpeciesLO:nSpeciesHI))<(one-consistprec)*q(1))) &
              .OR. q(1)<=small)THEN
            dt = 0.25*dt
            SrcProtect=2
            IF (Verbosity(2) == 2) THEN
               WRITE(1,*) '--- src: Performing species consistency protection #',i
            END IF
            RETURN
         END IF
         ! renormalize tracers to total density
         IF(nSpeciesLO/=nSpeciesHI) THEN
            q(nSpeciesLO:nSpeciesHI) = &
                 q(nSpeciesLO:nSpeciesHI) * q(1) / SUM(q(nSpeciesLO:nSpeciesHI))
         END IF
         ! temperature protection
         CALL EOS_vars(q,T=Temp,ke=ke,mu=mu,gamma=gamm)
         IF(Temp <= zero) THEN
            IF(dt>ci%dt*dtmin .AND. i>0) THEN
               dt = 0.25*dt
               SrcProtect=2
               IF (Verbosity(2) == 2) THEN
                  WRITE(1,*) '--- src: Performing pressure protection #',i
               END IF
               RETURN
            ELSE ! can't make timestep any smaller, use mintemp
               q(iE) = ke + MinTemp*q(1)/(EOSConstants*mu)
               IF (Verbosity(2) == 2) THEN
                  WRITE(1,*) '--- src: Performing hard pressure protection #',i
               END IF
               SrcProtect=0
               RETURN
            END IF
         ! maintain temperature to at least a few K
         ELSE IF(Temp<half*MinTemp) THEN
            q(iE) = ke + half*MinTemp*q(1)/(EOSConstants*mu)
         END IF
      END IF
    CONTAINS
    ! returns gamma 
    ! if present, gammais returns with value of isentropic gamma
    FUNCTION ratio_heat(q,gammais)
      !USE TF
      REAL(KIND=qprec), DIMENSION(:), INTENT(IN) :: q
      REAL(KIND=qprec) :: gamma_neq,ratio_heat,gammaisin
      REAL(KIND=qprec), OPTIONAL, INTENT(OUT) :: gammais
      REAL(KIND=qprec) :: dgamma_neq,dgammaisin

      IF(icooling == 2) THEN
         IF(PRESENT(gammais)) THEN
            CALL EOS_vars(REAL(q,qprec),gamma=dgamma_neq,gammais=dgammaisin)
            gammais=dgammaisin
         ELSE
            CALL EOS_vars(REAL(q,qprec),gamma=dgamma_neq)
         END IF
         gamma_neq=dgamma_neq
         ratio_heat = gamma_neq
      ELSE
         SELECT CASE(iEOS)
         CASE(0)
            ratio_heat = gamma
            gammaisin = gamma
         CASE(1)
            gammaisin = gamma
            ratio_heat = gammac
         CASE DEFAULT
            !CALL TFEOS_SC(q(:),ratio_heat,gammaisin)
            PRINT*,'EOS option not implemented in source terms'
            STOP
         END SELECT
      END IF
      IF(PRESENT(gammais)) gammais=gammaisin
    END FUNCTION ratio_heat

    ! compute pressure, depending on what microphysics we have going on
    ! chi =  derivative of pressure wrt conserved variables, if present 
    !        -- sometimes needed geometric jacobian terms
    ! retuns magnetic+themal pressure in pseudo-MHD case
    FUNCTION Press(q,chi,gamma,gammais)
      REAL(KIND=qprec), DIMENSION(:), INTENT(IN) :: q
      REAL(KIND=qprec), DIMENSION(:), OPTIONAL, INTENT(OUT) :: chi
      REAL(KIND=qprec), OPTIONAL,INTENT(OUT) :: gamma,gammais
      REAL(KIND=qprec) :: Press
      ! Internal variables
      REAL (KIND=qPrec) :: gamma1, ke, invq1

      IF(PRESENT(gammais)) gammais = ratio_heat(q,gammais=gammais)
      gamma1 = ratio_heat(q)-1.
      IF(PRESENT(gamma)) gamma=gamma1+1.

      invq1 = 1./q(1)
      ke = 0.5*DOT_PRODUCT(q(2:iSpeedHI),q(2:iSpeedHI))*invq1
      Press = gamma1*(q(iE) - ke)

      IF(PRESENT(chi)) THEN
         chi = 0.
         chi(1)        = gamma1*ke*invq1
         chi(2:iSpeedHI) = -gamma1*q(2:iSpeedHI)/q(1)
         chi(iE)       = gamma1
      END IF
    END FUNCTION Press
    END FUNCTION SrcProtect

  ! ------------------
  ! Auxiliary routines
  ! ------------------
  SUBROUTINE Stiff(q,dqdt,qScale,Precision,nrVars,nDim,t,dtTry,dtDone,dtNext,ci)
    ! Fourth-order Rosenbrock step for integrating stiff o.d.e.'s, with
    ! monitoring of local truncation error to adjust stepsize. Input are 
    ! the state vector q(:) and its derivative dqdt(:) at the starting
    ! value of t. Also input are the stepsize to be attempted
    ! dtTry, the required accuracy Precision, and the vector qScale(:) against
    ! which the error is scaled. On output, q and t are replaced by their new 
    ! values, dtDone is the stepsize that was actually accomplished, and dtNext 
    ! is the estimated next stepsize. Derivatives is the user-supplied
    ! subroutine that computes the inhomogeneous part of the conservation
    ! equations, while Cool_Jacobian is a user-supplied subroutine that computes
    ! the Jacobi matrix of Derivatives of the right-hand side with respect to
    ! the components of q.
    ! Parameters: Increm and Decrem are the largest and smallest factors by which
    ! stepsize can change in one step; ErrCon=(Increm/Safety)**(1/ExpIncrem) and
    ! handles the case when MaxError about = 0.

    ! Interface declarations
    TYPE(CELLINFO), INTENT(IN) :: ci
    INTEGER, INTENT(IN) :: NrVars,nDim
    REAL (KIND=qPrec), INTENT(INOUT) :: t
    REAL (KIND=qPrec), INTENT(IN)    :: dtTry,Precision
    REAL (KIND=qPrec), INTENT(OUT)   :: dtDone,dtNext
    REAL (KIND=qPrec), DIMENSION(:), INTENT(INOUT) :: q
    REAL (KIND=qPrec), DIMENSION(:), INTENT(IN)    :: dqdt,qScale
    ! Parameter declarations
    INTEGER, PARAMETER           :: MaxAttempts = 5000
    REAL (KIND=qPrec), PARAMETER :: Safety = 0.9, Increm = 1.5, Decrem = 0.5, &
         ExpIncrem = -0.25, ExpDecrem = -1.0/3.0,  &
         ErrCon = 0.1296
    REAL (KIND=qPrec), PARAMETER :: Gam = 0.5, A21 = 2.0, A31 = 48.0/25.0,    &
         A32 = 6.0/25.0,     C21 = -8.0,           &
         C31 = 372.0/25.0,   C32 = 12.0/5.0,       &
         C41 = -112.0/125.0, C42 = -54.0/125.0,    &
         C43 = -2.0/5.0, B1 = 19.0/9.0, B2 = 0.5,  &
         B3 = 25.0/108.0,    B4 = 125.0/108.0,     &
         E1 = 17.0/54.0,     E2 = 7.0/36.0,        &
         E3 = 0.0, E4 = 125.0/108.0, C1X = 0.5,    &
         C2X = -3.0/2.0,     C3X = 121.0/50.0,     &
         C4X = 29.0/250.0, A2X = 1.0, A3X = 3.0/5.0
    ! Internal declarations
    INTEGER i,jtry
    INTEGER, DIMENSION(3) :: ipro
    REAL (KIND=qPrec) :: MaxError,dt,tSaved,diag
    INTEGER, DIMENSION(NrVars) :: PermIndex
    REAL (KIND=qPrec), DIMENSION(NrVars):: dfdt,dqTemp,qSaved,Error,g1,g2,g3,g4
    REAL (KIND=qPrec), DIMENSION(NrVars,NrVars) :: A,dfdq
    !
    tSaved    = t
    qSaved(:) = q(:)
    ! Obtain the Jacobian matrix dfdq and vector dfdt
    dt = dtTry
    Iterations: DO jtry=1,MaxAttempts
       ! Set up the matrix 1-Gam*dt*f'
       A(:,:) = -dfdq(:,:)
       diag = 1/(Gam*dt)
       FORALL(i=1:NrVars) A(i,i) = A(i,i) + diag
       ! LU-decompose the matrix
       CALL LUdecompose(A,NrVars,PermIndex)
       !
       g1 = dqdt + dt*C1X*dfdt                    ! Set up right-hand side for g1
       CALL LUBackSubst(A,NrVars,PermIndex,g1)     ! Solve for g1 via LU back-substitution
       q  = qSaved + A21*g1                       ! Compute intermetiate values
       t  = tSaved + A2X*dt                       ! of q and t

       ! first protection check
       ipro(1)=SrcProtect(q,ci,dt,1)
       IF(ipro(1)==2) CYCLE Iterations
       
       ! Compute dqdt at the intermediate values
       CALL Cool_Derivatives(q,dqTemp,ci)
       ! Repeat for g2
       g2 = dqTemp + dt*C2X*dfdt + C21*g1/dt
       CALL LUBackSubst(A,NrVars,PermIndex,g2)
       q  = qSaved + A31*g1 + A32*g2
       t  = tSaved + A3X*dt

       ! second protection check
       ipro(2)=SrcProtect(q,ci,dt,2)
       IF(ipro(2)==2) CYCLE Iterations

       CALL Cool_Derivatives(q,dqTemp,ci)
       ! Find g3
       g3 = dqTemp + dt*C3X*dfdt + (C31*g1 + C32*g2)/dt
       CALL LUBackSubst(A,NrVars,PermIndex,g3)
       ! Find g4
       g4 = dqTemp + dt*C4X*dfdt + (C41*g1 + C42*g2 + C43*g3)/dt
       CALL LUBackSubst(A,NrVars,PermIndex,g4)
       ! Get 4th-order estimate of q and error estimate
       q     = qSaved + B1*g1 + B2*g2 + B3*g3 + B4*g4

       ! Perform the final pressure and density check
       ipro(3)=SrcProtect(q,ci,dt,3)
       IF(ipro(3)==2) CYCLE Iterations

       ! q is fine, estimate the error
       Error = E1*g1 + E2*g2 + E3*g3 + E4*g4
       t = tSaved + dt
       IF (t == tSaved) THEN
          PRINT *,'!!! src ERROR: Stepsize not significant'
          STOP
       END IF
       ! Evaluate accuracy
       MaxError = MAXVAL(ABS(Error/MAX(qScale*Precision,SMALL)))
       IF (MaxError <= 1.d0) THEN
          ! Step succeeded. Compute the next time step and return
          dtDone = dt
          dtNext = MERGE(Safety*dt*MaxError**ExpIncrem, Increm*dt,            &
               MaxError >= ErrCon)
          RETURN
       ELSE
          dtNext = Safety*dt*MaxError**ExpDecrem
          dt = MAX(dtNext,Decrem*dt)
       END IF
    END DO Iterations ! Go back and retry step
    ! Should not get here ever
    PRINT *,'!!! src ERROR: Desired accuracy not reached in the allowed ',    &
         'number of iterations'
    STOP
  END SUBROUTINE Stiff

  SUBROUTINE LUDecompose(A,NrVars,PermIndex)
    ! The subroutine implements the LU decomposition of the matrix A using the
    ! outer product Gauss elimination by Golub, G.H., and Van Loan, C.F. 1989, 
    ! Matrix Computations, 2nd ed. (Baltimore: Johns Hopkins University Press).
    ! Given a matrix A(1:NrVars,1:NrVars), replace it by the LU decomposition of
    ! a rowwise permutation of itself.
    ! A and NrVars are input. A is output; PermIndex(1:NrVars) is an output
    ! vector that records the row permutation effected by the partial pivoting
    !
    ! Interface declarations
    INTEGER, INTENT(IN)            :: NrVars
    INTEGER, DIMENSION(NrVars), INTENT(OUT)           :: PermIndex
    REAL (KIND=qPrec), DIMENSION(:,:), INTENT(INOUT) :: A
    ! Internal declarations
    INTEGER iMax,j
    REAL (KIND=qPrec), PARAMETER :: Small = 1.0d-20
    REAL (KIND=qPrec), DIMENSION(NrVars) :: RowScaling,dummy
    !
    ! Loop over rows to get the implicit scaling information
    RowScaling = MAXVAL(ABS(A),dim=2)
    IF (ANY(RowScaling == 0.d0)) THEN
       PRINT *,'!!! src ERROR: Singular matrix in LUDecompose'
       STOP
    END IF
    RowScaling = 1.0/RowScaling   ! Save the scaling
    DO j=1,NrVars
       ! Find the pivot row; SUM is needed just to turn an array produced by
       ! MAXLOC into a scalar
       iMax = (j-1) + SUM(MAXLOC(RowScaling(j:NrVars)*ABS(A(j:NrVars,j))))
       ! Do we need to interchange rows?
       IF (j /= iMax) THEN
          dummy = A(iMax,:)
          A(iMax,:) = A(j,:)
          A(j,:)    = dummy
          RowScaling(iMax) = RowScaling(j)   ! Also interchange the scale factor
       END IF
       PermIndex(j) = iMax
       ! If the pivot element is 0 the matrix is singular (at least to the
       ! precision of the algorithm). We will substitute Small for 0
       IF (A(j,j) == 0.d0) A(j,j) = Small
       ! Now, finally, divide by the pivot element
       A(j+1:NrVars,j) = A(j+1:NrVars,j)/A(j,j)
       ! Reduce the remaining submatrix
       A(j+1:NrVars,j+1:NrVars) = A(j+1:NrVars,j+1:NrVars) -                      &
            SPREAD(A(j+1:NrVars,j),dim=2,ncopies = NrVars-j) * &
            SPREAD(A(j,j+1:NrVars),dim=1,ncopies = NrVars-j)
    END DO
  END SUBROUTINE LUDecompose

  SUBROUTINE LUBackSubst(A,NrVars,PermIndex,B)
    ! Solve the set of NrVars linear equations A*X = B. A is input, not as the
    ! original matrix A but rather as its LU decomposition, determined by the
    ! routine LUDecompose. B(1:NrVars) is input as the right-hand side vector B,
    ! and on output contains the solution vector X.
    !
    ! Interface declarations
    INTEGER, INTENT(IN):: NrVars
    INTEGER, DIMENSION(NrVars), INTENT(IN)              :: PermIndex
    REAL (KIND=qPrec), DIMENSION(:,:), INTENT(IN)      :: A
    REAL (KIND=qPrec), DIMENSION(NrVars), INTENT(INOUT) :: B
    ! Internal declarations
    INTEGER i,ii,ll
    REAL (KIND=qPrec) :: sum
    ! When ii is set to a positive value, it will become the index of the
    ! first nonvanishing element of B. First forward substitution is done,
    ! while unscrambling the permutations along the way.
    ii = 0
    DO i=1,NrVars
       ll  = PermIndex(i)
       sum = B(ll)
       B(ll) = B(i)
       IF (ii /= 0) THEN
          sum = sum - DOT_PRODUCT(A(i,ii:i-1),B(ii:i-1))
       ELSE IF (sum /= 0.d0) THEN
          ! A nonzero element was encountered, so from now on we will have to
          ! do the dot product above
          ii = i
       END IF
       B(i) = sum
    END DO
    ! Now perform the backsubstitution
    DO i=NrVars,1,-1
       B(i) = (B(i) - DOT_PRODUCT(A(i,i+1:NrVars),B(i+1:NrVars)))/A(i,i)
    END DO
  END SUBROUTINE LUBackSubst
