!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    i_processing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!----------------------------------------
!
! Subroutines related to the processing aspect of bear2fix:
!
!  * GetFrames
!  * GetScales
!  * GetPrompts
!  * GetUserfield
!
!  * ReadFrames
!  * ProcessFrames
!
!  * ProcessHydro(lWrite)
!  * ProcessMHD(lWrite)
!  * ProcessHydroLLE(lWrite)
!
!----------------------------------------

  ! prompt for default, return frame & frameend
SUBROUTINE GetFrames
  INTEGER :: iErr,iErr2,iErr3,iErr4,iErr5,iErr6
  IF (lInteractive) THEN
    WRITE(*,'(A)',ADVANCE='NO')'select data set (Enter -1 for all frames, -2 to specify a range or -3 to use defaults) '
    default=frame
    READ*,default
    frameinput=default
    frame=default
  ELSE
    default=frameinput
  END IF


  step=1
  SELECT CASE(frameinput)
  CASE(-1,-3,-4,-5,-6) ! want all frames
     n=0;frame=0
     iErr=0
     DO WHILE (iErr==0 .OR. iErr2==0 .OR. iErr3==0 .OR. iErr4==0 .OR. iErr5==0 .OR. iErr6==0)
        WRITE(TreeFile,'(A10,I5.5,A4)')'./out/tree',n,'.dat'
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr)
        CLOSE(UNIT=77)
        WRITE(TreeFile,'(A12,I5.5,A4)')'./out/chombo',n,'.hdf'
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr2)
        CLOSE(UNIT=77)
        write(TreeFile,'(A10,I3.3A4)') "out/W_000_",  n , ".bov"
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr3)
        CLOSE(UNIT=77)
        write(TreeFile,'(A6,I3.3A4)') "out/W_" , n , ".bov"
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr4)
        CLOSE(UNIT=77)
        write(TreeFile,'(A10,I4.4A4)') "out/W_000_",  n , ".bov"
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr5)
        CLOSE(UNIT=77)
        write(TreeFile,'(A6,I4.4A4)') "out/W_" , n , ".bov"
        OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr6)
        CLOSE(UNIT=77)
        n=n+1
     END DO
     frameend=n-2
  CASE(-2) ! want range of frames
     IF (lInteractive) THEN
       PRINT*,'what frames? (set end frame=-1 to select all frames after begin frame)'
       WRITE(*,'(A)',ADVANCE='NO')' begin frame, end frame? '
       READ*,frame,frameend
     END IF
     
     IF(frameend==-1) THEN
        n=frame
        iErr=0
        DO WHILE (iErr==0 .OR. iErr2==0 .OR. iErr3==0 .OR. iErr4==0)
           WRITE(TreeFile,'(A10,I5.5,A4)')'./out/tree',n,'.dat'
           OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr)
           CLOSE(UNIT=77)
           WRITE(TreeFile,'(A12,I5.5,A4)')'./out/chombo',n,'.hdf'
           OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr2)
           CLOSE(UNIT=77)
           write(TreeFile,'(A10,I3.3A4)') "out/W_000_",  n , ".bov"
           OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr3)
           CLOSE(UNIT=77)
           write(TreeFile,'(A6,I3.3A4)') "out/W_" , n , ".bov"
           OPEN(UNIT=77,FILE=TreeFile,STATUS='OLD',IOSTAT=iErr4)
           CLOSE(UNIT=77)
           n=n+1
        END DO
        
        frameend=n-2
        IF(frameend < frame) THEN
           PRINT*,' !!!Error, begin frame greater than # frames in directory.'
           STOP
        ENDIF
        PRINT '(A24,i4.1,A4,i4.1)','    will process frames ',frame,' to ',frameend
     ENDIF
  CASE DEFAULT ! want specific frame, do nothing
     frameend=frame
  END SELECT

  IF(frameend<0) THEN
     PRINT*,'No output data found.'
     STOP
  END IF
END SUBROUTINE GetFrames


  SUBROUTINE GetScales
    IF (lInteractive) THEN
     PRINT*,'What frame scaling?'
     PRINT*,'0=relative'
     PRINT*,'1=absolute'
     PRINT*,'2=zeroed'
     WRITE(*,'(A)',ADVANCE='NO') ' 3=user specified ' 
     READ*,scales
     
     SELECT CASE(scales)
     CASE(0,1,2)
     CASE(3)
        WRITE(*,'(A)',ADVANCE='NO') ' Enter: minvalue,maxvalue ' 
        READ*, MinOverall,MaxOverall
     CASE DEFAULT
        PRINT*,'Error: invalid selection of scales'
        STOP
     END SELECT
   END IF
  END SUBROUTINE GetScales


  SUBROUTINE GetPrompts
    IF (lInteractive) THEN
      WRITE(*,'(A)',ADVANCE='NO')'select MaxLevel to input (-1) for all '
      READ*,MaxLevel

      PRINT*,'What application?'
      PRINT*,'1=hydro astro'
      PRINT*,'2=mhd astro'
      WRITE(*,'(A)',ADVANCE='NO') ' 3=hydro lab (LLE) '
      READ*, application

      PRINT*,'What operation?'
      PRINT*,'-1=parse process.data file'
      PRINT*,'0=write fixed grid chombo'
      PRINT*,'1=write fixed grid hdf'
      PRINT*,'2=write fixed grid fits'
      PRINT*,'3=plot log10 density (2D only) '
      PRINT*,'4=plot cut '
      PRINT*,'5=plot other field '
      PRINT*,'6=Schlieren '
      PRINT*,'7=user coded '
      PRINT*,'8=JetSet contest '
      PRINT*,'9=create fixed grid init files '
      PRINT*,'10=compare chombo files in out and ref ' 
      PRINT*,'11=Check Results of Test Problems ' 
      PRINT*,'12=create fixed grid BOV emission files '
      PRINT*,'13=produce ASCII file to read into Shape for emission maps '
      WRITE(*,'(A)',ADVANCE='NO') ' 14=produce bov files of magnetic potential for use in visit (2D only)'
      READ*, operation
      IF (ANY(operation == (/3,4,5,6/))) THEN
         IF (application == 2) THEN
            WRITE(*,'(A)', ADVANCE='NO') 'Stream lines? 0=>no, 1=>yes '
            READ*, iStream
         END IF         
         WRITE(*,'(A)', ADVANCE='NO') 'duplex? 0=>no, 1=>yes '
         READ*, duplex
         IF (duplex == 1) THEN
            WRITE(*,'(A)', ADVANCE='NO') 'symmetry axis? 0=>x, 1=>y, 2=>both'
            READ*, symaxis
         END IF
      END IF
   END IF

    SELECT CASE(operation)
    CASE(3) ! log10(n)
       llog=.true.
       userfield=0
       IF(application .eq. 3) userfield=1
       qnvars=1
       IF(iStream/=0) qnvars=NrVars
       IF (lInteractive) THEN
         WRITE(*,'(A)',ADVANCE='NO') ' max hotbox level (-1 for no hotboxes) ' 
         READ*, MaxHotBox
       END IF
    CASE(4) ! plot cut
       userfield=1 !this is only to speed up the createfixedgrid call below

       CALL CreateFixedGrid(frame)
       IF (lInteractive) THEN
         PRINT*,' cut position x1,y1,x2,y2?'
         WRITE(*,'(A13,I5.2,A1,I5.2,A2)',ADVANCE='NO') ' (grid extent',mxfix(1),',',mxfix(2),') ' 
         READ*, cutp1(1),cutp1(2),cutp2(1),cutp2(2)
       END IF
       CALL GetUserfield
       cutfield=userfield
       IF (lInteractive) THEN
         WRITE(*,'(A)',ADVANCE='NO') ' max hotbox level (-1 for no hotboxes) ' 
         READ*, MaxHotBox
         WRITE(*,'(A)',ADVANCE='NO') ' plot label ' 
         READ*, plotlabel
       END IF
       IF(cutfield<=3) THEN
          qnvars=1
       ELSE
          qnvars = NrVars
       ENDIF
    CASE(5) ! plot other field
       CALL GetUserfield
       IF(iStream/=0) qnvars=NrVars
       IF (lInteractive) THEN
         WRITE(*,'(A)',ADVANCE='NO') ' max hotbox level (-1 for no hotboxes) ' 
         READ*, MaxHotBox
       END IF
    CASE(6) ! schlieren
       qnvars=NrVars
       userfield=-1
    CASE(7)
        CALL PV(0.)
    CASE(8)
        CALL ProcessContestOptions
    CASE(9)
        IF (lInteractive) THEN
          WRITE(*,'(A)',ADVANCE='NO') ' number of processors to regrid to?' 
          READ*, np
        END IF
    CASE(-1:1, 10)
       qnvars=-1
    CASE(11)
       IF (lInteractive) CALL GetiTest()
       qnvars=-1
    CASE(12)
       qnvars=NrVars
       CALL GetEmissionData
       lEmissBOV=.TRUE.
    CASE(13)
       qnvars=NrVars
    CASE(14)
       qnvars=NrVars
    CASE DEFAULT ! outputting fixed grids
       IF (lInteractive) THEN
         WRITE(*,'(A)',ADVANCE='NO') ' Enter number of fields to process (-1) for all fields '
         READ*,qnvars
       END IF
    END SELECT
  END SUBROUTINE GetPrompts


  ! prompt for userfield if operation == 5
  SUBROUTINE GetUserfield
    IF (lInteractive) THEN
      WRITE(*,'(A)',ADVANCE='NO') ' What field? (-1 to see a list)  '
      READ*,userfield
      IF(userfield .eq. -1) THEN
         PRINT*,' Possible fields (as defined in bear2fixIO.f90):'
         PRINT*,'   0 - Number density [particles/cc]'
         PRINT*,'   1 - Density [g/cc]'
         PRINT*,'   2 - X velocity [cm/s]'
         PRINT*,'   3 - Y velocity [cm/s]'
         PRINT*,'   4 - Z velocity [cm/s] (if 3D)'
         PRINT*,'   5 - Total velocity [cm/s]'
         PRINT*,'   6 - Thermal energy [?]'
         PRINT*,'   7 - Pressure [?]'
         PRINT*,'   8 - Temperature [K]'
         PRINT*,'   9 - Sound Speed [cm/s]'
         PRINT*,'  10 - Emission [?]'
         PRINT*,'  12 - Total magnetic energy [?]'
         PRINT*,'  13 - Total energy [?]'
         PRINT*,'  14 - X B-field'
         PRINT*,'  15 - Y B-field'
         PRINT*,'  16 - Z B-field'
         PRINT*,'  17 - Vorticity []'
         PRINT*,'  18 - Plasma beta'
         PRINT*,'  19 - Tracer 1 (if present)'
         PRINT*,'  20 - Tracer 2 (if present)'
         WRITE(*,'(A)',ADVANCE='NO')'  21 - Tracer 3 (if present) '
         READ*,userfield
      END IF

      WRITE(*,'(A)',ADVANCE='NO') ' Log10? (T/F) '
      READ*,llog
    END IF

    qnvars = NrVars
  END SUBROUTINE GetUserfield


  SUBROUTINE GetEmissionData
    INTEGER i,emissionoutput
    WRITE(*,'(A)',ADVANCE='NO') ' Enter the number of transitions to process: '
    READ(*,'(I1)') numspecies

    ALLOCATE(species(numspecies)     &
            ,spec_char(numspecies) &
            ,lev1_char(numspecies) &
            ,lev2_char(numspecies) &
            ,exlevels(numspecies,2))

    DO i=1,numspecies
       ! Get the element & ionization state desired
       WRITE (*,'(A)',ADVANCE='NO') ' Enter the element desired (OI = 1, OII = 2, NI = 3, NII = 4, SII = 5, H_alpha = 6): '
       READ (*,'(I1)') species(i)
       !species(i) = 6
       IF (species(i) == 1) THEN
          spec_char(i) = 'O_I'
       ELSEIF (species(i) == 2) THEN
          spec_char(i) = 'OII'
       ELSEIF (species(i) == 3) THEN
          spec_char(i) = 'N_I'
       ELSEIF (species(i) == 4) THEN
          spec_char(i) = 'NII'
       ELSEIF (species(i) == 5) THEN
          spec_char(i) = 'SII'
       ELSEIF (species(i) == 6) THEN
          spec_char(i) = 'Hal'
       END IF

       ! Get the initial and final excitation levels
       WRITE (*,'(A)',ADVANCE='NO') ' Enter the initial and final excitation levels init, fin: '
       READ (*,*) exlevels(i,:)
       !exlevels = [4,3]
       IF (species(i) == 6) exlevels(i,:) = [3,2]
       lev1_char(i) = ACHAR(exlevels(i,1)+48)
       lev2_char(i) = ACHAR(exlevels(i,2)+48)
    END DO

    WRITE(*,'(A)',ADVANCE='NO') ' Output? 0:BOV, 1:Shape ASCII, 2:both '
    READ*,emissionOutput
    SELECT CASE(emissionOutput)
    CASE(0)
       lEmissBOV=.TRUE.
    CASE(1)
       lEmissShape=.TRUE.
    CASE(2)
       lEmissBOV=.TRUE.;lEmissShape=.TRUE.
    CASE DEFAULT
       PRINT*,'Error.'
       STOP
    END SELECT


  END SUBROUTINE GetEmissionData



  SUBROUTINE ReadFrames(stride)
    INTEGER :: stride

    DO n=frame,frameend,stride
       PRINT*,' reading frame ',n
       ! read the data
       pnvars=qnvars

       CALL CreateFixedGrid(n)
       !
       SELECT CASE(application)
       CASE(1)
          CALL ProcessHydro(.FALSE.)
       CASE(2)
          CALL ProcessMHD(.FALSE.)
       CASE(3)
          CALL ProcessHydroLLE(.FALSE.)
       CASE DEFAULT
          PRINT*,'Bad Application ID, line 276'
       END SELECT

       IF(n==frame) THEN !This is the initial frame
          SELECT CASE(operation)
          CASE(1)
          CASE(2,6,7) ! fixed grid FITS, Schlieren, user coded
             PRINT*,'Bad Operation ID; martin removed a STOP command from here'
             !STOP
          CASE(3,5) ! log10(rho); field 'userfield', stored in pFix(1)
             MinOverall = MINVAL(pFix(:,:,:,1))
             MaxOverall = MAXVAL(pFix(:,:,:,1))
          CASE(4) ! cut of field 'cutfield' ('userfield')
             IF(userfield<=3 .OR. userfield==5) THEN
                MinOverall = MINVAL(pFix(:,:,:,1))
                MaxOverall = MAXVAL(pFix(:,:,:,1))
             ELSE
                MinOverall = MINVAL(pFix(:,:,:,cutfield))
                MaxOverall = MAXVAL(pFix(:,:,:,cutfield))
             ENDIF
          CASE DEFAULT
             PRINT*,'Bad Operation ID line 296, "stop" removed by mhuarte'
             !STOP
          END SELECT
       ELSE
          SELECT CASE(operation)
          CASE(1)
          CASE(2,6,7) ! fixed grid FITS, Schlieren, user coded
             PRINT*,'Bad Operation ID; martin removed a STOP command from here'
             !STOP
          CASE(3,5) ! log10(rho); field 'userfield', stored in pFix(1)
             MinOverall = MIN(MinOverall,MINVAL(pFix(:,:,:,1)))
             MaxOverall = MAX(MaxOverall,MAXVAL(pFix(:,:,:,1)))
          CASE(4)! cut of field 'cutfield' ('userfield')
             IF(userfield<=3 .OR. userfield==5) THEN
                MinOverall = MIN(MinOverall,MINVAL(pFix(:,:,1,1)))
                MaxOverall = MAX(MaxOverall,MAXVAL(pFix(:,:,1,1)))
             ELSE
                MinOverall = MIN(MinOverall,MINVAL(pFix(:,:,1,cutfield)))
                MaxOverall = MAX(MaxOverall,MAXVAL(pFix(:,:,1,cutfield)))
             ENDIF
          CASE DEFAULT
             PRINT*,'Bad Operation ID, line 316'
          END SELECT
       ENDIF
    END DO

    IF(scales==2) MinOverall=0

!!$     IF(operation==3 .or. Llog) THEN
!!$        IF(MinOverall .gt. 0) THEN 
!!$           MinOverall = log10(MinOverall)
!!$        ELSE
!!$           PRINT*,'Error: MinOverall invalid for log plot'
!!$           PRINT*,'MinOverall, MaxOverall ',MinOverall, MaxOverall
!!$           STOP
!!$        END IF
!!$        MaxOverall = log10(MaxOverall)
!!$     ENDIF
    MinOverall = MinOverall*(1-SIGN(1.,MinOverall)*0.01)
    MaxOverall = MaxOverall*(1+SIGN(1.,MaxOverall)*0.01)
  END SUBROUTINE ReadFrames


  SUBROUTINE ProcessFrames
    REAL, DIMENSION(:,:,:,:), POINTER :: qfix2, temp
    IF (operation == -1) THEN
       CALL setup_vars()
       CALL init_diagnostics((/frame,frameend/))
    END IF
    DO n=frame,frameend
       CALL CreateFixedGrid(n)
       IF (operation == -1) THEN
          CALL cons_to_prim(gi_fixed%q)
          CALL run_diagnostics(n)
       ELSEIF (operation == 0) THEN
          CALL MakeChomboFile(n)
       ELSEIF (operation == 9) THEN
          CALL SplitGrid(np,n)
       ELSEIF (operation == 10) THEN

          !CALL cons_to_prim(qfix)
          temp=>qfix
          NULLIFY(qfix)
          NULLIFY(gi_fixed%q)
          outdir='ref'
          CALL CreateFixedGrid(n)
          !CALL cons_to_prim(qfix)
          qfix2=>qfix
          qfix=>temp
          gi_fixed%q=>qfix
          CALL compare(qfix,qfix2)
          DEALLOCATE(qfix2)
          outdir='out'

       ELSEIF (operation == 11) THEN
          IF (TestNeedsRefFrame()) THEN
             temp=>qfix
             NULLIFY(qfix)
             NULLIFY(gi_fixed%q)
             outdir='ref'
             CALL CreateFixedGrid(n)
             !CALL cons_to_prim(qfix)
             qfix2=>qfix
             qfix=>temp
             gi_fixed%q=>qfix
             CALL DoTest(gi_fixed,qfix2)
             DEALLOCATE(qfix2)
             outdir='out'
          ELSE
             CALL DoTest(gi_fixed)
          END IF
       ELSEIF (operation == 12) THEN
          CALL ProcessEmiss
       ELSEIF (operation == 13) THEN
          CALL PrintASCIIColumnFile(n)
       ELSEIF (operation == 14) THEN
          CALL PrintBovOfPotential(n)
       ELSE
          SELECT CASE(application)
          CASE(1)
             CALL ProcessHydro(.FALSE.)
          CASE(2)
             CALL ProcessMHD(.FALSE.)
          CASE(3)
             CALL ProcessHydroLLE(.FALSE.)
          CASE DEFAULT
             PRINT*,'Bad Application ID, line 396'
          END SELECT

          SELECT CASE(operation)
          CASE(1) ! fixed grid HDF
             CALL CreateFixedGrid(n)
             PRINT*,"Creating qFix"
             CALL WriteHDF(n)
             PRINT*,"Writing Fixed HDF5"
          CASE(2) ! fixed grid FITS
             CALL WriteFITS(PRIM)
          CASE(3,5) ! log10(rho); plot of field 'userfield'
             IF(scales .eq. 0) THEN
                MinOverall = MINVAL(pFix(:,:,:,1))
                MaxOverall = MAXVAL(pFix(:,:,:,1))
             END IF

             CALL PlotFrame(1,mxFix(1),1,mxFix(2),1,maxHotBox,MinOverall,MaxOverall)
          CASE(4) ! cut of field 'cutfield' ('userfield')
             IF(cutfield<=3 .OR. cutfield==5) THEN
                IF(scales==0) THEN 
                   MinOverall=MINVAL(pFix(:,:,:,1))
                   MaxOverall=MAXVAL(pFix(:,:,:,1))
                ENDIF
                CALL PlotCut(1,mxFix(1),1,mxFix(2),1,maxHotBox,MinOverall,MaxOverall)
             ELSE
                IF(scales==0) THEN 
                   MinOverall=MINVAL(pFix(:,:,:,cutfield))
                   MaxOverall=MAXVAL(pFix(:,:,:,cutfield))
                ENDIF
                CALL PlotCut(1,mxFix(1),1,mxFix(2),cutfield,maxHotBox,MinOverall,MaxOverall)
             ENDIF
          CASE(6) ! schlieren
             IF(ndim==2) THEN
                 ![BDS][20100826]:  Added user-defined ranges for schlieren plotting.
                 IF (scales == 3) THEN
                     CALL PlotSchlieren(1,mxFix(1),1,mxFix(2), MinOverall, MaxOverall)
                 ELSE
                     CALL PlotSchlieren(1,mxFix(1),1,mxFix(2))
                 END IF
             END IF
             IF(ndim==3) CALL PlotSchlieren3D(1,mxFix(1),1,mxFix(2))

          CASE(8) ! user option, code somthing here

             SELECT CASE (i_jetset_option)

             CASE(0)    ! Kelvin-Helmholtz test.

                !CALL PrintASCIIcolumnFILE(n, 5.d0/3)
                CALL PrintGenericJetsetFile(n, 5.d0/3)
                !CALL WriteToKHEnergyFile

             CASE(1)    ! Radiative shock test.

                CALL PrintRadShockFlowFile(n, 5.d0/3)
                CALL WriteToRadShockPositionFile(5.d0/3)

             CASE DEFAULT
                PRINT *, "Noooooo..."
             END SELECT

          CASE(7)
             CALL writeLevelArea(n)
          CASE DEFAULT
             PRINT*,'Bad Operation ID 454'
          END SELECT
       END IF
       DEALLOCATE(qFix)
       
    END DO
  END SUBROUTINE ProcessFrames


  SUBROUTINE ProcessHydro(lWrite)
    USE BearIO
    LOGICAL lWrite
    INTEGER :: i,j,k,nv
    !
    CALL PrimitiveInit(qnvars)
    ! fill the primitive field
    !rho,vx,vy,Temp
    
    IF(qnvars>=1) CALL Primitive(qn,1)
    IF(qnvars>=2) CALL Primitive(qvx,2)
    IF(qnvars>=3) CALL Primitive(qvy,3)
    IF(iE==5 .AND. qnvars>=4) CALL Primitive(qvz,4)
    IF(iE /= 0) THEN 
       CALL Primitive(qEth,iE)
    ENDIF

    ! tracer fields
    DO nv=nTracerLo,nTracerHi
       DO i=1,mxFix(1);DO j=1,mxFix(2);DO k=1,mxFix(3)
          pFix(i,j,k,nv) = qFix(i,j,k,nv)/qfix(i,j,k,1)
       END DO;END DO;END DO
    END DO

    ! Redefine the primitive array index 1 to be something else if operation == 5
    IF(operation .eq. 5) THEN
       SELECT CASE(userfield)
       CASE(qn,qrho,qvx,qvy,qvz,qv,qEth,qP,qTemp,qcs,qemission, &
            mBtot, mEtot,mbx,mby,mbz,qvorticity,beta)
          CALL Primitive(userfield,1)
       CASE(19:21) ! Tracers
          pFix(:,:,:,1) = pFix(:,:,:,iE+(userfield-18))
       CASE DEFAULT
       END SELECT
    END IF
    IF(Llog) pFix(:,:,:,1) = log10(pFix(:,:,:,1))
    ! write out primitive computational values in file pfixxxxx.hdf
    !IF(lWrite) CALL WriteHDF(PRIM)
  END SUBROUTINE ProcessHydro


  SUBROUTINE ProcessMHD(lWrite)
    USE BearIO
    LOGICAL lWrite
    ! construct a fixed grid for primitive values with 3 fields
    CALL PrimitiveInit(qnvars)
    ! fill the primitive field
    !rho,vx,vy,vz,bx,by,bz,Etotal
    IF(qnvars>=1) CALL Primitive(qn,1)
    IF(qnvars>=2) CALL Primitive(qvx,2)
    IF(qnvars>=3) CALL Primitive(qvy,3)
    IF(qnvars>=4) CALL Primitive(qvz,4)
    IF(qnvars>=5) CALL Primitive(qP,5)
    IF(qnvars>=6) CALL Primitive(mbx,6)
    IF(qnvars>=7) CALL Primitive(mby,7)
    IF(qnvars>=8) CALL Primitive(mbz,8)
    !IF(qnvars>=9) CALL Primitive(beta,9)
    !

    ! Redefine the primitive array index 1 to be something else if operation == 5
    IF(operation .eq. 5) THEN
       SELECT CASE(userfield)
       CASE(qn,qrho,qvx,qvy,qvz,qv,qEth,qP,qTemp,qcs,qemission, &
            mBtot,mEtot,mbx,mby,mbz,qvorticity,beta)
          CALL Primitive(userfield,1)
       CASE(19:21) ! Tracers
          pFix(:,:,:,1) = pFix(:,:,:,iE+(userfield-990))
       CASE DEFAULT
       END SELECT
    END IF

    IF(Llog) pFix(:,:,:,1) = log10(pFix(:,:,:,1))
    ! write out primitive computational values in file pfixxxxx.hdf
    !IF(lWrite) CALL WriteHDF(PRIM)
  END SUBROUTINE ProcessMHD


  ! Just like ProcessHydro except calls for qrho, not qn
  SUBROUTINE ProcessHydroLLE(lWrite)
    USE BearIO
    LOGICAL lWrite
    INTEGER :: i,j,k,nv
    !
    CALL PrimitiveInit(qnvars)
    ! fill the primitive field
    !rho,vx,vy,Temp
    IF(qnvars>=1) CALL Primitive(qrho,1)
    IF(qnvars>=2) CALL Primitive(qvx,2)
    IF(qnvars>=3) CALL Primitive(qvy,3)
    IF(iE==5 .AND. qnvars>=4) CALL Primitive(qvz,4)
    IF(qnvars>=iE) THEN 
       CALL Primitive(qP,iE)
    ENDIF

    ! tracer fields
    DO nv=iE+1,qnvars
       DO i=1,mxFix(1);DO j=1,mxFix(2);DO k=1,mxFix(3)
          pFix(i,j,k,nv) = qFix(i,j,k,nv)/qFix(i,j,k,1)
       END DO;END DO;END DO
    END DO

    ! Redefine the primitive array index 1 to be something else if operation == 5
    IF(operation .eq. 5) THEN
       SELECT CASE(userfield)
       CASE(qn,qrho,qvx,qvy,qvz,qv,qEth,qP,qTemp,qcs,qemission, &
            mBtot,mEtot,mbx,mby,mbz,qvorticity,beta)
          CALL Primitive(userfield,1)
          PRINT*,MAXVAL(pFix(:,:,:,1))
       CASE(19:21) ! Tracers
          pFix(:,:,:,1) = pFix(:,:,:,iE+(userfield-990))
       CASE DEFAULT
       END SELECT
    END IF

    IF(Llog) pFix(:,:,:,1) = log10(pFix(:,:,:,1))

    ! write out primitive computational values in file pfixxxxx.hdf
    !IF(lWrite) CALL WriteHDF(PRIM)
  END SUBROUTINE ProcessHydroLLE


  SUBROUTINE ProcessEmiss
    USE BearIO
    INTEGER :: i,j,k,nv
    REAL,ALLOCATABLE,DIMENSION(:,:,:,:) :: NTXvals
    !
    PRINT*,'Begin processing emission'

    CALL PrimitiveInit(5)
    ! fill the primitive field
    !    Ne : Tracer02/muH 
    !    T  : temperature (K)
    !    Xh : Tracer01/(Tracer01+Tracer02)
    !  nTot : (Tracer01 + 2*Tracer02)/muH
    !  nHTot: (Tracer01 + Tracer02)/muH
    ! where muH = 1.00794
    
    ! Ne
    FORALL(i=1:mxFix(1),j=1:mxFix(2),k=1:mxFix(3))
       pFix(i,j,k,1) = (qFix(i,j,k,NTracerLO+1)/muH)*nScale
    END FORALL
    ! T
    CALL Primitive(qTemp,2)
    ! Xh
    FORALL(i=1:mxFix(1),j=1:mxFix(2),k=1:mxFix(3))
       pFix(i,j,k,3) = qFix(i,j,k,nTracerLO+1)/(qFix(i,j,k,nTracerLO)+qFix(i,j,k,nTracerLO+1))
    END FORALL
    ! ensure ionization fraction is between ~0 and 1
    WHERE(pFix(:,:,:,3) <= 0.0) pFix(:,:,:,3) = 1e-10
    WHERE(pFix(:,:,:,3) > 1.0) pFix(:,:,:,3) = 1.0
    
    ! nHTot
    FORALL(i=1:mxFix(1),j=1:mxFix(2),k=1:mxFix(3))
       pFix(i,j,k,4) = ((qFix(i,j,k,nTracerLO) + qFix(i,j,k,nTracerLO+1))/muH)*nScale
    END FORALL

    ! Calculate emission
    ! This will create separate emission BOV files for each transition requested.
    ! These files then may be read in 'angledemission' and plotted.
    ! ** As of 10/2010, angledemission is hard-coded to do Halpha and Sii, only.
    PRINT*,mxFix(1),mxFix(2),mxFix(3)
    PRINT*,SIZE(pFix)
    ALLOCATE(NTXvals(5,mxFix(1),mxFix(2),mxFix(3)))
    NTXVals(1,:,:,:) = pFix(:,:,:,1)
    NTXVals(2,:,:,:) = pFix(:,:,:,2)
    NTXVals(3,:,:,:) = pFix(:,:,:,3)
    NTXVals(4,:,:,:) = pFix(:,:,:,4)

    PRINT*, "ne=", NTXVals(1,:,:,:)
    PRINT*, "T=", NTXVals(2,:,:,:)
    PRINT*, "XH=", NTXVals(3,:,:,:)
    PRINT*, "nHTot=", NTXVals(4,:,:,:)
    ! CalcEmiss also does the BOV and/or Shape output
    CALL CalcEmiss(NTXVals,mxFix(1),mxFix(2),mxFix(3),species,spec_char,exlevels,lev1_char,lev2_char)
!    pFix(:,:,:,5) = NTXVals(5,:,:,:)

    ! Also create velocity and rho^2 BOV files.
    !    vx : x-velocity [km/s]
    !    vy : y-velocity [km/s]
    !    vz : z-velocity [km/s]
    ! rho^2 : density [cgs]
    pFix=0d0
    CALL Primitive(qvx,1)
    CALL Primitive(qvy,2)
    IF(nDim==3) CALL Primitive(qvz,3)
    CALL Primitive(qrho,nDim+1)
    pFix(:,:,:,nDim+1)=pFix(:,:,:,nDim+1)**2

!    CALL WriteVars(pFix(:,:,:,1:nDim+1),T)
  END SUBROUTINE ProcessEmiss

!!$  SUBROUTINE WriteVars(array,lBOV)
!!$    REAL(KIND=qPrec),DIMENSION(:,:,:,:),POINTER :: array
!!$    LOGICAL :: lBOV
!!$    
!!$    IF(lBOV) THEN
!!$       ! Write BOV
!!$    ELSE
!!$       ! Write ASCII
!!$    END IF
!!$  END SUBROUTINE WriteVars



  SUBROUTINE ProcessContestOptions

    INTEGER :: i_err

	! Prompt user to select AstroBEAR standard or Chombo file inputs.
    IF (lInteractive) THEN
      WRITE(*, '(A)') "Select the type of JetSet contest output you wish to print:"
      WRITE(*, '(A)') "     0)  Kelvin-Helmholtz."
      WRITE(*, '(A)') "     1)  Radiative Shock."

      WRITE(*,'(A)',ADVANCE='NO')'Please input your selection:  '
      READ*, i_jetset_option
    END IF

    IF (i_jetset_option == 0) THEN

        OPEN(UNIT=314,FILE='./out/kh_energy.dat',STATUS='REPLACE',IOSTAT=i_err)

        IF (i_err /= 0) THEN
            PRINT *, "ProcessContestOptions error: unable to open file 'kh_energy.dat'."
            STOP
        END IF

        CLOSE(UNIT=314, IOSTAT=i_err)

        IF (i_err /= 0) THEN
            PRINT *, "ProcessContestOptions error: unable to close file 'kh_energy.dat'."
            STOP
        END IF

    ELSE IF (i_jetset_option == 1) THEN

        OPEN(UNIT=314,FILE='./out/radshock_pos.dat',STATUS='REPLACE',IOSTAT=i_err)

        IF (i_err /= 0) THEN
            PRINT *, "ProcessContestOptions error: unable to open file '.radshock_pos.dat'."
            STOP
        END IF

        CLOSE(UNIT=314, IOSTAT=i_err)

        IF (i_err /= 0) THEN
            PRINT *, "ProcessContestOptions error: unable to close file 'radshock_pos.dat'."
            STOP
        END IF

    END IF

  END SUBROUTINE ProcessContestOptions
