#!/bin/bash
####################################################################################
# Script name: buildtests.s
# Created:  20110223 by Brandon D. Shroyer
# Purpose:  Compile AstroBEAR 2.0 executables for a list of standard test problems,
#           create run directories for these executables, and run a series of tests.
# Notes:    This script assumes that the tests are being run on bluehive.  It also
#           assumes that:
#               a) All required PBS scripts are in the CODE_DIR/test_scripts/ directory,
#               b) All tests in the test list correspond to actual problem directories in CODE_DIR/modules.
####################################################################################


# Declare the base scratch directory.  This is where all of the problem directories
# will be set up.  This is currently set up to run in the group space on bluehive.
PROBLEM_BASE_DIR=/scratch/afrank_group

# Set the code directory to the current directory.  This could be changed to a
# command line parameter in the future.
CODE_DIR=.

# A list of the basic tests to be run.  These tests should match the names of the
# problem directories in CODE_DIR/modules.
basic_tests=( 'FieldLoopAdvection' 'OrbitingParticles' 'UniformCollapse' )
test_count=${#basic_tests[@]}

# Move to the appropriate code directory.
cd ${CODE_DIR}

# Link to the test makefile.
rm Makefile.inc
ln -s Makefile.inc.bluehive_test Makefile.inc

# Copy the PBS scripts to the problem base directory.
cp test_scripts/*.pbs ${PROBLEM_BASE_DIR}

# Loop over the list of tests.
for (( i=0;i<$test_count;i++)); do

    test=${basic_tests[${i}]}

    # Link to the selected test's problem directory.
    cd ${CODE_DIR}/modules
    rm Problem
    ln -s ${test} Problem

    # Compile the code using the specified problem.
    cd ..
    make clean
    make
    if [ -f astrobear ];  then
      echo ${test} "make done."
    else
      echo ${test} "failed to compile.  Aborting"
      exit 1
    fi

    # Loop over all the desired processor counts and levels.
    for nprocs in 1 8 16; do

        for level in 0 2; do

            # Specify the name of the problem directory for this test.  The directory name will include
            # the test's name, plus the number of processors and the number of levels of refinement.
            run_dir="${PROBLEM_BASE_DIR}/${test}_test_${nprocs}proc_${level}amr_clean"

            # If this problem directory does not exist, then create it.  This includes creating the
            # out/ sub directory and copying over the cooling tables.
            if [ ! -e ${run_dir} ]; then
                mkdir ${run_dir}
                mkdir ${run_dir}/out
                cp -r ${CODE_DIR}/source/TABLES ${run_dir}
            fi

	
            # Copy over the newly-compiled executable.
               cp astrobear ${run_dir}

            # Copy the problem data files to the run directory.
            cp modules/${test}/*.data ${run_dir}

            # Copy the global.data file for the appropriate AMR level over to the problem directory.
            cp modules/${test}/global.${level}amr.data ${run_dir}/global.data

        done # End of the level loop.

        # Create a restart setup for eight-processor jobs.  The PBS scripts will copy over output files from
        # the 2-AMR level run to use in the restart run.
        if [ ${nprocs} -eq 8 ]; then

            # Specify the restart directory name.
            restart_dir="${PROBLEM_BASE_DIR}/${test}_test_${nprocs}proc_2amr_restart"

            # If the restart directory does not exist, then create it.
            if [ ! -e ${restart_dir} ]; then
                mkdir ${restart_dir}
                mkdir ${restart_dir}/out
                cp -r ${CODE_DIR}/source/TABLES ${restart_dir}
            fi

            # Copy over the newly-compiled executable.
            cp astrobear ${restart_dir}

            # Copy the problem data files to the problem directory.
            cp modules/${test}/*.data ${restart_dir}

            # Copy the restart-configured global.data file to the problem directory.
            cp modules/${test}/global.restart.data ${restart_dir}/global.data
        fi

    done  # End of the nprocs loop.

    echo ${test} "copy done."

done

echo -e "\n"

cd ${PROBLEM_BASE_DIR}

# Open up permissions on PBS scripts so that they can be overwritten next time.
chmod 777 test_*proc.pbs

for i in 1 8 16; do
    echo -e "Submitting ${i}-processor job...\c"
    qsub test_${i}proc.pbs
    echo "done."
done
