!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    message_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file message_declarations.f90
!! @brief Main file for module MessageDeclarations

!> @defgroup MessageDeclarations Message Declarations
!! @brief Defines the data structures used by the MPI packing and transmission layers, as well as the methods that operate on them.
!! @ingroup Communication

!> Contains routines for managing messages
!! @author Brandon D. Shroyer
!! @date 8-21-2010
!! @ingroup MessageDeclarations
MODULE MessageDeclarations

   USE GlobalDeclarations

   IMPLICIT NONE
   PRIVATE

   PUBLIC RequestList, CreateMpiRequest, ClearAllRequests, ProcessorList, AddMpiRequest
   PUBLIC MessageBlock, CreateMessageBlock, ClearMessageBlocks, ClearAllMessageBlocks, AddMessageBlock
   PUBLIC PackedMessage, CreatePackedMessageObject, DestroyPackedMessageObject
   PUBLIC MessageList, DestroyMessageListObject
   PUBLIC StageMessageGroup, CreateMessageGroupObject, DestroyMessageGroupObject, pStageMessageGroup
   PUBLIC AppendProcessorToList, AddProcessorToMGList, AddMessageToList
   PUBLIC TERMINATIONBOX, TERMINATION_BOX_BYTES

   INTEGER, PUBLIC, PARAMETER :: STD_BUFFER_SIZE = 16384    ! 16 KB



   INTEGER, PUBLIC, PARAMETER :: I_FIRST_REQUESTS = 1
   INTEGER, PUBLIC, PARAMETER :: I_CURRENT_REQUESTS = 2
   INTEGER, PUBLIC, PARAMETER :: I_LAST_REQUESTS = 3

   !INTEGER, PUBLIC, PARAMETER :: TAG_LEVEL_MULTIPLIER = 200
   !INTEGER, PUBLIC, PARAMETER :: TAG_STAGE_MULTIPLIER = 2000
   INTEGER, PUBLIC, PARAMETER :: MAX_MESSAGE_PIECES = 100 !max number of messages for one stage

   !> Contains lists of mpi_requests
   TYPE RequestList
      INTEGER :: request
      TYPE(RequestList), POINTER :: next
      LOGICAL :: completed
   END TYPE RequestList

   !> Contains a list of data blocks for receiving messages
   TYPE MessageBlock
      CHARACTER, ALLOCATABLE, DIMENSION(:) :: buffer
      INTEGER :: block_offset
      INTEGER :: buffer_size
      INTEGER :: block_id
      TYPE(MessageBlock), POINTER :: next
   END TYPE MessageBlock

   !> Contains data needed for a single processor to processor message.
   TYPE PackedMessage
      INTEGER :: remote_proc
      INTEGER :: level
      INTEGER :: header
      INTEGER :: offset
      INTEGER :: nMessages
      INTEGER :: tag
      INTEGER :: nRequests
      INTEGER :: buffer_size
      TYPE(RequestList), POINTER :: requests
      TYPE(RequestList), POINTER :: current_request
      TYPE(RequestList), POINTER :: last_request
      TYPE(MessageBlock), POINTER :: blocks
      TYPE(MessageBlock), POINTER :: current_block
      TYPE(MessageBlock), POINTER :: last_block
      LOGICAL :: closed
      LOGICAL :: lSend
      LOGICAL :: first_block_received
      LOGICAL :: lMultiBlock
   END TYPE PackedMessage

   !> Contains a list of packed messages
   Type MessageList
      TYPE(PackedMessage), POINTER :: self
      TYPE(MessageList), POINTER :: next
   End Type MessageList

   !> Contains a list of processors
   TYPE ProcessorList
      INTEGER :: self
      TYPE(ProcessorList), POINTER :: next
   END TYPE ProcessorList

   !> Contains information about all messages in a given stage
   TYPE StageMessageGroup
      TYPE(ProcessorList), POINTER :: proclist
      TYPE(ProcessorList), POINTER :: last_proc
      TYPE(MessageList), POINTER :: messages
      TYPE(MessageList), POINTER :: last_message
      INTEGER :: nPackedMessages    
      LOGICAL :: lSend
      INTEGER :: level
      INTEGER :: iStageTag
   End TYPE StageMessageGroup

   !> Contains a pointer to a StageMessageGroup
   TYPE pStageMessageGroup
      TYPE(StageMessageGroup), POINTER :: p
   END TYPE pStageMessageGroup


CONTAINS

 
  !> @name RequestList routines
  !! @{

   !> Adds a new MpiRequest structure to the PackedMessage structure.
   !! @param mpi_request An MPI request handle.
   !! @param last_request The request list to which request will be appended.
   !! @param first_request An optional parameter indicating the first node in the last_request request list.
   !! @param current_request Another optional parameter providing another pointer that will point to the new request.
   SUBROUTINE CreateMpiRequest(mpi_request, last_request, first_request, current_request)

      TYPE(RequestList), POINTER :: last_request
      TYPE(RequestList), POINTER, OPTIONAL :: first_request
      TYPE(RequestList), POINTER, OPTIONAL :: current_request
      INTEGER :: mpi_request

      INTEGER :: iErr

!  TEsting
      iErr = 0

     IF (.NOT. ASSOCIATED(last_request)) THEN
         ALLOCATE(last_request, STAT=iErr)

         IF (PRESENT(first_request)) THEN
            first_request => last_request
         END IF

         IF (PRESENT(current_request)) THEN
            current_request => last_request
         END IF

     ELSE
         IF (ASSOCIATED(last_request%next)) THEN
            PRINT *, "CreateMpiRequest error: last_request%next is associated."
            STOP
         END IF

         ALLOCATE(last_request%next, STAT=iErr)

         last_request => last_request%next
      END IF

      IF (iErr /= 0) THEN
         PRINT *, "CreateMpiRequest() error: unable to allocate MPI Request object."
         STOP
      END IF

      last_request%request = mpi_request
      last_request%completed = .FALSE.
      NULLIFY(last_request%next)

   END SUBROUTINE CreateMpiRequest


   !> Deletes elements from an MPI RequestList structure, starting with the referenced request.
   !! @param request_list A reference to a request list.
   !   SUBROUTINE ClearRequestList(request_list)
   RECURSIVE SUBROUTINE ClearRequestList(request_list)

      TYPE(RequestList), POINTER :: request_list


      IF (.NOT. ASSOCIATED(request_list))  RETURN

      IF (ASSOCIATED(request_list%next))  CALL ClearRequestList(request_list%next)

      NULLIFY(request_list%next)
      DEALLOCATE(request_list)
      NULLIFY(request_list)

   END SUBROUTINE ClearRequestList

   !> Deletes all MPI message requests associated with a message object.
   !! @param message The PackedMessage object to be cleared of requests.
   SUBROUTINE ClearAllRequests(message)

      TYPE(PackedMessage), POINTER :: message


      NULLIFY(message%last_request)
      NULLIFY(message%current_request)
      CALL ClearRequestList(message%requests)
      NULLIFY(message%requests)
      message%nRequests = 0

   END SUBROUTINE ClearAllRequests

   !> Adds an MPI request to the message object.
   !! @param message A packed message object to hold the request.
   !! @param request An MPI request handle.
   !! @param init_message A logical flag indicating whether or not this is being called when the message is initialized.
   SUBROUTINE AddMpiRequest(message, request, init_message)

      TYPE(PackedMessage), POINTER :: message
      INTEGER :: request
      LOGICAL :: init_message

      IF (init_message) THEN
         CALL CreateMpiRequest(request, message%last_request, message%requests, message%current_request)
      ELSE
         CALL CreateMpiRequest(request, message%last_request)
      END IF

      message%nRequests = message%nRequests + 1

   END SUBROUTINE AddMpiRequest


   !> @}

   !> @name MessageBlock routines
   !! @{

   !> Adds a new MessageBlock structure to the the block list passed in.
   !! @param message_size The size of the new block to be created.
   !! @param last_block The MessageBlock list to which the new block will be appended.
   !! @param first_block An optional MessageBlock pointer that will point to a freshly-created MessageBlock list.
   !! @param current_block Another optional MessageBlock pointer that will point to a freshly-created MessageBlock list.
   SUBROUTINE CreateMessageBlock(message_size, last_block, first_block, current_block)

      INTEGER :: message_size
      TYPE(MessageBlock), POINTER :: last_block
      TYPE(MessageBlock), POINTER, OPTIONAL :: first_block
      TYPE(MessageBlock), POINTER, OPTIONAL :: current_block

      INTEGER :: iErr


      iErr = 0

      IF (message_size <= 0) THEN
          PRINT *, "CreateMessageBlock() error: invalid message buffer size ', i, '.')", message_size
      END IF

      IF (.NOT. ASSOCIATED(last_block)) THEN
         ALLOCATE(last_block, STAT=iErr)
         IF (PRESENT(first_block))  first_block => last_block
         IF (PRESENT(current_block))  current_block => first_block

      ELSE

         IF (ASSOCIATED(last_block%next)) THEN
            PRINT *, "CreateMessageBlock error: last_block%next associated."
            STOP
         END IF

         ALLOCATE(last_block%next, STAT=iErr)
         last_block => last_block%next

      END IF

      IF (iErr /= 0) THEN
         PRINT *, "CreateMessageBlock() error: unable to allocate new message block."
         STOP
      END IF

      ALLOCATE(last_block%buffer(message_size))
      CALL CheckAllocation(MessageAllocator, message_size)
      
      last_block%block_offset = 0
      last_block%buffer_size = 0
      last_block%block_id = 0
      NULLIFY(last_block%next)

   END SUBROUTINE CreateMessageBlock

   !> Deletes elements from a MessageBlock list structure, starting with the referenced request.
   !! @param block_list A reference to a message block list list.
   RECURSIVE SUBROUTINE ClearMessageBlocks(block_list)

      TYPE(MessageBlock), POINTER :: block_list


      IF (.NOT. ASSOCIATED(block_list))  RETURN

      IF (ASSOCIATED(block_list%next))  CALL ClearMessageBlocks(block_list%next)
      CALL CheckDeAllocation(MessageAllocator, size(block_list%buffer))
      DEALLOCATE(block_list%buffer)
      DEALLOCATE(block_list)
      NULLIFY(block_list)

   END SUBROUTINE ClearMessageBlocks

   !> Deletes all message blocks associated with a message object.
   !! @param message The PackedMessage object to be cleared of message blocks.
   SUBROUTINE ClearAllMessageBlocks(message)

      TYPE(PackedMessage), POINTER :: message


      NULLIFY(message%last_block)
      NULLIFY(message%current_block)
      IF (ASSOCIATED(message%blocks))  CALL ClearMessageBlocks(message%blocks)
      NULLIFY(message%blocks)
      Message%nMessages = 0

   END SUBROUTINE ClearAllMessageBlocks

   !> Adds a new message block to the message object.
   !! @param message A packed message object to hold the new message block.
   !! @param init_message A logical flag indicating whether or not this is being called when the message is initialized.
   !! @param message_size The size of the new message block's buffer.
   SUBROUTINE AddMessageBlock(message, init_message, message_size)

      TYPE(PackedMessage), POINTER :: message
      LOGICAL :: init_message
      INTEGER :: message_size
      INTEGER :: iErr


      IF (init_message) THEN
         CALL CreateMessageBlock(message_size, message%last_block, message%blocks, message%current_block)
      ELSE
         CALL CreateMessageBlock(message_size, message%last_block)
      END IF

      message%nMessages = message%nMessages + 1
      message%last_block%block_id = message%nMessages

      ! If the message has TAG_LEVEL_MULTIPLIER block or more, then the blocks run the risk
      ! of colliding with messages on other levels or overwriting blocks that belong to other
      ! messages.  Therefore, if this new block pushes the message to the TAG_LEVEL_MULTIPLIER
      ! limit, then kill the program (with a clear-ish explanation of what happened).
      !IF (message%nMessages >= TAG_LEVEL_MULTIPLIER) THEN
      IF (message%nMessages >= MAX_MESSAGE_PIECES) THEN
          PRINT "('AddMessageBlock() error::message block overrun (count=', i1, ').')", message%nMessages
          PRINT "('   AddMessageBlock() error::message tag = ', i5, '.')", message%tag
          PRINT "('   AddMessageBlock() error::buffer size = ', i1, '.')", message%buffer_size
          PRINT "('   AddMessageBlock() error::lSend = ', l, '.')", message%lSend
          PRINT *
          PRINT *, "*** ERROR ***:  This run failed because your message block size was too small.  ", &
                   "message block size and try again."
          CALL MPI_FINALIZE(iErr)
          STOP
      END IF

   END SUBROUTINE AddMessageBlock

   !> @}

   !> @name PackedMessage routines
   !! @{

   !> Initializes a PackedMessage data structure.
   !! @param level The level on which the creating process is working.
   !! @param remote_proc The ID of the processor with which this object will interact.
   !! @param tag The MPI message tag this object's MPI message will have.
   !! @param lSend Flag whether message is to be sent or received
   !! @param message_size The size of the message buffer to be created.
   !! @param message A pointer for the new message object.
   SUBROUTINE CreatePackedMessageObject(level, remote_proc, tag, lSend, message_size, message)

      INTEGER :: level
      INTEGER :: remote_proc
      INTEGER :: tag
      LOGICAL :: lSend
      INTEGER :: message_size
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: iErr


!      IF (ASSOCIATED(message)) THEN
!         PRINT *, "CreatePackedMessage() error:  message object is already associated."
!         STOP
!      END IF

      IF (remote_proc == MPI_id) THEN
          PRINT "('CreatePackedMessage() error: Proc ', i4, ' attempting to post a message for itself.')", remote_proc
          STOP
      END IF

      IF (message_size <= 0) THEN
          !PRINT "('CreatePackedMessage(', i1, '=>', i1,', level=', i2, ', tag=', i6, ') error: invalid message size ', i10, '.')", MPI_id, remote_proc, level, tag, message_size
          PRINT *, "Error: invalid message size"
          STOP
      END IF

      ALLOCATE(message, STAT=iErr)

      IF (iErr /= 0) THEN
         PRINT *, "CreatePackedMessage() error: unable to allocate message object."
         STOP
      END IF

      message%level = level
      message%tag = tag
      message%remote_proc = remote_proc
      message%offset = 0
      message%nMessages = 0
      message%nRequests = 0
      message%lSend = lSend
      message%buffer_size = message_size

      message%closed = .FALSE.

      NULLIFY(message%requests)
      NULLIFY(message%current_request)
      NULLIFY(message%last_request)
      NULLIFY(message%blocks)
      NULLIFY(message%current_block)
      NULLIFY(message%last_block)

   END SUBROUTINE CreatePackedMessageObject

   !> Deallocates the message object and all associated data structures.
   !! @param message A message object to be deallocated.
   SUBROUTINE DestroyPackedMessageObject(message)

      TYPE(PackedMessage), POINTER :: message


      IF (.NOT. ASSOCIATED(message))  RETURN

      CALL ClearAllRequests(message)
      CALL ClearAllMessageBlocks(message)
      DEALLOCATE(message)
      NULLIFY(message)

   END SUBROUTINE DestroyPackedMessageObject

   !> @}

   !> @name MessageList routines
   !! @{

   !> Appends a MessageList object (with accompanying PackedMessage object) to the end of a list.
   !! @param message The PackedMessage object to be appended to the list (this must be an associated object).
   !! @param msg_list The MessageList to receive a new member.
   !! @param last_message The last message in the message list
   SUBROUTINE AddMessageToList(message,last_message,msg_list)
      TYPE(MessageList), POINTER :: last_message
      TYPE(PackedMessage), POINTER :: message
      TYPE(MessageList), POINTER, OPTIONAL :: msg_list
      INTEGER :: iErr


      IF (.NOT. ASSOCIATED(last_message)) THEN     
         ALLOCATE(last_message, STAT=iErr)
         NULLIFY(last_message%next)
         NULLIFY(last_message%self)
         IF (iErr /= 0) THEN
            PRINT *, "AddMessage() error: unable to allocate last_message list object."
            STOP
         END IF
         IF (present(msg_list)) msg_list=>last_message
      ELSE
         IF (ASSOCIATED(last_message%next)) THEN
            PRINT *, "Error - last message next allocated"
            STOP
         END IF
         ALLOCATE(last_message%next, STAT=iErr)
         IF (iErr /= 0) THEN
             PRINT *, "AddMessage() error: unable to allocate last_message%next list object."
             STOP
         END IF
         last_message=>last_message%next         
         NULLIFY(last_message%next)
         NULLIFY(last_message%self)

      END IF
      IF (ASSOCIATED(message)) THEN
         last_message%self=>message
      ELSE
         ALLOCATE(last_message%self, STAT=iErr)
         
         IF (iErr /= 0) THEN
            PRINT *, "AddMessage() error: unable to allocate last_message%self object."
            STOP
         END IF         
         message=>last_message%self         
      END IF
   END SUBROUTINE AddMessageToList

   !> Destroys a MessageList object and all its children.  DO NOT USE THIS SUBROUTINE AT THE PARSING LAYER OR ABOVE.
   !! @param message_list A pointer to the message list being destroyed.
   RECURSIVE SUBROUTINE DestroyMessageListObject(message_list)

      TYPE(MessageList), POINTER :: message_list


      IF (.NOT. ASSOCIATED(message_list))  RETURN

      IF (ASSOCIATED(message_list%next))  CALL DestroyMessageListObject(message_list%next)
      IF (ASSOCIATED(message_list%self))  CALL DestroyPackedMessageObject(message_list%self)

      DEALLOCATE(message_list)
      NULLIFY(message_list)

   END SUBROUTINE DestroyMessageListObject

   !> @}

   !> @name StageMessageGroup routines
   !! @{

   !> Creates a StageMessageGroup object.  Note that this does NOT populate it with messages.
   !! @param sm_group The StageMessageGroup object being filled out.
   !! @param iStageTag The MPI tag identifier of the stage this StageMessageGroup is handling.
   !! @param lSend Set to true if this is a sending operation; false if it is a receiving operation.
   !! @param level The level of the current transmissions.
   SUBROUTINE CreateMessageGroupObject(sm_group, iStageTag, lSend, level)

      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER :: iStageTag
      LOGICAL :: lSend
      INTEGER :: level

      INTEGER :: iErr

!      write(*,*) "creating message group object", iStageTag, lSend, level
      IF (ASSOCIATED(sm_group)) THEN
         PRINT *, "CreateMessageGroup() error: message group already allocated."
         print*, iStageTag, lSend, level
         STOP
      END IF

      ALLOCATE(sm_group, STAT=iErr)
      
      IF (iErr /= 0) THEN
         PRINT *, "CreateMessageGroup() error: unable to allocate StageMessageGroup object."
      END IF

      NULLIFY(sm_group%proclist)
      NULLIFY(sm_group%last_proc)                
      NULLIFY(sm_group%messages)
      NULLIFY(sm_group%last_message)

      sm_group%nPackedMessages = 0
      sm_group%iStageTag = iStageTag
      sm_group%lSend = lSend
      sm_group%level = level

   END SUBROUTINE CreateMessageGroupObject

   !> Releases all memory referenced by a message group.  DO NOT USE THIS SUBROUTINE AT THE PARSING LAYER OR ABOVE.
   !! @param sm_group The StageMessageGroup object to be destroyed.
   SUBROUTINE DestroyMessageGroupObject(sm_group)
      TYPE(StageMessageGroup), POINTER :: sm_group


      IF (.NOT. ASSOCIATED(sm_group))  RETURN

      NULLIFY(sm_group%last_proc)
      NULLIFY(sm_group%last_message)

      IF (ASSOCIATED(sm_group%messages)) CALL DestroyMessageListObject(sm_group%messages)
      IF (ASSOCIATED(sm_group%proclist)) CALL ClearProcessorList(sm_group%proclist)

      DEALLOCATE(sm_group)
      NULLIFY(sm_group)

   END SUBROUTINE DestroyMessageGroupObject

   !> @}

   !> @name ProcessorList routines
   !! @{

   !> Appends a new processor to the processor list.  This subroutine will not fail on a duplicate entry, but it will not append duplicates.
   !! @param proc_list The processor list to receive a new entry.
   !! @param proc_id The processor ID be appended to the list.
   !! @param last_proc The last processor in the list
   SUBROUTINE AppendProcessorToList(proc_id, last_proc, proc_list)
      TYPE(ProcessorList), POINTER :: last_proc
      TYPE(ProcessorList), POINTER, OPTIONAL :: proc_list
      INTEGER :: proc_id
      INTEGER :: iErr

      IF (.NOT. ASSOCIATED(Last_proc)) THEN     
        ALLOCATE(Last_proc, STAT=iErr)
        NULLIFY(Last_proc%next)
        IF (iErr /= 0) THEN
           PRINT *, "AppendProcessorToList() error: unable to allocate Last_proc list object."
           STOP
         END IF
         IF (present(proc_list)) proc_list=>last_proc
      ELSE
         IF (ASSOCIATED(Last_proc%next)) THEN
            PRINT *, "Error - last proc_id next allocated"
            STOP
         END IF
         ALLOCATE(Last_proc%next, STAT=iErr)
         IF (iErr /= 0) THEN
             PRINT *, "AddProc_id() error: unable to allocate Last_proc%next list object."
             STOP
         END IF
         Last_proc=>Last_proc%next         
         NULLIFY(Last_proc%next)
      END IF
      Last_proc%self=proc_id
   END SUBROUTINE AppendProcessorToList


   !> Appends a Processor to a MessageGroup's processor list
   !! @param sm_group The MessageGroup object
   !! @param proc_id The processor ID
   SUBROUTINE AddProcessorToMGList(sm_group, proc_id)
      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER :: proc_id


      IF (.NOT. ASSOCIATED(sm_group)) THEN
         PRINT *, "AddProcessorToMGList() error: no message group associated."
         STOP
      END IF

      CALL AppendProcessorToList(proc_id, sm_group%last_proc, sm_group%proclist)

   END SUBROUTINE AddProcessorToMGList





   !> Deallocates and nullifies a processor list, starting with the given ProcessorList structure.
   !! @param proc_list The head of the processor list to be cleared.
   RECURSIVE SUBROUTINE ClearProcessorList(proc_list)

      TYPE(ProcessorList), POINTER :: proc_list


      IF (.NOT. ASSOCIATED(proc_list))  RETURN

      IF (ASSOCIATED(proc_list%next))  CALL ClearProcessorList(proc_list%next)

      DEALLOCATE(proc_list)
      NULLIFY(proc_list)

   END SUBROUTINE ClearProcessorList

   !> @}


END MODULE MessageDeclarations


