!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    mpi_packing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file mpi_packing.f90
!! @brief Main file for module MpiPacking

!> @defgroup MpiPacking MpiPacking
!! @brief Defines the packing interface that organizes data from the parsing layer into MPI message blocks.
!! @ingroup Communication

!> Defines the packing interface that organizes data from the parsing layer into MPI message blocks.
!! @ingroup MpiPacking
!! @author Brandon D. Shroyer
!! @date 8-22-2010
MODULE MpiPacking

   USE TreeDeclarations
   USE GlobalDeclarations
   USE MessageDeclarations
   USE MpiTransmission
   USE Boundary

   IMPLICIT NONE
!   INCLUDE 'mpif.h'
   PRIVATE

   PUBLIC CreatePackedMessage, ClosePackedMessage, DestroyPackedMessage
   PUBLIC CreateMessageGroup, CloseMessageGroup, DestroyMessageGroup, ExtractMessageFromGroup
   PUBLIC MGBlockOnFirstMessages, MGFinishMessageGroupMessages
   PUBLIC GetProcListAsArray, RemoveFirstMessageFromGroup, SendTerminationBox, PackTerminationBox
   PUBLIC StrictGetNextBox
   PUBLIC PACK_INTEGER_SIZE, PACK_FLOAT_SIZE, PACK_DOUBLE_SIZE, PACK_BOX_SIZE

   PUBLIC PackData, UnpackData, PackList, UnpackList
   !    PUBLIC PackInt0D, PackInt1D, PackInt2D, PackInt3D
   !    PUBLIC PackFloat0D, PackFloat1D, PackFloat2D, PackFloat3D
   !    PUBLIC PackDouble0D, PackDouble1D, PackDouble2D, PackDouble3D, PackDouble4D



   !> Generic interface for Packing Data
   INTERFACE PackData
      MODULE PROCEDURE    PackInt0D, PackInt1D, PackInt2D, PackInt3D, PackInt4D, &
           PackFloat0D, PackFloat1D, PackFloat2D, PackFloat3D, PackFloat4D, &
           PackDouble0D, PackDouble1D, PackDouble2D, PackDouble3D, PackDouble4D, &
           PackBox, PackBoundary
   END INTERFACE

   !> Generic interface for UnPacking Data
   INTERFACE UnpackData
      MODULE PROCEDURE    UnpackInt0D, UnpackInt1D, UnpackInt2D, UnpackInt3D, UnpackInt4D, &
           UnpackFloat0D, UnpackFloat1D, UnpackFloat2D, UnpackFloat3D, UnpackFloat4D, &
           UnpackDouble0D, UnpackDouble1D, UnpackDouble2D, UnpackDouble3D, UnpackDouble4D, &
           UnpackBox, UnPackBoundary

   END INTERFACE


   !> Generic interface for Packing Lists
   INTERFACE PackList
      MODULE PROCEDURE PackIntList, PackFloatList, PackDoubleList
   END INTERFACE

   !> Generic interface for UnPacking lists
   INTERFACE UnpackList
      MODULE PROCEDURE UnpackIntList, UnpackFloatList, UnpackDoubleList
   END INTERFACE

   PUBLIC GetNextBox
CONTAINS

   !> @name PACKEDMESSAGE ROUTINES
   !! @{

   !> Creates a PackedMessage object and initializes the MPI communications associated with it.
   !! @param level The level on which the creating process is working.
   !! @param remote_proc The ID of the processor with which this object will interact.
   !! @param tag The MPI message tag this object's MPI message will have.
   !! @param lSend A logical flag; true indicates that this is a send; false indicates a receive.
   !! @param message A pointer for the new message object.
   !! @param message_size An optional integer value indicating the size of the packed message buffer.
   SUBROUTINE CreatePackedMessage(level, remote_proc, tag, lSend, message, message_size)

      INTEGER :: level
      INTEGER :: remote_proc
      INTEGER :: tag
      LOGICAL :: lSend
      TYPE(PackedMessage), POINTER :: message
      INTEGER, OPTIONAL :: message_size

      INTEGER :: msize


      ! If there is no message size passed in, then use the standard buffer size for this message.  This will be the
      ! default option for tree messages, which are unlikely to need massive buffer sizes.
      IF (PRESENT(message_size)) THEN
         msize = message_size
      ELSE
         msize = STD_BUFFER_SIZE
      END IF

      CALL CreatePackedMessageObject(level, remote_proc, tag, lSend, msize, message)

      IF (PRESENT(message_size)) THEN
         message%lMultiBlock=.false.
      ELSE
         message%lMultiBlock=.true.         
      END IF
      

      IF (lSend) THEN
         CALL SetUpSend(message)
      ELSE
         CALL SetUpRecv(message)
      END IF

   END SUBROUTINE CreatePackedMessage

   !> Closes all MPI traffic associated with a message and then deallocates the object.
   !! @param message The message object to be closed out.
   SUBROUTINE DestroyPackedMessage(message)

      TYPE(PackedMessage), POINTER :: message


      IF (.NOT. ASSOCIATED(message))  RETURN

      ! Close out the current message block.
      IF (.NOT. message%closed) THEN
         CALL ClosePackedMessage(message)
      END IF

      ! Finish sending any unsent messages.
      IF (message%nMessages > 0 .AND. message%lSend) THEN
         CALL WaitOnMessageBlocks(message)
      END IF

      CALL DestroyPackedMessageObject(message)

   END SUBROUTINE DestroyPackedMessage


   !> Sends the contents of the current message block and then indicates that the message is closed.
   !! @param message The message object to be closed.
   SUBROUTINE ClosePackedMessage(message)

      TYPE(PackedMessage), POINTER :: message


      message%closed = .TRUE.    

      IF (message%lSend) THEN
          CALL SendPackedMessage(message)
      ELSE         
          IF (message%lMultiBlock) CALL CancelLastMessageBlockRecv(message)
      END IF

   END SUBROUTINE ClosePackedMessage

   !> Returns wheter a message is closed
   !! @param message The message to check
   LOGICAL FUNCTION MessageIsClosed(message)

      TYPE(PackedMessage), POINTER :: message

      MessageIsClosed = message%closed

   END FUNCTION MessageIsClosed

   !> @}


   !> @name MESSAGEGROUP ROUTINES
   !! @{

   !> Initializes all the values in a StageMessageGroup.
   !! @param sm_group The StageMessageGroup object being filled out.
   !! @param iStageTag The tag identifier of the stage this StageMessageGroup is handling.
   !! @param lSend Set to true if this is a sending operation; false if it is a receiving operation.
   !! @param level The level of the current transmissions.
   SUBROUTINE CreateMessageGroup(sm_group, iStageTag, lSend, level)

      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER :: iStageTag
      LOGICAL :: lSend
      INTEGER :: level

      ! Allocate and assemble message group object.
      CALL CreateMessageGroupObject(sm_group, iStageTag, lSend, level)

   END SUBROUTINE CreateMessageGroup

   !> Closes all messages within a given message group.
   !! @param sm_group The StageMessageGroup object whose members are to be closed.
   SUBROUTINE CloseMessageGroup(sm_group)

      TYPE(StageMessageGroup), POINTER :: sm_group

      TYPE(MessageList), POINTER :: msg_list
      TYPE(RequestList), POINTER :: req_list

      msg_list => sm_group%messages
      
      DO WHILE (ASSOCIATED(msg_list))
         IF (.NOT. msg_list%self%closed)  CALL ClosePackedMessage(msg_list%self)

         msg_list => msg_list%next
      END DO

   END SUBROUTINE CloseMessageGroup

   !> Safely dallocates all memory associated with a message group.
   !! @param sm_group The group being destroyed.
   SUBROUTINE DestroyMessageGroup(sm_group)

      TYPE(StageMessageGroup), POINTER :: sm_group


      IF (.NOT. ASSOCIATED(sm_group)) THEN
          print*, sm_group%level
          PRINT *, "DestroyMessageGroup() error: message_group not associated."
          STOP
      END IF

!      IF (.NOT. MessageGroupIsAllClosed(sm_group))  CALL CloseMessageGroup(sm_group)
      CALL CloseMessageGroup(sm_group)
      CALL MGFinishMessageGroupMessages(sm_group)
      CALL DestroyMessageGroupObject(sm_group)

   END SUBROUTINE DestroyMessageGroup

   !> Adds a PackedMessage object to a message group's list.
   !! @param sm_group The message group to receive a new message.
   !! @param message A PackedMessage object to add to the list.
   SUBROUTINE AddMessageToGroupList(sm_group, message)

      TYPE(StageMessageGroup), POINTER :: sm_group
      TYPE(PackedMessage), POINTER :: message

      CALL AddMessageToList(message, sm_group%last_message, sm_group%messages)
!      IF (.NOT. ASSOCIATED(sm_group%messages) .AND. ASSOCIATED(sm_group%last_message))  sm_group%messages => sm_group%last_message
      sm_group%nPackedMessages = sm_group%nPackedMessages + 1
!      PRINT *, "nPackedMessages=", sm_group%nPackedMessages
   END SUBROUTINE AddMessageToGroupList

   !> Takes a processor ID and returns the corresponding message object from the message group.  If the object doesn't exist, it will be created.
   !! @param sm_group The message group from which the object will be retrieved.
   !! @param proc_id The processor ID of the message to be extracted.
   !! @param message A PackedMessage pointer that will be associated by this subroutine.
   !! @param message_size An optional integer that sets the size of the message buffer.  If this is not present, then the standard buffer will be used.
   SUBROUTINE ExtractMessageFromGroup(sm_group, proc_id, message, message_size)

      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER :: proc_id
      TYPE(PackedMessage), POINTER :: message
      INTEGER, OPTIONAL :: message_size

      TYPE(MessageList), POINTER :: msg_list
      INTEGER :: msize


      NULLIFY(message)

      msg_list => sm_group%messages

      IF (proc_id == MPI_id) THEN
          PRINT *, "ExtractMessageFromGroup() error: proc_id ", proc_id, "matched process rank."
          STOP
      END IF

      ! If a message with this processor ID already exists in the list, simply return that message.
      DO WHILE (ASSOCIATED(msg_list))
         IF (msg_list%self%remote_proc == proc_id) THEN
            message => msg_list%self
            EXIT
         END IF
         msg_list => msg_list%next
      END DO

      ! If no message was found with the given processor ID, then create a new PackedMessageObject.
      IF (.NOT. ASSOCIATED(message)) THEN

         IF (PRESENT(message_size)) THEN
             msize = message_size
         ELSE
             msize = STD_BUFFER_SIZE
         END IF

         CALL CreatePackedMessage(sm_group%level, proc_id, sm_group%iStageTag, sm_group%lSend, message, msize)

         IF (PRESENT(message_size)) THEN
            message%lMultiBlock=.false.
         ELSE
            message%lMultiBlock=.true.         
         END IF

         CALL AddMessageToGroupList(sm_group, message)
         CALL AddProcessorToMGList(sm_group, proc_id)
      END IF

   END SUBROUTINE ExtractMessageFromGroup

   !> Closes and deletes the first message associated with a message group.
   !! @param sm_group The StageMessageGroup object being targeted.
   SUBROUTINE RemoveFirstMessageFromGroup(sm_group)

      TYPE(StageMessageGroup), POINTER :: sm_group

      TYPE(MessageList), POINTER :: dead_list


      dead_list => sm_group%messages
      sm_group%messages => sm_group%messages%next
      NULLIFY(dead_list%next)
      CALL DestroyMessageListObject(dead_list)            ! Will close and destroy the PackedMessage object as well.
      sm_group%nPackedMessages = sm_group%nPackedMessages - 1

   END SUBROUTINE RemoveFirstMessageFromGroup


   !> Returns true if all the packed messages in a given message group are closed.
   !! @param sm_group The message group being tested.
   LOGICAL FUNCTION MessageGroupIsAllClosed(sm_group)

      TYPE(StageMessageGroup) :: sm_group

      TYPE(MessageList), POINTER :: msg_list


      MessageGroupIsAllClosed = .TRUE.

      msg_list => sm_group%messages

      ! Loop over messages in group to see if they are closed.  Return if any of them aren't.
      DO WHILE (ASSOCIATED(msg_list))
         IF (.NOT. msg_list%self%closed) THEN
            MessageGroupIsAllClosed = .FALSE.
            EXIT
         END IF
         msg_list => msg_list%next
      END DO

   END FUNCTION MessageGroupIsAllClosed

   !> Waits until at least one message from the specified group has received its first message block.  Returns a reference to that message.
   !! @param sm_group The message group being queried for a new message.
   !! @param message A PackedMessage object pointer that has received a message.  If this returns null, then all messages have received at least one block.
   SUBROUTINE MGBlockOnFirstMessages(sm_group, message)

      TYPE(StageMessageGroup) :: sm_group
      TYPE(PackedMessage), POINTER :: message


      CALL WaitOnAnyMessageGroupRecv(sm_group, message, I_FIRST_REQUESTS)

   END SUBROUTINE MGBlockOnFirstMessages

   !> Blocks on all message requests until they complete.  Note that it blocks on sends and recvs differently.
   !! @param sm_group The message group whose requests are being completed.
   SUBROUTINE MGFinishMessageGroupMessages(sm_group)

      TYPE(StageMessageGroup), POINTER :: sm_group

      TYPE(PackedMessage), POINTER :: message
      LOGICAL :: lRepeat

      NULLIFY(message)
      lRepeat = .TRUE.

       IF (sm_group%lSend)  CALL WaitOnAllMessageGroupSends(sm_group)

   END SUBROUTINE MGFinishMessageGroupMessages

   !> Retrieves a StageMessageGroup's processor list as an array.  This subroutine assumes that the processor list has no duplicate entries.
   !! @param sm_group The message group whose list is being retrieved.
   !! @param proc_array A 1D integer array pointer that will be associated with the processor array.
   SUBROUTINE GetProcListAsArray(sm_group, proc_array)

      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER, POINTER, DIMENSION(:) :: proc_array

      INTEGER :: proc_count
      TYPE(ProcessorList), POINTER :: proc_list
      INTEGER :: iErr
      INTEGER :: m


      IF (ASSOCIATED(proc_array)) THEN
         DEALLOCATE(proc_array)
         NULLIFY(proc_array)
      END IF

      proc_count = 0

      proc_list => sm_group%proclist

      ! Establish the size of the processor array.  We have to do this because we can't 
      ! guarantee that there will only be one message for each processor in a message group.
      DO WHILE (ASSOCIATED(proc_list))
         proc_count = proc_count + 1
         proc_list => proc_list%next
      END DO

      IF (proc_count > 0) THEN
         ALLOCATE(proc_array(proc_count), STAT=iErr)

         IF (iErr /= 0) THEN
            PRINT *, "GetProcListAsArray() error: unable to allocate processor array."
            STOP
         END IF

         proc_list => sm_group%proclist
         m = 1

         ! Populate the outgoing processor array.
         DO WHILE (ASSOCIATED(proc_list))
            proc_array(m) = proc_list%self
            proc_list => proc_list%next
            m=m+1
         END DO

      END IF

   END SUBROUTINE GetProcListAsArray


   !> Packs a termination box into a packed message array.  Usually this occurs at the end of the message.
   !! @param message A packed message object to be ende
   SUBROUTINE PackTerminationBox(message)
      TYPE(PackedMessage), POINTER :: message

      CALL PackData(message, TERMINATIONBOX)

   END SUBROUTINE PackTerminationBox


   !> Sends a termination box to each message of a message group
   !! @param MessageGroup Message group to pack termination boxes to
   SUBROUTINE SendTerminationBox(MessageGroup)
      TYPE(StageMessageGroup), POINTER :: MessageGroup
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:), POINTER :: ProcList
      INTEGER :: i
      INTEGER :: list_size
      NULLIFY(ProcList)
      CALL GetProcListAsArray(MessageGroup, ProcList)
      IF (ASSOCIATED(ProcList)) THEN
         list_size = SIZE(ProcList)
      ELSE
         list_size = 0
      END IF

      DO i=1, list_size
         CALL ExtractMessageFromGroup(MessageGroup, ProcList(i), message)

         CALL PackData(message, TERMINATIONBOX)
      END DO

      IF (ASSOCIATED(ProcList)) THEN
          DEALLOCATE(ProcList)
          NULLIFY(ProcList)
      END IF

   END SUBROUTINE SendTerminationBox


   !> @}


   !> @name PACKING ROUTINES
   !! @{

   !> Packs an integer scalar into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_data a single integer.
   SUBROUTINE PackInt0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: type_data
      INTEGER, DIMENSION(1) :: array_data
      array_data(1)=type_data
      CALL PackInt1D(message, array_data)
   END SUBROUTINE PackInt0D

   !> Packs a 1D integer array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 1D integer array.
   SUBROUTINE PackInt1D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block

      array_size = SIZE(type_array)

      remaining_data = array_size               ! Will keep track of unpacked data in this array.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         ! If the current block is full, then send the current message.
         IF (message%last_block%block_offset + PACK_INTEGER_SIZE > message%buffer_size) THEN !We can't pack any more integers
            CALL SendPackedMessage(message)
         END IF

         ! Calculate the remaining block space (in integers, not bytes).
         remaining_block_data =  FLOOR((message%buffer_size -  message%last_block%block_offset) * 1.0/PACK_INTEGER_SIZE)

         ! Determine the size of the block that will be packed in this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Pack the array data into the current message block.
         CALL MPI_PACK(type_array(array_position), &
                       next_block, &
                       MPI_INTEGER, &
                       message%last_block%buffer, &
                       message%buffer_size, &
                       message%last_block%block_offset, &
                       MPI_COMM_WORLD, &
                       iErr)

         ! Increment the array counters by bytes or integers.
         array_position = array_position + next_block
        ! Decrement the remaining data count (in integers).                   
         remaining_data = remaining_data - next_block
      END DO

   END SUBROUTINE PackInt1D

   !> Packs a 2D integer array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 2D integer array.
   SUBROUTINE PackInt2D(message, type_array) 

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:) :: type_array

      CALL PackInt1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))

   END SUBROUTINE PackInt2D

   !> Packs a 3D integer array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 3D integer array.
   SUBROUTINE PackInt3D(message, type_array) 

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:,:) :: type_array


      CALL PackInt1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))

   END SUBROUTINE PackInt3D

   !> Packs a 4D integer array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 3D integer array.
   SUBROUTINE PackInt4D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:,:,:) :: type_array


      CALL PackInt1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))

   END SUBROUTINE PackInt4D


   !> Packs an floating-point scalar into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_data a single floating-point.
   SUBROUTINE PackFloat0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      REAL :: type_data
      REAL, DIMENSION(1) :: array_data
      array_data(1)=type_data
      CALL PackFloat1D(message, array_data)
   END SUBROUTINE PackFloat0D

   !> Packs a 1D floating-point array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 1D floating-point array.
   SUBROUTINE PackFloat1D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block

      array_size = SIZE(type_array)

      remaining_data = array_size               ! Will keep track of unpacked data in this array.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         ! If the current block is full, then send the current message.
         IF (message%last_block%block_offset + PACK_FLOAT_SIZE > message%buffer_size) THEN !We can't pack any more integers
            CALL SendPackedMessage(message)
         END IF

         ! Calculate the remaining block space (in integers, not bytes).
         remaining_block_data =  FLOOR((message%buffer_size -  message%last_block%block_offset) * 1.0/PACK_FLOAT_SIZE)

         ! Determine the size of the block that will be packed in this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Pack the array data into the current message block.
         CALL MPI_PACK(type_array(array_position), &
                       next_block, &
                       MPI_REAL, &
                       message%last_block%buffer, &
                       message%buffer_size, &
                       message%last_block%block_offset, &
                       MPI_COMM_WORLD, &
                       iErr)

        ! Decrement the remaining data count (in integers).                   
         remaining_data = remaining_data - next_block
         array_position = array_position + next_block
      END DO
   END SUBROUTINE PackFloat1D

   !> Packs a 2D floating-point array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 2D floating-point array.
   SUBROUTINE PackFloat2D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:) :: type_array
      CALL PackFloat1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackFloat2D

   !> Packs a 3D floating-point array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 3D floating-point array.
   SUBROUTINE PackFloat3D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:,:) :: type_array
      CALL PackFloat1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackFloat3D

   !> Packs a 3D floating-point array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 3D floating-point array.
   SUBROUTINE PackFloat4D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:,:,:) :: type_array
      CALL PackFloat1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackFloat4D

   !> Packs an double-precision scalar into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_data a single double-precision.
   SUBROUTINE PackDouble0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPREC) :: type_data
      REAL(KIND=qPREC), DIMENSION(1) :: array_data
      array_data(1)=type_data
      CALL PackDouble1D(message, array_data)
   END SUBROUTINE PackDouble0D

   !> Packs a 1D double-precision array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 1D double-precision array.
   SUBROUTINE PackDouble1D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPREC), DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block

      array_size = SIZE(type_array)

      remaining_data = array_size               ! Will keep track of unpacked data in this array.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         ! If the current block is full, then send the current message.
         IF (message%last_block%block_offset + PACK_DOUBLE_SIZE > message%buffer_size) THEN !We can't pack any more integers
            CALL SendPackedMessage(message)
         END IF

         ! Calculate the remaining block space (in integers, not bytes).
         remaining_block_data =  FLOOR((message%buffer_size -  message%last_block%block_offset) * 1.0/PACK_DOUBLE_SIZE)

         ! Determine the size of the block that will be packed in this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Pack the array data into the current message block.
         CALL MPI_PACK(type_array(array_position), &
                       next_block, &
                       MPI_DOUBLE_PRECISION, &
                       message%last_block%buffer, &
                       message%buffer_size, &
                       message%last_block%block_offset, &
                       MPI_COMM_WORLD, &
                       iErr)

        ! Decrement the remaining data count (in integers).                   
         remaining_data = remaining_data - next_block
         array_position = array_position + next_block
      END DO

   END SUBROUTINE PackDouble1D

   !> Packs a 2D double-precision array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 2D double-precision array.
   SUBROUTINE PackDouble2D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:) :: type_array
      CALL PackDouble1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackDouble2D

   !> Packs a 3D double-precision array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 3D double-precision array.
   SUBROUTINE PackDouble3D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:,:) :: type_array
      CALL PackDouble1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackDouble3D

   !> Packs a 4D double-precision array into the given packed message.
   !! @param message The packed message object to receive the data.
   !! @param type_array a 4D double-precision array.
   SUBROUTINE PackDouble4D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:,:,:) :: type_array
      CALL PackDouble1D(message, RESHAPE(type_array, (/ SIZE(type_array) /)))
   END SUBROUTINE PackDouble4D


   !> Unpacks a Boundaries Type from the specified message using several lower-level unpacking routines.
   !! @param message A PackedMessage object.
   !! @param boundary A Boundaries object
   SUBROUTINE PackBoundary(message, boundary)
      TYPE(PackedMessage), POINTER :: message
      TYPE(Boundaries) :: boundary
      INTEGER :: i
      DO i=1,nDim         
        CALL PackDouble4D(message, boundary%side(i)%data)
     END DO
   END SUBROUTINE PackBoundary

   !> Packs a NodeBox into the specified message using several lower-level packing routines.
   !! @param message A PackedMessage object.
   !! @param box A NodeBox object.
   SUBROUTINE PackBox(message, box)
      TYPE(NodeBox) :: box
      TYPE(PackedMessage), POINTER :: message
      CALL PackInt2D(message,box%mGlobal)
      CALL PackInt0D(message,box%MPI_ID)
   END SUBROUTINE PackBox

   !> @}

   
   !> @name UNPACKING ROUTINES 
   !! @{

   !> Extracts an integer scalar from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_data The scalar being filled by the packed message.
   SUBROUTINE UnpackInt0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: type_data
      INTEGER, DIMENSION(1) :: array_data
      CALL UnPackInt1D(message, array_data)
      type_data=array_data(1)
   END SUBROUTINE UnpackInt0D

   !> Extracts a 1D integer array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackInt1D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block


      array_size = SIZE(type_array)

      remaining_data = array_size                       ! Indicates how much data remains to be unpacked.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         IF (message%current_block%block_offset + PACK_INTEGER_SIZE > message%current_block%buffer_size) THEN
            CALL ReceivePackedMessage(message)
         END IF

         ! Calculate the number of integers remaining in the current message block.
         remaining_block_data =  FLOOR((message%current_block%buffer_size - message%current_block%block_offset) * 1.0 / PACK_INTEGER_SIZE)

         ! Determine the size of the block to be read in during this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Extract the data from the message buffer.
         CALL MPI_UNPACK(message%current_block%buffer, &
              message%current_block%buffer_size, &
              message%current_block%block_offset, &
              type_array(array_position), &
              next_block, &
              MPI_INTEGER, &
              MPI_COMM_WORLD, &
              iErr)

         ! Increment the array counters by bytes or integers.
         array_position = array_position + next_block
         ! Decrement the remaining data counters by bytes or integers.
         remaining_data = remaining_data - next_block
      END DO

   END SUBROUTINE UnpackInt1D

   !> Extracts a 2D integer array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackInt2D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:) :: type_array
      INTEGER, ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackInt1D(message, data_array)
      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array = RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)
   END SUBROUTINE UnpackInt2D

   !> Extracts a 3D integer array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackInt3D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:,:) :: type_array
      INTEGER, ALLOCATABLE, DIMENSION(:) :: data_array

      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackInt1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)
   END SUBROUTINE UnpackInt3D

   !> Extracts a 4D integer array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackInt4D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:,:,:,:) :: type_array
      INTEGER, ALLOCATABLE, DIMENSION(:) :: data_array

      ALLOCATE(data_array(SIZE(type_array)))  ! This is where the unpacked data will be stored.
      CALL UnpackInt1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)

   END SUBROUTINE UnpackInt4D

   !> Extracts an floating-point scalar from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_data The scalar being filled by the packed message.
   SUBROUTINE UnpackFloat0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      REAL :: type_data
      REAL, DIMENSION(1) :: array_data
      CALL UnPackFloat1D(message, array_data)
      type_data=array_data(1)
   END SUBROUTINE UnpackFloat0D


   !> Extracts a 1D floating-point array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackFloat1D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block


      array_size = SIZE(type_array)

      remaining_data = array_size                       ! Indicates how much data remains to be unpacked.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         IF (message%current_block%block_offset + PACK_FLOAT_SIZE > message%current_block%buffer_size) THEN
            CALL ReceivePackedMessage(message)
         END IF

         ! Calculate the number of integers remaining in the current message block.
         remaining_block_data =  FLOOR((message%current_block%buffer_size - message%current_block%block_offset) * 1.0 / PACK_FLOAT_SIZE)

         ! Determine the size of the block to be read in during this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Extract the data from the message buffer.
         CALL MPI_UNPACK(message%current_block%buffer, &
              message%current_block%buffer_size, &
              message%current_block%block_offset, &
              type_array(array_position), &
              next_block, &
              MPI_REAL, &
              MPI_COMM_WORLD, &
              iErr)

         ! Increment the array counters by bytes or integers.
         array_position = array_position + next_block
         ! Decrement the remaining data counters by bytes or integers.
         remaining_data = remaining_data - next_block
      END DO

   END SUBROUTINE UnpackFloat1D

   !> Extracts a 2D floating-point array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackFloat2D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:) :: type_array
      REAL, ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackFloat1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))

      DEALLOCATE(data_array)

   END SUBROUTINE UnpackFloat2D

   !> Extracts a 3D floating-point array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackFloat3D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:,:) :: type_array
      REAL, ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackFloat1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))

      DEALLOCATE(data_array)

   END SUBROUTINE UnpackFloat3D

   !> Extracts a 4D floating-point array from a packed message.  Note
   !! that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackFloat4D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:,:,:,:) :: type_array

      REAL, ALLOCATABLE, DIMENSION(:) :: data_array

      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.

      CALL UnpackFloat1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)
   END SUBROUTINE UnpackFloat4D

   !> Extracts an double-precision scalar from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_data The scalar being filled by the packed message.
   SUBROUTINE UnpackDouble0D(message, type_data)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPREC) :: type_data
      REAL(KIND=qPREC), DIMENSION(1) :: array_data
      CALL UnPackDouble1D(message, array_data)
      type_data=array_data(1)
   END SUBROUTINE UnpackDouble0D

   !> Extracts a 1D double-precision array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackDouble1D(message, type_array)

      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPREC), DIMENSION(:) :: type_array

      INTEGER :: array_size, array_position, iErr
      INTEGER :: remaining_data, remaining_block_data, next_block


      array_size = SIZE(type_array)

      remaining_data = array_size                       ! Indicates how much data remains to be unpacked.
      array_position= lBound(type_array,1)

      DO WHILE (remaining_data > 0)

         IF (message%current_block%block_offset + PACK_DOUBLE_SIZE > message%current_block%buffer_size) THEN
            CALL ReceivePackedMessage(message)
         END IF

         ! Calculate the number of integers remaining in the current message block.
         remaining_block_data =  FLOOR((message%current_block%buffer_size - message%current_block%block_offset) * 1.0 / PACK_DOUBLE_SIZE)

         ! Determine the size of the block to be read in during this pass.
         next_block = MIN(remaining_data, remaining_block_data)

         ! Extract the data from the message buffer.
         CALL MPI_UNPACK(message%current_block%buffer, &
              message%current_block%buffer_size, &
              message%current_block%block_offset, &
              type_array(array_position), &
              next_block, &
              MPI_DOUBLE_PRECISION, &
              MPI_COMM_WORLD, &
              iErr)

         ! Increment the array counters by bytes or integers.
         array_position = array_position + next_block
         ! Decrement the remaining data counters by bytes or integers.
         remaining_data = remaining_data - next_block
      END DO

   END SUBROUTINE UnpackDouble1D

   !> Extracts a 2D double-precision array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackDouble2D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:) :: type_array
      REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackDouble1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)

   END SUBROUTINE UnpackDouble2D

   !> Extracts a 3D double-precision array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackDouble3D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:,:) :: type_array
      REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackDouble1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)

   END SUBROUTINE UnpackDouble3D

   !> Extracts a 4D double-precision array from a packed message.  Note that the input pointer must be associated.
   !! @param message The packed message from which the data will be extracted.
   !! @param type_array The array being filled by the packed message.
   SUBROUTINE UnpackDouble4D(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:,:,:,:) :: type_array
      REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:) :: data_array
      ALLOCATE(data_array(SIZE(type_array)))    ! This is where the unpacked data will be stored.
      CALL UnpackDouble1D(message, data_array)

      ! Reshape the data to fit into type_array.  Note that this relies on RESHAPE being a symmetric transformation;
      ! if it's not then we'll have to do this the hard way (cell by cell).
      type_array=RESHAPE(data_array, SHAPE(type_array))
      DEALLOCATE(data_array)
   END SUBROUTINE UnpackDouble4D

   !> Unpacks a Boundaries Type from the specified message using several lower-level unpacking routines.
   !! @param message A PackedMessage object.
   !! @param boundary A Boundaries object
   SUBROUTINE UnpackBoundary(message, boundary)
      TYPE(PackedMessage), POINTER :: message
      TYPE(Boundaries) :: boundary
      INTEGER :: i

      DO i=1,nDim
        CALL UnPackDouble4D(message, boundary%side(i)%data)
     END DO

   END SUBROUTINE UnpackBoundary

   !> Unpacks a NodeBox into the specified message using several lower-level unpacking routines.
   !! @param message A PackedMessage object.
   !! @param box A NodeBox object.
   SUBROUTINE UnpackBox(message,box)

      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeBox) :: box

      CALL UnpackInt2D(message,box%mGlobal)
      CALL UnpackInt0D(message,box%MPI_ID)

   END SUBROUTINE UnpackBox


   LOGICAL FUNCTION GetNextBox(message,mGlobal)
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(3,2) :: mGlobal

!      LOGICAL GetNextBox


      mGlobal = TERMINATIONBOX
      CALL UnPackData(message, mGlobal)
      GetNextBox = (.NOT. ALL(mGlobal == TERMINATIONBOX))
   END FUNCTION GetNextBox


   LOGICAL FUNCTION StrictGetNextBox(message, mGlobal, caller)

      INTEGER, DIMENSION(3,2) :: mGlobal
      TYPE(PackedMessage), POINTER :: message
      CHARACTER(*) :: caller
      LOGICAL :: gnb_result

      gnb_result = GetNextBox(message, mGlobal)

      IF (gnb_result .AND. ANY(mGlobal < 1)) THEN
         PRINT *, caller, "::StrictGetNextBox() error: bad box [", mGlobal, "]."
         PRINT *, "   level = ", message%level, ", remote_proc = ", message%remote_proc
         STOP
      END IF

      StrictGetNextBox = gnb_result

   END FUNCTION StrictGetNextBox


   !> @}


   !> @name SIZED LIST PACKING ROUTINES
   !! @{

   !> Packs an integer list of arbitrary size into the given message.  Unlike PackInt1D, PackIntList includes the size of the array.
   !! @param message A PackedMessage object.
   !! @param array A 1D integer array.
   SUBROUTINE PackIntList(message, array)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:) :: array


      CALL packint0d(message,size(array))
      CALL packint1D(message,array)
   END SUBROUTINE PackIntList


   !> Packs a floating-point list of arbitrary size into the given message.  Unlike PackFloat1D, PackIntList includes the size of the array.
   !! @param message A PackedMessage object.
   !! @param array A 1D floating-point array.
   SUBROUTINE PackFloatList(message, array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:) :: array

      CALL packint0d(message,size(array))
      CALL packfloat1D(message,array)

   END SUBROUTINE PackFloatList

   !> Packs a double-precision list of arbitrary size into the given message.  Unlike PackDouble1D, PackDoubleList includes the size of the array.
   !! @param message A PackedMessage object.
   !! @param array A 1D double-precision array.
   SUBROUTINE PackDoubleList(message, array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:) :: array

      CALL PackInt0D(message,size(array))
      CALL PackDouble1D(message,array)

   END SUBROUTINE PackDoubleList


   !> @}
   
   !> @name SIZED LIST UNPACKING ROUTINES
   !! @{

   !> Unpacks an integer list, using the list size given in the buffer.  Assumes that the list size precedes the list in the buffer.
   !! @param message A PackedMessage object.
   !! @param type_array A pointer for a 1D integer array.
   SUBROUTINE UnpackIntList(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(:), POINTER :: type_array

      INTEGER :: n


      CALL UnpackInt0D(message,n)

!PRINT *, "UnpackList::n = ", n
      ALLOCATE(type_array(n))

!PRINT *, "UnpackList::old type_array = ", type_array

      CALL UnpackInt1D(message,type_array)

!PRINT *, "UnpackList::new type_array = ", type_array

   END SUBROUTINE UnpackIntList

   !> Unpacks an floating-point list, using the list size given in the buffer.  Assumes that the list size precedes the list in the buffer.
   !! @param message A PackedMessage object.
   !! @param type_array A pointer for a 1D floating-point array.
   SUBROUTINE UnpackFloatList(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL, DIMENSION(:), POINTER :: type_array

      INTEGER :: n


      CALL UnpackInt0D(message,n)
      ALLOCATE(type_array(n))
      CALL UnpackFloat1D(message,type_array)

   END SUBROUTINE UnpackFloatList

   !> Unpacks an double-precision list, using the list size given in the buffer.  Assumes that the list size precedes the list in the buffer.
   !! @param message A PackedMessage object.
   !! @param type_array A pointer for a 1D double-precision array.
   SUBROUTINE UnpackDoubleList(message, type_array)
      TYPE(PackedMessage), POINTER :: message
      REAL(KIND=qPrec), DIMENSION(:), POINTER :: type_array

      INTEGER :: n


      CALL UnpackInt0D(message,n)
      ALLOCATE(type_array(n))
      CALL UnpackDouble1D(message,type_array)

   END SUBROUTINE UnpackDoubleList
   !> @}

END MODULE MpiPacking

