!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    mpi_transmission.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file mpi_transmission.f90
!! @brief Main file for module MpiTransmission

!> @defgroup MpiTransmission MpiTransmission
!! @ingroup Communication
!! @brief Manages the MPI sends and receives of packed messages.

!> Manages the MPI sends and receives of packed messages.
!! @author Brandon D. Shroyer
!! @date 8-22-2010
!! @ingroup MpiTransmission
MODULE MpiTransmission

   USE MessageDeclarations
   USE GlobalDeclarations
#if defined PTH
   USE PthDeclarations
#endif
   USE Timing
   IMPLICIT NONE
   PRIVATE

   PUBLIC SetUpSend, SetUpRecv
   PUBLIC SendPackedMessage, ReceivePackedMessage, CancelLastMessageBlockRecv
   PUBLIC WaitOnMessageBlocks, WaitOnAnyMessageGroupRecv, WaitOnAllMessageGroupSends
   PUBLIC GetMGAllRequestsArray
CONTAINS

   !> Sets up the message to be sent.  For normal MPI, this means just adding a message block.
   !! @param message A packed message to be sent.
   SUBROUTINE SetUpSend(message)

      TYPE(PackedMessage), POINTER :: message

      ! Verify that all required objects are allocated.
      IF (.NOT. ASSOCIATED(message)) THEN
         PRINT *, "SetUpSend() error: message not associated."
         STOP
      END IF

      CALL AddMessageBlock(message, .NOT. ASSOCIATED(message%last_block), message%buffer_size)

   END SUBROUTINE SetUpSend

   !> Sends the current block of the specified message and creates a new block if necessary.
   !! @param message The packed message to be sent.
   SUBROUTINE SendPackedMessage(message)

      TYPE(PackedMessage), POINTER :: message

      INTEGER :: mpi_request
      INTEGER :: iErr
      INTEGER :: send_buffer_size


      ! Verify that all required objects are allocated.
      IF (.NOT. ASSOCIATED(message)) THEN
         PRINT *, "SendPackedMessage() error: message not associated."
         STOP

      ELSE IF (.NOT. ASSOCIATED(message%last_block)) THEN
         PRINT *, "SendPackedMessage() error: no message block allocated."
         STOP

      END IF

      ! Allocate the send buffer as a 1:offset array.
      send_buffer_size = message%last_block%block_offset
      message%last_block%buffer_size = send_buffer_size

      ! Call a non-blocking send.  The unique tags have been removed from this scheme so that the
      ! buffers will be received in the order they are sent.     
      
      CALL MPI_ISEND(message%last_block%buffer, &
           send_buffer_size, &
           MPI_PACKED, &
           message%remote_proc, &
           !message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, &
           (message%tag*(MaxDepth+3) + (message%level + 2)) * MAX_MESSAGE_PIECES + message%nMessages, &
           MPI_COMM_WORLD, &
           mpi_request, &
           iErr)

!      CALL MPI_SSEND(message%last_block%buffer, &
!           send_buffer_size, &
!           MPI_PACKED, &
!           message%remote_proc, &
!           message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, &
!           MPI_COMM_WORLD, &
!           iErr)
!      mpi_request = 0

      ! Store the MPI request handle in the message object.
      CALL AddMpiRequest(message, mpi_request, .NOT. ASSOCIATED(message%last_request))

      ! If the message is closed, then allocate the next message block.  This assumes that it
      ! is exceedingly unlikely that a message will be exactly one block long (i.e., that the
      ! odds of sending an additional block with nothing in it are negligible).
      IF (.NOT. message%closed)  CALL SetUpSend(message)
!      write(*,*) 'done'
   END SUBROUTINE SendPackedMessage

   !> Wait until all blocks associated with the packed message are sent.
   !! @param message The packed message object whose sends are being completed.
   SUBROUTINE WaitOnMessageBlocks(message)
     USE GlobalDeclarations
      TYPE(PackedMessage), POINTER :: message

      INTEGER, POINTER, DIMENSION(:) :: req_array
      INTEGER :: iErr


      IF (.NOT. ASSOCIATED(message)) THEN
         PRINT *, "WaitOnMessageBlocks() error: no message associated."
         STOP
      END IF

      ! Retrieve the request list in array form.
      NULLIFY(req_array)
      CALL GetRequestArray(message, req_array, .TRUE., .TRUE.)

!PRINT *
!PRINT *, "req_array = ", req_array
!PRINT *, "SIZE(req_array) = ", SIZE(req_array)
!PRINT *, "message level = ", message%level
!PRINT *, "MPI_WAITALL: req_array size = ", SIZE(req_array)
!PRINT *, "MPI_WAITALL: sm_group%StageTag = ", message%tag
!PRINT *, "MPI_WAITALL: modified tag = ", message%tag + (message%level + 2) * 100
!PRINT *, "message%nMessages = ", message%nMessages
!PRINT *

      ! Complete all outstanding waits.
# if defined PTH
      IF (iThreaded == THREADED) THEN
         CALL pth_MPI_WAITALL(message%nMessages, req_array, iErr)
      ELSE
         CALL StartTimer(iBarrier, message%level)
         CALL MPI_WAITALL(message%nMessages, req_array, MPI_STATUSES_IGNORE, iErr)
         CALL StopTimer(iBarrier, message%level)
      END IF
# else
      CALL StartTimer(iBarrier, message%level)
      CALL MPI_WAITALL(message%nMessages, req_array, MPI_STATUSES_IGNORE, iErr)
      CALL StopTimer(iBarrier, message%level)

# endif

      IF (ASSOCIATED(req_array)) THEN
          DEALLOCATE(req_array)
          NULLIFY(req_array)
      END IF

   END SUBROUTINE WaitOnMessageBlocks

   !> Sets up an MPI receive for the specified message.  For delta-MPI, this means creating a new message blcok and calling a delta-recv.
   !! @param message The packed message object intended to receive data.
   SUBROUTINE SetUpRecv(message)

     TYPE(PackedMessage), POINTER :: message

     INTEGER :: iErr
     INTEGER :: mpi_request


     IF (.NOT. ASSOCIATED(message)) THEN
        PRINT *, "SetUpRecv() error: no message object associated with pointer."
        STOP
     END IF

     ! Create a new message block.
     CALL AddMessageBlock(message, .NOT. ASSOCIATED(message%last_block), message%buffer_size)

     !      PRINT *, "Recv ASSOCIATED(message) = ", ASSOCIATED(message)
     !      PRINT *, "Recv ASSOCIATED(buffer) = ", ASSOCIATED(message%last_block)
     !      PRINT *, "Recv remote_proc = ", message%remote_proc
     !      PRINT *, "Recv nMessages = ", message%nMessages
     !      PRINT *, "Recv message tag = ", message%tag
     !      PRINT *, "Recv message level = ", message%level
     !      PRINT *, "Recv modified tag = ", message%tag + (message%level + 2) * 100 + message%nMessages - 1

     ! Call a non-blocking receive on the current message buffer.
     CALL MPI_IRECV(message%last_block%buffer, &
          message%buffer_size, &
          MPI_PACKED, &
          message%remote_proc, &
          !message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, &
          (message%tag*(MAXDEPTH+3) + (message%level + 2)) * MAX_MESSAGE_PIECES + message%nMessages, &
          MPI_COMM_WORLD, &
          mpi_request, &
          ierr)

     CALL AddMpiRequest(message, mpi_request, .NOT. ASSOCIATED(message%last_request))

     !      write(COMM_LOG_HANDLE,*) 'IReceiving', MPI_WTIME()-InitTime, 100*message%level+(message%nMessages-1)+message%tag/1000, message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, message%remote_proc, message%last_request%request


     !IF ((message%tag == 7000) .AND. (message%level == 1) .AND. (MPI_id == 0)) THEN
     !PRINT "('SetUpRecv::request ', i4, ' added by proc ', i1, '.')", message%last_request%request, message%remote_proc
     !END IF
   END SUBROUTINE SetUpRecv

   !> Receives a packed message
   !! @param message The message to receive
   SUBROUTINE ReceivePackedMessage(message)

      TYPE(PackedMessage), POINTER :: message

      INTEGER :: last_message
      INTEGER :: iErr
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: mpi_status
      INTEGER :: message_size


      IF (message%current_request%completed) THEN
         IF (ASSOCIATED(message%current_request%next)) THEN
            message%current_request => message%current_request%next
            message%current_block => message%current_block%next
         ELSE
            write(*,*) 'Error - trying to unpack additional message blocks without posting receives'
            write(*,*) 'If message is not a multi-block message - this could be due to a pre-calculation error'
            write(*,*) 'message%lMultiBlock=', message%lMultiBlock, message%tag, message%level, MPI_ID
            STOP
         END IF
      END IF
# if defined PTH
      IF (iThreaded == THREADED) THEN
         CALL pth_MPI_WAIT(message%current_request%request, mpi_status, iErr)
      ELSE
         CALL StartTimer(iBarrier, message%level)
         CALL MPI_WAIT(message%current_request%request, mpi_status, iErr)
         CALL StopTimer(iBarrier, message%level)
      END IF
# else

      CALL StartTimer(iBarrier, message%level)
      CALL MPI_WAIT(message%current_request%request, mpi_status, iErr)
      CALL StopTimer(iBarrier, message%level)
# endif
      message%current_request%completed = .TRUE.

      ! Get the message's size in bytes and cache it in the message's buffer size.
      CALL MPI_GET_COUNT(mpi_status, MPI_PACKED, message_size, iErr)

      message%current_block%buffer_size = message_size

      IF (message%lMultiBlock) THEN
         ! Post the next receive; if it turns out to be unnecessary it will be
         ! cancelled when ClosePackedMessage() is called.
         CALL SetUpRecv(message)
      END IF
   END SUBROUTINE ReceivePackedMessage

   !> Cancels last message block
   !! @param message The message to cancel the last block for
   SUBROUTINE CancelLastMessageBlockRecv(message)

       TYPE(PackedMessage), POINTER :: message

       INTEGER :: iErr
       INTEGER, DIMENSION(MPI_STATUS_SIZE) :: mpi_status

       ! A new receive is always posted upon completion of the old one; this call to MPI_CANCEL()
       ! eliminates the final receive.
!       write(*,*) MPI_ID, message%last_request%request
       CALL MPI_CANCEL(message%last_request%request, iErr)

       CALL StartTimer(iBarrier, message%level)
       CALL MPI_WAIT(message%last_request%request, mpi_status, iErr)
       CALL StopTimer(iBarrier, message%level)
       ! A new, extraneous message block will have been posted for the new receive; these lines do not
       ! delete the block, but it does clear out any statistics that might otherwise be misleading.
       message%last_request%request = 0
       message%last_block%buffer_size = 0

       message%nMessages = message%nMessages - 1
        
   END SUBROUTINE CancelLastMessageBlockRecv

   !> Waits until at least one message from the specified group has received a message block.  Returns a reference to that message.
   !! @param sm_group The message group being queried for a new message.
   !! @param message A PackedMessage object pointer that has received a message.  If this returns null, then all messages have received at least one block.
   !! @param request_type (first_request, current_request, last_request)
   SUBROUTINE WaitOnAnyMessageGroupRecv(sm_group, message, request_type)

      TYPE(StageMessageGroup) :: sm_group
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: request_type

      INTEGER :: i
      INTEGER :: message_count
      INTEGER, POINTER, DIMENSION(:) :: req_array
      INTEGER :: iErr
      INTEGER :: req_index
      TYPE(MessageList), POINTER :: msg_list
      INTEGER, DIMENSION(MPI_STATUS_SIZE) :: mpi_status
      CHARACTER, DIMENSION(MPI_MAX_ERROR_STRING) :: err_string
      INTEGER :: iErrClass
      INTEGER :: message_size



      req_index = 0
      mpi_status = 0

      NULLIFY(message)

      message_count = sm_group%nPackedMessages
      
      IF (message_count == 0) RETURN
      ! Setting the lOpenOnly flag to true ensures that only open requests will be returned;
      ! otherwise, a completed request would make the MPI_WAITANY call immediately return true 
      ! even if open requests remained.
      CALL GetMGFirstRequestArray(sm_group, req_array, .TRUE.)

      ! If there are any open requests, then wait until one completes and then return a reference
      ! to that request's message.
      IF (ASSOCIATED(req_array)) THEN
!PRINT "('WaitOnAnyMessageGroupRecv(MPI_id = ', i1, '; level = ', i2, ')::req_array_size = ', i6, '.')", MPI_id, sm_group%level, SIZE(req_array)
!PRINT *, "WaitOnAnyMsgGrpRecv(", MPI_id, ",", sm_group%level, ") req_array = ", req_array
!PRINT *, "WaitOnAnyMsgGrpRecv(", MPI_id, ",", sm_group%level, ") req_index = ", req_index
# if defined PTH
         IF (iThreaded == THREADED) THEN
            CALL pth_MPI_WAITANY(SIZE(req_array), req_array, req_index, mpi_status, iErr)
         ELSE
            CALL StartTimer(iBarrier, sm_group%level)
            CALL MPI_WAITANY(SIZE(req_array), req_array, req_index, mpi_status, iErr)
            CALL StopTimer(iBarrier, sm_group%level)

         END IF
# else
         CALL StartTimer(iBarrier, sm_group%level)
         CALL MPI_WAITANY(SIZE(req_array), req_array, req_index, mpi_status, iErr)
         CALL StopTimer(iBarrier, sm_group%level)

# endif
!PRINT *, "WaitOnAnyMsgGrpRecv(", MPI_id, ",", sm_group%level, ", ", req_index, ") done."
!PRINT *
         msg_list => sm_group%messages

         ! Iterate through the list until the appropriate index has been reached.
         IF (req_index /= MPI_UNDEFINED) THEN

            i=0
            DO WHILE (ASSOCIATED(msg_list))
               IF (ASSOCIATED(msg_list%self%requests)) THEN
                  IF (.NOT. msg_list%self%requests%completed) THEN
                     i=i+1
                     IF (i == req_index) EXIT
                  END IF
               END IF
               msg_list => msg_list%next
            END DO

            ! Obtain the message from the message list.
            IF (ASSOCIATED(msg_list)) THEN

                message => msg_list%self
 
!                IF (message%tag == 9000) write(*,'(A,I,A,100I)') 'message', MPI_ID, 'recvwait', message%requests%request, mpi_status

                message%requests%completed = .TRUE.
                message%requests%request = 0
                message%first_block_received = .TRUE.

                ! Get the message's size in bytes and cache it in the message's buffer size.
                CALL MPI_GET_COUNT(mpi_status, MPI_PACKED, message_size, iErr)
                message%blocks%buffer_size = message_size
!                write(COMM_LOG_HANDLE,*) 'Received', MPI_WTIME()-InitTime, 100*message%level+(message%nMessages-1)+message%tag/1000, message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, message%remote_proc
                
                IF (message%lMultiBlock) THEN
                   ! Post the next receive; if it turns out to be unnecessary then
                   ! it will be cancelled when the message is closed.
                   CALL SetUpRecv(message)
                END IF
            END IF

         END IF

         DEALLOCATE(req_array)
         NULLIFY(req_array)

      END IF

   END SUBROUTINE WaitOnAnyMessageGroupRecv

   !> Waits for all current message blocks to keep sending.  Intended for use right before deallocating a StageMessageGroup object.
   !! @param sm_group The StageMessageGroupObject whose messages are 
   SUBROUTINE WaitOnAllMessageGroupSends(sm_group)
     USE GlobalDeclarations
      TYPE(StageMessageGroup), POINTER :: sm_group

      INTEGER, POINTER, DIMENSION(:) :: req_array
      INTEGER :: iErr


      ! This subroutine is NOT conducive to processing receives, so don't even try.
      IF (.NOT. sm_group%lSend)  RETURN

      ! Retrieve all open requests associated with this group's messages.
      CALL GetMGAllRequestsArray(sm_group, req_array, .TRUE., .TRUE.)

      ! Launch a wait all on all sends.
      IF (ASSOCIATED(req_array)) THEN

!         IF (sm_group%iStageTag == 9000) write(*,'(A,I,A,100I)') 'message', MPI_ID, 'sendwait', req_array
# if defined PTH
         IF (iThreaded == THREADED) THEN
            CALL pth_MPI_WAITALL(SIZE(req_array), req_array, iErr)
         ELSE

            CALL StartTimer(iBarrier, sm_group%level)
            CALL MPI_WAITALL(SIZE(req_array), req_array, MPI_STATUSES_IGNORE, iErr)
            CALL StopTimer(iBarrier, sm_group%level)
         END IF
# else
         CALL StartTimer(iBarrier, sm_group%level)
         CALL MPI_WAITALL(SIZE(req_array), req_array, MPI_STATUSES_IGNORE, iErr)
         CALL StopTimer(iBarrier, sm_group%level)

# endif
!         IF (sm_group%iStageTag == 9000) write(*,'(A,I,A,100I)') 'message', MPI_ID, 'sendwait_done', req_array
         DEALLOCATE(req_array)
         NULLIFY(req_array)
      END IF
      
!      write(COMM_LOG_HANDLE,*) 'Finisehd', MPI_WTIME()-InitTime, 100*message%level+(message%nMessages), message%tag + (message%level + 2) * TAG_LEVEL_MULTIPLIER + message%nMessages, message%remote_proc
   END SUBROUTINE WaitOnAllMessageGroupSends




   !> Returns the request list as a 1D integer array.
   !! @param message The packed message object to have its requests retrieved.
   !! @param req_array A 1D integer array pointer to be set to the request array.
   !! @param lOpenOnly An optional parameter; when set to true, the returned array will have completed requests zeroed out.
   !! @param lFlagAsCompleted When set to true, this optional parameter flags all open requests as closed.  BE SURE TO FOLLOW THIS OPERATION IMMEDIATELY WITH A WAITALL.
   SUBROUTINE GetRequestArray(message, req_array, lOpenOnly, lFlagAsCompleted)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, POINTER, DIMENSION(:) :: req_array, temp_array

      TYPE(RequestList), POINTER :: target_request
      INTEGER :: i
      LOGICAL, OPTIONAL :: lOpenOnly, lFlagAsCompleted
      INTEGER :: iErr
      LOGICAL :: open_only, flag_as_completed

      IF (PRESENT(lOpenOnly)) THEN
         open_only = lOpenOnly
      ELSE
         open_only = .FALSE.
      END IF

      IF (PRESENT(lFlagAsCompleted)) THEN
         flag_as_completed = lFlagAsCompleted
      ELSE
         flag_as_completed = .FALSE.
      END IF

      IF (.NOT. ASSOCIATED(message))  THEN
         PRINT *, "GetRequestArray() error: message object not associated."
         STOP
      END IF

      ! Clear the request array if it's already associated.
      IF (ASSOCIATED(req_array)) THEN
         DEALLOCATE(req_array)
         NULLIFY(req_array)
      END IF

      ! Allocate enough space for all of this message's message blocks.
      ALLOCATE(temp_array(message%nMessages), STAT=iErr)
      NULLIFY(req_array)
      IF (iErr /= 0) THEN
         PRINT *, "GetRequestArray() error: unable to allocate request array."
         STOP
      END IF

!      req_array = 0
      i = 0

      target_request => message%requests

      ! Iterate over the request list and store the request handles in the array.
      DO WHILE (ASSOCIATED(target_request))

         ! Filter out requests that have already been completed.
         IF (.NOT. (open_only .AND. target_request%completed)) THEN
            i=i+1
            temp_array(i) = target_request%request
            IF (flag_as_completed)  target_request%completed = .TRUE.
         END IF

         target_request => target_request%next
      END DO
      
      IF (i > 0) THEN
         ALLOCATE(req_array(i))
         req_array=temp_array(1:i)
      END IF
      DEALLOCATE(temp_array)
   END SUBROUTINE GetRequestArray

   !> Retrieves the first message block requests for all members of a message group and presents them as an array.
   !! @param sm_group The group whose requests are being retrieved.
   !! @param req_array A pointer to an array that will be allocated (or not, depending).
   !! @param lOpenOnly An optional parameter; when set to true, the returned array will have completed requests zeroed out.
   SUBROUTINE GetMGFirstRequestArray(sm_group, req_array, lOpenOnly)

      TYPE(StageMessageGroup) :: sm_group
      INTEGER, POINTER, DIMENSION(:) :: req_array, temp_array
      LOGICAL, OPTIONAL :: lOpenOnly

      TYPE(MessageList), POINTER :: msg_list
      INTEGER :: iErr
      INTEGER :: m
      LOGICAL :: open_only


      IF (PRESENT(lopenOnly)) THEN
         open_only = lOpenOnly
      ELSE
         open_only = .FALSE.
      END IF

!      IF (ASSOCIATED(req_array)) THEN
!         DEALLOCATE(req_array)
!         NULLIFY(req_array)
!      END IF

      ALLOCATE(temp_array(sm_group%nPackedMessages), STAT=iErr)
      NULLIFY(req_array)
      IF (iErr /= 0) THEN
         PRINT "('WaitOnMessageGroup() error: unable to allocate request array of size ', i2, '.')", sm_group%nPackedMessages
         STOP
      END IF

      !req_array = 0

      msg_list => sm_group%messages
      m = 0

      ! Count up the number of currently allocated messages
      DO WHILE (ASSOCIATED(msg_list))
         !PRINT *, "open_only = ", open_only
         !PRINT *, "ASSOCIATED(msg_list) = ", ASSOCIATED(msg_list)
         !PRINT *, "ASSOCIATED(msg_list%self) = ", ASSOCIATED(msg_list%self)
         !PRINT *, "ASSOCIATED(msg_list%self%requests) = ", ASSOCIATED(msg_list%self%requests)
         !PRINT *, "ASSOCIATED(msg_list%self%last_request) = ", ASSOCIATED(msg_list%self%last_request)
         !PRINT *, "completed = ", msg_list%self%requests%completed
         ! If only open requests are desired, then filter out the finished requests.
         IF (ASSOCIATED(msg_list%self%requests)) THEN
            IF (.NOT. (open_only .AND. msg_list%self%requests%completed))  THEN
               m=m+1
               temp_array(m) = msg_list%self%requests%request
            END IF
         END IF
         msg_list => msg_list%next
      END DO
      IF (m > 0) THEN
         ALLOCATE(req_array(m))
         req_array=temp_array(1:m)
      END IF
      DEALLOCATE(temp_array)
   END SUBROUTINE GetMGFirstRequestArray

   !> Retrieves the current message block requests for all members of a message group and presents them as an array.
   !! @param sm_group The group whose requests are being retrieved.
   !! @param req_array A pointer to an array that will be allocated (or not, depending).
   !! @param lOpenOnly An optional parameter; when set to true, the returned array will have completed requests zeroed out.
   SUBROUTINE GetMGCurrentRequestArray(sm_group, req_array, lOpenOnly)

      TYPE(StageMessageGroup) :: sm_group
      INTEGER, POINTER, DIMENSION(:) :: req_array, temp_array
      LOGICAL, OPTIONAL :: lOpenOnly

      TYPE(MessageList), POINTER :: msg_list
      INTEGER :: iErr
      INTEGER :: m
      LOGICAL :: open_only


      IF (PRESENT(lopenOnly)) THEN
         open_only = lOpenOnly
      ELSE
         open_only = .FALSE.
      END IF

      IF (ASSOCIATED(req_array)) THEN
         DEALLOCATE(req_array)
         NULLIFY(req_array)
      END IF

      ALLOCATE(temp_array(sm_group%nPackedMessages), STAT=iErr)

      IF (iErr /= 0) THEN
         PRINT *, "WaitOnMessageGroup() error: unable to allocate request array."
         STOP
      END IF

      !req_array = 0

      msg_list => sm_group%messages
      m = 0
      ! Count up the number of currently allocated messages
      DO WHILE (ASSOCIATED(msg_list))
         ! If only open requests are desired, then filter out the finished
         ! requests.
         IF (ASSOCIATED(msg_list%self%current_request)) THEN
            IF (.NOT. (open_only .AND. msg_list%self%current_request%completed)) THEN
               m=m+1
               temp_array(m) = msg_list%self%current_request%request
            END IF
         END IF
         msg_list => msg_list%next
      END DO

      IF (m > 0) THEN
         ALLOCATE(req_array(m))
         req_array=temp_array(1:m)
      END IF
      DEALLOCATE(temp_array)
   END SUBROUTINE GetMGCurrentRequestArray


   !> Retrieves the last message block requests for all members of a message group and presents them as an array.
   !! @param sm_group The group whose requests are being retrieved.
   !! @param req_array A pointer to an array that will be allocated (or not, depending).
   !! @param lOpenOnly An optional parameter; when set to true, the returned array will have completed requests zeroed out.
   SUBROUTINE GetMGLastRequestArray(sm_group, req_array, lOpenOnly)

      TYPE(StageMessageGroup) :: sm_group
      INTEGER, POINTER, DIMENSION(:) :: req_array, temp_array
      LOGICAL, OPTIONAL :: lOpenOnly

      TYPE(MessageList), POINTER :: msg_list
      INTEGER :: iErr
      INTEGER :: m
      LOGICAL :: open_only


      IF (PRESENT(lopenOnly)) THEN
         open_only = lOpenOnly
      ELSE
         open_only = .FALSE.
      END IF

      IF (ASSOCIATED(req_array)) THEN
         DEALLOCATE(req_array)
         NULLIFY(req_array)
      END IF

      ALLOCATE(temp_array(sm_group%nPackedMessages), STAT=iErr)
      NULLIFY(req_array)
      IF (iErr /= 0) THEN
         PRINT *, "WaitOnMessageGroup() error: unable to allocate request array."
         STOP
      END IF

!      req_array = 0

      msg_list => sm_group%messages
      m = 0

      ! Count up the number of currently allocated messages
      DO WHILE (ASSOCIATED(msg_list))
         ! If only open requests are desired, then filter out the finished requests.
         IF (ASSOCIATED(msg_list%self%last_request)) THEN
            IF (.NOT. (open_only .AND. msg_list%self%last_request%completed)) THEN
               m=m+1
               temp_array(m) = msg_list%self%last_request%request               
            END IF
         END IF
         msg_list => msg_list%next
      END DO
      IF (m > 0) THEN
         ALLOCATE(req_array(m))
         req_array=temp_array(1:m)
      END IF
      DEALLOCATE(temp_array)
   END SUBROUTINE GetMGLastRequestArray


   !> Retrieves all message block requests for all members of a message group and presents them as an array.
   !! @param sm_group The group whose requests are being retrieved.
   !! @param req_array A pointer to an array that will be allocated (or not, depending).
   !! @param lOpenOnly An optional parameter; when set to true, the returned array will have completed requests zeroed out.
   !! @param lFlagAsCompleted When set to true, this optional parameter flags all open requests as closed.  BE SURE TO FOLLOW THIS OPERATION IMMEDIATELY WITH A WAITALL.
   SUBROUTINE GetMGAllRequestsArray(sm_group, req_array, lOpenOnly, lFlagAsCompleted)

      TYPE(StageMessageGroup) :: sm_group
      INTEGER, POINTER, DIMENSION(:) :: req_array, temp_array
      LOGICAL, OPTIONAL :: lOpenOnly
      LOGICAL, OPTIONAL ::lFlagAsCompleted

      TYPE(MessageList), POINTER :: msg_list
      TYPE(PackedMessage), POINTER :: message
      TYPE(RequestList), POINTER :: req_ptr
      INTEGER :: iErr
      INTEGER :: message_count
      INTEGER :: m
      LOGICAL :: open_only
      LOGICAL :: flag_as_completed


      IF (PRESENT(lOpenOnly)) THEN
         open_only = lOpenOnly
      ELSE
         open_only = .FALSE.
      END IF

      IF (PRESENT(lFlagAsCompleted)) THEN
         flag_as_completed = lFlagAsCompleted
      ELSE
         flag_as_completed = .FALSE.
      END IF

      ! Initialize tracking variables.
!      IF (ASSOCIATED(req_array)) THEN
!         DEALLOCATE(req_array)
!         NULLIFY(req_array)
!      END IF

      message_count = 0

      msg_list => sm_group%messages

      ! Accumulate the number of message blocks in this StageMessageGroup.
      DO WHILE (ASSOCIATED(msg_list))
!         message_count = message_count + msg_list%self%nMessages
         message_count = message_count + msg_list%self%nRequests

!req_ptr => msg_list%self%requests
!DO WHILE (ASSOCIATED(req_ptr))
!req_count = req_count + 1
!PRINT *, "req_counting::req(", req_count, ") = ", req_ptr%request
!PRINT *, "req_counting::req(", req_count, ") completed = ", req_ptr%completed
!req_ptr => req_ptr%next
!END DO 

         msg_list => msg_list%next
      END DO

      ALLOCATE(temp_array(message_count), STAT=iErr)
      NULLIFY(req_array)
      IF (iErr /= 0) THEN
         PRINT *, "WaitOnMessageGroup() error: unable to allocate request array."
         STOP
      END IF

!      req_array = 0

      msg_list => sm_group%messages
      m = 0

      ! Count up the number of currently allocated messages
      DO WHILE (ASSOCIATED(msg_list))

         message => msg_list%self
         req_ptr => message%requests

         DO WHILE (ASSOCIATED(req_ptr))
            ! If only open requests are desired, then filter out the finished requests.
            IF (.NOT. (open_only .AND. req_ptr%completed)) THEN
               m=m+1
               temp_array(m) = req_ptr%request
               IF (flag_as_completed)  req_ptr%completed = .TRUE.
            END IF

            req_ptr => req_ptr%next
         END DO

         msg_list => msg_list%next

      END DO

      IF (m > 0) THEN
         ALLOCATE(req_array(m))
         req_array=temp_array(1:m)
      END IF
      DEALLOCATE(temp_array)
   END SUBROUTINE GetMGAllRequestsArray

END MODULE MpiTransmission


