!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    boundary.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file boundary.f90
!! @brief Main file for module Boundary

!> @defgroup Boundary Boundary
!! @brief Module for manipulating boundary type objects
!! @ingroup DataDeclarations

!> Module for manipulating boundary type objects
!! @ingroup Boundary
MODULE Boundary
  USE GlobalDeclarations
  IMPLICIT NONE
  PUBLIC

  !> Defines two faces of an info structure (left and right, top and bottom, etc...)
  TYPE Face
    SEQUENCE
    REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: data
  END type Face
  
  !> Defines the boundary of an info structure
  TYPE Boundaries
    SEQUENCE
    TYPE(Face), DIMENSION(:), POINTER :: side
  END type Boundaries

  !> Defines a pointer to boundary types - so boundaries can be put in arrays
  TYPE pBoundaries
    SEQUENCE
    TYPE(Boundaries), POINTER :: p
  END type PBoundaries
  
CONTAINS

  !> Allocates fixups based on bounds
  !! @param fixups boundaries object
  !! @param mB bounds of grid
  SUBROUTINE AllocBoundaries(fixups,mB)
    TYPE(Boundaries), POINTER :: fixups
    INTEGER, DIMENSION(3,2) :: mB, ip
    INTEGER :: i, iErr

    IF (.NOT. ASSOCIATED(fixups)) THEN
        ALLOCATE(fixups, STAT=iErr)
        NULLIFY(fixups%side)
        IF (iErr /= 0) THEN
            PRINT *, "AllocBoundaries() error: unable to allocate fixups object."
            STOP
        END IF
    END IF

    IF (.NOT. ASSOCIATED(fixups%side)) THEN
        ALLOCATE(fixups%side(nDim), STAT=iErr)
        
        IF (iErr /= 0) THEN
            PRINT *, "AllocBoundaries() error: unable to allocate fixups%side object."
            STOP
        END IF
     END IF
     ip=mb
     DO i=1,nDim
        ip(i,:)=(/1,2/)        
        NULLIFY(fixups%side(i)%data)
        ALLOCATE(fixups%side(i)%data(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),nFlux), STAT=iErr)
        CALL CheckAllocation(InfoAllocator, size(fixups%side(i)%data)*8, 'boundary')
        IF (iErr /= 0) THEN
           PRINT "('AllocBoundaries() error:  unable to allocate fixups%side(',i1,')%data object.')", i
           STOP
        END IF
        ip(i,:)=mb(i,:)
      END DO
  END SUBROUTINE AllocBoundaries

  !> Deallocates fixups
  !! @param fixups boundaries object
  SUBROUTINE DeAllocBoundaries(fixups)
    TYPE(Boundaries), POINTER :: fixups
    INTEGER :: i
    IF (.NOT. ASSOCIATED(fixups))  RETURN

    IF (ASSOCIATED(fixups%side)) THEN
        DO i=1,nDim
           IF (ASSOCIATED(fixups%side(i)%data)) THEN
              CALL CheckDeAllocation(InfoAllocator, size(fixups%side(i)%data)*8)
              DEALLOCATE(fixups%side(i)%data)
              NULLIFY(fixups%side(i)%data)
           END IF
        END DO

        DEALLOCATE(fixups%side)
        NULLIFY(fixups%side)

    END IF

    DEALLOCATE(fixups)
    NULLIFY(fixups)

  END SUBROUTINE DeAllocBoundaries

END MODULE boundary

