!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    data_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file data_declarations.f90
!! @brief Main file for module DataDeclarations

!> @defgroup DataDeclarations Data Declarations
!! @brief Module defining data info structure and basic operations
!! @ingroup DataOps


!> Module defining data info structure and basic operations
!! @ingroup DataDeclarations
!================================================================================
! Module Name:  DataDeclarations
! Module File:  data_declarations.f90
! Purpose:      Describe the InfoDef structure and the methods that operate upon it.
! Created:      20100625 by Brandon Shroyer.
! Modified:     20100628 Sync flux methods added by Jonathan Carroll.
!================================================================================
MODULE DataDeclarations
  USE GlobalDeclarations
  USE Boundary
  USE SlopeLim
  IMPLICIT NONE
  PRIVATE

  PUBLIC :: InfoDef

  PUBLIC :: InitInfo, InitialInitInfo, AllocChildFixups, BackupInfo

  PUBLIC :: NullifyInfoFields, DestroyInfo, DeAllocChildFixups

  PUBLIC :: BoxOverlap, LevelUp, LevelDown, GetMBounds, stretch, stretchaux, MapToLevel

  PUBLIC :: CalcOverlaps, CalcAuxOverlaps, CalcEMFOverlaps, CalcFluxOverlaps, &
            GhostOverlap, CalcPhysicalOverlaps, CalcCellOverlaps, CellPos, &
            PosCell, storefixupfluxes, storeemfs, MapBoxToInfo, MapBoxToLevel, &
            ProlongateCellCenteredData, expand

  !> Info data type declaration
  TYPE InfoDef
     SEQUENCE
     ! Defined by AMR module
     INTEGER :: level                                                  ! AMR level
     INTEGER, DIMENSION(3,2) :: mGlobal                                ! Global Index of child on level
     INTEGER, DIMENSION(3,2) :: mBounds                                ! Index of child within parent grid
     INTEGER, DIMENSION(3,2) :: mthbc                                  ! boundary conditions (should be 3x2)
     REAL(KIND=xPrec), DIMENSION(3,2) :: xBounds                       ! physical bounds of grid
     INTEGER, DIMENSION(3) :: mX                                       ! Size of grid
     !     REAL(KIND=qPrec) :: CostPerCell                                   ! Average computational cost per cell for this info
     !     REAL(KIND=qPrec) :: OldCostPerCell                                ! Average computational cost per cell for this info
     REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: CostPerGrid        ! Average computational cost per grid for this info
     REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: costmap          ! stores the sub tree cost per cell
     REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: Parentcostmap    ! accumulates cost for the parent
     INTEGER, DIMENSION(:,:,:), POINTER :: ChildMask                   ! Contains child ID's for refined cells

     ! Defined by advance module
     INTEGER, DIMENSION(:,:,:), POINTER :: ErrFlag                     ! Pointer to Refinement Flags

     ! Updated by processing module
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: diagnostics      ! Pointer to cell-centered quantities

     ! Updated by advance module
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: q                ! Pointer to cell-centered quantities
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: aux              ! Pointer to B-fields (face centered)
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: qParent          ! Pointer to parents cell-centered fields
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: auxParent        ! Pointer to parents face centered fields (used for first reconstruction)
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: qChild           ! Back up of q for child prolongation
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: auxChild         ! Back up of aux for child prolongation
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: MassFlux         ! Pointer to mass flux arrays needed to make self gravity 2nd order

     ! Fluxes stored to preserve interlevel and intralevel conservation
     TYPE(Boundaries), POINTER :: fixupflux                            ! Pointer to fluxes at grid edges (for synchronization with neighbors)
     TYPE(Boundaries), POINTER :: parentfixup                          ! coarsened fluxes for parents (for synchronization with parents)
     TYPE(pBoundaries), POINTER, DIMENSION(:) :: childfixups            ! Pointers to coarse fluxes applied at childrens' edges to be differenced with children's parentfixup

     ! EMF's stored to preserve interlevel and intralevel Divergence of B
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: emf              ! Pointer to emf's (edge centered)
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: parentemf        ! Accumulated emf's for parent
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: childemf         ! Pointer to accumulated child emf's (edge centered)

  END TYPE InfoDef

CONTAINS

  !> @name Info creation routines
  !! @{

  !> This routine allocates info array structures.  
  !! @details Basically every pointer except childfixups gets allocated
  !! Childfixups should be allocated when children are created and deallocated when children are destroyed.
  !! This routine expects Info%level, Info%Child_ID, Info%mGlobal, Info%mBounds, & Info%mthbc to be already set.
  SUBROUTINE InitInfo(Info, level, mGlobal, ParentmGlobal)
    TYPE(InfoDef), POINTER :: Info
    INTEGER, DIMENSION(3,2) :: mGlobal
    INTEGER, DIMENSION(3,2), OPTIONAL :: ParentmGlobal
    INTEGER :: level,rmbc
    INTEGER, DIMENSION(3,2) :: mB    !Bounds for cell-centered grid values (costmap, qChild, ErrFlag)
    INTEGER, DIMENSION(3,2) :: mA    !Bounds for face/edge centered grid values (AuxChild, ChildEmf)
    INTEGER, DIMENSION(3,2) :: mCext !Extended bounds for mass flux
    INTEGER, DIMENSION(3,2) :: mBext !Extended bounds for grid values (q)
    INTEGER, DIMENSION(3,2) :: mAext !Extended bounds for aux values (aux,emf)
    INTEGER, DIMENSION(3,2) :: mC    !1-cell extended cell-centered values (childmask)
    INTEGER, DIMENSION(3,2) :: PmB   !Parents Cell centered bounds (qParent, ParentFixup)
    INTEGER, DIMENSION(3,2) :: PmA   !Parents face/edge centered bounds (ParentEMF, auxParent)

    if (level < 0) THEN
       write(*,*) "shouldn't be allocating info"
       STOP
    END if
    ALLOCATE(Info)

    ALLOCATE(Info%CostPerGrid(1:levels(level)%steps))
    !    ALLOCATE(Info%CostPerGrid(1:levels(Info%level)%steps))

    CALL NullifyInfoFields(Info)
    Info%level=level
    !    Info%CostPerCell=1
    !    Info%OldCostPerCell=1
    !    Info%level=level
    Info%mGlobal=mGlobal     
    IF (Present(ParentmGlobal)) THEN
       Info%mBounds=GetmBounds(mGlobal, ParentmGlobal, Info%level)
    ELSE
       Info%mBounds=mGlobal
    END IF
    Info%mX=mGlobal(:,2)-mGlobal(:,1)+1
    Info%xBounds(:,1)=GxBounds(:,1)+(mGlobal(:,1)-1)*levels(level)%dx
    Info%xBounds(:,2)=GxBounds(:,1)+(mGlobal(:,2))*levels(level)%dx
    mB=1
    mA=1
    mAext=1
    mBext=1
    mC=1
    pmA=1
    mCext=1
    mB(1:nDim,2)=Info%mX(1:nDim)
    mA(1:nDim,2)=Info%mX(1:nDim)+1
    PmB=Info%mBounds
    PmA(1:nDim,2)=PmB(1:nDim,2)+1
    PmA(1:nDim,1)=PmB(1:nDim,1)
    mC(1:nDim,1)=0;mC(1:nDim,2)=Info%mX(1:nDim)+1
    rmbc=levels(level-1)%pmbc*levels(level-1)%CoarsenRatio
    mBext(1:nDim,1)=mB(1:nDim,1)-rmbc;mBext(1:nDim,2)=mB(1:nDim,2)+rmbc
    mAext(1:nDim,1)=mA(1:nDim,1)-rmbc;mAext(1:nDim,2)=mA(1:nDim,2)+rmbc
    mCext(1:nDim,1)=1-levels(Info%level)%ambc(1);mCext(1:nDim,2)=Info%mX(1:nDim)+1+levels(Info%level)%ambc(1) !range of mass flux

    !    Info%CostPerGrid = Info%CostPerCell * PRODUCT(Info%mX)

    !    ALLOCATE(Info%costmap(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),2))   
    !    Info%costmap=0
    IF (Info%level >= 0) THEN
       CALL AllocBoundaries(Info%fixupflux,mB)   
       ALLOCATE(Info%q(mBext(1,1):mBext(1,2),mBext(2,1):mBext(2,2),mBext(3,1):mBext(3,2),NrVars))
       CALL CheckAllocation(InfoAllocator, size(Info%q)*8, 'q')
       INfo%q(:,:,:,:) = 0d0/0d0
       IF (lStoreMassFlux) THEN
          ALLOCATE(Info%MassFlux(mCext(1,1):mCext(1,2), mCext(2,1):mCext(2,2), mCext(3,1):mCext(3,2),1:nDim))
          CALL CheckAllocation(InfoAllocator, size(Info%massflux)*8,"massflux")
          Info%MassFlux=0
       END IF

       IF (MaintainAuxArrays) THEN
          ALLOCATE(Info%aux(mAext(1,1):mAext(1,2),mAext(2,1):mAext(2,2),mAext(3,1):mAext(3,2),1:nAux))      
          CALL CheckAllocation(InfoAllocator, size(Info%aux)*8,"aux")
          ALLOCATE(Info%emf(mAext(1,1):mAext(1,2),mAext(2,1):mAext(2,2),mAext(3,1):mAext(3,2),1:nEMF)) !Don't need space to receive child emf's                      
          CALL CheckAllocation(InfoAllocator, size(Info%emf)*8,"emf")
          Info%emf=0 !Clear emf so it can accumulate
       END IF

       IF (Info%level < MaxLevel) THEN
          ALLOCATE(Info%ErrFlag(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3,1):mC(3,2)))
          CALL CheckAllocation(InfoAllocator, size(Info%ErrFlag)*4,"ErrFlag")
          ALLOCATE(Info%qChild(mBext(1,1):mBext(1,2),mBext(2,1):mBext(2,2),mBext(3,1):mBext(3,2),nProlongate))
          CALL CheckAllocation(InfoAllocator, size(Info%qChild)*8,"qChild")
          IF (MaintainAuxArrays) THEN
             ALLOCATE(Info%childemf(mA(1,1):mA(1,2),mA(2,1):mA(2,2),mA(3,1):mA(3,2),1:nEMf))
             CALL CheckAllocation(InfoAllocator, size(Info%childemf)*8,"childemf")
             ALLOCATE(Info%auxChild(mAext(1,1):mAext(1,2),mAext(2,1):mAext(2,2),mAext(3,1):mAext(3,2),1:nAux))
             CALL CheckAllocation(InfoAllocator, size(Info%auxchild)*8)
          END IF
       END IF
       ALLOCATE(Info%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3,1):mC(3,2)))
       CALL CheckAllocation(InfoAllocator, size(Info%childmask)*4,"childmask")
       Info%ChildMask=-1 !Assume we are isolated
       !        IF (Info%level == 0) Info%ChildMask=0

       IF (Info%level > 0) THEN
          !  Most things we need two copies of - the child's version and the parent's version - so they can be differenced or merged correctly.
          !  The cell centered quantities being the only exception.  Here the parent's values just get overwritten by child values...
          !  Two avoid allocating extra space and performing an extra copy on child grids that are on the same processor as their parent
          !  it is better to just have the child grids point to their parents
          !      IF (Present(Parent)) THEN  
          !        Info%qParent=>Parent%q
          !      ELSE
          ALLOCATE(Info%qParent(PmB(1,1):PmB(1,2),PmB(2,1):PmB(2,2),PmB(3,1):PmB(3,2),nRestrict))
          CALL CheckAllocation(InfoAllocator, size(Info%qParent)*8,"qParent")
          !      END IF
          CALL AllocBoundaries(Info%parentfixup,PmB)
          IF (MaintainAuxArrays) THEN
             ALLOCATE(Info%parentemf(PmA(1,1):PmA(1,2),PmA(2,1):PmA(2,2),PmA(3,1):PmA(3,2),nEMF))
             CALL CheckAllocation(InfoAllocator, size(Info%Parentemf)*8,"Parentemf")
          END IF
       END IF
    END IF

    !    IF (Info%level > -2) THEN
    !       mB=Info%mBounds
    !       ALLOCATE(Info%ParentCostMap(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1))
    !    END IF
  END SUBROUTINE InitInfo


  !> This routine allocates initial info array structures.  
  !! @details Basically every pointer except childfixups gets allocated
  ! Childfixups should be allocated when children are created and deallocated when children are destroyed.
  ! This routine expects Info%level, Info%Child_ID, Info%mGlobal, Info%mBounds, & Info%mthbc to be already set.
  SUBROUTINE InitialInitInfo(Info, level, mGlobal, ParentmGlobal)
    TYPE(InfoDef), POINTER :: Info
    INTEGER, DIMENSION(3,2) :: mGlobal
    INTEGER, DIMENSION(3,2), OPTIONAL :: ParentmGlobal
    INTEGER :: level, rmbc
    INTEGER, DIMENSION(3,2) :: mB    !Bounds for cell-centered grid values (costmap, qChild, ErrFlag)
    INTEGER, DIMENSION(3,2) :: mA    !Bounds for face/edge centered grid values (AuxChild, ChildEmf)
    INTEGER, DIMENSION(3,2) :: mBext !Extended bounds for grid values (q)
    INTEGER, DIMENSION(3,2) :: mAext !Extended bounds for aux values (aux,emf)
    INTEGER, DIMENSION(3,2) :: mC    !1-cell extended cell-centered values (childmask)
    INTEGER, DIMENSION(3,2) :: PmB   !Parents Cell centered bounds (qParent, ParentFixup)
    INTEGER, DIMENSION(3,2) :: PmA   !Parents face/edge centered bounds (ParentEMF, auxParent)

    IF (Present(ParentmGlobal)) THEN
       CALL InitInfo(Info, level, mGlobal, ParentmGlobal)
    ELSE
       CALL InitInfo(Info, level, mGlobal)
    END IF
    !    Info%costmap(:,:,:,1)=0d0 !tiny(1e0)!d0!1e-10!tiny(1e0)!0!1!2**((nDim+1)*(MaxLevel-max(level,0)))
    !    Info%costmap(:,:,:,2)=0d0

    IF (Info%level > 0) THEN
       IF (MaintainAuxArrays) THEN
          PmA=Info%mBounds

          PmA(1:nDim,2)=PmA(1:nDim,2)+1
          ALLOCATE(Info%auxParent(PmA(1,1):PmA(1,2),PmA(2,1):PmA(2,2),PmA(3,1):PmA(3,2),nAux))
          CALL CheckAllocation(InfoAllocator, size(Info%auxParent)*8, "auxParent")
       END IF
    END IF

  END SUBROUTINE InitialInitInfo


  !> Routine for storing applied x-fluxes to be coarsened and synchronized with other patches
  !! @param info info structure
  !! @param ms index range
  !! @param f array of fluxes
  SUBROUTINE storefixupfluxes(Info, mS, dim, f, fields)
    TYPE(InfoDef) :: Info
    INTEGER :: mS(3,2), mT(3,2), dim, pos(3), edges(2), ip(3,2), iq(3,2), childleft, childright,i,j,k, l,posl(3)
    REAL(KIND=qPREC), DIMENSION(:,:,:,:) :: f
    INTEGER, DIMENSION(:), OPTIONAL :: fields
    INTEGER :: fluxloc, mbc

    IF (lStoreMassFlux) THEN
       fluxloc=0
       IF (present(fields)) THEN
          DO j=1,size(fields)
             IF (fields(j) == 1) THEN
                fluxloc=j
             END IF
          END DO
       ELSE
          fluxloc=1
       END IF
       IF (fluxloc /= 0) THEN
          mbc=levels(Info%level)%ambc(levels(Info%level)%step)
          mT=mS
          mT(1:nDim,1)=max(mS(1:nDim,1),1-mbc)
          mT(1:nDim,2)=min(mS(1:nDim,2),Info%mX(1:nDim)+mbc)
          mT(dim,2)=min(mS(dim,2), Info%mX(dim)+mbc+1)
          iq=mT-spread(mS(:,1),2,2)+1
          Info%MassFlux(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2), dim) = &
               f(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2),fluxloc)
       END IF
    END IF


    mT=mS
    mT(:,1)=max(mS(:,1),1)
    mT(:,2)=min(mS(:,2),Info%mX)
    mT(dim,2)=min(mS(dim,2), Info%mX(dim)+1)
    ip=mT
    iq=mT
    edges=(/1,Info%mX(dim)+1/)
    DO i=1,2
       IF (mT(dim,1) <= edges(i) .AND. edges(i) <= mT(dim,2)) THEN
          ip(dim,:)=i
          iq=mT-spread(mS(:,1),2,2)+1
          iq(dim,:)=edges(i)-mS(dim,1)+1
          IF (present(fields)) THEN
             DO j=1,size(fields)
                fluxloc=invFluxFields(fields(j))
                IF (fluxloc /= 0) THEN
                   Info%fixupflux%side(dim)%data(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),fluxloc) = &
                        Info%fixupflux%side(dim)%data(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),fluxloc) + & 
                        f(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2),j)
                END IF
             END DO
          ELSE
             Info%fixupflux%side(dim)%data(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:) = &
                  Info%fixupflux%side(dim)%data(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:) + & 
                  f(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2),FluxFields)
          END IF

       END IF
    END DO
    IF (Info%level < MaxLevel) THEN !store values at child boundaries
       DO i=mT(1,1),mT(1,2)
          DO j=mT(2,1),mT(2,2)
             DO k=mT(3,1),mT(3,2)
                pos=(/i,j,k/)
                posl=pos
                posl(dim)=pos(dim)-1
                childleft=Info%ChildMask(posl(1),posl(2),posl(3))
                childright=Info%ChildMask(pos(1),pos(2),pos(3))
                IF (childleft <= 0 .AND. childright <= 0) CYCLE
                IF (childleft > 0 .AND. childright > 0) CYCLE
                IF (childleft > 0) THEN !ChildCell left
                   posl(dim)=2
                   IF (present(fields)) THEN
                      DO l=1,size(fields)
                         fluxloc=invFluxFields(fields(l))
                         IF (fluxloc /= 0) THEN
                            Info%childfixups(childleft)%p%side(dim)%data(posl(1),posl(2),posl(3),fluxloc) = &
                                 Info%childfixups(childleft)%p%side(dim)%data(posl(1),posl(2),posl(3),fluxloc) + &
                                 f(pos(1)-mS(1,1)+1,pos(2)-mS(2,1)+1,pos(3)-mS(3,1)+1,l)
                         END IF
                      END DO
                   ELSE
                      Info%childfixups(childleft)%p%side(dim)%data(posl(1),posl(2),posl(3),:) = &
                           Info%childfixups(childleft)%p%side(dim)%data(posl(1),posl(2),posl(3),:) + &
                           f(pos(1)-mS(1,1)+1,pos(2)-mS(2,1)+1,pos(3)-mS(3,1)+1,FluxFields)
                   END IF
                ELSE
                   posl(dim)=1
                   IF (present(fields)) THEN
                      DO l=1,size(fields)
                         fluxloc=invFluxFields(fields(l))
                         IF (fluxloc /= 0) THEN
                            Info%childfixups(childright)%p%side(dim)%data(posl(1),posl(2),posl(3),fluxloc) = &
                                 Info%childfixups(childright)%p%side(dim)%data(posl(1),posl(2),posl(3),fluxloc) + &
                                 f(pos(1)-mS(1,1)+1,pos(2)-mS(2,1)+1,pos(3)-mS(3,1)+1,l)
                         END IF
                      END DO
                   ELSE
                      Info%childfixups(childright)%p%side(dim)%data(posl(1),posl(2),posl(3),:) = &
                           Info%childfixups(childright)%p%side(dim)%data(posl(1),posl(2),posl(3),:) + &
                           f(pos(1)-mS(1,1)+1,pos(2)-mS(2,1)+1,pos(3)-mS(3,1)+1,FluxFields)
                   END IF
                END IF
             END DO
          END DO
       END DO
    END IF
  END SUBROUTINE storefixupfluxes

  !> Stores emfs
  !! @param Info info structure
  !! @param mS indices of emf in info
  !! @param dim direction of emf component
  !! @param emf array of emf's
  SUBROUTINE StoreEmfs(Info, mS, dim, emf)
    TYPE(InfoDef) :: Info
    INTEGER :: mS(3,2), mT(3,2), dim,iq(3,2)
    REAL(KIND=qPREC), DIMENSION(:,:,:) :: emf
    !     return
    mT=mS
    mT(:,1)=max(mS(:,1),1)
    mT(:,2)=min(mS(:,2),Info%mX+1)
    mT(dim,2)=min(mS(dim,2),Info%mX(dim))
    iq=mT-spread(mS(:,1),2,2)+1
    IF (product(iq(:,2)-iq(:,1)+1) > 0) THEN
       Info%emf(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2),emfloc(dim)) = &
            Info%emf(mT(1,1):mT(1,2), mT(2,1):mT(2,2), mT(3,1):mT(3,2),emfloc(dim)) + &
            emf(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2))
    END IF
  END SUBROUTINE StoreEmfs


  !> Routine that duplicates the necessary parts of info to restart
  !! @param original original info
  !! @param info backup info
  SUBROUTINE BackupInfo(original, info, lRestore)
    TYPE(InfoDef), POINTER :: original, info
    LOGICAL :: lRestore
    INTEGER, DIMENSION(3,2) :: ip
    INTEGER, DIMENSION(3,2) :: mc
    INTEGER, DIMENSION(3,2) :: ParentmB

    IF (lRegridLevel(original%level)) THEN
       ip(:,1)=1
       ip(:,2)=original%mx
    ELSE
       ip=1
       ip(1:nDim,2)=original%mx(1:nDim)+levels(BaseLevel)%gmbc(1)
       ip(1:nDim,1)=1-levels(BaseLevel)%gmbc(1)
    END IF
    IF (.NOT. lRegridLevel(original%level) .AND. lRestore) THEN
       CALL DestroyInfo(info)
       CALL InitInfo(info,original%level, original%mGlobal, GetParentmGlobal(original%mGlobal, original%mBounds, original%level))
    ELSE
       IF (ASSOCIATED(info)) THEN
          write(*,*) 'whoops - info should not be associated in data_declarations'
          STOP
       END IF
       ALLOCATE(Info)
       CALL NullifyInfoFields(info)
       Info%level=original%level
       Info%mGlobal=original%mGlobal
       Info%mBounds=original%mBounds
       Info%xBounds=original%xBounds
       Info%mX=original%mX      
    END IF

    IF (.NOT. ALLOCATED(info%costpergrid)) ALLOCATE(Info%CostPerGrid(size(original%costpergrid)))
    Info%costpergrid=original%costpergrid

    !      IF (.NOT. ASSOCIATED(Info%Costmap)) ALLOCATE(Info%Costmap(ip(1),ip(2),ip(3),size(original%costmap, 4)))
    !      Info%Costmap(1:ip(1),1:ip(2),1:ip(3),:)= &
    !           original%costmap(1:ip(1),1:ip(2),1:ip(3),:)
    IF (info%level > -1) THEN
       IF (.NOT. ASSOCIATED(Info%q)) THEN
          ALLOCATE(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),size(original%q, 4)))
          CALL CheckAllocation(InfoAllocator, size(Info%q)*8, "backup q")
       END IF


       Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:)= &
            original%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),:)
       IF (MaintainAuxArrays) THEN         
          ip(1:nDim,2)=ip(1:nDim,2)+1
          IF (.NOT. ASSOCIATED(Info%aux)) THEN
             ALLOCATE(Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),size(original%aux, 4)))
             CALL CheckAllocation(InfoAllocator, size(Info%aux)*8, "backup aux")
          END IF
          Info%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2),ip(3,1):ip(3,2),:)=&
               original%aux(ip(1,1):ip(1,2), ip(2,1):ip(2,2),ip(3,1):ip(3,2),:)
       END IF
    END IF
    IF (.NOT. lRegridLevel(Info%level)) THEN !Need to also backup/restore childmask
       mC=1
       !         mC(nDim+1:3,:)=1
       mC(1:nDim,1)=0;mC(1:nDim,2)=Info%mX(1:nDim)+1
       IF (.NOT. ASSOCIATED(Info%ChildMask)) THEN
          ALLOCATE(Info%ChildMask(mc(1,1):mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2)))
          CALL CheckAllocation(InfoAllocator, size(Info%ChildMask)*4, "backup childmask")
       END IF
       Info%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3,1):mc(3,2))=&
            original%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3,1):mc(3,2))
    END IF
  END SUBROUTINE BackupInfo



  !> Allocates storage for fixupfluxes to be differenced with children fixupfluxes
  SUBROUTINE AllocChildFixups(Info, childgrids)
    TYPE(InfoDef), POINTER :: Info
    INTEGER, DIMENSION(:,:,:), POINTER :: childgrids
    TYPE(Boundaries), POINTER :: childfixup
    INTEGER :: i, n, tempmX(3), ip(3,2)
    ! If there was a child (own or neighbors) then there must still be a parent..
    WHERE(Info%ChildMask > 0 .OR. Info%ChildMask==NEIGHBORCHILD) Info%ChildMask=0
    IF (ASSOCIATED(childgrids)) THEN
       n=size(childgrids,3)
       IF (n > 0) THEN

          ALLOCATE(Info%childfixups(n))

          DO i=1,n
             NULLIFY(Info%childfixups(i)%p)
             ip=leveldown(childgrids(:,:,i), Info%level+1)-SPREAD(Info%mGlobal(:,1)-1,2,2)
             CALL AllocBoundaries(Info%childfixups(i)%p,ip)
             Info%ChildMask(ip(1,1):ip(1,2), &
                  ip(2,1):ip(2,2), &
                  ip(3,1):ip(3,2)) = i
          END DO

       END IF
    END IF
  END SUBROUTINE AllocChildFixups

  !> @}

  !> @name Info destruction routines
  !! @{

  !> This routine nullifies all info fields
  !! @param Info Info structure
  SUBROUTINE NullifyInfoFields(Info)

    TYPE(InfoDef) :: Info


    NULLIFY(Info%costmap, Info%ChildMask, Info%ErrFlag, &
         Info%q, Info%aux, Info%qParent, Info%auxParent, &
         Info%fixupflux, Info%parentfixup, Info%childfixups, Info%emf, &
         Info%parentemf, Info%childemf, Info%ParentCostMap, Info%AuxChild, Info%qChild, Info%massflux)

  END SUBROUTINE NullifyInfoFields

  !> This routine destroys info structures allocated by initinfo.  
  !! @param Info Info structure
  !! @details Basically every pointer except childfixups gets deallocated
  !! Childfixups should be allocated when children are created and deallocated when children are destroyed.
  SUBROUTINE DestroyInfo(Info)
    TYPE(InfoDef), POINTER :: Info

    IF (.NOT. ASSOCIATED(Info)) RETURN


    IF (ALLOCATED(Info%CostPerGrid)) THEN
       DEALLOCATE(Info%CostPerGrid)
    END IF

    IF (ASSOCIATED(Info%costmap)) THEN
       DEALLOCATE(Info%costmap)
       NULLIFY(Info%costmap)
    END IF
    IF (ASSOCIATED(Info%ParentCostMap)) THEN
       DEALLOCATE(Info%ParentCostMap)
       NULLIFY(Info%ParentCostMap)
    END IF
    !    IF (Info%level >= 0) THEN
    IF (ASSOCIATED(Info%q)) THEN
       CALL CheckDeAllocation(InfoAllocator, size(Info%q)*8)
       DEALLOCATE(Info%q)
       NULLIFY(Info%q)
    END IF

    IF (ASSOCIATED(Info%MassFlux)) THEN
       CALL CheckDeAllocation(InfoAllocator, size(Info%massflux)*8)
       DEALLOCATE(Info%MassFlux)
       NULLIFY(Info%MassFlux)
    END IF
    IF (ASSOCIATED(Info%ChildMask)) THEN
       CALL CheckDeAllocation(InfoAllocator, size(Info%childmask)*4)
       DEALLOCATE(Info%ChildMask)
       NULLIFY(Info%ChildMask)
    END IF

    IF (ASSOCIATED(Info%fixupflux)) CALL DeAllocBoundaries(Info%fixupflux)
    IF (MaintainAuxArrays) THEN
       IF (ASSOCIATED(Info%aux)) THEN
          CALL CheckDeAllocation(InfoAllocator, size(Info%aux)*8)
          DEALLOCATE(Info%aux)
          NULLIFY(Info%aux)
       END IF
       IF (ASSOCIATED(Info%emf)) THEN
          CALL CheckDeAllocation(InfoAllocator, size(Info%emf)*8)
          DEALLOCATE(Info%emf)
          NULLIFY(Info%emf)
       END IF
    END IF
    IF (Info%level > 0) THEN
       CALL DeAllocBoundaries(Info%parentfixup)          
       IF (ASSOCIATED(Info%qParent)) THEN
          CALL CheckDeAllocation(InfoAllocator, size(Info%qParent)*8)
          DEALLOCATE(Info%qParent)
          NULLIFY(Info%qParent)
       END IF
       IF (MaintainAuxArrays) THEN
          IF (ASSOCIATED(Info%parentemf)) THEN
             CALL CheckDeAllocation(InfoAllocator, size(Info%Parentemf)*8)
             DEALLOCATE(Info%parentemf)
             NULLIFY(Info%parentemf)
          END IF
          IF (ASSOCIATED(Info%auxParent)) THEN
             CALL CheckDeAllocation(InfoAllocator, size(Info%auxParent)*8)
             DEALLOCATE(Info%auxParent)
             NULLIFY(Info%auxParent)
          END IF
       END IF
    END IF
    IF (Info%level < MaxLevel) THEN

       IF (ASSOCIATED(Info%ErrFlag)) THEN
          CALL CheckDeAllocation(InfoAllocator, size(Info%ErrFlag)*4)
          DEALLOCATE(Info%ErrFlag)
          NULLIFY(Info%ErrFlag)
       END IF

       IF (ASSOCIATED(Info%qChild)) THEN
          CALL CheckDeAllocation(InfoAllocator, size(Info%qchild)*8)
          DEALLOCATE(Info%qChild)
          NULLIFY(Info%qChild)
       END IF

       IF (MaintainAuxArrays) THEN

          IF (ASSOCIATED(Info%auxChild)) THEN
             CALL CheckDeAllocation(InfoAllocator, size(Info%auxchild)*8)
             DEALLOCATE(Info%auxChild)
             NULLIFY(Info%auxChild)
          END IF

          IF (ASSOCIATED(Info%childemf)) THEN
             CALL CheckDeAllocation(InfoAllocator, size(Info%childemf)*8)
             DEALLOCATE(Info%childemf)       
             NULLIFY(Info%childemf)
          END IF
       END IF
       CALL DeAllocChildFixups(Info)
    END IF


    DEALLOCATE(Info)
    NULLIFY(Info)


  END SUBROUTINE DestroyInfo

  !> This routine deallocates info structures associated with children
  !! @param Info Info structure
  SUBROUTINE DeallocChildFixups(Info)

    TYPE(InfoDef), POINTER :: Info

    INTEGER :: i, n


    IF (.NOT. ASSOCIATED(Info))  RETURN

    IF (.NOT. ASSOCIATED(Info%childfixups))  RETURN
    DO i = 1, SIZE(Info%childfixups)
       CALL DeAllocBoundaries(Info%childfixups(i)%p)
    END DO
    DEALLOCATE(Info%childfixups)
    NULLIFY(Info%childfixups)
  END SUBROUTINE DeallocChildFixups

  !> @}

  !> @name OverlapCalculationRoutines Info Overlap Calculation Routines
  !! @{

  !> @brief Calculates number and bounds of overlaps for cell centered quantities between two grids given their locations
  !! @param InfomGlobal - Info%mGlobal
  !! @param SourcemGlobal - Source%mGlobal
  !! @param mTs - collection of info indices
  !! @param mSs - collection of source indices
  !! @param nOverlaps - number of ways grids overlap
  !! @param level - level
  !! @param nghost - number of ghost zones to add to consider when calculating overlaps
  SUBROUTINE CalcOverlaps(InfomGlobal,SourcemGlobal,mTs,mSs,nOverlaps,level,lPeriodic,nghost)
    ! Calculates overlap bounds of cell-centered quantities for
    !  overlapping grids.
    INTEGER, DIMENSION(3,2) :: InfomGlobal, SourcemGlobal
    INTEGER, DIMENSION(3,2) :: mO,mGlobal,iOffSet
    INTEGER, DIMENSION(3) :: pOffSet
    INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
    INTEGER, DIMENSION(27,3,2) :: MaxMTs, MaxMSs
    INTEGER :: i,j,k,rmbc,nOverlaps,level
    INTEGER, OPTIONAL :: nghost
    LOGICAL, DIMENSION(:) :: lPeriodic
    IF (present(nGhost)) THEN
       rmbc=nGhost
    ELSE
       rmbc = levels(level)%gmbc(levels(level)%step) !CoarsenRatio(level-1) * mbc
    END IF
    NULLIFY(mTs, mSs)
    nOverlaps=0
    mO=1
    !    mO(nDim+1:3,:)=1
    ioffset=0
    WHERE(lPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*levels(level)%mX(:)
             mGlobal(:,:)=SourcemGlobal(:,:)+SPREAD(pOffSet,2,2)

             mO(1:nDim,1)=max(InfomGlobal(1:nDim,1)-rmbc&
                  &,mGlobal(1:nDim,1)) 
             mO(1:nDim,2)=min(InfomGlobal(1:nDim,2)+rmbc&
                  &,mGlobal(1:nDim,2))

             IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                nOverlaps=nOverlaps+1
                MaxMTs(nOverlaps,:,:)=mO-Spread(  InfomGlobal(:,1)&
                     &,2,2)+1
                MaxMSs(nOverlaps,:,:)=mO-Spread(SourcemGlobal(:,1)&
                     &+pOffset,2,2)+1
             END IF

          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MTs(nOverlaps,3,2),MSs(nOverlaps,3,2))
       MTs=MaxMTs(1:nOverlaps,:,:)
       MSs=MaxMSs(1:nOverlaps,:,:)
    END IF
  END SUBROUTINE CalcOverlaps

  !> @brief Calculates number and bounds of overlaps for face centered quantities between two grids given their locations
  !! @param InfomGlobal - Info%mGlobal
  !! @param SourcemGlobal - Source%mGlobal
  !! @param mTs - collection of info indices
  !! @param mSs - collection of source indices
  !! @param nOverlaps - number of ways grids overlap
  !! @param level - level
  !! @param dir - Which face to consider
  !! @param nghost - number of ghost zones to add to consider when calculating overlaps
  SUBROUTINE CalcAuxOverlaps(InfomGlobal,SourcemGlobal,mTs,mSs,nOverlaps,level,dir,lPeriodic, nghost)
    ! Calculates overlap bounds of cell-centered quantities for
    !  overlapping grids.
    INTEGER, DIMENSION(3,2) :: InfomGlobal, SourcemGlobal
    INTEGER, DIMENSION(3,2) :: mO,mGlobal,iOffSet
    INTEGER, DIMENSION(3) :: pOffSet
    INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
    INTEGER, DIMENSION(27,3,2) :: MaxMTs, MaxMSs
    INTEGER :: i,j,k,rmbc,nOverlaps,level,l,dir
    INTEGER, OPTIONAL :: nghost
    LOGICAL, DIMENSION(:) :: lPeriodic
    IF (present(nGhost)) THEN
       rmbc=nGhost
    ELSE
       rmbc = levels(level)%gmbc(levels(level)%step) !CoarsenRatio(level-1) * mbc
    END IF
    NULLIFY(mTs, mSs)
    nOverlaps=0
    mO=1
    !    mO(nDim+1:3,:)=1
    ioffset=0
    WHERE(lPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*levels(level)%mX(:)
             mGlobal(:,:)=SourcemGlobal(:,:)+SPREAD(pOffSet,2,2)

             mO(1:nDim,1)=max(InfomGlobal(1:nDim,1)-rmbc,mGlobal(1:nDim,1)) 
             mO(1:nDim,2)=min(InfomGlobal(1:nDim,2)+rmbc,mGlobal(1:nDim,2))
             mO(dir,2)=mO(dir,2)+1 !Shift upper bound for Aux fields
             IF (ALL(mO(1:nDim,2) >= (mO(1:nDim,1)))) THEN 
                nOverlaps=nOverlaps+1
                MaxMTs(nOverlaps,:,:)=mO-Spread(  InfomGlobal(:,1)&
                     &,2,2)+1
                MaxMSs(nOverlaps,:,:)=mO-Spread(SourcemGlobal(:,1)&
                     &+pOffset,2,2)+1
                !                write(*,*) 'found overlap'
                !                write(*,'(6I5.2)') MaxMSs(nOverlaps,:,:)
                !                write(*,'(6I5.2)') MaxMTs(nOverlaps,:,:)
             END IF

          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MTs(nOverlaps,3,2),MSs(nOverlaps,3,2))
       MTs=MaxMTs(1:nOverlaps,:,:)
       MSs=MaxMSs(1:nOverlaps,:,:)
    END IF
  END SUBROUTINE CalcAuxOverlaps


  !> @brief Calculates number and bounds of overlaps for edge centered quantities between two grids given their locations
  !! @param InfomGlobal Info%mGlobal
  !! @param SourcemGlobal Source%mGlobal
  !! @param mTs collection of info indices
  !! @param mSs collection of source indices
  !! @param nOverlaps number of ways grids overlap
  !! @param offsets periodic offset of source grid
  !! @param level level
  !! @param dir direction of edge to consider
  SUBROUTINE CalcEMFOverlaps(InfomGlobal,SourcemGlobal,mTs,mSs,nOverlaps,offsets,level,dir, lPeriodic)
    ! Calculates overlap bounds of cell-centered quantities for
    !  overlapping grids.
    INTEGER, DIMENSION(3,2) :: InfomGlobal, SourcemGlobal
    INTEGER, DIMENSION(3,2) :: mO,mGlobal,iOffSet
    INTEGER, DIMENSION(3) :: pOffSet
    INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
    INTEGER, DIMENSION(:,:), POINTER :: offsets
    INTEGER, DIMENSION(27,3,2) :: MaxMTs, MaxMSs
    INTEGER, DIMENSION(27,3) :: MaxOffsets
    INTEGER :: i,j,k,nOverlaps,level,l,dir
    LOGICAL, DIMENSION(:) :: lPeriodic
    NULLIFY(mTs, mSs)
    nOverlaps=0
    mO=1
    !    mO(nDim+1:3,:)=1
    ioffset=0
    WHERE(lPeriodic(1:nDim)) ioffset(1:nDim,2)=1 !nperiodic_overlaps(1:nDim)
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*levels(level)%mX(:)
             mGlobal(:,:)=SourcemGlobal(:,:)+SPREAD(pOffSet,2,2)

             mO(1:nDim,1)=max(InfomGlobal(1:nDim,1)&
                  &,mGlobal(1:nDim,1)) 
             mO(1:nDim,2)=min(InfomGlobal(1:nDim,2)&
                  &,mGlobal(1:nDim,2))
             mO(:,2)=mO(:,2)+1 !Shift upper bound for emf fields  !This works for 2 or 3D
             mO(dir,2)=mO(dir,2)-1 !Shorten normal direction
             IF (ALL(mO(1:nDim,2) >= (mO(1:nDim,1)))) THEN 
                nOverlaps=nOverlaps+1
                MaxMTs(nOverlaps,:,:)=mO-Spread(  InfomGlobal(:,1)&
                     &,2,2)+1
                MaxMSs(nOverlaps,:,:)=mO-Spread(SourcemGlobal(:,1)&
                     &+pOffset,2,2)+1
                MaxOffsets(nOverlaps,:)=pOffSet
             END IF

          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MTs(nOverlaps,3,2),MSs(nOverlaps,3,2) &
            &,offsets(nOverlaps,3))
       MTs=MaxMTs(1:nOverlaps,:,:)
       MSs=MaxMSs(1:nOverlaps,:,:)
       offsets=MaxOffsets(1:nOverlaps,:)
    END IF
  END SUBROUTINE CalcEMFOverlaps

  !> @brief Calculates number and bounds of overlaps for face centered quantities between two grids given their locations
  !! @param InfomGlobal - Info%mGlobal
  !! @param SourcemGlobal - Source%mGlobal
  !! @param mTs - collection of info indices
  !! @param mSs - collection of source indices
  !! @param edges collection of edges
  !! @param nOverlaps - number of ways grids overlap
  !! @param level - level
  !! @param dir - normal direction to face quantity
  SUBROUTINE CalcFluxOverlaps(InfomGlobal,SourcemGlobal,mTs,mSs,edges,nOverlaps,level,dir,lPeriodic)
    ! Calculates overlap bounds of cell-centered quantities for
    !  overlapping grids.      
    INTEGER, DIMENSION(3,2) :: InfomGlobal, SourcemGlobal
    INTEGER, DIMENSION(3,2) :: mO,mGlobal,iOffSet
    INTEGER, DIMENSION(3) :: pOffSet
    INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
    INTEGER, DIMENSION(:), POINTER :: edges
    INTEGER, DIMENSION(2,3,2) :: MaxMTs, MaxMSs
    INTEGER, DIMENSION(2) :: Maxedges
    INTEGER :: i,j,k,nOverlaps,level,dir
    LOGICAL, DIMENSION(:) :: lPeriodic
    NULLIFY(mTs, mSs)
    nOverlaps=0
    mO=1
    !    mO(nDim+1:3,:)=1
    ioffset=0
    IF (lPeriodic(dir)) THEN
       ioffset(dir,2)=1!nperiodic_overlaps(dir)
       ioffset(dir,1)=-1!-nperiodic_overlaps(dir)
    END IF
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*levels(level)%mX(:)
             mGlobal(:,:)=SourcemGlobal(:,:)+SPREAD(pOffSet,2,2)
             IF (ALL(InfomGlobal==mGlobal)) CYCLE !Skip self...
             mO(1:nDim,1)=max(InfomGlobal(1:nDim,1),mGlobal(1:nDim&
                  &,1)) 
             mO(1:nDim,2)=min(InfomGlobal(1:nDim,2),mGlobal(1:nDim&
                  &,2))
             mO(dir,2)=mO(dir,2)+1 !Stretch bounds of overlap
             ! regions for face fields
             IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                nOverlaps=nOverlaps+1
                IF (nOverlaps > 2) THEN
                   write(*,*) 'too many overlaps'
                   STOP
                END IF

                MaxMTs(nOverlaps,:,:)=mO-Spread(  InfomGlobal(:,1)&
                     &,2,2)+1
                MaxMSs(nOverlaps,:,:)=mO-Spread(SourcemGlobal(:,1)&
                     &+pOffSet,2,2)+1
                IF (mO(dir,1)==InfomGlobal(dir,1)) THEN
                   MaxEdges(nOverlaps)=1
                ELSEIF (mO(dir,2)==InfomGlobal(dir,2)+1) THEN
                   MaxEdges(nOverlaps)=2
                ELSE
                   PRINT*, "error in flux overlaps"
                END IF
             END IF
          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MTs(nOverlaps,3,2),MSs(nOverlaps,3,2)&
            &,edges(nOverlaps))
       MTs=MaxMTs(1:nOverlaps,:,:)
       MSs=MaxMSs(1:nOverlaps,:,:)
       edges=MaxEdges(1:nOverlaps)
    END IF
  END SUBROUTINE CalcFluxOverlaps


  !> Calculates overlap regions between a physical box and an info structure.
  !! @param Info Info object
  !! @param PhysicalRegion Physical bounds of a box
  !! @param mSs bounds of overlap regions relative to Info
  !! @param nOverlaps number of overlap regions
  !! @param offsets Physical offsets of info region to use with each overlap
  SUBROUTINE CalcPhysicalOverlaps(Info, PhysicalRegion, mSs, nOverlaps, offsets, location, lPeriodic, rmbcOpt)
    REAL(KIND=qPREC), DIMENSION(3,2) :: PhysicalRegion
    TYPE(InfoDef) :: Info
    INTEGER :: nOverlaps, location
    REAL(KIND=qPREC), DIMENSION(:,:), POINTER :: offsets
    INTEGER, DIMENSION(:,:,:), POINTER :: mSs
    INTEGER :: rmbc, i, j, k, dir, edge
    REAL(KIND=qPREC), DIMENSION(27,3) :: MaxOffsets
    INTEGER, DIMENSION(27,3,2) :: MaxmSs
    INTEGER, DIMENSION(3,2) :: mO,iOffSet,ip
    REAL(KIND=qPREC), DIMENSION(3) :: pOffSet
    INTEGER, OPTIONAL :: rmbcOpt
    LOGICAL, DIMENSION(:) :: lPeriodic
    IF (PRESENT(rmbcOpt)) THEN
       rmbc=rmbcOpt
    ELSE
       rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)        
    END IF
    NULLIFY(mSs)
    nOverlaps=0
    mO=1
    !     mO(nDim+1:3,:)=1
    ioffset=0
    WHERE(lPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*(GxBounds(:,2)-GxBounds(:,1)) !Physical offset for info structure
             IF (location == IEVERYWHERE) THEN
                mO(1:nDim,1)=max(1-rmbc,ceiling((PhysicalRegion(1:nDim,1)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                             /levels(Info%level)%dx))
                mO(1:nDim,2)=min(Info%mX(1:nDim)+rmbc,ceiling((PhysicalRegion(1:nDim,2)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                             /levels(Info%level)%dx))
                IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                   nOverlaps=nOverlaps+1
                   MaxMSs(nOverlaps,:,:)=mO(:,:)
                   MaxOffsets(nOverlaps,:)=pOffset
                END IF
             ELSEIF (location == IBOUNDARIES) THEN
                DO dir=1,nDim
                   DO edge=1,2                    
                      IF (GhostOverlap(Info, dir, edge, ip)) THEN
                         mO(1:nDim,1)=max(ip(1:nDim,1),ceiling((PhysicalRegion(1:nDim,1)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                                      /levels(Info%level)%dx))
                         mO(1:nDim,2)=min(ip(1:nDim,2),ceiling((PhysicalRegion(1:nDim,2)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                                      /levels(Info%level)%dx))
                         IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                            nOverlaps=nOverlaps+1
                            MaxMSs(nOverlaps,:,:)=mO(:,:)
                            MaxOffsets(nOverlaps,:)=pOffset
                         END IF
                      END IF
                   END DO
                END DO
             ELSE
                DO dir=1,nDim
                   DO edge=1,2                    
                      IF (location == IBOUNDARY(dir,edge)) THEN
                         IF (GhostOverlap(Info, dir, edge, ip)) THEN
                            mO(1:nDim,1)=max(ip(1:nDim,1),ceiling((PhysicalRegion(1:nDim,1)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                                          /levels(Info%level)%dx))
                            mO(1:nDim,2)=min(ip(1:nDim,2),ceiling((PhysicalRegion(1:nDim,2)-(Info%xBounds(1:nDim,1)+pOffset(1:nDim))) &
                                         /levels(Info%level)%dx))
                            IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                               nOverlaps=nOverlaps+1
                               MaxMSs(nOverlaps,:,:)=mO(:,:)
                               MaxOffsets(nOverlaps,:)=pOffset
                            END IF
                         END IF
                      END IF
                   END DO
                END DO
             END IF
          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MSs(nOverlaps,3,2),offsets(nOverlaps,3))
       MSs=MaxMSs(1:nOverlaps,:,:)
       offsets=MaxOffsets(1:nOverlaps,:)
    END IF
  END SUBROUTINE CalcPhysicalOverlaps

  !> Calculates overlap regions between a indexed box and an info structure.
  !! @param Info Info object
  !! @param PhysicalRegion Physical bounds of a box (in its levels index space)
  !! @param mSs bounds of overlap regions relative to Info
  !! @param nOverlaps number of overlap regions
  !! @param offsets Physical offsets of info region to use with each overlap
  SUBROUTINE CalcCellOverlaps(Info, PhysicalRegion, mSs, nOverlaps, offsets, location, lPeriodic, rmbcOpt)
    INTEGER, DIMENSION(3,2) :: PhysicalRegion
    TYPE(InfoDef) :: Info
    INTEGER :: nOverlaps, location
    INTEGER, DIMENSION(:,:), POINTER :: offsets
    INTEGER, DIMENSION(:,:,:), POINTER :: mSs
    INTEGER :: rmbc, i, j, k, dir, edge
    INTEGER, DIMENSION(27,3) :: MaxOffsets
    INTEGER, DIMENSION(27,3,2) :: MaxmSs
    INTEGER, DIMENSION(3,2) :: mO,iOffSet,ip
    INTEGER, DIMENSION(3) :: pOffSet
    INTEGER, OPTIONAL :: rmbcOpt
    LOGICAL, DIMENSION(:) :: lPeriodic
    IF (PRESENT(rmbcOpt)) THEN
       rmbc=rmbcOpt
    ELSE
       rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)        
    END IF
    NULLIFY(mSs)
    nOverlaps=0
    mO=1
    ioffset=0
    WHERE(lPeriodic(1:nDim)) ioffset(1:nDim,2)=nperiodic_overlaps(1:nDim)
    ioffset(1:nDim,1)=-ioffset(1:nDim,2)
    DO i=ioffset(1,1),ioffset(1,2)
       DO j=ioffset(2,1),ioffset(2,2)
          DO k=ioffset(3,1),ioffset(3,2)
             pOffSet=(/i,j,k/)*(levels(Info%level)%mx(:)) !Physical offset for info structure
             IF (location == IEVERYWHERE) THEN
                mO(1:nDim,1)=max(1-rmbc,PhysicalRegion(1:nDim,1)-(Info%mGlobal(1:nDim,1)+pOffset(1:nDim))+1)
                mO(1:nDim,2)=min(Info%mX(1:nDim)+rmbc,PhysicalRegion(1:nDim,2)-(Info%mGlobal(1:nDim,1)+pOffset(1:nDim))+1)
                !                 write(*,'(A,7I)') 'A', MPI_ID, mO
                !                 IF (MPI_id == 3) THEN
                !                    write(*,'(A,7I)') 'B', PhysicalRegion
                !                    write(*,'(A,7I)') 'B', info%mGlobal
                !                    write(*,'(A,7I)') 'B', mO
                !                 END IF
                IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                   nOverlaps=nOverlaps+1
                   MaxMSs(nOverlaps,:,:)=mO(:,:)
                   MaxOffsets(nOverlaps,:)=pOffset
                END IF
             ELSEIF (location == IBOUNDARIES) THEN
                DO dir=1,nDim
                   DO edge=1,2                    
                      IF (GhostOverlap(Info, dir, edge, ip)) THEN
                         mO(1:nDim,1)=max(ip(1:nDim,1),PhysicalRegion(1:nDim,1)-(Info%mGlobal(:,1)+pOffset(1:nDim))+1)
                         mO(1:nDim,2)=min(ip(1:nDim,2),PhysicalRegion(1:nDim,2)-(Info%mGlobal(1:nDim,1)+pOffset(1:nDim))+1)
                         IF (ALL(mO(1:nDim,2) >= mO(1:nDim,1))) THEN
                            nOverlaps=nOverlaps+1
                            MaxMSs(nOverlaps,:,:)=mO(:,:)
                            MaxOffsets(nOverlaps,:)=pOffset
                         END IF
                      END IF
                   END DO
                END DO
             ELSE
                WRITE(*,*) "unrecognized location in data_declarations.f90 - stopping"
             END IF

          END DO
       END DO
    END DO
    IF (nOverlaps > 0) THEN
       ALLOCATE(MSs(nOverlaps,3,2),offsets(nOverlaps,3))
       MSs=MaxMSs(1:nOverlaps,:,:)
       offsets=MaxOffsets(1:nOverlaps,:)
    END IF
  END SUBROUTINE CalcCellOverlaps

  !> @}

  !> @name Geometric functions for grids
  !! @{

  !> Logical function that determines whether two boxes overlap.
  !! @param mGlobal1 bounds of box 1
  !! @param mGlobal2 bounds of box 2
  LOGICAL FUNCTION BoxOverlap(mGlobal1,mGlobal2)
    INTEGER, DIMENSION(3,2) :: mGlobal1, mGlobal2
    BoxOverlap= ALL(mGlobal1(1:nDim,2) >= mGlobal2(1:nDim,1) .AND. &
         mGlobal1(1:nDim,1) <= mGlobal2(1:nDim,2))
  END FUNCTION BoxOverlap

  !> Logical function that determines whether an info structure's ghost region is in a domain ghost region.
  !! @param Info Info structure
  !! @param dim Dimension to check for
  !! @param edge Edge to check (1=left, 2=right)
  !! @param ip indices of cells that are in the domain ghost region.
  FUNCTION GhostOverlap(Info, dim, edge,ip)
    TYPE(InfoDef) :: Info
    INTEGER, DIMENSION(3,2) :: ip, lGmGlobal
    INTEGER :: dim
    INTEGER :: edge
    LOGICAL :: GhostOverlap
    INTEGER :: level, rmbc
    level=Info%level
    rmbc=levels(level)%gmbc(levels(level)%step)
    ip=1

    !    ip(1:dim-1,1)=1  !No need to include corner ghost cells multiple times
    !    ip(dim:ndim,1)=1-rmbc
    ip(1:nDim,1)=1-rmbc
    !    ip(1:dim-1,2)=Info%mX(1:dim-1)
    !    ip(dim:nDim,2)=Info%mX(dim:nDim)+rmbc
    ip(1:nDim,2)=Info%mX(1:nDim)+rmbc
    lGmGlobal(:,1)=GmGlobal(:,1)
    lGmGlobal(:,2)=GmGlobal(:,2)*PRODUCT(levels(0:level-1)%CoarsenRatio)  
    IF (edge == 1) THEN
       ip(dim,2)=lGmGlobal(dim,1)-Info%mGlobal(dim,1)  !first cell on left boundary
       !      nCells=start-(1-rmbc)+1
    ELSE
       ip(dim,1)=(lGmGlobal(dim,2)+1)-(Info%mGlobal(dim,1)-1) !first ghost cell on right boundary
    END IF
    GhostOverlap=ALL(ip(:,2) >= ip(:,1))     
  END FUNCTION GhostOverlap


  !> This function returns the bounds of a box in the next higher amr level
  !! @param mGlobal bounds of box
  !! @param level current level of box
  FUNCTION LevelUp(mGlobal,level, levelnew_opt)
    INTEGER, DIMENSION(3,2) :: mGlobal, LevelUp
    INTEGER :: level, n, levelnew
    INTEGER, OPTIONAL :: levelnew_opt
    IF (PRESENT(levelnew_opt)) THEN
       levelnew=levelnew_opt
    ELSE
       levelnew=level+1
    END IF
    LevelUp=mGlobal
    DO n=level, levelnew-1
       LevelUp(1:nDim,1)=(LevelUp(1:nDim,1)-1)*levels(n)%CoarsenRatio+1
       LevelUp(1:nDim,2)=LevelUp(1:nDim,2)*levels(n)%CoarsenRatio
    END DO
  END FUNCTION LevelUp

  !> This function returns the bounds of a box in the next higher amr level
  !! @param mGlobal bounds of box
  !! @param level current level of box
  FUNCTION LevelDown(mGlobal,level,levelnew_opt)
    INTEGER, DIMENSION(3,2) :: mGlobal, LevelDown
    INTEGER :: level, n, levelnew
    INTEGER, OPTIONAL :: levelnew_opt
    IF (PRESENT(levelnew_opt)) THEN
       levelnew=levelnew_opt
    ELSE
       levelnew=level-1
    END IF
    LevelDown=mGlobal
    DO n=level, levelnew+1, -1
       LevelDown(1:nDim,1)=(LevelDown(1:nDim,1)-1)/levels(n-1)%CoarsenRatio+1
       LevelDown(1:nDim,2)=(LevelDown(1:nDim,2)-1)/levels(n-1)%CoarsenRatio+1
    END DO
  END FUNCTION LevelDown


  FUNCTION MapToLevel(index, a, b)
    INTEGER :: MapToLevel(2) !Cell indices in level b's mGlobal space
    INTEGER :: index !Cell index in level a's mGlobal space
    INTEGER :: a !current level
    INTEGER :: b !level to map to
    IF (a == b) THEN
       MapToLevel=index
    ELSEIF (a < b) THEN !need to determine bounds of child cells...
       MapToLevel(1)=(index-1)*PRODUCT(levels(a:b-1)%CoarsenRatio)+1
       MapToLevel(2)=(index)*PRODUCT(levels(a:b-1)%CoarsenRatio)
    ELSE !a > b
       MapToLevel(:)=(index-1)/PRODUCT(levels(b:a-1)%CoarsenRatio)+1
    END IF
  END FUNCTION MapToLevel


  !> This function returns the bounds of child within a parent
  !! @param mGlobal bounds of child box
  !! @param ParentmGlobal bounds of parent box
  !! @param level level of parent box
  FUNCTION GetMBounds(mGlobal, ParentmGlobal, level)
    INTEGER, DIMENSION(3,2) :: GetmBounds, mGlobal, ParentmGlobal
    INTEGER :: level
    GetmBounds=leveldown(mGlobal, level)-spread(ParentmGlobal(:,1)-1,2,2)
  END FUNCTION GetMBounds


  !> This function returns the bounds of child within a parent
  !! @param mGlobal bounds of child box
  !! @param ParentmGlobal bounds of parent box
  !! @param level level of parent box
  FUNCTION GetParentmGlobal(mGlobal, mBounds, level)
    INTEGER, DIMENSION(3,2) :: mBounds, mGlobal, GetParentmGlobal
    INTEGER :: level
    GetparentmGlobal=leveldown(mGlobal, level)-spread(mBounds(:,1)-1,2,2)
 END FUNCTION GetParentmGlobal


  !> This function stretches bounds in each direction by dist
  !! @param mB bounds of box
  !! @param dist distance to stretch
  FUNCTION stretch(mB,dist)
    INTEGER, DIMENSION(3,2) :: stretch, mB
    INTEGER :: dist
!    stretch(nDim+1:3,:)=1
    stretch=1
    stretch(1:nDim,1)=mB(1:nDim,1)-dist
    stretch(1:nDim,2)=mB(1:nDim,2)+dist
  END FUNCTION stretch

  !> This function stretches the bounds of aux fields in each direction by dist
  !! @param mB bounds of aux fields
  !! @param dist distance to stretch
  FUNCTION stretchaux(mB,dist)
    INTEGER, DIMENSION(3,2) :: stretchaux, mB
    INTEGER :: dist
!    stretchaux(nDim+1:3,:)=1
    stretchaux=1
    stretchaux(1:nDim,1)=mB(1:nDim,1)-dist
    stretchaux(1:nDim,2)=mB(1:nDim,2)+dist+1
  END FUNCTION stretchaux


  FUNCTION MapBoxToInfo(xB, Info, nghost)
     REAL(KIND=qPREC) :: xB(3,2)
     INTEGER :: MapBoxToInfo(3,2)
     INTEGER :: nGhost
     TYPE(InfoDef) :: Info
     MapBoxToInfo=ceiling((xb-spread(Info%xbounds(:,1),2,2))/levels(Info%level)%dx)
     MapBoxToInfo(1:nDim,1)=max(MapBoxToInfo(1:nDim,1), 1-nghost)
     MapBoxToInfo(1:nDim,2)=min(MapBoxToInfo(1:nDim,2), Info%mX(1:nDim)+nGhost)
  END FUNCTION MapBoxToInfo

  FUNCTION MapBoxToLevel(xB, level, nghost)
     INTEGER :: level, nghost
     INTEGER :: MapBoxToLevel(3,2)
     REAL(KIND=qPREC) :: xB(3,2)
     MapBoxToLevel=ceiling(xb-spread(GxBounds(:,1),2,2))/levels(level)%dx
     MapBoxToLevel(1:nDim,1)=max(MapBoxToLevel(1:nDim,1), 1-nGhost)
     MapBoxToLevel(1:nDim,2)=min(MapBoxToLevel(1:nDim,2), levels(level)%mX(1:nDim)+nGhost)
     write(*,*) xb, GxBounds, level, nGhost, MapBoxTolevel
  END FUNCTION MapBoxToLevel


  FUNCTION CellPos(Info,i,j,k)
     TYPE(InfoDef) :: Info
     REAL(KIND=qPREC), DIMENSION(3) :: CellPos
     INTEGER :: i,j,k
!     CellPos=Info%XBounds(:,1)+levels(Info%level)%dx*REAL((/i,j,k/)-half)
     CellPos=Info%XBounds(:,1)+merge(levels(Info%level)%dx*REAL((/i,j,k/)-half), (/0d0,0d0,0d0/), nDim >= (/1,2,3/))
  END FUNCTION CellPos

  FUNCTION PosCell(Info,x,y,z)
    TYPE(InfoDef) :: Info
     INTEGER, DIMENSION(3) :: PosCell
     REAL(KIND=qPREC) :: x,y,z
     PosCell(:)=ceiling(((/x,y,z/)-GxBounds(:,1))/levels(Info%level)%dx) - Info%mGlobal(:,1) + 1
!     write(*,'(A,4I)') 'poscell', poscell
!     write(*,'(A,4I)') 'xyz',x,y,z
!     write(*,'(A,4I)') 'mglobal', Info%mGlobal(:,1)
!     STOP

     PosCell(nDim+1:)=1
  END FUNCTION PosCell

!> @}


  SUBROUTINE ProlongateCellCenteredData(pdata, cdata, r, nGhost, method)
     REAL(KIND=qPREC), DIMENSION(:,:,:) :: pdata !parent data to prolongate (includes any ghost zones)
     REAL(KIND=qPREC), DIMENSION(:,:,:) :: cdata !child data to populate
     REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:) :: sided, slopes!work arrays for storing slopes
     INTEGER :: method !prolongation method
     INTEGER :: r !Refinement Ratio
     INTEGER :: nGhost !number of ghost parent cells available
     INTEGER, DIMENSION(3,2) :: ic !work array for child cells
     INTEGER, DIMENSION(3,2) :: ip, iq, ir, is !work arrays for parent cells
     REAL(KIND=xPrec), DIMENSION(:), ALLOCATABLE :: dx,hdx,hdx2 !cell spacing arrays
     INTEGER :: l(3) !offset array for filling in child cells
     INTEGER :: i, d, n !loop counters
     INTEGER :: nZones !number of zones to do method correctly

     !Calculate ip to prolongate NOT including ghost zones
     !ic has child cells

     ip=1
     ip(1:nDim,1)=nGhost+1
     ic=1
     DO i=1,nDim
        ip(i,2)=size(pdata, i) - nGhost
        ic(i,2)=size(cdata, i)
     END DO
     
     !Sanity check that number of ghost zones, refinement ratios, and sizes of arrays are consistent
     IF (ANY((ip(1:nDim,2)-ip(1:nDim,1)+1)*r /= ic(1:nDim,2)-ic(1:nDim,1)+1)) THEN
        write(*,*) 'mismatch between size of arrays, number of parent ghost zones, and refinement ratio.'
        write(*,*) 'parent bounds = ', ip(1:nDim,:)
        write(*,*) 'child bounds = ', ic(1:nDim,:)
        STOP
     END IF
     

     !First do constant prolongation regardless
     DO i=0,r**nDim-1  ! i loops over all r**nDim child subcells
        DO n=1,nDim    ! maps integer i to particular n-Dimensional offset array l
           l(n)=MOD(i/r**(n-1),r)
        END DO
        ic(1:nDim,1)=ic(1:nDim,1)+l(1:nDim)     !shift 
        ic(1:nDim,2)=ic(1:nDim,2)-r+1+l(1:nDim) !child cells
        cdata(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r) = &
             pdata(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))
        ic(1:nDim,1)=ic(1:nDim,1)-l(1:nDim)     !undo shift
        ic(1:nDim,2)=ic(1:nDim,2)+r-1-l(1:nDim)
     END DO

     IF (method /= CONSTANT_INTERP) THEN
        !need to calculate slopes etc.

        !first calculate deltas for prolongation to child cell centers in parent cell width units.
        ALLOCATE(dx(0:r-1), hdx(0:r-1),hdx2(0:r-1))
        DO i=0,r-1
           dx(i)= 0.5d0*( (2.d0*i+1.d0)/REAL(r,KIND=qprec) - 1.d0)
           hdx(i)=0.5d0*dx(i)
           hdx2(i)=dx(i)*hdx(i)
        END DO
        
        nZones=GetnGhost(method)
        
        ! Allocate work array for cell centered slopes
        ALLOCATE(slopes(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2)))

        ! Loop through each dimension and add slopes to child data
        DO d=1,nDim

           !ip is 1 sided slopes we need to calculate 
           ip(d,2)=ip(d,2)+1                   
           ALLOCATE(sided(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2)))

           !iq is 1 sided bounds we can AND need to calculate
           iq=ip 
           iq(d,1)=max(ip(d,1),1+nZones)
           iq(d,2)=min(ip(d,2),size(pdata,d)-nZones)
           !ir is offset along dimension
           ir=iq
           ir(d,:)=ir(d,:)-1
           sided(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2)) = &
                pdata(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2))-&
                pdata(ir(1,1):iq(1,2),ir(2,1):iq(2,2),ir(3,1):ir(3,2))


           !now ir is bounds we need but could not calculate for lower boundary
           ir(d,1)=ip(d,1)
           ir(d,2)=iq(d,1)-1

           IF (ir(d,2) >= ir(d,1)) THEN
              !is is slab to copy from
              is=ir
              is(d,:)=iq(d,1)
              sided(ir(1,1):ir(1,2), ir(2,1):ir(2,2), ir(3,1):ir(3,2)) = &
                spread(sum(sided(is(1,1):is(1,2), is(2,1):is(2,2), is(3,1):is(3,2)), d), d, ir(d,2)-ir(d,1)+1)
           END IF


           !now ir is bounds we need to extrapolate for upper boundary
           ir(d,1)=iq(d,2)+1
           ir(d,2)=ip(d,2)

           IF (ir(d,2) >= ir(d,1)) THEN
              !is is slab to copy from
              is=ir
              is(d,:)=iq(d,2)
              sided(ir(1,1):ir(1,2), ir(2,1):ir(2,2), ir(3,1):ir(3,2)) = &
                spread(sum(sided(is(1,1):is(1,2), is(2,1):is(2,2), is(3,1):is(3,2)), d), d, ir(d,2)-ir(d,1)+1)
           END IF

           ! now we shrink ip to refer to cell centered slopes we need (and can calculate from sided slopes)
           ip(d,2)=ip(d,2)-1
           
           SELECT CASE (method)
              
           CASE(CONSTANT_INTERP, MINMOD_INTERP, SUPERBEE_INTERP, VANLEER_INTERP, MC_INTERP, LINEAR_INTERP)
              
              ! iq is offset for calculated centered limited slopes from sided slopes
              iq=ip
              iq(d,:)=ip(d,:)+1
              
              slopes(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2)) = limiter( &
                   sided(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2)), &
                   sided(iq(1,1):iq(1,2), iq(2,1):iq(2,2), iq(3,1):iq(3,2)), method)
              
              
              DO i=0,r**nDim-1  ! i loops over all r**nDim child subcells
                 DO n=1,nDim    ! maps integer i to particular n-Dimensional offset array l
                    l(n)=MOD(i/r**(n-1),r)
                 END DO
                 ic(1:nDim,1)=ic(1:nDim,1)+l(1:nDim)
                 ic(1:nDim,2)=ic(1:nDim,2)-r+1+l(1:nDim)
                 cdata(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r) = &
                      cdata(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r) + &
                      dx(l(d))*slopes(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))
                 ic(1:nDim,1)=ic(1:nDim,1)-l(1:nDim)
                 ic(1:nDim,2)=ic(1:nDim,2)+r-1-l(1:nDim)
              END DO
           CASE(PARABOLIC_INTERP)
              iq=ip
              iq(d,:)=ip(d,:)+1
              DO i=0,r**nDim-1  ! i loops over all r**nDim child subcells
                 DO n=1,nDim    ! maps integer i to particular n-Dimensional offset array l
                    l(n)=MOD(i/r**(n-1),r)
                 END DO
                 ic(1:nDim,1)=ic(1:nDim,1)+l(1:nDim)
                 ic(1:nDim,2)=ic(1:nDim,2)-r+1+l(1:nDim)
                 cdata(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r) = &
                      cdata(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r) + &
                      (hdx(l(d))+hdx2(l(d)))*sided(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2)) + &
                      (hdx(l(d))-hdx2(l(d)))*sided(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))
                 ic(1:nDim,1)=ic(1:nDim,1)-l(1:nDim)
                 ic(1:nDim,2)=ic(1:nDim,2)+r-1-l(1:nDim)
              END DO
           
           END SELECT
           DEALLOCATE(sided)
        END DO
        DEALLOCATE(slopes)
        DEALLOCATE(dx,hdx,hdx2)
     END IF
     
  END SUBROUTINE ProlongateCellCenteredData

  FUNCTION expand(mB, nGhost)
     INTEGER, DIMENSION(3,2) :: expand, mB
     INTEGER :: nGhost
     expand=mB
     expand(1:nDim,1)=mB(1:nDim,1)-nGhost
     expand(1:nDim,2)=mB(1:nDim,2)+nGhost
  END FUNCTION expand
END MODULE DataDeclarations




