!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    data_info_ops.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file data_info_ops.f90
!! @brief Main file for module DataInfoOps

!> @defgroup DataInfoOps Data Info Operations
!! @brief Module for performing AMR related info operations
!! @ingroup DataOps

!> Module for performing AMR related info operations
!! @ingroup DataInfoOps

MODULE DataInfoOps
   USE GlobalDeclarations
   USE DataDeclarations
   USE PhysicsDeclarations
   USE Boundary
   USE HyperbolicDeclarations
   !  USE EllipticDeclarations
   IMPLICIT NONE

   !Pair-wise data operations
   PUBLIC :: ProlongateParentData, ApplyOverlap, ApplyChildData, ApplyInitialChildData, SyncFlux, ApplyGenericOverlap

   !Pair-wise data sub-operations (because of periodic bounds - and multiple stencil locations for fluxes and emfs)
   PUBLIC :: ApplySingleOverlap, ApplySingleAuxOverlap, ApplySingleCostMapOverlap, SyncSingleFlux, SyncSingleEmf  !Pairwise intralevel data operations ('communication')

   !Sincle grid data operations
   PUBLIC :: ProlongationFixup, AccumulateFlux, RestrictionFixup, CoarsenDataForParent, CoarsenInitialDataForParent, UpdateTimeDeriv, ClearFixupFlux, ClearParentFixup, ClearChildFixup

   PUBLIC :: ChildMaskOverlap, UpdateAux, UpdateChildMask, UpdateSelfChildMask

   PUBLIC :: GetChildCosts, GetSubTreeCost, GetMyCosts, NewSubGrids

CONTAINS

   !> @name PairWiseInfoOps Pair-wise Info Operations
   !! @{


   !> @brief Prolongates parents data onto new child grids
   !! @param Parent - Parent info structure
   !! @param Info - Child info structure
   SUBROUTINE ProlongateParentData(Parent,Info)
      USE SLOPELIM
      TYPE(InfoDef) :: Info
      TYPE(InfoDef) :: Parent
      INTEGER, DIMENSION(3,2) :: mB
      INTEGER r,n,rmbc,iError,i,j,m,nd,mbc,p,k
      INTEGER l(3),ic(3,2),ip(3,2),iq(3,2)
      REAL (KIND=xPrec) :: fact, dxp, dxc
      REAL (KIND=qPrec), ALLOCATABLE, DIMENSION(:,:,:,:) :: dqf,dqb&
           &,dauxf,dauxb
      REAL(KIND=xPrec), DIMENSION(:), ALLOCATABLE :: dx,hdx, hdx2
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: q, aux
      TYPE(InfoDef), POINTER :: InfoCopy
      !    Info%CostPerCell=Parent%CostPerCell
      !    Info%OldCostPerCell=Parent%CostPerCell

      IF (Info%level > 0) THEN
         IF (MaintainAuxArrays) Info%aux=UNDEFINED
         r=levels(Info%level-1)%CoarsenRatio
         mB=Info%mBounds
         ! Linear interpolation of coarse grid values into fine grid
         mbc=levels(Info%level-1)%pmbc!ceiling(real(rmbc)/real(r))!levels(Info%level-1)%CoarsenRatio
         rmbc=mbc*r !levels(Info%level)%ombc(1)
         ip=1
         ip(1:nDim,1)=mB(1:nDim,1)-mbc
         ip(1:nDim,2)=mB(1:nDim,2)+mbc
         ic=1        
         ic(1:nDim,1)=1-rmbc
         ic(1:nDim,2)=Info%mX(1:nDim)+rmbc
         DO m=1,nProlongate
            CALL ProlongateCellCenteredData(Parent%qChild(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2), m), &
                 Info%q(ic(1,1):ic(1,2), ic(2,1):ic(2,2), ic(3,1):ic(3,2), ProlongateFields(m)), &
                 r, 0, InterpMethod(ProlongateFields(m)))
         END DO

         IF (MaintainAuxArrays) THEN
            dxp=levels(Info%level-1)%dx
            dxc=levels(Info%level)%dx

            ALLOCATE(dx(0:r-1), hdx(0:r-1),hdx2(0:r-1))
            DO i=0,r-1
               dx(i)= 0.5d0*( (2.d0*i+1.d0)/REAL(r,KIND=qprec) - 1.d0)
               hdx(i)=0.5d0*dx(i)
               hdx2(i)=dx(i)*hdx(i)
            END DO

            DO i=1,nDim
               DO j=0,r**nDim-1
                  IF (MOD(j/r**(i-1),r)==1) CYCLE
                  DO n=1,nDim
                     l(n)=MOD(j/r**(n-1),r)  
                  END DO
                  ic(1:nDim,1)=1+l(1:nDim)-rmbc; ic(1:nDim,2)=Info%mX(1:nDim)-r+1+l(1:nDim)+rmbc;ic(i,2)=ic(i,2)+2
                  ip(1:nDim,1)=mB(1:nDim,1)-mbc; ip(1:nDim,2)=mB(1:nDim,2)+mbc;ip(i,2)=ip(i,2)+1
                  Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,i) =     &
                       Parent%auxChild(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,i)
               END DO
            END DO
            IF (ANY(InterpMethod(auxfields) /= SPREAD(CONSTANT_INTERP,1,naux))) THEN

               ip(1:nDim,1)=mB(1:nDim,1)-mbc; ip(1:nDim,2)=mB(1:nDim,2)+mbc+1
               ALLOCATE(dauxf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1), &
                    dauxb(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1), &
                    STAT=iError)
               IF (iError/=0) THEN
                  PRINT *,'Error: Unable to allocate daux work arrays in InitFields.'
                  STOP
               END IF
               ! Add first order correction (linear interpolation)
               DO m=1,nDim !B-field direction
                  IF (InterpMethod(auxfields(m)) == CONSTANT_INTERP) CYCLE
                  DO n=1,nDim
                     IF (n==m) CYCLE !Don't calculate gradients normal to field
                     l=0; l(n)=1
                     ip(1:nDim,1) = mB(1:nDim,1)-mbc
                     ip(1:nDim,2) = mB(1:nDim,2)+mbc
                     ip(m,2)=ip(m,2)+1

                     ! Compute derivative of parent array data along direction n

                     ! Forward differences
                     ip(n,1) = mB(n,1)-mbc; ip(n,2)=ip(n,2)-1
                     iq(1:nDim,:)=ip(1:nDim,:)+Spread(l(1:nDim),2,2)
                     IF (iCylindrical==NoCyl   .or.   n>1) THEN
                        dauxf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) = &
                             Parent%auxChild(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) - &
                             Parent%auxChild(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),m)

                        ! Extrapolate forward difference to last cell along this direction
                        ip(n,1)=mB(n,2)+mbc; ip(n,2)=mB(n,2)+mbc
                        iq(1:nDim,:)=ip(1:nDim,:)-Spread(l(1:nDim),2,2)
                        dauxf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) =  &
                             dauxf(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),1)

                        ! Backward differences
                        ip(n,1)=mB(n,1)-mbc+1;ip(n,2) = mB(n,2)+mbc
                        iq(1:nDim,:)=ip(1:nDim,:)-Spread(l(1:nDim),2,2)
                        dauxb(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) = &
                             Parent%auxChild(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),m) -        &
                             Parent%auxChild(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),m)

                        ! Extrapolate backward difference to first cell along this direction
                        ip(n,1)=mB(n,1)-mbc; ip(n,2)=mB(n,1)-mbc
                        iq(1:nDim,:)=ip(1:nDim,:)+Spread(l(1:nDim),2,2)
                        dauxb(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) =  &
                             dauxb(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),1)
                     ELSE
                        DO i=ip(1,1),ip(1,2)
                           dauxf(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) = &
                                (  Parent%auxChild(i+1,iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) - &
                                Parent%auxChild(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),m)   )*&
                                (Parent%xBounds(1,1)+i*dxp)

                        END DO
                        ! Extrapolate forward difference to last cell along this direction
                        i=mB(n,2)+mbc
                        !ip(n,1)=mB(n,2)+mbc; ip(n,2)=mB(n,2)+mbc
                        iq(2:nDim,:)=ip(2:nDim,:)-Spread(l(2:nDim),2,2)
                        dauxf(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) =  &
                             (  Parent%auxChild(i,iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) - &
                             Parent%auxChild(i-1,ip(2,1):ip(2,2),ip(3,1):ip(3,2),m)   )*&
                             (Parent%xBounds(1,1)+i*dxp)


                        ! Backward differences
                        ip(n,1)=mB(n,1)-mbc+1;ip(n,2) = mB(n,2)+mbc
                        iq(1:nDim,:)=ip(1:nDim,:)-Spread(l(1:nDim),2,2)
                        DO i=ip(1,1),ip(1,2)
                           dauxb(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) = &
                                (   Parent%auxChild(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),m) -        &
                                Parent%auxChild(i-1,iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) )*&
                                (Parent%xBounds(1,1)+(i-1)*dxp)
                        END DO
                        ! Extrapolate backward difference to first cell along this direction
                        i=mB(n,1)-mbc
                        !ip(n,1)=mB(n,1)-mbc; ip(n,2)=mB(n,1)-mbc
                        iq(2:nDim,:)=ip(2:nDim,:)+Spread(l(2:nDim),2,2)
                        dauxb(i,ip(2,1):ip(2,2),ip(3,1):ip(3,2),1) =  &
                             (   Parent%auxChild(i+1,ip(2,1):ip(2,2),ip(3,1):ip(3,2),m) -        &
                             Parent%auxChild(i,iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) )*&
                             (Parent%xBounds(1,1)+i*dxp)

                     ENDIF

                     SELECT CASE(InterpMethod(AuxFields(m)))
                     CASE(CONSTANT_INTERP, MINMOD_INTERP, SUPERBEE_INTERP, VANLEER_INTERP, MC_INTERP, LINEAR_INTERP)
                        ! Choose derivative to be used; choice is stored in dqf
                        dauxf=limiter(dauxf,dauxb,InterpMethod(AuxFields(m)))


                        ! Add correction term, taking into account offset of child cell center from
                        ! parent cell center
                        ip(n,1) = mB(n,1)-mbc
                        ip(n,2) = mB(n,2)+mbc

                        DO j=0,r**nDim-1
                           IF (MOD(j/r**(m-1),r)==1) CYCLE
                           DO nd=1,nDim
                              l(nd)=MOD(j/r**(nd-1),r)  
                           END DO
                           ic(1:nDim,1)=1+l(1:nDim)-rmbc; ic(1:nDim,2)=Info%mX(1:nDim)-r+1+l(1:nDim)+rmbc;ic(m,2)=ic(m,2)+2
                           IF (iCylindrical==NoCyl   .or.   n>1) THEN
                              Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) =     &
                                   Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) + &
                                   dx(l(n))*dauxf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1)
                           ELSE
                              p=ip(1,1)
                              DO i=ic(1,1),ic(1,2),r
                                 Info%aux(i,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) =     &
                                      Info%aux(i,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) + &
                                      dx(l(n))*dauxf(p,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1)/&
                                      (Info%xBounds(1,1)+(REAL(i)-half)*dxc)
                                 p=p+1
                              END DO
                           END IF
                        END DO
                     CASE(PARABOLIC_INTERP)
                        IF (iCylindrical>NoCyl   .AND.   n==1) THEN
                           print*,'data_info_ops.f90: Parabolic interpolation of bz + cylindrical terms not &
                                supported yet' ; stop
                        ELSE
                           Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) =     &
                                Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,m) + &
                                (hdx(l(n))+hdx2(l(n)))*dauxf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1) + &
                                (hdx(l(n))-hdx2(l(n)))*dauxb(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1)
                        END IF
                     END SELECT
                  END DO
               END DO
               DEALLOCATE(dauxb,dauxf)
            END IF
            DEALLOCATE(dx,hdx,hdx2) 
         END IF
 
         !       write(*,*) '==========Info%q=
         !       write(*,*) Info%q(:,:,:,1)
         !       write(*,*) Info%aux(:,:,:,1)
         !       write(*,*) Info%aux(:,:,:,2)

         IF (.NOT. lRegridLevel(Info%level)) THEN
            ALLOCATE(InfoCopy)
            InfoCopy=Info
            InfoCopy%q=>q
            InfoCopy%aux=>aux
            CALL ApplyOverlap(Info, InfoCopy, Info%level)
            DEALLOCATE(InfoCopy%q)
            IF (MaintainAuxArrays) DEALLOCATE(InfoCopy%aux)
            DEALLOCATE(InfoCopy)
         END IF

      END IF

   END SUBROUTINE ProlongateParentData



   !> @brief Applies Overlaps to new grids
   !! @param Info - New info structure
   !! @param Source - Old info structure
   !! @param n - level
   SUBROUTINE ApplyOverlap(Info,Source,n)
      ! Transfers data from Source within mbc cells of Info
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,n,dir,level
      level=Info%level
      CALL CalcOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,level,lHydroPeriodic,0)

      !    DO i=1,nOverlaps
      !       mT=mTs(i,:,:)
      !       mS=mSs(i,:,:)
      !       IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
      !       CALL ApplySingleCostMapOverlap(Info,Source,mS,mT)
      !    END DO

      IF (nOverlaps > 0) THEN
         DEALLOCATE(mTs,mSs)
         NULLIFY(mTs,mSs)
      END IF

      IF (n >-1) THEN
         CALL CalcOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,level,lHydroPeriodic, levels(level)%gmbc(levels(level)%step))

         DO i=1,nOverlaps
            mT=mTs(i,:,:)
            mS=mSs(i,:,:)
            IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
            CALL ApplySingleOverlap(Info, Source, mS, mT, GCopyFields)
         END DO

         IF (nOverlaps > 0) THEN
            DEALLOCATE(mTs,mSs)
            NULLIFY(mTs,mSs)
         END IF

         !Solution for potential has to come from base grid...
         !But we don't just want to prolongate solution onto finer grid - so we need to resolve for finer grid potential after regridding?  It should agree with old grid solution although that was calculated using the coarse phi-dot and the fine density field...  Probably should solve for the potential after regridding using the new potential

!         IF (n >= BaseLevel .AND. (n == 0 .OR. levels(n)%step > 1) .AND. EGVars > 0) THEN 

            ! Don't copy overlaps from BaseLevel since this will have been done after last elliptic solve
            ! If BaseLevel is -1 then level n grids will be new and we want to copy old values - before the first step
            ! Otherwise we only want to copy old values if we are not on the first step
         IF (n > BaseLevel .AND. ((n == 0 .AND. levels(n)%step <= 1) .OR. levels(n)%step > 1) .AND. EGVars > 0) THEN

            CALL CalcOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,level,lEllipticPeriodic,levels(level)%egmbc(levels(level)%step))
            DO i=1,nOverlaps
               mT=mTs(i,:,:)
               mS=mSs(i,:,:)
               IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
               CALL ApplySingleOverlap(Info, Source, mS, mT, EGCopyFields)
            END DO

            IF (nOverlaps > 0) THEN
               DEALLOCATE(mTs,mSs)
               NULLIFY(mTs,mSs)
            END IF
         END IF

         IF (MaintainAuxArrays) THEN
            DO dir=1,nDim
               CALL CalcAuxOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,level,dir,lHydroPeriodic)

               DO i=1,nOverlaps
                  mT=mTs(i,:,:)
                  mS=mSs(i,:,:)

                  IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE

                  CALL ApplySingleAuxOverlap(Info, Source, mS, mT,dir)

               END DO

               IF (nOverlaps > 0) THEN
                  DEALLOCATE(mTs,mSs)
                  NULLIFY(mTs,mSs)
               END IF

            END DO
         END IF
      END IF
      !    DO i=1, size(Info%q,4)
      !       write(*,*) "finished applying overlaps",i, minval(Info%q(:,:,:,i)), maxval(Info%q(:,:,:,i))
      !    END DO
   END SUBROUTINE ApplyOverlap


   !> @brief Applies Overlaps to new grids
   !! @param Info New info structure
   !! @param Source Old info structure
   !! @param n level
   !! @param fields Indices of info%q to ghost
   !! @param nghost Number of ghost zones to transfer
   SUBROUTINE ApplyGenericOverlap(Info,Source,n,fields,nghost,lPeriodic)
      ! Transfers data from Source within mbc cells of Info
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,n,dir
      INTEGER, DIMENSION(:) :: fields
      INTEGER :: nghost
      LOGICAL, DIMENSION(3) :: lPeriodic
      CALL CalcOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,Info%level,lPeriodic, nghost)
      IF (nOverlaps > 0) THEN
         DO i=1,nOverlaps
            mT=mTs(i,:,:)
            mS=mSs(i,:,:)
            IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
            CALL ApplySingleoverlap(Info,Source,mS,mT,fields)
         END DO
         DEALLOCATE(mTs,mSs)
         NULLIFY(mTs,mSs)
      END IF
   END SUBROUTINE ApplyGenericOverlap

   !> @brief Applies single overlap of cell-centered quantity %q
   !! @param Info target info structure
   !! @param Source source info structure
   !! @param mS indices of overlap region in source grid
   !! @param mT indices of overlap region in target grid
   !! @param fields Indices of info%q to ghost
   SUBROUTINE ApplySingleOverlap(Info, Source, mS, mT, fields)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, DIMENSION(:) :: fields
      Info%q(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),Fields)=&
           Source%q(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),Fields)

   END SUBROUTINE ApplySingleOverlap

   !> @brief Applies single overlap of aux field %aux
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param dir - Component of aux field to copy
   SUBROUTINE ApplySingleAuxOverlap(Info, Source, mS, mT, dir)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER :: dir
      Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dir)= &
           Source%aux(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)
   END SUBROUTINE ApplySingleAuxOverlap

   !> @brief Applies single overlap of cell-centered quantity %q
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   SUBROUTINE ApplySingleCostMapOverlap(Info,Source,mS,mT)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS

      Info%costmap(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1)=&
           Source%costmap(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),1)

      !    Info%CostPerCell=Info%CostPerCell+(Source%CostPerCell-Info%OldCostPerCell)*PRODUCT(mT(1:nDim,2)-mT(1:nDim,1)+1)&
      !         &/PRODUCT(Info%mX(1:nDim))
   END SUBROUTINE ApplySingleCostMapOverlap

   !> @brief Applies restricted data from children
   !! @param Info - target info structure
   !! @param Child - source child info structure
   !! @param ChildID - ID of the child (stored in ChildMask)
   !! @param n - level
   SUBROUTINE ApplyChildData(Info,Child,ChildID,n)
      TYPE(InfoDef) :: Info, Child
      INTEGER, DIMENSION(3,2) :: mG,ip,iq,ir,mB
      INTEGER :: i,edge,m,ChildID,n
      TYPE(Boundaries), POINTER :: parentfixups, childfixups
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: parentfixup, childfixup

      mB=Child%mBounds

      !    Info%CostMap(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1) = &
      !         Child%ParentCostmap(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1)

      IF (n > -1) THEN
         ! First update cell centered quantities from children
         Info%q(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), RestrictFields) = &
              Child%qParent(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2), :)

         ! Then apply differences between parent's stored fluxes and child's parentfixup
         mG=1
         mG(1:nDim,2)=Info%mX(1:nDim)    
         parentfixups=>Info%childfixups(ChildID)%p
         childfixups=>Child%parentfixup
         DO i=1,nDim
            parentfixup=>parentfixups%side(i)%data
            childfixup=>childfixups%side(i)%data
            DO edge=1,2

               iq=mB
               ir=mB
               iq(i,:)=edge
               ir(i,:)=Child%mBounds(i,edge)            
               ir(i,:)=ir(i,:)+(-1)**edge !neighboring cell

               FORALL(m=1:nFlux)
                  WHERE(Info%ChildMask(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2)) < 1)
                     Info%q(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2),FluxFields(m)) = &
                          Info%q(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2),FluxFields(m)) + &
                          (-1)**edge*(childfixup(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),m) - &
                          parentfixup(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),m))
                  END WHERE
               END FORALL
               IF (Child%mBounds(i,edge)==mG(i,edge)) THEN
                  Info%fixupflux%side(i)%data(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),:) = &
                       childfixup(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),:)
               END IF
            END DO
         END DO

         IF (MaintainAuxArrays) THEN
            ! Second store child emf's to be differenced later
            SELECT CASE(ndim)
            CASE(2)
               Info%childemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)+1,1,1)=&
                    child%parentemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)+1,1,1)
            CASE(3)
               Info%childemf(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)+1,mB(3,1):mB(3,2)+1,1) = &
                    child%parentemf(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)+1,mB(3,1):mB(3,2)+1,1)
               Info%childemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)+1,2) = &
                    child%parentemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)+1,2)
               Info%childemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)+1,mB(3,1):mB(3,2)  ,3) = &
                    child%parentemf(mB(1,1):mB(1,2)+1,mB(2,1):mB(2,2)+1,mB(3,1):mB(3,2)  ,3)
            END SELECT
         END IF

      END IF
   END SUBROUTINE ApplyChildData

   !> @brief Applies restricted data from children
   !! @param Info - target info structure
   !! @param Child - source child info structure
   !! @param n - level
   SUBROUTINE ApplyInitialChildData(Info,Child,n)
      TYPE(InfoDef) :: Info, Child
      INTEGER, DIMENSION(3,2) :: ip,mB
      INTEGER :: i,n

      mB=Child%mBounds
      !    Info%CostMap(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), 1) = &
      !         Child%ParentCostmap(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), 1)

      IF (n > -1) THEN
         ! First update cell centered quantities from children
         Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), RestrictFields) = &
              Child%qParent(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), :)

         ! Then update B-fields from children (From now on we'll only update the fluxes(emfs) for the B-fields)
         IF (MaintainAuxArrays) THEN
            DO i=1,nAux
               ip=mb
               ip(i,2)=mb(i,2)+1
               Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i) = &
                    child%auxParent(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i)
            END DO
            CALL UpdateAux(Info, mB)
         END IF
      END IF
   END SUBROUTINE ApplyInitialChildData

   !> @brief Synchronizes Fluxes with neighboring grid
   !! @param Info - New info structure
   !! @param Source - neighboring info structure
   SUBROUTINE SyncFlux(Info,Source)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, DIMENSION(3) :: offset
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
      INTEGER, DIMENSION(:,:), POINTER :: Offsets
      INTEGER, DIMENSION(:), POINTER :: edges
      INTEGER :: nOverlaps,i,dir,edge


      DO dir=1,nDim
         CALL CalcFluxOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,edges,nOverlaps,Info%level,dir,lHydroPeriodic)
         DO i=1,nOverlaps
            mT=mTs(i,:,:)
            mS=mSs(i,:,:)            
            edge=edges(i)
            IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
            CALL SyncSingleFlux(Info, Source, mT, mS,dir,edge)
         END DO
         IF (nOverlaps > 0) THEN
            DEALLOCATE(mTs,mSs,edges)
            NULLIFY(mTs,mSs,edges)
         END IF
      END DO

      IF (MaintainAuxArrays) THEN
         DO dir=1,nEMF
            CALL CalcEmfOverlaps(Info%mGlobal,Source%mGlobal,mTs,mSs,nOverlaps,offsets,Info%level,EmfDir(dir), lHydroPeriodic)
            DO i=1,nOverlaps
               mT=mTs(i,:,:)
               mS=mSs(i,:,:)
               offset=offsets(i,:)
               IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE
               CALL SyncSingleEMF(Info, Source, mT, mS, offset,EmfDir(dir))
            END DO
            IF (nOverlaps > 0) THEN
               DEALLOCATE(mTs,mSs, offsets)
               NULLIFY(mTs,mSs, offsets)
            END IF
         END DO
      END IF

      !    IF (nEllipticTransfers > 0) THEN
      !       CALL CalcOverlaps(Info%mGlobal, Source%mGlobal,mTs,mSs,nOverlaps,Info%level, lEllipticPeriodic)
      !       DO i=1,nOverlaps
      !          mT=mTs(i,:,:)
      !          mS=mSs(i,:,:)

      !          IF (LOC(Info)==LOC(Source) .AND. ALL(mT==mS)) CYCLE

      !          CALL ApplySingleOverlap(Info, Source, mS, mT,EllipticTransferFields)

      !       END DO
      !       IF (nOverlaps > 0) THEN
      !          DEALLOCATE(mTs,mSs)
      !          NULLIFY(mTs,mSs)
      !       END IF
      !    END IF
   END SUBROUTINE SyncFlux


   !> @brief Synchronizes flux between two grids
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param dir - Normal dimension to neighboring grid
   !! @param edge - Which side of info does source lie on?
   SUBROUTINE SyncSingleFlux(Info, Source, mT, mS, dir,edge)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER :: dir,edge
      IF (Info%level==MaxLevel) THEN
         CALL SyncMaxLevelHydroFlux(Info,Source,mT,mS,dir,edge)
      ELSE
         CALL SyncHydroFlux(Info,Source,mT,mS,dir,edge)
      END IF
   END SUBROUTINE SyncSingleFlux

   !> @brief Synchronizes a single component of the emf between grids
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param offset - The periodic offset of the source grid 
   !! (needed to determine whether or not the source grid has refined emf's)
   !! @param dir - Normal dimension to neighboring grid
   SUBROUTINE SyncSingleEMF(Info, Source, mT, mS, offset,dir)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, DIMENSION(3) :: offset
      INTEGER :: dir
      IF (Info%level==MaxLevel) THEN
         CALL SyncMaxLevelEMF(Info,Source,mT,mS,dir)
      ELSE
         CALL SyncEMF(Info,Source,mT,mS,offset,dir)
      END IF
   END SUBROUTINE SyncSingleEMF

   !> @brief Synchronizes flux between two grids
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param dir - Normal dimension to neighboring grid
   !! @param edge - Which side of info does source lie on?
   SUBROUTINE SyncHydroFlux(Info,Source,mT,mS,dir,edge)
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS, mF,mU,mC,mD,mV, mW, mb, ip, ic
      INTEGER :: dir,i,j,edge, r, l(3),n 
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: correction
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: infofixupflux,&
           & sourcefixupflux
      mF=mT
      mC=mT
      mD=mT
      mU=mT  
      mV=mS
      mW=mS
      mU(dir,:)=mT(dir,:)-1
      mW(dir,:)=mS(dir,:)-1
      mF(dir,:)=edge
      mV(dir,:)=3-edge

      IF (edge==1) THEN !mT(dim,1)==1
         mC(dir,:)=1
         mD(dir,:)=0
      ELSE ! (edge==2) !(mT(dim,2)==Info%mx(dim)+1) THEN
         mC(dir,:)=Info%mX(dir)
         mD(dir,:)=Info%mX(dir)+1
      END IF

      infofixupflux=>Info%fixupflux%side(dir)%data
      sourcefixupflux=>Source%fixupflux%side(dir)%data

      ALLOCATE(correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
           &,1):mF(3,2),nFlux))

!!! If cell is shared by child, then don't change fixupflux
!!! otherwise, assume neighbor has child and copy neighbors fixupflux
      FORALL(j=1:nFlux)
         WHERE (info%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3&
              &,1):mC(3,2))==0) ! Child mask for my fluxes
            WHERE(info%ChildMask(mD(1,1):mD(1,2),mD(2,1):mD(2,2),mD(3&
                 &,1):mD(3,2))==0)

               correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
                    &,1):mF(3,2),j) = half*(Sourcefixupflux(mV(1&
                    &,1):mV(1,2),mV(2,1):mV(2,2),mV(3,1):mV(3,2),j) -&
                    & Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2)&
                    &,mF(3,1):mF(3,2),j))

               !               Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
               !                    &,1):mF(3,2),j) = Infofixupflux(mF(1,1):mF(1,2)&
               !                    &,mF(2,1):mF(2,2),mF(3,1):mF(3,2),j) +&
               !                    & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
               !                    &,1):mF(3,2),j)

               !               Sourcefixupflux(mV(1,1):mV(1,2), mV(2,1):mV(2,2), mV(3&
               !                    &,1):mV(3,2),j) = Sourcefixupflux(mV(1,1):mV(1,2)&
               !                    &,mV(2,1):mV(2,2), mV(3,1):mV(3,2),j) -& 
               !                    & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
               !                    &,1):mF(3,2),j)

            ELSEWHERE
               correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
                    &,1):mF(3,2),j) = Sourcefixupflux(mV(1,1):mV(1,2)&
                    &,mV(2,1):mV(2,2),mV(3,1):mV(3,2),j) -&
                    & Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2)&
                    &,mF(3,1):mF(3,2),j)

               !               Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
               !                    &,1):mF(3,2),j) = Sourcefixupflux(mV(1,1):mV(1,2)&
               !                    &,mV(2,1):mV(2,2),mV(3,1):mV(3,2),j)


            END WHERE
            Info%q(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2)&
                 &,FluxFields(j)) = Info%q(mT(1,1):mT(1,2),mT(2&
                 &,1):mT(2,2),mT(3,1):mT(3,2),FluxFields(j)) +&
                 & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
                 &,1):mF(3,2),j)              

            Info%q(mU(1,1):mU(1,2),mU(2,1):mU(2,2),mU(3,1):mU(3,2)&
                 &,FluxFields(j)) = Info%q(mU(1,1):mU(1,2),mU(2&
                 &,1):mU(2,2),mU(3,1):mU(3,2),FluxFields(j)) -&
                 & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
                 &,1):mF(3,2),j)               


            !            Source%q(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2)&
            !                 &,FluxFields(j)) = Source%q(mS(1,1):mS(1,2),mS(2&
            !                 &,1):mS(2,2),mS(3,1):mS(3,2),FluxFields(j)) -&
            !                 & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
            !                 &,1):mF(3,2),j)              

            !            Source%q(mW(1,1):mW(1,2),mW(2,1):mW(2,2),mW(3,1):mW(3,2)&
            !                 &,FluxFields(j)) = Source%q(mW(1,1):mW(1,2),mW(2&
            !                 &,1):mW(2,2),mW(3,1):mW(3,2),FluxFields(j)) +&
            !                 & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
            !                 &,1):mF(3,2),j)               

         ELSEWHERE
            correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3,1):mF(3,2),j)=0
         END WHERE
      END FORALL
      IF (Info%level > 0) THEN !adjust parentfixups...
         mb=Info%mBounds
         r=levels(Info%level-1)%CoarsenRatio               
         ic=mF
         !         ic(1:nDim,2)=Info%mX(1:nDim)
         ip(:,1)=Info%mBounds(:,1)+(mF(:,1) - 1)/r
         ip(:,2)=Info%mBounds(:,1)-1+(mF(:,2))/r
         ip(dir,:)=edge
         DO j=0,r**nDim-1
            IF (MOD(j/r**(dir-1),r)==1) CYCLE
            DO n=1,nDim
               l(n)=MOD(j/r**(n-1),r)  
            END DO
            ic(1:nDim,1)=mF(1:nDim,1)+l(1:nDim)
            ic(1:nDim,2)=mF(1:nDim,2)-r+1+l(1:nDim)
            ic(dir,:)=edge            

            !PRINT *, "ASSOCIATED(Info%parentfixup%side(", i, ")%data) = ", ASSOCIATED(Info%parentfixup%side(i)%data)
            Info%parentfixup%side(dir)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) = &
                 Info%parentfixup%side(dir)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) + &
                 correction(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,:)
         END DO
      END IF


      DEALLOCATE(correction) 

   END SUBROUTINE SyncHydroFlux

   !> @brief Synchronizes flux between two grids on the maxlevel
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param dir - Normal dimension to neighboring grid
   !! @param edge - Which side of info does source lie on?
   SUBROUTINE SyncMaxLevelHydroFlux(Info,Source,mT,mS,dir,edge)
      ! This routine synchronizes fixupfluxes at the appropriate
      !  boundary based on mT and mS and
      ! updates any conserved quantities in the adjacent cells.
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS, mF,mU,mC,mD,mV, mW,mb,ip,ic
      INTEGER :: dir,i,j,edge, r,l(3), n
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: correction
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), POINTER :: infofixupflux,&
           & sourcefixupflux
      mF=mT
      mC=mT
      mD=mT
      mU=mT  
      mV=mS
      mW=mS
      mU(dir,:)=mT(dir,:)-1
      mW(dir,:)=mS(dir,:)-1
      mF(dir,:)=edge
      mV(dir,:)=3-edge

      IF (edge == 1) THEN !mT(dir,1)==1) THEN
         mC(dir,:)=1
         mD(dir,:)=0
      ELSEIF (edge == 2) THEN !(mT(dir,2)==Info%mx(dir)+1) THEN
         mC(dir,:)=Info%mX(dir)
         mD(dir,:)=Info%mX(dir)+1
      END IF
      infofixupflux=>Info%fixupflux%side(dir)%data
      sourcefixupflux=>Source%fixupflux%side(dir)%data
      ALLOCATE(correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
           &,1):mF(3,2),nFlux))



!!! If cell is shared by child, then don't change fixupflux
!!! otherwise, assume neighbor has child and copy neighbors fixupflux
      DO j=1,nFlux !FORALL(j=1:nFlux)

         correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3,1):mF(3,2)&
              &,j) = half*(Sourcefixupflux(mV(1,1):mV(1,2),mV(2&
              &,1):mV(2,2),mV(3,1):mV(3,2),j) - Infofixupflux(mF(1&
              &,1):mF(1,2),mF(2,1):mF(2,2),mF(3,1):mF(3,2),j))

         !         Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3,1):mF(3&
         !              &,2),j) = Infofixupflux(mF(1,1):mF(1,2),mF(2,1):mF(2,2)&
         !              &,mF(3,1):mF(3,2),j) + correction(mF(1,1):mF(1,2),mF(2&
         !              &,1):mF(2,2),mF(3,1):mF(3,2),j)


         !         Sourcefixupflux(mV(1,1):mV(1,2), mV(2,1):mV(2,2), mV(3&
         !              &,1):mV(3,2),j) = Sourcefixupflux(mV(1,1):mV(1,2)&
         !              &,mV(2,1):mV(2,2), mV(3,1):mV(3,2),j) -& 
         !              & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
         !              &,1):mF(3,2),j)


         Info%q(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2)&
              &,FluxFields(j)) = Info%q(mT(1,1):mT(1,2),mT(2,1):mT(2&
              &,2),mT(3,1):mT(3,2),FluxFields(j)) + correction(mF(1&
              &,1):mF(1,2),mF(2,1):mF(2,2),mF(3,1):mF(3,2),j)


         Info%q(mU(1,1):mU(1,2),mU(2,1):mU(2,2),mU(3,1):mU(3,2)&
              &,FluxFields(j)) = Info%q(mU(1,1):mU(1,2),mU(2,1):mU(2,2)&
              &,mU(3,1):mU(3,2),FluxFields(j)) - correction(mF(1,1):mF(1&
              &,2),mF(2,1):mF(2,2),mF(3,1):mF(3,2),j)               

         !         Source%q(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2)&
         !              &,FluxFields(j)) = Source%q(mS(1,1):mS(1,2),mS(2&
         !              &,1):mS(2,2),mS(3,1):mS(3,2),FluxFields(j)) -&
         !              & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
         !              &,1):mF(3,2),j)              

         !         Source%q(mW(1,1):mW(1,2),mW(2,1):mW(2,2),mW(3,1):mW(3,2)&
         !              &,FluxFields(j)) = Source%q(mW(1,1):mW(1,2),mW(2&
         !              &,1):mW(2,2),mW(3,1):mW(3,2),FluxFields(j)) +&
         !              & correction(mF(1,1):mF(1,2),mF(2,1):mF(2,2),mF(3&
         !              &,1):mF(3,2),j)               


         !    END FORALL
      END DO
      IF (Info%level > 0) THEN !adjust parentfixups...
         mb=Info%mBounds
         r=levels(Info%level-1)%CoarsenRatio               
         ic=mF
         ip=1
         !         ic(1:nDim,2)=Info%mX(1:nDim)
         ip(1:nDim,1)=Info%mBounds(1:nDim,1)+(mF(1:nDim,1) - 1)/r
         ip(1:nDim,2)=Info%mBounds(1:nDim,1)-1+(mF(1:nDim,2))/r
         ip(dir,:)=edge
         DO j=0,r**nDim-1
            IF (MOD(j/r**(dir-1),r)==1) CYCLE
            DO n=1,nDim
               l(n)=MOD(j/r**(n-1),r)  
            END DO
            ic(1:nDim,1)=mF(1:nDim,1)+l(1:nDim)
            ic(1:nDim,2)=mF(1:nDim,2)-r+1+l(1:nDim)
            ic(dir,:)=edge            
            !PRINT *, "ASSOCIATED(Info%parentfixup%side(", i, ")%data) = ", ASSOCIATED(Info%parentfixup%side(i)%data)
            Info%parentfixup%side(dir)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) = &
                 Info%parentfixup%side(dir)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) + &
                 correction(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,:)
         END DO
      END IF
      DEALLOCATE(correction) 
   END SUBROUTINE SyncMaxLevelHydroFlux

   !> @brief Synchronizes a single component of the emf between grids
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! @param offset - The periodic offset of the source grid 
   !! (needed to determine whether or not the source grid has refined emf's)
   !! @param dir - Normal dimension to neighboring grid
   SUBROUTINE SyncEMF(Info,Source,mT,mS,offset,dir)
      !This routine synchronizes any common emf's used by two
      ! adjacent grids along their boundary and updates adjacent B
      ! -fields.
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS, mA, mB, mC, mCm,tempmT,tempmS
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: correction
      LOGICAL, DIMENSION(:,:,:), ALLOCATABLE :: sourceauxchildmask,&
           & sourcechildmask
      INTEGER :: i,dim,dir,inorm,level,edge,j,k,l
      REAL(KIND=qPREC) :: dtdxdim, dtdxnorm, dx, ri,rl,rh
      INTEGER, DIMENSION(3) :: offset
      ! Find direction of shared boundary (If neighbor is diagonal
      !  this will pick the lower of the two dims - but it shouldn't
      !   matter)
      level=Info%level
      dx=levels(level)%dx
      SELECT CASE(nDim)

      CASE(2) !nDim
         !We have Source%mGlobal and we want to create an emf mask
         ! for sources children
         !Need to make a cell centered mask for source only
         !
         DO i=1,nDim
            mC(i,1)=mT(i,1)-1
            mC(i,2)=mT(i,2)
         END DO
         ALLOCATE(sourcechildmask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),1))
         sourcechildmask=.false.
         DO i=1,nDim
            mC(i,1)=max(mC(i,1),Source%mGlobal(i,1)+offset(i)-Info&
                 &%mGlobal(i,1)+1) !Ensure that left boundary is
            ! inside of source
            mC(i,2)=min(mC(i,2),Source%mGlobal(i,2)+offset(i)-Info&
                 &%mGlobal(i,1)+1)
         END DO
         sourcechildmask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),1)=(Info&
              &%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),1)&
              &==NEIGHBORCHILD)
         ALLOCATE(correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1),&
              & sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1))
         sourceauxchildmask=.false.         
         mC=mT              
         DO j=-1,0
            DO k=-1,0
               !              IF (i==dim .AND. j==-1) CYCLE
               mC(1,:)=mT(1,:)+j
               mC(2,:)=mT(2,:)+k
               sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1)=&
                    & sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2&
                    &,2),1) .OR. sourcechildmask(mC(1,1):mC(1,2),mC(2&
                    &,1):mC(2,2),1)
            END DO
         END DO
         mA=mT
         mB=mT
         mA(2 ,:)=mT(2  ,:)-1      !Shift for updating aux(inorm)
         mB(1,:)=mT(1,:)-1      !Shift for updating aux(dim)
         IF (iCylindrical==NoCyl) THEN
            WHERE (Info%childemf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) ==&
                 & undefined) !don't have refined data

               ! If Source%childemf is communicated then...
               !            WHERE (Source%childemf(mS(1,1):mS(1,2),mS(2
               !            ,1):mS(2,2),1,1,dir) /= undefined)
               !             !neighbor has refined data
               ! Else we can calculate sourceauxchildmask from childmask
               !  data in the ghost regions
               WHERE(sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2,2)&
                    &,1)) !neighbor has refined data

                  correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) =&
                       & Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1)&
                       &- Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)

                  Info%childemf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) =&
                       & Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1)

                  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) = Source&
                       &%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1)

                  ! And apply correction
                  Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)=&
                       & Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1&
                       &,1)+ correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),1,1    )

                  Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1,1)=&
                       & Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1&
                       &,1)- correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),1,1    )

                  Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,2  )=&
                       & Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,2 &
                       & )- correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),1,1    )

                  Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1,2  )=&
                       & Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1,2 &
                       & )+ correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),1,1    )


               ELSEWHERE !Neither grid has refined data yet - so take
                  ! minimum abs
                  WHERE(ABS(  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1&
                       &,1)) > ABS(Source%emf(mS(1,1):mS(1,2),mS(2&
                       &,1):mS(2,2),1,1)))

                     correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) =&
                          & Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1&
                          &,1)- Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2)&
                          &,1,1)

                     Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)=&
                          & Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1&
                          &,1)

                     ! And apply correction
                     Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)=&
                          & Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1&
                          &,1)+ correction(mT(1,1):mT(1,2)&
                          &,mT(2,1):mT(2,2),1,1    )

                     Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1,1)=&
                          & Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1&
                          &,1)- correction(mT(1,1):mT(1,2)&
                          &,mT(2,1):mT(2,2),1,1    )

                     Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,2  )=&
                          & Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1&
                          &,2  )- correction(mT(1,1):mT(1,2)&
                          &,mT(2,1):mT(2,2),1,1    )

                     Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1,2  )=&
                          & Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1&
                          &,2  )+ correction(mT(1,1):mT(1,2)&
                          &,mT(2,1):mT(2,2),1,1    )


                  END WHERE
               END WHERE
            END WHERE
         ELSE
            DO i=mT(1,1),mT(1,2)          
               j=i-mT(1,1)+mS(1,1)
               k=i-mT(1,1)+mA(1,1)
               l=i-mT(1,1)+mB(1,1)
               ri=(Info%xBounds(1,1)+(i-1)*dx)
               rl=1.d0/(Info%xBounds(1,1)+(REAL(i)-half)*dx)
               rh=1.d0/(Info%xBounds(1,1)+(REAL(i)+half)*dx)

               WHERE (Info%childemf(i,mT(2,1):mT(2,2),1,1) ==&
                    & undefined) !don't have refined data

                  ! If Source%childemf is communicated then...
                  !            WHERE (Source%childemf(j,mS(2
                  !            ,1):mS(2,2),1,1,dir) /= undefined)
                  !             !neighbor has refined data
                  ! Else we can calculate sourceauxchildmask from childmask
                  !  data in the ghost regions
                  WHERE(sourceauxchildmask(i,mT(2,1):mT(2,2)&
                       &,1)) !neighbor has refined data

                     correction(i,mT(2,1):mT(2,2),1,1) =&
                          & Source%emf(j,mS(2,1):mS(2,2),1,1)&
                          &- Info%emf(i,mT(2,1):mT(2,2),1,1)

                     Info%childemf(i,mT(2,1):mT(2,2),1,1) =&
                          & Source%emf(j,mS(2,1):mS(2,2),1,1)

                     Info%emf(i,mT(2,1):mT(2,2),1,1) = Source&
                          &%emf(j,mS(2,1):mS(2,2),1,1)

                     ! And apply correction
                     Info%aux(i,mT(2,1):mT(2,2),1,1)=&
                          & Info%aux(i,mT(2,1):mT(2,2),1&
                          &,1)+ correction(i,mT(2&
                          &,1):mT(2,2),1,1    )

                     Info%aux(k,mA(2,1):mA(2,2),1,1)=&
                          & Info%aux(k,mA(2,1):mA(2,2),1&
                          &,1)- correction(i,mT(2&
                          &,1):mT(2,2),1,1    )

                     Info%aux(i,mT(2,1):mT(2,2),1,2  )=&
                          & Info%aux(i,mT(2,1):mT(2,2),1,2 &
                          & )- correction(i,mT(2&
                          &,1):mT(2,2),1,1    )*&
                          ri*rh

                     Info%aux(l,mB(2,1):mB(2,2),1,2  )=&
                          & Info%aux(l,mB(2,1):mB(2,2),1,2 &
                          & )+ correction(i,mT(2&
                          &,1):mT(2,2),1,1    )*&
                          ri*rl


                  ELSEWHERE !Neither grid has refined data yet - so take
                     ! minimum abs
                     WHERE(ABS(  Info%emf(i,mT(2,1):mT(2,2),1&
                          &,1)) > ABS(Source%emf(j,mS(2&
                          &,1):mS(2,2),1,1)))

                        correction(i,mT(2,1):mT(2,2),1,1) =&
                             & Source%emf(j,mS(2,1):mS(2,2),1&
                             &,1)- Info%emf(i,mT(2,1):mT(2,2)&
                             &,1,1)

                        Info%emf(i,mT(2,1):mT(2,2),1,1)=&
                             & Source%emf(j,mS(2,1):mS(2,2),1&
                             &,1)

                        ! And apply correction
                        Info%aux(i,mT(2,1):mT(2,2),1,1)=&
                             & Info%aux(i,mT(2,1):mT(2,2),1&
                             &,1)+ correction(i&
                             &,mT(2,1):mT(2,2),1,1    )

                        Info%aux(k,mA(2,1):mA(2,2),1,1)=&
                             & Info%aux(k,mA(2,1):mA(2,2),1&
                             &,1)- correction(i&
                             &,mT(2,1):mT(2,2),1,1    )

                        Info%aux(i,mT(2,1):mT(2,2),1,2  )=&
                             & Info%aux(i,mT(2,1):mT(2,2),1&
                             &,2  )- correction(i&
                             &,mT(2,1):mT(2,2),1,1    )*&
                             ri*rh

                        Info%aux(l,mB(2,1):mB(2,2),1,2  )=&
                             & Info%aux(l,mB(2,1):mB(2,2),1&
                             &,2  )+ correction(i&
                             &,mT(2,1):mT(2,2),1,1    )*&
                             ri*rl


                     END WHERE
                  END WHERE
               END WHERE
            END DO
         END IF!icyl
         DEALLOCATE(correction,sourceauxchildmask,sourcechildmask) 

      CASE(3) !nDim
         DO i=1,nDim
            mC(i,1)=mT(i,1)-1!max(mT(i,1)-1,1+Source%mGlobal(i,1)
            !-Info%mGlobal(i,1))
            mC(i,2)=mT(i,2)!min(mT(i,2),  1+Source%mGlobal(i,2)-Info
            !%mGlobal(i,1))
         END DO
         mC(dir,:)=mT(dir,:) !don't need to shrink direction of emf!
         ALLOCATE(sourcechildmask(mC(1,1):mC(1,2),mC(2,1):mC(2,2)&
              &,mC(3,1):mC(3,2)))           
         sourcechildmask=.false.
         DO i=1,nDim
            !            IF (i == dir) cycle
            mC(i,1)=max(mC(i,1),Source%mGlobal(i,1)+offset(i)-Info&
                 &%mGlobal(i,1)+1) !Ensure that left boundary is
            ! inside of source
            mC(i,2)=min(mC(i,2),Source%mGlobal(i,2)+offset(i)-Info&
                 &%mGlobal(i,1)+1)
         END DO
         sourcechildmask(mC(1,1):mC(1,2),mC(2,1):mC(2,2),mC(3,1):mC(3&
              &,2))=(Info%ChildMask(mC(1,1):mC(1,2),mC(2,1):mC(2,2)&
              &,mC(3,1):mC(3,2))==NEIGHBORCHILD)

         ! sourceauxchildmask(:,:,:,:,1) is a flag for neighbor's
         !  emfs being refined or no
         ALLOCATE(correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
              &,1):mT(3,2),1), sourceauxchildmask(mT(1,1):mT(1,2)&
              &,mT(2,1):mT(2,2),mT(3,1):mT(3,2)))
         sourceauxchildmask=.false.
         dim=modulo(dir-2,3)+1
         inorm=6-dim-dir               !direction normal to edge
         ! and emf           
         mC=mT
         DO j=-1,0
            DO k=-1,0
               mC(dim,:)=mT(dim,:)+j
               mC(inorm,:)=mT(inorm,:)+k
               sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2,2)&
                    &,mT(3,1):mT(3,2))= sourceauxchildmask(mT(1&
                    &,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2))&
                    & .OR. sourcechildmask(mC(1,1):mC(1,2),mC(2&
                    &,1):mC(2,2),mC(3,1):mC(3,2))
            END DO
         END DO
         mA=mT
         mB=mT
         mA(dim  ,:)=mT(dim  ,:)-1      !Shift for updating
         ! aux(inorm)
         mB(inorm,:)=mT(inorm,:)-1      !Shift for updating
         ! aux(dim)

         !if have childemf - do nothing
         !else if neighbors have child - apply correction and
         ! store emf and update childemf
         !else calc correction and update emf...

         WHERE (Info%childemf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
              &,1):mT(3,2),dir) == undefined) !don't have refined
            ! data

            ! If Source%childemf is communicated then...
            !            WHERE (Source%childemf(mS(1,1):mS(1,2)
            !            ,mS(2,1):mS(2,2),mS(3,1):mS(3,2),1,dir)
            !            /= undefined) !neighbor has refined data
            ! Else we can calculate sourceauxchildmask from
            !  childmask data in the ghost regions
            WHERE(sourceauxchildmask(mT(1,1):mT(1,2),mT(2,1):mT(2&
                 &,2),mT(3,1):mT(3,2))) !neighbor has refined data

               correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                    &,1):mT(3,2),1   ) = Source%emf(mS(1,1):mS(1&
                    &,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)-&
                    & Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2)&
                    &,mT(3,1):mT(3,2),dir)

               Info%childemf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                    &,1):mT(3,2),dir) = Source%emf(mS(1,1):mS(1&
                    &,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)

               Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                    &,1):mT(3,2),dir) = Source%emf(mS(1,1):mS(1&
                    &,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)

               ! And apply correction
               Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                    &,1):mT(3,2),inorm)= Info%aux(mT(1,1):mT(1,2)&
                    &,mT(2,1):mT(2,2),mT(3,1):mT(3,2),inorm)+&
                    & correction(mT(1,1):mT(1,2),mT(2&
                    &,1):mT(2,2),mT(3,1):mT(3,2),1    )

               Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),mA(3&
                    &,1):mA(3,2),inorm)= Info%aux(mA(1,1):mA(1,2)&
                    &,mA(2,1):mA(2,2),mA(3,1):mA(3,2),inorm)-&
                    & correction(mT(1,1):mT(1,2),mT(2&
                    &,1):mT(2,2),mT(3,1):mT(3,2),1    )

               Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                    &,1):mT(3,2),dim  )= Info%aux(mT(1,1):mT(1,2)&
                    &,mT(2,1):mT(2,2),mT(3,1):mT(3,2),dim  )-&
                    & correction(mT(1,1):mT(1,2),mT(2&
                    &,1):mT(2,2),mT(3,1):mT(3,2),1    )

               Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3&
                    &,1):mB(3,2),dim  )= Info%aux(mB(1,1):mB(1,2)&
                    &,mB(2,1):mB(2,2),mB(3,1):mB(3,2),dim  )+&
                    & correction(mT(1,1):mT(1,2),mT(2&
                    &,1):mT(2,2),mT(3,1):mT(3,2),1    )


            ELSEWHERE !Neither grid has refined data yet - so take
               ! minimum abs
               WHERE(ABS(  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2&
                    &,2),mT(3,1):mT(3,2),dir)) > ABS(Source&
                    &%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3&
                    &,1):mS(3,2),dir)))

                  correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                       &,1):mT(3,2),1   ) = Source%emf(mS(1&
                       &,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3&
                       &,2),dir)- Info%emf(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),mT(3,1):mT(3,2),dir)

                  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                       &,1):mT(3,2),dir)= Source%emf(mS(1,1):mS(1&
                       &,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)

                  ! And apply correction
                  Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                       &,1):mT(3,2),inorm)= Info%aux(mT(1,1):mT(1&
                       &,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),inorm)&
                       &+ correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),mT(3,1):mT(3,2),1    )

                  Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),mA(3&
                       &,1):mA(3,2),inorm)= Info%aux(mA(1,1):mA(1&
                       &,2),mA(2,1):mA(2,2),mA(3,1):mA(3,2),inorm)&
                       &- correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),mT(3,1):mT(3,2),1    )

                  Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3&
                       &,1):mT(3,2),dim  )= Info%aux(mT(1,1):mT(1&
                       &,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dim  )&
                       &- correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),mT(3,1):mT(3,2),1    )

                  Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3&
                       &,1):mB(3,2),dim  )= Info%aux(mB(1,1):mB(1&
                       &,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),dim  )&
                       &+ correction(mT(1,1):mT(1,2),mT(2&
                       &,1):mT(2,2),mT(3,1):mT(3,2),1    )


               END WHERE
            END WHERE

         END WHERE

         DEALLOCATE(correction,sourceauxchildmask,sourcechildmask) 
      END SELECT
   END SUBROUTINE SyncEMF

   !> @brief Synchronizes a single component of the emf between two maxlevel grids
   !! @param Info - target info structure
   !! @param Source - source info structure
   !! @param mS - indices of overlap region in source grid
   !! @param mT - indices of overlap region in target grid
   !! (needed to determine whether or not the source grid has refined emf's)
   !! @param dir - Normal dimension to neighboring grid
   SUBROUTINE SyncMaxLevelEMF(Info,Source,mT,mS,dir)
      !This routine synchronizes any common emf's used by two
      ! adjacent grids along their boundary and updates adjacent B
      ! -fields.
      TYPE(InfoDef) :: Info, Source
      INTEGER, DIMENSION(3,2) :: mT, mS, mA, mB, mC, mCm
      REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: correction
      LOGICAL, DIMENSION(:,:,:), ALLOCATABLE :: sourceauxchildmask
      INTEGER :: i,dim,dir,inorm,level,j,k,l
      REAL(KIND=qPREC) :: dtdxdim, dtdxnorm, dx, ri, rh, rl
      ! Find direction of shared boundary (If neighbor is diagonal
      !  this will pick the lower of the two dims - but it shouldn't
      !   matter)
      level=Info%level
      dx=levels(level)%dx

      SELECT CASE(nDim)
      CASE(2) !nDim
         IF (dir /= 3) THEN
            WRITE(*,*) "Error in SyncMaxLevelEMF"
            STOP
         END IF
         mA=mT
         mB=mT
         mA(2,:)=mT(2,:)-1      !Shift for updating aux(inorm)
         mB(1,:)=mT(1,:)-1      !Shift for updating aux(dim)

         ALLOCATE(correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1))

         IF (iCylindrical==NoCyl) THEN
            WHERE(ABS(  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)) > & 
                 ABS(Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1)))

               correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) = &
                    Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1) - &
                    Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)

               Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) = &
                    Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),1,1)

               ! And apply correction
               Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1) = &
                    Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)+&
                    correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)

               Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1,1) = &
                    Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),1,1)-&
                    correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)

               Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,2) = &
                    Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,2)-&
                    correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)

               Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1,2) = &
                    Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),1,2)+&
                    correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),1,1)


            END WHERE
         ELSE
            DO i=mT(1,1),mT(1,2)          
               j=i-mT(1,1)+mS(1,1)
               k=i-mT(1,1)+mA(1,1)
               l=i-mT(1,1)+mB(1,1)
               ri=(Info%xBounds(1,1)+(i-1)*dx)
               rl=1.d0/(Info%xBounds(1,1)+(REAL(i)-half)*dx)
               rh=1.d0/(Info%xBounds(1,1)+(REAL(i)+half)*dx)

               WHERE(ABS(  Info%emf(i,mT(2,1):mT(2,2),1,1)) > & 
                    ABS(Source%emf(j,mS(2,1):mS(2,2),1,1)))

                  correction(i,mT(2,1):mT(2,2),1,1) = &
                       Source%emf(j,mS(2,1):mS(2,2),1,1) - &
                       Info%emf(i,mT(2,1):mT(2,2),1,1)

                  Info%emf(i,mT(2,1):mT(2,2),1,1) = &
                       Source%emf(j,mS(2,1):mS(2,2),1,1)

                  ! And apply correction
                  Info%aux(i,mT(2,1):mT(2,2),1,1) = &
                       Info%aux(i,mT(2,1):mT(2,2),1,1)+&
                       correction(i,mT(2,1):mT(2,2),1,1)

                  Info%aux(k,mA(2,1):mA(2,2),1,1) = &
                       Info%aux(k,mA(2,1):mA(2,2),1,1)-&
                       correction(i,mT(2,1):mT(2,2),1,1)

                  Info%aux(i,mT(2,1):mT(2,2),1,2) = &
                       Info%aux(i,mT(2,1):mT(2,2),1,2)-&
                       correction(i,mT(2,1):mT(2,2),1,1)*&
                       ri*rh

                  Info%aux(l,mB(2,1):mB(2,2),1,2) = &
                       Info%aux(l,mB(2,1):mB(2,2),1,2)+&
                       correction(i,mT(2,1):mT(2,2),1,1)*&
                       ri*rl

               END WHERE
            END DO
         END IF!icyl
         DEALLOCATE(correction) 

      CASE(3) !nDim

         ! sourceauxchildmask(:,:,:,:,1) is a flag for neighbor's
         !  emfs being refined or no
         ALLOCATE(correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1))

         dim=modulo(dir-2,3)+1
         inorm=6-dim-dir               !direction normal to edge
         ! and emf

         mA=mT
         mB=mT
         mA(dim  ,:)=mT(dim  ,:)-1      !Shift for updating
         ! aux(inorm)
         mB(inorm,:)=mT(inorm,:)-1      !Shift for updating
         ! aux(dim)

         !if have childemf - do nothing
         !else if neighbors have child - apply correction and
         ! store emf and update childemf
         !else calc correction and update emf...

         WHERE(ABS(  Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dir)) > &
              ABS(Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)))

            correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1) = &
                 Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir) - &
                 Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dir)

            Info%emf(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dir) = &
                 Source%emf(mS(1,1):mS(1,2),mS(2,1):mS(2,2),mS(3,1):mS(3,2),dir)

            ! And apply correction
            Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),inorm) = &
                 Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),inorm) + &
                 correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1)

            Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),mA(3,1):mA(3,2),inorm) = &
                 Info%aux(mA(1,1):mA(1,2),mA(2,1):mA(2,2),mA(3,1):mA(3,2),inorm) - &
                 correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1)

            Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dim) = &
                 Info%aux(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),dim) - &
                 correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1)

            Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),dim) = &
                 Info%aux(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),dim) + &
                 correction(mT(1,1):mT(1,2),mT(2,1):mT(2,2),mT(3,1):mT(3,2),1)


         END WHERE
         DEALLOCATE(correction)
      END SELECT

   END SUBROUTINE SyncMaxLevelEMF


   !> @}

   !> @name SingleInfoOps Single Info Operations
   !! @{

   !> @brief Solves for inner prolongated aux-fields in a divergence free way
   !! @param Info - prolongated info structure
   SUBROUTINE ProlongationFixup(Info)
      !Should only be called after last overlap of first round of overlaps
      TYPE(InfoDef) :: Info
      REAL(KIND=qprec), DIMENSION(:,:,:,:), POINTER :: aux
      INTEGER :: i,j,k,rmbc
      INTEGER,DIMENSION(3,2) :: mb
      REAL(KIND=qPREC), DIMENSION(12) :: temp

      ! These 2D and 3D matrices come from solving the divergence equation.  For a derivation
      ! of this, see Cunningham '09.  If it's not there, e-mail us and yell at us.

      REAL(KIND=qprec), PARAMETER, DIMENSION(4,8) :: A2D = &
           0.25d0*RESHAPE((/ &
           2.d0, 0.d0, 1.d0, 1.d0, & 
           0.d0, 2.d0, -1.d0, -1.d0, &
           2.d0, 0.d0, -1.d0, -1.d0, &
           0.d0, 2.d0, 1.d0, 1.d0, &
           1.d0, 1.d0, 2.d0, 0.d0, &
           -1.d0, -1.d0, 2.d0, 0.d0, &
           -1.d0, -1.d0, 0.d0, 2.d0, &
           1.d0, 1.d0, 0.d0, 2.d0 &
           /),(/4,8/) )

      REAL(KIND=qprec), PARAMETER, DIMENSION(12,24) :: A3D = &
           0.0625d0*RESHAPE((/ &
           8.d0, 0.d0, 0.d0, 0.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, &
           0.d0, 8.d0, 0.d0, 0.d0, 1.d0, 3.d0, 1.d0, 3.d0, -3.d0, -1.d0, -3.d0, -1.d0, &
           0.d0, 0.d0, 8.d0, 0.d0, -3.d0, -1.d0, -3.d0, -1.d0, 1.d0, 3.d0, 1.d0, 3.d0, &
           0.d0, 0.d0, 0.d0, 8.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, &
           8.d0, 0.d0, 0.d0, 0.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, -3.d0, -1.d0, &
           0.d0, 8.d0, 0.d0, 0.d0, -1.d0, -3.d0, -1.d0, -3.d0, 3.d0, 1.d0, 3.d0, 1.d0, &
           0.d0, 0.d0, 8.d0, 0.d0, 3.d0, 1.d0, 3.d0, 1.d0, -1.d0, -3.d0, -1.d0, -3.d0, &
           0.d0, 0.d0, 0.d0, 8.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, 1.d0, 3.d0, &
           3.d0, 1.d0, 3.d0, 1.d0, 8.d0, 0.d0, 0.d0, 0.d0, 3.d0, 3.d0, 1.d0, 1.d0, &
           1.d0, 3.d0, 1.d0, 3.d0, 0.d0, 8.d0, 0.d0, 0.d0, -3.d0, -3.d0, -1.d0, -1.d0, &
           -3.d0, -1.d0, -3.d0, -1.d0, 8.d0, 0.d0, 0.d0, 0.d0, -3.d0, -3.d0, -1.d0, -1.d0, &
           -1.d0, -3.d0, -1.d0, -3.d0, 0.d0, 8.d0, 0.d0, 0.d0, 3.d0, 3.d0, 1.d0, 1.d0, &
           -3.d0, -1.d0, -3.d0, -1.d0, 0.d0, 0.d0, 8.d0, 0.d0, 1.d0, 1.d0, 3.d0, 3.d0, &
           -1.d0, -3.d0, -1.d0, -3.d0, 0.d0, 0.d0, 0.d0, 8.d0, -1.d0, -1.d0, -3.d0, -3.d0, &
           3.d0, 1.d0, 3.d0, 1.d0, 0.d0, 0.d0, 8.d0, 0.d0, -1.d0, -1.d0, -3.d0, -3.d0, &
           1.d0, 3.d0, 1.d0, 3.d0, 0.d0, 0.d0, 0.d0, 8.d0, 1.d0, 1.d0, 3.d0, 3.d0, &
           3.d0, 3.d0, 1.d0, 1.d0, 3.d0, 3.d0, 1.d0, 1.d0, 8.d0, 0.d0, 0.d0, 0.d0, &
           -3.d0, -3.d0, -1.d0, -1.d0, -3.d0, -3.d0, -1.d0, -1.d0, 8.d0, 0.d0, 0.d0, 0.d0, &
           1.d0, 1.d0, 3.d0, 3.d0, -3.d0, -3.d0, -1.d0, -1.d0, 0.d0, 8.d0, 0.d0, 0.d0, &
           -1.d0, -1.d0, -3.d0, -3.d0, 3.d0, 3.d0, 1.d0, 1.d0, 0.d0, 8.d0, 0.d0, 0.d0, &
           -3.d0, -3.d0, -1.d0, -1.d0, 1.d0, 1.d0, 3.d0, 3.d0, 0.d0, 0.d0, 8.d0, 0.d0, &
           3.d0, 3.d0, 1.d0, 1.d0, -1.d0, -1.d0, -3.d0, -3.d0, 0.d0, 0.d0, 8.d0, 0.d0, &
           -1.d0, -1.d0, -3.d0, -3.d0, -1.d0, -1.d0, -3.d0, -3.d0, 0.d0, 0.d0, 0.d0, 8.d0, &
           1.d0, 1.d0, 3.d0, 3.d0, 1.d0, 1.d0, 3.d0, 3.d0, 0.d0, 0.d0, 0.d0, 8.d0 &
           /),(/12,24/) )

      aux=>Info%aux
      rmbc=levels(Info%level-1)%pmbc*levels(Info%level-1)%CoarsenRatio
      mb=1
      mb(1:nDim,1)=1-rmbc
      mb(1:nDim,2)=Info%mx(1:nDim)+rmbc

      IF (nDim == 2) THEN
         DO i=mb(1,1),mb(1,2),2
            DO j=mb(2,1),mb(2,2),2
               IF (aux(i+1,j,1,1)==UNDEFINED) THEN !Assume all
                  temp(1:4)=MATMUL(A2D, &            
                       (/aux(i  ,j  ,1,1),     &
                       aux(i  ,j+1,1,1),   &
                       aux(i+2,j  ,1,1),   &
                       aux(i+2,j+1,1,1), &
                       aux(i  ,j  ,1,2),     &
                       aux(i  ,j+2,1,2),   &
                       aux(i+1,j  ,1,2),   &
                       aux(i+1,j+2,1,2) /))
                  aux(i+1,j  ,1,1)=temp(1)
                  aux(i+1,j+1,1,1)=temp(2)
                  aux(i  ,j+1,1,2)=temp(3)
                  aux(i+1,j+1,1,2)=temp(4)
               END IF
            END DO
         END DO

      ELSE
         DO i=mb(1,1),mb(1,2),2
            DO j=mb(2,1),mb(2,2),2
               DO k=mb(3,1),mb(3,2),2
                  IF (aux(i+1,j,k,1) == UNDEFINED) THEN !Assume
                     temp = MATMUL(A3D, &
                          (/aux(i  ,j  ,k  ,1),&
                          aux(i  ,j  ,k+1,1),&
                          aux(i  ,j+1,k  ,1),&
                          aux(i  ,j+1,k+1,1),&
                          aux(i+2,j  ,k  ,1),&
                          aux(i+2,j  ,k+1,1),&
                          aux(i+2,j+1,k  ,1),&
                          aux(i+2,j+1,k+1,1),&
                          aux(i  ,j  ,k  ,2),&
                          aux(i  ,j  ,k+1,2),&
                          aux(i  ,j+2,k  ,2),&
                          aux(i  ,j+2,k+1,2),&
                          aux(i+1,j  ,k  ,2),&
                          aux(i+1,j  ,k+1,2),&
                          aux(i+1,j+2,k  ,2),&
                          aux(i+1,j+2,k+1,2),&
                          aux(i  ,j  ,k  ,3),&
                          aux(i  ,j  ,k+2,3),&
                          aux(i  ,j+1,k  ,3),&
                          aux(i  ,j+1,k+2,3),&
                          aux(i+1,j  ,k  ,3),&
                          aux(i+1,j  ,k+2,3),&
                          aux(i+1,j+1,k  ,3),&
                          aux(i+1,j+1,k+2,3)/))
                     aux(i+1,j  ,k  ,1) = temp(1)
                     aux(i+1,j  ,k+1,1) = temp(2)
                     aux(i+1,j+1,k  ,1) = temp(3)
                     aux(i+1,j+1,k+1,1) = temp(4)
                     aux(i  ,j+1,k  ,2) = temp(5)
                     aux(i  ,j+1,k+1,2) = temp(6)
                     aux(i+1,j+1,k  ,2) = temp(7)
                     aux(i+1,j+1,k+1,2) = temp(8)
                     aux(i  ,j  ,k+1,3) = temp(9)
                     aux(i  ,j+1,k+1,3) = temp(10)
                     aux(i+1,j  ,k+1,3) = temp(11)
                     aux(i+1,j+1,k+1,3) = temp(12)
                  END IF
               END DO
            END DO
         END DO
      END IF
      CALL UpdateAux(Info,mb)
   END SUBROUTINE ProlongationFixup


   !> @brief Coarsens data for Parent
   !! @param Info - Info structure
   !! @param n - Level
   SUBROUTINE CoarsenDataForParent(Info,n)
      TYPE(InfoDef) :: Info
      INTEGER :: n
      !    CALL CoarsenCostMapForParent(Info)      
      IF (n > 0) THEN
         CALL CoarsenQForParent(Info)
         CALL CoarsenFluxesForParent(Info)
      END IF
   END SUBROUTINE CoarsenDataForParent

   !> @brief Coarsens initial data for Parent
   !! @param Info - Info structure
   !! @param n - Level
   SUBROUTINE CoarsenInitialDataForParent(Info,n)
      TYPE(InfoDef) :: Info
      INTEGER :: n
      !    CALL CoarsenCostMapForParent(Info)
      IF (n > 0) THEN
         CALL CoarsenQForParent(Info)
         IF (MaintainAuxArrays) CALL CoarsenAuxForParent(Info)
      END IF
   END SUBROUTINE CoarsenInitialDataForParent

   !> @brief Coarsens costmap for parent
   !! @param Info - Info structure
   SUBROUTINE CoarsenCostMapForParent(Info)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mB
      INTEGER i,j,l(3),n,r
      INTEGER,DIMENSION(3,2) :: ic,ip
      REAL(KIND=qPrec) :: factor,factor_aux,factor_emf,factor_flux
      INTEGER :: ermbc
      REAL :: grid_size_factor

      ! Bounds within parent    
      mB=Info%mBounds
      r=levels(Info%level-1)%CoarsenRatio
      ic(ndim+1:3,:)=1
      Info%ParentCostMap(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1) = 0d0
      l=0

      ! hyperbolic_mbc has r layers of ghost zones on the first step, r-1 on the second step, etc.
      ! So ermbc will be r(r+1)/2 times hyperbolic_mbc (because sum(1:r) = r(r+1)/2).  Multiply the
      ! result by 2, since ghost cells appear on both sides of an edge.  Since this
      ! is being done in the parent's frame of reference, though, ermbc will be divided by r.

      ! average_mbc = 2 * (hyperbolic_mbc * sum(1:r)/r) = 2 * hyperbolic_mbc * r(r+1)/2r = (r+1) * hyperbolic_mbc
      !    ermbc = (r+1) * hyperbolic_mbc
      ermbc=0

      ! The grid size factor is the number of cells in the extended grid (core grid + ghost zones)
      ! divided by the number of cells in the core grid (no ghost zones).
      grid_size_factor = PRODUCT(mB(1:nDim,2)-mB(1:nDim,1) + 1 + ermbc) * 1.0 / &
           PRODUCT(mB(1:nDim,2)-mB(1:nDim,1) + 1)

      DO i=0,r**nDim-1  ! i loops over all child subcells

         DO n=1,nDim
            l(n)=MOD(i/r**(n-1),r)
         END DO

         ic(1:nDim,1)=1 + l(1:nDim)
         ic(1:nDim,2)=Info%mX(1:nDim) - r + l(1:nDim) + 1

         Info%ParentCostMap(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,1) =       &
              Info%ParentCostMap(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,1) +  &
              Info%CostMap(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,1) + &
              Info%CostMap(ic(1,1):ic(1,2):r, ic(2,1):ic(2,2):r, ic(3,1):ic(3,2):r,2)

      END DO
      IF (Info%level > -1) Info%ParentCostmap(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,1)=&
           Info%ParentCostmap(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,1)*REAL(r) !+&
      !         sum(Info%CostPerGrid(:))./product(mB(1:nDim,2)-mB(1:nDim,1)+1) 

   END SUBROUTINE CoarsenCostMapForParent

   !> @brief Coarsens cell centered q variable for parent
   !! @param Info - Info structure
   SUBROUTINE CoarsenQForParent(Info)
      TYPE(InfoDef) :: Info
      INTEGER i,l(3),n,r
      INTEGER,DIMENSION(3,2) :: ic,mB
      REAL(KIND=qPrec) :: factor

      ! Bounds within parent    
      mB=Info%mBounds
      r=levels(Info%level-1)%CoarsenRatio
      factor = 1d0/r**nDim
      ic(ndim+1:3,:)=1
      Info%qParent(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),:) = 0d0
      l=0
      DO i=0,r**nDim-1  ! i loops over all child subcells
         DO n=1,nDim
            l(n)=MOD(i/r**(n-1),r)
         END DO
         ic(1:nDim,1)=1 + l(1:nDim)
         ic(1:nDim,2)=Info%mX(1:nDim) - r + l(1:nDim) + 1
         Info%qParent(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,:) =       &
              Info%qParent(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,:) +  &
              Info%q(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,RestrictFields)
      END DO
      Info%qParent(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,:) =       &
           Info%qParent(mB(1,1):mB(1,2)  ,mB(2,1):mB(2,2)  ,mB(3,1):mB(3,2)  ,:) * factor
   END SUBROUTINE CoarsenQForParent

   !> @brief Coarsens fixupfluxes for parent
   !! @param Info - Info structure
   SUBROUTINE CoarsenFluxesForParent(Info)
      TYPE(InfoDef) :: Info
      INTEGER i,r
      INTEGER,DIMENSION(3,2) :: ic,ip,mB
      REAL(KIND=qPrec) :: factor_emf,factor_flux

      ! Bounds within parent    
      mB=Info%mBounds
      r=levels(Info%level-1)%CoarsenRatio
      ! Calculate the restriction factors.
      factor_emf = 1d0/r**(nDim-1) !(nDim-2) spatial plus time
      factor_flux = 1d0/r**(nDim)    !(nDim-1) spatial plus time
      ip(nDim+1:3,:)=1
      DO i=1,nDim
         Info%parentfixup%side(i)%data=Info%parentfixup%side(i)%data*factor_flux
      END DO
      IF (MaintainAuxArrays) THEN
         IF (nDim == 2) THEN
            ip(1:nDim,1)=mB(1:nDim,1); ip(1:nDim,2)=mB(1:nDim,2)+1;   
            Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1) = &
                 Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,1) * factor_emf
         ELSE
            ip(1:nDim,1)=mB(1:nDim,1); ip(1:nDim,2)=mB(1:nDim,2)+1;   
            DO i=1,nDim
               ip(i,2)=ip(i,2)-1 
               Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,i) = &
                    Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,i) * factor_emf
               ip(i,2)=ip(i,2)+1
            END DO
         END IF
      END IF
   END SUBROUTINE CoarsenFluxesForParent

   !> @brief Coarsens aux fields for parent
   !! @param Info - Info structure
   SUBROUTINE CoarsenAuxForParent(Info)
      TYPE(InfoDef) :: Info
      INTEGER i,j,k,r,l(3)
      INTEGER,DIMENSION(3,2) :: ic,ip,mB
      REAL(KIND=qPrec) :: factor,factor_aux

      ! Bounds within parent    
      mB=Info%mBounds
      r=levels(Info%level-1)%CoarsenRatio
      ! Calculate the restriction factors.
      factor_aux=1d0/r**(nDim-1)    !(nDim-1) spatial-1
      ic(ndim+1:3,:)=1
      ip(ndim+1:3,:)=1

      Info%auxParent=0
      l=0;
      DO i=1,nDim !B-field direction
         ip(1:nDim,1) = mB(1:nDim,1)
         ip(1:nDim,2) = mB(1:nDim,2)            
         ip(i,2)=ip(i,2)+1
         Info%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), i)=0
         DO j=0,r**nDim-1
            DO k=1,nDim
               l(k)=MOD(j/r**(k-1),r)  
            END DO
            IF (l(i)==1) CYCLE
            ic(1:nDim,1)=1 + l(1:nDim)
            ic(1:nDim,2)=Info%mX(1:nDim) - r + l(1:nDim) + 1
            ic(i,:)=(/1,Info%mX(i)+1/)
            Info%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), i) = &
                 Info%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), i) + &
                 Info%aux(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,i)
         END DO
         Info%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), i) = &
              Info%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), i)*factor_aux
      END DO
   END SUBROUTINE CoarsenAuxForParent

   !> @brief Accumulates fixupfluxes and emf's for parent
   !! @param Info - Info structure
   SUBROUTINE AccumulateFlux(Info)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: ip, ic, mb
      INTEGER :: i,j,l(3),r,edge,n    
      mb=Info%mBounds
      r=levels(Info%level-1)%CoarsenRatio      

      ic=1
      ic(1:nDim,2)=Info%mX(1:nDim)

      !PRINT *, "AccumulateFlux::level = ", Info%level
      !PRINT *, "ASSOCIATED(Info%parentfixup) = ", ASSOCIATED(Info%parentfixup)
      !PRINT *, "ASSOCIATED(Info%parentfixup%side) = ", ASSOCIATED(Info%parentfixup%side)

      DO i=1,nDim
         DO edge=1,2
            DO j=0,r**nDim-1
               IF (MOD(j/r**(i-1),r)==1) CYCLE
               DO n=1,nDim
                  l(n)=MOD(j/r**(n-1),r)  
               END DO
               ic(1:nDim,1)=1+l(1:nDim); ic(1:nDim,2)=Info%mX(1:nDim)-r+1+l(1:nDim);ic(i,:)=edge
               ip=mb; ip(i,:)=edge
               !PRINT *, "ASSOCIATED(Info%parentfixup%side(", i, ")%data) = ", ASSOCIATED(Info%parentfixup%side(i)%data)
               Info%parentfixup%side(i)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) = &
                    Info%parentfixup%side(i)%data(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,:) + &
                    Info%fixupflux%side(i)%data(ic(1,1):ic(1,2):r,ic(2,1):ic(2,2):r,ic(3,1):ic(3,2):r,:)
            END DO
         END DO
      END DO
      IF (MaintainAuxArrays) THEN
         IF (nDim == 2) THEN
            ip(1:nDim,1)=mb(1:nDim,1)
            ip(1:nDim,2)=mb(1:nDim,2)+1
            ic(1:nDim,1)=1
            ic(1:nDim,2)=Info%mX(1:nDim)+1
            Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,1,1) = &
                 Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,1,1) + &
                 Info%emf(ic(1,1):ic(1,2):2,ic(2,1):ic(2,2):2,1,1)
         ELSE
            DO i=1,3
               ip(1:nDim,1)=mb(1:nDim,1)
               ip(1:nDim,2)=mb(1:nDim,2)+1
               ip(i,2)=ip(i,2)-1
               ic(1:nDim,2)=Info%mX(1:nDim)+1
               ic(1:nDim,1)=1
               DO j=0,1
                  ic(i,1)=1+j
                  ic(i,2)=Info%mX(i)-1+j
                  Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,i) = &
                       Info%parentemf(ip(1,1):ip(1,2)  ,ip(2,1):ip(2,2)  ,ip(3,1):ip(3,2)  ,i) + &
                       Info%emf(ic(1,1):ic(1,2):2,ic(2,1):ic(2,2):2,ic(3,1):ic(3,2):2,i)
               END DO
            END DO
         END IF
      END IF
   END SUBROUTINE AccumulateFlux

   !> @brief Applies corrections form children emf's
   !! @param Info - Info structure
   SUBROUTINE RestrictionFixup(Info)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2):: ip,mB,ia,ib
      REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: correction
      INTEGER :: i,l(3),level
      REAL(KIND=qPREC) :: dx,rl,rh,ri
      level=Info%level
      dx=levels(level)%dx

      mB=1
      mB(:,2)=Info%mx
      ip=mB
      IF (Info%level < MaxLevel) THEN
         SELECT CASE (nDim)
         CASE(2)
            ip(1:nDim,2)=mB(1:nDim,2)+1          
            ALLOCATE(correction(ip(1,1):ip(1,2), ip(2,1):ip(2,2),1))
            IF (iCylindrical==NoCyl) THEN     
               WHERE (Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1,1) /= undefined)
                  correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1) = &
                       Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1,1)- &
                       Info%emf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1,1)

                  Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1,1)= &
                       Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1,1)+ &
                       correction(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1  )

                  Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)-1:ip(2,2)-1,1,1)= &
                       Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)-1:ip(2,2)-1,1,1)- &
                       correction(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1  )

                  Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1,2)= &
                       Info%aux(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1,2)- &
                       correction(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1  )

                  Info%aux(ip(1,1)-1:ip(1,2)-1,ip(2,1)  :ip(2,2)  ,1,2)= &
                       Info%aux(ip(1,1)-1:ip(1,2)-1,ip(2,1)  :ip(2,2)  ,1,2)+ &
                       correction(ip(1,1)  :ip(1,2)  ,ip(2,1)  :ip(2,2)  ,1  )

                  Info%emf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1,1)= &
                       Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),1,1)
               END WHERE
            ELSE
               DO i=ip(1,1),ip(1,2) 
                  ri=(Info%xBounds(1,1)+(i-1)*dx)
                  rl=1.d0/(Info%xBounds(1,1)+(REAL(i)-half)*dx)
                  rh=1.d0/(Info%xBounds(1,1)+(REAL(i)+half)*dx)
                  WHERE (Info%childemf(i,ip(2,1):ip(2,2),1,1) /= undefined)
                     correction(i,ip(2,1):ip(2,2),1) = &
                          Info%childemf(i,ip(2,1):ip(2,2),1,1)- &
                          Info%emf(i,ip(2,1):ip(2,2),1,1)

                     Info%aux(i,ip(2,1)  :ip(2,2)  ,1,1)= &
                          Info%aux(i,ip(2,1)  :ip(2,2)  ,1,1)+ &
                          correction(i,ip(2,1)  :ip(2,2)  ,1  )

                     Info%aux(i  ,ip(2,1)-1:ip(2,2)-1,1,1)= &
                          Info%aux(i  ,ip(2,1)-1:ip(2,2)-1,1,1)- &
                          correction(i  ,ip(2,1)  :ip(2,2)  ,1  )

                     Info%aux(i  ,ip(2,1)  :ip(2,2)  ,1,2)= &
                          Info%aux(i  ,ip(2,1)  :ip(2,2)  ,1,2)- &
                          correction(i  ,ip(2,1)  :ip(2,2)  ,1  )*&
                          rh*ri

                     Info%aux(i-1,ip(2,1)  :ip(2,2)  ,1,2)= &
                          Info%aux(i-1,ip(2,1)  :ip(2,2)  ,1,2)+ &
                          correction(i  ,ip(2,1)  :ip(2,2)  ,1  )*&
                          rl*ri

                     Info%emf(i,ip(2,1):ip(2,2),1,1)= &
                          Info%childemf(i,ip(2,1):ip(2,2),1,1)
                  END WHERE
               END DO
            END IF!icyl
            DEALLOCATE(correction)
         CASE(3)
            DO i=1,3
               ip(1:nDim,2)=mB(1:nDim,2)+1          
               ip(i,2)=ip(i,2)-1
               l(2:3)=modulo((/i,i+1/),3)+1
               ia=ip
               ib=ip
               ia(l(3),:)=ip(l(3),:)-1
               ib(l(2),:)=ip(l(2),:)-1
               ALLOCATE(correction(ip(1,1):ip(1,2), ip(2,1):ip(2,2),ip(3,1):ip(3,2)))
               WHERE (Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i) /= undefined)
                  correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2)) = &
                       Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i)- &
                       Info%emf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i)

                  Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),l(2))= &
                       Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),l(2))+ &
                       correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))

                  Info%aux(ia(1,1):ia(1,2),ia(2,1):ia(2,2),ia(3,1):ia(3,2),l(2))= &
                       Info%aux(ia(1,1):ia(1,2),ia(2,1):ia(2,2),ia(3,1):ia(3,2),l(2))- &
                       correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))

                  Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),l(3))= &
                       Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),l(3))- &
                       correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))

                  Info%aux(ib(1,1):ib(1,2),ib(2,1):ib(2,2),ib(3,1):ib(3,2),l(3))= &
                       Info%aux(ib(1,1):ib(1,2),ib(2,1):ib(2,2),ib(3,1):ib(3,2),l(3))+ &
                       correction(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))

                  Info%emf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i)= &
                       Info%childemf(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i)
               END WHERE
               DEALLOCATE(correction)
            END DO
         END SELECT
      END IF
      CALL UpdateAux(Info, mB)
   END SUBROUTINE RestrictionFixup

   !> @brief Updates time derivatives
   !! @param Info - Info structure
   SUBROUTINE UpdateTimeDeriv(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i, j
      DO i=1, nProlongate
         DO j=1, TDVars
            IF (ProlongateFields(i) == TimeDerivFields(j)) THEN
               Info%qchild(:,:,:,i)=Info%q(:,:,:,TimeDerivFields(j))
            END IF
         END DO
      END DO
   END SUBROUTINE UpdateTimeDeriv


   SUBROUTINE ClearFixupFlux(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i
      DO i=1,nDim
         Info%fixupflux%side(i)%data=0
      END DO
      IF (MaintainAuxArrays) Info%emf=0
      IF (lStoreMassFlux) Info%MassFlux=0
   END SUBROUTINE ClearFixupFlux


   SUBROUTINE ClearParentFixup(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i
      DO i=1,nDim
         Info%parentfixup%side(i)%data=0d0
      END DO
      IF (MaintainAuxArrays) Info%parentemf=0d0
   END SUBROUTINE ClearParentFixup

   SUBROUTINE ClearChildFixup(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j
      IF (ASSOCIATED(Info%childfixups)) THEN
         DO j=1,size(Info%childfixups)
            DO i=1,nDim
               Info%childfixups(j)%p%side(i)%data=0d0
            END DO
         END DO
      END IF
      IF (MaintainAuxArrays) THEN
         IF (Associated(Info%childemf)) Info%childemf=undefined
         Info%AuxChild=Info%aux
      END IF
      Info%qChild=Info%q(:,:,:,ProlongateFields)
   END SUBROUTINE ClearChildFixup


   !> This routine sets ChildMask in the ghost zones to 0 where neighbors exist.
   SUBROUTINE ChildMaskOverlap(Info,neighbormGlobal)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mO,neighbormGlobal,mGlobal,ioffset
      INTEGER :: i,j,k,l
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=1!nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      mO(nDim+1:3,:)=1
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=neighbormGlobal(:,:)+SPREAD((/i,j,k/)*levels(Info%level)%mX(:),2,2)

               mO(1:nDim,1)=max(Info%mGlobal(1:nDim,1)-1,mGlobal(1:nDim,1)) 
               mO(1:nDim,2)=min(Info%mGlobal(1:nDim,2)+1,mGlobal(1:nDim,2))
               IF (ALL(mO(1:nDim,1) <= mO(1:nDim,2))) THEN
                  mO(1:nDim,:)=mO(1:nDim,:)-Spread( Info%mGlobal(1:nDim,1),2,2)+1
                  Info%ChildMask(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=0
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE ChildMaskOverlap


   !> This routine sets ChildMask to NEIGHBORCHILD in the ghost zones where neighbor children exist.
   SUBROUTINE UpdateChildMask(Info,neighborchildmGlobal)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mO,neighbormGlobal,mGlobal,ioffset,neighborchildmGlobal
      INTEGER :: i,j,k,l
      ioffset=0
      WHERE(lAnyPeriodic(1:nDim)) ioffset(1:nDim,2)=1!nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      mO(nDim+1:3,:)=1
      neighbormGlobal(nDim+1:3,:)=1
      neighbormGlobal(1:nDim,2)=neighborchildmGlobal(1:nDim,2)/levels(Info%level)%CoarsenRatio
      neighbormGlobal(1:nDim,1)=(neighborchildmGlobal(1:nDim,1)-1)/levels(Info%level)%CoarsenRatio+1
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)

               mGlobal(:,:)=neighbormGlobal(:,:)+SPREAD((/i,j,k/)*levels(Info%level)%mX(:),2,2)

               mO(1:nDim,1)=max(Info%mGlobal(1:nDim,1)-1,mGlobal(1:nDim,1)) 
               mO(1:nDim,2)=min(Info%mGlobal(1:nDim,2)+1,mGlobal(1:nDim,2))
               IF (ALL(mO(1:nDim,1) <= mO(1:nDim,2))) THEN
                  mO(1:nDim,:)=mO(1:nDim,:)-Spread( Info%mGlobal(1:nDim,1),2,2)+1
                  Info%ChildMask(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=NEIGHBORCHILD
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE UpdateChildMask


   !> This routine sets ChildMask in the ghost zones to NEIGHBORCHILD where periodioc copies of children exist.
   SUBROUTINE UpdateSelfChildMask(Info,childmGlobal)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mO,mymGlobal,mGlobal,ioffset,childmglobal
      INTEGER :: i,j,k,l
      ioffset=0
      WHERE(lHydroPeriodic(1:nDim)) ioffset(1:nDim,2)=1!nperiodic_overlaps(1:nDim)
      ioffset(1:nDim,1)=-ioffset(1:nDim,2)
      mO(nDim+1:3,:)=1
      mymglobal(nDim+1:3,:)=1
      mymglobal(1:nDim,2)=childmglobal(1:nDim,2)/levels(Info%level)%CoarsenRatio
      mymglobal(1:nDim,1)=(childmglobal(1:nDim,1)-1)/levels(Info%level)%CoarsenRatio+1
      DO i=ioffset(1,1),ioffset(1,2)
         DO j=ioffset(2,1),ioffset(2,2)
            DO k=ioffset(3,1),ioffset(3,2)
               if (ALL((/i,j,k/)==0)) CYCLE
               mGlobal(:,:)=mymglobal(:,:)+SPREAD((/i,j,k/)*levels(Info%level)%mX(:),2,2)

               mO(1:nDim,1)=max(Info%mGlobal(1:nDim,1)-1,mGlobal(1:nDim,1)) 
               mO(1:nDim,2)=min(Info%mGlobal(1:nDim,2)+1,mGlobal(1:nDim,2))
               IF (ALL(mO(1:nDim,1) <= mO(1:nDim,2))) THEN
                  mO(1:nDim,:)=mO(1:nDim,:)-Spread( Info%mGlobal(1:nDim,1),2,2)+1
                  Info%ChildMask(mO(1,1):mO(1,2), mO(2,1):mO(2,2), mO(3,1):mO(3,2))=NEIGHBORCHILD
               END IF
            END DO
         END DO
      END DO
   END SUBROUTINE UpdateSelfChildMask

   SUBROUTINE UpdateAux(Info, mB)
      TYPE(InfoDef) :: Info
      INTEGER :: i, mB(3,2), ip(3,2)
      DO i=1,nAux
         ip=mb
         ip(i,:)=mb(i,:)+1
         Info%q(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2),auxFields(i))=half*(&
              Info%aux(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2),i)+&
              Info%aux(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),i))
      END DO
   END SUBROUTINE UpdateAux



   RECURSIVE SUBROUTINE GetSplitSolution(Info,n,mB,nGrids, TotalCost, solution)
      TYPE(InfoDef) :: Info
      INTEGER :: n
      INTEGER, DIMENSION(3,2) :: mB, mB1, mB2
      INTEGER :: nGrids, nGrids1, nGrids2
      REAL(KIND=qPREC) :: TotalCost, TotalCost1, TotalCost2
      INTEGER, DIMENSION(:,:,:), POINTER :: solution, solution1, solution2
      REAL(KIND=qPREC), PARAMETER :: FillRatioTolerance=0.95
      LOGICAL :: HaveSplit
      TotalCost=ChildAdvanceCost(mB, Info%level)
      !      Write(*,*) 'mB= ', mB

      IF (GridFlagRatio(Info,mB) < FillRatioTolerance .AND. n > 0) THEN
         !Print *, 'Before FindBestSplit_i'
         CALL FindBestSplit_i(Info, mB, mB1, mB2, HaveSplit)
         !         Print *, 'HaveSplit=', HaveSplit
         IF (HaveSplit) Then
            CALL GetSplitSolution(Info, n-1, mB1, nGrids1, TotalCost1, solution1)
            CALL GetSplitSolution(Info, n-1, mB2, nGrids2, TotalCost2, solution2)
            IF (TotalCost1+TotalCost2 > TotalCost .OR. .NOT. HaveSplit) THEN       
               nGrids=1
               ALLOCATE(solution(3,2,1))
               solution(:,:,1)=mB                      
            ELSE
               !               write(*,*) "A", nGrids1, nGrids2
               nGrids=nGrids1+nGrids2
               ALLOCATE(solution(3,2,nGrids))
               solution(:,:,1:nGrids1)=solution1
               solution(:,:,nGrids1+1:nGrids)=solution2
               TotalCost=TotalCost1+TotalCost2
            END IF
            DEALLOCATE(solution1, solution2)

         ELSE
            nGrids=1
            ALLOCATE(solution(3,2,1))
            solution(:,:,1)=mB                
         END IF
      ELSE
         nGrids=1
         ALLOCATE(solution(3,2,1))
         solution(:,:,1)=mB                
      END IF
   END SUBROUTINE GetSplitSolution


   SUBROUTINE FindBestSplit_i(Info, mB, mB1new, mB2new, HaveSplit)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mB, mB1, mB2, mB1new, mB2new, ip
      LOGICAL :: HaveSplit
      INTEGER :: j, edge, level, i,  mx(3), splitdir, splitindx, maxm, maxdim, gaploc(3), maxsignature,odd
      REAL(KIND=qPREC) :: gapcost(3)
      INTEGER, DIMENSION(:,:), ALLOCATABLE :: signatures
      INTEGER, DIMENSION(:), ALLOCATABLE :: dSignatures
      ! Loop until no better grid splitting can be found
      HaveSplit=.FALSE.
      level=Info%level
      mx=mB(:,2)-mB(:,1)+1
      !      write(*,*) ' mx= ', mx
      maxdim=SUM(maxloc(mX(1:nDim))) !Get the largest dimension
      IF (mx(maxdim) < 2) RETURN
      HaveSplit=.true.
      splitdir=maxdim !Initialize default split dir
      splitindx=mx(maxdim)/2 ! Initialize default split loc
      !Look for holes from center out
      maxm=maxval(mX)
      gapcost=huge(gapcost)
      !     write(*,*) 'maxm=', maxm
      ALLOCATE(signatures(ndim,maxm))
      gaploc=0
      DO i=1,nDim
         ip=mB
         !   write(*,*) 'mX=', mX,'i=', i
         DO j=1,mX(i)
            ip(i,:)=j
            signatures(i,j)=sum(Info%ErrFlag(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2)))
            !signatures(i,1:mX(i))=sum(sum(Info%ErrFlag(mB(1,1):mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2)), mod(i,3)+1), mod(i+1,3)+1)       
         END DO
         odd = mod(mx(i),2)
         DO j=0, mx(i)/2-2+odd
            !            write(*,*) 'j=', j
            IF (signatures(i,mX(i)/2+j+1-odd) == 0) THEN
               CALL SplitShrink(Info, mB, i, mB(i,1)-1+mx(i)/2+j+1-odd, mB1, mB2)
               gapcost(i)=ChildAdvanceCost(mB1,level) + ChildAdvanceCost(mB2,level)
               gaploc(i)=mB(i,1)-1+mx(i)/2+j+1-odd
               EXIT
            END IF
            IF (signatures(i,mX(i)/2-j) == 0) THEN
               CALL SplitShrink(Info, mB, i, mB(i,1)-1+mx(i)/2-j, mB1, mB2)
               gapcost(i)=ChildAdvanceCost(mB1,level) + ChildAdvanceCost(mB2,level)
               gaploc(i)=mB(i,1)-1+mx(i)/2-j
               EXIT
            END IF
         END DO
      END DO

      IF (ANY(gaploc > 0)) THEN !Found a gap
         splitdir=sum(minloc(gapcost)) !Optimal split direction
         splitindx=gaploc(splitdir) !Location
         CALL SplitShrink(Info, mB, splitdir, splitindx, mB1new, mB2new) !Get subgrid bounds
         !         print *, 'find a gap'
      ELSE
         !Find inflections
         ALLOCATE(dSignatures(1:maxm))
         dSignatures=0
         !Search through second derivatives
         DO i=1,nDim
            dSignatures(1:mx(i)-1)=abs(signatures(i,2:mx(i))-(signatures(i,1:mx(i)-1)))
            maxsignature=-1
            odd=mod(mx(i),2)
            DO j=0, mx(i)/2-1
               !Write(*,*) 'j=', j
               IF (dSignatures(mX(i)/2-j) > maxsignature) THEN
                  CALL SplitShrink(Info, mB, i, mB(i,1)-1+mx(i)/2-j, mB1, mB2)
                  !                  Write(*,*) 'found inflection at ', mx(i)/2-j, ' dsig=',dSignatures(mX(i)/2-j)
                  gapcost(i)=ChildAdvanceCost(mB1,level) + ChildAdvanceCost(mB2,level)
                  gaploc(i)=mB(i,1)-1+mx(i)/2-j
                  maxsignature=dSignatures(mX(i)/2-j)
               END IF
               IF (dsignatures(mX(i)/2+j+odd) > maxsignature) THEN
                  !                  Write(*,*) 'found inflection at ', mx(i)/2+j+odd, ' dsig=',dSignatures(mX(i)/2+j+odd)
                  CALL SplitShrink(Info, mB, i, mB(i,1)-1+mx(i)/2+j+odd, mB1, mB2)
                  gapcost(i)=ChildAdvanceCost(mB1,level) + ChildAdvanceCost(mB2,level)
                  gaploc(i)=mB(i,1)-1+mx(i)/2+j+odd
                  maxsignature=dSignatures(mX(i)/2+j+odd)
               END IF
            END DO
         END DO
         splitdir=sum(minloc(gapcost)) !Optimal split direction
         splitindx=gaploc(splitdir) !Location
         !Write(*,*) 'found inflection splitdir=', splitdir, ' &splitindx=', splitindx
         CALL SplitShrink(Info, mB, splitdir, splitindx, mB1new, mB2new) !Get subgrid bounds
         !         write(*,'(A,8I5)') 'new suggested grids are', mB1new(1:2,:), mB2new(1:2,:)
         DEALLOCATE(dSignatures)
      END IF
      DEALLOCATE(signatures)
   END SUBROUTINE FindBestSplit_i


   SUBROUTINE SplitShrink(Info, mB, splitdir, splitindx, mB1new, mB2new)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mB, mB1new, mB2new, ip
      INTEGER :: splitdir, splitindx, m, edge
      mB1new=mB(:,:)
      mB2new=mB(:,:)         

      mB1new(splitdir,2) = splitindx
      mB2new(splitdir,1) = splitindx+1

      DO m=1,ndim
         DO edge=1,2
            IF (.NOT. (m == splitdir .AND. edge == 1)) THEN
               !shrink mB1 along dim and edge
               ip=mB1new
               ip(m,:)=mB1new(m,edge)
               DO WHILE (ALL(Info%ErrFlag(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))==0))
                  ip(m,:)=ip(m,:)-(-1)**edge
               END DO
               mB1new(m,edge)=ip(m,1)
            END IF
            IF (.NOT. (m == splitdir .AND. edge == 2)) THEN
               !shrink mB2 along dim and edge
               ip=mB2new
               ip(m,:)=mB2new(m,edge)
               DO WHILE (ALL(Info%ErrFlag(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))==0))
                  ip(m,:)=ip(m,:)-(-1)**edge
               END DO
               mB2new(m,edge)=ip(m,1)
            END IF
         END DO
      END DO
   END SUBROUTINE SplitShrink


   SUBROUTINE FindBestSplit(Info, mB, mB1new, mB2new, HaveSplit)
      TYPE(InfoDef) :: Info
      INTEGER, DIMENSION(3,2) :: mB, mB1, mB2, mB1new, mB2new, ip
      LOGICAL :: HaveSplit
      INTEGER :: n, m, edge, level, i, i1,i2
      REAL(KIND=qPREC) :: MinSplitCost, SplitCost
      ! Loop until no better grid splitting can be found
      MinSplitCost=HUGE(MinSplitCost)
      HaveSplit=.FALSE.
      level=Info%level
      ! check splitting position along x direction ===============
      !write(*,'(A,4I5)') 'considering grid ', mB(1:2,:)


      DO n=1,ndim
         i1=mB(n,1); i2=mB(n,2)

         DO i=i1,i2-1

            mB1=mB(:,:)
            mB2=mB(:,:)         

            mB1(n,:)=i
            if(all(Info%ErrFlag(mB1(1,1):mB1(1,2), mB1(2,1):mB1(2,2), mB1(3,1):mB1(3,2))==0)) CYCLE

            CALL SplitShrink(Info, mB, n, i, mB1, mB2)
            ! Shrink along y direction
            SplitCost = ChildAdvanceCost(mB1,level) + ChildAdvanceCost(mB2,level);             
            !write(*,'(A, 8I5, E25.15)') 'evaluating ', mB1(1:2,:), mB2(1:2,:), SplitCost   
            if(SplitCost < MinSplitCost) then
               MinSplitCost = SplitCost
               HaveSplit = .TRUE.
               mB2new=mB2
               mB1new=mB1
               !write(*,*) 'x spitting at ', i, splitcost
               !write(*,'(100I5)') mB1(1:2,:), mB2(1:2,:)
               !write(*,*) 'mB now = ', mB
            end if
         END DO
      END DO
      !=====================================

   END SUBROUTINE FindBestSplit


   !> This routine choose either OldNewSubGrids which corresponding to the Old grid generating Algo,
   !! and NewNewSubGrids which corresponding to the New grid-generating Algo

   SUBROUTINE NewSubGrids(Info, nSubGrids, child_box_array)
      Type(InfoDef), POINTER :: Info
      INTEGER, POINTER, DIMENSION(:,:,:) :: child_box_array
      ! Internal declarations
      INTEGER :: nSubGrids
      IF (lUseOriginalNewSubGrids) THEN
         CALL OldNewSubGrids(Info, nSubGrids, child_box_array)
      ELSE
         CALL NewNewSubGrids(Info, nSubGrids, child_box_array)
      END IF
   END SUBROUTINE NewSubGrids

   !> Old grid-generating algo
   !! This routine creates an array of desired refinement regions based on the errflag array
   !! @param Info Info structure
   !! @param nSubGrids Number of created regions
   !! @param child_box_array Array containing created regions
   SUBROUTINE OldNewSubGrids(Info, nSubGrids, child_box_array)
      ! Implementation of Berger-Rigoutsos algorithm (IEEE Trans. Systems, Man &
      ! Cyber., 21(5):1278-1286, 1991

      ! Interface declarations
      TYPE (InfoDef), POINTER :: Info
      INTEGER, POINTER, DIMENSION(:,:,:) :: child_box_array

      ! Internal declarations
      INTEGER :: nSubGrids
      INTEGER nGrid,min_level
      INTEGER, DIMENSION(3,2) :: mBounds
      INTEGER, DIMENSION(3,2,MAX_SUBGRIDS) :: mSubBounds

      INTEGER, PARAMETER :: MAX_SPLIT_PASSES=40
      LOGICAL HaveSplit,CanSplitGrid,CanSplit(MAX_SUBGRIDS)
      INTEGER i,n,iError,i1,i2,i1L,i2L,iGrid,inflect,del,maxm,minm,npass,level,nd
      INTEGER iSplit(MAX_DIMS),iErr,DomDecSplits(4)
      INTEGER, ALLOCATABLE, DIMENSION (:,:) :: Signature,ddSignature
      REAL FillRatio,DesiredFillRatio,rand

      DomDecSplits=0

      mBounds(1:nDim,1)=1
      mBounds(1:nDim,2)=Info%mX(1:nDim)

      level=Info%level
      mSubBounds = 1
      DesiredFillRatio = levels(level)%DesiredFillRatios
      nSubGrids=0
      ! Compute fill ratio for this grid
      FillRatio=GridFlagRatio(Info,mBounds)

      IF (FillRatio==zero) RETURN

      ! Allocate space for signatures
      maxm=MAXVAL(Info%mX(1:nDim))
      ALLOCATE(Signature(maxm,nDim),ddSignature(maxm,nDim), STAT=iError)
      IF (iError /= 0) THEN
         PRINT *,'NewSubGrids() error: Unable to allocate signature arrays.'
         STOP
      END IF

      Signature=0
      ddSignature=0

      ! Initialize list of subgrids
      nGrid=1
      CanSplit(:)=.TRUE.

      mSubBounds(1:nDim,1:2,nGrid)=mBounds(1:nDim,1:2)
      iGrid=1

      ! Loop until no better grid splitting can be found
      DO WHILE (nGrid<MAX_SUBGRIDS .AND. iGrid<=nGrid)
         npass=0
         DO WHILE (CanSplit(iGrid) .AND. npass<MAX_SPLIT_PASSES)
            npass=npass+1
            CALL CalcSignatures(Info,mSubBounds(:,:,iGrid),maxm,Signature)

            ! Trim unflagged points on the edges of this grid
            DO n=1,nDim
               i1=mSubBounds(n,1,iGrid); i2=mSubBounds(n,2,iGrid);
               DO WHILE ( Signature(i1,n)==0 .AND. i1<mSubBounds(n,2,iGrid) .AND. &
                    i2-i1+1 > MinimumGridPoints )
                  i1=i1+1
               END DO
               DO WHILE ( Signature(i2,n)==0 .AND. i2>mSubBounds(n,1,iGrid)  .AND. &
                    i2-i1+1 > MinimumGridPoints )
                  i2=i2-1
               END DO

               mSubBounds(n,1,iGrid)=i1
               mSubBounds(n,2,iGrid)=i2
            END DO

            FillRatio=GridFlagRatio(Info,mSubBounds(:,:,iGrid))

            minm=MINVAL(mSubBounds(1:nDim,2,iGrid)-mSubBounds(1:nDim,1,iGrid))+1

            CALL CalcSignatures(Info,mSubBounds(:,:,iGrid),maxm,Signature)

            ! Look for holes along which to split grid
            iSplit=0
            HaveSplit=.FALSE.

            DO n=1,nDim
               i1=mSubBounds(n,1,iGrid); i2=mSubBounds(n,2,iGrid)

               i1L=i1;i2L=i2               
               DO i=i1L,i2L-1
                  IF (Signature(i,n)==0 .AND. MIN(i-i1+1,i2-i) >= MinimumGridPoints) THEN
                     iSplit=0
                     iSplit(n)=i
                     HaveSplit=.TRUE.
                     EXIT
                  END IF
               END DO

               IF (HaveSplit) EXIT
            END DO


            IF (FillRatio<DesiredFillRatio .AND. .NOT. HaveSplit) THEN
               ! No split along a hole. Try split along inflection point
               DO n=1,nDim

                  i1=mSubBounds(n,1,iGrid)
                  i2=mSubBounds(n,2,iGrid)

                  DO i=i1+1,i2-1
                     ddSignature(i,n)=Signature(i-1,n)-2*Signature(i,n)+Signature(i+1,n)
                  END DO

               END DO

               inflect=0

               DO n=1,nDim
                  i1=mSubBounds(n,1,iGrid)
                  i2=mSubBounds(n,2,iGrid)
                  i1L=i1;i2L=i2

                  DO i=i1L,i2L-1
                     del=ABS(ddSignature(i+1,n)-ddSignature(i,n))
                     IF (del>inflect .AND. MIN(i-i1+1,i2-i) >= MinimumGridPoints) THEN
                        inflect=del
                        iSplit=0
                        iSplit(n)=i
                        HaveSplit=.TRUE.
                     END IF
                  END DO

               END DO

            END IF
            IF (HaveSplit) THEN
               ! Split the grid along a determined line
               DO n=1,nDim
                  IF (iSplit(n)>0 .AND. &
                       MIN( mSubBounds(n,2,iGrid)-iSplit(n) , &
                       iSplit(n)-mSubBounds(n,1,iGrid)+1 ) >= MinimumGridPoints) THEN
                     ! Add a new subgrid to the end of the grid list
                     nGrid=nGrid+1

                     CanSplit(nGrid)=.TRUE.
                     mSubBounds(1:nDim,1:2,nGrid)=mSubBounds(1:nDim,1:2,iGrid)
                     mSubBounds(n,1,nGrid)=iSplit(n)+1
                     ! Replace current grid with a subgrid
                     mSubBounds(n,2,iGrid)=iSplit(n)

                     EXIT
                  END IF
               END DO
            ELSE
               ! Mark grid if no split is possible
               CanSplit(iGrid)=.FALSE.
            END IF
         END DO
         iGrid=iGrid+1
      END DO

      DEALLOCATE(Signature,ddSignature,STAT=iError)
      IF (iError /= 0) THEN
         PRINT *,'Error deallocating signatures arrays in NewSubGrids'
         STOP
      END IF

      DO iGrid=1,nGrid
         IF (ALL(Info%ErrFlag(mSubBounds(1,1,iGrid):mSubBounds(1,2,iGrid), &
              mSubBounds(2,1,iGrid):mSubBounds(2,2,iGrid), &
              mSubBounds(3,1,iGrid):mSubBounds(3,2,iGrid))==0)) THEN
            DO i=iGrid+1,nGrid
               mSubBounds(:,:,i-1)=mSubBounds(:,:,i)
            END DO
            nGrid=nGrid-1
         END IF
      END DO

      ALLOCATE(child_box_array(3,2,nGrid), STAT=iErr)

      child_box_array(:,:,1:nGrid) = mSubBounds(:,:,1:nGrid)
      nSubGrids=nGrid

   END SUBROUTINE OldNewSubGrids

   !> New grid-generating Algo
   !! This routine creates an array of desired refinement regions based on the errflag array
   !! @param Info Info structure
   !! @param nSubGrids Number of created regions
   !! @param child_box_array Array containing created regions
   SUBROUTINE NewNewSubGrids(Info, nSubGrids, child_box_array)
      ! Implementation of Berger-Rigoutsos algorithm (IEEE Trans. Systems, Man &
      ! Cyber., 21(5):1278-1286, 1991

      ! Interface declarations
      TYPE (InfoDef), POINTER :: Info
      INTEGER, POINTER, DIMENSION(:,:,:) :: child_box_array

      ! Internal declarations
      INTEGER :: nSubGrids
      INTEGER nGrid,min_level
      INTEGER, DIMENSION(3,2) :: mBounds
      INTEGER, DIMENSION(3,2,MAX_SUBGRIDS) :: mSubBounds

      INTEGER, PARAMETER :: MAX_SPLIT_PASSES=40
      LOGICAL HaveSplit,CanSplitGrid,CanSplit
      INTEGER i,n,iError,i1,i2,i1L,i2L,iGrid,inflect,del,maxm,minm,npass,level,nd,m, edge
      INTEGER iSplit(MAX_DIMS),iErr,DomDecSplits(4)
      INTEGER, ALLOCATABLE, DIMENSION (:,:) :: Signature,ddSignature

      REAL FillRatio,DesiredFillRatio,rand
      REAL(KIND=qPREC) :: TotalCost

      INTEGER, DIMENSION(3,2) :: mB, ip

      mB(:,1)=1
      mB(:,2)=Info%mX

      level=Info%level
      !print *, "nSubGrids=", nSubGrids

      nSubGrids=0
      ! Compute fill ratio for this grid
      FillRatio=GridFlagRatio(Info,mB)

      IF (FillRatio==zero) RETURN

      ! Initialize list of subgrids

      DO m=1,ndim
         DO edge=1,2
            !shrink mB1 along dim and edge
            ip=mB
            ip(m,:)=mB(m,edge)
            DO WHILE (ALL(Info%ErrFlag(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2))==0))
               ip(m,:)=ip(m,:)-(-1)**edge
            END DO
            mB(m,edge)=ip(m,1)
         END DO
      END DO

      !print *, "nSubGrids=", nSubGrids
      CALL GetSplitSolution(Info, MAX_SPLIT_PASSES, mB, nSubGrids, TotalCost, child_box_array)

   END SUBROUTINE NewNewSubGrids

   SUBROUTINE CalcSignatures(Info,mSubBounds,maxm,Signature)
      ! Interface declarations
      TYPE (InfoDef) :: Info
      INTEGER mSubBounds(MAX_DIMS,2),maxm
      INTEGER Signature(maxm,nDim)
      ! Internal declarations
      INTEGER i,n,i1(MAX_DIMS),i2(MAX_DIMS)
      Signature=0
      i1=1; i2=1
      DO n=1,nDim
         i1(1:nDim)=mSubBounds(1:nDim,1)
         i2(1:nDim)=mSubBounds(1:nDim,2)
         DO i=mSubBounds(n,1),mSubBounds(n,2)
            i1(n)=i; i2(n)=i
            Signature(i,n)=SUM(Info%ErrFlag(i1(1):i2(1), &
                 i1(2):i2(2), &
                 i1(3):i2(3)) )
         END DO
      END DO
   END SUBROUTINE CalcSignatures

   REAL FUNCTION GridFlagRatio(Info,mBounds)
      ! Interface declarations
      TYPE (InfoDef) :: Info
      INTEGER mBounds(MAX_DIMS,2)
      ! Internal declarations
      REAL Total,Flagged

      Total=PRODUCT(mBounds(1:nDim,2)-mBounds(1:nDim,1)+1)
      mBounds(nDim+1:MAX_DIMS,:)=1
      Flagged = SUM(REAL(Info%ErrFlag(mBounds(1,1):mBounds(1,2), &
           mBounds(2,1):mBounds(2,2), &
           mBounds(3,1):mBounds(3,2)) ))
      GridFlagRatio=Flagged/Total


   END FUNCTION GridFlagRatio

   FUNCTION  GetMyCosts(Info, step)
      TYPE(InfoDef) :: Info
      INTEGER :: mx(3), bc, i, step
      REAL(KIND=qPREC) :: GetMyCosts
      GetMyCosts=0
      bc=levels(Info%level)%ambc(step)
      mx=1
      mx(1:nDim)=Info%mx(1:nDim)+2*bc
      GetMyCosts=GetMyCosts+AdvanceCost(mx)
   END Function GetMyCosts

   ! !> This routine calculates the cost of a child grid with bounds mB
   ! !! @param info Info structure
   ! !! @param mB Bounds of child in parent indexing space
   ! !! @param n level of parent
   REAL FUNCTION GetChildCosts(mB,level)
      INTEGER, DIMENSION(3,2) :: mB
      INTEGER :: n, r,i,bc,mx(3), level
      GetChildCosts=0
      IF (level >= 0) THEN
         DO i=1,levels(level)%steps
            bc=levels(level)%ambc(i)
            mx=1
            mx(1:nDim)=(mB(1:nDim,2)-mB(1:nDim,1)+1)+2*bc
            GetChildCosts=GetChildCosts+AdvanceCost(mx)
         END DO
      ELSE
         GetChildCosts=product(mB(:,2)-mB(:,1)+1)*tiny(1e0)
      END IF
   END FUNCTION GetChildCosts
   !> This routine calculates the subtree cost associated with a refinement region
   !! @param info Info structure
   !! @param mB Bounds of child in parent indexing space
   !! @param n level of parent
   REAL FUNCTION GetSubTreeCost(info, mB)
      TYPE(InfoDef), POINTER :: info
      INTEGER, DIMENSION(3,2) :: mB
      GetSubTreeCost=SUM(info%costmap(mB(1,1):mB(1,2), mB(2,1):mB(2,2), mB(3,1):mB(3,2),1))
   END FUNCTION GetSubTreeCost
   !> @}

END MODULE DataInfoOps
 
