!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    data_level_ops.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir data
!! @brief Contains amr-related modules for managing info structures.

!> @file data_level_ops.f90
!! @brief Main file for module DataLevelOps

!> @defgroup DataOps Data Operations
!! @ingroup AMR

!> @defgroup DataLevelOps Data Level Operations
!> @brief Main module for managing AMR related data operations by level
!! @ingroup DataOps

!> Main module for managing AMR related data operations by level
!! @ingroup DataLevelOps
MODULE DataLevelOps
   USE DataInfoOps
   USE HyperbolicControl
   USE ModuleControl
   USE HyperbolicDeclarations
   USE TreeDeclarations
   USE Scheduling
   USE Timing
   USE DataLevelComms
   IMPLICIT NONE
   PRIVATE

   !Pair-wise data operations   
   PUBLIC ProlongateParentsData, ApplyOverlaps, ApplyChildrenData, ApplyInitialChildrenData, SyncFluxes

   !Single node data operations   
   PUBLIC InitInfos, InitialInitInfos, AfterOverlaps, UpdateChildMasks, SetErrFlags, RestrictionFixups, &
        AccumulateFluxes, CoarsenDataForParents, CoarsenInitialDataForParents, InitGrids, AdvanceGrids, &
        ApplyPhysicalBCs, AfterFixups, ChildMaskOverlaps, ScheduledAdvanceGrids, CompleteAdvanceGrids, WaitingAdvances, UpdateTimeDerivs, ClearFixupFluxes, ClearParentFixups, ClearChildFixups

   PUBLIC InitRestartGrids, UpdateMeanDensity

   PUBLIC GenericTransfer,  TransferTimeDerivs

CONTAINS 

   !> Manages Ghosting of fields in q on grids on level n for nghost cells
   !! @param n level
   !! @param fields index array to transfer
   !! @param nghost number of ghost cells to fill
   SUBROUTINE GenericTransfer(n,fields,nghost, lPeriodic, lNeighbors_opt)
      INTEGER :: n   
      INTEGER, DIMENSION(:) :: fields
      INTEGER :: nghost
      LOGICAL, DIMENSION(3) :: lPeriodic
      LOGICAL :: lNeighbors     !ghosting with neighbors or overlaps?
      LOGICAL, OPTIONAL :: lNeighbors_opt
      IF (PRESENT(lNeighbors_opt)) THEN
         lNeighbors=lNeighbors_opt
      ELSE
         lNeighbors=.true.
      END IF
      CALL PostRecvGenericData(n, fields, nghost, lPeriodic, lNeighbors)  
      CALL PostSendGenericData(n,fields,nghost, lPeriodic, lNeighbors)
      CALL ApplyGenericOverlaps(n,fields,nghost, lPeriodic, lNeighbors)
      CALL CompRecvGenericData(n,fields)
      CALL CompSendGenericData(n)
   END SUBROUTINE GenericTransfer

  SUBROUTINE TransferTimeDerivs(n)
     INTEGER :: n
     IF (TDVars > 0) THEN
        CALL GenericTransfer(n, TimeDerivFields, levels(0)%egmbc(1), (/.true.,.true.,.true./), lNeighbors_opt=.false.)
     END IF
  END SUBROUTINE TransferTimeDerivs

   !> Ghosts fields in q on local grids on level n for nghost cells
   !! @param n level
   !! @param fields index array to transfer
   !! @param nghost number of ghost cells to fill
   SUBROUTINE ApplyGenericOverlaps(n,fields,nghost, lPeriodic, lNeighbors)
      INTEGER :: n,step
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(NodeDefList), POINTER :: nodelist, neighborlist
      INTEGER, DIMENSION(:) :: fields
      INTEGER :: nghost
      LOGICAL, DIMENSION(3) :: lPeriodic
      LOGICAL :: lNeighbors
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         IF (lNeighbors) THEN
            neighborlist=>node%neighbors
         ELSE
            neighborlist=>node%overlaps
         END IF
         DO WHILE (associated(neighborlist))
            neighbor=>neighborlist%self 
            IF (neighbor%box%MPI_ID == MPI_ID) CALL ApplyGenericOverlap(node%info,neighbor%info,n,fields,nghost, lPeriodic) 
            neighborlist=>neighborlist%next
         END DO
         CALL ApplyGenericOverlap(node%info,node%info,n,fields,nghost, lPeriodic)
         nodelist=>nodelist%next 
      END DO
   END SUBROUTINE ApplyGenericOverlaps




   !> @name Pair-wise Data Operations
   !! @{

   !> Prolongate data from the node's parent
   !! @param   n level
   SUBROUTINE ProlongateParentsData(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, parent
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iProlongateParentsData, n)

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         parent => node%parent
         IF (.NOT. ASSOCIATED(parent)) THEN
            write(*,*) "Error - no parent associated in ProlongateParentsData"
            STOP
         END IF

         IF (parent%box%MPI_id == MPI_id)  CALL ProlongateParentData(parent%info, node%info)

         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iProlongateParentsData, n)

   END SUBROUTINE ProlongateParentsData


   !> Synchronize data with node's neighbors
   !! @param n level
   !! @param step current level step
   SUBROUTINE ApplyOverlaps(n,step)
      INTEGER :: n,step
      TYPE(NodeDef), POINTER :: node, overlap
      TYPE(NodeDefList), POINTER :: nodelist, overlaplist
      CALL StartTimer(iApplyOverlaps,n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         overlaplist=>node%overlaps 
         DO WHILE (associated(overlaplist))
            overlap=>overlaplist%self 
            IF (overlap%box%MPI_ID == MPI_ID) CALL ApplyOverlap(node%info,overlap%info,n) 
            overlaplist=>overlaplist%next
         END DO
         IF (step == 2 .OR. .NOT. lRegridLevel(n)) CALL ApplyOverlap(node%info,node%info,n)
         nodelist=>nodelist%next 
      END DO
      CALL StopTimer(iApplyOverlaps,n)          
   END SUBROUTINE ApplyOverlaps

   !> Apply restricted data from the node's children
   !! @param n level
   SUBROUTINE ApplyChildrenData(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, child
      TYPE(NodeDefList), POINTER :: nodelist, childlist
      CALL StartTimer(iApplyChildrenData, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         childlist => node%children
         DO WHILE (ASSOCIATED(childlist))
            child => childlist%self
            IF (child%box%MPI_ID == MPI_ID) CALL ApplyChildData(node%info, child%info, GetChildID(node,child),n)
            childlist => childlist%next
         END DO
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iApplyChildrenData, n)

   END SUBROUTINE ApplyChildrenData

   !> Apply restricted initial data from the node's children
   !! @param n level
   SUBROUTINE ApplyInitialChildrenData(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, child
      TYPE(NodeDefList), POINTER :: nodelist, childlist

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         childlist => node%children
         DO WHILE (ASSOCIATED(childlist))
            child => childlist%self
            IF (child%box%MPI_ID == MPI_ID) CALL ApplyInitialChildData(node%info, child%info, n)
            childlist => childlist%next
         END DO
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE ApplyInitialChildrenData


   !> Synchronize fluxes with neighbors
   !! @param n level  
   SUBROUTINE SyncFluxes(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(NodeDefList), POINTER :: nodelist, neighborlist
      CALL StartTimer(iSyncFluxes, n)

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         neighborlist=>node%neighbors 
         DO WHILE (associated(neighborlist))
            neighbor=>neighborlist%self
            IF (neighbor%box%MPI_ID == MPI_ID) CALL SyncFlux(node%info,neighbor%info) 
            neighborlist=>neighborlist%next
         END DO

         CALL SyncFlux(node%info,node%info)
         nodelist=>nodelist%next 

      END DO
      CALL StopTimer(iSyncFluxes, n)
   END SUBROUTINE SyncFluxes

   !> @}

   !> @name Single-Node Data Operations
   !! @{

   !> Fill in missing prolongated aux fields and calculate the cost of the grid.
   !! @param n level
   SUBROUTINE AfterOverlaps(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iAfterOverlaps, n)
      IF (MaintainAuxArrays) THEN
         nodelist=>Nodes(n)%p 
         DO WHILE (associated(nodelist))
            node=>nodelist%self
            IF (MaintainAuxArrays) CALL ProlongationFixup(node%info)
            nodelist=>nodelist%next
         END DO
      END IF
      CALL StopTimer(iAfterOverlaps, n)
   END SUBROUTINE AfterOverlaps

   !> Apply Physical Boundary Conditions
   !! @param n level
   SUBROUTINE ApplyPhysicalBCs(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iApplyPhysicalBCs, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         IF (.NOT. ASSOCIATED(node%info)) THEN
            print*, 'woops',MPI_ID, n,node%box, node%iD
            STOp
         END IF

         CALL ApplyPhysicalBC(node%info)         
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iApplyPhysicalBCs, n)

   END SUBROUTINE ApplyPhysicalBCs


   !> Apply fixup emf collected from all children
   !! @param n level
   SUBROUTINE RestrictionFixups(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iRestrictionFixups, n)

      IF (MaintainAuxArrays) THEN
         nodelist=>Nodes(n)%p 
         DO WHILE (associated(nodelist))
            node=>nodelist%self

            CALL RestrictionFixup(node%info)
            nodelist=>nodelist%next
         END DO
      END IF
      CALL StopTimer(iRestrictionFixups, n)

   END SUBROUTINE RestrictionFixups


   !> Update Time Derivatives in qchild
   !! @param n level
   SUBROUTINE UpdateTimeDerivs(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self            
         CALL UpdateTimeDeriv(node%info)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE UpdateTimeDerivs

   !> Clear Fixup Fluxes and emfs
   !! @param n level 
   SUBROUTINE ClearFixupFluxes(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self            
         CALL ClearFixupFlux(node%info)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE ClearFixupFluxes

   !> Clear Parent Fixup Fluxes and emfs
   !! @param n level 
   SUBROUTINE ClearParentFixups(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self            
         CALL ClearParentFixup(node%info)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE ClearParentFixups

   !> Clear Child Fixup Fluxes and emfs
   !! @param n level 
   SUBROUTINE ClearChildFixups(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self            
         CALL ClearChildFixup(node%info)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE ClearChildFixups

   !> Call module routines after updating level with children data
   !! @param n level
   SUBROUTINE AfterFixups(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iAfterFixups, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         CALL AfterFixup(node%info)
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iAfterFixups, n)
   END SUBROUTINE AfterFixups


   !> Update ChildMask to NEIGHBOR_CHILD where neighbor's have children
   !! @param n level   
   SUBROUTINE UpdateChildMasks(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, neighbor,neighborchild, child
      TYPE(NodeDefList), POINTER :: nodelist
      TYPE(NodeDefList), POINTER :: neighborlist
      TYPE(NodeDefList), POINTER :: neighborchildlist
      TYPE(NodeDefList), POINTER :: childlist
      CALL StartTimer(iUpdateChildMasks, n)

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         neighborlist=>node%neighbors
         DO WHILE (associated(neighborlist))
            neighbor=>neighborlist%self
            neighborchildlist=>neighbor%children
            DO WHILE (associated(neighborchildlist))
               neighborchild=>neighborchildlist%self
               CALL UpdateChildMask(node%info, neighborchild%box%mGlobal)
               neighborchildlist=>neighborchildlist%next
            END DO

            neighborlist=>neighborlist%next
         END DO
         childlist=>node%children
         DO WHILE (ASSOCIATED(childlist))
            child=>childlist%self
            CALL UpdateSelfChildMask(node%info,child%box%mglobal)
            childlist=>childlist%next
         END DO
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iUpdateChildMasks, n)
   END SUBROUTINE UpdateChildMasks


   !> Set the Error Flags
   !! @param n level
   SUBROUTINE SetErrFlags(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iSetErrFlags, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL SetErrFlag(node%info) 
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iSetErrFlags, n)
   END SUBROUTINE SetErrFlags

   !> Coarsen fluxes and cell-centered data for parents
   !! @param n level   
   SUBROUTINE CoarsenDataForParents(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iCoarsenDataForParents,n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL CoarsenDataForParent(node%info, n) 
         nodelist=>nodelist%next

        CALL StopTimer(iCoarsenDataForParents,n)
      END DO
   END SUBROUTINE CoarsenDataForParents


   !> Coarsen cell-centered and face-centered data for parents
   !! @param n level   
   SUBROUTINE CoarsenInitialDataForParents(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL CoarsenInitialDataForParent(node%info, n) 
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE CoarsenInitialDataForParents

   !> Store fluxes for this time step in parent accumulator
   !! @param n level   
   SUBROUTINE AccumulateFluxes(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      CALL StartTimer(iAccumulateFluxes, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL AccumulateFlux(node%info) 
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iAccumulateFluxes, n)
   END SUBROUTINE AccumulateFluxes

   !> Initialize Info structures
   !! @param n level   
   SUBROUTINE InitInfos(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(NodeDefList), POINTER :: nodelist
      TYPE(NodeDefList), POINTER :: neighborlist
      CALL StartTimer(iInitInfos, n)
      NumCellsByLevel(n)=0
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL InitInfo(node%info, n,node%box%mGlobal, node%parent%box%mGlobal)
         NumCellsByLevel(n)=NumCellsByLevel(n)+product(node%info%mx)
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iInitInfos, n)

   END SUBROUTINE InitInfos

   !> Set ChildMask to 0 internally and where neighbors exist
   !> @param n level
   SUBROUTINE ChildMaskOverlaps(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(NodeDefList), POINTER :: nodelist
      TYPE(NodeDefList), POINTER :: neighborlist
      CALL StartTimer(iChildMaskOverlaps,n)
           
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self
         neighborlist=>node%neighbors
         DO WHILE (associated(neighborlist))
            neighbor=>neighborlist%self
            CALL ChildMaskOverlap(node%info, neighbor%box%mGlobal)
            neighborlist=>neighborlist%next
         END DO
         CALL ChildMaskOverlap(node%info, node%box%mGlobal)
         nodelist=>nodelist%next
      END DO
      CALL StopTimer(iChildMaskOverlaps,n)
   END SUBROUTINE ChildMaskOverlaps

   !> Initialize Infos the first time.
   !! @param n level   
   SUBROUTINE InitialInitInfos(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(NodeDefList), POINTER :: nodelist
      TYPE(NodeDefList), POINTER :: neighborlist
      NumCellsByLevel(n)=0
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL InitialInitInfo(node%info, n,node%box%mGlobal, node%parent%box%mGlobal)
         NumCellsByLevel(n)=NumCellsByLevel(n)+product(node%info%mx)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE InitialInitInfos


   !> Initialize data at beginning of simulation
   !! @param n level   
   SUBROUTINE InitGrids(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist

      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL GridInit(node%info)
         nodelist=>nodelist%next
      END DO
   END SUBROUTINE InitGrids

   !! @brief An adaptation of InitGrids geared toward restarts.
   !! @param level The level of the grids being initialized.
   SUBROUTINE InitRestartGrids(level)
      INTEGER :: level

      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist

      nodelist=>Nodes(level)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL GridInit(node%info)
         nodelist=>nodelist%next
      END DO

   END SUBROUTINE InitRestartGrids

   !> Routine to advance grids
   !! @param n level  
   SUBROUTINE AdvanceGrids(n)
      INTEGER :: n,i
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      if (levels(n)%dt == 0) RETURN
      CALL StartTimer(iAdvanceGrids, n)
      nodelist=>Nodes(n)%p 
      DO WHILE (associated(nodelist))
         node=>nodelist%self 
         CALL Advance(node%info)                 
         nodelist=>nodelist%next 
      END DO
      levels(n)%tnow=levels(n)%tnow+levels(n)%dt
      CALL StopTimer(iAdvanceGrids, n)                  
   END SUBROUTINE AdvanceGrids


   !> Routine that schedules advances for level
   !! @param n level  
   SUBROUTINE ScheduledAdvanceGrids(n)
      INTEGER :: n,i
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      INTEGER :: level
      AdvanceNodeListByLevel(n)%p=>Nodes(n)%p 
      IF (n == MaxLevel .AND. MaxLevel > 0) THEN
         CALL StartTimer(iScheduledAdvanceGrids, n)
         CALL CalcEstimatedWaitingTimes(n)
         CALL StopTimer(iScheduledAdvanceGrids, n)
      END IF
   END SUBROUTINE ScheduledAdvanceGrids

   !> Routine that finishes advancing grids for a level
   !! @param n level
   SUBROUTINE CompleteAdvanceGrids(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      if (levels(n)%dt == 0) RETURN
!      CALL StartTimer(iCompleteAdvancedGrids, n)
      CALL StartTimer(iAdvanceGrids, n)

      DO WHILE (ASSOCIATED(AdvanceNodeListByLevel(n)%p))             
         node=>AdvanceNodeListByLevel(n)%p%self            
         CALL Advance(node%info, .true.)
         AdvanceNodeListByLevel(n)%p=>AdvanceNodeListByLevel(n)%p%next
      END DO
      levels(n)%tnow=levels(n)%tnow+levels(n)%dt

      CALL StopTimer(iAdvanceGrids, n)
!      CALL StopTimer(iCompleteAdvancedGrids, n)                  

   END SUBROUTINE CompleteAdvanceGrids

   !> Routine that advances level n grids and pre-advances coarser grids if there is time
   !! @param n
   SUBROUTINE WaitingAdvances(n)
      INTEGER :: n, level,i
      TYPE(NodeDef), POINTER :: node
      REAL(KIND=qPREC) :: WorkDoneThisStep
!      CALL StartTimer(iWaitingAdvances, n)
      level=n-1
      if (levels(n)%dt == 0) RETURN
!      IF (WaitingTimes(n) <= 0) RETURN
      
      IF (StartTime+SyncTime(n) < mpi_wtime()) THEN
!         write(*,'(A,I4,A,12E15.3)') 'exiting late on processor ', MPI_ID, ' at ', mpi_wtime()-StartTime-Synctime(n), ((/WorkDoneByLevel(i),WorkLoadByLevelPerStep(i,levels(i)%step)/), i=0,MaxLevel)         
         RETURN
      END IF
!      TimeAvailableToSolver=WaitingTimes(n)
      TimeAvailableToSolver=StartTime+SyncTime(n) !-mpi_wtime() !WaitingTimes(n)
!      write(*,*) MPI_ID, 'waiting advances ', n, WaitingTimes(n)
      DO WHILE (level >= 0)
         DO WHILE (ASSOCIATED(AdvanceNodeListByLevel(level)%p))             
         CALL StartTimer(iAdvanceGrids, level)

            node=>AdvanceNodeListByLevel(level)%p%self
            CALL Advance(node%info)
!            TimeAvailableToSolver=TimeAvailableToSolver-TimeUsedBySolver(level)
            IF (NodeCompleted(level)) THEN
               AdvanceNodeListByLevel(level)%p=>AdvanceNodeListByLevel(level)%p%next
            ELSE
               CALL StopTimer(iAdvanceGrids, level)
               EXIT
            END IF
            CALL StopTimer(iAdvanceGrids, level)
         END DO

         IF (StartTime+SyncTime(n) < mpi_wtime()) EXIT !TimeAvailableToSolver <= 0) EXIT
         level=level-1
      END DO
      IF (mpi_wtime() < StartTime+SyncTime(n)) THEN
!         write(*,'(A,I4,A,12E15.3)') 'exiting early on processor ', MPI_ID, ' at ', mpi_wtime()-StartTime-SyncTime(n), ((/WorkDoneByLevel(i), WorkLoadByLevelPerStep(i,levels(i)%step)/), i=0,MaxLevel)
!      write(*,'(I4,A,E25.15)') MPI_ID, 'exiting WaitingAdvances at t = ', mpi_wtime()-StartTime
      END IF
      !      CALL StopTimer(iWaitingAdvances, n)
   END SUBROUTINE WaitingAdvances
   !> @}

  ! Routine for calculating mean density on root level
  SUBROUTINE UpdateMeanDensity(n)
    INTEGER :: n
    REAL(KIND=qPREC) :: total_mass
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    INTEGER :: iErr
    TYPE(InfoDef), POINTER :: info
    INTEGER :: i, mB(3,2)
    total_mass=0
    DO i=0,n
       nodelist=>Nodes(i)%p
       DO WHILE(ASSOCIATED(nodelist))
          info=>nodelist%self%info
          mB=1
          mB(1:nDim,2)=Info%mX(1:nDim)
          IF (i < n) THEN
             total_mass=total_mass+SUM(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),irho), Info%ChildMask(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3)) <= 0)*levels(i)%dx**nDim
          ELSE
             total_mass=total_mass+SUM(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),irho))*levels(i)%dx**nDim 
          END IF
          nodelist=>nodelist%next
       END DO
    END DO
    CALL StartTimer(iBarrier, 0)
    CALL MPI_ALLREDUCE(MPI_IN_PLACE, total_mass, 1, MPI_DOUBLE_PRECISION, MPI_SUM,levels(0)%MPI_COMM, iErr)
    CALL StopTimer(iBarrier, 0)
    mean_density=total_mass/(product(GxBounds(1:nDim,2)-GxBounds(1:nDim,1)))
!    IF (MPI_ID == 0) write(*,*) "mean density=", mean_density, total_mass, n
  END SUBROUTINE UpdateMeanDensity


END MODULE DataLevelOps

