!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    data_parsing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file data_parsing.f90
!! @brief Main file for module DataParsing

!> @defgroup DataParsing Data Parsing
!! @brief Performs necessary communication required for pair-wise info operations 
!! and calls appropriate pair-wise info routine when unparsing messages.
!! Intermediate between DataLevelComms and DataInfoOps
!! @ingroup DataComms

!> Performs necessary communication required for pair-wise info operations 
!! and calls appropriate pair-wise info routine when unparsing messages.
!! Intermediate between DataLevelComms and DataInfoOps
!! @ingroup DataParsing
MODULE DataParsing

   USE MPIPacking
   USE MessageDeclarations
   USE DataInfoOps
   USE TreeDeclarations
   IMPLICIT NONE

CONTAINS

   !> @name ProlongateParentDataParsing Routines for parsing data for ProlongateParentData
   !! @{

   !> Precalculates the buffer size needed to send child data to each processor.
   !! @param level The level of the nodes doing the sending.
   !! @param node The local node
   !! @param child The external child node
   INTEGER FUNCTION SendChildrenData_Precalculate(level, node, child)

      INTEGER :: level
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: child

      INTEGER, DIMENSION(3,2) :: mB
      INTEGER, DIMENSION(3,2) :: ip
      INTEGER :: accumulator
      INTEGER :: rmbc

      ! child%box%mGlobal
      accumulator = (6 * PACK_INTEGER_SIZE)

      IF (level > -1) THEN
         rmbc=levels(level)%pmbc
          ! node%info%qChild
          mB = GetChildmBounds(node, child%box%mGlobal, level)
          ip = stretch(mB, rmbc)
          accumulator = accumulator + &
              (SIZE(node%info%qChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:nProlongate)) * PACK_DOUBLE_SIZE)

          ! node%info%auxChild
          IF (MaintainAuxArrays) THEN
              ip = stretchaux(mB,rmbc)
              accumulator = accumulator + &
                 (SIZE(node%info%auxChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:nAux)) * PACK_DOUBLE_SIZE)
          END IF

      END IF

      SendChildrenData_Precalculate = accumulator

   END FUNCTION SendChildrenData_Precalculate

   !> @brief Packs data for children necessary for ProlongateParentData
   !! (CostPerCell, [level > 0: qChild, ChildMask, [MaintainAuxArrays: aux]])
   !! @param message The message to pack to
   !! @param node The local node
   !! @param child The external child node
   SUBROUTINE SendChildrenData(message, node, child)
      TYPE(NodeDef), POINTER :: node, child
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(3,2) :: mB, ip
      INTEGER :: rmbc
      INTEGER :: counter = 0

      CALL PackData(message, child%box%mGlobal) !Given the child - the parent can be found - but not vice-versa

      IF (message%level > -1) THEN
         rmbc=levels(message%level)%pmbc
         mB=GetChildmBounds(node,child%box%mGlobal,message%level)
         ip=stretch(mB, rmbc)

         CALL PackData(message, node%info%qChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),:))
         IF (MaintainAuxArrays) THEN
            ip=stretchaux(mB,rmbc)
            
            CALL PackData(message, node%info%auxChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),:))
         END IF
      END IF
   END SUBROUTINE SendChildrenData

   !> Calculate the amount of buffer space a specific node will need to receive for RecvParentsData().
   !! @param parent_level The level of node's parent.
   !! @param node The node structure whose parent is sending data.
   INTEGER FUNCTION RecvParentsData_Precalculate(parent_level, node)

      INTEGER :: parent_level
      TYPE(NodeDef), POINTER :: node

      INTEGER :: accumulator
      INTEGER, DIMENSION(3,2) :: mB
      INTEGER, DIMENSION(3,2) :: ip
      INTEGER, DIMENSION(3) :: ip_mx
      INTEGER :: rmbc

      ! Add the size of the child grid, which will be sent over and caught by StrictGetNextBox().
      accumulator = (6 * PACK_INTEGER_SIZE)

      IF (parent_level > -1) THEN
          rmbc=levels(parent_level)%pmbc
          ! Use the node's bounds within its parent to calculate the dimensions of the prolongated parent data that will
          ! be sent over.
          mB = LevelDown(node%box%mGlobal, parent_level+1, parent_level)
          ip = stretch(mB, rmbc)
          
          ip_mx = ip(:,2) - ip(:,1) + 1

          ! parent%info%qChild
          accumulator = accumulator + (PRODUCT(ip_mx) * nProlongate * PACK_DOUBLE_SIZE)

          ! parent%info%auxChild
          IF (MaintainAuxArrays) THEN
              ip = stretchaux(mB, rmbc)
              ip_mx = ip(:,2) - ip(:,1) + 1
              accumulator = accumulator + (PRODUCT(ip_mx) * nAux * PACK_DOUBLE_SIZE)
          END IF

       END IF

       RecvParentsData_Precalculate = accumulator

     END FUNCTION RecvParentsData_Precalculate


   !> @brief UnPacks data from Parents necessary for ProlongateParentData
   !! (CostPerCell, [level > 0: qChild, ChildMask, [MaintainAuxArrays: aux]])
   !! @param message The message to unpack from
   SUBROUTINE RecvParentsData(message)
     TYPE(NodeDef), POINTER :: node, parent
     TYPE(PackedMessage), POINTER :: message
     TYPE(NodeBox) :: node_box
     INTEGER :: level
     INTEGER, DIMENSION(3,2) :: mB, ip
     TYPE(InfoDef), POINTER :: ParentInfo
     INTEGER :: rmbc

     node_box%MPI_ID=MPI_ID
     level=message%level+1
     rmbc=levels(message%level)%pmbc
    DO WHILE(StrictGetNextBox(message, node_box%mGlobal, "RecvParentsData"))

        CALL StrictFindNode(level,node_box,node, "RecvParentsData(node)")
        parent=>node%parent

        ALLOCATE(parentInfo)
        CALL NullifyInfoFields(parentInfo)

        IF (message%level > -1) THEN
           mB=node%Info%mBounds

           ip=stretch(mB,rmbc)

           ALLOCATE(parentInfo%qChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nProlongate))

           CALL UnPackData(message, parentInfo%qChild)

           IF (MaintainAuxArrays) THEN
              ip=stretchaux(mB,rmbc)
              ALLOCATE(parentInfo%auxChild(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nAux))
              CALL UnPackData(message, parentInfo%auxChild)
           END IF
        END IF

        CALL ProlongateParentData(ParentInfo, Node%Info)
        IF (message%level > -1) THEN
           DEALLOCATE(parentInfo%qChild)
           IF (MaintainAuxArrays) DEALLOCATE(parentInfo%auxChild)
        END IF
        DEALLOCATE(parentInfo)

     END DO

   END SUBROUTINE RecvParentsData
   !> @}

   !> @name ApplyOverlapDataParsing Routines for parsing data for ApplyOverlap
   !! @{


   !> Calculate the amount of buffer space a specific node will need to receive for SendOverlapData().
   !! @param level The level of the overlapping data being transferred.
   !! @param node The local node sending overlaps.
   !! @param node The placeholder node for the overlap on another system.
   INTEGER FUNCTION SendOverlapData_Precalculate(level, node, overlap)

      INTEGER :: level
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: overlap

      INTEGER :: accumulator
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,dir,j
      INTEGER :: EGCopyFieldsSize


      ! Add the size of overlap%box%mGlobal.
      accumulator = PACK_BOX_SIZE

      IF (level > -1) THEN

         CALL CalcOverlaps(overlap%box%mGlobal, node%box%mGlobal, mTs, mSs, nOverlaps, level, lHydroPeriodic, levels(level)%gmbc(levels(level)%step))

         ! nOverlaps again.
         accumulator = accumulator + PACK_INTEGER_SIZE

         IF (nOverlaps > 0) THEN
            !Add the size of the overlap arrays.
            accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

            DO i=1,nOverlaps
               mS=mSs(i,:,:)
               mT=mTs(i,:,:)

               ! node%info%q using overlap source boundaries and the size of the GCopyFields array.
               accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * GVars * PACK_DOUBLE_SIZE)

            END DO

            DEALLOCATE(mTs,mSs)
            NULLIFY(mTs,mSs)
         END IF


         IF (level > BaseLevel .AND. ((level == 0 .AND. levels(level)%step <= 1) .OR. levels(level)%step > 1) .AND. EGVars > 0) THEN
!         IF (level >= BaseLevel .AND. (level == 0 .OR. levels(level)%step > 1) .AND. EGVars > 0) THEN 

            CALL CalcOverlaps(overlap%box%mGlobal, node%box%mGlobal, mTs, mSs, nOverlaps, level, lEllipticPeriodic, levels(level)%egmbc(levels(level)%step))

            ! nOverlaps again.
            accumulator = accumulator + PACK_INTEGER_SIZE

            IF (nOverlaps > 0) THEN
               !Add the size of the overlap arrays.
               accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

               EGCopyFieldsSize = SIZE(EGCopyFields)

               DO i=1,nOverlaps
                  mS=mSs(i,:,:)
                
                     accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * EGCopyFieldsSize * PACK_DOUBLE_SIZE)
               END DO
               DEALLOCATE(mTs,mSs)
               NULLIFY(mTs,mSs)
            END IF
         END IF

         IF (MaintainAuxArrays) THEN
            DO dir=1,nDim
               CALL CalcAuxOverlaps(overlap%box%mGlobal, node%box%mGlobal, mTs, mSs, nOverlaps, level, dir,lHydroPeriodic)

               ! nOverlaps again.
               accumulator = accumulator + PACK_INTEGER_SIZE

               IF (nOverlaps > 0) THEN
                  !Add the size of the overlap arrays.
                  accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

                  DO i=1,nOverlaps
                     mS=mSs(i,:,:)
                     ! node%info%aux using overlap source dimensions and the dir array.
                     accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * PACK_DOUBLE_SIZE)
                  END DO

                  DEALLOCATE(mTs,mSs)
                  NULLIFY(mTs,mSs)
               END IF
            END DO
         END IF

      END IF
      SendOverlapData_Precalculate = accumulator

   END FUNCTION SendOverlapData_Precalculate


   !> @brief Packs data for overlaps necessary for ApplyOverlap
   !! (CostPerCell, CostMap, [level > -1: q,[MaintainAuxArrays: aux]])
   !! @param message The message to pack to
   !! @param node The local node
   !! @param overlap The external overlap node
   SUBROUTINE SendOverlapData(message, node, overlap)
      TYPE(NodeDef), POINTER :: node, overlap
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,dir,j

      CALL PackData(message, overlap%box%mGlobal)

      IF (message%level > -1) THEN

         CALL CalcOverlaps(overlap%box%mGlobal, node%box%mGlobal,mTs,mSs,nOverlaps,message%level,lHydroPeriodic, levels(message%level)%gmbc(levels(message%level)%step))
         CALL PackData(message, nOverlaps)

         IF (nOverlaps > 0) THEN
            CALL PackData(message, mTs)
            CALL PackData(message, mSs)

            DO i=1,nOverlaps
               mT=mTs(i,:,:)
               mS=mSs(i,:,:)
               CALL PackData(message, node%info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),GCopyFields))

            END DO
            DEALLOCATE(mTs,mSs)
            NULLIFY(mTs,mSs)
         END IF

         IF (message%level > BaseLevel .AND. ((message%level == 0 .AND. levels(message%level)%step <= 1) .OR. levels(message%level)%step > 1) .AND. EGVars > 0) THEN
!         IF (message%level >= BaseLevel .AND. (message%level == 0 .OR. levels(message%level)%step > 1) .AND. EGVars > 0) THEN 

            CALL CalcOverlaps(overlap%box%mGlobal, node%box%mGlobal,mTs,mSs,nOverlaps,message%level, lEllipticPeriodic, levels(message%level)%egmbc(levels(message%level)%step))
            CALL PackData(message, nOverlaps)
            
            IF (nOverlaps > 0) THEN
               CALL PackData(message, mTs)
               CALL PackData(message, mSs)
               
               DO i=1,nOverlaps
                  mT=mTs(i,:,:)
                  mS=mSs(i,:,:)
                  DO j=1,size(EGCopyFields)
                     CALL PackData(message, node%info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),EGCopyFields(j)))
                  END DO
               END DO
               DEALLOCATE(mTs,mSs)
               NULLIFY(mTs,mSs)
            END IF
         END IF

         IF (MaintainAuxArrays) THEN
            DO dir=1,nDim
               CALL CalcAuxOverlaps(overlap%box%mGlobal, node%box%mGlobal,mTs,mSs,nOverlaps,message%level,dir,lHydroPeriodic)
               CALL PackData(message, nOverlaps)

               IF (nOverlaps > 0) THEN
                  CALL PackData(message, mTs)
                  CALL PackData(message, mSs)

                  DO i=1,nOverlaps
                     mT=mTs(i,:,:)
                     mS=mSs(i,:,:)
                     CALL PackData(message, node%info%aux(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),dir:dir))

                  END DO
                  DEALLOCATE(mTs,mSs)
                  NULLIFY(mTs,mSs)
               END IF
            END DO
         END IF

      END IF

   END SUBROUTINE SendOverlapData

   !> Calculate the amount of buffer space a specific node will need to receive for RecvOverlapData().
   !! @param level The level of the overlapping data being transferred.
   !! @param node The local node sending overlaps.
   !! @param node The placeholder node for the overlap on another system.
   INTEGER FUNCTION RecvOverlapData_Precalculate(level, node, overlap)

      INTEGER :: level
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: overlap

      INTEGER :: accumulator
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,dir,j
      INTEGER :: EGCopyFieldsSize


      ! The size of a localnodebox global array.
      accumulator = 6 * PACK_INTEGER_SIZE

      IF (level > -1) THEN

          CALL CalcOverlaps(node%box%mGlobal, overlap%box%mGlobal, mTs, mSs, nOverlaps, &
                            level, lHydroPeriodic, levels(level)%gmbc(levels(level)%step))

          ! nOverlaps
          accumulator = accumulator + PACK_INTEGER_SIZE

          IF (nOverlaps > 0) THEN

             !Add the size of the overlap arrays.
             accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

             DO i=1,nOverlaps
                mS=mSs(i,:,:)
                mT=mTs(i,:,:)

                ! overlapInfo%q using source array and Gvars field.
                accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * GVars * PACK_DOUBLE_SIZE)
             END DO

             DEALLOCATE(mTs,mSs)
             NULLIFY(mTs,mSs)
          END IF

         IF (level > BaseLevel .AND. ((level == 0 .AND. levels(level)%step <= 1) .OR. levels(level)%step > 1) .AND. EGVars > 0) THEN
!          IF (level >= BaseLevel .AND. (level == 0 .OR. levels(level)%step > 1) .AND. EGVars > 0) THEN 

              CALL CalcOverlaps(node%box%mGlobal, overlap%box%mGlobal, mTs, mSs, nOverlaps, level, lEllipticPeriodic, levels(level)%egmbc(levels(level)%step))
                
              ! nOverlaps
              accumulator = accumulator + PACK_INTEGER_SIZE

              IF (nOverlaps > 0) THEN
                  
                  !Add the size of the overlap arrays.
                  accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

                  DO i=1,nOverlaps
                     mS=mSs(i,:,:)
                     DO j=1,EGVars
                        ! overlapInfo%q using source array and EGCopyFields(j) field.
                        accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * PACK_DOUBLE_SIZE)
                     END DO
                  END DO

                  DEALLOCATE(mTs,mSs)
                  NULLIFY(mTs,mSs)
              END IF
          END IF

          IF (MaintainAuxArrays) THEN
              DO dir=1,nDim
                  CALL CalcAuxOverlaps(node%box%mGlobal, overlap%box%mGlobal, mTs, mSs, nOverlaps, level, dir,lHydroPeriodic)

                  ! nOverlaps again.
                  accumulator = accumulator + PACK_INTEGER_SIZE

                  IF (nOverlaps > 0) THEN
                      !Add the size of the overlap arrays.
                      accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)
                      DO i=1,nOverlaps
                         mS=mSs(i,:,:)
                         ! overlapInfo%aux using source array and dir field.
                         accumulator = accumulator + (PRODUCT(mS(:,2) - mS(:,1) + 1) * PACK_DOUBLE_SIZE)
                      END DO

                      DEALLOCATE(mTs,mSs)
                      NULLIFY(mTs,mSs)
                  END IF
              END DO
          END IF
      END IF
      RecvOverlapData_Precalculate = accumulator

   END FUNCTION RecvOverlapData_Precalculate

   !> @brief UnPacks data from overlaps necessary for ApplyOverlaps
   !! @param message The message to unpack from
   SUBROUTINE RecvOverlaps(message)
      TYPE(NodeDef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message
      TYPE(InfoDef) :: overlapInfo
      TYPE(NodeBox) :: localnodebox
      INTEGER :: level, nOverlaps,i,dir
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
      INTEGER, DIMENSION(3,2) :: mT, mS

      INTEGER :: j


      localnodebox%MPI_ID=MPI_ID
      level=message%level
      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvOverlaps"))

         CALL NullifyInfoFields(overlapInfo)

         CALL StrictFindNode(level, localnodebox,node, "RecvOverlaps(node)")

         IF (message%level > -1) THEN
            CALL UnPackData(message, nOverlaps)
            IF (nOverlaps > 0) THEN
               ALLOCATE(mTs(nOverlaps,3,2),mSs(nOverlaps,3,2))
               CALL UnPackData(message, mTs)
               CALL UnPackData(message, mSs)

               DO i=1,nOverlaps
                  mT=mTs(i,:,:)
                  mS=mSs(i,:,:)
                  ALLOCATE(overlapinfo%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),GVars))
                  CALL UnPackData(message, overlapinfo%q)
                  CALL ApplySingleOverlap(node%info,overlapInfo,mS,mT,GCopyFields)
                  DEALLOCATE(overlapInfo%q)
                  NULLIFY(overlapInfo%q)
               END DO
               DEALLOCATE(mTs,mSs)
               NULLIFY(mTs,mSs)
            END IF

            IF (message%level > BaseLevel .AND. ((message%level == 0 .AND. levels(message%level)%step <= 1) .OR. levels(message%level)%step > 1) .AND. EGVars > 0) THEN
!            IF (message%level >= BaseLevel .AND. (message%level == 0 .OR. levels(message%level)%step > 1) .AND. EGVars > 0) THEN 

               CALL UnPackData(message, nOverlaps)
               
               IF (nOverlaps > 0) THEN
                  ALLOCATE(mTs(nOverlaps,3,2),mSs(nOverlaps,3,2))
                  CALL UnPackData(message, mTs)
                  CALL UnPackData(message, mSs)
                  
                  DO i=1,nOverlaps
                     mT=mTs(i,:,:)
                     mS=mSs(i,:,:)
                     DO j=1,EGVars
                        ALLOCATE(overlapinfo%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),EGCopyFields(j):EGCopyFields(j)))
                        CALL UnPackData(message, overlapinfo%q)
                        CALL ApplySingleOverlap(node%info,overlapInfo,mS,mT,EGCopyFields(j:j))
                        DEALLOCATE(overlapInfo%q)
                        NULLIFY(overlapInfo%q)
                     END DO
                  END DO
                  DEALLOCATE(mTs,mSs)
                  NULLIFY(mTs,mSs)
               END IF
            END IF
            
            IF (MaintainAuxArrays) THEN
               DO dir=1,nDim
                  CALL UnPackData(message, nOverlaps)
                  IF (nOverlaps > 0) THEN
                     ALLOCATE(mTs(nOverlaps,3,2),mSs(nOverlaps,3,2))
                     CALL UnPackData(message, mTs)
                     CALL UnPackData(message, mSs)
                     DO i=1,nOverlaps
                        mT=mTs(i,:,:)
                        mS=mSs(i,:,:)
                        ALLOCATE(overlapinfo%aux(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),dir:dir))
                        CALL UnPackData(message, overlapinfo%aux)
                        CALL ApplySingleAuxOverlap(node%info,overlapInfo,mS,mT,dir)
                        DEALLOCATE(overlapInfo%aux)
                        NULLIFY(overlapInfo%aux)
                     END DO
                     DEALLOCATE(mTs,mSs)
                     NULLIFY(mTs,mSs)
                  END IF
               END DO
            END IF
         END IF
      END DO
   END SUBROUTINE RecvOverlaps

   !> @}

   !> @name ApplyChildDataParsing Routines for parsing data for ApplyChildData
   !! @{

   !> Calculate the cost of sending a node's parent data to its parent.
   !! @param child_level The level of the node whose data will be sent.
   !! @parent The parent who will receive the node's data..
   !! @node The node whose data will be sent to its parent.
   INTEGER FUNCTION SendParentsData_Precalculate(child_level, parent, node)

       INTEGER :: child_level
       TYPE(NodeDef), POINTER :: parent
       TYPE(NodeDef), POINTER :: node

       INTEGER :: accumulator
       INTEGER, DIMENSION(3,2) :: ip
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER :: n


       ! Initializes the accumulator with the cost of the node's global array.
       accumulator = 6 * PACK_INTEGER_SIZE

       IF (child_level > 0) THEN

           ip = GetChildmBounds(parent,node%box%mGlobal,child_level - 1)

           ip_mx(:) = ip(:,2) - ip(:,1) + 1

           ! The cost of the child's parentEMF fields.
           IF (MaintainAuxArrays) THEN
               ip_mx(1:nDim)=ip_mx(1:nDim)+1
               accumulator = accumulator + (PRODUCT(ip_mx) * nEMF * PACK_DOUBLE_SIZE)
               ip_mx(1:nDim)=ip_mx(1:nDim)-1
           END IF

           accumulator = accumulator + (PRODUCT(ip_mx) * nRestrict * PACK_DOUBLE_SIZE)

           ! The projected cost of the parentFixup data that will be transferred from the child.
           DO n=1,nDim
              ip_mx(n) = 2   ! The AllocBoundary() routine would set the array size along this dimension to 2.
              accumulator = accumulator + (PRODUCT(ip_mx) * nFlux * PACK_DOUBLE_SIZE)
              ip_mx(n)=ip(n,2)-ip(n,1)+1
           END DO

       END IF

       SendParentsData_Precalculate = accumulator

   END FUNCTION SendParentsData_Precalculate

   !> @brief Packs data for parent necessary for ApplyChildData 
   !! (mGlobal, ParentCostmap, [level > -1: qParent, parentfixup, [MaintainAuxArrays: parentemf]])
   !! @param message The message to pack to
   !! @param node The local node
   SUBROUTINE SendParentsData(message, node)
      TYPE(NodeDef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message

      CALL PackData(message, node%box%mGlobal)  !Child Box should already exit on the parent proc
      IF (message%level > -1) THEN
         IF (MaintainAuxArrays) CALL PackData(message, node%Info%ParentEMF)
         CALL PackData(message, node%Info%qParent)
         CALL PackData(message, node%Info%ParentFixup)
      END IF      
   END SUBROUTINE SendParentsData

   !> Calculates the amount of buffer space required for the parent to receive this child's parent data.
   !! @param parent_level The level of the parent node.
   !! @param node The parent node.
   !! @param child The child node that will be sending its data to the parent.
   INTEGER FUNCTION RecvChildrenData_Precalculate(parent_level, node, child)
       INTEGER :: parent_level
       TYPE(NodeDef), POINTER :: node
       TYPE(NodeDef), POINTER :: child

       INTEGER :: accumulator
       INTEGER, DIMENSION(3,2) :: ip
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER :: n


       ! Initialize the accumulator to the size of the child's six-integer global array.
       accumulator = 6 * PACK_INTEGER_SIZE

       ip = GetChildmBounds(node,child%box%mGlobal,parent_level)
       ip_mx(:) = ip(:,2) - ip(:,1) + 1

       ! On grid-level nodes, there will be EMF and fixup data to transfer to parents.  This
       ! counts against the message size.
       IF (parent_level > -1) THEN

           ! The cost of the child's parentEMF fields.
           IF (MaintainAuxArrays) THEN
               ip_mx(1:nDim)=ip_mx(1:nDim)+1
               accumulator = accumulator + (PRODUCT(ip_mx) * nEMF * PACK_DOUBLE_SIZE)
               ip_mx(1:nDim)=ip_mx(1:nDim)-1
           END IF

           ! The cost of the qParent data.
           accumulator = accumulator + (PRODUCT(ip_mx) * nRestrict * PACK_DOUBLE_SIZE)

           ! The projected cost of the parentFixup data that will be transferred from the child.
           DO n=1,nDim
              ip_mx(n) = 2   ! The AllocBoundary() routine would set the array size along this dimension to 2.
              accumulator = accumulator + (PRODUCT(ip_mx) * nFlux * PACK_DOUBLE_SIZE)
              ip_mx(n)=ip(n,2)-ip(n,1)+1
           END DO
       END IF

       RecvChildrenData_Precalculate = accumulator

   END FUNCTION RecvChildrenData_Precalculate

   !> @brief UnPacks data from children necessary for ApplyChildrenData 
   !! (mGlobal, ParentCostmap, [level > -1: qParent, parentfixup, [MaintainAuxArrays: parentemf]])
   !! @param message The message to unpack from
   SUBROUTINE RecvChildrenData(message)
      TYPE(NodeDef), POINTER :: node, child
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeBox) :: childbox
      INTEGER :: level, j
      INTEGER, DIMENSION(3,2) :: ip
      TYPE(InfoDef), POINTER :: ChildInfo


      childbox%MPI_ID=message%remote_proc
      level=message%level

      DO WHILE(StrictGetNextBox(message, childbox%mGlobal, "RecvChildrenData"))
         CALL StrictFindNode(level+1,childbox,child, "RecvChildrenData(child)")
         node=>child%parent

         ALLOCATE(ChildInfo)                 
         CALL NullifyInfoFields(ChildInfo)
         ChildInfo%level=level+1
         ChildInfo%mBounds(:,:)=GetChildmBounds(node,childbox%mGlobal,level)
         ip=ChildInfo%mBounds

         IF (level > -1) THEN
            ALLOCATE(ChildInfo%qParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nRestrict))

            IF (MaintainAuxArrays) THEN
               ip(1:nDim,2)=ip(1:nDim,2)+1
               ALLOCATE(ChildInfo%parentEmf(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1:nEMF))
               CALL UnPackData(message, ChildInfo%ParentEmf)
               ip(1:nDim,2)=ip(1:nDim,2)-1
            END IF

            CALL AllocBoundaries(ChildInfo%ParentFixup, ip)            

            CALL UnPackData(message, ChildInfo%qParent)
            CALL UnPackData(message, ChildInfo%ParentFixup)
         END IF

         CALL ApplyChildData(node%info, ChildInfo, GetChildID(node, child), level)
         IF (level > -1) THEN
            DEALLOCATE(ChildInfo%qParent)
            IF (MaintainAuxArrays) THEN
               DEALLOCATE(ChildInfo%parentemf)
            END IF
            CALL DeAllocBoundaries(ChildInfo%ParentFixup)
         END IF
         DEALLOCATE(ChildInfo)
         NULLIFY(ChildInfo)
      END DO

   END SUBROUTINE RecvChildrenData
   !> @}

   !> @name ApplyInitialChildDataParsing Routines for parsing data for ApplyIntialChildData
   !! @{

   !> Calculate the cost of sending a node's parent data to its parent (first timestep only).
   !! @param child_level The level of the node whose data will be sent.
   !! @node The node whose data will be sent to its parent.
   INTEGER FUNCTION SendParentsInitialData_Precalculate(child_level, node)

       INTEGER :: child_level
       TYPE(NodeDef), POINTER :: node

       INTEGER :: accumulator
       INTEGER, DIMENSION(3,2) :: ip
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER :: n


       ! Initializes the accumulator with the cost of the node's global array.
       accumulator = PACK_BOX_SIZE


       ! On grid-level nodes, there will be cell-centered and face-centered data to transfer to parents.  This
       ! counts against the message size.
       IF (child_level > 0) THEN

           ip = GetChildmBounds(node%parent,node%box%mGlobal,child_level - 1)
           ip_mx(:) = ip(:,2) - ip(:,1) + 1

           accumulator = accumulator + (PRODUCT(ip_mx) * nRestrict * PACK_DOUBLE_SIZE)

           IF (MaintainAuxArrays) THEN
               ip(1:nDim,2)=ip(1:nDim,2)+1
               ip_mx(:) = ip(:,2) - ip(:,1) + 1
               accumulator = accumulator + (PRODUCT(ip_mx) * nAux * PACK_DOUBLE_SIZE)
               ip(1:nDim,2)=ip(1:nDim,2)-1
           END IF
       END IF

       SendParentsInitialData_Precalculate = accumulator

   END FUNCTION SendParentsInitialData_Precalculate

   !> @brief Packs data for parent necessary for ApplyInitialChildData
   !! (mGlobal, ParentCostmap, [level > -1: qParent, [MaintainAuxArrays: auxParent]])
   !! @param message The message to pack to
   !! @param node The local node
   SUBROUTINE SendParentsInitialData(message, node)
      TYPE(NodeDef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message

      CALL PackData(message, node%box%mGlobal)  !Child Box should already exit on the parent proc

      IF (message%level > -1) THEN
         CALL PackData(message, node%Info%qParent)
         IF (MaintainAuxArrays)  CALL PackData(message, node%Info%auxParent)
      END IF      

   END SUBROUTINE SendParentsInitialData

   !> Calculates the amount of buffer space required for the parent to receive this child's parent data (first timestep only).
   !! @param parent_level The level of the parent node.
   !! @param node The parent node.
   !! @param child The child node that will be sending its data to the parent.
   INTEGER FUNCTION RecvInitialChildrenData_Precalculate(parent_level, node, child)
       INTEGER :: parent_level
       TYPE(NodeDef), POINTER :: node
       TYPE(NodeDef), POINTER :: child

       INTEGER :: accumulator
       INTEGER, DIMENSION(3,2) :: ip
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER :: n


       accumulator = PACK_BOX_SIZE


       ! On grid-level nodes, there will be EMF and fixup data to transfer to parents.  This
       ! counts against the message size.
       IF (parent_level > -1) THEN

           ip = GetChildmBounds(node,child%box%mGlobal,parent_level)
           ip_mx(:) = ip(:,2) - ip(:,1) + 1

           ! The cost of the qParent data.
           accumulator = accumulator + (PRODUCT(ip_mx) * nRestrict * PACK_DOUBLE_SIZE)

           ! The cost of the child's parentEMF fields.
           IF (MaintainAuxArrays) THEN
               ip(1:nDim,2)=ip(1:nDim,2)+1
               ip_mx(:) = ip(:,2) - ip(:,1) + 1
               accumulator = accumulator + (PRODUCT(ip_mx) * nAux * PACK_DOUBLE_SIZE)
               ip(1:nDim,2)=ip(1:nDim,2)-1
           END IF

       END IF

       RecvInitialChildrenData_Precalculate = accumulator

   END FUNCTION RecvInitialChildrenData_Precalculate

   !> @brief UnPacks data from children necessary for ApplyInitialChildData
   !! (mGlobal, ParentCostmap, [level > -1: qParent, [MaintainAuxArrays: auxParent]])
   !! @param message The message to unpack from
   SUBROUTINE RecvInitialChildrenData(message)
     TYPE(NodeDef), POINTER :: node, child
     TYPE(PackedMessage), POINTER :: message
     TYPE(NodeBox) :: childbox
     INTEGER :: level
     INTEGER, DIMENSION(3,2) :: ip
     TYPE(InfoDef), POINTER :: ChildInfo

     childbox%MPI_ID=message%remote_proc
     level=message%level

     DO WHILE(StrictGetNextBox(message, childbox%mGlobal, "RecvInitialChildrenData"))

        NULLIFY(child)
        CALL StrictFindNode(level+1,childbox,child, "RecvInitialChildrenData(child)")

        node=>child%parent

        ALLOCATE(ChildInfo)                 
        CALL NullifyInfoFields(ChildInfo)

        ChildInfo%level=level+1
        ChildInfo%mBounds(:,:)=GetChildmBounds(node,childbox%mGlobal,level)
        ip=ChildInfo%mBounds

        IF (level > -1) THEN
           ALLOCATE(ChildInfo%qParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nRestrict))
           CALL UnPackData(message, ChildInfo%qParent)

           IF (MaintainAuxArrays) THEN
              ip(1:nDim,2)=ip(1:nDim,2)+1
              ALLOCATE(ChildInfo%auxParent(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),nAux))
              CALL UnPackData(message, ChildInfo%auxParent)
           END IF
        END IF

        CALL ApplyInitialChildData(node%info, ChildInfo, level)
        IF (level > -1) THEN
           DEALLOCATE(ChildInfo%qParent)
           IF (MaintainAuxArrays) THEN
              DEALLOCATE(ChildInfo%auxParent)
           END IF
        END IF
        DEALLOCATE(ChildInfo)
        NULLIFY(ChildInfo)

     END DO

   END SUBROUTINE RecvInitialChildrenData

   !>@}

   !> Precalculates the amount of buffer space needed to send a node's flux data to its neighbor.
   !! @param level The level of the two nodes exchanging fluxes.
   !! @param node The local node.
   !! @param neighbor The node's neighbor.
   INTEGER FUNCTION SendFluxes_Precalculate(level, node, neighbor)

       INTEGER :: level
       TYPE(NodeDef), POINTER :: node
       TYPE(NodeDef), POINTER :: neighbor
       INTEGER :: dir,nOverlaps,i,edge
       INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
       INTEGER, DIMENSION(3,2) :: mT, mS,ip
       INTEGER, DIMENSION(:), POINTER :: edges
       INTEGER, DIMENSION(:,:), POINTER :: Offsets 
       INTEGER :: accumulator
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER, DIMENSION(3) :: ms_mx


       ! node%box%mGlobal size.
       accumulator = 6 * PACK_INTEGER_SIZE

       ! neighbor%box%mGlobal size.
       accumulator = accumulator + 6 * PACK_INTEGER_SIZE

       DO dir=1,nDim

           ! Calculates the flux overlaps.
           CALL CalcFluxOverlaps(neighbor%box%mGlobal, node%box%mGlobal,mTs,mSs,edges,nOverlaps,level,dir,lHydroPeriodic)

           ! Holds the number of overlaps.
           accumulator = accumulator + PACK_INTEGER_SIZE

           IF (nOverlaps > 0) THEN

               !Add the size of the overlap and edge arrays.
               accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs) + SIZE(edges)) * PACK_INTEGER_SIZE)

               DO i=1,nOverlaps
                   mT=mTs(i,:,:)
                   mS=mSs(i,:,:)
                   edge=edges(i)
                   ip=mS
                   ip(dir,:)=3-edge

                   ip_mx(:) = ip(:,2) - ip(:,1) + 1
 
                   ! Add space for the contents of a single fixup flux side (node%info%fixupflux%side(dir)%data).
                   accumulator = accumulator + (PRODUCT(ip_mx) * SIZE(node%info%fixupflux%side(dir)%data, 4) * PACK_DOUBLE_SIZE)
 
               END DO
    
              DEALLOCATE(mTs,mSs,edges)
              NULLIFY(mTs,mSs,edges)
           END IF
       END DO

       IF (MaintainAuxArrays) THEN
           DO dir=1,nEMF
               CALL CalcEmfOverlaps(neighbor%box%mGlobal,node%box%mGlobal,mTs,mSs,nOverlaps,offsets, level,EmfDir(dir),lHydroPeriodic)

               ! Another nOverlaps value.
               accumulator = accumulator + PACK_INTEGER_SIZE

               IF (nOverlaps > 0) THEN

                   !Add the size of the overlap and offset arrays.
                   accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs) + SIZE(offsets)) * PACK_INTEGER_SIZE)

                   DO i=1,nOverlaps
                       mT=mTs(i,:,:)
                       mS=mSs(i,:,:)

                       ! node%Info%emf
                       ms_mx(:) = mS(:,2) - mS(:,1) + 1
                       accumulator = accumulator + PRODUCT(ms_mx) * PACK_DOUBLE_SIZE
                   END DO

                   DEALLOCATE(mTs,mSs,offsets)
                   NULLIFY(mTs,mSs,offsets)
               END IF
           END DO
       END IF

       SendFluxes_Precalculate = accumulator
        
   END FUNCTION SendFluxes_Precalculate

   !> @name SyncFluxesParsing Routines for parsing data for SyncFluxes
   !! @{

   !> @brief Packs data for neighbors necessary for SyncFluxes
   !! @param message The message to pack to
   !! @param node The local node
   !! @param neighbor The external neighbor node
   SUBROUTINE SendFluxes(message, node, neighbor)
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: dir,nOverlaps,i,edge
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
      INTEGER, DIMENSION(3,2) :: mT, mS,ip
      INTEGER, DIMENSION(:), POINTER :: edges
      INTEGER, DIMENSION(:,:), POINTER :: Offsets 


      CALL PackData(message, neighbor%box%mGlobal)
      CALL PackData(message, node%box%mGlobal)
      DO dir=1,nDim
         CALL CalcFluxOverlaps(neighbor%box%mGlobal, node%box%mGlobal,mTs,mSs,edges,nOverlaps,message%level,dir,lHydroPeriodic)
         CALL PackData(message, nOverlaps)
         IF (nOverlaps > 0) THEN
            CALL PackData(message, mTs)
            CALL PackData(message, mSs)
            CALL PackData(message, edges)
            DO i=1,nOverlaps
               mT=mTs(i,:,:)
               mS=mSs(i,:,:)
               edge=edges(i)
               ip=mS
               ip(dir,:)=3-edge

               CALL PackData(message, node%info%fixupflux%side(dir)%data(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),:))
            END DO
            DEALLOCATE(mTs,mSs,edges)
            NULLIFY(mTs,mSs,edges)
         END IF
      END DO
      IF (MaintainAuxArrays) THEN
         DO dir=1,nEMF
            CALL CalcEmfOverlaps(neighbor%box%mGlobal,node%box%mGlobal,mTs,mSs,nOverlaps,offsets,message%level,EmfDir(dir), lHydroPeriodic)
            CALL PackData(message, nOverlaps)
            IF (nOverlaps > 0) THEN
               CALL PackData(message, mTs)
               CALL PackData(message, mSs)
               CALL PackData(message, offsets)               
               DO i=1,nOverlaps
                     mT=mTs(i,:,:)
                     mS=mSs(i,:,:)

                  CALL PackData(message,node%info%emf(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),EmfLoc(emfdir(dir)):EmfLoc(emfdir(dir))))
               END DO
               DEALLOCATE(mTs,mSs,offsets)
               NULLIFY(mTs,mSs,offsets)
            END IF
         END DO
      END IF
   END SUBROUTINE SendFluxes


   !> Precalculates the amount of buffer space needed to receive a neighbor's flux data.
   !! @param level The level of the two nodes exchanging fluxes.
   !! @param node The local node.
   !! @param neighbor The node's neighbor.
   INTEGER FUNCTION RecvFluxes_Precalculate(level, node, neighbor)

       INTEGER :: level
       TYPE(NodeDef), POINTER :: node
       TYPE(NodeDef), POINTER :: neighbor

       INTEGER :: dir,nOverlaps,i,edge,j
       INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
       INTEGER, DIMENSION(3,2) :: mT, mS,ip
       INTEGER, DIMENSION(:), POINTER :: edges
       INTEGER, DIMENSION(:,:), POINTER :: Offsets    
       INTEGER, DIMENSION(3) :: offset

       INTEGER :: accumulator
       INTEGER, DIMENSION(3) :: ip_mx
       INTEGER, DIMENSION(3) :: ms_mx


       ! neighbor mGlobal
       accumulator = 6 * PACK_INTEGER_SIZE

       ! node mGlobal
       accumulator = accumulator + 6 * PACK_INTEGER_SIZE

       DO dir=1,nDim

           ! Unpack nOverlaps
           accumulator = accumulator + PACK_INTEGER_SIZE

           ! Calculates the flux overlaps (reversed from SendFluxes).
           CALL CalcFluxOverlaps(node%box%mGlobal, neighbor%box%mGlobal,mTs,mSs,edges,nOverlaps,level,dir,lHydroPeriodic)

           IF (nOverlaps > 0) THEN

               !Add the size of the overlap and edge arrays.
               accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs) + SIZE(edges)) * PACK_INTEGER_SIZE)

               DO i=1,nOverlaps
                   mS=mSs(i,:,:)
                   edge=edges(i)
                   ip=mS
                   ip(dir,:)=3-edge
 
                   ip_mx(:) = ip(:,2) - ip(:,1) + 1
 
                   ! neighborinfo%fixupflux%side%data
                   accumulator = accumulator + (PRODUCT(ip_mx) * nFlux * PACK_DOUBLE_SIZE)
 
                END DO
 
                DEALLOCATE(mTs,mSs,edges)
                NULLIFY(mTs,mSs,edges)
            END IF
        END DO
 
        IF (MaintainAuxArrays) THEN

            DO dir=1,nEMF

               ! add nOverlaps space.
               accumulator = accumulator + PACK_INTEGER_SIZE

               ! Calculate overlaps (the inverse of the SendFluxes_Precalculate operation).
               CALL CalcEmfOverlaps(node%box%mGlobal,neighbor%box%mGlobal,mTs,mSs,nOverlaps,offsets,level,EmfDir(dir), lHydroPeriodic)

               IF (nOverlaps > 0) THEN

                   !Add the size of the overlap and offsets arrays.
                   accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs) + SIZE(offsets)) * PACK_INTEGER_SIZE)

                   DO i=1,nOverlaps
                       mS=mSs(i,:,:)
                       offset=offsets(i,:)                     

                       ms_mx(:) = mS(:,2) - mS(:,1) + 1

                       ! Add space for neighborinfo%emf(emfloc(dir)).
                       accumulator = accumulator + PRODUCT(ms_mx) * PACK_DOUBLE_SIZE

                   END DO

                   DEALLOCATE(mTs,mSs,offsets)
                   NULLIFY(mTs,mSs,offsets)

               END IF
           END DO
       END IF

       RecvFluxes_Precalculate = accumulator

   END FUNCTION RecvFluxes_Precalculate

   !> @brief UnPacks data from neighbors necessary for SyncFluxes
   !! @param message The message to unpack from
   SUBROUTINE RecvFluxes(message)
      TYPE(NodeDef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: dir,nOverlaps,i,level,edge,j
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
      INTEGER, DIMENSION(3,2) :: mT, mS,ip
      INTEGER, DIMENSION(:), POINTER :: edges
      INTEGER, DIMENSION(:,:), POINTER :: Offsets    
      INTEGER, DIMENSION(3) :: offset
      TYPE(NodeBox) :: localnodebox
      TYPE(InfoDef), POINTER :: neighborinfo


      localnodebox%MPI_ID=MPI_ID
      level=message%level

      NULLIFY(NeighborInfo)
      ALLOCATE(NeighborInfo)
      CALL NullifyInfoFields(NeighborInfo)

      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvFluxes"))

         CALL StrictFindNode(level,localnodebox,node, "RecvFluxes")
         CALL UnPackData(message,NeighborInfo%mGlobal)
         ALLOCATE(neighborinfo%fixupflux)
         DO dir=1,nDim

            CALL UnPackData(message, nOverlaps)


            IF (nOverlaps > 0) THEN

               ALLOCATE(neighborinfo%fixupflux%side(dir:dir))
               ALLOCATE(mTs(nOverlaps,3,2))
               ALLOCATE(mSs(nOverlaps,3,2))
               ALLOCATE(edges(nOverlaps))

               CALL UnPackData(message, mTs)
               CALL UnPackData(message, mSs)
               CALL UnPackData(message, edges)

               DO i=1,nOverlaps
                  mT=mTs(i,:,:)
                  mS=mSs(i,:,:)
                  edge=edges(i)
                  ip=mS
                  ip(dir,:)=3-edge
                  ALLOCATE(neighborinfo%fixupflux%side(dir)%data(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2), nFlux))

                  CALL UnPackData(message, neighborinfo%fixupflux%side(dir)%data)

                  CALL SyncSingleFlux(node%info,neighborinfo,mT,mS,dir,edge)
                  DEALLOCATE(neighborinfo%fixupflux%side(dir)%data)
                  NULLIFY(neighborinfo%fixupflux%side(dir)%data)
               END DO

               DEALLOCATE(mTs,mSs,edges)
               NULLIFY(mTs,mSs,edges)

               DEALLOCATE(neighborinfo%fixupflux%side)
               NULLIFY(neighborinfo%fixupflux%side)
            END IF
         END DO

         DEALLOCATE(neighborinfo%fixupflux)
         NULLIFY(neighborinfo%fixupflux)

         IF (MaintainAuxArrays) THEN
            DO dir=1,nEMF
               CALL UnPackData(message, nOverlaps)

               IF (nOverlaps > 0) THEN

                  ALLOCATE(mTs(nOverlaps,3,2), mSs(nOverlaps,3,2), offsets(nOverlaps,3))

                  CALL UnPackData(message, mTs)
                  CALL UnPackData(message, mSs)
                  CALL UnPackData(message, offsets)               

                  DO i=1,nOverlaps
                     mT=mTs(i,:,:)
                     mS=mSs(i,:,:)
                     offset=offsets(i,:)                     
                     ALLOCATE(neighborinfo%emf(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),EmfLoc(emfdir(dir)):EmfLoc(emfdir(dir))))
                     CALL UnPackData(message,neighborinfo%emf(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),EmfLoc(emfdir(dir))))

                     CALL SyncSingleEMF(node%info, neighborinfo, mT, mS, offset, EmfDir(dir))
                     DEALLOCATE(neighborinfo%emf)
                     NULLIFY(neighborinfo%emf)
                  END DO

                  DEALLOCATE(mTs,mSs,offsets)
                  NULLIFY(mTs,mSs,offsets)

               END IF
            END DO
         END IF
      END DO

      DEALLOCATE(NeighborInfo)
      NULLIFY(NeighborInfo)

   END SUBROUTINE RecvFluxes
   
   !> @}

   !> @name ApplyGenericOverlapsParsing Routines for parsing data for ApplyGenericOverlap
   !! @{

   !> Calculates the buffer space required to transfer generic data from a node to a specific neighbor.
   !! @param level The level of the two nodes swapping data.
   !! @param node The node whose data will be sent.
   !! @param neighbor The neighbor who will receive node's data.
   !! @param fields An array listing the indices of the fields to be transferred.
   !! @param nghost The number of ghost cells to be considered for overlap.
   INTEGER FUNCTION SendGenericData_Precalculate(level, node, neighbor, fields, nghost, lPeriodic)

      INTEGER :: level
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: neighbor
      INTEGER, DIMENSION(:) :: fields
      INTEGER ::  nghost
      LOGICAL, DIMENSION(3) :: lPeriodic
      INTEGER :: accumulator
      INTEGER, DIMENSION(3,2) :: mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,n
      INTEGER, DIMENSION(3) :: ms_mx


      ! Pack neighbor%box%mGlobal
      accumulator = 6 * PACK_INTEGER_SIZE      

      ! nOverlaps
      accumulator = accumulator + PACK_INTEGER_SIZE

      ! Obtain the overlap arrays that will be used by the actual SendGenericData() subroutine.
      CALL CalcOverlaps(neighbor%box%mGlobal, node%box%mGlobal,mTs,mSs,nOverlaps,level,lPeriodic, nghost)

      IF (nOverlaps > 0) THEN

          !Add the size of the overlap and offsets arrays.
          accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

          DO i=1,nOverlaps
              mS=mSs(i,:,:)
              ms_mx(:) = mS(:,2) - mS(:,1) + 1

              ! Add the cost of the overlaps for the generic variables.
              accumulator = accumulator + (PRODUCT(ms_mx) * SIZE(fields) * PACK_DOUBLE_SIZE)
          END DO

          DEALLOCATE(mTs,mSs)
          NULLIFY(mTs,mSs)            

      END IF

      SendGenericData_Precalculate = accumulator

   END FUNCTION SendGenericData_Precalculate

   !> @brief Packs data for overlaps necessary for GenericOverlap
   !! (CostPerCell, CostMap, [level > -1: q,[MaintainAuxArrays: aux]])
   !! @param message The message to pack to
   !! @param node The local node
   !! @param neighbor The external neighbor node
   !! @param fields The set of fields in q to transfer
   !! @param nghost The number of ghost cells to transfer
   SUBROUTINE SendGenericData(message, node, neighbor, fields, nghost, lPeriodic)
      TYPE(NodeDef), POINTER :: node, neighbor
      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,n,dir,j
      INTEGER, DIMENSION(:) :: fields
      INTEGER ::  nghost
      LOGICAL, DIMENSION(3) :: lPeriodic
      CALL CalcOverlaps(neighbor%box%mGlobal, node%box%mGlobal,mTs,mSs,nOverlaps,message%level,lPeriodic, nghost)
      CALL PackData(message, neighbor%box%mGlobal)
      CALL PackData(message, nOverlaps)
      IF (nOverlaps > 0) THEN
         CALL PackData(message, mTs)
         CALL PackData(message, mSs)         
         DO i=1,nOverlaps
            mT=mTs(i,:,:)
            mS=mSs(i,:,:)
            DO j=1,size(fields)
               CALL PackData(message, node%info%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),fields(j)))
            END DO
         END DO
         DEALLOCATE(mTs,mSs)
         NULLIFY(mTs,mSs)            
      END IF
   END SUBROUTINE SendGenericData


   !> Calculates the buffer space required for a node to receive elliptic data from a specific neighbor.
   !! @param level The level of the two nodes swapping data.
   !! @param node The node receiving data.
   !! @param neighbor The neighbor who will sending data to node.
   !! @param fields An array listing the indices of the fields to be transferred.
   !! @param nghost The number of ghost cells to be considered for overlap.
   INTEGER FUNCTION RecvGenericData_Precalculate(level, node, neighbor, fields, nghost, lPeriodic)

      INTEGER :: level
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDef), POINTER :: neighbor
      INTEGER, DIMENSION(:) :: fields
      INTEGER ::  nghost

      INTEGER :: accumulator
      INTEGER, DIMENSION(3,2) :: mS
      INTEGER, POINTER, DIMENSION(:,:,:) :: mTs, mSs
      INTEGER :: nOverlaps,i,n
      INTEGER, DIMENSION(3) :: ms_mx
      LOGICAL, DIMENSION(3) :: lPeriodic

      ! Pack neighbor%box%mGlobal
      accumulator = 6 * PACK_INTEGER_SIZE      

      ! Obtain the overlap arrays that will be used by the actual RecvGenericData() subroutine.  This is
      ! basically the inverse of the CalcOverlaps() call in SendGenericData--note the node and neighbor
      ! positions in the subroutine call.
      CALL CalcOverlaps(node%box%mGlobal, neighbor%box%mGlobal,mTs,mSs,nOverlaps,level,lPeriodic, nghost)

      ! nOverlaps
      accumulator = accumulator + PACK_INTEGER_SIZE

      IF (nOverlaps > 0) THEN

          !Add the size of the overlap and offsets arrays.
          accumulator = accumulator + ((SIZE(mTs) + SIZE(mSs)) * PACK_INTEGER_SIZE)

          DO i=1,nOverlaps
              mS=mSs(i,:,:)
              ms_mx(:) = mS(:,2) - mS(:,1) + 1

              ! Add the cost of the overlaps for the generic variables.
              accumulator = accumulator + (PRODUCT(ms_mx) * SIZE(fields) * PACK_DOUBLE_SIZE)
          END DO

          DEALLOCATE(mTs,mSs)
          NULLIFY(mTs,mSs)            

      END IF

      RecvGenericData_Precalculate = accumulator

   END FUNCTION RecvGenericData_Precalculate

   !> @brief UnPacks data from neighbors necessary for GenericOverlaps
   !! @param message The message to unpack from
   !! @param fields The fields in q to unpack
   SUBROUTINE RecvGenericData(message,fields)
      TYPE(NodeDef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message
      TYPE(InfoDef) :: neighborInfo
      TYPE(NodeBox) :: localnodebox
      INTEGER :: level, nOverlaps,i,dir,j
      INTEGER, DIMENSION(:,:,:), POINTER :: mTs, mSs
      INTEGER, DIMENSION(3,2) :: mT, mS
      INTEGER, DIMENSION(:) :: fields
      localnodebox%MPI_ID=MPI_ID
      level=message%level
      DO WHILE(StrictGetNextBox(message, localnodebox%mGlobal, "RecvGenericData"))
         CALL StrictFindNode(level,localnodebox,node, "RecvGenericData(node)")
         CALL UnPackData(message, nOverlaps)
         
         IF (nOverlaps > 0) THEN
            
            ALLOCATE(mTs(nOverlaps,3,2), mSs(nOverlaps,3,2))
               
            CALL UnPackData(message, mTs)
            CALL UnPackData(message, mSs)
            
            DO i=1,nOverlaps
               mT=mTs(i,:,:)
               mS=mSs(i,:,:)                 
               DO j=1,size(fields)
                  ALLOCATE(neighborinfo%q(mS(1,1):mS(1,2), mS(2,1):mS(2,2), mS(3,1):mS(3,2),fields(j):fields(j)))
                  CALL UnPackData(message, neighborinfo%q)
                  CALL ApplySingleOverlap(node%info,neighborinfo,mS,mT,fields(j:j))
                  DEALLOCATE(neighborinfo%q)
               END DO
            END DO
            DEALLOCATE(mTs,mSs)
            NULLIFY(mTs,mSs)
         END IF
      END DO
   END SUBROUTINE RecvGenericData
   !> @}
END MODULE DataParsing
