!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    slopelim.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file slopelim.f90
!! @brief Main file for module SlopeLim

!> @defgroup SlopeLim Slope Limiters
!! @brief Module for calculating slope limiters
!! @ingroup DataDeclarations

!> Module for calculating slope limiters
!! @ingroup SlopeLim

MODULE SlopeLim


  USE GlobalDeclarations
  IMPLICIT NONE

  PUBLIC
  INTEGER, PARAMETER :: CONSTANT_INTERP=0
  INTEGER, PARAMETER :: MINMOD_INTERP=1
  INTEGER, PARAMETER :: SUPERBEE_INTERP=2
  INTEGER, PARAMETER :: VANLEER_INTERP=3
  INTEGER, PARAMETER :: MC_INTERP=4
  INTEGER, PARAMETER :: PARABOLIC_INTERP=5
  INTEGER, PARAMETER :: LINEAR_INTERP=6
  INTEGER, PARAMETER :: SPECTRAL_PROLONGATION=7

CONTAINS

  ! implements various limiters.
  PURE ELEMENTAL FUNCTION limiter(a,b,mlim)
    REAL(KIND=qprec),INTENT(IN) :: a,b
    INTEGER,INTENT(IN) :: mlim
    REAL(KIND=qprec) :: limiter, r
    !
    SELECT CASE(mlim)
    CASE(LINEAR_INTERP) ! no limiter
       limiter = 0.5d0*(a+b)
       RETURN
    END SELECT
    !
    limiter=0.d0
    IF(a*b<=0.d0) RETURN
    !
    ! flatten to 1st order if noise apears if limiter is indicated
    ! by two digits (last digit specifies limiter)
    IF(mlim>10 .AND. ABS(a)<=1.d-12 .AND. ABS(b)<=1.d-12) RETURN
    !
    SELECT CASE(mlim)
    CASE(CONSTANT_INTERP)  ! constant
       limiter = 0.d0
    CASE(MINMOD_INTERP) ! minmod
       limiter = MIN(ABS(b),ABS(a))
    CASE(SUPERBEE_INTERP) ! superbee
       limiter = MAX(MIN(ABS(b),2.d0*ABS(a)),MIN(2.d0*ABS(b),ABS(a)))
    CASE(VanLeer_INTERP) ! vanLeer
       limiter = 2.d0*ABS(a*b)/(ABS(b)+ABS(a))
    CASE(MC_INTERP) ! MC
       limiter = MIN(2.d0*ABS(b),0.5d0*(ABS(a)+ABS(b)),2.d0*ABS(a))
    END SELECT
    limiter=limiter*SIGN(1.d0,a)
  END FUNCTION limiter


  FUNCTION GetnGhost(method)
     INTEGER :: GetnGhost, method
     SELECT CASE(method)
     CASE(CONSTANT_INTERP)
        GetnGhost=0
     CASE(MINMOD_INTERP, SUPERBEE_INTERP, VANLEER_INTERP, MC_INTERP, LINEAR_INTERP, PARABOLIC_INTERP)
        GetnGhost=1
     END SELECT
  END FUNCTION GetnGhost

END MODULE SlopeLim
