!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    elliptic_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir elliptic
!! @brief Contains modules for solving elliptic equations

!> @defgroup Elliptic Elliptic Solvers
!! @brief Group of modules for solving elliptic/hyperbolic equations

!> @file elliptic_control.f90
!! @brief Main file for module EllipticControl

!> @defgroup EllipticControl Elliptic Control
!! @brief Module for managing elliptic solves
!! @ingroup Elliptic

!> Module for managing elliptic solves
!! @ingroup EllipticControl
MODULE EllipticControl
  USE Timing
  USE GlobalDeclarations
  USE CommunicationDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE Poisson
  USE PhysicsDeclarations
  USE EllipticComms
  IMPLICIT NONE
  PUBLIC EllipticInit, EllipticSetup, EllipticSolve, EllipticDestroy, ApplyEllipticBC
  INTEGER :: isolve=0
CONTAINS


  !> @name Initialization routines
  !! @{

  !> Initializes variables needed by elliptic solvers and by control module
  SUBROUTINE EllipticInit
     INTEGER :: i
     INTEGER :: iErr
     lEllipticPeriodic=.false.
     NrEllipticObjects=0
     ALLOCATE(elliptic_maxspeed(0:MaxLevel))     
     elliptic_maxspeed = 0.d0
     lPoissonSolve=(lSelfGravity)
     IF (lPoissonSolve) THEN
        NrEllipticObjects=NrEllipticObjects+1
        iPoissonSolve=NrEllipticObjects
     END IF     
     IF (NrEllipticObjects > 0) THEN
        ALLOCATE(EllipticObjects(NrEllipticObjects))
     END IF
     IF (iPoissonSolve .ne. 0) CALL Poisson_Init(EllipticObjects(iPoissonSolve))
     DO i=1,NrEllipticObjects
        ALLOCATE(EllipticObjects(i)%LevelObjects(0:MaxLevel))
     END DO
     lAnyPeriodic=lAnyPeriodic .OR. lEllipticPeriodic
  END SUBROUTINE EllipticInit
  !> @}

  !> @name Main control routines for a level  
  !! @{

  !> Solves elliptic equations at simulation start on level n 
  !! @param n level
   SUBROUTINE InitialElliptic(n)
      INTEGER :: n
      TYPE(NodeDef), POINTER :: node
      TYPE(NodeDefList), POINTER :: nodelist
      INTEGER :: matrixID, ierr
      REAL(KIND=qPREC) :: temp
      CALL StartTimer(iElliptic, n)
      temp=levels(n)%dt
      levels(n)%dt=0d0
      IF (lElliptic) THEN
         CALL EllipticSetup(n)
         CALL EllipticSolve(n)
         CALL EllipticDestroy(n)
         CALL PostElliptic(n)
!         IF (nEllipticTransferFields > 0) THEN
!            CALL EllipticTransfer(n, EllipticTransferFields, levels(n)%egmbc(1))
!         END IF
      END IF
      levels(n)%dt=temp
      CALL StopTimer(iElliptic, n)
   END SUBROUTINE InitialElliptic

  !> Solves elliptic equations on level n
  !! @param n level
   SUBROUTINE Elliptic(n)
     INTEGER :: n, iErr
      CALL StartTimer(iElliptic, n)
      IF (lElliptic) THEN
         CALL MPI_ALLREDUCE(lRequestRestart, RestartStep, 1, MPI_LOGICAL, MPI_LOR, MPI_COMM_WORLD, iErr)
         IF (RestartStep) THEN
!            IF (levels(n)%step > 1) THEN !We already setup the grid etc...
!               CALL EllipticDestroy(n)
!            END IF
!           IF (MPI_ID == 0) write(*,*) 'Skipping elliptic solve since restart was requested'
            RETURN
         END IF
!         IF (levels(n)%step == 1) THEN
            CALL EllipticSetup(n)
!         END IF
         CALL EllipticSolve(n)
!         IF (levels(n)%step == levels(n)%steps) THEN
            CALL EllipticDestroy(n)
!         END IF
         CALL PostElliptic(n)
      END IF
      CALL StopTimer(iElliptic, n)                  
   END SUBROUTINE Elliptic

  !> @}


  !> @name Level operations for all elliptic equations
  !! @{

  !> Creates hypre grid structure for level n
  !! @param n level
  SUBROUTINE EllipticSetup(n)
     INTEGER :: n
     TYPE(NodeDefList), POINTER :: nodelist
     INTEGER :: iErr
     INTEGER :: i, nnodes, totalnodes, mb(3,2), minmb(3), maxmb(3), minmbo(3), maxmbo(3), minsize, maxsize, minsizeo, maxsizeo
     REAL(8) :: datasize, totaldatasize
     CHARACTER(LEN=21) :: s_filename
     !First create the grid for this level to be used by all elliptic equations
     CALL C_StructGridCreate(levels(n)%MPI_COMM, nDim, levels(n)%Grid, iErr); CALL CheckErr('C_StructGridCreate',iErr)
     nodelist=>nodes(n)%p
     datasize=0
     nnodes=0
     minmb=2**10
     maxmb=-2**10
     minsize=2**10
     maxsize=-2**10
     DO WHILE (associated(nodelist))
        CALL C_StructGridSetExtents(levels(n)%Grid, nodelist%self%box%mGlobal(1:nDim,1), nodelist%self%box%mGlobal(1:nDim,2), iErr)
        CALL CheckErr('C_StructGridSetExtents',iErr)
!        mB=nodelist%self%box%mGlobal(1:nDim,:)
!        datasize=datasize+product(mB(:,2)-mB(:,1) + 1)
!        minmb=min(minmb, mB(:,1))
!        maxmb=max(maxmb, mB(:,2))
!        minsize=min(minsize, product(mB(:,2)-mB(:,1)+1))
!        maxsize=max(maxsize, product(mB(:,2)-mB(:,1)+1))
!        nnodes=nnodes+1
        nodelist=>nodelist%next
     END DO

!!     CALL MPI_REDUCE(nnodes, totalnodes, 1, MPI_INTEGER, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
!     CALL MPI_REDUCE(datasize, totaldatasize, 1, MPI_DOUBLE_PRECISION, MPI_SUM, 0, MPI_COMM_WORLD, iErr)
!     CALL MPI_REDUCE(minmb, minmbo, 3, MPI_INTEGER, MPI_MIN, 0, MPI_COMM_WORLD, iErr)
!     CALL MPI_REDUCE(maxmb, maxmbo, 3, MPI_INTEGER, MPI_MAX, 0, MPI_COMM_WORLD, iErr)
!     CALL MPI_REDUCE(minsize, minsizeo, 1, MPI_INTEGER, MPI_MIN, 0, MPI_COMM_WORLD, iErr)
!     CALL MPI_REDUCE(maxsize, maxsizeo, 1, MPI_INTEGER, MPI_MAX, 0, MPI_COMM_WORLD, iErr)
     
!     IF (MPI_ID == 0) write(*,*) n, totalnodes, totaldatasize, minmbo, maxmbo, minsizeo, maxsizeo

!     CALL MPI_BARRIER(MPI_COMM_WORLD, iErr)
     IF (ANY(lEllipticPeriodic(1:nDim))) CALL C_StructGridSetPeriodic(levels(n)%Grid,levels(n)%mX(1:nDim), iErr) 
     !If Self Gravity and Thermal Diffusion are not both periodic this will break
     CALL CheckErr('StructGridSetPeriodic',iErr)
     CALL C_StructGridAssemble(levels(n)%Grid, iErr); CALL CheckErr('C_StructGridAssemble',iErr)
     !Next setup matrices for different elliptic objects
     DO i=1,NrEllipticObjects
        CALL C_StructMatrixCreate(levels(n)%MPI_COMM, levels(n)%Grid, EllipticObjects(i)%Stencil, EllipticObjects(i)%LevelObjects(n)%Matrix, iErr) ;CALL CheckErr('C_StructMatrixCreate',iErr)     
        CALL C_StructMatrixInitialize(EllipticObjects(i)%LevelObjects(n)%Matrix, iErr); CALL CheckErr('C_StructMatrixInitialize',iErr)
        IF (i == iPoissonSolve) THEN
           CALL Poisson_Setup(n,EllipticObjects(i)%LevelObjects(n))
        END IF
        CALL C_StructMatrixAssemble(EllipticObjects(i)%LevelObjects(n)%Matrix, iErr);CALL CheckErr('C_StructMatrixAssemble',iErr)
        IF (EllipticObjects(i)%hVerbosity > 0) THEN
           write(s_filename,'(A7,I2.2,A1,I2.2,A1,I4.4,A4)') "matrix_",n,"_",i,"_",isolve,".dat"
           CALL C_StructMatrixPrint(s_filename, EllipticObjects(i)%LevelObjects(n)%Matrix, 0, ierr)
           CALL CheckErr('C_SStructMatrixPrint', iErr)
        END IF   
     END DO
  END SUBROUTINE EllipticSetup

  !> Solves elliptic equations on level n
  !! @param n level
  SUBROUTINE EllipticSolve(n)
     INTEGER :: n
     INTEGER :: i
     INTEGER :: ierr, iters
     CHARACTER(LEN=25) :: s_filename
     LOGICAL :: done_solving
     isolve=isolve+1
     elliptic_maxspeed(n)=0
     DO i=1,NrEllipticObjects
        done_solving=.false.
        CALL PreSolveComm(i,n,done_solving)
        DO WHILE (.not. done_solving)
           CALL C_StructVectorCreate(levels(n)%MPI_COMM, levels(n)%Grid, EllipticObjects(i)%LevelObjects(n)%VariableVector, iErr) ;CALL CheckErr('C_StructVectorCreate',iErr)
           CALL C_StructVectorCreate(levels(n)%MPI_COMM, levels(n)%Grid, EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr) ;CALL CheckErr('C_StructVectorCreate',iErr)
           CALL C_StructVectorInitialize(EllipticObjects(i)%LevelObjects(n)%VariableVector, iErr); CALL CheckErr('C_StructVectorInitialize',iErr)
           CALL C_StructVectorInitialize(EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr); CALL CheckErr('C_StructVectorInitialize', iErr)
           IF (i == iPoissonSolve) THEN
              CALL Poisson_LoadVectors(n,EllipticObjects(iPoissonSolve)%LevelObjects(n))     
           END IF
           CALL C_StructVectorAssemble(EllipticObjects(i)%LevelObjects(n)%VariableVector, iErr) ;CALL CheckErr('C_StructVectorAssemble',iErr)
           CALL C_StructVectorAssemble(EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr) ;CALL CheckErr('C_StructVectorAssemble',iErr)
           IF (EllipticObjects(i)%hVerbosity > 0) THEN
              write(s_filename,'(A11,I2.2,A1,I2.2,A1,I4.4,A4)') "rhs_vector_",n,"_",i,"_",isolve,".dat"
              CALL C_StructVectorPrint(s_filename, EllipticObjects(i)%LevelObjects(n)%VariableVector, 0, ierr)
              CALL CheckErr('C_StructMatrixPrint', iErr)
              write(s_filename,'(A11,I2.2,A1,I2.2,A1,I4.4,A4)') "lhs_vector_",n,"_",i,"_",isolve,".dat"
              CALL C_StructVectorPrint(s_filename, EllipticObjects(i)%LevelObjects(n)%SolutionVector, 0, ierr)
              CALL CheckErr('C_StructVectorPrint', iErr)
           END IF

           SELECT CASE(EllipticObjects(i)%solver) !Choose correct solver
           CASE(StructPCG)
              CALL C_StructPCGCreate(levels(n)%MPI_COMM, EllipticObjects(i)%LevelObjects(n)%Solver, iErr); CALL CheckErr('C_SStructPCGCreate',iErr)
              
              ! set solver parameters
              CALL C_StructPCGSetTol(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%tolerance, iErr)
              CALL CheckErr('C_StructPCGSetTol',iErr)

              ! set solver parameters
              CALL C_StructPCGSetMaxIter(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%MaxIters, iErr)
              CALL CheckErr('C_StructPCGSetMaxIter',iErr)
              
              CALL C_StructPCGSetPrintLevel(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%printlevel, iErr)
              CALL CheckErr('C_StructPCGSetPrintLevel',iErr)
              
              ! setup and solve
              CALL C_StructPCGSetup(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%LevelObjects(n)%Matrix, EllipticObjects(i)%LevelObjects(n)%VariableVector, EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr)
              CALL CheckErr('C_StructPCGSetup',iErr)
              CALL C_StructPCGSolve(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%LevelObjects(n)%Matrix, EllipticObjects(i)%LevelObjects(n)%VariableVector, EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr)
              CALL CheckErr('C_StructPCGSolve',iErr)
              CALL C_StructPCGGetNumIterations(EllipticObjects(i)%LevelObjects(n)%Solver, Iters, iErr)
              CALL CheckErr('C_StructPCGSolve',iErr)
              CALL CheckIters(Iters, EllipticObjects(i)%MaxIters)
              CALL C_StructPCGDestroy(EllipticObjects(i)%LevelObjects(n)%Solver,iErr)
              CALL CheckErr('C_StructPCGDestroy',iErr)
           CASE(StructGMRES)
              
              CALL C_StructGMRESCreate(levels(n)%MPI_COMM, EllipticObjects(i)%LevelObjects(n)%Solver, iErr); CALL CheckErr('C_SStructGMRESCreate',iErr)       
              
              ! set solver parameters
              CALL C_StructGMRESSetTol(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%tolerance, iErr)
              CALL CheckErr('C_StructGMRESSetTol',iErr)

              CALL C_StructGMRESSetMaxIter(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%maxIters, iErr)
              CALL CheckErr('C_StructGMRESSetMaxIter',iErr)
              
              CALL C_StructGMRESSetPrintLevel(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%printlevel, iErr)
              CALL CheckErr('C_StructGMRESSetPrintLevel',iErr)
              
              ! setup and solve
              CALL C_StructGMRESSetup(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%LevelObjects(n)%Matrix, EllipticObjects(i)%LevelObjects(n)%VariableVector, EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr)
              CALL CheckErr('C_StructGMRESSetup',iErr)
              CALL C_StructGMRESSolve(EllipticObjects(i)%LevelObjects(n)%Solver, EllipticObjects(i)%LevelObjects(n)%Matrix, EllipticObjects(i)%LevelObjects(n)%VariableVector, EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr)
              CALL CheckErr('C_StructGMRESSolve',iErr)
              CALL C_StructGMRESGetNumIterations(EllipticObjects(i)%LevelObjects(n)%Solver, Iters, iErr)
              CALL CheckErr('C_StructPCGSolve',iErr)
              CALL CheckIters(Iters, EllipticObjects(i)%MaxIters)
              CALL C_StructGMRESDestroy(EllipticObjects(i)%LevelObjects(n)%Solver,iErr)
              CALL CheckErr('C_StructGMRESDesetroy',iErr)
              
           END SELECT
           
           !Choose correct routine to unload vector objects
           IF (i == iPoissonSolve) THEN
              CALL Poisson_UnLoadVectors(n,EllipticObjects(iPoissonSolve)%LevelObjects(n))     
           END IF
           CALL C_StructVectorDestroy(EllipticObjects(i)%LevelObjects(n)%VariableVector, iErr) ;CALL CheckErr('C_StructVectorDestroy',iErr)
           CALL C_StructVectorDestroy(EllipticObjects(i)%LevelObjects(n)%SolutionVector, iErr) ;CALL CheckErr('C_StructVectorDestroy',iErr)        
           CALL BetweenSolveComm(i,n,done_solving)
        END DO
        CALL PostSolveComm(i,n)
     END DO
  END SUBROUTINE EllipticSolve

  !> Destroys hypre grid object for level n
  !! @param n level
  SUBROUTINE EllipticDestroy(n)
     INTEGER :: n
     TYPE(NodeDefList), POINTER :: nodelist
     INTEGER :: ierr
     INTEGER :: i
     !First create the grid for this level to be used by all elliptic equations
     DO i=1,NrEllipticObjects
        CALL C_StructMatrixDestroy(EllipticObjects(i)%LevelObjects(n)%Matrix, iErr) ;CALL CheckErr('C_StructMatrixDestroy',iErr)     
     END DO
     CALL C_StructGridDestroy(levels(n)%Grid, iErr); CALL CheckErr('C_StructGridDesetroy',iErr)
  END SUBROUTINE EllipticDestroy

  !> Performs any necessary grid updates post elliptic solves on level n
  !! @param n level   
  SUBROUTINE PostElliptic(n)
     INTEGER :: i
     INTEGER :: n
     DO i=1,NrEllipticObjects
        IF (i == iPoissonSolve) THEN
           CALL PoissonPostElliptic(n)
        END IF
     END DO
  END SUBROUTINE PostElliptic
  !> @}  


  !> @name Level operations for individual elliptic equations
  !! @{

  !> Performs any presolve communication
  !! @param i Handle for elliptic equation
  !! @param n level
  !! @param done_solving Logical flag that determins whether a solve needs to be performed
  SUBROUTINE PreSolveComm(i,n,done_solving)
     INTEGER :: i
     INTEGER :: n
     LOGICAL :: done_solving
     IF (i == iPoissonSolve) THEN
        CALL PoissonPreSolveComm(n,done_solving)
     END IF
  END SUBROUTINE PreSolveComm

  !> Performs any between solve communication in the case of sub-cycling
  !! @param i Handle for elliptic equation
  !! @param n level
  !! @param done_solving Logical flag that determins whether a solve needs to be performed
  SUBROUTINE BetweenSolveComm(i,n,done_solving)
     INTEGER :: i
     INTEGER :: n
     LOGICAL :: done_solving
     IF (i == iPoissonSolve) THEN
        CALL PoissonBetweenSolveComm(n,done_solving)
     END IF
   END SUBROUTINE BetweenSolveComm

  !> Performs any post solve communication
  !! @param i Handle for elliptic equation
  !! @param n level
  SUBROUTINE PostSolveComm(i,n)
     INTEGER :: i
     INTEGER :: n
     LOGICAL :: done_solving
     IF (i == iPoissonSolve) THEN
        CALL PoissonPostSolveComm(n)
     END IF
  END SUBROUTINE PostSolveComm



  SUBROUTINE ApplyEllipticBC(n)
     INTEGER :: n
     CALL StartTimer(iApplyEllipticBCs, n)
     IF (lPoissonSolve) CALL PoissonSetBC(n)
     CALL StopTimer(iApplyEllipticBCs, n)
  END SUBROUTINE ApplyEllipticBC



  !> @}



  
END MODULE EllipticControl

