!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    elliptic_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file elliptic_declarations.f90
!! @brief Main file for module EllipticDeclarations

!> @defgroup EllipticDeclarations Elliptic Declarations
!! @brief Module containing variables used by elliptic solvers
!! @ingroup Elliptic

!> Module containing variables used by elliptic solvers
!! @ingroup EllipticDeclarations
MODULE EllipticDeclarations
   USE GlobalDeclarations
   USE PhysicsDeclarations
!   USE DataLevelComms
!   USE CommunicationControl
   IMPLICIT NONE
   PUBLIC
   SAVE

   !> Stores handles to Hypre entities associated with a given level
   TYPE EllipticLevelObjectDef
      SEQUENCE
      INTEGER(8) :: Matrix !This should be persistent over two time steps by level
      INTEGER(8) :: Solver  ! These are locally created and destroyed by level
      INTEGER(8) :: VariableVector !These are locally created and destroyed and don't need to be stored by level
      INTEGER(8) :: SolutionVector !These are locally created and destroyed and don't need to be stored by level
   END TYPE EllipticLevelObjectDef

   !> Stores handles to Hypre entities associated with a particular equation
   TYPE EllipticObjectDef
      SEQUENCE
      TYPE(EllipticLevelObjectDef), DIMENSION(:), ALLOCATABLE :: LevelObjects
      REAL(8)  :: tolerance=1e-6     ! convergence tolerance
      INTEGER(8) :: Stencil          ! This should be constant over all levels and does not need to be in 
      INTEGER :: Interface      ! type of HYPRE interface
      INTEGER :: solver=1         ! type of HYPRE solver
      INTEGER :: printLevel     ! verbosity
      INTEGER :: maxIters=1000       ! maximum number of iterations
      INTEGER :: hVerbosity=0     ! how verbose user-side HYPRE should be
   END TYPE EllipticObjectDef


   TYPE(EllipticObjectDef), DIMENSION(:), POINTER :: EllipticObjects

   INTEGER,PARAMETER,PUBLIC :: StructInterface=1,SStructInterface=2
   INTEGER,PARAMETER,PUBLIC :: StructPCG=1, StructGMRES=2

   INTEGER :: NrEllipticObjects, iPoissonSolve

   LOGICAL :: lPoissonSolve
   REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: elliptic_maxspeed

  INTEGER :: nEllipticLo, nEllipticHi  ! Elliptic indices
  INTEGER :: NrEllipticVars=0

  INTEGER, PARAMETER :: MAXELLIPTICLENGTH = 20
  INTEGER, PARAMETER :: MAXELLIPTICVARS=40

  CHARACTER(LEN = MAXELLIPTICLENGTH), DIMENSION(0:MAXELLIPTICVARS) :: EllipticNames


CONTAINS

  !> Adds an elliptic field and assigns it to slot i
  !! @param i integer to assign next slot
  !! @param str optional string to define variable 

  SUBROUTINE AddElliptic(i, str)
     INTEGER :: i
     CHARACTER(LEN=*), OPTIONAL :: str
     i=nEllipticLo+NrEllipticVars
     NrEllipticVars=NrEllipticVars+1     
     IF (NrEllipticVars <= MaxEllipticVars) THEN
        IF (Present(str)) THEN
           EllipticNames(NrEllipticVars)=str(1:min(len(str), len(EllipticNames(NrEllipticVars))))
        ELSE
           write(EllipticNames(NrEllipticvars), '(A10I3.3)') 'Elliptic', NrEllipticVars
        END IF
     ELSE
        IF (MPI_ID == 0) THEN
           PRINT*, 'Warning: Unable to store name of elliptic variable without increasing MaxElliptics in processing_declarations.f90'
        END IF
     END IF
  END SUBROUTINE AddElliptic

  !> Returns a string containing the name of the elliptic
  !! @param i index of elliptic
  FUNCTION EllipticName(i)
     INTEGER :: i
     CHARACTER(LEN=MAXELLIPTICLENGTH) :: EllipticName

     IF (i <= MaxEllipticVars) THEN
        EllipticName=EllipticNames(i)
     ELSE
        write(EllipticName, '(A10I3.3)') 'Elliptic', NrEllipticVars
     END IF
     
  END FUNCTION EllipticName

  !> Checks the error code returned by hypre calls
  !! @param errString string passed in by calling routine
  !! @param iErr error to check returned by hypre
  SUBROUTINE CheckErr(errString,iErr)
    CHARACTER(*) :: errString
    INTEGER :: iErr
    CHARACTER(LEN=1000) :: description
    IF(iErr/=0 .AND. .NOT. lRequestRestart) THEN
       PRINT'(A,I5,A,A)','*** HYPRE returned an error on ',MPI_id,' at ',errString
       PRINT'(A,I8)','    with error code ',iErr
!       CALL Hypre_DescribeError(ierr, description)
!       write(*,*) description
!       STOP
       IF (.NOT. lRequestRestart) THEN
          PRINT*, 'Processor', MPI_ID, 'requesting restart'
       END IF
       lRequestRestart=.true.
    END IF
  END SUBROUTINE CheckErr

  SUBROUTINE CheckIters(n,m)
    INTEGER :: n,m
    IF (n == m) THEN
       IF (MPI_ID == 0) write(*,*) 'maximum # of iterations reached by hyper solve'
       IF (lRestartOnHypreMaxIters) THEN
          IF (MPI_ID == 0) write(*,*) 'requesting restart'
          lRequestRestart=.true.
       END IF
!    ELSE
!       IF (MPI_ID == 0) write(*,*) 'took ', n, 'out of ', m, 'iterations'
    END IF
  END SUBROUTINE CheckIters
END MODULE EllipticDeclarations
