//#########################################################################
//		
//    Copyright (C) 2003-2012 Department of Physics and Astronomy,
//                            University of Rochester,
//                            Rochester, NY
//
//    extended_hypre_wrapper.c is part of AstroBEAR.
//
//    AstroBEAR is free software: you can redistribute it and/or modify	  
//    it under the terms of the GNU General Public License as published by 
//    the Free Software Foundation, either version 3 of the License, or    
//    (at your option) any later version.
//
//    AstroBEAR is distributed in the hope that it will be useful, 
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
//
//#########################################################################
/*! \file extended_hypre_wrapper.c
\brief Provide a Fortran-friendly interface for C-based Hypre functions.
*/

/*
    File Name:  hypre_wrapper.c

    Created:    20090603 by Brandon D. Shroyer

    Purpose:    Provide a Fortran-friendly interface for C-based Hypre functions.

    Description:The principle problem handled by this file is the fact that Fortran passes
                function arguments by reference, whereas C passes them by value.  To that
                end, this library is simply a series of functions that take reference
                inputs and then "clean them up" before using them to execute a HYPRE 
                subroutine.  This clean-up process consists of dereferencing simple types
                where necessary, and performing type casting on the derived types passed
                in by Fortran.

                Note the underscore after each wrapper function name.  Apparently this
                is required to make Linux Fortran compilers recognize these C constructs
                as functions.

                Also note that Fortran function names are case-insensitive, whereas C
                functions are not.  The Fortran compilers we use convert Fortran
                code to lower-case before passing it to other libraries, which is why
                all the wrapper function names are all lower-case.

    Notes:      [BDS][20090604]:  Due to type conversion problems, we're using MPI_COMM_WORLD
                                  to initialize the Hypre structures now.  We hope to figure
                                  out the problem someday, so we're leaving the mpicomm
                                  parameters in place, but they currently have no effect
                                  on the construction of the Hypre data structures.

*/
#include "extended_hypre_wrapper.h"
#include "HYPRE_struct_ls.h"
#include "HYPRE_sstruct_ls.h"
#include <stdio.h>
#include <assert.h>

/******************************  StructGrid functions ***********************************/

void c_structgridcreate_(long int * mpicomm, int * dim, long int * grid, int * ierr) {

    HYPRE_StructGrid cgrid;

    *ierr = HYPRE_StructGridCreate(MPI_COMM_WORLD, *dim, &cgrid);

    *grid = (long int)cgrid;
}

void c_structgridsetextents_(long int * grid, int * lower, int * upper, int * ierr) {

    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);

    *ierr = HYPRE_StructGridSetExtents(cgrid, lower, upper);
}

void c_structgridsetperiodic_(long int * grid, int * periodicoffset, int * ierr) {

    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);
    
    *ierr = HYPRE_StructGridSetPeriodic(cgrid, periodicoffset);
}


void c_structgridassemble_(long int * grid, int * ierr) {

    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);

    *ierr = HYPRE_StructGridAssemble(cgrid);
}

void c_structgriddestroy_(long int * grid, int * ierr) {

    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);

    *ierr = HYPRE_StructGridDestroy(cgrid);
}

/*****************************  End StructGrid functions ********************************/

/******************************  StructStencil functions ********************************/

void c_structstencilcreate_(int * dim, int * size, long int * stencil, int * ierr) {

    HYPRE_StructStencil cstencil;

    *ierr = HYPRE_StructStencilCreate(*dim, *size, &cstencil);

    *stencil = (long int)cstencil;
}

void c_structstencilsetelement_(long int * stencil, int *entry, int *offset, int * ierr) {

    HYPRE_StructStencil cstencil = (HYPRE_StructStencil)(*stencil);

    *ierr = HYPRE_StructStencilSetElement(cstencil, *entry, offset);
}

void c_structstencildestroy_(long int * stencil, int * ierr) {

    HYPRE_StructStencil cstencil = (HYPRE_StructStencil)(*stencil);

    *ierr = HYPRE_StructStencilDestroy(cstencil);
}

/****************************  End StructStencil functions ******************************/

/*******************************  StructMatrix functions ********************************/

void c_structmatrixcreate_(long int * mpicomm, long int * grid, long int * stencil, long int * matrix, int * ierr) {
 
    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);
    HYPRE_StructStencil cstencil = (HYPRE_StructStencil)(*stencil);
    HYPRE_StructMatrix cmatrix;

    *ierr = HYPRE_StructMatrixCreate(MPI_COMM_WORLD, cgrid, cstencil, &cmatrix);

    *matrix = (long int)cmatrix;
}

void c_structmatrixinitialize_(long int * matrix, int * ierr) {

    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);

    *ierr = HYPRE_StructMatrixInitialize(cmatrix);
}

void c_structmatrixsetboxvalues_(long int * matrix, int * lower, int * upper, int * nentries, int * entries, double * values, int * ierr) {

    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);

    /*    printf("matrix is %d\n",*matrix);
    printf("lower is %d %d %d\n",lower[0],lower[1],lower[2]);
    printf("upper is %d %d %d\n",upper[0],upper[1],upper[2]);
    printf("nentries is %d\n",*nentries);
    printf("entries is %d %d %d %d %d\n",entries[0],entries[1],entries[2],entries[3],entries[4]);

    printf("the first few entries in the matrix are %lf %lf %lf %lf %lf %lf %lf %lf %lf",values[0],values[1],values[2],values[3],values[4],values[5],values[6],values[7],values[8],values[9]); */
    *ierr = HYPRE_StructMatrixSetBoxValues (cmatrix, lower, upper, *nentries, entries, values);

}

void c_structmatrixassemble_(long int * matrix, int * ierr) {

    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);

    *ierr = HYPRE_StructMatrixAssemble(cmatrix);
}

void c_structmatrixdestroy_(long int * matrix, int * ierr) {

    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);

    *ierr = HYPRE_StructMatrixDestroy(cmatrix);
}

/*****************************  End StructMatrix functions ******************************/

/*******************************  StructVector functions ********************************/

void c_structvectorcreate_(long int * mpicomm, long int * grid, long int * vector, int * ierr) {

    HYPRE_StructGrid cgrid = (HYPRE_StructGrid)(*grid);
    HYPRE_StructVector cvector;
    
    *ierr = HYPRE_StructVectorCreate (MPI_COMM_WORLD, cgrid, &cvector);

    *vector = (long int)cvector;
}

void c_structvectorinitialize_(long int * vector, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorInitialize(cvector);

}

void c_structvectorsetboxvalues_(long int * vector, int * lower, int * upper, double * values, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorSetBoxValues (cvector, lower, upper, values);
}

void c_structvectorgetboxvalues_(long int * vector, int * lower, int * upper, double * values, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorGetBoxValues(cvector, lower, upper, values);
}


void c_structvectorassemble_(long int * vector, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorAssemble(cvector);

}

void c_structvectorprint_(char * filename, long int * vector, int * var3, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorPrint(filename, cvector, *var3);

}

void c_structmatrixprint_(char * filename, long int * matrix, int * all, int * ierr) {
    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);
    *ierr = HYPRE_StructMatrixPrint(filename, cmatrix, *all);
}


void c_structvectordestroy_(long int * vector, int * ierr) {

    HYPRE_StructVector cvector = (HYPRE_StructVector)(*vector);

    *ierr = HYPRE_StructVectorDestroy(cvector);
}

/*****************************  End StructVector functions ******************************/

/*******************************  PCG solver functions **********************************/

void c_structpcgcreate_(long int * mpicomm, long int * solver, int * ierr) {

    HYPRE_StructSolver csolver;

    *ierr = HYPRE_StructPCGCreate(MPI_COMM_WORLD, &csolver);

    *solver = (long int)csolver;
}

void c_structpcgsetup_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);
    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);
    HYPRE_StructVector c_bvector = (HYPRE_StructVector)(*bvector);
    HYPRE_StructVector c_xvector = (HYPRE_StructVector)(*xvector);

    *ierr = HYPRE_StructPCGSetup(csolver, cmatrix, c_bvector, c_xvector);
}

void c_structpcgsolve_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);
    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);
    HYPRE_StructVector c_bvector = (HYPRE_StructVector)(*bvector);
    HYPRE_StructVector c_xvector = (HYPRE_StructVector)(*xvector);

    *ierr = HYPRE_StructPCGSolve(csolver, cmatrix, c_bvector, c_xvector);
}

void c_structpcgsettol_(long int * solver, double * tol, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructPCGSetTol(csolver, *tol);
}

void c_structpcgsetmaxiter_(long int * solver, int * max_iter, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructPCGSetMaxIter(csolver, *max_iter);
}

void c_structpcggetnumiterations_(long int * solver, int * num_iter, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructPCGGetNumIterations(csolver, num_iter);
}


void c_structpcgsetprintlevel_(long int * solver, int * level, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructPCGSetPrintLevel(csolver, *level);

}

void c_structpcgdestroy_(long int * solver, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructPCGDestroy(csolver);
}

/*****************************  End PCG solver functions ********************************/


/*********************************  GMRES solver functions ************************************/

void c_structgmrescreate_(long int * comm, long int * solver, int * ierr) {

    MPI_Comm commval = (MPI_Comm)(*comm);
    HYPRE_StructSolver csolver;

    *ierr = HYPRE_StructGMRESCreate(MPI_COMM_WORLD, &csolver);

    *solver = (long int)csolver;

}

void c_structgmresdestroy_(long int * solver, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructGMRESDestroy(csolver);
}

void c_structgmressetup_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);
    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);
    HYPRE_StructVector c_bvector = (HYPRE_StructVector)(*bvector);
    HYPRE_StructVector c_xvector = (HYPRE_StructVector)(*xvector);

    *ierr = HYPRE_StructGMRESSetup(csolver, cmatrix, c_bvector, c_xvector);
}

void c_structgmressolve_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);
    HYPRE_StructMatrix cmatrix = (HYPRE_StructMatrix)(*matrix);
    HYPRE_StructVector c_bvector = (HYPRE_StructVector)(*bvector);
    HYPRE_StructVector c_xvector = (HYPRE_StructVector)(*xvector);

    *ierr = HYPRE_StructGMRESSolve(csolver, cmatrix, c_bvector, c_xvector);
}

void c_structgmressettol_(long int * solver, double * tol, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructGMRESSetTol(csolver, *tol);
}

void c_structgmressetmaxiter_(long int * solver, int * max_iter, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructGMRESSetMaxIter(csolver, *max_iter);
}

void c_structgmresgetnumiterations_(long int * solver, int * num_iter, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructGMRESGetNumIterations(csolver, num_iter);
}

void c_structgmressetprintlevel_(long int * solver, int * level, int * ierr) {

    HYPRE_StructSolver csolver = (HYPRE_StructSolver)(*solver);

    *ierr = HYPRE_StructGMRESSetPrintLevel(csolver, *level);

}


/********************************* End GMRES solver functions *********************************/


/**====================================================================================**/
/**============================= SEMI-STRUCTURED GRIDS ================================**/
/**====================================================================================**/


/*******************************  SStructGrid functions *********************************/

void c_sstructgridcreate_(long int * mpicomm, int * dim, int * nparts, long int * grid, int * ierr) {

    HYPRE_SStructGrid cgrid;

    *ierr = HYPRE_SStructGridCreate(MPI_COMM_WORLD, *dim, *nparts, &cgrid);

    *grid = (long int)cgrid;
}

void c_sstructgridsetextents_(long int * grid, int * part, int * lower, int * upper, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);

    *ierr = HYPRE_SStructGridSetExtents(cgrid, *part, lower, upper);
}

void c_sstructgridsetvariables_(long int * grid, int * part, int * nvars, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);
    HYPRE_SStructVariable vartypes[1] = {HYPRE_SSTRUCT_VARIABLE_CELL};


    *ierr = HYPRE_SStructGridSetVariables(cgrid, *part, *nvars, vartypes);
}

void c_sstructgridassemble_(long int * grid, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);

    *ierr = HYPRE_SStructGridAssemble(cgrid);
}

void c_sstructgridsetperiodic_(long int * grid, int * part, int * periodicoffset, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);

    *ierr = HYPRE_SStructGridSetPeriodic(cgrid, *part, periodicoffset);
}

void c_sstructgriddestroy_(long int * grid, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);

    *ierr = HYPRE_SStructGridDestroy(cgrid);
}

void c_sstructgridsetnumghost_(long int * grid, int * num_ghost, int * ierr) {
    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);

    *ierr = HYPRE_SStructGridSetNumGhost(cgrid, num_ghost);
}

/*****************************  End SStructGrid functions *******************************/

/******************************  SStructStencil functions *******************************/

void c_sstructstencilcreate_(int * dim, int * size, long int * stencil, int * ierr) {

    HYPRE_SStructStencil cstencil;

    *ierr = HYPRE_SStructStencilCreate(*dim, *size, &cstencil);

    *stencil = (long int)cstencil;
}

void c_sstructstencilsetentry_(long int * stencil, int *entry, int *offset, int * var, int * ierr) {

    HYPRE_SStructStencil cstencil = (HYPRE_SStructStencil)(*stencil);

    *ierr = HYPRE_SStructStencilSetEntry(cstencil, *entry, offset, *var);
}

void c_sstructstencildestroy_(long int * stencil, int * ierr) {

    HYPRE_SStructStencil cstencil = (HYPRE_SStructStencil)(*stencil);

    *ierr = HYPRE_SStructStencilDestroy(cstencil);
}

/****************************  End SStructStencil functions *****************************/

/*******************************  SStructGraph functions ********************************/

void c_sstructgraphcreate_(long int * mpicomm, long int * grid, long int * graph, int * ierr) { 

/*    MPI_Comm commval = (MPI_Comm)(*mpicomm);*/
    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);
    HYPRE_SStructGraph cgraph;

    *ierr = HYPRE_SStructGraphCreate(MPI_COMM_WORLD, cgrid, &cgraph);

    *graph = (long int)cgraph;
}

void  c_sstructgraphsetstencil_(long int * graph, int * part, int * var, long int * stencil, int * ierr) {

    HYPRE_SStructGraph cgraph = (HYPRE_SStructGraph)(*graph);
    HYPRE_SStructStencil cstencil = (HYPRE_SStructStencil)(*stencil);

    *ierr = HYPRE_SStructGraphSetStencil(cgraph, *part, *var, cstencil);
}

void c_sstructgraphaddentries_(long int * graph, int * part, int * index, int * var, int * to_part, int * to_index, int * to_var, int * ierr) {

    HYPRE_SStructGraph cgraph = (HYPRE_SStructGraph)(*graph);
    int cfrom_index[3], cto_index[3];

    cfrom_index[0] = index[0];
    cfrom_index[1] = index[1];
    cfrom_index[2] = index[2];

    cto_index[0] = to_index[0];
    cto_index[1] = to_index[1];
    cto_index[2] = to_index[2];


/*printf("SStructGraphAddEntries([%d, %d, %d] => [%d, %d, %d].\n", index[0], index[1], index[2], to_index[0], to_index[1], to_index[2]);*/

/*if (((cto_index[0] == 1) && (cto_index[1] == 64)) || ((cfrom_index[0] == 1) && (cfrom_index[1] == 64))) {*/
/*printf("SStructGraphAddEntries[graph=%d, part=%d, index=[%d, %d, %d], var=%d, to_part=%d, to_index=[%d, %d, %d], to_var=%d.\n", *graph, *part, cfrom_index[0], cfrom_index[1], cfrom_index[2], *var, *to_part, cto_index[0], cto_index[1], cto_index[2], *to_var);
}*/
    *ierr = HYPRE_SStructGraphAddEntries(cgraph, *part, cfrom_index, *var, *to_part, cto_index, *to_var);

}

void c_sstructgraphassemble_(long int * graph, int * ierr) {

    HYPRE_SStructGraph cgraph = (HYPRE_SStructGraph)(*graph);

    *ierr = HYPRE_SStructGraphAssemble(cgraph);
}

void c_sstructgraphdestroy_(long int * graph, int * ierr) {

    HYPRE_SStructGraph cgraph = (HYPRE_SStructGraph)(*graph);

    *ierr = HYPRE_SStructGraphAssemble(cgraph);
}

/*****************************  End SStructGraph functions ******************************/

/******************************  SStructMatrix functions ********************************/

void c_sstructmatrixcreate_(long int * mpicomm, long int * graph, long int * matrix, int * ierr) {

    HYPRE_SStructGraph cgraph = (HYPRE_SStructGraph)(*graph);
    HYPRE_SStructMatrix cmatrix;

    *ierr = HYPRE_SStructMatrixCreate(MPI_COMM_WORLD, cgraph, &cmatrix);

    *matrix = (long int)cmatrix;
}

void c_sstructmatrixsetobjecttype_(long int * matrix, int * obj_type, int * ierr) {

    int ctype;

    switch(*obj_type) {
        case F_HYPRE_STRUCT:
            ctype = HYPRE_STRUCT;
            break;
        case F_HYPRE_SSTRUCT:
            ctype = HYPRE_SSTRUCT;
            break;
        case F_HYPRE_PARCSR:
            ctype = HYPRE_PARCSR;
            break;
        default:
            printf("C_SStructMatrixSetObjectType error: invalid Fortran object code %d.\n", *obj_type);
    }
        
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);

    *ierr = HYPRE_SStructMatrixSetObjectType(cmatrix, ctype);
}

void c_sstructmatrixgetobject_(long int * matrix, long int * object, int * ierr) {

    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    HYPRE_ParCSRMatrix cobject;

    *ierr = HYPRE_SStructMatrixGetObject(cmatrix, (void **)&cobject);

    *object = (long int)(cobject);

}

void c_sstructmatrixinitialize_(long int * matrix, int * ierr) {

    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);

    *ierr = HYPRE_SStructMatrixInitialize(cmatrix);
}

void c_sstructmatrixsetboxvalues_(long int * matrix, int * part, int * lower, int * upper, int * var, int * nentries, int * entries, double * values, int * ierr) {

    int dummy[1];
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);

    *ierr = HYPRE_SStructMatrixSetBoxValues (cmatrix, *part, lower, upper, *var, *nentries, entries, values);
}

void c_sstructmatrixsetvalues_(long int * matrix, int * part, int * index, int * var, int * nentries, int * entries, double * values, int * ierr) {

    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    int array_length = *nentries;
    int centries[*nentries];
    double cvalues[*nentries];
    int dummy[1];
    int i;

    for(i = 0; i < array_length; i++) {
        centries[i] = entries[i];
        cvalues[i] = values[i];
    }

/*
printf("SStructMatrixSetValues(%ld, %d, [%d, %d, %d], %d, %d, [%d, %d], [%f, %f]).\n", *matrix, *part, index[0], index[1], index[2], *var, *nentries, centries[0], centries[1], cvalues[0], cvalues[1]);
*/

    *ierr = HYPRE_SStructMatrixSetValues(cmatrix, *part, index, *var, *nentries, centries, cvalues);

}

void c_sstructmatrixassemble_(long int * matrix, int * ierr) {
    
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);

    *ierr = HYPRE_SStructMatrixAssemble(cmatrix);
}

void c_sstructmatrixdestroy_(long int * matrix, int * ierr) {

    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);

    *ierr = HYPRE_SStructMatrixDestroy(cmatrix);
}

void c_sstructmatrixprint_(char * filename, long int * matrix, int * all, int * ierr) {
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    *ierr = HYPRE_SStructMatrixPrint(filename, cmatrix, *all);
}
/****************************  End SStructMatrix functions *****************************/

/******************************  SStructVector functions *******************************/

void c_sstructvectorcreate_(long int * mpicomm, long int * grid, long int * vector, int * ierr) {

    HYPRE_SStructGrid cgrid = (HYPRE_SStructGrid)(*grid);
    HYPRE_SStructVector cvector;
    
    *ierr = HYPRE_SStructVectorCreate (MPI_COMM_WORLD, cgrid, &cvector);

    *vector = (long int)cvector;
}

void c_sstructvectorinitialize_(long int * vector, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorInitialize(cvector);

}

void c_sstructvectorsetboxvalues_(long int * vector, int * part, int * lower, int * upper, int * var, double * values, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorSetBoxValues (cvector, *part, lower, upper, *var, values);
}

void c_sstructvectorgetboxvalues_(long int * vector, int * part, int * lower, int * upper, int * var, double * values, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorGetBoxValues(cvector, *part, lower, upper, *var, values);
}

void c_sstructvectorsetobjecttype_(long int * vector, int * obj_type, int * ierr) {

    int ctype;

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    switch(*obj_type) {
        case F_HYPRE_STRUCT:
            ctype = HYPRE_STRUCT;
            break;
        case F_HYPRE_SSTRUCT:
            ctype = HYPRE_SSTRUCT;
            break;
        case F_HYPRE_PARCSR:
            ctype = HYPRE_PARCSR;
            break;
        default:
            printf("C_SStructVectorSetObjectType error: invalid Fortran object code %d.\n", *obj_type);
    }


    *ierr = HYPRE_SStructVectorSetObjectType(cvector, ctype);

}

void c_sstructvectorgetobject_(long int * vector, long int * object, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);
    HYPRE_ParVector cobject;

    *ierr = HYPRE_SStructVectorGetObject(cvector, (void **)&cobject);

    *object = (long int)(*object);

}

void c_sstructvectorgather_(long int * vector, int * ierr) {
    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorGather(cvector);
}

void c_sstructvectorassemble_(long int * vector, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorAssemble(cvector);

}

void c_sstructvectorprint_(char * filename, long int * vector, int * var3, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorPrint(filename, cvector, *var3);

}

void c_sstructvectordestroy_(long int * vector, int * ierr) {

    HYPRE_SStructVector cvector = (HYPRE_SStructVector)(*vector);

    *ierr = HYPRE_SStructVectorDestroy(cvector);
}

/*****************************  End StructVector functions ******************************/

/*******************************  PCG solver functions **********************************/

void c_sstructpcgcreate_(long int * mpicomm, long int * solver, int * ierr) {

    HYPRE_SStructSolver csolver;

    *ierr = HYPRE_SStructPCGCreate(MPI_COMM_WORLD, &csolver);

    *solver = (long int)csolver;
}

void c_sstructpcgsetup_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    HYPRE_SStructVector c_bvector = (HYPRE_SStructVector)(*bvector);
    HYPRE_SStructVector c_xvector = (HYPRE_SStructVector)(*xvector);

    *ierr = HYPRE_SStructPCGSetup(csolver, cmatrix, c_bvector, c_xvector);
}

void c_sstructpcgsolve_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    HYPRE_SStructVector c_bvector = (HYPRE_SStructVector)(*bvector);
    HYPRE_SStructVector c_xvector = (HYPRE_SStructVector)(*xvector);

    *ierr = HYPRE_SStructPCGSolve(csolver, cmatrix, c_bvector, c_xvector);
}

void c_sstructpcgsettol_(long int * solver, double * tol, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructPCGSetTol(csolver, *tol);
}

void c_sstructpcgsetprintlevel_(long int * solver, int * level, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructPCGSetPrintLevel(csolver, *level);

}

void c_sstructpcgdestroy_(long int * solver, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructPCGDestroy(csolver);
}

/*****************************  End PCG solver functions ********************************/


/*******************************  ParCSRPCG solver functions **********************************/

void c_parcsrpcgcreate_(long int * comm, long int * solver, int * ierr) {

    MPI_Comm commval = (MPI_Comm)(*comm);
    HYPRE_Solver csolver;

    *ierr = HYPRE_ParCSRPCGCreate(MPI_COMM_WORLD, &csolver);

    *solver = (long int)csolver;

}

void c_parcsrpcgsettol_(long int * solver, double * tol, int * ierr) {
    HYPRE_Solver csolver = (HYPRE_Solver)(*solver);

    *ierr = HYPRE_ParCSRPCGSetTol(csolver, *tol);
}

void c_parcsrpcgsetup_(long int * solver, long int * matrix, long int * bvector, long int * xvector, int * ierr) {
    HYPRE_Solver csolver = (HYPRE_Solver)(*solver);
    HYPRE_ParCSRMatrix cmatrix = (HYPRE_ParCSRMatrix)(*matrix);
    HYPRE_ParVector cbvector = (HYPRE_ParVector)(*bvector);
    HYPRE_ParVector cxvector = (HYPRE_ParVector)(*xvector);

    *ierr = HYPRE_ParCSRPCGSetup(csolver, cmatrix, cbvector, cxvector);
}

void c_parcsrpcgsolve_(long int * solver, long int * matrix, long int * bvector, long int * xvector, int * ierr) {
    HYPRE_Solver csolver = (HYPRE_Solver)(*solver);
    HYPRE_ParCSRMatrix cmatrix = (HYPRE_ParCSRMatrix)(*matrix);
    HYPRE_ParVector cbvector = (HYPRE_ParVector)(*bvector);
    HYPRE_ParVector cxvector = (HYPRE_ParVector)(*xvector);

    *ierr = HYPRE_ParCSRPCGSolve(csolver, cmatrix, cbvector, cxvector);
}

void c_parcsrpcgsetprintlevel_(long int * solver, int * print_level, int * ierr) {
    HYPRE_Solver csolver = (HYPRE_Solver)(*solver);

    *ierr = HYPRE_ParCSRPCGSetPrintLevel(csolver, *print_level);
}

void c_parcsrpcgdestroy_(long int * solver, int * ierr) {
    HYPRE_Solver csolver = (HYPRE_Solver)(*solver);

    *ierr = HYPRE_ParCSRPCGDestroy(csolver);
}

void c_parcsrgotime_(long int * matrix, long int * varvector, long int * solvector, double * tol, int * printlevel, int * ierr) {
   HYPRE_Solver          csolver;
   HYPRE_ParCSRMatrix    A;
   HYPRE_ParVector       b;
   HYPRE_ParVector       x;
   int commrank;

   *ierr = MPI_Comm_rank(MPI_COMM_WORLD, &commrank);

   HYPRE_SStructMatrix Amatrix = (HYPRE_SStructMatrix)(*matrix);
   HYPRE_SStructVector bvector = (HYPRE_SStructVector)(*varvector);
   HYPRE_SStructVector xvector = (HYPRE_SStructVector)(*solvector);

   *ierr = HYPRE_SStructMatrixGetObject(Amatrix, (void **) &A);
   *ierr = HYPRE_SStructVectorGetObject(bvector, (void **) &b);
   *ierr = HYPRE_SStructVectorGetObject(xvector, (void **) &x);

   *ierr = HYPRE_ParCSRPCGCreate(MPI_COMM_WORLD, &csolver);
   assert(*ierr == 0);

   *ierr = HYPRE_ParCSRPCGSetTol(csolver, *tol);
   assert(*ierr == 0);

   *ierr = HYPRE_ParCSRPCGSetPrintLevel(csolver, *printlevel);
   assert(*ierr == 0);

   *ierr = HYPRE_ParCSRPCGSetup(csolver, A, b, x);
   assert(*ierr == 0);
   
   *ierr = HYPRE_ParCSRPCGSolve(csolver, A, b, x);
   assert(*ierr == 0);

   *ierr = HYPRE_ParCSRPCGDestroy(csolver);
   assert(*ierr == 0);

}

/*****************************  End ParCSRPCG solver functions ********************************/

/*********************************  GMRES solver functions ************************************/

void c_sstructgmrescreate_(long int * comm, long int * solver, int * ierr) {

    MPI_Comm commval = (MPI_Comm)(*comm);
    HYPRE_SStructSolver csolver;

    *ierr = HYPRE_SStructGMRESCreate(MPI_COMM_WORLD, &csolver);

    *solver = (long int)csolver;

}

void c_sstructgmresdestroy_(long int * solver, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructGMRESDestroy(csolver);
}

void c_sstructgmressetup_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    HYPRE_SStructVector c_bvector = (HYPRE_SStructVector)(*bvector);
    HYPRE_SStructVector c_xvector = (HYPRE_SStructVector)(*xvector);

    *ierr = HYPRE_SStructGMRESSetup(csolver, cmatrix, c_bvector, c_xvector);
}

void c_sstructgmressolve_(long int * solver, long int * matrix, long int * bvector, long int *xvector, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);
    HYPRE_SStructMatrix cmatrix = (HYPRE_SStructMatrix)(*matrix);
    HYPRE_SStructVector c_bvector = (HYPRE_SStructVector)(*bvector);
    HYPRE_SStructVector c_xvector = (HYPRE_SStructVector)(*xvector);

    *ierr = HYPRE_SStructGMRESSolve(csolver, cmatrix, c_bvector, c_xvector);
}

void c_sstructgmressettol_(long int * solver, double * tol, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructGMRESSetTol(csolver, *tol);
}

void c_sstructgmressetprintlevel_(long int * solver, int * level, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructGMRESSetPrintLevel(csolver, *level);

}

void c_sstructgmressetkdim_(long int * solver, int * k_dim, int * ierr) {

    HYPRE_SStructSolver csolver = (HYPRE_SStructSolver)(*solver);

    *ierr = HYPRE_SStructGMRESSetKDim(csolver, *k_dim);

}

/********************************* End GMRES solver functions *********************************/


/******************************** Start PFMG solver functions *********************************/

/********************************* End PFMG solver functions **********************************/
