!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    multipole.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file multipole.f90
!! @brief Main file for module multipole

!> @defgroup Multipole Multipole solver
!! @brief Module for approximating phi at boundaries using multipole method
!! @ingroup SelfGravity


!> Module for approximating phi at boundaries using multipole method
!! @ingroup MultiPole
module multipole
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE CommonFunctions
  USE Timing
  IMPLICIT NONE
  COMPLEX(8), DIMENSION(:), ALLOCATABLE :: moments
  INTEGER, DIMENSION(9,2) :: momentfactors
  REAL(KIND=qPREC) :: CM_00,CM_10,CM_11,CM_20,CM_21,CM_22
  LOGICAL :: lDipole, lQuadrupole,lReflect(3), lMultiPole(3)
  REAL(KIND=qPREC) :: massfact, costheta_fact, sintheta_fact, cosphi_fact, sinphi_fact, cos2phi_fact, sin2phi_fact, MultiPole_COM(3)
  REAL(KIND=qPREC) :: multipole_radius
  SAVE
contains

  !> Initializes MultiPole variables
   SUBROUTINE InitMultiPoleMoments
      REAL(KIND=qPREC) :: dV
      lDipole=.true.
      lQuadrupole=.true.
      !            lDipole=.false.
      !            lQuadrupole=.false.
      dV=levels(0)%dx**nDim
      IF (nDim == 3) THEN
         IF (lDipole) THEN
            IF (lQuadrupole) THEN
               ALLOCATE(moments(9))
            ELSE
               ALLOCATE(moments(4))
            END IF
         ELSE
            ALLOCATE(moments(1))
         END IF
      ELSE
         IF (lDipole) THEN
            write(*,*) 'WARNING*** Dipole expansion not supported in 2D'
         END IF
         lDipole=.false.
         lQuadrupole=.false.
         IF (lDipole) THEN
            IF (lQuadrupole) THEN
               ALLOCATE(moments(5))
            ELSE
               ALLOCATE(moments(3))
            END IF
         ELSE
            ALLOCATE(moments(1))
         END IF
      END IF


      !      CM_00=-4d0*Pi*ScaleGrav*dV
      !      CM_10=-4d0*Pi*ScaleGrav*dV
      !      CM_11=-.5d0*4d0*Pi*ScaleGrav*dV
      !      CM_20=-.25d0*4d0*Pi*ScaleGrav*dV
      !      CM_21=-3d0/2d0*4d0*Pi*ScaleGrav*dV
      !      CM_22=-3d0/8d0*4d0*Pi*ScaleGrav*dV
      CM_00=-ScaleGrav*dV
      CM_10=-ScaleGrav*dV
      CM_11=-.5d0*ScaleGrav*dV
      CM_20=-.25d0*ScaleGrav*dV
      CM_21=-3d0/2d0*ScaleGrav*dV
      CM_22=-3d0/8d0*ScaleGrav*dV

      IF (nDim == 3) THEN

         !-----------------------------------------------------------------------!
         !             ***   Symmetry of Spherical Harmonics ***                 !
         !-----------------------------------------------------------------------!
         ! Term                                                Symmetry Real/Imag!
         !-----------------------------------------------------------------------!
         ! Y_0                                                     xyz / xyz     !
         !-----------------------------------------------------------------------!
         ! Y_1_-1 = sqrt(1-z^2)*(x-iy)                              yz / xz      !
         ! Y_1_0 = z                                                xy / xy      !
         ! Y_1_1 = sqrt(1-z^2)*(x+iy)                               yz / xz      !
         !-----------------------------------------------------------------------!
         ! Y_2_-2 = (1-z^2)*(x-iy)^2 = (1-z^2)*(x^2+y^2-2*i*x*y)   xyz / z       !
         ! Y_2_-1 = z*sqrt(1-z^2)*(x-iy)                             y / x       !
         ! Y_2_0 = (3*z^2-1)                                       xyz / xyz     !
         ! Y_2_1 = z*sqrt(1-z^2)*(x-iy)                              y / x       !
         ! Y_2_-2 = (1-z^2)*(x+iy)^2 = (1-z^2)*(x^2+y^2+2*i*x*y)   xyz / z       !
         !-----------------------------------------------------------------------!


         !----------------------------------------------------------|       
         !   ***  Multiplication factors given grid symmetry  ***   |
         !----------------------------------------------------------|
         ! symmetry |  0  |    -1    0    1   |   -2  -1   0   1   2  |
         ! none     | 1/1 |    1/1  1/1  1/1  |   1/1 1/1 1/1 1/1 1/1 |
         ! x only   | 2/2 |    0/2  2/2  0/2  !   2/0 0/2 2/2 0/2 2/0 |
         ! y only   | 2/2 |    2/0  2/2  2/0  |   2/0 0/2 2/2 0/2 2/0 |
         ! z only   | 2/2 |    2/2  0/0  2/2  |   2/2 0/0 2/2 0/0 2/2 | 
         ! xy       | 4/4 |    0/0  4/4  0/0  |   4/0 0/4 4/4 0/4 4/0 |
         ! xz       | 4/4 |    0/4  0/0  0/4  |   4/0 0/0 4/4 0/0 4/0 | 
         ! yz       | 4/4 |    4/0  0/0  4/0  |   4/0 0/0 4/4 0/0 4/0 |
         ! xyz      | 8/8 |    0/0  0/0  0/0  |   8/0 0/0 8/8 0/0 8/0 |
         !----------------------------------------------------------|
         MomentFactors(:,:)=1
         IF (lReflect(1)) MomentFactors=MomentFactors*reshape((/2,2, 0,2, 2,2, 0,2, 2,0, 0,2, 2,2, 0,2, 2,0/),(/9,2/),(/0/),(/2,1/))
         IF (lReflect(2)) MomentFactors=MomentFactors*reshape((/2,2, 2,0, 2,2, 2,0, 2,0, 0,2, 2,2, 0,2, 2,0/),(/9,2/),(/0/),(/2,1/))
         IF (lReflect(3)) MomentFactors=MomentFactors*reshape((/2,2, 2,2, 0,0, 2,2, 2,2, 0,0, 2,2, 0,0, 2,2/),(/9,2/),(/0/),(/2,1/))
      ELSE
         MomentFactors(:,:)=1
         IF (lReflect(1)) MomentFactors(1,:)=MomentFactors(1,:)*(/2,2/)
         IF (lReflect(2)) MomentFactors(1,:)=MomentFactors(1,:)*(/2,2/)        
      END IF
   END SUBROUTINE InitMultiPoleMoments


  !> Routine for calculating multipole moments from root level density distribution
  SUBROUTINE CalcMultiPoleMoments()
    REAL(KIND=qPREC) :: rmass(4)
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    INTEGER :: iErr
    TYPE(InfoDef), POINTER :: info
    REAL(KIND=qPREC) :: x,y,z,s,r,costheta,sintheta,cosphi,sinphi,cos2phi,sin2phi,r2,r3, dx, dy, dz, rho, rrho, rrrho, pos(3), dm, theta
    INTEGER :: jmin, jmax, kmin, kmax, imin, imax, dim, i, j, k
    INTEGER, DIMENSION(3,2) :: ip
    nodelist=>Nodes(0)%p
    !      massfact=1
    rmass=0
    moments=0
    DO WHILE(ASSOCIATED(nodelist))
       info=>nodelist%self%info
       DO dim=1,nDim
          IF (lMultiPole(dim)) THEN
             ip(:,1)=1
             ip(:,2)=Info%mX(:)
             DO i=1, Info%mX(dim)
                ip(dim,:)=i
                x=Info%xBounds(dim,1)+(REAL(i)-half)*levels(0)%dx
                dm=SUM(Info%q(ip(1,1):ip(1,2), ip(2,1):ip(2,2), ip(3,1):ip(3,2),1))
                IF (dim == 1) rmass(4)=rmass(4)+dm
                rmass(dim) = rmass(dim) + dm*x
             END DO
          END IF
       END DO
       nodelist=>nodelist%next
    END DO
    CALL StartTimer(iBarrier, 0)
    CALL MPI_ALLREDUCE(MPI_IN_PLACE, rmass, 4, MPI_DOUBLE_PRECISION, MPI_SUM, levels(0)%MPI_COMM, iErr)
    CALL StopTimer(iBarrier, 0)
    multipole_com=rmass(1:3)/rmass(4)
    DO dim=1,nDim
       IF (lReflect(dim)) THEN !ANY(Gmthbc(dim,edge) == (/REFLECT_WALL, REFLECT_BPARALLEL, REFLECT_CYLINDRICAL/))) THEN
          multipole_com(dim)=GxBounds(dim,1)
       END IF
    END DO
    multipole_radius=huge(multipole_radius)
    DO dim=1,nDim
       IF (lMultiPole(dim)) THEN
          multipole_radius=min(multipole_radius,abs(multipole_com(dim)-GxBounds(dim,2)))
       END IF
    END DO
!    write(*,*) multipole_com, multipole_radius
    SELECT CASE(nDim)
    CASE(2)
       z=GxBounds(3,1)
       nodelist=>Nodes(0)%p
       DO WHILE(ASSOCIATED(nodelist))
          info=>nodelist%self%info

          dx=multipole_radius
          imin=max(1,floor((multipole_com(1)-dx-Info%xBounds(1,1))/levels(0)%dx+half))
          imax=min(Info%mX(1),ceiling((multipole_com(1)+dx-Info%xbounds(1,1))/levels(0)%dx+half))           
          DO i=imin,imax
             x=Info%Xbounds(1,1)+(REAL(i)-half)*levels(0)%dx - multipole_com(1)

             dy=sqrt(multipole_radius**2-x**2)
             jmin=max(1,floor((multipole_com(2)-dy-Info%xbounds(2,1))/levels(0)%dx+half))
             jmax=min(Info%mX(2),ceiling((multipole_com(2)+dy-Info%xbounds(2,1))/levels(0)%dx+half))
             DO j=jmin,jmax
                y=Info%xbounds(2,1)+(REAL(j)-half)*levels(0)%dx - multipole_com(2)
                pos=(/x,y,z/)

                r=sqrt(SUM(pos(1:nDim)**2))
                IF (r < multipole_radius) THEN

                   rho=Info%q(i,j,1,1)
                   moments(1)=moments(1)+rho
!                   write(*,*) moments(1), rho, massfact
                   IF (lDipole) THEN
                      moments(2)=moments(2)+rho*x
                      moments(3)=moments(3)+rho*y
                      IF (lQuadrupole) THEN
                         rrho=rho/2d0*r**2
                         theta=getphi(x,y)
                         moments(4)=moments(4)+rrho*(cos(2d0*theta))
                         moments(5)=moments(5)+rrho*(sin(2d0*theta))
                      END IF
                   END IF
                END IF
             END DO
          END DO
          nodelist=>nodelist%next
       END DO
    CASE(3)




       nodelist=>Nodes(0)%p
       DO WHILE(ASSOCIATED(nodelist))
          info=>nodelist%self%info

          dx=multipole_radius
          imin=max(1,floor((multipole_com(1)-dx-Info%xBounds(1,1))/levels(0)%dx+half))
          imax=min(Info%mX(1),ceiling((multipole_com(1)+dx-Info%xbounds(1,1))/levels(0)%dx+half))           
          DO i=imin,imax
             x=Info%Xbounds(1,1)+(REAL(i)-half)*levels(0)%dx - multipole_com(1)

             dy=sqrt(multipole_radius**2-x**2)
             jmin=max(1,floor((multipole_com(2)-dy-Info%xbounds(2,1))/levels(0)%dx+half))
             jmax=min(Info%mX(2),ceiling((multipole_com(2)+dy-Info%xbounds(2,1))/levels(0)%dx+half))
             DO j=jmin,jmax
                y=Info%xbounds(2,1)+(REAL(j)-half)*levels(0)%dx - multipole_com(2)

                dz=sqrt(dy**2-y**2)
                kmin=max(1,floor((multipole_com(3)-dz-Info%xbounds(3,1))/levels(0)%dx+half))
                kmax=min(Info%mX(3),ceiling((multipole_com(3)+dz-Info%xbounds(3,1))/levels(0)%dx+half))              
                DO k=kmin,kmax                              
                   z=Info%xbounds(3,1)+(REAL(k)-half)*levels(0)%dx - multipole_com(3)

                   pos=(/x,y,z/)
                   r=sqrt(SUM(pos(1:nDim)**2))
                   IF (r < multipole_radius) THEN


                      rho=Info%q(i,j,k,1)
                      moments(1)=moments(1)+rho


                      IF (lDiPole) THEN
                         s=sqrt(x**2+y**2)
                         costheta=z/r
                         sintheta=s/r
                         cosphi=x/s
                         sinphi=y/s
                         IF (lQuadrupole) THEN
                            cos2phi=(cosphi**2-sinphi**2)
                            sin2phi=2d0*cosphi*sinphi
                         END IF
                         cosphi=cosphi
                         sinphi=sinphi
                         rrho=r*rho
                         moments(2)=moments(2)+sintheta*cmplx(cosphi,+sinphi) * rrho
                         moments(3)=moments(3)+costheta * rrho
                         moments(4)=moments(4)+sintheta*cmplx(cosphi,-sinphi) * rrho

                         IF (lQuadrupole) THEN
                            rrrho=r*rrho
                            cos2phi=(cosphi**2-sinphi**2) * cos2phi_fact
                            sin2phi=2d0*cosphi*sinphi * sin2phi_fact
                            moments(5)=moments(5)+sintheta**2*cmplx(cos2phi,sin2phi)*rrrho
                            moments(6)=moments(6)+sintheta*costheta*cmplx(cosphi,+sinphi) * rrrho
                            moments(7)=moments(7)+(3d0*(z/r)**2-1d0) * rrrho
                            moments(8)=moments(8)+sintheta*costheta*cmplx(cosphi,-sinphi) * rrrho
                            moments(9)=moments(9)+sintheta**2*cmplx(cos2phi,-sin2phi) * rrrho
                         END IF
                      END IF
                   END IF
                END DO
             END DO
          END DO
          nodelist=>nodelist%next
       END DO
    END SELECT
    DO i=1,size(moments)
       moments(i)=cmplx(real(moments(i))*momentfactors(i,1),aimag(moments(i))*momentfactors(i,2))
    END DO
    CALL MPI_ALLREDUCE(MPI_IN_PLACE, moments, size(moments), MPI_DOUBLE_COMPLEX, MPI_SUM, levels(0)%MPI_COMM, iErr)
!    write(*,*) 'moments=', moments*levels(0)%dx**nDim
    IF (nDim == 3) THEN
       moments(1)=moments(1)*CM_00
!@       write(*,*) 'moments_now', moments(1), CM_00, ScaleGrav, massfact
      IF (lDipole) THEN
          moments(2)=moments(2)*CM_11
          moments(3)=moments(3)*CM_10
          moments(4)=-moments(4)*CM_11
          IF (lQuadrupole) THEN
             moments(5)=moments(5)*CM_22
             moments(6)=moments(6)*CM_21
             moments(7)=moments(7)*CM_20
             moments(8)=-moments(8)*CM_21
             moments(9)=moments(9)*CM_22
          END IF
       END IF
    ELSE
!       write(*,*) 'moments=', moments*levels(0)%dx**nDim
       moments(:)=-moments(:)*2d0*CM_00

    END IF
!    moments(2:)=0d0
  END SUBROUTINE CalcMultiPoleMoments


  !> Routine for setting phi in the ghost regions of a grid based on the moments
  !! @param Info Info object
  !! @param ip Bounds of Info to update
  SUBROUTINE SetPhi(Info, ip)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k
    INTEGER, DIMENSION(3,2) :: ip
    REAL(KIND=qPREC) :: x,y,z,s,r,costheta,sintheta,cosphi,sinphi,r2,r3,s2
!    write(*,*) 'setting phi', ip
    DO i=ip(1,1),ip(1,2)
       x=Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx - multipole_com(1)

       DO j=ip(2,1), ip(2,2)
          y=Info%xbounds(2,1)+(REAL(j)-half)*levels(Info%level)%dx - multipole_com(2)
          s2=x**2+y**2
          s=sqrt(s2)
          cosphi=x/s
          sinphi=y/s
          DO k=ip(3,1),ip(3,2)
             IF (nDim == 3) THEN
                z=Info%xbounds(3,1)+(REAL(k)-half)*levels(Info%level)%dx - multipole_com(3)
                r2=(s**2+z**2)
                r=sqrt(r2)
                costheta=z/r
                sintheta=s/r
             END IF
!             write(*,*) r, moments(1)
             IF (nDim == 3) THEN
                Info%q(i,j,k,iPhiGas)=moments(1)/r
                
                IF (lDiPole) THEN
                   Info%q(i,j,k,iPhiGas)=Info%q(i,j,k,iPhiGas)+&
                        abs(moments(2)*sintheta*cmplx(cosphi,-sinphi) + &
                        moments(3)*costheta + &
                        moments(4)*sintheta*cmplx(cosphi,+sinphi))/r2
                   IF (lQuadrupole) THEN
                      r3=r2*r
                      Info%q(i,j,k,iPhiGas)=Info%q(i,j,k,iPhiGas)+ &
                           abs(moments(5)*(sintheta*cmplx(cosphi,-sinphi))**2 + &
                           moments(6)*sintheta*costheta*cmplx(cosphi,-sinphi) + &
                           moments(7)*(3d0*costheta**2-1d0) + &
                           moments(8)*sintheta*costheta*cmplx(cosphi,+sinphi) + &
                           moments(9)*(sintheta*cmplx(cosphi,+sinphi))**2)/r3
                   END IF
                END IF
             ELSE
                Info%q(i,j,k,iPhiGas)=moments(1)*log(s/R2DEff)
!                write(*,*) moments(1), s, Info%q(i,j,k,iPhiGas) !REAL(moments(1)*log(s/R2Deff))
                IF (lDiPole) THEN
!                   Info%q(i,j,k,iPhiGas)=Info%q(i,j,k,iPhiGas)-&
!                        (moments(2)*cosphi+moments(3)*sinphi)/s
                   IF (lQuadrupole) THEN
!                      Info%q(i,j,k,iPhiGas)=Info%q(i,j,k,iPhiGas)-&
!                           (moments(4)*(cosphi**2-sinphi**2)+moments(5)*2d0*sinphi*cosphi)/s2
                   END IF
                END IF
             END IF
             Info%q(i,j,k,iPhiDot)=0d0
!             Info%q(i,j,k,iPhi)=Info%q(i,j,k,iPhiGas)+Info%q(i,j,k,iPhiSinks)
          END DO
       END DO
    END DO
!    write(*,*) maxval(Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),iPhiGas))
!    write(*,*) ip
!    pause
  END SUBROUTINE SetPhi
END module multipole
