!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    poisson.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir self_gravity
!! @brief Contains modules for solving the equation of self gravity.

!> @file poisson.f90
!! @brief Main file for module Poisson

!> @defgroup SelfGravity Self Gravity
!! @brief Module group for handling equations of self gravity
!! @ingroup Elliptic

!> @defgroup Poisson Poisson
!! @brief Module for solving Poisson's equation
!! @ingroup SelfGravity

!> Module for solving Poisson's equation
!! @ingroup Poisson
MODULE Poisson
  USE EllipticDeclarations
  USE GlobalDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE PhysicsDeclarations
  USE MultiPole
  USE EllipticComms
  IMPLICIT NONE
  INTEGER, DIMENSION(:,:), ALLOCATABLE :: offsets
  INTEGER, DIMENSION(:), ALLOCATABLE :: StencilValues
  INTEGER, DIMENSION(:), ALLOCATABLE :: iEntries
  INTEGER, DIMENSION(3,2) :: Poisson_mthbc
  REAL(KIND=qPREC) :: PoissonScale
  INTEGER,PARAMETER,PUBLIC :: ZEROSLOPE=0,REFLECTING=1,PERIODIC=2,MULTIPOLE_EXPANSION=3,USERSPECIFIED=4
  LOGICAL, DIMENSION(3) :: lPoissonPeriodic
  INTEGER :: UpdateFreq=1
  SAVE
  ! Initializes local variables and creates stencil needed for poisson equation
  !   INCLUDE 'mpif.h'
  INCLUDE 'fftw3.f'
CONTAINS

  !> @name Initialization routines
  !! @{

  !> Initializes/Reads in variables needed by Poisson Module and sets up EllipticObject
  SUBROUTINE Poisson_Init(EllipticObject)
    ! EllipticObject%stencil - handle for hyper stencil created here
    ! EllipticObject%Interface - (Currently only supports StructInterface==1)
    ! EllipticObject%Solver - (StuctPCG=1, StructGMRES=2)
    ! EllipticObject%tolerance - (Solver tolerance)
    ! EllipticObject%MaxIters  - (Maximum # of iterations)
    ! EllipticObject%hverbosity - (hypre verbosity)

    TYPE(EllipticObjectDef) :: EllipticObject
    INTEGER :: iErr
    INTEGER :: i,j,m
    INTEGER :: solver         ! type of HYPRE solver
    INTEGER :: printLevel     ! verbosity
    INTEGER :: maxIters       ! maximum number of iterations
    INTEGER :: hVerbosity     ! how verbose user-side HYPRE should be
    REAL    :: tolerance      ! convergence tolerance
    INTEGER, DIMENSION(3,2) :: mthbc
    NAMELIST /PoissonData/ solver,tolerance,printlevel,MaxIters,hverbosity, mthbc

    ! Read in scheme value.
    READ(GLOBAL_DATA_HANDLE,NML=PoissonData,IOStat=iErr)

    IF(iErr/=0) THEN
       PRINT *, "PoissonInit() error:  unable to read ", GLOBAL_DATA_FILE, "."
       STOP
    END IF
    PoissonScale=4d0*Pi*ScaleGrav
    EllipticObject%Interface=StructInterface
    EllipticObject%Solver=solver
    EllipticObject%tolerance=tolerance
    EllipticObject%PrintLevel=printlevel
    EllipticObject%MaxIters=MaxIters
    EllipticObject%hverbosity=hverbosity
    Poisson_mthbc=mthbc   
    CALL Checkmthbc(mthbc)    
!    lPoissonPeriodic=ANY(mthbc(1:nDim,:)==2,2)
    lEllipticPeriodic(1:nDim)=lEllipticPeriodic(1:nDim) .OR. lPoissonPeriodic(1:nDim) !ANY(mthbc(1:nDim,:)==2,2)
    lGravityPeriodic=ANY(lPoissonPeriodic)
!    write(*,*) 'lEllipticPeriodic=', lEllipticPeriodic
    !      write(*,*) mthbc, lReflect
    ! Sets up stencil offsets for poisson solve
    ALLOCATE(offsets(nDim,0:2*nDim))
    ALLOCATE(StencilValues(0:2*nDim))
    ALLOCATE(iEntries(1:2*nDim+1))
    iEntries=(/(i,i=0,2*nDim)/)

    CALL C_StructStencilCreate(ndim, 2*nDim+1, EllipticObject%stencil, iErr) ;CALL CheckErr('C_StructStencilCreate',iErr)
    offsets(:,0)=0
    stencilvalues(0)=-2*nDim
    m=0
    DO i=1,nDim
       DO j=-1,1,2
          m=m+1
          offsets(:,m)=0
          offsets(i,m)=j
          stencilvalues(m)=1
       END DO
    END DO
    DO m=0,2*nDim
       CALL C_StructStencilSetElement(EllipticObject%stencil, m,offsets(:,m), iErr)
       CALL CheckErr('C_StructStencilSetElement',iErr)
    END DO
    CALL InitMultiPoleMoments
  END SUBROUTINE Poisson_Init
  !> @}

  !> @name Communication routines
  !! @{

  !> Performs communications needed before first solve
  !! @param n level
  !! @param done_solving Logical flag that determines whether a solve needs to be performed
  SUBROUTINE PoissonPreSolveComm(n,done_solving)
    INTEGER :: n
    LOGICAL :: done_solving
!    IF (n == 0 .AND. ALL(lPoissonPeriodic(1:nDim))) CALL UpdateMeanDensity(n)
    done_solving=.false.
  END SUBROUTINE PoissonPreSolveComm

  !> Performs communication in between solves
  !! @param n level
  !! @param done_solving Logical flag that determines whether a solve needs to be performed
  SUBROUTINE PoissonBetweenSolveComm(n,done_solving)
    INTEGER :: n
    LOGICAL :: done_solving
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist     
    done_solving=.true.
  END SUBROUTINE PoissonBetweenSolveComm


  !> Performs communication post solves
  !! @param n level
  SUBROUTINE PoissonPostSolveComm(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node, overlap
    TYPE(NodeDefList), POINTER :: nodelist, overlaplist

    ! Need to ghost variables q(:,:,:,iPhi) and q(:,:,:,iPhiDot) into regions to calculate source terms.
!    CALL EllipticTransfer(n,(/iPhiGas,iPhi,iPhiDot/),levels(n)%gmbc(levels(n)%step+1)+1)
  END SUBROUTINE PoissonPostSolveComm

  !> @}

  !> @name Poisson Level Operations
  !! @{

  !> Loops over grids setting matrix values for poisson equation
  !! @param n level
  !! @param EllipticLevelObject handle to Elliptic Level Object
  SUBROUTINE Poisson_Setup(n,EllipticLevelObject)
    INTEGER :: n
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist        
    nodelist=>nodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL PoissonMatrixSetBoxValues(n,node%info,EllipticLevelObject)
       nodelist=>nodelist%next
    END DO
  END SUBROUTINE Poisson_Setup

  !> Loops over grids setting vector values
  !! @param n level
  !! @param EllipticLevelObject handle to Elliptic Level Object
  SUBROUTINE Poisson_LoadVectors(n,EllipticLevelObject)
    INTEGER :: n
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist  
    nodelist=>nodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL PoissonVectorSetBoxValues(n,node%info,EllipticLevelObject)
       nodelist=>nodelist%next
    END DO
  END SUBROUTINE Poisson_LoadVectors

  !> Loops over grids retrieving vector values
  !! @param n level
  !! @param EllipticLevelObject handle to Elliptic Level Object
  SUBROUTINE Poisson_UnLoadVectors(n,EllipticLevelObject)
    INTEGER :: n
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist        
    nodelist=>nodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL PoissonVectorGetBoxValues(n,node%info,EllipticLevelObject)
       nodelist=>nodelist%next
    END DO
!    write(*,*) 'solved for phigas and updated phidot and phi on level', n 
  END SUBROUTINE Poisson_UnLoadVectors


  !> Sets boundary values for calculating source terms
  !! @param n level
  SUBROUTINE PoissonSetBC(n)
    INTEGER :: n, rmbc
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    rmbc=levels(n)%egmbc(levels(n)%step)
    IF (n == 0 .AND. ANY(lMultiPole)) CALL CalcMultiPoleMoments()
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       IF (ANY(lReflect(1:nDim))) CALL ApplyPoissonPhysicalBC(node%info, rmbc)
       IF (ANY(lMultiPole(1:nDim))) CALL ApplyAnalyticBC(node%info, rmbc)
       nodelist=>nodelist%next 
    END DO
 END SUBROUTINE PoissonSetBC

  !> Post Elliptic routines
  !! @param n level
  SUBROUTINE PoissonPostElliptic(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    CALL EllipticTransfer(n, (/iPhiGas, iPhiDot/), levels(n)%egmbc(levels(n)%step))

    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       CALL PoissonGetMaxSpeed(node%info)
       nodelist=>nodelist%next 
    END DO

!    write(*,*) 'updating mass fluxes with new potential on level', n
    
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       CALL SelfGravCorrection(node%info)
       nodelist=>nodelist%next 
    END DO


  END SUBROUTINE PoissonPostElliptic

  !> Updates phi with analytic solution at boundaries prior to solving
  !! @param n level
  SUBROUTINE UpdatePhiAtBoundaries(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    nodelist=>Nodes(n)%p
    DO WHILE (ASSOCIATED(nodelist))
       CALL ApplyAnalyticBC(nodelist%self%info, 1)
       nodelist=>nodelist%next
    END DO
  END SUBROUTINE UpdatePhiAtBoundaries


  !> Corrects momentum and energy sources to use time centered phi
  SUBROUTINE SelfGravCorrection(info)
    TYPE(InfoDef) :: Info
    INTEGER, DIMENSION(3,2) :: mb ! bounds to update
    INTEGER :: i,j,k,l,m,mbc,gpmb(3,3,2) ! size of ghost regions
    REAL(KIND=qPREC) :: dx, iPhiSign(2),dtdx
    INTEGER, DIMENSION(2) :: iPhiIndex(2)
    REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: f2x_, f2y_, f2z_, phi
    REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: gradphix_, gradphiy_, gradphiz_
    IF (levels(Info%level)%dt == 0) RETURN
!    RETURN
    dx=levels(Info%level)%dx
    mb=1
    mbc=levels(Info%level)%ambc(levels(Info%level)%step)

    !Expand bounds to include row of ghost zones for phi
    mb(1:nDim,1)=1-mbc-1
    mB(1:nDim,2)=Info%mx(1:nDim)+mbc+1
    
    allocate(phi(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),2))
    phi(:,:,:,2)=Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), iPhiGas)
    phi(:,:,:,1)=phi(:,:,:,2)-Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2), iPhiDot)*levels(Info%level)%dt
!    IF (Info%level == 0) THEN
!       write(*,*) mb
!       write(*,*) minval(phi), maxval(phi)
!    END IF
    !Set bounds for grad phi
    gpmb=1
    DO i=1,nDim
       gpmb(i,1:nDim,1)=mb(1:nDim,1)
       gpmb(i,1:nDim,2)=mb(1:nDim,2)
       gpmb(i,i,1)=mb(i,1)+1
    END DO
!    write(*,*) 'gpmb=', gpmb
    !Readjust bounds for region of q to update
    mb(1:nDim,1)=mB(1:nDim,1)+1
    mB(1:nDim,2)=mB(1:nDim,2)-1
!    write(*,*) 'reset mb to', mB
    ALLOCATE(gradphix_(gpmb(1,1,1):gpmb(1,1,2), gpmb(1,2,1):gpmb(1,2,2), gpmb(1,3,1):gpmb(1,3,2)))
    ALLOCATE(f2x_(mb(1,1):mb(1,2)+1,mb(2,1):mb(2,2),mb(3,1):mb(3,2),1:nDim))
    f2x_=0d0
    IF (nDim >= 2) THEN
       ALLOCATE(gradphiy_(gpmb(2,1,1):gpmb(2,1,2), gpmb(2,2,1):gpmb(2,2,2), gpmb(2,3,1):gpmb(2,3,2)))
       ALLOCATE(f2y_(mb(1,1):mb(1,2),mb(2,1):mb(2,2)+1,mb(3,1):mb(3,2),1:nDim))
       f2y_=0d0
       IF (nDim >= 3) THEN
          ALLOCATE(gradphiz_(gpmb(3,1,1):gpmb(3,1,2), gpmb(3,2,1):gpmb(3,2,2), gpmb(3,3,1):gpmb(3,3,2)))
          ALLOCATE(f2z_(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2)+1,1:nDim))
          f2z_=0d0
       END IF
    END IF
    
    iPhiSign=(/-1d0,1d0/)
    
    DO l=1,2
       FORALL (i=gpmb(1,1,1):gpmb(1,1,2), j=gpmb(1,2,1):gpmb(1,2,2), k=gpmb(1,3,1):gpmb(1,3,2))
          gradphix_(i,j,k)=(phi(i,j,k,l)-phi(i-1,j,k,l))/dx
       END FORALL
       IF (nDim >= 2) THEN
          FORALL (i=gpmb(2,1,1):gpmb(2,1,2), j=gpmb(2,2,1):gpmb(2,2,2), k=gpmb(2,3,1):gpmb(2,3,2))
             gradphiy_(i,j,k)=(phi(i,j,k,l)-phi(i,j-1,k,l))/dx 
          END FORALL
          IF (nDim >= 3) THEN
             FORALL (i=gpmb(3,1,1):gpmb(3,1,2), j=gpmb(3,2,1):gpmb(3,2,2), k=gpmb(3,3,1):gpmb(3,3,2))
                gradphiz_(i,j,k)=(phi(i,j,k,l)-phi(i,j,k-1,l))/dx 
             END FORALL
          END IF
       END IF
       dtdx=half*levels(Info%level)%dt/dx*iPhiSign(l)
       IF (nDim == 1) THEN
          mB(1,2)=mb(1,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2x_(i,j,k,1) = f2x_(i,j,k,1) + &
                  dtdx*(.125d0/Pi/ScaleGrav*gradphix_(i,j,k)**2+half*mean_density*(phi(i,j,k,l)+phi(i-1,j,k,l)))
          END FORALL
          mB(1,2)=mb(1,2)-1
       ELSEIF (nDim == 2) THEN
          mB(1,2)=mb(1,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2x_(i,j,k,1) = f2x_(i,j,k,1) + dtdx*(.125d0/Pi/ScaleGrav*&
                  (gradphix_(i,j,k)**2-(.25d0*sum(gradphiy_(i-1:i,j:j+1,k)))**2)+&
                  half*mean_density*SUM(phi(i-1:i,j,k,l)))
             f2x_(i,j,k,2) = f2x_(i,j,k,2) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphix_(i,j,k)*.25d0*sum(gradphiy_(i-1:i,j:j+1,k))))
          END FORALL
          mB(1,2)=mb(1,2)-1
          mB(2,2)=mb(2,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2y_(i,j,k,2) = f2y_(i,j,k,2) + dtdx*(.125d0/Pi/ScaleGrav*&
                  (gradphiy_(i,j,k)**2-(.25d0*sum(gradphix_(i:i+1,j-1:j,k)))**2)+&
                  half*mean_density*SUM(phi(i,j-1:j,k,l)))
             f2y_(i,j,k,1) = f2y_(i,j,k,1) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphiy_(i,j,k)*.25d0*sum(gradphix_(i:i+1,j-1:j,k))))
          END FORALL
          mB(2,2)=mb(2,2)-1
       ELSE! nDim == 3
          mB(1,2)=mb(1,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2x_(i,j,k,1) = f2x_(i,j,k,1) + dtdx*(.125d0/Pi/ScaleGrav*&
                  (gradphix_(i,j,k)**2-(.25d0*sum(gradphiy_(i-1:i,j:j+1,k)))**2-(.25d0*sum(gradphiz_(i-1:i,j,k:k+1)))**2)+&
                  half*mean_density*SUM(phi(i-1:i,j,k,l)))
             f2x_(i,j,k,2) = f2x_(i,j,k,2) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphix_(i,j,k)*.25d0*sum(gradphiy_(i-1:i,j:j+1,k))))
             f2x_(i,j,k,3) = f2x_(i,j,k,3) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphix_(i,j,k)*.25d0*sum(gradphiz_(i-1:i,j,k:k+1))))
          END FORALL
          mB(1,2)=mb(1,2)-1
          mB(2,2)=mb(2,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2y_(i,j,k,2) = f2y_(i,j,k,2) + dtdx*(.125d0/Pi/ScaleGrav*&
                  (gradphiy_(i,j,k)**2-(.25d0*sum(gradphix_(i:i+1,j-1:j,k)))**2-(.25d0*sum(gradphiz_(i,j-1:j,k:k+1)))**2)+&
                  half*mean_density*SUM(phi(i,j-1:j,k,l)))
             f2y_(i,j,k,1) = f2y_(i,j,k,1) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphiy_(i,j,k)*.25d0*sum(gradphix_(i:i+1,j-1:j,k))))
             f2y_(i,j,k,3) = f2y_(i,j,k,3) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphiy_(i,j,k)*.25d0*sum(gradphiz_(i,j-1:j,k:k+1))))
          END FORALL
          mB(2,2)=mb(2,2)-1
          mB(3,2)=mb(3,2)+1
          FORALL (i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             f2z_(i,j,k,3) = f2z_(i,j,k,3) + dtdx*(.125d0/Pi/ScaleGrav*&
                  (gradphiz_(i,j,k)**2-(.25d0*sum(gradphix_(i:i+1,j,k-1:k)))**2-(.25d0*sum(gradphiy_(i,j:j+1,k-1:k)))**2)+&
                  half*mean_density*SUM(phi(i,j,k-1:k,l)))
             f2z_(i,j,k,1) = f2z_(i,j,k,1) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphiz_(i,j,k)*.25d0*sum(gradphix_(i:i+1,j,k-1:k))))
             f2z_(i,j,k,2) = f2z_(i,j,k,2) + dtdx*(.25d0/Pi/ScaleGrav*&
                  (gradphiz_(i,j,k)*.25d0*sum(gradphiy_(i,j:j+1,k-1:k))))
          END FORALL
          mB(3,2)=mb(3,2)-1
       END IF
    END DO
    FORALL(i=mB(1,1):mB(1,2),j=mb(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nDim)
       Info%q(i,j,k,imom(m))=Info%q(i,j,k,imom(m))+f2x_(i,j,k,m)-f2x_(i+1,j,k,m)
    END FORALL
    mB(1,2)=mB(1,2)+1
    CALL storefixupfluxes(Info,mB,1,f2x_,imom(1:nDim))
    mb(1,2)=mb(1,2)-1
    DEALLOCATE(f2x_, gradphix_)
    IF (nDim >= 2) THEN
       FORALL(i=mB(1,1):mB(1,2),j=mb(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nDim)
          Info%q(i,j,k,imom(m))=Info%q(i,j,k,imom(m))+f2y_(i,j,k,m)-f2y_(i,j+1,k,m)
       END FORALL
       mB(2,2)=mB(2,2)+1
       CALL storefixupfluxes(Info,mB,2,f2y_,imom(1:nDim))
       mB(2,2)=mB(2,2)-1
       DEALLOCATE(f2y_, gradphiy_)
       IF (nDim >= 3) THEN
          FORALL(i=mB(1,1):mB(1,2),j=mb(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nDim)
             Info%q(i,j,k,imom(m))=Info%q(i,j,k,imom(m))+f2z_(i,j,k,m)-f2z_(i,j,k+1,m)
          END FORALL
          mB(3,2)=mB(3,2)+1
          CALL storefixupfluxes(Info,mB,3,f2z_,imom(1:nDim))
          mb(3,2)=mb(3,2)-1
          DEALLOCATE(f2z_, gradphiz_)
       END IF
    END IF

    IF (iE /= 0) THEN
       FORALL(i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
          Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-.25d0*( &
               Info%MassFlux(i,j,k,1)*(phi(i,j,k,2)-phi(i,j,k,1)-phi(i-1,j,k,2)+phi(i-1,j,k,1)) + &
               Info%MassFlux(i+1,j,k,1)*(phi(i+1,j,k,2)-phi(i+1,j,k,1)-phi(i,j,k,2)+phi(i,j,k,1)))
       END FORALL
       IF (nDim >= 2) THEN
          FORALL(i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
             Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-.25d0*( &
                  Info%MassFlux(i,j,k,2)*(phi(i,j,k,2)-phi(i,j,k,1)-phi(i,j-1,k,2)+phi(i,j-1,k,1)) + &
                  Info%MassFlux(i,j+1,k,2)*(phi(i,j+1,k,2)-phi(i,j+1,k,1)-phi(i,j,k,2)+phi(i,j,k,1)))
          END FORALL
          IF (nDim == 3) THEN
             FORALL(i=mb(1,1):mb(1,2), j=mb(2,1):mb(2,2), k=mb(3,1):mb(3,2))
                Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-.25d0*( &
                     Info%MassFlux(i,j,k,3)*(phi(i,j,k,2)-phi(i,j,k,1)-phi(i,j,k-1,2)+phi(i,j,k-1,1)) + &
                     Info%MassFlux(i,j,k+1,3)*(phi(i,j,k+1,2)-phi(i,j,k+1,1)-phi(i,j,k,2)+phi(i,j,k,1)))
             END FORALL
          END IF
       END IF
    END IF
    DEALLOCATE(phi)
  END SUBROUTINE SelfGravCorrection
  !> @}


  !> @name Poisson Info Operations
  !! @{

  !> Routine for setting up matrixboxvalues for given info structure.  Matrix handle stored in EllipticLevelObject%matrix
  !! @param n level
  !! @param Info Info object
  !! @param EllipticLevelObject Elliptic Level Object
  SUBROUTINE PoissonMatrixSetBoxValues(n,Info,EllipticLevelObject)
    INTEGER :: n ! Level
    TYPE(InfoDef) :: Info 
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject

    INTEGER :: nvalues,nCells
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: matrixvalues
    INTEGER :: i,j,k,m,edge,p,temp,ip(3),l
    LOGICAL, DIMENSION(3,2) :: internal_to_grid, internal_to_domain
    INTEGER :: ierr
    nvalues=size(offsets,2)
    nCells=product(Info%mX(1:nDim))
    ALLOCATE(matrixvalues(nCells*nvalues))

    matrixvalues=RESHAPE(SPREAD(StencilValues,2,nCells),shape(matrixvalues)) !default values
    m=1-nvalues
    DO k=Info%mGlobal(3,1),Info%mGlobal(3,2)
       internal_to_grid(3,1) = (k > Info%mGlobal(3,1))
       internal_to_grid(3,2) = (k < Info%mGlobal(3,2))
       IF (.NOT. internal_to_grid(3,1)) internal_to_domain(3,1) = lPoissonPeriodic(3) .OR. k > 1
       IF (.NOT. internal_to_grid(3,2)) internal_to_domain(3,2) = lPoissonPeriodic(3) .OR. k < levels(n)%mX(3)

       DO j=Info%mGlobal(2,1),Info%mGlobal(2,2)
          internal_to_grid(2,1) = (j > Info%mGlobal(2,1))
          internal_to_grid(2,2) = (j < Info%mGlobal(2,2))
          IF (.NOT. internal_to_grid(2,1)) internal_to_domain(2,1) = lPoissonPeriodic(2) .OR. j > 1
          IF (.NOT. internal_to_grid(2,2)) internal_to_domain(2,2) = lPoissonPeriodic(2) .OR. j < levels(n)%mX(2)

          DO i=Info%mGlobal(1,1), Info%mGlobal(1,2)
             internal_to_grid(1,1) = (i > Info%mGlobal(1,1))
             internal_to_grid(1,2) = (i < Info%mGlobal(1,2))
             IF (.NOT. internal_to_grid(1,1)) internal_to_domain(1,1) = lPoissonPeriodic(1) .OR. i > 1
             IF (.NOT. internal_to_grid(1,2)) internal_to_domain(1,2) = lPoissonPeriodic(1) .OR. i < levels(n)%mX(1)
             m=m+nvalues
             IF (ALL(internal_to_grid)) CYCLE
             ip=(/i,j,k/)-Info%mGlobal(1:3,1)+1
             p=0
             DO l=1,nDim
                temp=ip(l)
                DO edge=1,2
                   p=p+1
                   IF (internal_to_grid(l,edge)) CYCLE
                   IF (internal_to_domain(l,edge)) THEN !zero out connections to ancestors
                      ip(l)=temp+(-1)**edge
                      IF (isAncestor(Info%ChildMask(ip(1),ip(2),ip(3)))) matrixvalues(m+p) = 0 !zero out matrix connections to other levels
                      
                   ELSE !physical boundary condition exists
                      matrixvalues(m+p) = 0                       
                      SELECT CASE (Poisson_mthbc(l,edge))
                      CASE(MULTIPOLE_EXPANSION)
                         !Do nothing - handled by source terms
                      CASE(REFLECTING)
                         !matrixvalues(m+p-(-1)**edge)=matrixvalues(m+p-(-1)**edge)+StencilValues(p)
                         matrixvalues(m)=matrixvalues(m)+StencilValues(p)
                      CASE(ZEROSLOPE)
                         matrixvalues(m)=matrixvalues(m)-StencilValues(p)
                      END SELECT
                   END IF
                END DO
                ip(l)=temp
             END DO
          END DO
       END DO
    END DO
    CALL C_StructMatrixSetBoxValues(EllipticLevelObject%Matrix, Info%mGlobal(:,1), Info%mGlobal(:,2), nvalues, iEntries, matrixValues, iErr)
    CALL CheckErr('C_StructMatrixSetBoxValues',iErr)
    DEALLOCATE(matrixValues)
  END SUBROUTINE PoissonMatrixSetBoxValues

  !> Routine for setting up lhs and rhs vectors for given info structure.  
  !! @param n level
  !! @param Info Info object
  !! @param EllipticLevelObject Elliptic Level Object
  !! @details Vector handles stored in EllipticLevelObject%SolutionVector and  EllipticLevelObject%VariableVector
  SUBROUTINE PoissonVectorSetBoxValues(n,Info,EllipticLevelObject)
     USE EOS
    INTEGER :: n
    TYPE(InfoDef) :: Info
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject

    INTEGER ::nCells
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: vectorvalues
    INTEGER :: i,j,k,m,edge,p,temp,mb(3,2),ip(3),l,mbc
    LOGICAL, DIMENSION(3,2) :: internal_to_grid, internal_to_domain
    INTEGER :: iErr
    REAL(KIND=qPREC), DIMENSION(3) :: pos
    REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: oldphi
    mbc=levels(Info%level)%egmbc(levels(Info%level)%step)
    mb(nDim+1:3,:)=1
    mb(1:nDim,1)=1-mbc
    mb(1:nDim,2)=Info%mX(1:nDim)+mbc
    CALL Protectq(Info, mb, 'setboxvalues')

    nCells=product(Info%mX(1:nDim))
    !Update phiGas with predictor and store old phigas in phidot
    IF (levels(n)%dt > 0) THEN
       ALLOCATE(OldPhi(Info%mX(1),Info%mX(2),Info%mX(3)))
       Oldphi(1:Info%mX(1),1:Info%mX(2), 1:Info%mX(3))= &
            Info%q(1:Info%mX(1),1:Info%MX(2),1:Info%mX(3),iPhiGas)
       Info%q(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2),iPhiGas)=&
            Info%q(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2),iPhiGas)+&
            levels(n)%dt*Info%q(mb(1,1):mb(1,2),mb(2,1):mb(2,2),mb(3,1):mb(3,2),iPhiDot)
       Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiDot)=Oldphi
       DEALLOCATE(oldphi)
    END IF
    
    ALLOCATE(vectorvalues(nCells))

    IF (ALL(lPoissonPeriodic(1:nDim))) THEN
       vectorvalues=reshape(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),irho)-mean_density,shape(vectorvalues))*levels(n)%dx**2*PoissonScale
 
    ELSEIF (ALL(lMultipole(1:nDim))) THEN
       m=0
       DO k=1,Info%mX(3)
          DO j=1,Info%mX(2)
             DO i=1,Info%mX(1)
                m=m+1
                pos=Info%xbounds(:,1)+(REAL((/i,j,k/))-half)*levels(Info%level)%dX - Multipole_COM(:)
                IF (SUM(pos(1:nDim)**2) < multipole_radius**2) THEN
                   vectorvalues(m)=Info%q(i,j,k,1)*levels(n)%dx**2*PoissonScale
                ELSE
                   vectorvalues(m)=0
                END IF
             END DO
          END DO
       END DO
    ELSE
       vectorvalues=reshape(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),irho),shape(vectorvalues))*levels(n)%dx**2*PoissonScale
    END IF

    m=0
    DO k=Info%mGlobal(3,1),Info%mGlobal(3,2)
       internal_to_grid(3,1) = (k > Info%mGlobal(3,1))
       internal_to_grid(3,2) = (k < Info%mGlobal(3,2))
       IF (.NOT. internal_to_grid(3,1)) internal_to_domain(3,1) = lPoissonPeriodic(3) .OR. k > 1
       IF (.NOT. internal_to_grid(3,2)) internal_to_domain(3,2) = lPoissonPeriodic(3) .OR. k < levels(n)%mX(3)

       DO j=Info%mGlobal(2,1),Info%mGlobal(2,2)
          internal_to_grid(2,1) = (j > Info%mGlobal(2,1))
          internal_to_grid(2,2) = (j < Info%mGlobal(2,2))
          IF (.NOT. internal_to_grid(2,1)) internal_to_domain(2,1) = lPoissonPeriodic(2) .OR. j > 1
          IF (.NOT. internal_to_grid(2,2)) internal_to_domain(2,2) = lPoissonPeriodic(2) .OR. j < levels(n)%mX(2)

          DO i=Info%mGlobal(1,1), Info%mGlobal(1,2)
             internal_to_grid(1,1) = (i > Info%mGlobal(1,1))
             internal_to_grid(1,2) = (i < Info%mGlobal(1,2))
             IF (.NOT. internal_to_grid(1,1)) internal_to_domain(1,1) = lPoissonPeriodic(1) .OR. i > 1
             IF (.NOT. internal_to_grid(1,2)) internal_to_domain(1,2) = lPoissonPeriodic(1) .OR. i < levels(n)%mX(1)
             m=m+1
             IF (ALL(internal_to_grid)) CYCLE
             ip=(/i,j,k/)-Info%mGlobal(:,1)+1
             p=0
             DO l=1,nDim
                temp=ip(l)
                DO edge=1,2
                   p=p+1
                   IF (internal_to_grid(l,edge)) CYCLE
                   ip(l)=temp+(-1)**edge
                   IF ((internal_to_domain(l,edge) .AND. isAncestor(Info%childmask(ip(1),ip(2),ip(3)))) .OR. ((.NOT. internal_to_domain(l,edge)) .AND. (Poisson_mthbc(l,edge) == USERSPECIFIED .OR. Poisson_mthbc(l,edge) == MULTIPOLE_EXPANSION))) THEN !zero out connections to ancestors
                      vectorvalues(m) = vectorvalues(m)-StencilValues(p)*Info%q(ip(1),ip(2),ip(3),iPhiGas) !add boundary term
                   END IF
                END DO
                ip(l)=temp
             END DO
          END DO
       END DO
    END DO
    CALL C_StructVectorSetBoxValues(EllipticLevelObject%VariableVector, Info%mGlobal(:,1), Info%mGlobal(:,2),vectorValues, iErr)
    CALL CheckErr('C_StructVectorSetBoxValues',iErr)
    vectorvalues=reshape(Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiGas),shape(vectorvalues))
    CALL C_StructVectorSetBoxValues(EllipticLevelObject%SolutionVector, Info%mGlobal(:,1), Info%mGlobal(:,2), vectorvalues, iErr)
    CALL CheckErr('C_StructVectorSetBoxValues',iErr)

    DEALLOCATE(vectorValues)

  END SUBROUTINE PoissonVectorSetBoxValues

  !> Routine for extracting solution from lhs vector.
  !! @param n level
  !! @param Info Info object
  !! @param EllipticLevelObject Elliptic Level Object
  !! @details Handle stored in EllipticLevelObject%SolutionVector
  SUBROUTINE PoissonVectorGetBoxValues(n,Info,EllipticLevelObject)
    INTEGER :: n
    TYPE(InfoDef) :: Info
    TYPE(EllipticLevelObjectDef) :: EllipticLevelObject
    INTEGER ::nCells, mb(3,2), mbc
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: vectorvalues
    INTEGER :: iErr
    !     RETURN
    nCells=product(Info%mX(1:nDim))
    ALLOCATE(vectorvalues(nCells))

    CALL C_StructVectorGetBoxValues(EllipticLevelObject%SolutionVector, Info%mGlobal(:,1), Info%mGlobal(:,2),vectorValues, iErr)
    CALL CheckErr('C_StructVectorGetBoxValues',iErr)

    mbc=levels(Info%level)%egmbc(levels(Info%level)%step)
    mb(nDim+1:3,:)=1
    mb(1:nDim,1)=1-mbc
    mb(1:nDim,2)=Info%mX(1:nDim)+mbc
    !Update iPhiGas
    Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiGas)=reshape(vectorvalues,Info%mX)
    IF (levels(n)%dt > 0) THEN !predictive phidot
       !Update iPhiDot using old Phi stored in iPhiDot and new phi
       Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiDot)=&
            (Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiGas) - &
            Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiDot))/levels(n)%dt
    ELSE
!       Info%q(:,:,:,iPhiDot)=0
    END IF
    DEALLOCATE(vectorValues)

  END SUBROUTINE PoissonVectorGetBoxValues


  !> Self Gravity Source routine.
  !! @param Info Info Object
  SUBROUTINE PoissonGetMaxSpeed(Info)
     USE ParticleDeclarations
     USE ParticleControl
     TYPE(InfoDef) :: Info            
     INTEGER :: level
     INTEGER, DIMENSION(3,2) :: ip,ir,il
     INTEGER :: m
     level=Info%level
     ip(nDim+1:3,:)=1
     ip(1:nDim,1)=1!-levels(level)%gmbc(levels(level)%step)
     ip(1:nDim,2)=Info%mX(1:nDim)!+levels(level)%gmbc(levels(level)%step)
     ir=ip
     il=ip
     DO m=1,nDim
        ir(m,1)=ip(m,1)+1
        il(m,2)=ip(m,2)-1         
        elliptic_maxspeed(level)=max(elliptic_maxspeed(level),sqrt(nDim*&
             maxval(abs(Info%q(ir(1,1):ir(1,2),ir(2,1):ir(2,2),ir(3,1):ir(3,2),iPhiGas)-&
             Info%q(il(1,1):il(1,2),il(2,1):il(2,2),il(3,1):il(3,2),iPhiGas)))))
        ir(m,:)=ip(m,:)
        il(m,:)=ip(m,:)
     END DO
!~     write(*,*) 'elliptic_maxspeed=', elliptic_maxspeed(level)
  END SUBROUTINE PoissonGetMaxSpeed


  !> Applies analytic boundary conditions where necessary
  !! @param Info Info object
  SUBROUTINE ApplyAnalyticBC(Info, rmbc)
    ! Interface declarations  
    TYPE (InfoDef) :: Info  
    ! Internal declarations    
    INTEGER rmbc, level,dim,edge,start,ncells,i,dir
    INTEGER, DIMENSION(3,2):: lGmGlobal, ip
    level=Info%level
!    rmbc=levels(level)%egmbc(1)!levels(level)%step)
    lGmGlobal(:,1)=GmGlobal(:,1)
    lGmGlobal(:,2)=GmGlobal(:,2)*PRODUCT(levels(0:level-1)%CoarsenRatio)  
    ip=1
    DO dim=1,nDim
       DO edge=1,2
          IF (Poisson_mthbc(dim,edge) == MULTIPOLE_EXPANSION) THEN
             IF (edge == 1) THEN
                start=lGmGlobal(dim,1)-Info%mGlobal(dim,1)  !first cell on left boundary
                nCells=start-(1-rmbc)+1
             ELSE
                start=(lGmGlobal(dim,2)+1)-(Info%mGlobal(dim,1)-1) !first ghost cell on right boundary
                nCells=Info%mx(dim)+rmbc-start+1
             END IF
             DO i=1,nDim
                IF (i==dim) CYCLE
                ip(i,1)=1-rmbc
                ip(i,2)=Info%mX(i)+rmbc
             END DO
             dir=(-1)**edge !(edge*2-3)  !direction of edge (1=>-1, 2=>1)
             ip(dim,edge)=start+dir*(nCells-1)
             ip(dim,3-edge)=start           
             IF (nCells > 0) CALL SetPhi(Info, ip)
          END IF
       END DO
    END DO
  END SUBROUTINE ApplyAnalyticBC

  !> Applies physical boundary conditions where necessary
  !! @param Info Info object
  SUBROUTINE ApplyPoissonPhysicalBC(Info,rmbc)
    ! Interface declarations  
    TYPE (InfoDef) :: Info  
    ! Internal declarations    
    INTEGER rmbc, level,dim,edge,start,ncells
    INTEGER, DIMENSION(3,2):: lGmGlobal
    level=Info%level
    lGmGlobal(:,1)=GmGlobal(:,1)
    lGmGlobal(:,2)=GmGlobal(:,2)*PRODUCT(levels(0:level-1)%CoarsenRatio)  
    DO dim=1,nDim
       DO edge=1,2
          IF (Poisson_mthbc(dim,edge)==REFLECTING) THEN
             IF (edge == 1) THEN
                start=lGmGlobal(dim,1)-Info%mGlobal(dim,1)  !first cell on left boundary
                nCells=start-(1-rmbc)+1
             ELSE
                start=(lGmGlobal(dim,2)+1)-(Info%mGlobal(dim,1)-1) !first ghost cell on right boundary
                nCells=Info%mx(dim)+rmbc-start+1
             END IF
             IF (nCells > 0) CALL MirrorPhiGhost(dim,edge,start,nCells,Info)
          END IF
       END DO
    END DO
  END SUBROUTINE ApplyPoissonPhysicalBC

  !> @}

  !> @name Miscellaneous routines
  !! @{

  ! Consistency Check on Poisson boundary conditions
  SUBROUTINE Checkmthbc(mthbc)
    INTEGER, DIMENSION(:,:) :: mthbc
    INTEGER :: dim,edge
    lMultiPole=.false.
    lReflect=.false.

    IF (ANY(mthbc(1:nDim,:)==MULTIPOLE_EXPANSION)) THEN
       IF (ANY(ALL(mthbc(1:nDim,:)/=MULTIPOLE_EXPANSION, 2))) THEN
          PRINT*, 'must have at least one boundary in each dimension being multipole if any are multipole'
          STOP
       END IF
    END IF
    DO dim=1,nDim
       DO edge=1,2
          lReflect(dim) = lReflect(dim) .OR. mthbc(dim,edge) == REFLECTING !ANY(Gmthbc(dim,edge) == (/REFLECT_WALL, REFLECT_BPARALLEL, REFLECT_CYLINDRICAL/))
          lMultiPole(dim) = lMultiPole(dim) .OR. mthbc(dim,edge) == MULTIPOLE_EXPANSION
       END DO
    END DO
    lPoissonPeriodic(1:nDim)=ANY(mthbc(1:nDim,:)==PERIODIC,2)
    IF (ALL(lPoissonPeriodic(1:nDim))) lNeedMeanDensity=.true.
    IF (ANY(lPoissonPeriodic(1:nDim) .AND. (mthbc(1:nDim,1) /= mthbc(1:nDim,2)))) THEN
       write(*,*) 'Boundaries must be periodic on both sides of any given direction'
       STOP
    END IF
!    write(*,*) lMultiPole
  END SUBROUTINE Checkmthbc

  SUBROUTINE MirrorPhiGhost(dim, edge, start, nCells, Info)
    INTEGER :: dim, start, nCells, edge,dir, level, rmbc,i,j
    TYPE(InfoDef) :: Info
    INTEGER, DIMENSION(3,2) :: ip,iq,ir,is
    LOGICAL :: lFirst
    INTEGER, DIMENSION(2) :: AuxParFields
    REAL(KIND=qPREC) :: aux_psign, aux_nsign
    INTEGER, DIMENSION(4) :: ReflectVars
    INTEGER :: nReflect, MirrorFields(3), nmirror
    IF (nCells == 0) RETURN
    nMirror=2
    MirrorFields(1:2)=(/iPhiGas, iPhiDot/)
    level=Info%level
    dir=(-1)**edge !(edge*2-3)  !direction of edge (1=>-1, 2=>1)
    ip(nDim+1:3,:)=1
    iq(nDim+1:3,:)=1
    ! Stretch bounds by nCells
    DO i=1,nDim
       IF (i==dim) CYCLE
       ip(i,1)=1-nCells
       ip(i,2)=Info%mX(i)+nCells
       iq(i,:)=ip(i,:)
    END DO
    DO j=1,nCells
       iq(dim,:)=start+dir*(j-1)
       ip(dim,:)=start-dir*(j)
       Info%q(iq(1,1):iq(1,2),iq(2,1):iq(2,2),iq(3,1):iq(3,2),MirrorFields(1:nMirror))=&
            Info%q(ip(1,1):ip(1,2),ip(2,1):ip(2,2),ip(3,1):ip(3,2),MirrorFields(1:nMirror))
    END DO
  END SUBROUTINE MirrorPhiGhost


  SUBROUTINE SelfGravCorrections(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    TYPE(InfoDef), POINTER :: info
    nodelist=>Nodes(n)%p
    DO WHILE(ASSOCIATED(nodelist))
       info=>nodelist%self%info
       CALL SelfGravCorrection(info)
       nodelist=>nodelist%next
    END DO
  END SUBROUTINE SelfGravCorrections
  
  !> @}


  SUBROUTINE FFTSOLVE(Info)
    TYPE(InfoDef) :: Info
    COMPLEX(8), dimension(:,:,:), allocatable :: in ! maps to transform of density
    COMPLEX(8), dimension(:,:,:), allocatable :: phi !
    INTEGER :: i,j,k,frame,ikx,iky,ikz
    INTEGER(8) :: p1, p2
    REAL, DIMENSION(:), ALLOCATABLE :: kx,ky,kz
    REAL :: C1,dk(3)
    ALLOCATE(in(Info%mX(1),Info%mX(2),Info%mX(3)))
    ALLOCATE(phi(Info%mx(1),Info%mX(2),Info%mX(3)))
    C1=4d0*Pi*ScaleGrav/(PRODUCT(Info%mX(:))) !extra scaling factors (scale_grav is just G_ scaled to computational units)
    dk(:)=2d0*Pi/(GXBounds(1:3,2)-GxBounds(1:3,1))/(sqrt(C1))
    CALL dfftw_plan_dft_3d(p1, Info%mx(1), Info%mx(2), Info%mx(3), in, phi, FFTW_FORWARD, FFTW_ESTIMATE)  
    CALL dfftw_plan_dft_3d(p2, Info%mx(1), Info%mx(2), Info%mx(3), phi, in, FFTW_BACKWARD, FFTW_ESTIMATE) 

    in=Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),1)
    CALL dfftw_execute(p1)

    ALLOCATE (kx(Info%mX(1)),ky(Info%mX(2)),kz(Info%mX(3)))
    DO ikx = 0, Info%mX(1)-1
       kx(ikx+1)=dk(1)*REAL((mod((ikx+Info%mX(1)/2),Info%mX(1))-Info%mX(1)/2),8)
    END DO
    DO iky = 0, Info%mX(2)-1
       ky(iky+1)=dk(2)*REAL((mod((iky+Info%mX(2)/2),Info%mX(2))-Info%mX(2)/2),8)
    END DO
    DO ikz = 0, Info%mX(3)-1
       kz(ikz+1)=dk(3)*REAL((mod((ikz+Info%mX(3)/2),Info%mX(3))-Info%mX(3)/2),8)
    END DO
    DO i=1,Info%mX(1)
       DO j=1,Info%mX(2)
          DO k=1,Info%mX(3)
             phi(i,j,k)=-1d0*phi(i,j,k)/REAL(kx(i)**2+ky(j)**2+kz(k)**2)
          END DO
       END DO
    END DO
    phi(1,1,1)=0d0 !periodic bc's
    CALL dfftw_execute(p2)
    Info%q(1:Info%mX(1),1:Info%mX(2),1:Info%mX(3),iPhiGas)=REAL(in)
  END SUBROUTINE FFTSOLVE

END MODULE Poisson
