!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    bviscosity.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE BViscositySrc
  USE GlobalDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE ExplicitDeclarations
  USE PhysicsDeclarations
  USE EOS

CONTAINS

  SUBROUTINE BViscosityGridAdvance(n,stepping)
    INTEGER :: n,stepping
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist        
    IF(mpi_id==0)THEN
       PRINT*, "Advancing BV Solver on Grid..."
    END IF
    nodelist=>nodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL BViscositySource(n,node%info,stepping)
       nodelist=>nodelist%next
    END DO

  END SUBROUTINE BViscosityGridAdvance

  SUBROUTINE BViscosityGridGetMaxSpeed(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       CALL BViscosityGetMaxSpeed(node%info)
       nodelist=>nodelist%next 
    END DO

  END SUBROUTINE BViscosityGridGetMaxSpeed


  SUBROUTINE BViscositySource(n,Info,stepping)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k,stepping,n,mx,my,mz,mbc,zmbc,msx(3,2),msy(3,2),msz(3,2)
    INTEGER,ALLOCATABLE,DIMENSION(:) :: iflux
    REAL(KIND=qPREC) :: dx,dt
    REAL(KIND=qPREC),ALLOCATABLE,DIMENSION(:) :: vvec,vdq
    REAL(KIND=qPREC),ALLOCATABLE,DIMENSION(:,:) :: bgvx,bgvy,bgvz
    REAL(KIND=qPREC),ALLOCATABLE,DIMENSION(:,:,:,:) :: xflux,yflux,zflux
    
    mx=Info%mX(1);my=Info%mX(2);mz=Info%mX(3)
    mbc=levels(n)%ambc(levels(n)%step)-stepping
    SELECT CASE(nDim)
    CASE(2)
       zmbc=0
    CASE(3)
       zmbc=mbc
    END SELECT
    dx=levels(n)%dx; dt=levels(n)%dt
    
    msx=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc+1,my+mbc,mz+zmbc/),(/3,2/))
    msy=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc,my+mbc+1,mz+zmbc/),(/3,2/))
    IF(nDim==3) msz=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc,my+mbc,mz+zmbc+1/),(/3,2/))
    
    ALLOCATE(iflux(1:nDim+1))
    iflux(1:nDim)=imom(1:nDim);iflux(nDim+1)=iE
    
    ALLOCATE(bgvx(1:ndim,1:ndim)); ALLOCATE(bgvy(1:ndim,1:ndim))
    ALLOCATE(vvec(1:ndim)); ALLOCATE(vdq(1:ndim+1))
    ALLOCATE(xflux(msx(1,1):msx(1,2),msx(2,1):msx(2,2),msx(3,1):msx(3,2),1:nDim+1))
    ALLOCATE(yflux(msy(1,1):msy(1,2),msy(2,1):msy(2,2),msy(3,1):msy(3,2),1:nDim+1))
    IF(nDim==3) ALLOCATE(bgvz(1:ndim,1:ndim))
    IF(nDim==3) ALLOCATE(zflux(msz(1,1):msz(1,2),msz(2,1):msz(2,2),msz(3,1):msz(3,2),1:nDim+1))
    
    DO k=msx(3,1),msx(3,2); DO j=msx(2,1),msx(2,2); DO i=msx(1,1),msx(1,2)
       IF(nDim==3)THEN
          CALL bvTensor3D(n,Info,i,j,k,1,bgvx)
       ELSE
          CALL bvTensor2D(n,Info,i,j,k,1,bgvx)
       END IF
       vvec(1:nDim)=0.5*(Info%q(i-1,j,k,imom(1:nDim))/Info%q(i-1,j,k,1)+Info%q(i,j,k,imom(1:nDim))/Info%q(i,j,k,1))
       xflux(i,j,k,1:nDim)=bgvx(1,:)
       xflux(i,j,k,nDim+1)=DOT_PRODUCT(bgvx(:,1),vvec)
    END DO; END DO; END DO
    
    DO k=msy(3,1),msy(3,2); DO j=msy(2,1),msy(2,2); DO i=msy(1,1),msy(1,2)
       IF(nDim==3)THEN
          CALL bvTensor3D(n,Info,i,j,k,2,bgvy)
       ELSE
          CALL bvTensor2D(n,Info,i,j,k,2,bgvy)
       END IF
       vvec(1:nDim)=0.5*(Info%q(i,j-1,k,imom(1:nDim))/Info%q(i,j-1,k,1)+Info%q(i,j,k,imom(1:nDim))/Info%q(i,j,k,1))
       yflux(i,j,k,1:nDim)=bgvy(2,:)
       yflux(i,j,k,nDim+1)=DOT_PRODUCT(bgvy(:,2),vvec)
    END DO; END DO; END DO
    
    IF(nDim==3) THEN
       DO k=msz(3,1),msz(3,2); DO j=msz(2,1),msz(2,2); DO i=msz(1,1),msz(1,2)
          IF(nDim==3)THEN
             CALL bvTensor3D(n,Info,i,j,k,3,bgvz)
          ELSE
             CALL bvTensor2D(n,Info,i,j,k,3,bgvz)
          END IF
       END DO; END DO; END DO
       vvec(1:nDim)=0.5*(Info%q(i,j,k-1,imom(1:nDim))/Info%q(i,j,k-1,1)+Info%q(i,j,k,imom(1:nDim))/Info%q(i,j,k,1))
       zflux(i,j,k,1:nDim)=bgvz(3,:)
       zflux(i,j,k,nDim+1)=DOT_PRODUCT(bgvz(:,3),vvec)
    END IF
    
    DEALLOCATE(bgvx);DEALLOCATE(bgvy)
    DEALLOCATE(vvec)

    IF(nDim==3) THEN
       DEALLOCATE(bgvz)
    END IF
    
    DO k=1-zmbc,mz+zmbc; DO j=1-mbc,my+mbc; DO i=1-mbc,mx+mbc
       vdq(:)=-dt*(xflux(i+1,j,k,:)-xflux(i,j,k,:)+yflux(i,j+1,k,:)-yflux(i,j,k,:))/dx
       IF(nDim==3) vdq(:)=vdq(:)-dt*(zflux(i,j,k+1,:)-zflux(i,j,k,:))/dx
       Info%q(i,j,k,imom(1:nDim))=vdq(1:nDim)+Info%q(i,j,k,imom(1:nDim))
       Info%q(i,j,k,iE)=vdq(nDim+1)+Info%q(i,j,k,iE)
    END DO; END DO; END DO
    
    DEALLOCATE(vdq)
    
    CALL storefixupfluxes(Info,msx,1,xflux,iflux)
    CALL storefixupfluxes(Info,msy,2,yflux,iflux)
    IF(nDim==3) CALL storefixupfluxes(Info,msz,3,zflux,iflux)
    
    DEALLOCATE(xflux); DEALLOCATE(yflux)
    IF(nDim==3) DEALLOCATE(zflux)
    DEALLOCATE(iflux)
    
  END SUBROUTINE BViscositySource
  
  SUBROUTINE bvTensor2D(n,Info,i,j,k,dir,bv)
    TYPE(InfoDef) :: Info
    INTEGER:: i,j,k,l,dir
    REAL(KIND=qPREC) :: bvmag,ath,templ,tempr
    REAL(KIND=qPREC), DIMENSION(2,2) :: ii,bb,dvel,bv
    
    ath=1d0/3d0; ii=RESHAPE((/1,0,0,1/),(/2,2/));
    dx=levels(n)%dx
    
    CALL bbTensor2D(Info,i,j,k,dir,bb)
    CALL VelocityDiffTensor2D(Info,i,j,k,dir,dvel); dvel=dvel/dx
    
    bv=bb-0.3*ii
    bvmag=(bb(1,1)-ath)*dvel(1,1)+(bb(2,2)-ath)*dvel(2,2)+bb(1,2)*(dvel(1,2)+dvel(2,1))
    CALL getT(Info,i,j,k,tempr)
    SELECT CASE(dir)
    CASE(1)
       CALL getT(Info,i-1,j,k,templ)
    CASE(2)
       CALL getT(Info,i,j-1,k,templ)
    END SELECT
    
    bvmag=-3d0*bviscosity*(0.5*(templ+tempr))**2.5*bvmag
    bv=bvmag*bv
    
  END SUBROUTINE bvTensor2D
  
  
  SUBROUTINE bvTensor3D(n,Info,i,j,k,dir,bv)
    TYPE(InfoDef) :: Info
    INTEGER :: n,i,j,k,dir
    REAL(KIND=qPREC),DIMENSION(3,3) :: bv
    
    SELECT CASE(dir)
    CASE(1)
    CASE(2)
    CASE(3)
    END SELECT


  END SUBROUTINE bvTensor3D


  SUBROUTINE bbTensor2D(Info,i,j,k,dir,bb)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k,dir
    REAL(KIND=qPREC) :: bbx,bby,bstr
    REAL(KIND=qPREC), DIMENSION(2,2) :: bb
    
    bb(:,:)=RESHAPE((/1d0/3d0,1d0,1d0,1d0/3d0/),(/2,2/))
    IF(VisType==1) RETURN
    SELECT CASE(dir)
    CASE(1)
       bbx=Info%aux(i,j,k,1)
       bby=0.25*(Info%aux(i-1,j,k,2)+Info%aux(i,j,k,2)+Info%aux(i-1,j+1,k,2)+Info%aux(i,j+1,k,2))
    CASE(2)
       bbx=0.25*(Info%aux(i,j-1,k,1)+Info%aux(i,j,k,1)+Info%aux(i+1,j-1,k,1)+Info%aux(i+1,j,k,1))
       bby=Info%aux(i,j,k,2)
    END SELECT
    bstr=bbx**2+bby**2
    IF(bstr==0d0) RETURN
    bb(1,1)=bbx**2/bstr; bb(2,2)=bby**2/bstr
    bb(1,2)=bbx*bby/bstr; bb(2,1)=bb(1,2)
    
  END SUBROUTINE bbTensor2D


  SUBROUTINE VelocityDiffTensor2D(Info,i,j,k,dir,dvel)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k,l,dir
    REAL(KIND=qPREC) :: vl(1:ndim), vr(1:ndim)
    REAL(KIND=qPREC),DIMENSION(2,2) :: dvel
    
    SELECT CASE(dir)
    CASE(1)
       dvel(1,1:ndim)=Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1)-Info%q(i-1,j,k,imom(1:ndim))/Info%q(i-1,j,k,1)
       vl(1:ndim)=0.5*(Info%q(i-1,j,k,imom(1:ndim))/Info%q(i-1,j,k,1)-Info%q(i-1,j-1,k,imom(1:ndim))/Info%q(i-1,j-1,k,1)+&
            Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1)-Info%q(i,j-1,k,imom(1:ndim))/Info%q(i,j-1,k,1))
       vr(1:ndim)=0.5*(Info%q(i-1,j+1,k,imom(1:ndim))/Info%q(i-1,j+1,k,1)-Info%q(i-1,j,k,imom(1:ndim))/Info%q(i-1,j,k,1)+&
            Info%q(i,j+1,k,imom(1:ndim))/Info%q(i,j+1,k,1)-Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1))
       DO l=1,ndim
          dvel(2,l)=MCLimiter(vl(l),vr(l))
       END DO
    CASE(2)
       vl(1:ndim)=0.5*(Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1)-Info%q(i-1,j,k,imom(1:ndim))/Info%q(i-1,j,k,1)+&
            Info%q(i,j-1,k,imom(1:ndim))/Info%q(i,j-1,k,1)-Info%q(i-1,j-1,k,imom(1:ndim))/Info%q(i-1,j-1,k,1))
       vr(1:ndim)=0.5*(Info%q(i+1,j,k,imom(1:ndim))/Info%q(i+1,j,k,1)-Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1)+&
            Info%q(i+1,j-1,k,imom(1:ndim))/Info%q(i+1,j-1,k,1)-Info%q(i,j-1,k,imom(1:ndim))/Info%q(i,j-1,k,1))
       DO l=1,ndim
          dvel(1,l)=MCLimiter(vl(l),vr(l))
       END DO
       dvel(2,1:ndim)=(Info%q(i,j,k,imom(1:ndim))/Info%q(i,j,k,1)-Info%q(i,j-1,k,imom(1:ndim))/Info%q(i,j-1,k,1))
    END SELECT
  END SUBROUTINE VelocityDiffTensor2D


  SUBROUTINE VelocityDiffTensor3D(Info,i,j,k,dir,dvdx)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k,dir
    REAL(KIND=qPREC),DIMENSION(3,3) :: dvdx
    
    SELECT CASE(dir)
    CASE(1)
    CASE(2)
    CASE(3)
    END SELECT
    

  END SUBROUTINE VelocityDiffTensor3D


  REAL FUNCTION MCLimiter(xl,xr)
    REAL(KIND=qPREC) :: xl, xr
    
    IF((xl+xr==0d0).or.(xl*xr<0d0)) THEN
       MCLimiter=0d0
       RETURN
    END IF

    MCLimiter = 2d0*xl*xr/(xl+xr)
    RETURN
    
  END FUNCTION MCLimiter
  

  SUBROUTINE getT(Info,i,j,k,temp)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k
    REAL(KIND=qPREC) :: temp

    temp=(gamma-1d0)*(Info%q(i,j,k,iE)-0.5*SUM(Info%q(i,j,k,imom(1:nDim))**2)/Info%q(i,j,k,1)&
         -0.5*SUM(Info%q(i,j,k,iBx:iBx+nDim-1)**2))/Info%q(i,j,k,1)
    
  END SUBROUTINE getT


  SUBROUTINE BViscosityGetMaxSpeed(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: level, m
    REAL(KIND=qPREC) :: dx
    
    level=Info%level
    dx=levels(level)%dx
    
    explicit_maxspeed(level) = max(explicit_maxspeed(level),2d0*resistivity/dx)

  END SUBROUTINE BViscosityGetMaxSpeed



END MODULE BViscositySrc



