!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    conductive.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE ConductiveSrc
   USE GlobalDeclarations
   USE TreeDeclarations
   USE DataDeclarations
   USE ExplicitDeclarations
   USE PhysicsDeclarations
   USE EOS

   REAL(KIND=qPREC), DIMENSION(3,3), PARAMETER :: az = RESHAPE((/4.0,3.0,2.0,&
      0.765270225971218, 0.724613694800218, 0.647648420009554, &
      0.084729774028782, 0.125386305199781, 0.20235157990446/),(/3,3/))

 CONTAINS   
   
   SUBROUTINE ConductiveGridAdvance(n,stepping)
     INTEGER :: n,stepping
     TYPE(NodeDef), POINTER :: node
     TYPE(NodeDefList), POINTER :: nodelist        
     IF(mpi_id==0)THEN
        PRINT*, "Advancing Conductive Solver on Grid..."
     END IF
     nodelist=>nodes(n)%p
     DO WHILE (associated(nodelist))
        node=>nodelist%self
        CALL ConductiveSource(n,node%info,stepping)
        nodelist=>nodelist%next
     END DO

   END SUBROUTINE ConductiveGridAdvance

   SUBROUTINE ConductiveGridGetMaxSpeed(n)
     INTEGER :: n
     TYPE(NodeDef), POINTER :: node
     TYPE(NodeDefList), POINTER :: nodelist
     
     nodelist=>Nodes(n)%p 
     DO WHILE (associated(nodelist))
        node=>nodelist%self 
        CALL ConductiveGetMaxSpeed(node%info)
        nodelist=>nodelist%next 
     END DO
     
   END SUBROUTINE ConductiveGridGetMaxSpeed


   SUBROUTINE ConductiveSource(n,Info,stepping)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,stepping,n,mx,my,mz,mbc,zmbc
     INTEGER :: msx(3,2),msy(3,2),msz(3,2)
     INTEGER :: efield(1)
     LOGICAL :: ndim3
     REAL(KIND=qPREC) :: dx,dt,deltaT,chic,chir
     REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:,:) :: qx, qy, qz
     REAL(KIND=qPrec) :: fb(3), diT(3)
     mx=Info%mX(1);my=Info%mX(2);mz=Info%mX(3)
     mbc=levels(n)%ambc(levels(n)%step)-stepping
     ndim3=.false.
     SELECT CASE(nDim)
     CASE(2)
        zmbc=0
     CASE(3)
        zmbc=mbc; ndim3=.true.
     END SELECT
     dx=levels(n)%dx; dt=levels(n)%dt; efield=iE
     
     msx=reshape((/1-mbc,mx+mbc+1,1-mbc,my+mbc,1-zmbc,mz+zmbc/),(/3,2/))
     msy=reshape((/1-mbc,mx+mbc,1-mbc,my+mbc+1,1-zmbc,mz+zmbc/),(/3,2/))
     IF(ndim3) msz=reshape((/1-mbc,mx+mbc,1-mbc,my+mbc,1-zmbc,mz+zmbc+1/),(/3,2/))
     
     
     ALLOCATE(qx(1-mbc:mx+mbc+1,1-mbc:my+mbc,1-zmbc:mz+zmbc,1))
     ALLOCATE(qy(1-mbc:mx+mbc,1-mbc:my+mbc+1,1-zmbc:mz+zmbc,1))
     IF(ndim3) ALLOCATE(qz(1-mbc:mx+mbc,1-mbc:my+mbc,1-zmbc:mz+zmbc+1,1))

     DO i=1-mbc,mx+mbc+1; DO j=1-mbc,my+mbc; DO k=1-zmbc,mz+zmbc
        CALL FindFacedTdx(Info,diT,i,j,k,1)
        CALL FindFaceB(Info,fb,i,j,k,1)
        CALL FindConductivity(Info,chic,chir,i,j,k,1)
        !PRINT*, "qx", i,j,k,diT(:)
        qx(i,j,k,1) = -chic*fb(1)*(fb(1)*diT(1)+fb(2)*diT(2))-chir*diT(1)
        IF(ndim3) qx(i,j,k,1)=qx(i,j,k,1)-chic*fb(1)*fb(3)*diT(3)
        IF(lThermalFluxLimiter) CALL FindSaturationFlux(Info,i,j,k,fb,qx(i,j,k,1),1)
        IF((lIsoDiff).and.(ConType<2)) qx(i,j,k,1) = qx(i,j,k,1)-chic*isodiffratio*diT(1)
     END DO; END DO; END DO
     DO i=1-mbc,mx+mbc; DO j=1-mbc,my+mbc+1; DO k=1-zmbc,mz+zmbc
        CALL FindFacedTdx(Info,diT,i,j,k,2)
        CALL FindFaceB(Info,fb,i,j,k,2)
        CALL FindConductivity(Info,chic,chir,i,j,k,2)
        !PRINT*, "qy", i,j,k,diT(:)
        qy(i,j,k,1) = -chic*fb(2)*(fb(1)*diT(1)+fb(2)*diT(2))-chir*diT(2)
        IF(ndim3) qy(i,j,k,1)=qy(i,j,k,1)-chic*fb(2)*fb(3)*diT(3)
        IF(lThermalFluxLimiter) CALL FindSaturationFlux(Info,i,j,k,fb,qy(i,j,k,1),2)
        IF((lIsoDiff).and.(ConType<2)) qy(i,j,k,1) = qy(i,j,k,1)-chic*isodiffratio*diT(2)
     END DO; END DO; END DO
     IF(ndim3) THEN
        DO i=1-mbc,mx+mbc; DO j=1-mbc,my+mbc; DO k=1-zmbc,mz+zmbc+1
           CALL FindFacedTdx(Info,diT,i,j,k,3)
           CALL FindFaceB(Info,fb,i,j,k,3)
           CALL FindConductivity(Info,chic,chir,i,j,k,3)
           qz(i,j,k,1) = -chic*fb(3)*(fb(1)*diT(1)+fb(2)*diT(2)+fb(3)*diT(3))-chir*diT(3)
           IF(lThermalFluxLimiter) CALL FindSaturationFlux(Info,i,j,k,fb,qz(i,j,k,1),3)
           IF((lIsoDiff).and.(ConType<2)) qz(i,j,k,1) = qz(i,j,k,1)-chic*isodiffratio*diT(3)
        END DO; END DO; END DO
     END IF
     
     CALL storefixupfluxes(Info,msx,1,qx,efield)
     CALL storefixupfluxes(Info,msy,2,qy,efield)
     IF(ndim3) CALL storefixupfluxes(Info,msz,3,qz,efield)
     
     DO i=1-mbc,mx+mbc; DO j=1-mbc,my+mbc; DO k=1-zmbc,mz+zmbc
        deltaT=(dt/dx)*(qx(i+1,j,k,1)-qx(i,j,k,1)+qy(i,j+1,k,1)-qy(i,j,k,1))
        IF(ndim3) deltaT=deltaT+(dt/dx)*(qz(i,j,k+1,1)-qz(i,j,k,1))
        !PRINT*, "Ratio of Change is: ", i,j,(Info%q(i,j,k,1)*deltaT/(gamma-1d0))/Info%q(i,j,k,iE)
        Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-Info%q(i,j,k,1)*deltaT/(gamma-1d0)
     END DO; END DO; END DO

    ! DO i=1,mx; DO j=1,my
    !    PRINT*, i,j,qx(i,j,1,1),qy(i,j,1,1)
    ! END DO; END DO
     
     DEALLOCATE(qx); DEALLOCATE(qy)
     IF(ndim3) DEALLOCATE(qz)
     
   END SUBROUTINE ConductiveSource

   SUBROUTINE FindFacedTdx(Info,diT,i,j,k,m)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,m
     REAL(KIND=qPREC) :: diT(3)
     REAL(KIND=qPREC) :: mT(-1:1,-1:1,-1:1)
     
     CALL FindBodyT(Info,mT,i,j,k)
     diT=0d0
     !PRINT*
     !PRINT*, i,j,"mT is:"
     !PRINT*, mT(:,1,0)
     !PRINT*, mT(:,0,0)
     !PRINT*, mT(:,-1,0)
     !PRINT*

     SELECT CASE(m)
     CASE(1)
        diT(1)=mT(0,0,0)-mT(-1,0,0)
        diT(2)=0.25*(mT(0,1,0)-mT(0,-1,0)+mT(-1,1,0)-mT(-1,-1,0))
        IF(nDim==3) diT(3)=0.25*(mT(0,0,1)-mT(0,0,-1)+mT(-1,0,1)-mT(-1,0,-1))
     CASE(2)
        diT(1)=0.25*(mT(1,0,0)-mT(-1,0,0)+mT(1,-1,0)-mT(-1,-1,0))
        diT(2)=mT(0,0,0)-mT(0,-1,0)
        IF(nDim==3) diT(3)=0.25*(mT(0,0,1)-mT(0,0,-1)+mT(0,-1,1)-mT(0,-1,-1))
     CASE(3)
        diT(1)=0.25*(mT(1,0,0)-mT(-1,0,0)+mT(1,0,-1)-mT(-1,0,-1))
        diT(2)=0.25*(mT(0,1,0)-mT(0,-1,0)+mT(0,1,-1)-mT(0,-1,-1))
        IF(nDim==3) diT(3)=mT(0,0,0)-mT(0,0,-1)
     END SELECT
     
   END SUBROUTINE FindFacedTdx
   
   SUBROUTINE FindFaceB(Info,fb,i,j,k,m)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,m
     REAL(KIND=qPREC) :: ff(3),fl,fb(3)
     ff=0d0;fb=0d0;fl=0d0
     SELECT CASE(m)
     CASE(1)
        ff(1)=Info%aux(i,j,k,1)
        ff(2)=0.25*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2)+Info%aux(i-1,j,k,2)+Info%aux(i-1,j+1,k,2))
        IF(nDim==3) ff(3)=0.25*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3)+Info%aux(i-1,j,k,3)+Info%aux(i-1,j,k+1,3))
     CASE(2)
        ff(1)=0.25*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1)+Info%aux(i,j-1,k,1)+Info%aux(i+1,j-1,k,1))
        ff(2)=Info%aux(i,j,k,2)
        if(nDim==3) ff(3)=0.25*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3)+Info%aux(i,j-1,k,3)+Info%aux(i,j-1,k+1,3))
     CASE(3)
        ff(1)=0.25*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1)+Info%aux(i,j,k-1,1)+Info%aux(i+1,j,k-1,1))
        ff(2)=0.25*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2)+Info%aux(i,j,k-1,2)+Info%aux(i,j+1,k-1,2))
        IF(nDim==3) ff(3)=Info%aux(i,j,k,3)
     END SELECT

     fl=SQRT(SUM(ff(:)**2))
     IF(fl > 0d0) THEN
        fb(:)=ff(:)/fl
     ELSE
        fb(m)=1d0
     END IF

   END SUBROUTINE FindFaceB

   SUBROUTINE FindBodyT(Info,mT,i,j,k)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,l,m,n
     REAL(KIND=qPREC) :: mT(-1:1,-1:1,-1:1)
     
     mT=0d0

     IF(nDim==2)THEN
        DO l=-1,1; DO m=-1,1
           mT(l,m,0)=(gamma-1d0)*(Info%q(i+l,j+m,k,iE)-0.5*SUM(Info%q(i+l,j+m,k,&
                imom(1:ndim))**2)/Info%q(i+l,j+m,k,1)-0.5*SUM(Info%q(i+l,j+m,k,iBx:iBx+nDim-1)**2))/Info%q(i+l,j+m,k,1)
        END DO; END DO
     ELSE IF(nDim==3) THEN
        DO l=-1,1; DO m=-1,1; DO n=-1,1
           mT(l,m,n)=(gamma-1d0)*(Info%q(i+l,j+m,k+n,iE)-0.5*SUM(Info%q(i+l,j+m,k+n,&
                imom(1:ndim))**2)/Info%q(i+l,j+m,k+n,1)-0.5*SUM(Info%q(i+l,j+m,k+n,iBx:iBx+nDim-1)**2))/Info%q(i+l,j+m,k+n,1)
        END DO; END DO; END DO
     ELSE
        PRINT*, "Conductive Solver is 2D, 3D only!"
        STOP
     END IF
   END SUBROUTINE FindBodyT
   
   SUBROUTINE FindConductivity(Info,chic,chir,i,j,k,dir)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,ii,jj,kk,dir,shift(3)
     REAL(KIND=qPREC) :: rho,temp1,temp2,temp,conpara,concross,magp,chic,chir
     
     chic=0d0;chir=0d0
     IF(ConType<2)THEN
        chic=conductivity
        RETURN
     END IF

     shift=0;shift(dir)=-1
     ii=i+shift(1);jj=j+shift(2);kk=k+shift(3)
     
     rho=0.5*(Info%q(i,j,k,1)+Info%q(ii,jj,kk,1))
     temp1=(gamma-1d0)*(Info%q(i,j,k,iE)-0.5*SUM(Info%q(i,j,k,&
          imom(1:ndim))**2)/Info%q(i,j,k,1)-0.5*SUM(Info%q(i,j,k,iBx:iBx+nDim-1)**2))/Info%q(i,j,k,1)
     temp2=(gamma-1d0)*(Info%q(ii,jj,kk,iE)-0.5*SUM(Info%q(ii,jj,kk,&
          imom(1:ndim))**2)/Info%q(ii,jj,kk,1)-0.5*SUM(Info%q(ii,jj,kk,iBx:iBx+nDim-1)**2))/Info%q(ii,jj,kk,1)
     temp=0.5*(temp1+temp2)
     
     magp=0.25*(Info%q(i,j,k,iBx)+Info%q(ii,jj,kk,iBx))**2+&
          0.25*(Info%q(i,j,k,iBy)+Info%q(ii,jj,kk,iBy))**2+&
          0.25*(Info%q(i,j,k,iBz)+Info%q(ii,jj,kk,iBz))**2
     
     conpara=SpitzerConductivity*temp**2.5
     concross=CrossConductivity*rho**2/(temp**0.5*magp)
     
     chir=concross; chic=conpara-concross
     
   END SUBROUTINE FindConductivity
   
   SUBROUTINE FindSaturationFlux(Info,i,j,k,fb,qf,dir)
     TYPE(InfoDef) :: Info
     INTEGER :: i,j,k,ii,jj,kk,dir,shift(3)
     REAL(KIND=qPREC) :: rho,temp1,temp2,temp,qf,qs,phi,fb(3)
     
     shift=0;shift(dir)=-1
     ii=i+shift(1);jj=j+shift(2);kk=k+shift(3)
     
     phi=0.3
     
     rho=0.5*(Info%q(i,j,k,1)+Info%q(ii,jj,kk,1))
     temp1=(gamma-1d0)*(Info%q(i,j,k,iE)-0.5*SUM(Info%q(i,j,k,&
          imom(1:ndim))**2)/Info%q(i,j,k,1)-0.5*SUM(Info%q(i,j,k,iBx:iBx+nDim-1)**2))/Info%q(i,j,k,1)
     temp2=(gamma-1d0)*(Info%q(ii,jj,kk,iE)-0.5*SUM(Info%q(ii,jj,kk,&
          imom(1:ndim))**2)/Info%q(ii,jj,kk,1)-0.5*SUM(Info%q(ii,jj,kk,iBx:iBx+nDim-1)**2))/Info%q(ii,jj,kk,1)
     temp=0.5*(temp1+temp2)
     qs=5d0*fb(dir)*phi*rho*(gamma*temp)**1.5
     
     CALL ThermalFluxLimiter(qs,qf)
     
   END SUBROUTINE FindSaturationFlux

   SUBROUTINE ThermalFluxLimiter(s,r)
     INTEGER :: ln
     REAL(KIND=qPREC) :: s,r,x
     
     ln=1; x=abs(r/s)

     IF(x.gt.az(2,ln)) x=(x+az(3,ln))/(1d0+(x+az(3,ln))**az(1,ln))**(1d0/az(1,ln))

     r=SIGN(x*s,r)

   END SUBROUTINE ThermalFluxLimiter

   SUBROUTINE ConductiveGetMaxSpeed(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: level, m
    REAL(KIND=qPREC) :: dx
    
    level=Info%level
    dx=levels(level)%dx
    
    explicit_maxspeed(level) = max(explicit_maxspeed(level),2d0*conductivity/dx)

   END SUBROUTINE ConductiveGetMaxSpeed

 END MODULE ConductiveSrc










