!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    resistive.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE ResistiveSrc
  USE GlobalDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE ExplicitDeclarations
  USE PhysicsDeclarations
  USE EOS

CONTAINS

  SUBROUTINE ResistiveGridAdvance(n,stepping)
    INTEGER :: n,stepping
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist        
    !alternative solution: REAL ALLOCATABLE :: q,aux
    IF(MPI_ID==0)   PRINT*, "Advancing Resistive Solver on Grid..."
    nodelist=>nodes(n)%p
    DO WHILE (associated(nodelist))
       node=>nodelist%self
       CALL ResistiveSource(n,node%info,stepping)
       !alternative solution: store q and aux
       nodelist=>nodelist%next
    END DO

  END SUBROUTINE ResistiveGridAdvance

  SUBROUTINE ResistiveGridGetMaxSpeed(n)
    INTEGER :: n
    TYPE(NodeDef), POINTER :: node
    TYPE(NodeDefList), POINTER :: nodelist
    
    nodelist=>Nodes(n)%p 
    DO WHILE (associated(nodelist))
       node=>nodelist%self 
       CALL ResistiveGetMaxSpeed(node%info)
       nodelist=>nodelist%next 
    END DO

  END SUBROUTINE ResistiveGridGetMaxSpeed

  SUBROUTINE ResistiveSource(n,Info,stepping)
    TYPE(InfoDef) :: Info
    LOGICAL :: consist_en=.true.
    INTEGER :: n
    INTEGER :: i,j,k,z_off,stepping,mx,my,mz,mbc,zmbc
    INTEGER :: msx(3,2),msy(3,2),msz(3,2), mjx(3,2), mjy(3,2), mjz(3,2)
    INTEGER, ALLOCATABLE, DIMENSION(:) :: iflux,bflux
    REAL(KIND=qPREC) :: dx,dt,check_en,off_en
    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: new_b
    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:) :: jx, jy, jz
    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:,:) :: xflux, yflux, zflux
    REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:,:) :: fjx, fjy

    mx=Info%mX(1);my=Info%mX(2);mz=Info%mX(3)
 
    mbc=levels(n)%ambc(levels(n)%step)-stepping

    IF(MPI_ID==0) THEN
       PRINT*, "level =",n,"mbc =",mbc
    END IF
        
    SELECT CASE(nDim)
    CASE(2)
       zmbc=0;z_off=0
    CASE(3)
       zmbc=mbc;z_off=1
    END SELECT
    dx=levels(n)%dx; dt=levels(n)%dt
    
    ALLOCATE(iflux(1))
    iflux(:)=iE

    ! bounds for edge centered currents
    mjx=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc,my+mbc+1,mz+zmbc+z_off/),(/3,2/))
    mjy=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc+1,my+mbc,mz+zmbc+z_off/),(/3,2/))
    mjz=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc+1,my+mbc+1,mz+zmbc/),(/3,2/))

    ALLOCATE(jx(mjx(1,1):mjx(1,2),mjx(2,1):mjx(2,2),mjx(3,1):mjx(3,2)))
    ALLOCATE(jy(mjy(1,1):mjy(1,2),mjy(2,1):mjy(2,2),mjy(3,1):mjy(3,2)))
    ALLOCATE(jz(mjz(1,1):mjz(1,2),mjz(2,1):mjz(2,2),mjz(3,1):mjz(3,2)))

    jx=resistivity;jy=resistivity;jz=resistivity
    ! if resistivity is not constant
    IF(ResType==2)THEN
       DO i=mjx(1,1),mjx(1,2); DO j=mjx(2,1),mjx(2,2); DO k=mjx(3,1),mjx(3,2)
          CALL RealResistivity(Info,i,j,k,jx(i,j,k),dx,1)
       END DO; END DO; END DO
       DO i=mjy(1,1),mjy(1,2); DO j=mjy(2,1),mjy(2,2); DO k=mjy(3,1),mjy(3,2)
          CALL RealResistivity(Info,i,j,k,jy(i,j,k),dx,2)
       END DO; END DO; END DO
       DO i=mjz(1,1),mjz(1,2); DO j=mjz(2,1),mjz(2,2); DO k=mjz(3,1),mjz(3,2)
          CALL RealResistivity(Info,i,j,k,jz(i,j,k),dx,3)
       END DO; END DO; END DO
    END IF
   
    ! j = curl B
    DO i=mjx(1,1),mjx(1,2); DO j=mjx(2,1),mjx(2,2); DO k=mjx(3,1),mjx(3,2)
       IF(nDim==2)THEN
          jx(i,j,k) = -jx(i,j,k)*(Info%q(i,j-1,k,iBz)-Info%q(i,j,k,iBz))/dx
       ELSE
          jx(i,j,k) = -jx(i,j,k)*((Info%aux(i,j,k,2)-Info%aux(i,j,k-1,2))+(Info%aux(i,j-1,k,3)-Info%aux(i,j,k,3)))/dx
       END IF
       !Info%emf(i,j,k,iEx) = Info%emf(i,j,k,iEx)+jx(i,j,k)*dt/dx
    END DO; END DO; END DO

    DO i=mjy(1,1),mjy(1,2); DO j=mjy(2,1),mjy(2,2); DO k=mjy(3,1),mjy(3,2)
       IF(nDim==2)THEN
          jy(i,j,k) = -jy(i,j,k)*(Info%q(i,j,k,iBz)-Info%q(i-1,j,k,iBz))/dx
       ELSE
          jy(i,j,k) = -jy(i,j,k)*((Info%aux(i,j,k,3)-Info%aux(i-1,j,k,3))+(Info%aux(i,j,k-1,1)-Info%aux(i,j,k,1)))/dx
       END IF
       !Info%emf(i,j,k,iEy) = Info%emf(i,j,k,iEy)+jy(i,j,k)*dt/dx
    END DO; END DO; END DO

    DO i=mjz(1,1),mjz(1,2); DO j=mjz(2,1),mjz(2,2); DO k=mjz(3,1),mjz(3,2)
       jz(i,j,k) = -jz(i,j,k)*((Info%aux(i,j,k,1)-Info%aux(i,j-1,k,1))+(Info%aux(i-1,j,k,2)-Info%aux(i,j,k,2)))/dx
       !Info%emf(i,j,k,iEz) = Info%emf(i,j,k,iEz)+jz(i,j,k)*dt/dx
    END DO; END DO; END DO

    ! store x,y emfs.
    IF(nDim==2)THEN
       ALLOCATE(fjx(mjy(1,1):mjy(1,2),mjy(2,1):mjy(2,2),mjy(3,1):mjy(3,2),1))
       ALLOCATE(fjy(mjx(1,1):mjx(1,2),mjx(2,1):mjx(2,2),mjx(3,1):mjx(3,2),1))
       ALLOCATE(bflux(1))
       bflux(:)=iBz
       fjx(:,:,:,1)=jy(:,:,:)*dt/dx
       fjy(:,:,:,1)=-jx(:,:,:)*dt/dx
       CALL storefixupfluxes(Info,mjy,1,fjx,bflux)
       CALL storefixupfluxes(Info,mjx,2,fjy,bflux)
       DEALLOCATE(bflux)
       DEALLOCATE(fjx); DEALLOCATE(fjy)
    ELSE
       CALL StoreEmfs(Info,mjx,1,jx*dt/dx)
       CALL StoreEmfs(Info,mjy,2,jy*dt/dx)
    END IF

    ! store z emf. The same for 2D and 3D
    CALL StoreEmfs(Info,mjz,3,jz*dt/dx)

    ! bounds for face centered fluxes
    msx=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc+1,my+mbc,mz+zmbc/),(/3,2/))
    msy=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc,my+mbc+1,mz+zmbc/),(/3,2/))
    msz=reshape((/1-mbc,1-mbc,1-zmbc,mx+mbc,my+mbc,mz+zmbc+z_off/),(/3,2/))

    ALLOCATE(xflux(msx(1,1):msx(1,2),msx(2,1):msx(2,2),msx(3,1):msx(3,2),1))
    ALLOCATE(yflux(msy(1,1):msy(1,2),msy(2,1):msy(2,2),msy(3,1):msy(3,2),1))
    IF(nDim==3) ALLOCATE(zflux(msz(1,1):msz(1,2),msz(2,1):msz(2,2),msz(3,1):msz(3,2),1))

    DO i=msx(1,1),msx(1,2); DO j=msx(2,1),msx(2,2); DO k=msx(3,1),msx(3,2)
       IF(nDim==2)THEN
          Info%aux(i,j,k,1) = Info%aux(i,j,k,1)+(jz(i,j,k)-jz(i,j+1,k))*dt/dx
          xflux(i,j,k,1)=-0.25*(jz(i,j+1,k)+jz(i,j,k))*(Info%q(i-1,j,k,iBy)+Info%q(i,j,k,iBy))         
       ELSE
          Info%aux(i,j,k,1) = Info%aux(i,j,k,1)+((jy(i,j,k+1)-jy(i,j,k))+(jz(i,j,k)-jz(i,j+1,k)))*dt/dx
          xflux(i,j,k,1)=0.25*(jy(i,j,k+1)+jy(i,j,k))*(Info%q(i-1,j,k,iBz)+Info%q(i,j,k,iBz))-0.25*(jz(i,j+1,k)+jz(i,j,k))*(Info%q(i-1,j,k,iBy)+Info%q(i,j,k,iBy))
       END IF
    END DO; END DO; END DO
    DO i=msy(1,1),msy(1,2); DO j=msy(2,1),msy(2,2); DO k=msy(3,1),msy(3,2)
       IF(nDim==2)THEN
          Info%aux(i,j,k,2) = Info%aux(i,j,k,2)+(jz(i+1,j,k)-jz(i,j,k))*dt/dx
          yflux(i,j,k,1)=0.25*(jz(i+1,j,k)+jz(i,j,k))*(Info%q(i,j-1,k,iBx)+Info%q(i,j,k,iBx))
       ELSE
          Info%aux(i,j,k,2) = Info%aux(i,j,k,2)+((jz(i+1,j,k)-jz(i,j,k))+(jx(i,j,k)-jx(i,j,k+1)))*dt/dx
          yflux(i,j,k,1)=0.25*(jz(i+1,j,k)+jz(i,j,k))*(Info%q(i,j-1,k,iBx)+Info%q(i,j,k,iBx))-0.25*(jx(i,j,k+1)+jx(i,j,k))*(Info%q(i,j-1,k,iBz)+Info%q(i,j,k,iBz))
       END IF
    END DO; END DO; END DO
    DO i=msz(1,1),msz(1,2); DO j=msz(2,1),msz(2,2); DO k=msz(3,1),msz(3,2)
       IF(nDim==2)THEN
          Info%q(i,j,k,iBz) = Info%q(i,j,k,iBz)+((jx(i,j+1,k)-jx(i,j,k))+(jy(i,j,k)-jy(i+1,j,k)))*dt/dx
       ELSE
          Info%aux(i,j,k,3) = Info%aux(i,j,k,3)+((jx(i,j+1,k)-jx(i,j,k))+(jy(i,j,k)-jy(i+1,j,k)))*dt/dx
          zflux(i,j,k,1)=0.25*(jx(i,j+1,k)+jx(i,j,k))*(Info%q(i,j,k-1,iBy)+Info%q(i,j,k,iBy))-0.25*(jy(i+1,j,k)+jy(i,j,k))*(Info%q(i,j,k-1,iBx)+Info%q(i,j,k,iBx))
       END IF
    END DO; END DO; END DO

    DEALLOCATE(jx); DEALLOCATE(jy); DEALLOCATE(jz)

    !ALLOCATE(new_b(1:nDim))
    
    off_en=0d0
    DO k=1-zmbc, mz+zmbc; DO j=1-mbc, my+mbc; DO i=1-mbc, mx+mbc
       !check_en=Info%q(i,j,k,iE)-0.5*SUM(Info%q(i,j,k,iBx:iBz)**2)
       IF(nDim==2)THEN
          Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-dt*((xflux(i+1,j,k,1)-xflux(i,j,k,1))+(yflux(i,j+1,k,1)-yflux(i,j,k,1)))/dx
       ELSE
          Info%q(i,j,k,iE)=Info%q(i,j,k,iE)-dt*((xflux(i+1,j,k,1)-xflux(i,j,k,1))+(yflux(i,j+1,k,1)-yflux(i,j,k,1))+(zflux(i,j,k+1,1)-zflux(i,j,k,1)))/dx
       END IF
       !new_b(1) = 0.5*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
       !new_b(2) = 0.5*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
       !new_b(3) = 0.5*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
       !off_en=max(off_en,abs(check_en-(Info%q(i,j,k,iE)-0.5*SUM(new_b(:)**2)))/check_en)
    END DO; END DO; END DO
    
    !IF(off_en.gt.1d-3) PRINT*, MPI_ID, "Energy Inconsistency Found: ", off_en
    
    CALL storefixupfluxes(Info,msx,1,xflux,iflux)
    CALL storefixupfluxes(Info,msy,2,yflux,iflux)
    IF(nDim==3) CALL storefixupfluxes(Info,msz,3,zflux,iflux)
    
    DEALLOCATE(iflux)!; DEALLOCATE(new_b)
    DEALLOCATE(xflux); DEALLOCATE(yflux); 
    IF(nDim==3) DEALLOCATE(zflux)

    ! update field components in q.
    DO i=1-mbc,mx+mbc; DO j=1-mbc,my+mbc; DO k=1-zmbc,mz+zmbc
       Info%q(i,j,k,iBx)=.5*(Info%aux(i,j,k,1)+Info%aux(i+1,j,k,1))
       Info%q(i,j,k,iBy)=.5*(Info%aux(i,j,k,2)+Info%aux(i,j+1,k,2))
       IF(nDim==3) Info%q(i,j,k,iBz)=.5*(Info%aux(i,j,k,3)+Info%aux(i,j,k+1,3))
    END DO; END DO; END DO

  END SUBROUTINE ResistiveSource


  SUBROUTINE RealResistivity(Info,i,j,k,resi,dx,dir)
    TYPE(InfoDef) :: Info
    INTEGER :: i,j,k,l,m,n,dir
    REAL(KIND=qPREC) :: dx,resi,temp,t,r,r0,rho,rhotemp,Cln,pos(3)

    IF(dir==1)THEN
       pos=Info%xBounds(1:3,1)+(/REAL(i)-half,REAL(j)-1d0,REAL(k)-1d0/)*dx
    ELSE IF(dir==2)THEN
       pos=Info%xBounds(1:3,1)+(/REAL(i)-1d0,REAL(j)-half,REAL(k)-1d0/)*dx
    ELSE IF(dir==3)THEN
       pos=Info%xBounds(1:3,1)+(/REAL(i)-1d0,REAL(j)-1d0,REAL(k)-half/)*dx
    END IF
    r=sqrt(pos(1)**2+pos(2)**2)
    r0=0.5
    resi=resistivity*(0.1+0.9*exp(-r**2/r0**2))
    
    !n=1;t=0d0;rho=0d0
    !DO l=-1,0; DO m=-1,0
    !   SELECT CASE(dir)
    !   CASE(1)
    !      CALL getT(Info,i,j+l,k+m,temp)
    !      rhotemp = Info%q(i,j+l,k+m,1)
    !   CASE(2)
    !      CALL getT(Info,i+l,j,k+m,temp)
    !      rhotemp = Info%q(i+l,j,k+m,1)
    !   CASE(3)
    !      CALL getT(Info,i+l,j+m,k,temp)
    !      rhotemp = Info%q(i+l,j+m,k,1)
    !   END SELECT
    !   t=t+temp; rho=rho+rhotemp; n=n+1
    !END DO; END DO
    !t=0.25*t;rho=0.25*rho
    !
    !   IF(t*tScale.le.4.2e5)THEN
    !      Cln=16.3+1.5*LOG(t*tScale)-0.5*LOG(rho*rScale/(mH*Xmu))
    !   ELSE
    !      Cln=22.8+LOG(t*tScale)-0.5*LOG(rho*rScale/(mH*Xmu))
    !   END IF
    
    ! resistivity = 0.14*me**0.5*c^2, scaled
    !resi=SpitzerResistivity*Xmu*Cln*FZeff*t**(-1.5)

  END SUBROUTINE RealResistivity

  SUBROUTINE ResistiveGetMaxSpeed(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: level, m
    REAL(KIND=qPREC) :: dx
    
    level=Info%level
    dx=levels(level)%dx
    
    explicit_maxspeed(level) = max(explicit_maxspeed(level),2d0*resistivity/dx)

  END SUBROUTINE ResistiveGetMaxSpeed


  !SUBROUTINE getT(Info,i,j,k,temp)
  !   TYPE(InfoDef) :: Info
  !   INTEGER :: i,j,k
  !   REAL(KIND=qPREC) :: temp
  
  !   temp=(gamma-1d0)*(Info%q(i,j,k,iE)-0.5*SUM(Info%q(i,j,k,2:4)**2)/Info%q(i,j,k,1)&
  !    -0.5*SUM(Info%q(i,j,k,iBx:iBz)**2))/Info%q(i,j,k,1)
  
  !END SUBROUTINE getT
  
  
END MODULE ResistiveSrc



