!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    global_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file global_declarations.f90
!! @brief Main file for module GlobalDeclarations

!> @defgroup GlobalDeclarations GlobalDeclarations
!! @brief Public variables that need to be accessed throughout the program.

!> Public variables that need to be accessed throughout the program.
!! @ingroup GlobalDeclarations

Module GlobalDeclarations
   IMPLICIT NONE
   SAVE

   PUBLIC

   INCLUDE 'mpif.h'

   !> @name Parameters that everyone should be aware of
   !! @{

   INTEGER, PARAMETER :: qPrec = SELECTED_REAL_KIND(14,32)
   INTEGER, PARAMETER :: xPrec = SELECTED_REAL_KIND(14,32)
   INTEGER, PARAMETER :: ROOT_LEVEL=0
   INTEGER, PARAMETER :: EXTRAPOLATED_BOUND = 1
   INTEGER, PARAMETER :: PERIODIC_BOUND = 2
   INTEGER, PARAMETER :: REFLECT_WALL = 3
   INTEGER, PARAMETER :: REFLECT_BPARALLEL = 4
   INTEGER, PARAMETER :: REFLECT_CYLINDRICAL = 5
   INTEGER, PARAMETER :: INTERNAL_BOUND = 999
   INTEGER, PARAMETER :: MaxDepth = 16
   INTEGER, PARAMETER :: MAX_SUBGRIDS = 2048
   INTEGER, PARAMETER :: MAX_DIMS = 3
   INTEGER, PARAMETER :: NEIGHBORCHILD = -999
   REAL(KIND=qPREC), PARAMETER :: UNDEFINED=-1e30, half=.5d0, one=1d0, two=2d0
   REAL(KIND=qPrec), PARAMETER :: ZERO = 0d0
   INTEGER, PARAMETER :: IEVERYWHERE = 0, IBOUNDARIES = 1
   INTEGER, PARAMETER, DIMENSION(3,2) :: IBOUNDARY=RESHAPE((/2,3,4,5,6,7/),(/3,2/))
   INTEGER, PARAMETER :: NON_THREADED = -1, PSEUDO_THREADED = 0, THREADED = 1

   !> @}

   !> @name Level data type declarations
   !! @{

   !> Defines a level and its parameters
   TYPE LevelDef
      REAL(KIND=qPREC) :: dx
      INTEGER :: CoarsenRatio=2
      REAL(KIND=qPREC) :: qTolerance=1e-3
      REAL(KIND=qPrec) :: DesiredFillRatios
      INTEGER, DIMENSION(3) :: gmbc !nmbc(i) is used for determining neighbors at the end of step i [nmbc(1) must be >= ombc(2)] since neighbors become overlaps
      INTEGER, DIMENSION(2) :: egmbc 
      INTEGER, DIMENSION(2) :: ombc 
      INTEGER, DIMENSION(2) :: ambc 
      INTEGER :: nmbc 
      INTEGER :: pmbc !The number of ghost cells required for Info%q (used to store phi)
      REAL(KIND=qPREC) :: dt
      REAL(KIND=qPREC) :: tnow
      INTEGER :: step
      INTEGER :: steps
      INTEGER, DIMENSION(3) :: mX=1
      INTEGER(8) :: Grid
      INTEGER :: MPI_COMM=MPI_COMM_WORLD
      INTEGER :: id
      INTEGER :: CurrentLevelStep=0
   END TYPE LevelDef
   TYPE(LevelDef), DIMENSION(:), ALLOCATABLE :: levels
   !> @}

   !> @name Domain data type declarations
   !! @{

   !> Defines a physical rectangular region.
   TYPE DomainDef
      INTEGER, DIMENSION(3,2) :: mGlobal
      INTEGER, DIMENSION(3,2) :: mthbc
   END TYPE DomainDef
   TYPE(DomainDef), DIMENSION(:), ALLOCATABLE :: Domains
   !> @}

   !> @name Variables needed by DataInfoOps to synchronize tree
   !! @{
   INTEGER :: NrVars
   INTEGER :: GVars   !Two different distances for ghost zoning...
   INTEGER :: EGVars  !Two regions must be grouped in q... (1:GVars) & (GVars+1:GVars+EGVars)
   INTEGER :: TDVars  !Number of time derivative variables
   INTEGER :: nFlux
   INTEGER :: nProlongate
   INTEGER :: nRestrict
   INTEGER :: nEMF
   INTEGER :: nAux
   INTEGER :: nEllipticTransfers
   INTEGER, DIMENSION(:), ALLOCATABLE :: TimeDerivFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: GCopyFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: EGCopyFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: ProlongateFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: RestrictFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: FluxFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: invFluxFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: auxFields
   INTEGER, DIMENSION(:), ALLOCATABLE :: InterpMethod
   INTEGER, DIMENSION(:), ALLOCATABLE :: EllipticTransferFields
   INTEGER, DIMENSION(3) :: EmfLoc                               !EmfLoc(i) stores indice in info%emf for component i of emf
   INTEGER, DIMENSION(3) :: EmfDir                               !EmfDir(i) stores component of emf stored in Info%emf(i)
   LOGICAL ::  MaintainAuxArrays
   !> @}

   !> @name Miscellaneous variables - probably belong elsewhere
   !! @{
   INTEGER :: MPI_ID=0, MPI_NP
   INTEGER :: iThreaded=-1
   REAL(KIND=qPREC), DIMENSION(2) :: LevelBalance=(/0d0,0d0/)
   LOGICAL :: lSkipProfile=.true.
   LOGICAL :: lKnapSack=.false.
   LOGICAL :: lTimingLog=.false.
   LOGICAL :: lParticles=.false.
   LOGICAL :: lElliptic=.false.
   LOGICAL :: lExplicit=.false.
   INTEGER :: MinimumGridPoints=4
   LOGICAL :: lStressTest=.false.
   REAL, DIMENSION(:), ALLOCATABLE :: WorkLoad
   REAL :: MySpeedFactor=1
   REAL(8) :: StartTime
   LOGICAL, DIMENSION(3) :: lEllipticPeriodic=.false.
   LOGICAL, DIMENSION(3) :: lHydroPeriodic=.false.
   LOGICAL :: lStoreMassFlux=.false.                              !Switch whether or not mass fluxes are stored
   LOGICAL :: lNeedMeanDensity=.false.
   INTEGER, DIMENSION(3) :: nperiodic_overlaps=1   ! Number of times to periodically stack data for filling ghost zones in periodic directions.  
   !> @}

   !> @name Grid global parameters
   !! @{
   LOGICAL, DIMENSION(-2:MaxDepth) :: lRegridLevel
   INTEGER :: ndim=2
   INTEGER :: MaxLevel=0
   INTEGER :: LastStaticLevel=-1
   LOGICAL :: lUseOriginalNewSubGrids=.false.
   INTEGER :: FinestLevel
   INTEGER :: RestartLevel
   INTEGER :: BaseLevel=-2
   INTEGER :: nDomains=1
   REAL(KIND=qPREC), DIMENSION(3,2) :: GxBounds ! Global bounds
   INTEGER, DIMENSION(3) :: GmX=(/32,32,1/)     ! Global resolution
   INTEGER, DIMENSION(3,2) :: GmGlobal     ! Global resolution
   INTEGER, DIMENSION(3,2) :: Gmthbc = 1
   LOGICAL, DIMENSION(3) :: lAnyPeriodic=.false.
   INTEGER :: hyperbolic_mbc  ! Number of ghost cells used by hyperbolic solver   !lost each hyperbolic step
   INTEGER :: elliptic_mbc    ! Number of ghost cells required by elliptic solver !not lost
   INTEGER :: source_mbc      ! Number of ghost cells required by source steps    !not lost
   INTEGER :: particle_mbc    ! Number of ghost cells required by particleupdates !not lost
   INTEGER :: afterstep_mbc   ! Number of ghost cells required by explicit solver !lost each explicit step
   !> @}

   !> @name Time stepping variables
   !! @{
   REAL(KIND=qPrec) :: initial_maxspeed
   REAL(KIND=qPrec), DIMENSION(3) :: cfl_vars = (/1.0,.3,.5/)
   INTEGER :: current_frame
   INTEGER :: start_frame = 0
   INTEGER :: final_frame
   INTEGER :: restart_frame
   REAL(KIND=qPREC) :: current_time, start_time=0d0, final_time, restart_time
   LOGICAL :: RestartStep=.false.
   LOGICAL :: lRequestRestart=.false.
   !> @}

   REAL(KIND=qPREC) :: InitTime


   !> @name IO-Relevant variables.
   !! @{
   LOGICAL :: lRestart=.false.
   LOGICAL :: lRegrid=.false.
   LOGICAL :: lPostProcess=.false.
   LOGICAL :: lReOutput=.false.
   INTEGER :: iDataFileType=0
   LOGICAL :: lPrintDebugFrame=.false.
   !> @}

   !> @name NameList Variables
   !! @{
   NAMELIST /GlobalData/ MaxLevel, LastStaticLevel, lUseOriginalNewSubGrids, MinimumGridPoints, &
        lRestart, lPostProcess, lRegrid, restart_frame, start_time, final_time, final_frame, GmX, &
        GxBounds, Gmthbc, initial_maxspeed, cfl_vars, iThreaded, LevelBalance, lKnapSack, lTimingLog, lStressTest, MaintainAuxArrays, nDim, lSkipProfile

   INTEGER, PARAMETER :: GLOBAL_DATA_HANDLE = 76
   CHARACTER(LEN=11), PARAMETER :: GLOBAL_DATA_FILE = "global.data"
   INTEGER, PARAMETER :: PHYSICS_DATA_HANDLE = 73
   CHARACTER(LEN=12), PARAMETER :: PHYSICS_DATA_FILE = "physics.data"
   INTEGER, PARAMETER :: MODULES_DATA_HANDLE = 74
   CHARACTER(LEN=12), PARAMETER :: MODULES_DATA_FILE = "modules.data"
   INTEGER, PARAMETER :: COMMUNICATION_DATA_HANDLE = 77
   CHARACTER(LEN=18), PARAMETER :: COMMUNICATION_DATA_FILE = "communication.data"
   INTEGER, PARAMETER :: PROFILE_DATA_HANDLE = 78
   CHARACTER(LEN=12), PARAMETER :: PROFILE_DATA_FILE = "profile.data"
   INTEGER, PARAMETER :: SCALES_DATA_HANDLE = 79
   CHARACTER(LEN=11), PARAMETER :: SCALES_DATA_FILE = "scales.data"
   INTEGER, PARAMETER :: PROBLEM_DATA_HANDLE = 80 !Generic data handle for individual use modules
   INTEGER, PARAMETER :: TIMER_LOG_HANDLE = 81
   INTEGER, PARAMETER :: COMM_LOG_HANDLE = 82
   INTEGER, PARAMETER :: PROCESS_DATA_HANDLE = 83
   CHARACTER(LEN=12), PARAMETER :: PROCESS_DATA_FILE = "process.data"
   INTEGER, PARAMETER :: TOTALS_DATA_HANDLE = 84
   CHARACTER(LEN=16), PARAMETER :: TOTALS_DATA_FILE = "out/totals.dat"
   INTEGER, PARAMETER :: HISTOGRAM_DATA_HANDLE = 85
   INTEGER, PARAMETER :: BOV_DATA_HANDLE = 86
   INTEGER, PARAMETER :: PPM_DATA_HANDLE = 87
   INTEGER, PARAMETER :: BONDI_DATA_HANDLE = 88
   INTEGER, PARAMETER :: SPECTRA_DATA_HANDLE = 89
   INTEGER, PARAMETER :: LAYOUT_DATA_HANDLE = 90
   INTEGER, PARAMETER :: CAMERA_DATA_HANDLE = 91
   !> @}

   INTEGER :: InfoAllocator=1, MessageAllocator=2, SweepAllocator=3
   REAL(KIND=qPREC) :: allocator(3)=0, maxallocation(3)=0
   REAL(KIND=qPREC) :: NumCellsByLevel(0:MaxDepth)

   INTEGER, PARAMETER :: PACK_INTEGER_SIZE = 4
   INTEGER, PARAMETER :: PACK_FLOAT_SIZE = 4
   INTEGER, PARAMETER :: PACK_DOUBLE_SIZE = 8
   INTEGER, PARAMETER :: PACK_BOX_SIZE = 24     ! 6 * PACK_INTEGER_SIZE
   INTEGER, PARAMETER, DIMENSION(3,2) :: TERMINATIONBOX = RESHAPE((/-1,-1,-1,-1,-1,-1/), (/3, 2/))
   INTEGER, PARAMETER :: TERMINATION_BOX_BYTES = 24     ! size of six integers

  INTEGER, PARAMETER :: BINBYVOLUME=-999, BINBYMASS=-998
  REAL(KIND=qPREC), PARAMETER :: MINOVERALL=1e30, MAXOVERALL=-1e30
  INTEGER, PARAMETER :: LOGSCALE=0, LINEARSCALE=1

CONTAINS

   !> Checks if a value is NAN
   !! @param x value to check
   PURE ELEMENTAL FUNCTION ISNAN(x)
      LOGICAL :: ISNAN
      REAL(KIND=qPrec), INTENT(IN) :: x
      ISNAN=.FALSE.
      IF(x .ne. x) ISNAN=.TRUE.
   END FUNCTION ISNAN

   !> Checks if a value is NAN or INF
   !! @param x value to check
   PURE ELEMENTAL FUNCTION ISINFNAN(x)
      LOGICAL :: ISINFNAN
      REAL(KIND=qprec), INTENT(IN) :: x
      ISINFNAN=.FALSE.
      IF(ISNAN(x) .OR. ABS(x)>HUGE(x)) ISINFNAN=.TRUE.
   END FUNCTION ISINFNAN

   !> Converts .TRUE. and .FALSE. to -1 and 0, respectively (shouldn't Fortran have this already?).
   !! @param flag The logical value to be converted.
   INTEGER FUNCTION BoolToInt(flag)

      LOGICAL :: flag

      IF (flag) THEN
         BoolToInt = -1
      ELSE
         BoolToInt = 0
      END IF

   END FUNCTION BoolToInt


   SUBROUTINE OutputIntArray(a)
      INTEGER, DIMENSION(:,:) :: a
      INTEGER :: i,logb,b
      CHARACTER(LEN=22) :: X
      CHARACTER(LEN=16) :: Y
      b=maxval(abs(a))
      DO i=0, 9
         IF (size(a, 1) < 10**i) EXIT
      END DO
      DO logb=0,9
         IF (b < 10**logb) EXIT
      END DO
      WRITE(Y,'(A5,I1,A10)') '(A1,I',i,',A1,I1,A1)'
      write(X,Y) '(',size(a, 1),'I',logb+2,')'
      write(*,'(A,I3,A1,I3,A)') '------------Integer array ', size(a,1), 'x',size(a,2),'------------------'
      write(*,X) a !transpose(a)
   END SUBROUTINE OutputIntArray

   SUBROUTINE OutputRealArray(a)
      REAL, DIMENSION(:,:) :: a
      INTEGER :: i,logb,b
      CHARACTER(LEN=17) :: X
      CHARACTER(LEN=9) :: Y
      b=maxval(abs(a))
      DO i=0, 9
         IF (size(a, 1) < 10**i) EXIT
      END DO
      DO logb=0,9
         IF (b < 10**logb) EXIT
      END DO
      WRITE(Y,'(A5,I1,A3)') '(A1,I',i,',A6)'
      write(X,Y) '(',size(a, 1),'E15.3)'
      write(*,'(A,I3,A1,I3,A)') '------------Real array ', size(a,1), 'x',size(a,2),'------------------'
      write(*,X) a !transpose(a)
   END SUBROUTINE OutputRealArray


   SUBROUTINE OutputDoubleArray(a)
      REAL(8), DIMENSION(:,:) :: a
      INTEGER :: i,logb,b
      CHARACTER(LEN=17) :: X
      CHARACTER(LEN=10) :: Y
      b=maxval(abs(a))
      DO i=0, 9
         IF (size(a, 1) < 10**i) EXIT
      END DO
      DO logb=0,9
         IF (b < 10**logb) EXIT
      END DO
      WRITE(Y,'(A5,I1,A4)') '(A1,I',i,',A7)'
      write(X,Y) '(',size(a, 1),'E25.16)'
      write(*,'(A,I3,A1,I3,A)') '------------Double array ', size(a,1), 'x',size(a,2),'------------------'
      write(*,X) a !transpose(a)
   END SUBROUTINE OutputDoubleArray


   SUBROUTINE CheckAllocation(i, size, caller, err)
      INTEGER, OPTIONAL :: err
      INTEGER :: i, size, j
      CHARACTER(LEN=*), OPTIONAL :: caller
      IF (present(err)) THEN
         IF (err /= 0) THEN
            IF (present(caller)) THEN
               write(*,*) 'check allocation failed with err ', err, 'size = ', size, caller
            ELSE
               write(*,*) 'check allocation failed with err ', err, 'size = ', size
            END IF
            STOP
         END IF
      END IF
      allocator(i)=allocator(i)+REAL(size,KIND=qPREC)
      maxallocation(i)=max(maxallocation(i), allocator(i))
   END SUBROUTINE CheckAllocation

   SUBROUTINE CheckDeAllocation(i, size, err, caller)
      INTEGER, OPTIONAL :: err
      CHARACTER(LEN=*), OPTIONAL :: caller
      INTEGER :: i, size
      IF (present(err)) THEN
         IF (err /= 0) THEN
            IF (present(caller)) THEN
               write(*,*) 'check allocation failed with err ', err, 'size = ', size, caller
            ELSE
               write(*,*) 'check allocation failed with err ', err, 'size = ', size
            END IF
            STOP
         END IF
      END IF
      allocator(i)=allocator(i)-REAL(size,KIND=qPREC)
   END SUBROUTINE CheckDeAllocation


   FUNCTION printsize(x)
      REAL(KIND=qPREC) :: x
      CHARACTER(LEN=10) :: printsize
      IF (abs(x) == 0) THEN
         write(printsize,'(A10)') '  ------  '
      ELSEIF (abs(x) > 1024d0**3) THEN
         write(printsize, '(F7.1,A3)') x/real(1024d0**3), ' gb'
      ELSEIF (abs(x) > 1024d0**2) THEN
         write(printsize, '(F7.1,A3)') x/real(1024d0**2), ' mb'
      ELSEIF (abs(x) > 1024d0) THEN
         write(printsize, '(F7.1,A3)') x/1024d0, ' kb'
      ELSE
         write(printsize, '(I4,A3)') NINT(x), ' by'
      END IF
   END FUNCTION printsize

   FUNCTION printtime(x)
      REAL(KIND=qPREC) :: x
      CHARACTER(LEN=11) :: printtime
      REAL(KIND=qPREC), PARAMETER :: kyr_=31556926000
      REAL(KIND=qPREC), PARAMETER :: yr_=31556926
      REAL(KIND=qPREC), PARAMETER :: month_=yr_/12d0
      REAL(KIND=qPREC), PARAMETER :: day_=24d0*3600d0
      REAL(KIND=qPREC), PARAMETER :: hr_=3600d0
      REAL(KIND=qPREC), PARAMETER :: min_=60d0

      IF (abs(x) == 0) THEN
         write(printtime,'(A10)') '  ------  '
      ELSEIF (abs(x) > 1000*kyr_) THEN
         write(printtime, '(A5)') 'never'
      ELSEIF (abs(x) > kyr_) THEN
         write(printtime, '(F7.1,A4)') x/kyr_, ' kyr'
      ELSEIF (abs(x) > yr_) THEN
         write(printtime, '(F7.1,A4)') x/yr_, ' yr '
      ELSEIF (abs(x) > month_) THEN
         write(printtime, '(F7.1,A4)') x/month_, ' mo '
      ELSEIF (abs(x) > day_) THEN
         write(printtime, '(F7.1,A4)') x/day_, ' day'
      ELSEIF (abs(x) > hr_) THEN
         write(printtime, '(F7.1,A4)') x/hr_, ' hr '
      ELSEIF (abs(x) > min_) THEN
         write(printtime, '(F7.1,A4)') x/min_, ' min'
      ELSE
         write(printtime, '(F7.1,A4)') x, ' s  '
      END IF
   END FUNCTION printtime

   SUBROUTINE CheckSlabSymmetric(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:) :: q
      INTEGER :: i,j
      DO i=1,size(q,1)
         DO j=1,size(q,3)
            IF (maxval(q(i,:,j)) > minval(q(i,:,j))) THEN
               write(*,*) 'found a problem with field ', j, 'at index ', i, maxval(q(i,:,j)), minval(q(i,:,j))
               write(*,'(100E29.20)') q(i,:,j)
            END IF
         END DO
      END DO
   END SUBROUTINE CheckSlabSymmetric

END Module GlobalDeclarations

