!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    MUSCL_scheme.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MUSCL
!! @brief Directory containing files used by MUSCLScheme

!> @file MUSCL_scheme.f90
!! @brief Main file for module MUSCLScheme

!> @defgroup MUSCLScheme MUSCLScheme
!! @brief Module for updating grids by with MUSCL schemes 
!! @ingroup Hyperbolic

!===============================================================================
! Module Name:		MUSCLScheme
! Module File:		MUSCL_scheme.f90
! Purpose:		Implements a numerical scheme for solving 
!                       hydrodynamic equations.
! Public Methods:	ReadMUSCLDomainData(), MUSCLadvance()
! Created:		by Jonathan Carroll-Nellenback and Baowei Liu 
!===============================================================================

!> Module for updating grids with MUSCL Hancock 
!! @ingroup MUSCLScheme
!! @par Include Files:
!! MUSCL_scheme_.f90
MODULE MUSCLScheme
   USE GlobalDeclarations
   USE DataDeclarations
   USE HyperbolicDeclarations
   USE SourceControl
   USE SourceDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE ModuleControl
   USE RiemannSolvers
   USE DataInfoOps
   USE TreeDeclarations
   IMPLICIT NONE
   PRIVATE
   SAVE 
   PUBLIC MUSCLInit, MUSCLadvance
   INTEGER :: nFields,nSteps
   INTEGER, DIMENSION(:,:), ALLOCATABLE :: wdi, fdi, dim_orders
   LOGICAL :: lUseLimiter=.true.
CONTAINS
   
   SUBROUTINE MUSCLInit()
      NAMELIST/MUSCLData/ lUseLimiter
      INTEGER :: i,d 
      READ(SOLVER_DATA_HANDLE,NML=MUSCLData)
      hyperbolic_mbc=2

      IF (nDim == 1) THEN      
         nSteps=1
         ALLOCATE(dim_orders(nDim,nSteps))
         dim_orders=reshape((/1/),(/nDim,nSteps/))
         nFields=3
         ALLOCATE(wdi(nDim,nFields+NrTracerVars), fdi(nDim,nFields+NrTracerVars))
         wdi(1,1:nFields)=(/1,3,2/)
         fdi(1,1:nFields)=(/1,3,2/)
      ELSEIF (nDim == 2) THEN
         nSteps=2
         ALLOCATE(dim_orders(nDim,nSteps))
         dim_orders=reshape((/1,2, 2,1/),(/nDim,nSteps/))

         nFields=4
         ALLOCATE(wdi(nDim,nFields+NrTracerVars), fdi(nDim,nFields+NrTracerVars))
         wdi(1,1:nFields)=(/1,4,2,3/)
         fdi(1,1:nFields)=(/1,3,4,2/)
         wdi(2,1:nFields)=(/1,4,3,2/)
         fdi(2,1:nFields)=(/1,4,3,2/)
      ELSEIF (nDim == 3) THEN
         nSteps=6
         ALLOCATE(dim_orders(nDim,nSteps))
         dim_orders=reshape((/1,2,3, 1,3,2, 2,1,3, 2,3,1, 3,1,2, 3,2,1/),(/nDim,nSteps/))

         nFields=5
         ALLOCATE(wdi(nDim,nFields+NrTracerVars), fdi(nDim,nFields+NrTracerVars))
         wdi(1,1:nFields)=(/1,5,2,3,4/)
         fdi(1,1:nFields)=(/1,3,4,5,2/)
         wdi(2,1:nFields)=(/1,5,3,4,2/)
         fdi(2,1:nFields)=(/1,5,3,4,2/)
         wdi(3,1:nFields)=(/1,5,4,2,3/)
         fdi(3,1:nFields)=(/1,4,5,3,2/)
      END IF
      IF (NrTracerVars > 0) THEN
         DO d=1,nDim
            wdi(d,nFields+1:nFields+NrTracerVars)=(/(nFields+i, i=1, NrTracerVars)/)
            fdi(d,nFields+1:nFields+NrTracerVars)=(/(nFields+i, i=1, NrTracerVars)/)
         END DO
         nFields=nFields+NrTracerVars
      END IF

      write(*,*) 'nFields=', nFields-NrTracerVars
      write(*,*) 'NrTracerVars=', NrTracerVars
      write(*,*) 'wdi=', wdi(1,:)
      write(*,*) 'fdi=', fdi(1,:)
      write(*,*) 'mhigh_1=', m_high+1
      ! stop
   END SUBROUTINE MUSCLInit

SUBROUTINE MUSCLAdvance(Info, dom_range, lComplete)
  TYPE (InfoDef) :: Info
  LOGICAL :: lComplete
  INTEGER :: level
  INTEGER :: mx  !number of cells in data to update
  REAL(KIND=qPREC) :: dt, hdt !time and half time to advance for
  INTEGER, DIMENSION(3,2) :: dom_range, ghost_range
  INTEGER, DIMENSION(3,2) :: row, grow
  INTEGER, DIMENSION(3,2) :: ms 
  INTEGER :: d, i, j, k, m
  INTEGER ::  dir
  INTEGER , DIMENSION(2) :: pdirs
  REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: data !array that goes from 2-hyperbolic_mbc to mx+hyperbolic_mbc that needs to be updated from 1:mx
  REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: fluxes !array that has fluxes from 1:mx+1  
  REAL(KIND=qPREC) :: dtdx

  ghost_range=dom_range
!  write(*,*) 'dom_range=', dom_range
  ghost_range(1:nDim,:)=ghost_range(1:nDim,:)+spread(hyperbolic_mbc*(/-1,1/),1,nDim)
!  write(*,*) spread(hyperbolic_mbc*(/-1,1/),nDim,1)
!  write(*,*) 'ghost range=', ghost_range
  dt=levels(Info%level)%dt
  hdt=.5d0*dt

  dtdx=dt/levels(Info%level)%dx

  CALL BeforeStep(Info)
  CALL Src(Info, ghost_range, levels(Info%level)%tnow, hdt)

  DO d=1,nDim

     ! direction of update
     dir=dim_orders(d,modulo(levels(Info%level)%CurrentLevelStep-1, nSteps)+1)

!     write(*,*) 'doing pass for dimension ',dir, levels(Info%level)%CurrentLevelStep

     ! normal directions
     pdirs=modulo(dir+(/0,1/),3)+1

     ! grow is range of Info%q to map onto data including ghost zones
     grow(dir,:)=ghost_range(dir,:)

     ! row is range of Info%q to update
     row(dir,:)=dom_range(dir,:)

     ! ms is bounds for returned fluxes needed to update Info%q
     ms(dir,:)=dom_range(dir,:)+(/0,1/)

     ! mx is number of cells to update
     mx=row(dir,2)-row(dir,1)+1

     ALLOCATE(data(1-hyperbolic_mbc:mx+hyperbolic_mbc, nFields), fluxes(1:mx+1, nFields))

     DO j=ghost_range(pdirs(1),1), ghost_range(pdirs(1),2)
        !update bounds in transverse directions to update
        grow(pdirs(1),:)=j
        row(pdirs(1),:)=j
        mS(pdirs(1),:)=j

!        write(*,*) 'j=', j
        DO k=ghost_range(pdirs(2),1), ghost_range(pdirs(2),2)
           !update bounds in transverse directions to update
           grow(pdirs(2),:)=k
           row(pdirs(2),:)=k
           mS(pdirs(2),:)=k

           !remap bounds of ghost zones onto data array and map fields in q to correct order (rho, E, v_perp, v_parallel... using wdi(:,:)
           data(1-hyperbolic_mbc:mx+hyperbolic_mbc,1:nFields)= &
                reshape(Info%q(grow(1,1):grow(1,2), grow(2,1):grow(2,2), grow(3,1):grow(3,2), wDi(dir,:)),(/mx+2*hyperbolic_mbc,nFields/))


           !calculate fluxes using data
           CALL calc_fluxes(data, mx, levels(Info%level)%dt, levels(Info%level)%dx, fluxes, maxspeed(Info%level))

!           FORALL(i=lbound(fluxes,1):ubound(fluxes, 1))
              !remap fluxes back to order of fields in q
              fluxes(:,:)=fluxes(:,fdi(dir,:))*dtdx
!           end forall

           !pass fluxes to storefixupfluxes in case they need to be synchronized or restricted...
           CALL storefixupfluxes(Info,mS,dir,reshape(fluxes,(/ms(:,2)-ms(:,1)+1,nFields/)), (/(m,m=1,nFields)/))           

           !Update q with those fluxes
           Info%q(row(1,1):row(1,2),row(2,1):row(2,2),row(3,1):row(3,2),1:nFields)= &
                Info%q(row(1,1):row(1,2),row(2,1):row(2,2),row(3,1):row(3,2),1:nFields)+ &
                reshape((fluxes(1:mx,1:nFields)-fluxes(2:mx+1,1:nFields)), (/row(:,2)-row(:,1)+1,nFields/))
        END DO
     END DO

  !   DO i=1, Info%mX(dir)
  !      write(*,'(I,10E20.5)') i, Info%q(i,1,1,:)
  !   END DO
     !Now we can shrink the ghost_range in the direction we just updated
     ghost_range(dir,:)=ghost_range(dir,:)+hyperbolic_mbc*(/1,-1/)

     !free up work arrays for this direction
     deallocate(data, fluxes)
  END DO

  CALL Src(Info, dom_range, levels(Info%level)%tnow+hdt, hdt)
  
  CALL AfterStep(Info)

END SUBROUTINE MUSCLAdvance

subroutine calc_fluxes(data, mx, dt, dx, fluxes, maxspeed)
  
  REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: data, fluxes
  REAL(KIND=qPREC), POINTER, DIMENSION(:,:) :: w, dw
  REAL(KIND=qPREC), POINTER, DIMENSION(:) :: wl, wr, fl, fr, df, qr, ql
  INTEGER :: mx  !number of cells in data to update
  INTEGER :: i, mbc, j
  REAL(KIND=qPREC) :: dt, dx, dtdx, hdt  !time and half time to advance for
  REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: f
  REAL(KIND=qPREC) :: maxspeed, dright, dleft, dcenter
  !order of fields in data is rho, E, px, [py, [pz]]
  mbc=hyperbolic_mbc
  ALLOCATE(w(1-mbc:mx+mbc, nFields))
  ALLOCATE(dw(1-mbc+1:mx+mbc-1, nFields))
  !ALLOCATE(wl(1-mbc:mx+mbc))
  !ALLOCATE(wr(1-mbc+1:mx+mbc+1))
  ALLOCATE(wl(nFields))
  ALLOCATE(wr(nFields))
  ALLOCATE(f(nFields))
  ALLOCATE(fr(nFields))
  ALLOCATE(fl(nFields))
  ALLOCATE(df(nFields))
  ALLOCATE(qr(nFields))
  ALLOCATE(ql(nFields))

  dtdx=dt/dx

  DO i=1-mbc,mx+mbc
     CALL cons_to_prim_MUSCL(data(i,:), w(i,:))
  END DO

  DO i=1-mbc+1,mx+mbc-1
     IF (lUseLimiter) THEN
        DO j=1,nFields
           dright=w(i+1,j)-w(i,j)
           dleft=w(i,j)-w(i-1,j)
           dcenter=.5d0*(dright+dleft)
           IF (SIGN(1d0, dleft) == SIGN(1d0, dright)) THEN
              dw(i,j)=sign(min(2d0*abs(dleft), 2d0*abs(dright), abs(dcenter)), dcenter)
           ELSE
              dw(i,j)=0d0           
           END IF
        END DO
     ELSE
        dw(i,:)=.5d0*(w(i+1,:)-w(i-1,:))
     END IF
!     dw(i,:)=0d0
!     write(*,'(I,10E20.5)') i, dw(i,:)
  END DO

  DO i=1,mx+1
     wl = w(i-1,:)+.5d0*dw(i-1,:)
     wr = w(i,:)-.5d0*dw(i,:)
     maxspeed=max(maxspeed, calc_flux(wl, wr, f))
!     write(*,'(A,I)') 'i = ', i
!     write(*,'(A,10E20.5)') 'w(i-1) = ',w(i-1,:) 
!     write(*,'(A,10E20.5)') 'dw(i-1) = ',dw(i-1,:) 
!     write(*,'(A,10E20.5)') 'wl = ', wl
!     write(*,'(A,10E20.5)') 'w(i) = ',w(i,:) 
!     write(*,'(A,10E20.5)') 'dw(i) = ',dw(i,:) 
!     write(*,'(A,10E20.5)') 'wr = ', wr
!     write(*,'(A,10E20.5)') 'f = ', f
     fluxes(i,:)=f
  END DO

  DEALLOCATE(w, dw, wl, wr, f, fr, fl, df, qr, ql)
end subroutine calc_fluxes 


subroutine cons_to_prim_MUSCL(q,w)
  REAL(KIND=qPREC),DIMENSION(:) :: q,w
  w = q
  w(2)=gamma1*(q(2)-half*sum(q(3:m_high+1)**2)/q(1))
  w(3:m_high+1)=q(3:m_high+1)/q(1)
  !w(nDim+2+1:nFields)=q(nDim+2+1:nFields)
   
END subroutine cons_to_prim_MUSCL

subroutine prim_to_cons_MUSCL(w,q)
  REAL(KIND=qPREC),DIMENSION(:) :: w,q
  q = w
  q(2)=gamma7*w(2)+half*sum(w(3:m_high+1)**2)*w(1)
  q(3:m_high+1)=w(3:m_high+1)*w(1)
  !w(nDim+2+1:nFields)=q(nDim+2+1:nFields)
   
END subroutine prim_to_cons_MUSCL

END MODULE MUSCLScheme
