!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    hyperbolic_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir hyperbolic
!! @brief Directory containing modules for solving hyperbolic equations with source terms

!> @defgroup Hyperbolic Hyperbolic Solvers
!! @brief Group of modules for solving hyperbolic equations with source terms

!> @file hyperbolic_control.f90
!! @brief Main file for module HyperbolicControl

!> @defgroup HyperbolicControl Hyperbolic Control
!! @brief Module for managing different hyperbolic solvers
!! @ingroup Hyperbolic


!> Module for managing different hyperbolic solvers
!! @ingroup HyperbolicControl

!============================================================================================
! Module Name:        SolverControl
! Module File:        solver_control.f90
! Purpose:              Provide an interface for the rest of the code to use when 
!                       initializing and using numerical solvers.  Handles the 
!                       selection of a scheme and the reading in of appropriate data.
! Public Methods:    SolverInit(), advance()
! Created:            20100705 by Brandon D. Shroyer
!============================================================================================
MODULE HyperbolicControl

  USE DataDeclarations
  USE HyperbolicDeclarations
  USE PhysicsDeclarations
  USE SweepScheme
  USE MUSCLScheme
  USE Timing
  IMPLICIT NONE
  PRIVATE

  PUBLIC HyperbolicInit, advance, PRofileAdvance, AdvanceStackSize

CONTAINS

  !> @brief Read data in from the SOLVER_DATA_FILE and use it to initialize a solver
  !!        based on which scheme is selected.
  !!        Bootstraps initialization by reading in the SolverData namelist,
  !!        and then uses that data to select the appropriate scheme.  
  !!        Calls scheme implementation files to finish initialization.


  !> Read data in from the SOLVER_DATA_FILE and use it to initialize a solver based on which scheme is selected.
  ! Created:    20100705 by Brandon D. Shroyer
  ! Notes:    Bootstraps initialization by reading in the SolverData namelist,
  !           and then uses that data to select the appropriate scheme.  
  !           Calls scheme implementation files to finish initialization.
  SUBROUTINE HyperbolicInit()
    INTEGER :: iErr
    ALLOCATE(maxspeed(0:MaxLevel), maxwavespeed(0:Maxlevel), maxsolverspeed(0:MaxLevel), AdvanceStencil(0:MaxLevel), t_startadvance(0:MaxLevel))


    ALLOCATE(WorkDoneByLevel(-1:MaxLevel))
    ALLOCATE(WorkDoneByGrid(-1:MaxLevel))
    ALLOCATE(InternalCellUpdates(0:MaxLevel))
    ALLOCATE(CellUpdates(0:MaxLevel))

    WorkDoneByLevel=0d0
    WorkDoneByGrid=0d0
    InternalCellUpdates=0
    CellUpdates=0

    ! Open data file (quit if not successful).
    OPEN(UNIT=SOLVER_DATA_HANDLE,FILE=SOLVER_DATA_FILE,IOSTAT=iErr)

    IF (iErr /= 0) THEN
       PRINT *, "SolverInit() error: unable to open ", SOLVER_DATA_FILE, "."
       STOP
    END IF

    ! Read in scheme value.
    READ(SOLVER_DATA_HANDLE,NML=SolverData,IOStat=iErr)

    IF(iErr/=0) THEN
       PRINT *, "SolverInit() error:  unable to read ", SOLVER_DATA_FILE, "."
       STOP
    END IF


    IF ((iSolver == 1 .OR. iSolver == 3) .AND. (lMHD .OR. (iEOS == EOS_ISOTHERMAL))) THEN
       PRINT*,'method(1) = 1 and iSolver = ', iSolver, 'is only supported for adiabatic hydro.  ', &
              'Please verify that iCooling /= 4 and that lMHD = False '
    END IF
    IF (iSolver== 0) THEN
       iSolver=2
       IF (lMHD) iSolver = iSolver + 4
       IF (iEOS == EOS_ISOTHERMAL) iSolver = iSolver + 2
    END IF

    ! Select the appropriate scheme option--currently there are only two, so
    ! there's no need for a select-case statement.
    IF (iScheme == SWEEP_SCHEME_ID) THEN
       CALL SweepReadDomainData()

    ELSE IF (iScheme == MUSCL_SCHEME_ID) THEN
       CALL MUSCLInit()

    ELSE
       PRINT "('SolverInit() error: invalid scheme option ', i3, '.')", iScheme
       STOP

    END IF

    CLOSE(SOLVER_DATA_HANDLE, IOStat=iErr)

    IF (iErr /= 0) THEN
       PRINT *, "SolverInit() error:  unable to close ", SOLVER_DATA_FILE, "."
    END IF


    ALLOCATE(NodeCompleted(0:MaxLevel))
    ALLOCATE(tused_this_grid(0:MaxLevel))
  END SUBROUTINE HyperbolicInit


  FUNCTION AdvanceStackSize(n)
     INTEGER :: n
     INTEGER :: AdvanceStackSize
     SELECT CASE(iScheme)
     CASE(SWEEP_SCHEME_ID)
        AdvanceStackSize=SweepAdvanceStackSize(n)
     CASE DEFAULT
        PRINT*, 'error - ischeme not recognized'
     END SELECT

  END FUNCTION AdvanceStackSize

  !> Performs one (level-appropriate) step-advance on the input grid.
  !! @details Does no actual work on its own; rather, it calls the appropriate 
  !! scheme's advance method.
  !! @param Info Info structure
  !! @param partialOK_opt Optional logical flag that says partial updates of info are ok.
  SUBROUTINE advance(Info, lCompleteOpt)
    ! Interface declarations
    TYPE (InfoDef) :: Info
    LOGICAL :: partialOK, lComplete
    INTEGER :: bc, mB(3,2), i
    LOGICAL, OPTIONAL :: lCompleteOpt
    IF (Present(lCompleteOpt)) THEN
       lComplete=lCompleteOpt
    ELSE
       lComplete=.false.
    END IF
    ! choose which scheme
    !    bc(1:nDim) = levels(level)%gmbc(levels(level)%step)-hyperbolic_mbc
    !    dom_range= reshape((/1, 1, 1, Info%mX(1),Info%mX(2), Info%mX(3)/),(/3,2/))
    !    dom_range=expand(1,bc(1), expand(2,bc(2), expand(3,bc(3), dom_range)))
    ! Number of ghost zones that need to be updated by hyperbolic advance
!    if (levels(Info%level)%dt == 0) THEN
!       write(*,*) "shouldn't be here"
!       write(*,*) bc
!       STOP
!    end if
    
    bc=levels(Info%level)%ambc(levels(Info%level)%step) 
    mB=1
    mB(1:nDim,2)=Info%mX(1:nDim)+bc
    mB(1:nDim,1)=1-bc

!    DO i=1, 100
!       Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)=2d0*&
!            Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)
!       Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)=.5d0*&
!            Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)
!       Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)=1.001*&
!            Info%q(mb(1,1):mb(1,2), mb(2,1):mb(2,2), mb(3,1):mb(3,2),1:NrHydroVars)
!    END DO
!    RETURN
!    IF (MPI_ID == 0) write(*,*) 'advancing', mB
    SELECT CASE(iScheme)
    CASE(MUSCL_SCHEME_ID)
       CALL MUSCLAdvance(Info,mB, lComplete)
    CASE(SWEEP_SCHEME_ID)
       CALL sweepAdvance(Info,mB, lComplete)
    CASE DEFAULT
       PRINT "('advance() error: invalid scheme option ', i3, '.')", iScheme
       STOP

    END SELECT
!    Info%OldCostPerCell=Info%CostPerCell
!    Info%CostPerCell= 1
  END SUBROUTINE advance


  SUBROUTINE ProfileAdvance
     USE ModuleControl
     USE CommonFunctions
     !    USE IOBOV
     INTEGER :: i,j,k,l,s, ierr, m
     INTEGER :: ncase,nvar,nmax,nk,nsize, npoints
     INTEGER, DIMENSION(3,2) :: mGlobal
     REAL(KIND=qPREC) :: tstart, temp, temp2
     TYPE(InfoDef), POINTER :: Info
     REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: Times,sls,errls
     INTEGER, DIMENSION(:), ALLOCATABLE :: points
     REAL(KIND=qPREC), DIMENSION(:,:), ALLOCATABLE :: LeastSquares
     REAL(KIND=qPREC) :: AdvanceCoeffsTemp(8), err
     CHARACTER(LEN=40) :: FileName
     INTEGER :: filehandle=1324, tempithreaded, curr_progress, old_progress
     REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: MeanAdvanceCoeffs
     LOGICAL :: lFound
     NAMELIST /ProfileData/ AdvanceCoeffsTemp     ! This namelist will probably grow over time.   
     levels(0)%step=levels(0)%steps
     !     levels(0)%gmbc(1)=hyperbolic_mbc
     !     levels(0)%dx=1
     levels(0)%dt=1e-10
     tempiThreaded=iThreaded
     iThreaded = NON_THREADED
     ! This section computes the least square from sampled mpi_wtime
     !     ncase=ndim**(6-ndim);nvar=2**nDim;nmax=nDim**(9-2*nDim);nk=27**(nDim-2);nsize=6-ndim


     IF (nDim == 1) THEN
        npoints=12
        nvar=2
        ALLOCATE(AdvanceGridTimes(npoints,1,1))
     ELSE IF (nDim == 2) THEN
        npoints=11
        ALLOCATE(AdvanceGridTimes(npoints,npoints,1))
        nvar=2
     ELSE
        npoints=6
        ALLOCATE(AdvanceGridTimes(npoints,npoints,npoints))
        nvar=1
     END IF
     ncase=min(4,npoints)**nDim

     ALLOCATE(AdvanceCoeffs(nvar), MeanAdvanceCoeffs(nvar)) !Advance Coefficients
     OPEN(UNIT=PROFILE_DATA_HANDLE, IOSTAT=ierr, file=PROFILE_DATA_FILE, status="old")
     lFound=.false.
     IF (ierr==0) THEN !Found the file
        IF (MPI_ID == 0) PRINT *,'Found existing profile.data file.'
        lFound=.true.
        READ(PROFILE_DATA_HANDLE, NML=ProfileData)
        AdvanceCoeffs(1)=AdvanceCoeffsTemp(1)
        AdvanceCoeffs(2:size(AdvanceCoeffs))=AdvanceCoeffsTemp(2:size(AdvanceCoeffs))*AdvanceCoeffs(1)
        READ(PROFILE_DATA_HANDLE, '(10E25.16)') AdvanceGridTimes
        CLOSE(PROFILE_DATA_HANDLE)
        IF (.NOT. lSkipProfile) THEN
           ! Test profile.data quickly for a small 32**nDim grid
           mGlobal=1
           ! Get Advance Times (solution) for sample points
           mGlobal(1:nDim,2)=2**(5)
           CALL InitInfo(Info, 0, mGlobal)
           IF (MaintainAuxArrays) Info%aux(:,:,:,:)=0
           Info%q(:,:,:,:)=0d0
           Info%q(:,:,:,1)=1d0
           IF (iE .ne. 0d0) Info%q(:,:,:,iE)=1d0
           CALL sweepAdvance(Info,mGlobal,lComplete=.true., lProfile_opt=.true.)
           tstart=MPI_Wtime()
           CALL sweepAdvance(Info,mGlobal,lComplete=.true., lProfile_opt=.true.)
           tstart=MPI_Wtime()-tstart
           CALL DestroyInfo(Info)
           err=ABS(tstart-AdvanceCost(mGlobal(:,2)))/tstart
           CALL MPI_ALLREDUCE(MPI_IN_PLACE, err, 1, MPI_DOUBLE_PRECISION, MPI_MAX, MPI_COMM_WORLD, iErr)
           IF (err > .1d0) THEN
              lFound=.false.
           END IF
           IF (MPI_ID == 0 .AND. .NOT. lFound) THEN
              write(*,*) 'profile.data is off by ', nint(err*100d0), '%. Generating a new profile'
              write(*,*) 'If you wish to skip re-profiling - consider adding lSkipProfile=.true. to your global.data file'
           END IF
        END IF
     END IF

     !       IF (MPI_ID == 0) CALL WriteBOV3DScalar('advancecoeffs', (/0d0,0d0,0d0/)-half,(/npoints, npoints, npoints/)-half, 0d0, AdvanceGridTimes, 'advance_times')
     IF (.NOT. lfound) THEN
        ALLOCATE(Times(1:ncase)) ! Array to store solution for fit points
        ALLOCATE(LeastSquares(ncase,nvar))
        AdvanceCoeffs=0
        mGlobal=1
        ! Get Advance Times (solution) for sample points
        mGlobal(1:nDim,2)=2**(5)
        IF (.NOT. lSkipProfile) THEN
           CALL InitInfo(Info, 0, mGlobal)
           IF (MaintainAuxArrays) Info%aux(:,:,:,:)=0
           Info%q(:,:,:,:)=0d0
           Info%q(:,:,:,1)=1d0
           IF (iE .ne. 0d0) Info%q(:,:,:,iE)=1d0
           CALL sweepAdvance(Info,mGlobal,lComplete=.true.,lProfile_opt=.true.)
           CALL DestroyInfo(Info)
        END IF
        CALL MPI_BARRIER(MPI_COMM_WORLD, ierr) !want these to happen at the same time

        IF (lSkipProfile) THEN
           DO i=1,npoints
              DO j=1, npoints        
                 DO k=1, npoints
                    IF (nDim == 1 .AND. j > 1) CYCLE
                    IF (nDim <= 2 .AND. k > 1) CYCLE
                    AdvanceGridTimes(i,j,k)=2**(i+j+k-3)
                 END DO
              END DO
           END DO
        ELSE
           IF (MPI_ID == 0)  write(*,*) 'profiling...'
           s=1
           l=0
           curr_progress=0
           DO i=1, npoints
              DO j=1, npoints        
                 DO k=1, npoints
                    IF (nDim == 1 .AND. j > 1) CYCLE
                    IF (nDim <= 2 .AND. k > 1) CYCLE
                    mglobal(1:3,2)=2**((/i,j,k/)-1)
                    IF (nDim <= 2) mGlobal(3,2)=1
                    IF (nDim <= 1) mGlobal(2,2)=1
                    CALL InitInfo(Info, 0, mGlobal)
                    Info%q(:,:,:,:)=0d0
                    Info%q(:,:,:,1)=1d0
                    IF (iE .ne. 0d0) Info%q(:,:,:,iE)=1d0
                    IF (MaintainAuxArrays) Info%aux=0d0
                    tstart=MPI_Wtime()
                    CALL sweepAdvance(Info,mGlobal,lComplete=.true.,lProfile_opt=.true.)
                    AdvanceGridTimes(i,j,k)=MPI_Wtime()-tstart
                    !                 IF (i > npoints-4 .AND. (nDim < 2 .OR. j > npoints-4) .AND. (nDim < 3 .OR. k > npoints-4)) THEN
                    !                    Times(s)=AdvanceGridTimes(i,j,k)
                    !                    LeastSquares(s,:)=(/product(Info%mX), 1/)
                    !                    s=s+1
                    !                 END IF
                    CALL DestroyInfo(Info)
                    l=l+1
                    curr_progress=nint(40d0*real(l)/real(npoints**nDim))
                    IF (MPI_ID == 0) write(*,'(1A1,A1,I3,A2,40A1,A1,$)')  char(13), ' ', nint(100d0*real(curr_progress)/40d0), '%|', ('|', m=1, curr_progress), (' ', m=curr_progress+1,40), '|'
                 END DO
              END DO
           END DO
           IF (MPI_ID == 0) write(*,*)
        END IF

        !Call the least square fitting function, 
        !sls stores the solution vector in an order of (xy,x,y,1) or (xyz,xy,yz,zx,x,y,z,1)
        !errls stores the relative error for the ncase sampled data
!        CALL LSSolver(LeastSquares,Times,AdvanceCoeffs) ! 

        AdvanceCoeffs(1)=AdvanceGridTimes(npoints,merge(npoints,1,nDim >= 2) ,merge(npoints,1,nDim == 3))/(2d0**(npoints-1))**ndim

        DEALLOCATE(LeastSquares,Times) ! Array to store solution for fit points

        mGlobal=1

        IF (MPI_NP > 1) THEN
           CALL MPI_ALLREDUCE(AdvanceCoeffs, MeanAdvanceCoeffs, size(AdvanceCoeffs), MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, iErr)
           CALL MPI_ALLREDUCE(MPI_IN_PLACE, AdvanceGridTimes, size(AdvanceGridTimes), MPI_DOUBLE_PRECISION, MPI_SUM, MPI_COMM_WORLD, iErr)
           AdvanceGridTimes=AdvanceGridTimes/REAL(MPI_NP)
           MeanAdvanceCoeffs=MeanAdvanceCoeffs/REAL(MPI_NP)
           AdvanceCoeffs=MeanAdvanceCoeffs
        END IF
        AdvanceGridTimes=log(AdvanceGridTimes)

        IF (MPI_ID == 0 .AND. .NOT. lSkipProfile) THEN 
           AdvanceCoeffsTemp=0d0
           AdvanceCoeffsTemp(1)=AdvanceCoeffs(1)
           AdvanceCoeffsTemp(2:size(AdvanceCoeffs))=AdvanceCoeffs(2:size(AdvanceCoeffs))/AdvanceCoeffs(1)
           open(UNIT=PROFILE_DATA_HANDLE, file=PROFILE_DATA_FILE, status="unknown")
           WRITE(PROFILE_DATA_HANDLE, NML=ProfileData)
           WRITE(PROFILE_DATA_HANDLE, '(10E25.16)') AdvanceGridTimes
           CLOSE(PROFILE_DATA_HANDLE)
        END IF

     END IF
     IF (1 == 2) THEN
        CALL MPI_BARRIER(MPI_COMM_WORLD, iErr)
        write(*,*) "AdvanceCoeffs = ", AdvanceCoeffs
        DO i=0,10,1
           DO j=0,10,1
              DO k=1,1,1
                 IF (nDim == 2 .AND. k /= 1) CYCLE
                 mGlobal(1:3,2)=(/2**i,2**j,k/)
                 !                mGlobal(1:3,2)=(/i,j,k/)
                 CALL InitInfo(Info, 0, mGlobal)
                 CALL GridInit(Info)
                 tstart=MPI_Wtime()
                 CALL sweepAdvance(Info,mGlobal,lComplete=.true.,lProfile_opt=.true.)
                 temp=MPI_Wtime()-tstart
                 temp2=SimpleAdvanceCost(mGlobal(:,2))
                 temp2=AdvanceCost(mGlobal(:,2))
                 !                write(*,'(2I6,2E13.5,F13.5)') 2**i,2**j,temp,temp2, 2d0*((temp-temp2)/(temp+temp2))*100
                 write(*,'(2I6,2E13.5,F13.5)') 2**i,2**j, 2d0*((temp-temp2)/(temp+temp2))*100
                 !                write(*,*) i,j,temp2
                 !                write(*,*) i,j,temp2 !temp2/(Product(REAL(mglobal(1:nDim,2)))*AdvanceCoeffs(1))
                 CALL DestroyInfo(Info)
              END DO
           END DO
           write(*,*) 
        END DO
     END IF
     iThreaded = TempiThreaded


     AdvanceTimer%Accumulator(:)=0
     AdvancePredictor%Accumulator(:)=0
     MySpeedFactor=1d0
     !    CALL MPI_BARRIER(MPI_COMM_WORLD, ierr)
     !    STOP
  END SUBROUTINE ProfileAdvance


END MODULE HyperbolicControl


