!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    hyperbolic_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file hyperbolic_declarations.f90
!! @brief Main file for module HyperbolicDeclarations

!> @defgroup HyperbolicDeclarations Hyperbolic Declarations
!! @brief Contains variables and parameters used to initialize a hyperbolic solver.
!! @ingroup Hyperbolic

!> Contains variables and parameters used to initialize a hyperbolic solver.
!! @ingroup HyperbolicDeclarations

!============================================================================================
! Module Name:		HyperbolicDeclarations
! Module File:		hyperbolic_declarations.f90
! Purpose:			Contains variables and parameters used to initialize a hyperbolic solver.
! Created:			20100705 by Brandon D. Shroyer
!============================================================================================
MODULE HyperbolicDeclarations
  USE GlobalDeclarations
  USE DataDeclarations
    IMPLICIT NONE
    SAVE

	PUBLIC

    INTEGER :: iScheme          ! 0 = MUSCL scheme (MUSCLScheme)
                                ! 1 = Sweep scheme (SweepScheme)
                                
    INTEGER :: iSolver = 0      ! Choice of Riemann Solver
    INTEGER, PARAMETER :: MUSCL_SCHEME_ID = 0
    INTEGER, PARAMETER :: SWEEP_SCHEME_ID = 1
	
    INTEGER, PARAMETER :: SOLVER_DATA_HANDLE = 50   ! File handle for data file.
    CHARACTER(LEN=11), PARAMETER :: SOLVER_DATA_FILE = "solver.data" !A constant string containing the name of the solver's data file.

    NAMELIST/SolverData/iScheme,iSolver     ! This namelist will probably grow over time.

	! Global variables to be used in MUSCL and sweep schemes.
    REAL(Kind=qPREC), DIMENSION(:), ALLOCATABLE, PUBLIC :: maxspeed
    REAL(Kind=qPREC), DIMENSION(:), ALLOCATABLE, PUBLIC :: maxsolverspeed
    REAL(Kind=qPREC), DIMENSION(:), ALLOCATABLE, PUBLIC :: maxwavespeed


!    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: AdvanceClockByLevel

    LOGICAL, DIMENSION(:), ALLOCATABLE :: NodeCompleted  !Flag indicating whether nodelist for each level should be advanced.
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: AdvanceCoeffs    
    REAL(KIND=qPREC), DIMENSION(:,:,:), ALLOCATABLE :: AdvanceGridTimes

    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: tused_this_grid
    INTEGER, DIMENSION(:), ALLOCATABLE :: AdvanceStencil
    REAL(8), DIMENSION(:), ALLOCATABLE :: t_startadvance
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: WorkDoneByLevel
    REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: WorkDoneByGrid

   INTEGER(8), DIMENSION(:), ALLOCATABLE :: InternalCellUpdates !NumCellUpdatesByLevel, EffectiveCellUpdatesByLevel
   INTEGER(8), DIMENSION(:), ALLOCATABLE :: CellUpdates !NumCellUpdatesByLevel, EffectiveCellUpdatesByLevel


    INTEGER :: AdvanceState
    REAL(KIND=qPREC) :: tStopAdvance
    INTEGER, PARAMETER :: STOPPING = 0, RESUMING = 1, RUNNING = 2
    REAL(KIND=qPREC), PARAMETER :: FOREVER=huge(1d0)


CONTAINS

   FUNCTION SimpleAdvanceCost(mx)
      INTEGER :: mx(:)
      REAL(KIND=qPREC) :: SimpleAdvanceCost
      IF (ANY(mx < 1)) THEN
         SimpleAdvanceCost=0
         RETURN
      ELSE
         SimpleAdvanceCost=AdvanceCoeffs(1)*product(mx(:))
      END IF
   END FUNCTION SimpleAdvanceCost


   FUNCTION AdvanceCost(mx)
      INTEGER :: mx(:)
      REAL(KIND=qPREC) :: idx(3), AdvanceCost
      idx=log(real(mx, KIND=qPREC))/log(2d0)+1d0
      IF (ANY(mx < 1)) THEN
         AdvanceCost=0
         RETURN
      ELSEIF (ANY(idx > REAL(shape(AdvanceGridTimes)+.0001))) THEN
         AdvanceCost=(SimpleAdvanceCost(mx))
      ELSE
         IF (nDim == 1) THEN
            AdvanceCost=exp(Interp1D(idx(1), AdvanceGridTimes(:,1,1)))
         ELSEIF (nDim == 2) THEN
            AdvanceCost=exp(Interp2D(idx(1:2), AdvanceGridTimes(:,:,1)))
         ELSE
            AdvanceCost=exp(Interp3D(idx(1:3), AdvanceGridTimes(:,:,:)))
         END IF
      END IF
   END FUNCTION AdvanceCost

   

   FUNCTION Interp1D(x, array)
      REAL(KIND=qPREC) :: array(:), Interp1D, x
      INTEGER :: iLo, iHi
      iLo=min(max(floor(x),1),size(array)-1)
      iHI=iLo+1
      Interp1D=array(iLo)+(array(iHi)-array(iLo))*(x-REAL(iLo, KIND=qpREC))
   END FUNCTION Interp1D


   FUNCTION Interp2D(x, array)
      REAL(KIND=qPREC) :: x(2), array(:,:), Interp2D, fr(2)
      INTEGER :: iLo(2), iHi(2)
      iLo=min(max(floor(x),1),shape(array)-1)
      iHI=iLo+1
      fr(1)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iLo(2)))
      fr(2)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iHi(2)))
      Interp2D=Interp1D(x(2)-REAL(iLo(2))+1d0,fr(:))
   END FUNCTION Interp2D


   FUNCTION Interp3D(x, array)
      REAL(KIND=qPREC) :: x(3), array(:,:,:), Interp3D, fr(2,2), gr(2)
      INTEGER :: iLo(3), iHi(3)
      iLo=min(max(floor(x),1),shape(array)-1)
      iHI=iLo+1
      fr(1,1)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iLo(2),iLo(3)))
      fr(2,1)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iHi(2),iLo(3)))
      fr(1,2)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iLo(2),iHi(3)))
      fr(2,2)=Interp1D(x(1)-REAL(iLo(1))+1d0,array(iLo(1):iHi(1),iHi(2),iHi(3)))
      gr(1)=Interp1D(x(2)-REAL(iLo(2))+1d0,fr(:,1))
      gr(2)=Interp1D(x(2)-REAL(iLo(2))+1d0,fr(:,2))
      Interp3D=Interp1D(x(3)-REAL(iLo(3))+1d0, gr)
   END FUNCTION Interp3D




  FUNCTION ChildAdvanceCost(mB,level)
    INTEGER :: i
    INTEGER :: mB(3,2)
    INTEGER :: mx2(3), mx(3)
    INTEGER :: level
    REAL(KIND=qPREC) :: ChildAdvanceCost
    ChildAdvanceCost=0
    mx=mB(:,2)-mB(:,1)+1
    IF (ANY(mx < 1)) THEN
       RETURN
    ELSE
       mx2=1
       DO i=1, levels(level+1)%steps
          mx2(1:nDim)=levels(level)%CoarsenRatio*mx(1:nDim)+2d0*levels(level+1)%ambc(i)
          !mx2(1:nDim)=levels(level)%CoarsenRatio*mx(1:nDim)
          ChildAdvancecost=ChildAdvanceCost+AdvanceCost(mx2)
       END DO
    END IF
  END FUNCTION ChildAdvanceCost

 END MODULE HyperbolicDeclarations

