!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    riemann_solvers.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file riemann_solvers.f90
!! @brief Main file for module RiemannSolvers

!> @defgroup RiemannSolvers Riemann Solvers
!! @brief Module containing various Riemann Solvers
!! @ingroup Hyperbolic


!> Module containing various Riemann Solvers
!! @ingroup RiemannSolvers
MODULE RiemannSolvers
   USE GlobalDeclarations
   USE HyperbolicDeclarations
   USE PhysicsDeclarations
   USE EOS
   IMPLICIT NONE
   INTEGER, PARAMETER, PUBLIC :: iHLL=3, iHLLC=2, iHLLD=6, iExactRS=1, iHLLC_ISO=4, iHLLD_ISO=8, iHLL_MHD=7
CONTAINS

  !> Calculate flux based on solver selected
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter
  FUNCTION calc_flux(wl, wr, f, lambda_max)
    REAL(KIND=qPREC) :: calc_flux
    REAL(KIND=qPrec), DIMENSION(:) :: f, wl, wr
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    INTEGER :: i
    IF (iSolver == 0) THEN !select default solver
       IF (lMHD) THEN 
          IF (iEOS == EOS_ISOTHERMAL) THEN
             iSolver=iHLLD_ISO
          ELSE
             iSolver = iHLLD
          END IF
       ELSE
          IF (iEOS == EOS_ISOTHERMAL) THEN
             iSolver = iHLLC_ISO
          ELSE
             iSolver = iHLLC
          END IF
       END IF
    END IF
    IF (present(lambda_max)) THEN
       SELECT CASE (iSolver)
       CASE (iHLL)
          calc_flux=HLL(wl,wr,f,lambda_max)
       CASE(iHLL_MHD)
          calc_flux=HLL_MHD(wl,wr,f,lambda_max)
       CASE (iHLLC)
          calc_flux=HLLC(wl,wr,f,lambda_max)
       CASE (iHLLC_ISO)
          calc_flux=HLLC_ISO(wl,wr,f,lambda_max)
       CASE (iHLLD)
          calc_flux=HLLD(wl,wr,f,lambda_max)
       CASE (iHLLD_ISO)
          calc_flux=HLLD_ISO(wl,wr,f,lambda_max)
       CASE (iExactRS)
          calc_flux=ExactRS(wl,wr,f,lambda_max)
       END SELECT
    ELSE
       SELECT CASE (iSolver)
       CASE (iHLL)
          calc_flux=HLL(wl,wr,f)
       CASE(iHLL_MHD)
          calc_flux=HLL_MHD(wl,wr,f)
       CASE (iHLLC)
          calc_flux=HLLC(wl,wr,f)
       CASE (iHLLC_ISO)
          calc_flux=HLLC_ISO(wl,wr,f)
       CASE (iHLLD)
          calc_flux=HLLD(wl,wr,f)
       CASE (iHLLD_ISO)
          calc_flux=HLLD_ISO(wl,wr,f)
       CASE (iExactRS)
          calc_flux=ExactRS(wl,wr,f)
       END SELECT
    END IF
    DO i=1,size(f)
       IF (ISNAN(f(i))) THEN
          IF (.NOT. lRequestRestart) THEN
             write(*,*) 'processor', MPI_ID, 'requesting restart due to nan in flux'
             write(*,*) 'wl=', wl
             write(*,*) 'wr=', wr
          END IF
          lRequestRestart=.true.

!          write(*,*) 'found nan flux'
!          write(*,'(15E13.2)') f
!          write(*,'(15E13.2)') wl
!          write(*,'(15E13.2)') wr
!       STOP
       END IF

    END DO
  END FUNCTION CALC_FLUX

  !> Calculate Isothermal version of the HLLC flux
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter
  FUNCTION HLLC_ISO(wl,wr,f, lambda_max)
    !wl and wr = (/rho,vx,vy,vz,Bx,By,Bz/)
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: f, wl, wr
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: ul, ur, usl, usr, usc, fl, fr
    REAL(KIND=qPrec) :: SL, SR, DM, SM, MM, pTr, pTl,ml, mr, f2l, f2r, sqrtdl, sqrtdr, um
    INTEGER :: n
    REAL(KIND=qPREC) :: HLLC_ISO
    n=SIZE(wl)
    ALLOCATE(ul(n), ur(n), usl(n), usr(n), fl(n), fr(n))

    !Einfeldt version
    sqrtdl=SQRT(wl(1))
    sqrtdr=SQRT(wr(1))
    um=(sqrtdl*wl(2)+sqrtdr*wr(2))/(sqrtdl+sqrtdr)
    SL = MIN(wl(2), um) - Iso_Speed
    SR = MAX(wr(2), um) + Iso_Speed

    !Davis Version
    !  SL=min(wl(2),wr(2))-Iso_Speed
    !  SR=max(wl(2),wr(2))+Iso_Speed   


    ! H Viscosity protection
    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
    END IF

    IF (SL >= 0d0) THEN
       CALL F_prim_Iso(wl, f)
    ELSE IF (SR <= 0d0) THEN
       CALL F_prim_Iso(wr, f)
    ELSE !Solve for common star variables DM, SM, and f(1:2)
       pTr=wr(1)*Iso_Speed2
       pTl=wl(1)*Iso_Speed2
       mr=wr(1)*wr(2)
       ml=wl(1)*wl(2)
       f2r=mr*wr(2)+pTr
       f2l=ml*wl(2)+pTl
       DM = ((SR*wr(1)-SL*wl(1))-(mr-ml))/(SR-SL)
       MM = ((SR*mr-SL*ml)-(f2r-f2l))/(SR-SL)
       f(1)=((SR*ml-SL*mr)+SR*SL*(wr(1)-wl(1)))/(SR-SL)
       f(2)=((SR*f2l-SL*f2r)+SR*SL*(mr-ml))/(SR-SL)
       SM=f(1)/DM  !advective choice for middle speed (instead of MM/DM)

       IF (SM >= 0) THEN !Need to find sleft values
          usl(1:2)=(/   DM, &       
               MM /)
          usl(3:m_high)=DM*wl(3:m_high)
          usl(m_high+1:n)=DM*wl(m_high+1:n)/wl(1)

          CALL F_prim_Iso(wl,fl)
          CALL rprim_to_cons_Iso(wl, ul)
          fl = fl+SL*(usl-ul)          
          f=fl
       END IF
       IF (SM <= 0) THEN
          usr(1:2)=(/   DM, &       
               MM /)

          usr(3:m_high)=DM*wr(3:m_high)
          usr(m_high+1:n)=DM*wr(m_high+1:n)/wr(1)

          CALL F_prim_Iso(wr,fr)
          CALL rprim_to_cons_Iso(wr, ur)
          fr = fr+SR*(usr-ur)
          f=fr
       END IF
       IF (SM == 0d0) THEN
          f=.5d0*(fl+fr)
       END IF
          
    END IF
    HLLC_ISO=max(ABS(SL),ABS(SR))
    DEALLOCATE(ul, ur, usl, usr, fl, fr)
  END FUNCTION HLLC_ISO


  !> Calculate Isothermal version of the HLLD flux
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter
  FUNCTION HLLD_ISO(wl,wr,f, lambda_max)
    !wl and wr = (/rho,vx,vy,vz,Bx,By,Bz/)
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: f, wl, wr
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: ul, ur, usl, usr, usc
    REAL(KIND=qPrec) :: SL, SR, DM, SM, AM, MM, SsL, SsR, CFL, CFR, sqrtdr, sqrtdl, &
         pTr, pTl, Bx, Bx_2, temp, temp2, ml, mr, f2l, f2r, X, um, Bm(3), bx2, b2, a2b2, CFM, sbx, Xi
    INTEGER :: n, i
    REAL(KIND=qPREC) :: HLLD_ISO
    mr=0
    ml=0
    n=SIZE(wl)
    ALLOCATE(ul(n), ur(n), usl(n), usr(n), usc(n))
    Bx = half*(wr(5)+wl(5))
    CFL = fast_speed_Iso(wl)
    CFR = fast_speed_Iso(wr)      
    Bx_2 = Bx**2
    IF (Bx == 0) THEN
       sBx=0d0
    ELSE
       sBx=SIGN(1d0,Bx)
    END IF

    !Einfeldt version (more computationally expensive but more accurate)
    sqrtdl=SQRT(wl(1))
    sqrtdr=SQRT(wr(1))
    um=(sqrtdl*wl(2)+sqrtdr*wr(2))/(sqrtdl+sqrtdr)
    Bm=(sqrtdl*wl(5:7)+sqrtdr*wr(5:7))/(sqrtdl+sqrtdr)
    dm=sqrtdl*sqrtdr
    bx2=Bm(1)**2/dm;
    b2=DOT_PRODUCT(Bm(2:3),Bm(2:3))/dm+bx2     
    a2b2 = (Iso_Speed2+b2)/2d0
    CFM = sqrt((a2b2+sqrt(a2b2**2-Iso_Speed2*bx2)))

    SL = MIN(wl(2)-CFL, um-CFM)
    SR = MAX(wr(2)+CFR, um+CFM)

    !Davis Version
    !    SL=min(wl(2)-CFL,wr(2)-CFR)
    !    SR=max(wl(2)+CFL,wr(2)+CFR)


    ! H Viscosity protection
    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
    END IF


    IF (SL >= 0d0) THEN
       CALL F_prim_MHD_Iso(wl, f)
    ELSE IF (SR <= 0d0) THEN
       CALL F_prim_MHD_Iso(wr, f)
    ELSE !Solve for common star variables DM, SM, and f(1:2)
       pTr=wr(1)*Iso_Speed2+half*(Bx_2+DOT_PRODUCT(wr(6:7),wr(6:7)))
       pTl=wl(1)*Iso_Speed2+half*(Bx_2+DOT_PRODUCT(wl(6:7),wl(6:7)))
       mr=wr(1)*wr(2)
       ml=wl(1)*wl(2)
       f2r=mr*wr(2)+pTr-Bx_2
       f2l=ml*wl(2)+pTl-Bx_2
       DM = ((SR*wr(1)-SL*wl(1))-(mr-ml))/(SR-SL)
       MM = ((SR*mr-SL*ml)-(f2r-f2l))/(SR-SL)
       f(1)=((SR*ml-SL*mr)+SR*SL*(wr(1)-wl(1)))/(SR-SL)
       f(2)=((SR*f2l-SL*f2r)+SR*SL*(mr-ml))/(SR-SL)

       SM=f(1)/DM  !advective choice for middle speed (instead of MM/DM)
       AM=ABS(BX)/sqrt(DM) !star alfven speed
       SsL=SM-AM !Slow waves
       SsR=SM+AM !Slow waves
       IF (SsR >= 0) THEN !Need to find sleft values
          IF (miniscule(SL-SsL)) THEN
             temp2=0d0
             temp=1d0
          ELSE
             temp=1d0/((SL-SsL)*(SL-SsR))
             temp2=Bx*(SM-wl(2))*temp
             temp=(wl(1)*(SL-wl(2))**2 - Bx_2)*temp/DM
          END IF

          usl(1:7)=(/        DM, &       
               MM, &
               DM*wl(3)-temp2*wl(6), &
               DM*wl(4)-temp2*wl(7), &
               Bx, &
               wl(6)*temp, &
               wl(7)*temp &
               /) 
          usl(8:n) = DM*wl(8:n)/wl(1)                                        
       END IF
       IF (SsL <= 0) THEN !Need to find sright values
          IF (miniscule(SR-SsR)) THEN
             temp2=0d0
             temp=1d0
          ELSE
             temp=1d0/((SR-SsR)*(SR-SsL))
             temp2=Bx*(SM-wr(2))*temp
             temp=(wr(1)*(SR-wr(2))**2 - Bx_2)*temp/DM
          END IF


          usr(1:7)=(/        DM, &       
               MM, &
               DM*wr(3)-temp2*wr(6), &
               DM*wr(4)-temp2*wr(7), &
               Bx, &
               wr(6)*temp, &
               wr(7)*temp &
               /)                                         
          usr(8:n) = DM*wr(8:n)/wr(1)                                        
       END IF

       !Now need to find region
       IF (SsR >= 0 .AND. Ssl <= 0) THEN !in center region
          X=sbx*sqrt(DM) !sign(sqrt(DM),Bx)
          Xi=1d0/X
          IF (Bx==0) Xi=0
          usc(3:7) = (/ half*((usl(3)+usr(3))+X*(usr(6)-usl(6))), &
               half*((usl(4)+usr(4))+X*(usr(7)-usl(7))), &
               Bx, &
               half*((usr(6)+usl(6))+(usr(3)-usl(3))*Xi), &
               half*((usr(7)+usl(7))+(usr(4)-usl(4))*Xi) &
               /)

          f(3:7) = (/   usc(3)*SM-Bx*usc(6)     ,&
               usc(4)*SM-Bx*usc(7)     ,&
               0d0                     ,&
               usc(6)*SM-Bx*usc(3)/DM  ,&
               usc(7)*SM-BX*usc(4)/DM  &
               /)
          f(8:n)=((SR*wl(2)*wl(8:n)-SL*wr(2)*wr(8:n))+SR*SL*(wr(8:n)-wl(8:n)))/(SR-SL)
       ELSEIF (SsL >= 0) THEN !in sleft region
          CALL F_prim_MHD_Iso(wl,f)
          CALL rprim_to_cons_MHD_Iso(wl, ul)
          f = f+SL*(usl-ul)
       ELSE !in right region
          CALL F_prim_MHD_Iso(wr,f)
          CALL rprim_to_cons_MHD_Iso(wr, ur)
          f = f+SR*(usr-ur)
       END IF
    END IF
    f(5)=0d0 !kill normal magnetic field
    HLLD_ISO=max(ABS(SL),ABS(SR))
!    DO i=1,size(f)
!       IF (ISNAN(f(i))) THEN
!          write(*,*) 'HLLD_ISO returned a nan'
!          write(*,'(A,10E25.16)') 'wl=', wl
!          write(*,'(A,10E25.16)') 'wr=', wr
!          write(*,'(A,10E25.16)') 'f=', f
!          write(*,*) Sl, Ssl, SM, SsR, SR, ((SR*ml-SL*mr)+SR*SL*(wr(1)-wl(1)))/(SR-SL), DM, mr, ml
!          write(*,*) '============================='
!          STOP
!       END IF
!    END DO
    DEALLOCATE(ul, ur, usl, usr, usc)
  END FUNCTION HLLD_ISO


  !> Calculate HLLD flux for an ideal eos
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter 
  FUNCTION HLLD(wl,wr,f,lambda_max)
    REAL(KIND=qPREC) :: HLLD
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: f, wl, wr
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: ul, ur, rr, rl, rm, usl, usr, ussl, ussr, fl, fr
    REAL(KIND=qPrec) :: SL, SR, SM, SsL, SsR, sqrtdl,sqrtdr,sqrtdsl, sqrtdsr, CFL, CFR, CFM, &
         pTr, pTl, pT, Bx, Bx_2, dsr, dsl, temp, Brat, vrat, tempr, templ, a2, aT2, b2, X, sumsqrtds, sBx
    REAL(KIND=qPrec), DIMENSION(3) :: vss, Bss
    REAL(KIND=qPrec), DIMENSION(2) :: db
    INTEGER :: n,i
    n=SIZE(wl)
    ALLOCATE(ul(n), ur(n), rr(n), rl(n), rm(n), usl(n), usr(n), ussl(n), ussr(n), fl(n), fr(n))
    CALL rprim_to_cons_MHD(wl,ul)
    CALL rprim_to_cons_MHD(wr,ur)

    Bx = half*(wr(6)+wl(6))
    IF (Bx == 0) THEN
       sBx=0d0
    ELSE
       sBx=SIGN(1d0,Bx)
    END IF
    sqrtdl=SQRT(wl(1))
    sqrtdr=SQRT(wr(1))      
    CFL = fast_speed(wl)
    CFR = fast_speed(wr)      
    Bx_2 = Bx**2

    !Einfeldt version (more computationally expensive but more accurate)
    CALL prim_to_Roe_MHD(wl,rl)
    CALL prim_to_Roe_MHD(wr,rr)
    sqrtdl=SQRT(wl(1))
    sqrtdr=SQRT(wr(1))
    rm(2:)=(sqrtdl*rl(2:)+sqrtdr*rr(2:))/(sqrtdl+sqrtdr)
    rm(1)=sqrtdl*sqrtdr
    CFM=fast_speed_Roe(rm) 
    SL = MIN(wl(3)-CFL, rm(3)-CFM)
    SR = MAX(wr(3)+CFR, rm(3)+CFM)

    !Davis Version
!        SL=min(wl(3)-CFL,wr(3)-CFR)
!        SR=max(wl(3)+CFL,wr(3)+CFR)

    !protection to ensure proper behavior when flow is supersonic


    ! H Viscosity protection
    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
!    write(*,'(A,4E25.16)') 'A', SL, SR, CFL, CFR
!    write(*,'(A,4E25.16)') 'B', rm(3), CFM
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
!       write(*,'(A,4E25.16)') 'C', SL, SR, lambda_max
    END IF

    IF (SL >= 0d0) THEN  !Region 1
       CALL F_prim_MHD(wl, f)
    ELSE IF (SR <= 0d0) THEN !Region 6
       CALL F_prim_MHD(wr, f)
    ELSE !Regions 2-5
       pTr=wr(2)+half*DOT_PRODUCT(wr(6:8),wr(6:8))
       pTl=wl(2)+half*DOT_PRODUCT(wl(6:8),wl(6:8))
       tempr=(SR-wr(3))*wr(1)
       templ=(SL-wl(3))*wl(1)
       SM = ((tempr*wr(3)-templ*wl(3))-(pTr-pTl))/(tempr - templ)
       pT = ((tempr*pTl - templ*pTr) + tempr*templ*(wr(3)-wl(3)))/(tempr-templ)
       dsl=templ/(SL-SM)
       dsr=tempr/(SR-SM)
       sqrtdsl = SQRT(dsl)
       sqrtdsr = SQRT(dsr)
       SsL=SM-ABS(Bx)/sqrtdsl
       SsR=SM+ABS(Bx)/sqrtdsr  
       IF (SsL <= 0.d0) THEN !Regions !3-5
          IF (tempr*(SR-SM) == Bx_2) THEN               
             vrat = 0d0
             Brat = 1d0
          ELSE
             temp = 1.d0/(tempr*(SR-SM)-Bx_2)
             vrat = (SM-wr(3))*temp
             Brat = (tempr**2/wr(1) - Bx_2)*temp
          END IF
          usr(1:8) = (/dsr, &
               0.d0, &
               dsr*SM, &
               dsr*(wr(4)-Bx*wr(7)*vrat), &
               dsr*(wr(5)-Bx*wr(8)*vrat), &
               Bx, &
               wr(7)*Brat, &
               wr(8)*Brat /)
          usr(2) = ((SR-wr(3))*ur(2)-pTr*wr(3)+pT*SM+Bx* &
               (DOT_PRODUCT(wr(3:5),wr(6:8))-DOT_PRODUCT(usr(3:5),usr(6:8))/dsr))/(SR-SM)
          usr(9:n) = dsr*wr(9:n)/wr(1)
       END IF
       IF (SsR >= 0d0) THEN !Regions 2-4
          IF (templ*(SL-SM) == Bx_2) THEN
             vrat=0.0
             Brat=1d0
          ELSE
             temp = 1.d0/(templ*(SL-SM)-Bx_2)
             vrat = (SM-wl(3))*temp
             Brat = (templ**2/wl(1) - Bx_2)*temp
          END IF
          usl(1:8) = (/dsl, &
               0.d0, &
               dsl*SM, &
               dsl*(wl(4)-Bx*wl(7)*vrat), &
               dsl*(wl(5)-Bx*wl(8)*vrat), &
               Bx, &
               wl(7)*Brat, &
               wl(8)*Brat /)
          usl(2) =   ((SL-wl(3))*ul(2)-pTl*wl(3)+pT*SM+Bx* &
               (DOT_PRODUCT(wl(3:5),wl(6:8))-DOT_PRODUCT(usl(3:5),usl(6:8))/dsl))/(SL-SM)
          usl(9:n) = dsl*wl(9:n)/wl(1)
       END IF
       IF (SsL <= 0.d0 .AND. SsR >= 0.d0) THEN !Regions 3-4
          sumsqrtds = sqrtdsl+sqrtdsr
          vss = (/SM, &
               (usl(4)/sqrtdsl+usr(4)/sqrtdsr + (usr(7)-usl(7))*sbx)/(sumsqrtds), &
               (usl(5)/sqrtdsl+usr(5)/sqrtdsr + (usr(8)-usl(8))*sbx)/(sumsqrtds)/)
          Bss = (/Bx, &                     
               (sqrtdsl*usr(7)+sqrtdsr*usl(7)+(sqrtdsl/sqrtdsr*usr(4)-sqrtdsr/sqrtdsl*usl(4))*sbx)/ &
               sumsqrtds, &
               (sqrtdsl*usr(8)+sqrtdsr*usl(8)+(sqrtdsl/sqrtdsr*usr(5)-sqrtdsr/sqrtdsl*usl(5))*sbx)/ &
               sumsqrtds/)
       END IF

       IF (SM >= 0.d0) THEN           !Regions 2-3
          CALL F_prim_MHD(wl,fl)
          IF (SsL >= 0.d0) THEN       !Region 2
             fl = fl+SL*(usl-ul)
          ELSE                        !Region 3
             ussl(1:8) = (/usl(1), &
                  usl(2)-sbx*(DOT_PRODUCT(usl(3:5),usl(6:8))/sqrtdsl-sqrtdsl*DOT_PRODUCT(vss,Bss)), &                     
                  usl(1)*vss(1), &
                  usl(1)*vss(2), &
                  usl(1)*vss(3), &                   
                  Bss(1), &
                  Bss(2), &
                  Bss(3)/)
             ussl(9:n)=usl(9:n)
             fl = fl+SsL*ussl - (SsL-SL)*usl-SL*ul
          END IF
          f=fl
       END IF
       IF (SM <= 0.d0) THEN       !Regions 4-5
          CALL F_prim_MHD(wr,fr)
          IF (SsR <= 0.d0) THEN       !Region 5
             fr = fr+SR*(usr-ur)
          ELSE                        !Region 4
             ussr(1:8) = (/usr(1), &
                  usr(2)+sbx*(DOT_PRODUCT(usr(3:5),usr(6:8))/sqrtdsr-sqrtdsr*DOT_PRODUCT(vss,Bss)), &
                  usr(1)*vss(1), &
                  usr(1)*vss(2), &
                  usr(1)*vss(3), &
                  Bss(1), &
                  Bss(2), &
                  Bss(3)/)
             ussr(9:n)=usr(9:n)
             fr = fr+SsR*ussr - (SsR-SR)*usr-SR*ur
          END IF
          f=fr
       END IF
       IF (SM == 0.d0) THEN
          f=.5d0*(fl+fr)
       END IF
    END IF
    f(6)=0d0 !kill normal magnetic flux
    HLLD = MAX(ABS(SL),ABS(SR))!, ABS(wleft(2))+CL, ABS(wright(2))+CR)
    DEALLOCATE(ul, ur, rr, rl, rm, usl, usr, ussl, ussr, fl, fr)

  END FUNCTION HLLD

  !> Calculate HLLC flux for an ideal gas eos
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter
  FUNCTION HLLC(wl,wr,f, lambda_max)
    REAL(KIND=qPREC) :: HLLC
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: wl, wr, f 
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: ul, ur, rl, rr, rm, usl, usr, fl, fr
    REAL(KIND=qPrec) :: SL, SR, SM, sqrtdl,sqrtdr, CL, CR, pT, tempr, templ, a, dsl, dsr
    INTEGER :: n, x(3)
    n = SIZE(f)

    ALLOCATE(ur(n), ul(n), rl(n), rr(n), rm(n), usl(n), usr(n), fl(n), fr(n))
    CL=sound_speed(wl)
    CR=sound_speed(wr)

    !Einfeldt version
    CALL prim_to_Roe(wl,rl)
    CALL prim_to_Roe(wr,rr)
    sqrtdl=SQRT(wl(1))
    sqrtdr=SQRT(wr(1))
    rm(2:)=(sqrtdl*rl(2:)+sqrtdr*rr(2:))/(sqrtdl+sqrtdr)
    rm(1)=sqrtdl*sqrtdr
    a=sound_speed_Roe(rm) 
    SL = MIN(wl(3)-CL, rm(3)-a)
    SR = MAX(wr(3)+CR, rm(3)+a)

    !Davis Version
    !  SL=min(wl(3)-CL,wr(3)-CR)
    !  SR=max(wl(3)+CL,wr(3)+CR)

    !protection to ensure proper behavior when flow is supersonic
    !  IF (present(lambda_max)) THEN
    !     IF (lambda_max <. SR .OR. lambda_max <  -SL) write(*,*) SL, SR, lambda_max
    !  END IF

    ! H Viscosity protection
    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
    END IF


    tempr = (SR-wr(3))*wr(1)
    templ = (SL-wl(3))*wl(1)
    SM = ((tempr*wr(3)-templ*wl(3))-(wr(2)-wl(2)))/(tempr - templ)
    pT = ((tempr*wl(2)-templ*wr(2))+tempr*templ*(wr(3)-wl(3)))/ &
         (tempr-templ)
    IF (SL >= 0d0) THEN
       CALL F_prim(wl, f)
    ELSE IF (SR <= 0d0) THEN
       CALL F_prim(wr, f)
    ELSE 
       IF (SM >= 0d0) THEN
          CALL F_prim(wl, fl)
          CALL rprim_to_cons(wl,ul)
          dsl = templ/(SL-SM)
          usl(1:3)=(/dsl,((SL-wl(3))*ul(2)-wl(2)*wl(3)+pT*SM)/(SL-SM),dsl*SM/)
          usl(4:m_high+1)=dsl*wl(4:m_high+1)
          usl(m_high+2:n)=dsl*wl(m_high+2:n)/wl(1)
          fl = fl+SL*(usl-ul)
          f=fl
       END IF
       IF (SM <= 0d0) THEN
          CALL F_prim(wr,fr)
          CALL rprim_to_cons(wr,ur)
          dsr=tempr/(SR-SM)
          usr(1:3)=(/dsr,((SR-wr(3))*ur(2)-wr(2)*wr(3)+pT*SM)/(SR-SM),dsr*SM/)
          usr(4:m_high+1)=dsr*wr(4:m_high+1)
          usr(m_high+2:n)=dsr*wr(m_high+2:n)/wr(1)
          fr= fr + SR*(usr-ur)
          f=fr
       END IF
       IF (SM == 0d0) THEN
          f=.5d0*(fr+fl)
       END IF
    END IF
    !      IF (MAX(ABS(SL), ABS(SR)) > 90000) THEN
    !         write(*,*) wl, wr, CL, CR, SL, SR
    !      END IF
    HLLC = max(ABS(SL),ABS(SR))
    DEALLOCATE(ur, ul, rl, rr, rm, usl, usr, fl, fr)
  END FUNCTION HLLC

  !> Calculate HLL flux for an ideal EOS
  !! @param wleft left primitive state
  !! @param wright right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter

  FUNCTION HLL(wleft,wright,f, lambda_max)
    REAL(KIND=qPREC) :: HLL
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: wleft, wright, f
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: uleft, uright, fl, fr, rl, rr, rm
    REAL(KIND=qPrec) :: SL, SR, sqrtdl,sqrtdr, CL, CR, a
    INTEGER :: n
    n=size(f)
    ALLOCATE(uleft(n), uright(n), fl(n), fr(n), rl(n), rr(n), rm(n))
    CALL rprim_to_cons(wleft,uleft)
    CALL rprim_to_cons(wright,uright)
    sqrtdl=SQRT(wleft(1))
    sqrtdr=SQRT(wright(1))
    CL=sound_speed(wleft)
    CR=sound_speed(wright)

    !Einfeldt version
    CALL prim_to_Roe(wleft,rl)
    CALL prim_to_Roe(wright,rr)
    sqrtdl=SQRT(wleft(1))
    sqrtdr=SQRT(wright(1))
    rm(2:)=(sqrtdl*rl(2:)+sqrtdr*rr(2:))/(sqrtdl+sqrtdr)
    rm(1)=sqrtdl*sqrtdr
    a=sound_speed_Roe(rm) 
    SL = MIN(wleft(3)-CL, rm(3)-a)
    SR = MAX(wright(3)+CR, rm(3)+a)

    !Davis Version
    !  SL=min(wleft(3)-CL,wright(3)-CR)
    !  SR=max(wleft(3)+CL,wright(3)+CR)

    !protection to ensure proper behavior when flow is supersonic

    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
    END IF

    IF (SL >= 0d0) THEN
       CALL F_prim(wleft, f)
    ELSE IF (SR <= 0d0) THEN
       CALL F_prim(wright,f)
    ELSE
       CALL rprim_to_cons(wleft,uleft)
       CALL rprim_to_cons(wright,uright)
       CALL F_prim(wleft, fl)
       CALL F_prim(wright, fr)
       f = ((SR*fl-SL*fr)+SR*SL*(uright-uleft))/(SR-SL)
    END IF
    HLL = MAX(ABS(SL),ABS(SR))
    DEALLOCATE(uleft, uright, fl, fr, rl, rr, rm)
  END FUNCTION HLL

  !> Calculate HLL flux for an ideal EOS
  !! @param wleft left primitive state
  !! @param wright right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter

  FUNCTION HLL_MHD(wleft,wright,f, lambda_max)
    REAL(KIND=qPREC) :: HLL_MHD
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: wleft, wright, f
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: uleft, uright, fl, fr, rl, rr, rm
    REAL(KIND=qPrec) :: SL, SR, sqrtdl,sqrtdr, CL, CR, a
    INTEGER :: n
    n=size(f)
    ALLOCATE(uleft(n), uright(n), fl(n), fr(n), rl(n), rr(n), rm(n))
    CALL rprim_to_cons_MHD(wleft,uleft)
    CALL rprim_to_cons_MHD(wright,uright)
    sqrtdl=SQRT(wleft(1))
    sqrtdr=SQRT(wright(1))
    CL=fast_speed(wleft)
    CR=fast_speed(wright)

    !Einfeldt version
    CALL prim_to_Roe_MHD(wleft,rl)
    CALL prim_to_Roe_MHD(wright,rr)
    sqrtdl=SQRT(wleft(1))
    sqrtdr=SQRT(wright(1))
    rm(2:)=(sqrtdl*rl(2:)+sqrtdr*rr(2:))/(sqrtdl+sqrtdr)
    rm(1)=sqrtdl*sqrtdr
    a=fast_speed_Roe(rm) 
    SL = MIN(wleft(3)-CL, rm(3)-a)
    SR = MAX(wright(3)+CR, rm(3)+a)

    !Davis Version
    !  SL=min(wleft(3)-CL,wright(3)-CR)
    !  SR=max(wleft(3)+CL,wright(3)+CR)

    !protection to ensure proper behavior when flow is supersonic

    ! Sanders 9a
!    IF (present(lambda_max)) THEN
!       SL=SIGN(max(abs(SL),lambda_max), SL)
!       SR=SIGN(max(abs(SR),lambda_max), SR)
!    END IF

    ! Sanders 9b
    IF (present(lambda_max)) THEN
       SL=SIGN(abs(SL)+lambda_max, SL)
       SR=SIGN(abs(SR)+lambda_max, SR)
    END IF

    IF (SL >= 0d0) THEN
       CALL F_prim_MHD(wleft, f)
    ELSE IF (SR <= 0d0) THEN
       CALL F_prim_MHD(wright,f)
    ELSE
!       CALL rprim_to_cons(wleft,uleft)
!       CALL rprim_to_cons(wright,uright)
       CALL F_prim_MHD(wleft, fl)
       CALL F_prim_MHD(wright, fr)
       f = ((SR*fl-SL*fr)+SR*SL*(uright-uleft))/(SR-SL)
    END IF
    HLL_MHD = MAX(ABS(SL),ABS(SR))
    DEALLOCATE(uleft, uright, fl, fr, rl, rr, rm)
 END FUNCTION HLL_MHD

  !> Calculate the exact solution given an ideal EOS
  !! @param wl left primitive state
  !! @param wr right primitive state
  !! @param f flux
  !! @param lambda_max optional H-viscosity parameter
  FUNCTION ExactRS(wl, wr, f, lambda_max)
    REAL(KIND=qPREC) :: ExactRS,max_speed
    REAL(KIND=qPREC), OPTIONAL :: lambda_max
    REAL(KIND=qPrec), DIMENSION(:) :: wl, wr, f
    REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: wstar
    ALLOCATE (wstar(SIZE(wl)))   
    CALL riemann(wl,wr,wstar,max_speed)
    CALL f_prim(wstar,f)
    ExactRS=max_speed
    DEALLOCATE(wstar)
  END FUNCTION ExactRS



  !> Calculate interface state for exact riemann solver
  !! @param WLeft left primitive state
  !! @param WRight right primitive state
  !! @param WMiddle interface primitive state
  SUBROUTINE riemann(WLeft,WRight,WMiddle,max_speed)
    REAL(KIND=qPrec), DIMENSION(:) :: WLeft, WRight, WMiddle
    REAL(KIND=qPrec), DIMENSION(3) :: Left, Right, Middle    
    REAL(KIND=qPrec) :: S, UM,max_speed
    S=0.0
    Left(1:3)=WLeft(1:3)
    Right(1:3)=WRight(1:3)
    CALL vacuum_solve(Left, Right, Middle, UM, S,max_speed)
    Wmiddle(1:3)=Middle(1:3)
    IF (SIZE(WLeft) >= 4) THEN
       IF (WMiddle(3) >= 0) THEN
          WMiddle(4:)=WLeft(4:)
       ELSE
          WMiddle(4:)=WRight(4:)
       END IF
    END IF
  END SUBROUTINE riemann

  !> Routine for solving for interface state
  !! @param Left Left primitive state
  !! @param Right Right primitive state
  !! @param Middle interface primitive state
  !! @param UM velocity of interface state
  !! @param S Sample speed
  SUBROUTINE  vacuum_solve(Left, Right, Middle, UM, S, max_speed)
    USE GlobalDeclarations
    REAL(KIND=qPrec), DIMENSION(3) :: Left, Right, Middle, Vacuum
    REAL(KIND=qPrec) :: S, CL, CR, PM, UM, SHL, SHR, UML, UMR, C,max_speed
    INTEGER :: solver
    Vacuum = (/REAL(0, KIND=qPrec), REAL(0,KIND=qPrec), REAL(0,KIND=qPrec) /)
    Right(1)=MAX(Right(1),0d0)
    Left(1)=MAX(Left(1),0d0)
    Right(2)=MAX(Right(2),MinDensity*MinTemp)
    Left(2)=MAX(Left(2),MinDensity*MinTemp)

    IF (Left(1) > MinDensity .AND. Right(1) > MinDensity) THEN !check for vacuum generation
       CL = SQRT(gamma*Left(2)/Left(1))
       CR = SQRT(gamma*Right(2)/Right(1))
       IF(gamma2*(CL+CR) <= Right(3)-Left(3)) THEN
          solver = 1
          SHR = Right(3)+CR
          SHL = Left(3)-CL
          UMR = Right(3)-gamma2*CR
          UML = Left(3)+gamma2*CL
          max_speed = MAX(ABS(SHR), ABS(SHL))
          IF (S<=SHL) THEN
             Middle = Left
          ELSE IF (S <= UML) THEN
             C=gamma9*(CL+gamma12*(Left(3)-S))
             Middle=(/Left(1)*(C/CL)**gamma2, Left(2)*(C/CL)**gamma10, gamma9*(CL+gamma12*Left(3)+S)/)
          ELSE IF (S <= UMR) THEN
             Middle = Vacuum
          ELSE IF (S < SHR) THEN
             C=gamma9*(CR-gamma12*(Right(3)-S))
             Middle = (/Right(1)*(C/CR)**gamma2, Right(2)*(C/CR)**gamma10,gamma9*(-CR+gamma12*Right(3)+S) /)
          ELSE
             Middle = Right
          END IF
       ELSE
          solver = 2
          CALL STARPU(Left,Right, CL, CR, PM, UM) 
          CALL SAMPLE(PM, UM, S, Left, Right, Middle, CL, CR, max_speed)
       END IF

    ELSE IF (Right(1) >  MinDensity) THEN 
       solver = 3
       CR = SQRT(gamma*Right(2)/Right(1))
       SHR = Right(3)+CR
       UM = Right(3)-gamma2*CR
       max_speed = MAX(abs(SHR), abs(UM))
       IF (S <= UM) THEN ! Sampled point in vacuum state
          Middle = Vacuum
       ELSE IF (S < SHR) THEN ! Sampled point in right fan
          C=gamma9*(CR-gamma12*(Right(3)-S))
          Middle = (/Right(1)*(C/CR)**gamma2, Right(2)*(C/CR)**gamma10,gamma9*(-CR+gamma12*Right(3)+S) /)
       ELSE !Sampled point in right state
          Middle = Right
       END IF

    ELSE IF (Left(1) > MinDensity) THEN
       solver = 4
       CL = SQRT(gamma*Left(2)/Left(1))
       SHL = Left(3)-CL
       UM = Left(3)+gamma2*CL
       max_speed = MAX(abs(SHL), abs(UM))
       IF (S >= UM) THEN ! Sampled point in vacuum state
          Middle = Vacuum
       ELSE IF (S > SHL) THEN ! Sampled point in left fan
          C=gamma9*(CL+gamma12*(Left(3)-S))
          Middle=(/Left(1)*(C/CL)**gamma2, Left(2)*(C/CL)**gamma10,gamma9*(CL+gamma12*Left(3)+S)/)
       ELSE !Sampled point in left state
          Middle = Left
       END IF

    ELSE !both vacuum
       solver = 5
       Middle=Vacuum
       max_speed=0d0
    END IF

    IF (Middle(1)+Middle(2)+Middle(3) < 1e20) THEN
    ELSE
       write(*,*)  Left, Right, Middle, solver
       STOP
    END IF
  END SUBROUTINE vacuum_solve

  !> Calculate velocity of interface state
  SUBROUTINE STARPU(Left, Right, CL, CR, PM, UM)
    REAL(KIND=qPrec), DIMENSION(3) :: Left, Right
    REAL(KIND=qPrec) :: CL, CR, CHANGE, FL, FLD, FR, FRD, POLD, PSTART, UDIFF, UM, PM
    INTEGER I, NRITER
    NRITER=100
    CALL guessp(Left, Right, CL, CR, PSTART)
    POLD = PSTART
    UDIFF = Right(3)-Left(3)
    DO I=1,NRITER
       CALL PREFUN(FL, FLD, POLD, Left, CL)
       CALL PREFUN(FR, FRD, POLD, Right, CR)
       PM=POLD-(FL+FR+UDIFF)/(FLD+FRD)
       IF (PM <= 0.0) PM=MinDensity*MinTemp
       CHANGE=2.0*ABS((PM-POLD)/(PM+POLD))
       IF (CHANGE <= 1e-10) EXIT
       IF (I == NRITER ) THEN
          write(*,*)  I
          write(*,*)  "Left=", Left
          write(*,*)  "Right=", Right
          write(*,*)  "PSTART=", PSTART
          write(*,*)  "PM=", PM
          write(*,*)  "POLD=", POLD
          write(*,*)  "Change=", CHANGE
          write(*,*)  "CL=", CL
          write(*,*)  "CR=", CR
          write(*,*)  "STARPU reached max iterations in riemann_solvers.f90"
          STOP
       END IF
       POLD = PM
    END DO
    UM=half*(Left(3)+Right(3)+FR-FL)
  END SUBROUTINE STARPU

  !> No idea
  SUBROUTINE SAMPLE(PM, UM, S, Left, Right, Middle, CL, CR, max_speed)
    REAL(KIND=qPrec), DIMENSION(3) :: Left, Right, Middle, MiddleR, MiddleL
    REAL(KIND=qPrec) :: DL, UL, PL, CL, DR, UR, PR, CR, P, D, U, S, UM, PM
    REAL(KIND=qPrec) :: C, CML, CMR, PML, PMR, SHL, SHR, SL, SR, STL, STR,Smax
    REAL(KIND=qPREC) :: max_speed
    SHL = Left(3) - CL
    SHR = Right(3) + CR
    PML = PM/Left(2)
    PMR = PM/Right(2)
    SL = Left(3) - CL*SQRT(gamma8*PML+gamma3)
    SR = Right(3) + CR*SQRT(gamma8*PMR+gamma3)
    max_speed=0d0
    IF (PM<=Left(2)) THEN  !Left rarefaction
       max_speed=MAX(max_speed,ABS(SHL))  
    ELSE                   !Left shock
       max_speed=MAX(max_speed,ABS(SL))
    END IF
    IF (PM<=Right(2)) THEN !Right rarefaction
       max_speed=MAX(max_speed,ABS(SHR))
    ELSE                   !Right shock
       max_speed=MAX(max_speed,ABS(SR))
    END IF
    max_speed=MAX(max_speed,ABS(UM)) !Entropy wave

    IF (S <= UM) THEN         !Sample left of contact
       IF (PM <= Left(2)) THEN   !Left rarefaction
          IF (S <= SHL) THEN        !Sample in left state
             Middlel = Left
          ELSE                  
             CML=CL*(PML)**gamma3
             STL=UM-CML
             IF (S > STL) THEN       !Sample in star left state
                Middlel = (/Left(1)*(PML)**(gamma6), PM, UM /)
             ELSE                    !Sample in left fan
                C=gamma9*(CL+gamma12*(Left(3)-S))
                Middlel=(/Left(1)*(C/CL)**gamma2, Left(2)*(C/CL)**gamma10, gamma9*(CL+gamma12*Left(3)+S)/)
             END IF
          END IF
       ELSE                      !Left shock
          IF (S <= SL) THEN         !Sample in left state
             Middlel = Left
          ELSE                      !Sample in star left state
             Middlel = (/Left(1)*(PML+gamma11)/(PML*gamma11+1d0), PM, UM/)
          END IF
       END IF
       middle=middlel
    END IF
    IF (S >= UM) THEN         !Sample right of contact
       IF (PM > Right(2)) THEN   !Right shock
          IF (S >= SR) THEN         !Sample in right state
             MiddleR = Right
          ELSE                      !Sample in star right state
             MiddleR = (/Right(1)*(PMR+gamma11)/(PMR*gamma11+1d0), PM, UM/)
          END IF
       ELSE                      !Right rarefaction
          IF (S >= SHR) THEN        !Sample in right state
             MiddleR = Right
          ELSE
             CMR = CR*(PMR)**gamma3
             STR = UM + CMR
             IF (S <= STR) THEN      !Sample in star right state
                MiddleR = (/Right(1)*(PMR)**(gamma6), PM, UM/)
             ELSE                    !Sample in right fan
                C=gamma9*(CR-gamma12*(Right(3)-S))
                MiddleR = (/Right(1)*(C/CR)**gamma2,Right(2)*(C/CR)**gamma10, gamma9*(-CR+gamma12*Right(3)+S) /)
             END IF
          END IF
       END IF
       middle=middleR
    END IF
    IF (S == UM) THEN
       Middle=.5d0*(middleL+middleR)
    END IF
  END SUBROUTINE SAMPLE

  !> Completely clueless
  SUBROUTINE guessp(Left,Right,CL,CR,PM)
    REAL(KIND=qPrec) :: PM, PQ, UM, GEL, GER, CL, CR, UDIFF, PPV, PMIN, PMAX, QMAX, QUSER, PTL, PTR
    REAL(KIND=qPrec), DIMENSION(3) :: Left, Right
    QUSER=2.0
    UDIFF=Right(3)-Left(3)
    PPV=half*(Left(2)+Right(2))-0.125*UDIFF*(Left(1)+Right(1))*(CL+CR)
    PMIN=MIN(Left(2),Right(2))
    PMAX=MAX(Left(2),Right(2))
    QMAX=PMAX/PMIN
    IF (QMAX <= QUSER .AND. (PMIN <=PPV .AND. PPV <= PMAX)) THEN !PVRS
       PM=PPV  
    ELSE
       IF (PPV < PMIN) THEN                                       !TRRS
          PQ=(Left(2)/Right(2))**gamma3
          UM=(PQ*Left(3)/CL+Right(3)/CR+gamma2*(PQ-1d0))/(PQ/CL+1d0/CR)
          PTL=1d0+gamma12*(Left(3)-UM)/CL
          PTR=1d0+gamma12*(UM-Right(3))/CR
          PM=half*(Left(2)*PTL**gamma10+Right(2)*PTR**gamma10)
       ELSE                                                       !TSRS
          GEL=SQRT((gamma9/Left(1))/(gamma11*Left(2)+PPV))
          GER=SQRT((gamma9/Right(1))/(gamma11*Right(2)+PPV))
          PM=(GEL*Left(2)+GER*Right(2)-UDIFF)/(GEL+GER)
       END IF
    END IF
    PM = MAX(PM,MinDensity*MinTemp)                                   
  END SUBROUTINE guessp

  !> Called before having any fun
  SUBROUTINE PREFUN(F, FD, Pstar, WK, CK)
    REAL(KIND=qPrec) :: Pstar, F, FD, CK, PRAT, AK, BK, QRT
    REAL(KIND=qPrec), DIMENSION(3) :: WK
    IF (Pstar <= WK(2)) THEN
       PRAT=Pstar/WK(2)
       F=gamma2*CK*(PRAT**gamma3-1.0)
       FD=(1.0/(WK(1)*CK))*PRAT**(-gamma8)
    ELSE
       AK=gamma9/WK(1)
       BK=gamma11*WK(2)
       QRT=SQRT(AK/(BK+Pstar))
       F=(Pstar-WK(2))*QRT
       FD=(1d0-half*(Pstar-WK(2))/(BK+Pstar))*QRT
    END IF
  END SUBROUTINE PREFUN

END MODULE RiemannSolvers

