!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    stencil_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE StencilControl
USE GlobalDeclarations
USE StencilDeclarations
USE SchemeDeclarations
USE Scheduling
# if defined PTH
     USE PthDeclarations
# endif
IMPLICIT NONE
CONTAINS
!> @file stencil_control.f90 
!! @brief Module for managing stencil dependencies


!> Sets up stencil dependencies and ranges
SUBROUTINE setup()
  INTEGER :: dom_range(3,2), i
  CALL sweep_()
  CALL SetStencilNrVars
  CALL sort_dependencies()
  Stencil(aftersweepstep)%lead=0
  Stencil(aftersweepstep)%trail=0
  Stencil(aftersweepstep)%range=0
  CALL set_dependencies()
  CALL set_codependencies()
  CALL finalize_dependencies()
  CALL set_descriptions()      
  hyperbolic_mbc=Stencil(q)%range(1,2)
!  IF (MPI_ID == 0) CALL print_stencils
END SUBROUTINE setup


SUBROUTINE Clear_ranges()
   INTEGER :: dom_range(3,2)
   INTEGER :: i
   DO i=1,nStencils
      Stencil(slist(i))%range=RESHAPE( (/100,100,100,-100,-100,-100/), (/3,2/) )
   END DO
   dom_range=square((/0,0,0,0,0,0/))
   Stencil(aftersweepstep)%range=dom_range
   Stencil(aftersweepstep)%lead=0
   Stencil(aftersweepstep)%trail=0
END SUBROUTINE Clear_ranges

!> Writes stencil pieces to standard out
SUBROUTINE print_stencils
  INTEGER :: i
  TYPE(StencilDef), POINTER :: a
  write(*,'(A16,6A4,2A7,A6)') 'Description      ','  x1','  y1','  z1','  x2','  y2','  z2','|  rank','   lead','   lag'
  DO i=1, nStencils
     a=>Stencil(slist(i))
     write(*,'(A16,6I4,A2,I4,A1,3I6)') a%description, a%range, "| ", a%rank, " ", a%lead, a%lead-a%trail, slist(i)
  END DO
END SUBROUTINE print_stencils




   !> Sorts dependencies
   SUBROUTINE sort_dependencies()
      INTEGER :: i,j,k,l,m,next_rank
      INTEGER :: a, inc
      !sort list of dependencies in backwards causal order
      !a depends on b called before any b depends on ...
      !find thing(s) that are depended on by nothing...
      !do all dependencies of that.
      !find things that are depended on only by things already taken care of...
      !find the next component that has no unranked dependendants or groupings

      next_rank=1
      inc=1
      OuterLoop: DO m=1,nStencils
         InnerLoop: DO j=1,nStencils
            ! If j already has been ranked cycle
            IF (Stencil(j)%rank .ne. -1) CYCLE

            ! If any of the things that depend on j are not yet ranked then skip
            DO k=1,di
               IF ((dependencies(k)%p2==j) .AND. Stencil(dependencies(k)%p1)%rank == -1) CYCLE InnerLoop
            END DO

            ! If j is in an unranked group then cycle
            DO k=1,gi
               IF ((groupings(k)%p2 == j) .AND. Stencil(groupings(k)%p1)%rank == -1) CYCLE InnerLoop
            END DO


            ! If j is in a codepency then we would like to add both - so first check it's codependent partner
            DO l=1,ci
               IF (codependencies(l)%p1 == j .OR. codependencies(l)%p2 == j) THEN

                  i=merge(codependencies(l)%p1, codependencies(l)%p2, codependencies(l)%p2 == j)
                  DO k=1,di
                     IF ((dependencies(k)%p2==i) .AND. Stencil(dependencies(k)%p1)%rank == -1) CYCLE InnerLoop
                  END DO

                  ! If i is in an unranked group then cycle
                  DO k=1,gi
                     IF ((groupings(k)%p2 == i) .AND. Stencil(groupings(k)%p1)%rank == -1) CYCLE InnerLoop
                  END DO                     
                  
                  Stencil(i)%rank=next_rank
                  slist(inc)=i
                  inc=inc+1

!                  next_rank=next_rank+1
!                  m=m+1 !we can skip one m loop since we added an extra term
               END IF               
            END DO
                  

            slist(inc)=j
            inc=inc+1            
            Stencil(j)%rank=next_rank

            DO k=1,gi
               IF (groupings(k)%p1 == j) THEN
                  i=groupings(k)%p2
                  Stencil(i)%rank=next_rank
                  slist(inc)=i
                  inc=inc+1
               END IF
            END DO

            next_rank=next_rank+1
            exit
5        END DO InnerLoop
      END DO OuterLoop
   END SUBROUTINE sort_dependencies


   !> Routine for setting dependencies
   SUBROUTINE set_dependencies()
      INTEGER :: j,k,l,m
      DO k=1,nStencils

         DO j=1,ci
            IF (codependencies(j)%p1 == slist(k) .OR. codependencies(j)%p2 == slist(k)) THEN
!               i=merge(codependencies(j)%p1, codependencies(j)%p2, codependencies(j)%p2 == slist(k))
               CALL ExpandIfNeeded(Stencil(codependencies(j)%p1)%range, Stencil(codependencies(j)%p2)%range, codependencies(j)%range)
               CALL ExpandIfNeeded(Stencil(codependencies(j)%p2)%range, Stencil(codependencies(j)%p1)%range, -codependencies(j)%range)
               Stencil(codependencies(j)%p2)%lead=MAX(Stencil(codependencies(j)%p1)%lead+codependencies(j)%range(1,1), Stencil(codependencies(j)%p2)%lead)
               Stencil(codependencies(j)%p1)%lead=Stencil(codependencies(j)%p2)%lead-codependencies(j)%range(1,1)
            END IF
         END DO

         DO j=1,di
            IF (dependencies(j)%p1==slist(k)) THEN !need to adjust dependencies and groupings
               Stencil(dependencies(j)%p2)%lead=max(Stencil(dependencies(j)%p2)%lead, Stencil(dependencies(j)%p1)%lead+dependencies(j)%range(1,2))
               Stencil(dependencies(j)%p2)%trail=min(Stencil(dependencies(j)%p2)%trail, Stencil(dependencies(j)%p1)%lead+dependencies(j)%range(1,1))
               CALL ExpandIfNeeded(Stencil(dependencies(j)%p1)%range, Stencil(dependencies(j)%p2)%range, dependencies(j)%range)
            END IF
         END DO
         DO j=1,gi
            IF (groupings(j)%p1==slist(k)) THEN !need to adjust dependencies and groupings
               CALL ExpandIfNeeded(Stencil(groupings(j)%p1)%range, Stencil(groupings(j)%p2)%range, square((/0,0,0,0,0,0/)))
               Stencil(groupings(j)%p2)%lead=max(Stencil(groupings(j)%p2)%lead, Stencil(groupings(j)%p1)%lead)
               Stencil(groupings(j)%p2)%trail=min(Stencil(groupings(j)%p2)%trail, Stencil(groupings(j)%p1)%trail)               
            END IF
         END DO
      END DO
      MaxLead=maxval(-Stencil(1:nStencils)%range(1,1)+Stencil(1:nStencils)%Lead)
      MaxTrail=minval(Stencil(1:nStencils)%range(1,2)-Stencil(1:nStencils)%Lead)
   END SUBROUTINE set_dependencies


   !> Routine for managing codependencies
   SUBROUTINE set_codependencies()
      INTEGER :: j
      DO j=1,ci
         CALL ExpandIfNeeded(Stencil(codependencies(j)%p1)%range, Stencil(codependencies(j)%p2)%range, codependencies(j)%range)
         CALL ExpandIfNeeded(Stencil(codependencies(j)%p2)%range, Stencil(codependencies(j)%p1)%range, -codependencies(j)%range)
         Stencil(codependencies(j)%p2)%lead=MAX(Stencil(codependencies(j)%p1)%lead+codependencies(j)%range(1,1), Stencil(codependencies(j)%p2)%lead)
         Stencil(codependencies(j)%p1)%lead=Stencil(codependencies(j)%p2)%lead-codependencies(j)%range(1,1)
      END DO
   END SUBROUTINE set_codependencies

   !> Routine for finalizing dependencies
   SUBROUTINE finalize_dependencies()
      INTEGER :: j,k,l,m
      DO k=1,nStencils
         DO j=1,di
            IF (dependencies(j)%p1 == slist(k)) THEN !need to adjust dependencies and groupings
               Stencil(dependencies(j)%p2)%lead=max(Stencil(dependencies(j)%p2)%lead, Stencil(dependencies(j)%p1)%lead+dependencies(j)%range(1,2))
               Stencil(dependencies(j)%p2)%trail=min(Stencil(dependencies(j)%p2)%trail, Stencil(dependencies(j)%p1)%lead+dependencies(j)%range(1,1))
               CALL ExpandIfNeeded(Stencil(dependencies(j)%p1)%range, Stencil(dependencies(j)%p2)%range, dependencies(j)%range)
            END IF
         END DO
         DO j=1,gi
            IF (groupings(j)%p1 == slist(k)) THEN !need to adjust dependencies and groupings
               CALL ExpandIfNeeded(Stencil(groupings(j)%p1)%range, Stencil(groupings(j)%p2)%range, square((/0,0,0,0,0,0/)))
               Stencil(groupings(j)%p2)%lead=max(Stencil(groupings(j)%p2)%lead, Stencil(groupings(j)%p1)%lead)
               Stencil(groupings(j)%p2)%trail=min(Stencil(groupings(j)%p2)%trail, Stencil(groupings(j)%p1)%trail)
            END IF
         END DO
      END DO

   END SUBROUTINE finalize_dependencies



!> Shifts slabs for stencil
!! @param name Stencil piece
subroutine shift(index, level, iStencil, mB)
  INTEGER :: iStencil, level, index, mB(3,2)
  INTEGER :: temp,ii
  mB=0
  mB(1,:)=Stencil(iStencil)%lead
  mB(2:3,:)=LevelBuffers(level)%StencilBuffer(iStencil)%range(2:3,:)
  IF (index >= LevelBuffers(level)%StencilBuffer(iStencil)%start) THEN !Need to cycle in case values still used
     temp=LevelBuffers(level)%StencilBuffer(iStencil)%x(Stencil(iStencil)%trail)
     do ii=Stencil(iStencil)%trail, Stencil(iStencil)%lead-1
        Levelbuffers(level)%StencilBuffer(iStencil)%x(ii)=LevelBuffers(level)%StencilBuffer(iStencil)%x(ii+1)
     end do
     LevelBuffers(level)%StencilBuffer(iStencil)%x(Stencil(iStencil)%lead)=temp
  END IF
end subroutine shift

!> Shifts slabs for stencil and returns whether or not the stencil should be updated
!! @param index Current sweep position
!! @param name Stencil piece
!! @param i leading stencil position
LOGICAL FUNCTION istimeshift(index,level,iStencil,mB)
  INTEGER :: iStencil, level, index, mB(3,2)
  INTEGER :: temp,ii
  REAL(8) :: wtime
  mB=0
  !  write(*,*) 'level =', level

  istimeshift=.false.
  IF (iStencil == 0) RETURN
  IF (iThreaded == PSEUDO_THREADED) then ! .AND. tStopAdvance /= FOREVER) THEN
     IF (AdvanceState==STOPPING) RETURN
     IF (AdvanceState==RESUMING) THEN
        IF (AdvanceStencil(level) == iStencil) THEN
           AdvanceState = RUNNING
           AdvanceStencil(level) = 0
!           write(*,*) MPI_ID, index, level, 'resuming on stencil', Stencil(iStencil)%description
        ELSE
           RETURN
        END IF
     END IF
     IF (mpi_wtime() > tStopAdvance) THEN
        AdvanceState=STOPPING
        AdvanceStencil(level)=iStencil
!        write(*,*) MPI_ID, index, level, 'stopped on stencil', Stencil(iStencil)%description
        RETURN
     END IF
  END IF

  istimeshift=LevelBuffers(level)%StencilBuffer(iStencil)%start <= index .AND. index <= LevelBuffers(level)%StencilBuffer(iStencil)%finish
  IF (istimeshift) THEN

# if defined PTH
     IF (iThreaded == THREADED .AND. level < MaxLevel) CALL yield(level)
# endif

     mB(1,:)=Stencil(iStencil)%lead
     mB(2:3,:)=LevelBuffers(level)%StencilBuffer(iStencil)%range(2:3,:)
  END IF
  IF (index >= LevelBuffers(level)%StencilBuffer(iStencil)%start) THEN !Need to cycle in case values still used
     temp=LevelBuffers(level)%StencilBuffer(iStencil)%x(Stencil(iStencil)%trail)
     do ii=Stencil(iStencil)%trail, Stencil(iStencil)%lead-1
        Levelbuffers(level)%StencilBuffer(iStencil)%x(ii)=LevelBuffers(level)%StencilBuffer(iStencil)%x(ii+1)
     end do
     LevelBuffers(level)%StencilBuffer(iStencil)%x(Stencil(iStencil)%lead)=temp
  END IF

END FUNCTION istimeshift

!> Determines whether or not the stencil should be updated
!! @param index Current sweep position
!! @param name Stencil piece
!! @param i leading stencil position
LOGICAL FUNCTION istime(index,level,iStencil, mB)
  INTEGER :: iStencil, level, index, mB(3,2)
  INTEGER :: temp,ii
  mB=0
  istime=.false.
  IF (iStencil == 0) RETURN
  IF (iThreaded == PSEUDO_THREADED .AND. AdvanceState /= RUNNING) RETURN
  istime=LevelBuffers(level)%StencilBuffer(iStencil)%start <= index .AND. index <= LevelBuffers(level)%StencilBuffer(iStencil)%finish
  IF (istime) THEN
# if defined PTH
     IF (iThreaded == THREADED .AND. level < MaxLevel) CALL yield(level)
# endif
     mB(1,:)=Stencil(iStencil)%lead
     mB(2:3,:)=LevelBuffers(level)%StencilBuffer(iStencil)%range(2:3,:)
  END IF
END FUNCTION istime




FUNCTION BufferSizes(dom_range)
  INTEGER, DIMENSION(3,2) :: dom_range
  INTEGER :: i,mb(3,2), BufferSizes
  BufferSizes=0
  DO i=1,nStencils
     mb=dom_range+Stencil(i)%range
     BufferSizes=BufferSizes+(Stencil(i)%lead-Stencil(i)%trail+1)*PRODUCT(mb(2:3,2)-mb(2:3,1)+1)*PRODUCT(Stencil(i)%NrVars(:))
  END DO

END FUNCTION BufferSizes

SUBROUTINE initialize_buffer(Buffer, dom_range)
  TYPE(BufferDataDef) :: Buffer
  INTEGER, DIMENSION(3,2) :: dom_range
  INTEGER :: i,j,mb(3,2)
  ALLOCATE(Buffer%StencilBuffer(nStencils))
!  write(*,*) 'dom_range=', dom_range

  DO i=1,nStencils
     mb=dom_range+Stencil(i)%range
     Buffer%StencilBuffer(i)%range=mB
     Buffer%StencilBuffer(i)%start=mB(1,1)-Stencil(i)%lead
     Buffer%StencilBuffer(i)%finish=mB(1,2)-Stencil(i)%lead !mB(1,2)-Stencil(i)%lead     
     IF (PRODUCT(Stencil(i)%NrVars) > 0) THEN
        ALLOCATE(Buffer%StencilBuffer(i)%data(Stencil(i)%trail:Stencil(i)%lead,mb(2,1):mb(2,2),mb(3,1):mb(3,2),Stencil(i)%NrVars(1), Stencil(i)%NrVars(2)))
!        write(*,*) size(Buffer%StencilBuffer(i)%data)
        CALL CheckAllocation(SweepAllocator, size(Buffer%StencilBuffer(i)%data)*8)
        IF (Stencil(i)%clear) THEN
           Buffer%StencilBuffer(i)%data=0
        ELSE
!           Buffer%StencilBuffer(i)%data=0d0/0d0
           
        END IF
!        write(*,*) 'allocated data for buffer', stencil(i)%description, mB
     ELSE
!        write(*,*) 'skipped allocation for buffer', stencil(i)%description, stencil(i)%nrvars
     END IF
     DO j=Stencil(i)%trail, Stencil(i)%lead
        Buffer%StencilBuffer(i)%x(j)=j
     END DO
!     write(*,*) Stencil(i)%description, Buffer%StencilBuffer(i)%start, Buffer%StencilBuffer(i)%finish
  END DO
 ! write(*,*) 'init buffer loc', LOC(LevelBuffers(0)%StencilBuffer(w)), ASSOCIATED(LevelBuffers(0)%StencilBuffer(w)%data)

END SUBROUTINE initialize_buffer

SUBROUTINE Clear_Buffer(Buffer)
  TYPE(BufferDataDef) :: Buffer
  INTEGER :: i
  DO i=1,nStencils
!     write(*,*) Stencil(i)%description, Stencil(i)%NrVars, ASSOCIATED(Buffer%StencilBuffer(i)%data)
     IF (ASSOCIATED(Buffer%StencilBuffer(i)%data)) THEN
        CALL CheckDeAllocation(SweepAllocator, size(Buffer%StencilBuffer(i)%data)*8)
        DEALLOCATE(Buffer%StencilBuffer(i)%data)
     END IF
  END DO
  DEALLOCATE(Buffer%StencilBuffer)
END SUBROUTINE Clear_Buffer


END MODULE StencilControl
