!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    stencil_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
MODULE StencilDeclarations
USE GlobalDeclarations
IMPLICIT NONE
   
SAVE
   !> Contains information about stencil pieces used in the hyperbolic solve
   TYPE :: StencilDef
      INTEGER :: x(-10:10)
      INTEGER, DIMENSION(3,2) :: range=RESHAPE( (/100,100,100,-100,-100,-100/), (/ 3,2 /) )
      INTEGER :: lead=-100
      INTEGER :: trail=100
      INTEGER :: lag
      INTEGER :: rank=-1
      INTEGER :: NrVars(2)=0
      LOGICAL :: needed=.false.
      CHARACTER(LEN=16) :: description
      LOGICAL :: clear=.false.
   END TYPE StencilDef


   TYPE :: StencilBufferDef
      INTEGER, DIMENSION(3,2) :: range
      INTEGER :: start
      INTEGER :: finish
      REAL(KIND=qPREC), DIMENSION(:,:,:,:,:), POINTER :: data => NULL()
      INTEGER :: x(-10:10)
   END type StencilBufferDef

  TYPE BufferDataDef
     TYPE(StencilBufferDef), DIMENSION(:), POINTER :: StencilBuffer
  END TYPE BufferDataDef

  TYPE(BufferDataDef), DIMENSION(:), ALLOCATABLE :: LevelBuffers
  TYPE(BufferDataDef), POINTER :: ActiveBuffer


   !> Pointer to a stencil type
!   TYPE pStencilDef
!      INTEGER :: p
!   END TYPE pStencilDef


   !> Defines a dependency between two stencil pieces and a dependency range
   TYPE dependency
      INTEGER :: p1, p2
      INTEGER :: range(3,2)
   END TYPE dependency

   !> Defines a grouping of two stencil pieces 
   TYPE grouping
      INTEGER :: p1, p2
   END TYPE grouping

   !> Defines an adjustment to a stencil piece to limit the range
   TYPE adjustment
      INTEGER :: p1
      INTEGER :: range(3,2)
   END TYPE adjustment

   !> Defines a codependency between stencil pieces.  Offset stored twice in range
   TYPE codependency
      INTEGER :: p1, p2
      INTEGER :: range(3,2)
   END TYPE codependency


   INTEGER, PARAMETER :: nStencilsMax=200
   TYPE(StencilDef), DIMENSION(nStencilsMax), Target :: Stencil
   INTEGER :: nStencils=0
   
   INTEGER :: slist(nStencilsMax)
   INTEGER :: MaxLead, MaxTrail
   TYPE(dependency) :: dependencies(300)
   TYPE(adjustment) :: adjustments(50)
   TYPE(grouping) :: groupings(10)
   TYPE(codependency) :: codependencies(10)
   
   INTEGER :: di=0
   INTEGER :: gi=0
   INTEGER :: ci=0


  INTEGER, PARAMETER, DIMENSION(6) :: center_r=(/0,0,0,0,0,0/), center_l=(/-1,-1,0,0,0,0/), center_u=(/0,0,0,0,0,0/), &
       center_d=(/0,0,-1,-1,0,0/), center_b=(/0,0,0,0,-1,-1/), center_f=(/0,0,0,0,0,0/), &
       same=(/0,0,0,0,0,0/), exact=(/1000,1000,1000,1000,1000,1000/)


CONTAINS

!> Reshapes and transposes an array
!! @param a 6 element input array
FUNCTION square(a)
  INTEGER, DIMENSION(:) :: a
  INTEGER, DIMENSION(3,2) :: square
  square=reshape(a,(/3,2/),(/0/), (/2,1/))
END FUNCTION square

!> Creates a 6 element array with values determined by a
!! @param a offset
FUNCTION cube(a)
  INTEGER :: a, cube(6)
  cube=(/-a,a,-a,a,-a,a/)
END FUNCTION cube


!> Puts stencil pieces in a group
!! @param a stencil piece
!! @param b another stencil piece
SUBROUTINE set_group(a,b)
   INTEGER :: a,b
   gi=gi+1
   IF (a == 0) THEN      
      nStencils=nStencils+1
      a=nStencils
   END IF
   IF (b == 0) THEN
      nStencils=nStencils+1
      b=nStencils
   END IF
   groupings(gi)%p1=a
   groupings(gi)%p2=b
END SUBROUTINE set_group

!> Creates a dependency between a and b
!! @param a stencil piece
!! @param b another stencil piece
!! @param range index range of dependency
SUBROUTINE set_dependency(a, b, range)
   INTEGER :: a,b
   INTEGER, DIMENSION(6) :: range
   di=di+1
   IF (a == 0) THEN
      nStencils=nStencils+1
      a=nStencils
   END IF
   IF (b == 0) THEN
      nStencils=nStencils+1
      b=nStencils
   END IF
   dependencies(di)%p1=a
   dependencies(di)%p2=b
   dependencies(di)%range=square(range)
END SUBROUTINE set_dependency

!> Creates a codependency between a and b
!! @param a stencil piece
!! @param b another stencil piece
!! @param range offset of codependency
SUBROUTINE set_codependency(a,b,range)
   INTEGER :: a, b
   INTEGER, DIMENSION(3) :: range
   ci=ci+1
   IF (a == 0) THEN 
      nStencils=nStencils+1
      a=nStencils
   END IF
   IF (b == 0) THEN
      nStencils=nStencils+1
      b=nStencils
   END IF
   codependencies(ci)%p1=a
   codependencies(ci)%p2=b
   codependencies(ci)%range=SPREAD(range, 2, 2)
END SUBROUTINE set_codependency

!> Expands range2 if needed based on range1 and offsets
!! @param range1 1st range
!! @param range2 2nd range
!! @param offsets offsets from dependencies
SUBROUTINE ExpandIfNeeded(range1, range2, offsets)
  INTEGER, DIMENSION(3,2) :: range1, range2, offsets
  INTEGER :: i
  DO i=1,3
     range2(i,1)=min(range2(i,1), range1(i,1)+offsets(i,1))
     range2(i,2)=max(range2(i,2), range1(i,2)+offsets(i,2))
  END DO

END SUBROUTINE ExpandIfNeeded

!> Shrinks a range if needed
!! @param range1 available range
!! @param range2 range to adjust
!! @param offsets offsets from dependencies
SUBROUTINE shrink(range1, range2, offsets)
  INTEGER, DIMENSION(3,2) :: range1, range2, offsets
  INTEGER :: i
  DO i=1,3
     range2(i,1)=max(range2(i,1), range1(i,1)+offsets(i,1))
     range2(i,2)=min(range2(i,2), range1(i,2)+offsets(i,2))
  END DO

END SUBROUTINE shrink



!> Expands the range along the dimension dim a distance x
!! @param dim direction
!! @param x distance
!! @param range to expand
function expand(dim, x, range)
  integer, dimension(3,2), INTENT(IN) :: range
  integer, dimension(3,2)  :: expand
  integer,INTENT(IN)  :: dim, x
  expand=range
  expand(dim,1)=expand(dim,1)-x
  expand(dim,2)=expand(dim,2)+x
end function expand

!> Expands the 1st index of range along the dimension dim a distance x
!! @param dim direction
!! @param x distance
!! @param range to expand
function expand_left(dim, x, range)
  integer, dimension(4,2), INTENT(IN) :: range
  integer, dimension(4,2)  :: expand_left
  integer,INTENT(IN)  :: dim, x
  expand_left=range
  expand_left(dim,1)=expand_left(dim,1)-x
end function expand_left

!> Expands the 2nd index of range along the dimension dim a distance x
!! @param dim direction
!! @param x distance
!! @param range to expand
function expand_right(dim, x, range)
  integer, dimension(3,2), INTENT(IN) :: range
  integer, dimension(3,2)  :: expand_right
  integer,INTENT(IN)  :: dim, x
  expand_right=range
  expand_right(dim,2)=expand_right(dim,2)+x
end function expand_right



!> Converts a cell centered range into a face centered range and expands it along the dimension dim a distance x
!! @param dim direction
!! @param x distance
!! @param range to expand
function expandaux(dim, x, range)
  integer, dimension(3,2), INTENT(IN) :: range
  integer, dimension(3,2)  :: expandaux
  integer,INTENT(IN)  :: dim, x
  expandaux=range
  expandaux(dim,1)=expandaux(dim,1)-x
  expandaux(dim,2)=expandaux(dim,2)+x+1
end function expandaux

!> Converts a face centered range into a cell centered range and expands it along the dimension dim a distance x
!! @param dim direction
!! @param x distance
!! @param range to expand
function expandauxinv(dim, x, range)
  integer, dimension(3,2), INTENT(IN) :: range
  integer, INTENT(IN) :: x, dim
  integer, dimension(3,2) :: expandauxinv
  expandauxinv=range
  expandauxinv(dim,1)=expandauxinv(dim,1)-x
  expandauxinv(dim,2)=expandauxinv(dim,2)+x-1
end function expandauxinv



END MODULE StencilDeclarations
