!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    sweep_scheme.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir sweep
!! @brief Directory containing files used by SweepScheme

!> @file sweep_scheme.f90
!! @brief Main file for module SweepScheme

!> @defgroup SweepScheme SweepScheme
!! @brief Module for updating grids by sweeping across the grid
!! @ingroup Hyperbolic

!===============================================================================
! Module Name:		SweepScheme
! Module File:		sweep.f90
! Purpose:		Implements a numerical scheme for solving 
!                       hydrodynamic equations.
! Public Methods:	ReadSweepDomainData(), sweepadvance()
! Created:		by Jonathan Carroll.
! Notes:		MHD equations are handled by constrained transport 
!                       through the afterstep() function.
!===============================================================================

!> Module for updating grids by sweeping across the grid
!! @ingroup SweepScheme
!! @par Include Files:
!! sweep_scheme_.f90
MODULE SweepScheme

   USE GlobalDeclarations
   USE DataDeclarations
   USE HyperbolicDeclarations
   USE SourceControl
   USE SourceDeclarations
   USE PhysicsDeclarations
   USE EOS
   USE ModuleControl
   USE RiemannSolvers
   USE DataInfoOps
   USE Timing
   USE StencilDeclarations
   USE SchemeDeclarations
   USE StencilControl
   USE TreeDeclarations
   IMPLICIT NONE
   PRIVATE
   SAVE
   PUBLIC SweepReadDomainData, sweepadvance, SweepAdvanceStackSize

CONTAINS



  FUNCTION SweepAdvanceStackSize(n)
     INTEGER :: n, SweepAdvanceStackSize, ambc
     TYPE(NodeDefList), POINTER :: nodelist
     ambc=levels(n)%ambc(1)+hyperbolic_mbc
     nodelist=>Nodes(n)%p
     SweepAdvanceStackSize=0d0
     DO WHILE (ASSOCIATED(nodelist))
        SweepAdvanceStackSize=max(SweepAdvanceStackSize, BufferSizes(nodelist%self%info%mGlobal))
        nodelist=>nodelist%next
     END DO
     ! Advance stack size should multiplied by the size of qPREC, then by a safety factor, and then 1 MB
     SweepAdvanceStackSize=nint(SweepAdvanceStackSize*8*1.5)+1*1024*1024
  END FUNCTION SweepAdvanceStackSize


   !> Read in sweep-specific solver data and use it to initialize the system.
   SUBROUTINE SweepReadDomainData()
      INTEGER :: iErr, i

      NAMELIST/SweepData/ lChar_Limiters, lCTU, lCautious,lLimiter,lGudonov,ViscCD,lApplyLOF,LOF_ALPHA, iTracer, SourceMethod, InterpOrder, lApplyDiffusion, DIFF_ALPHA, lUsePPML

      ! Flags for various sweep options are initialized with defaults.	
      lLimiter=.true.
      lChar_Limiters=.true.
      lUsePPML=.false.
      lCTU=.true.
      lCautious=.true.
      lHLLTypeSolver=.false.
      iTracer=NONLAGRANGIAN
      lGudonov=.false.
      ViscCD = 0
      lApplyLOF = .false.
      LOF_ALPHA=.075
      DIFF_ALPHA=.1
      lApplyDiffusion=.false.
      InterpOrder = -1

      READ(SOLVER_DATA_HANDLE,NML=SweepData,IOStat=iErr)
      IF(iErr/=0) THEN
         PRINT*,'SweepReadDomainData() error:  unable to read SweepData namelist.'
         STOP
      END IF

      ! These methods really need to be listed by name.
      lIsothermal = iEOS == EOS_ISOTHERMAL

      ! Backwards compatability for interpolation method
      IF (lGudonov) THEN
         IF (InterpOrder == -1) THEN
            PRINT*, 'lGudonov is being depracated.  Use InterpOrder == 1 instead'
            InterpOrder = 1
         ELSEIF (InterpOrder == 1) THEN
            PRINT*, 'lGudonov is not needed in solver.data'
         ELSE
            PRINT*, 'lGudonov conflicts with InterpOrder /= 1.  Ignoring lGudonov!!!'
         END IF
      END IF
      IF (InterpOrder < 1) InterpOrder = 3
      IF (InterpOrder == 1) lGudonov=.true.
      IF (InterpOrder < 3) lUsePPML=.false.
      ! Check choice of riemann solvers
      IF (iSolver == 3  .OR. iSolver == 2 .OR. iSolver == 4 .OR. iSolver == 6 .OR. iSolver == 8) THEN
                lHLLTypeSolver=.true.
      END IF

!      write(*,*) 'iSolver=', iSolver

      !Setup variables for mapping q to fluxes in x y and z direction
      IF (lMHD) THEN
         NrWaves=NrCons-1
      ELSE
         NrWaves=NrCons
      END IF

      nSweepFlux=nFlux
      IF (iTracer == NONLAGRANGIAN) THEN
         NrWaves=NrWaves+NrTracerVars
         SweepCons=NrCons+NrTracerVars
         nSweepFlux=nFlux
      ELSE
         SweepCons=NrCons
         nSweepFlux=nFlux-NrTracerVars
      END IF

      ALLOCATE(SweepFluxFields(nSweepFlux))
      SweepFluxFields(1:nSweepFlux)=FluxFields(1:nSweepFlux)


      ALLOCATE(oneDx_i(NrWaves),oneDy_i(NrWaves),oneDz_i(NrWaves), &
           wDx_i(SweepCons),wDy_i(SweepCons),wDz_i(SweepCons), &
           fDx_i(SweepCons),fDy_i(SweepCons),fDz_i(SweepCons), STAT=iErr)

      IF (iErr /= 0) THEN
         PRINT *, "SweepReadDomainData() error: unable to allocate arrays."
         STOP
      END IF

      IF (lMHD) THEN
         IF (lIsothermal) THEN
            oneDx_i(1:6)=(/1,2,3,4,6,7/)
            oneDy_i(1:6)=(/1,3,4,2,7,5/)
            oneDz_i(1:6)=(/1,4,2,3,5,6/)
            wDx_i(1:7)=(/1,2,3,4,5,6,7/)
            wDy_i(1:7)=(/1,3,4,2,6,7,5/)
            wDz_i(1:7)=(/1,4,2,3,7,5,6/)
            fDx_i(1:7)=(/1,2,3,4,5,6,7/)
            fDy_i(1:7)=(/1,4,2,3,7,5,6/)
            fDz_i(1:7)=(/1,3,4,2,6,7,5/)
         ELSE
            oneDx_i(1:7)=(/1,5,2,3,4,7,8/)
            oneDy_i(1:7)=(/1,5,3,4,2,8,6/)
            oneDz_i(1:7)=(/1,5,4,2,3,6,7/)
            wDx_i(1:8)=(/1,5,2,3,4,6,7,8/)
            wDy_i(1:8)=(/1,5,3,4,2,7,8,6/)
            wDz_i(1:8)=(/1,5,4,2,3,8,6,7/)
            fDx_i(1:8)=(/1,3,4,5,2,6,7,8/)
            fDy_i(1:8)=(/1,5,3,4,2,8,6,7/)
            fDz_i(1:8)=(/1,4,5,3,2,7,8,6/)
         END IF
      ELSE
         IF (lIsothermal) THEN
            IF (nDim == 1) THEN
               oneDx_i(1:2)=(/1,2/)
               wDx_i(1:2)=(/1,2/)
               fDx_i(1:2)=(/1,2/)
            ELSE IF (nDim == 2) THEN
               IF (iCylindrical.ne.WithAngMom) THEN
                  oneDx_i(1:3)=(/1,2,3/)
                  oneDy_i(1:3)=(/1,3,2/)
                  wDx_i(1:3)=(/1,2,3/)
                  wDy_i(1:3)=(/1,3,2/)
                  fDx_i(1:3)=(/1,2,3/)
                  fDy_i(1:3)=(/1,3,2/)
               ELSE 
                  oneDx_i(1:4)=(/1,2,3,4/)
                  oneDy_i(1:4)=(/1,3,4,2/)
                  !                oneDz_i(1:4)=(/1,4,2,3/)
                  wDx_i(1:4)=(/1,2,3,4/)
                  wDy_i(1:4)=(/1,3,4,2/)
                  !                wDz_i(1:4)=(/1,4,2,3/)
                  fDx_i(1:4)=(/1,2,3,4/)
                  fDy_i(1:4)=(/1,4,2,3/)
                  !                fDz_i(1:4)=(/1,3,4,2/)
               END IF
            ELSE
               oneDx_i(1:4)=(/1,2,3,4/)
               oneDy_i(1:4)=(/1,3,4,2/)
               oneDz_i(1:4)=(/1,4,2,3/)
               wDx_i(1:4)=(/1,2,3,4/)
               wDy_i(1:4)=(/1,3,4,2/)
               wDz_i(1:4)=(/1,4,2,3/)
               fDx_i(1:4)=(/1,2,3,4/)
               fDy_i(1:4)=(/1,4,2,3/)
               fDz_i(1:4)=(/1,3,4,2/)
            END IF
         ELSE
            IF (nDim == 1) THEN
               oneDx_i(1:3)=(/1,3,2/)
               wDx_i(1:3)=(/1,3,2/)
               fDx_i(1:3)=(/1,3,2/)
            ELSE IF (nDim == 2) THEN
               IF (iCylindrical.ne.WithAngMom) THEN
                  oneDx_i(1:4)=(/1,4,2,3/)
                  oneDy_i(1:4)=(/1,4,3,2/)
                  wDx_i(1:4)=(/1,4,2,3/)
                  wDy_i(1:4)=(/1,4,3,2/)

                  fDx_i(1:4)=(/1,3,4,2/)
                  fDy_i(1:4)=(/1,4,3,2/)
               ELSE 
                  oneDx_i(1:5)=(/1,5,2,3,4/)
                  oneDy_i(1:5)=(/1,5,3,4,2/)
                  wDx_i(1:5)  =(/1,5,2,3,4/)
                  wDy_i(1:5)  =(/1,5,3,4,2/)
                  fDx_i(1:5)  =(/1,3,4,5,2/)
                  fDy_i(1:5)  =(/1,5,3,4,2/)
               END IF
            ELSE
               oneDx_i(1:5)=(/1,5,2,3,4/)
               oneDy_i(1:5)=(/1,5,3,4,2/)
               oneDz_i(1:5)=(/1,5,4,2,3/)
               wDx_i(1:5)=(/1,5,2,3,4/)
               wDy_i(1:5)=(/1,5,3,4,2/)
               wDz_i(1:5)=(/1,5,4,2,3/)
               fDx_i(1:5)=(/1,3,4,5,2/)
               fDy_i(1:5)=(/1,5,3,4,2/)
               fDz_i(1:5)=(/1,4,5,3,2/)
            END IF
         END IF
      END IF

      !  Set the number of conserved variables that are handled by sweep.  The
      !  magnetic fields are
      !  handled by the CT scheme rather than the sweep itself, so they are
      !  not included in the number
      !  of conserved variables.

      IF (NrTracerVars > 0 .AND. iTracer == NONLAGRANGIAN) THEN
         IF (lMHD) THEN
            OneDx_i(NrCons:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
            OneDy_i(NrCons:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
            OneDz_i(NrCons:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
         ELSE
            OneDx_i(NrCons+1:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
            OneDy_i(NrCons+1:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
            OneDz_i(NrCons+1:NrWaves) = (/(i,i=nTracerLo, nTracerHi)/)
         END IF
         wDx_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
         wDy_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
         wDz_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
         fDx_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
         fDy_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
         fDz_i(NrCons+1:SweepCons) = (/(i,i=nTracerLo, nTracerHi)/)
      END IF


      ALLOCATE(LevelBuffers(0:MaxLevel)) !Cache for each level advance
      ALLOCATE(index_start_bylevel(0:MaxLevel))  !Cache index for each level's sweep for pseudo threading    
      index_start_bylevel(:)=THEBEGINNING
      CALL setup()

   END SUBROUTINE SweepReadDomainData

   !> Advances the grid based on the available solvertime
   !! @param Info Info structure
   !! @param partialOK Logical optional flag that determins whether partial advances are ok
   SUBROUTINE sweepAdvance(Info, dom_range, lComplete, lProfile_opt)

# if defined PTH
      USE PthDeclarations
# endif
      USE Scheduling
      !    SAVE
      TYPE (InfoDef) :: Info
      LOGICAL :: lComplete, lProfile
      LOGICAL, OPTIONAL :: lProfile_opt
      !    REAL(KIND=qPREC), OPTIONAL :: TimeAvailableToSolver
      INTEGER :: index,level, index_start, index_finish
      REAL(KIND=qPREC), PARAMETER :: PPMC=1.25
      REAL(KIND=qPrec) :: cfl, CostPerSweep
      REAL(KIND=qPrec) :: dt,dtdx,dtdy,dtdz,hdtdx,hdtdy,hdtdz,hdt,qdtdx,qdtdy,qdtdz,two_thirds_dtdx, &
           two_thirds_dtdy, two_thirds_dtdz, three_dtdx, three_dtdy, three_dtdz,dx,qdt,hdx, dv, d2, d2R, d2L, d2C, d2m
      REAL(KIND=qPrec) :: t_loopstart, t_loopend

      REAL(8) :: tused
      INTEGER :: dom_range(3,2), bc(3)

      LOGICAL :: partialOK

      TYPE(StencilBufferDef), POINTER :: w_,qLx_,qRx_,qLy_,qRy_,qLz_,qRz_,fx_,fy_,fz_,q2Lx_,q2Rx_,q2Ly_,q2Ry_,q2Lz_,q2Rz_,f2x_,f2y_,f2z_,limiter_x_,limiter_y_,limiter_z_,limiter_ppm_,pT_,qex_,qey_,qez_,dqx_,dqy_,dqz_,ex_,ey_,ez_,ex_bar_,ey_bar_,ez_bar_,e2x_,e2y_,e2z_,e2x_bar_,e2y_bar_,e2z_bar_,ctfy_,ctfz_,ctfx_,ctf2x_,ctf2y_,ctf2z_,w2_,A2x_,A2y_,A2z_,Sx_,Sy_,Sz_,w3_,A3x_,A3y_,A3z_,SpeedsX_,SpeedsY_,SpeedsZ_,leftX_,leftY_,leftZ_,rightX_,rightY_,rightZ_,nWaves_,req_eigens_,adfx_,adfy_,adfz_,eta2x_,eta2y_,eta2z_,etax_,etay_,etaz_,q_,aux_,recon_,beforesweepstep_,aftersweepstep_,source_,source2_,cornerdiv_,gradphix_,gradphiy_,gradphiz_

      lProfile=.false.
      IF (PRESENT(lProfile_opt)) lProfile=lProfile_opt
      level=Info%level    

      dx=levels(level)%dx
      dv=dx**nDim
      dt=levels(level)%dt
      hdt=half*dt
      qdt=fourth*dt
      hdx=half*dx
      dtdx=dt/dx;dtdy=dtdx;dtdz=dtdx
      hdtdx=half*dtdx;hdtdy=half*dtdy;hdtdz=half*dtdz
      qdtdx=fourth*dtdx;qdtdy=fourth*dtdy;qdtdz=fourth*dtdz;
      two_thirds_dtdx=2d0/3d0*dtdx
      two_thirds_dtdy=2d0/3d0*dtdy
      two_thirds_dtdz=2d0/3d0*dtdz
      three_dtdx=3d0*dtdx
      three_dtdy=3d0*dtdy
      three_dtdz=3d0*dtdz
!      AdvanceTimer%LastStarted(level)=MPI_WTIME()

      t_startadvance(level)=mpi_wtime()
      IF (index_start_bylevel(level) == THEBEGINNING) THEN
         WorkDoneByGrid(level) = 0d0
         IF (.NOT. lProfile) CALL BeforeStep(Info)
# if defined PTH
         IF (iThreaded == THREADED .AND. level < MaxLevel) CALL yield(level)
# endif
         CALL initialize_buffer(LevelBuffers(level), dom_range)
         index_start_bylevel(level)=dom_range(1,1)-MaxLead
         AdvanceState=RUNNING
      ELSE !must be using scheduling
         IF (AdvanceStencil(level) /= 0) THEN
            AdvanceState = RESUMING
         ELSE
            AdvanceState = RUNNING
         END IF
      END IF

      IF (lComplete) THEN
         tStopAdvance=FOREVER
      ELSE
         tStopAdvance=TimeAvailableToSolver
      END IF
      
      CALL LoadLevelStencilBuffers(level)


      DO index=index_start_bylevel(level),dom_range(1,2)+MaxTrail

# if defined PTH
         IF (iThreaded == THREADED .AND. level < MaxLevel) CALL yield(level)
# endif
         CALL SweepBeforeStep(Info,index)
         CALL Init_prims(Info, index)
         CALL Reconstruct(Info, index)
         IF (ViscCD == ROE_VISCOSITY .OR. ViscCD == H_VISCOSITY) CALL HVisc(Info, index)
         IF (nDim >= 2) THEN
            CALL calc_fluxes(Info, index)
            IF (lMHD) THEN
               CALL calc_emf(Info, index)
               CALL updateB(Info, index)
               CALL update_fluxes(Info, index)
            END IF
            IF (lMHD .OR. (NrTracerVars > 0 .AND. iTracer==LAGRANGIAN)) CALL updatew2(Info,index)
            CALL CTU(Info,index)
            CALL calc_final_fluxes(Info, index)
            IF (NrTracerVars > 0 .AND. iTracer == LAGRANGIAN) CALL calc_tracer_fluxes(Info, index)
            CALL update_final_fluxes(Info, index)
            CALL update_final(Info,index)
            IF (lMHD) THEN
               CALL calc_final_emf(Info, index)
               CALL updateB_final(Info, index)
            END IF
         ELSE
            CALL calc_fluxes_noctu(Info, index)
            CALL update_final_fluxes(Info, index)
            CALL update_final(Info,index)
         END  IF
         CALL store_fixup_fluxes(Info, index)         
         CALL SweepAfterStep(Info,index)
         
         IF (AdvanceState == STOPPING) EXIT
      END DO

      IF (AdvanceState == RUNNING) THEN
         index_start_bylevel(level) = THEBEGINNING
         NodeCompleted(level)=.true.

         CALL Clear_Buffer(LevelBuffers(level))
         CALL UpdateAux(Info, dom_range)
         IF (.NOT. lProfile) CALL AfterStep(Info)
         maxspeed(level)=max(maxspeed(level), GetMaxSpeed(Info%q(1:Info%mx(1),1:Info%mx(2),1:Info%mx(3),:)))

         tused = mpi_wtime()-t_startadvance(level)
         WorkDoneByGrid(level)=WorkDoneByGrid(level)+tused
         WorkDoneByLevel(level)=WorkDoneByLevel(level)-WorkDoneByGrid(level)+Info%CostPerGrid(levels(Info%level)%step)

         IF (.NOT. lProfile) InternalCellUpdates(level)=InternalCellUpdates(level)+product(Info%mX(1:nDim))
         IF (.NOT. lProfile) CellUpdates(level)=CellUpdates(level)+product(dom_range(:,2)-dom_range(:,1)+1)
         AdvancePredictor%Accumulator(level)=AdvancePredictor%Accumulator(level)+info%costpergrid(levels(Info%level)%step)
!         AdvanceTimer%Accumulator(level)=AdvanceTimer%Accumulator(level)+WorkDoneByGrid(level)
!         MySpeedFactor=sum(AdvanceTimer%Accumulator(0:MaxLevel))/sum(AdvancePredictor%Accumulator(0:MaxLevel))
      ELSE
         index_start_bylevel(level) = index
         NodeCompleted(level)=.false.
         tused = mpi_wtime()-t_startadvance(level)
         WorkDoneByLevel(level)=WorkDoneByLevel(level)+tused
         WorkDoneByGrid(level)=WorkDoneByGrid(level)+tused
      END IF

   CONTAINS

      !> Calls routines that happen before a cell is updated
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates beforesweepstep from q
      SUBROUTINE SweepBeforeStep(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mS(3,2)    ! Bounds of slab to update in grid space

         IF (isTimeShift(index, Info%level, beforesweepstep, mB)) then
            ms(1,:)=index+mB(1,:)
            mS(2:3,:)=mB(2:3,:)

            IF (lSourceTerms) THEN
               ms(1,:)=index+mB(1,:)
               mS(2:3,:)=mB(2:3,:)
               CALL Src(Info, mS, levels(info%level)%tnow, hdt)
            END IF
            DO i=mB(1,1), mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,1:NrHydroVars) = &
                       Info%q(index+i,j,k,1:NrHydroVars)
               END FORALL
            END DO
            mS=mB
            mS(1,:)=mB(1,:)+index
         END IF
      END SUBROUTINE SweepBeforeStep

      !> Calls routines that happen after a cell is updated
      !! @param Info Info structure
      !! @param index Current sweep position
      !! updates aftersweepstep (info%q)  with w3 (info%q) with second strang step
      SUBROUTINE SweepAfterStep(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (IsTime(index, Info%level, aftersweepstep, mB)) THEN
            IF (lSourceTerms) THEN
               mb(1,:)=index+mB(1,:)
               CALL Src(Info, mb, levels(info%level)%tnow+hdt, hdt) 
            END IF
         END IF
      END SUBROUTINE SweepAfterStep


      !> Calculate primitive form of variables
      !! @param Info Info structure
      !! @param index Current sweep position
      !! updates w from beforesweepstep
      SUBROUTINE Init_prims(Info, index)  
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update

         IF (istimeshift(index, Info%level, w, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL cons_to_prim_2(beforesweepstep_%data(beforesweepstep_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1:SweepCons),w_%data(w_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1:SweepCons))
               IF (lCautious) CALL protect_all(w_%data(w_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO

         END IF
      END SUBROUTINE Init_prims

      !> Calculate predictor interface states
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates time centered qRx and qLx with 1D algorithms - and multi-D magnetic source terms from Garinder and Stone
      !! Calculates dqx, qex, limiters, and eigen system along the way
      !! Calculates gradphi if necessary and apply 1D grav source terms to 1D interface states
      SUBROUTINE Reconstruct(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mC(3,2)    ! Bounds of slab to update
         REAL(Kind=qPrec) :: dq, sdq2,qmhsq, lambda_min, lambda_max, dp
         INTEGER :: m,waves,n_waves
         REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: dleft, dright, dcenter, dw, dw6,dwmdw6, dwpdw6,dw_l,deltaq,aleft,aright,acenter
         REAL(KIND=qPREC), DIMENSION(:), POINTER :: q

         IF (InterpOrder == 1) THEN
            IF (istimeshift(index, Info%level, qRx, mB)) THEN              
               qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:)
               IF (lMHD .AND. nDim == 1) qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                    Info%q(1,1,1,iBx)
               IF (lMHD .AND. nDim >= 2) qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1)
            END IF
            IF (istimeshift(index, Info%level, qLx, mB)) THEN
               qLx_%data(qLx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1)-1:mB(1,2)-1),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:)
               IF (lMHD .AND. nDim == 1) qLx_%data(qLx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                    Info%q(1,1,1,iBx)
               IF (lMHD .AND. nDim >= 2) qLx_%data(qLx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1)
            END IF
            IF (istimeshift(index, Info%level, qRy, mB)) THEN
               qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:)
               IF (lMHD) qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBy) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),2)
            END IF
            IF (istimeshift(index, Info%level, qLy, mB)) THEN
               qLy_%data(qLy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1):mB(1,2)),mB(2,1)-1:mB(2,2)-1, mB(3,1):mB(3,2),1,:)
               IF (lMHD) qLy_%data(qLy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBy) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),2)
            END IF
            IF (istimeshift(index, Info%level, qRz, mB)) THEN
               qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:)
               IF (lMHD) qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBz) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),3)
            END IF
            IF (istimeshift(index, Info%level, qLz, mB)) THEN
               qLz_%data(qLz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    w_%data(w_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1)-1:mB(3,2)-1,1,:)
               IF (lMHD) qLz_%data(qLz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBz) = &
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),3)
            END IF

         ELSE !Interp order = 2 or 3

            IF (lLimiter .OR. lApplyLOF) THEN    
               CALL calc_limiters(Info, index)
            END IF
            IF (.NOT. request_eigens(Info, index) .AND. .NOT. lRequestRestart)  THEN
               write(*,*) "Request eigens failed at", index,j,k
               lRequestRestart=.true.
               RETURN
               !            STOP
            END IF
            ALLOCATE (dleft(NrWaves), dright(NrWaves), dcenter(NrWaves), dw(NrWaves), dw6(NrWaves), &
                 dwmdw6(NrWaves), dwpdw6(NrWaves), dw_l(NrWaves), deltaq(NrWaves), &
                 aleft(NrWaves),aright(NrWaves), acenter(NrWaves))



            ! First calculate limited slopes.  If using Characteristic Limiters - first project slopes onto eigen vectors, limit characteristics, and project back.


            IF(istimeshift(index, Info%level, dqx, mB)) THEN 
               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        DO m=1,NrWaves
                           !Calculate right, left, and the center differences.
                           dright(m)=w_%data(w_%x(i+1),j,k,1,oneDx_i(m))-w_%data(w_%x(i),j,k,1,oneDx_i(m))
                           dleft(m)=w_%data(w_%x(i),j,k,1,oneDx_i(m))-w_%data(w_%x(i-1),j,k,1,oneDx_i(m))
                           dcenter(m)=half*(dleft(m)+dright(m))
                        END DO
                        IF (.NOT. lUsePPML) THEN !limit 1st order slopes
                           IF (lChar_Limiters) THEN
                              !Map gradients to characteristic variables
                              n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,1))
                              DO m=1, n_waves
                                 aleft(m)=DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,m,:), dleft)
                                 aright(m)=DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,m,:), dright)
                                 ! Limit slopes in characteristice variables by method of VanLeer
                                 IF (SIGN(1d0,aleft(m)) == SIGN(1d0,aright(m))) THEN
                                    acenter(m)=DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,m,:), dcenter)                     
                                    acenter(m)=sign(min(2d0*abs(aleft(m)), 2d0*abs(aright(m)), abs(acenter(m))), acenter(m))
                                 ELSE
                                    acenter(m)=0d0
                                 END IF
                              END DO
                              ! Now what are we doing here?  We are symmetrizing this operation...
                              
                              
                              IF (abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,1)) > abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,n_waves))) THEN                           
                                 DO m=1,NrWaves
                                    dqx_%data(dqx_%x(i),j,k,1,m)=DOT_PRODUCT(rightX_%data(rightX_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves))
                                 END DO
                              ELSEIF (abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,1)) < abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,n_waves))) THEN 
                                 DO m=1,NrWaves
                                    dqx_%data(dqx_%x(i),j,k,1,m)=DOT_PRODUCT(rightX_%data(rightX_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1))
                                 END DO
                              ELSE !Do symmetric addition!
                                 DO m=1,NrWaves
                                    dqx_%data(dqx_%x(i),j,k,1,m)=half*(DOT_PRODUCT(rightX_%data(rightX_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves)) + DOT_PRODUCT(rightX_%data(rightX_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1)))
                                 END DO
                              END IF
                           ELSE
                              DO m=1,NrWaves
                                 IF (SIGN(1d0,dleft(m)) == SIGN(1d0,dright(m))) THEN
                                    dqx_%data(dqx_%x(i),j,k,1,m)=sign(min(2d0*abs(dleft(m)), 2d0*abs(dright(m)), abs(dcenter(m))), dcenter(m))
                                 ELSE
                                    dqx_%data(dqx_%x(i),j,k,1,m) = 0d0
                                 END IF
                              END DO
                           END IF
                        ELSE
                           dqx_%data(dqx_%x(i),j,k,1,1:NrWaves) = dcenter(1:NrWaves)
                        END IF
                        IF (lLimiter) THEN
                           IF (limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1) < 1d0) THEN
                              dqx_%data(dqx_%x(i),j,k,1,:)=dqx_%data(dqx_%x(i),j,k,1,:)*limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
            END IF

            IF(istimeshift(index, Info%level, dqy, mB)) THEN 

               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        DO m=1,NrWaves
                           dright(m)=w_%data(w_%x(i),j+1,k,1,oneDy_i(m))-w_%data(w_%x(i),j,k,1,oneDy_i(m))
                           dleft(m)=w_%data(w_%x(i),j,k,1,oneDy_i(m))-w_%data(w_%x(i),j-1,k,1,oneDy_i(m))
                           dcenter(m)=half*(dleft(m)+dright(m))
                        END DO
                        IF (.NOT. lUsePPML) THEN !limit 1st order slopes
                           IF (lChar_Limiters) THEN
                              n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,2))
                              DO m=1, n_waves
                                 aleft(m)=DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,m,:), dleft)
                                 aright(m)=DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,m,:), dright)
                                 IF (SIGN(1d0,aleft(m)) == SIGN(1d0,aright(m))) THEN
                                    acenter(m)=DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,m,:), dcenter)                     
                                    acenter(m)=sign(min(2d0*abs(aleft(m)), 2d0*abs(aright(m)), abs(acenter(m))), acenter(m))
                                 ELSE
                                    acenter(m)=0d0
                                 END IF
                              END DO
                              IF (abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,1)) > abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,n_waves))) THEN
                                 DO m=1,NrWaves
                                    dqy_%data(dqy_%x(i),j,k,1,m)=DOT_PRODUCT(rightY_%data(rightY_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves))
                                 END DO
                              ELSEIF (abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,1)) < abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,n_waves))) THEN
                                 DO m=1,NrWaves
                                    dqy_%data(dqy_%x(i),j,k,1,m)=DOT_PRODUCT(rightY_%data(rightY_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1))
                                 END DO
                              ELSE
                                 DO m=1,NrWaves
                                    dqy_%data(dqy_%x(i),j,k,1,m)=half*(DOT_PRODUCT(rightY_%data(rightY_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves)) +  DOT_PRODUCT(rightY_%data(rightY_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1)))
                                 END DO
                              END IF
                           ELSE
                              DO m=1,NrWaves
                                 IF (SIGN(1d0,dleft(m)) == SIGN(1d0,dright(m))) THEN
                                    dqy_%data(dqy_%x(i),j,k,1,m)=sign(min(2d0*abs(dleft(m)), 2d0*abs(dright(m)), abs(dcenter(m))), dcenter(m))
                                 ELSE
                                    dqy_%data(dqy_%x(i),j,k,1,m) = 0d0
                                 END IF
                              END DO
                           END IF
                        ELSE
                           dqy_%data(dqy_%x(i),j,k,1,1:NrWaves) = dcenter(1:NrWaves)
                        END IF
                     END DO
                  END DO
               END DO
            END IF

            IF(istimeshift(index, Info%level, dqz, mB)) THEN 

               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        DO m=1,NrWaves
                           dright(m)=w_%data(w_%x(i),j,k+1,1,oneDz_i(m))-w_%data(w_%x(i),j,k,1,oneDz_i(m))
                           dleft(m)=w_%data(w_%x(i),j,k,1,oneDz_i(m))-w_%data(w_%x(i),j,k-1,1,oneDz_i(m))
                           dcenter(m)=half*(dleft(m)+dright(m))
                        END DO
                        IF (.NOT. lUsePPML) THEN !limit 1st order slopes
                           IF (lChar_Limiters) THEN
                              n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,3))
                              DO m=1, n_waves
                                 aleft(m)=DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,m,:), dleft)
                                 aright(m)=DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,m,:), dright)
                                 IF (SIGN(1d0,aleft(m)) == SIGN(1d0,aright(m))) THEN
                                    acenter(m)=DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,m,:), dcenter)                     
                                    acenter(m)=sign(min(2d0*abs(aleft(m)), 2d0*abs(aright(m)), abs(acenter(m))), acenter(m))
                                 ELSE
                                    acenter(m)=0d0
                                 END IF
                              END DO
                              IF (abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,1)) > abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,n_waves))) THEN
                                 DO m=1,NrWaves
                                    dqz_%data(dqz_%x(i),j,k,1,m)=DOT_PRODUCT(rightZ_%data(rightZ_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves))
                                 END DO
                              ELSEIF (abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,1)) < abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,n_waves))) THEN
                                 DO m=1,NrWaves
                                    dqz_%data(dqz_%x(i),j,k,1,m)=DOT_PRODUCT(rightZ_%data(rightZ_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1))
                                 END DO
                              ELSE
                                 DO m=1,NrWaves
                                    dqz_%data(dqz_%x(i),j,k,1,m)=half*(DOT_PRODUCT(rightZ_%data(rightZ_%x(i),j,k,1:n_waves,m) , acenter(1:n_waves))+DOT_PRODUCT(rightZ_%data(rightZ_%x(i),j,k,n_waves:1:-1,m) , acenter(n_waves:1:-1)))
                                 END DO
                              END IF
                           ELSE
                              DO m=1,NrWaves
                                 IF (SIGN(1d0,dleft(m)) == SIGN(1d0,dright(m))) THEN
                                    dqz_%data(dqz_%x(i),j,k,1,m)=sign(min(2d0*abs(dleft(m)), 2d0*abs(dright(m)), abs(dcenter(m))), dcenter(m))
                                 ELSE
                                    dqz_%data(dqz_%x(i),j,k,1,m) = 0d0
                                 END IF
                              END DO
                           END IF
                        ELSE
                           dqz_%data(dqz_%x(i),j,k,1,1:NrWaves) = dcenter(1:NrWaves)
                        END IF
                     END DO
                  END DO
               END DO
            END IF


            IF (InterpOrder == 2) THEN

               IF (istimeshift(index, Info%level, qRx, mB)) THEN
                  CALL shift(index, Info%level, qLx, mC)

                  IF (ANY(mC(1,:) .ne. mB(1,:)+1)) THEN
                     write(*,*) "qRx and qLx are Codependent but there codependent rows are not being updated at the same time"
                     write(*,*) mC
                     write(*,*) mB
                     stop
                  end IF
                  DO i=mB(1,1),mB(1,2)
                     DO j=mB(2,1),mB(2,2)
                        DO k=mB(3,1),mB(3,2)
                           n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,1))
                           lambda_min=min(0d0,SpeedsX_%data(SpeedsX_%x(i),j,k,1,1))
                           lambda_max=max(0d0,SpeedsX_%data(SpeedsX_%x(i),j,k,1,n_waves))

                           dw(:)=dqx_%data(dqx_%x(i),j,k,1,:)
                           qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))=w_%data(w_%x(i),j,k,1,oneDx_i(:))-(half+hdtdx*lambda_min)*dw  
                           qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))=w_%data(w_%x(i),j,k,1,oneDx_i(:))+(half-hdtdx*lambda_max)*dw

                           DO waves=1, n_waves
                              IF (SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) < 0d0 .OR. lHLLTypeSolver) THEN
                                 deltaq=(lambda_min-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dw*hdtdx
                                 qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))=qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:)) + &
                                      DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,waves,:), deltaq(:))*rightX_%data(rightX_%x(i),j,k,waves,:)
                              END IF
                           END DO

                           DO waves=n_waves, 1,-1
                              IF (SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) > 0d0 .OR. lHLLTypeSolver) THEN
                                 deltaq=(lambda_max-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dw*hdtdx
                                 qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))=qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:)) + &
                                      DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,waves,:), deltaq(:))*rightX_%data(rightX_%x(i),j,k,waves,:)
                              END IF
                           END DO
                        END DO
                     END DO
                  END DO
                  IF (lMHD) THEN
                     IF (nDim == 1) THEN
                        qLx_%data(qLx_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2), mC(3,1):mC(3,2),1,iBx) = &
                             Info%q(1,1,1,iBx)
                        qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                             Info%q(1,1,1,iBx)
                     ELSE IF (nDim >= 2) THEN
                        qLx_%data(qLx_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBx) = &
                             Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),1)
                        qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBx) = &
                             Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1)
                     END IF
                  END IF
               END IF


               IF (nDim >= 2) THEN
                  if (istimeshift(index, Info%level, qRy, mB)) THEN
                     CALL shift(index, Info%level, qLy, mC)

                     IF (ANY(mC(1,:) .ne. mB(1,:))) THEN
                        write(*,*) "qRy and qLy are Codependent but there codependent rows are not being updated at the same time"
                        STOP
                     END IF
                     DO i=mB(1,1),mB(1,2)
                        DO j=mB(2,1),mB(2,2)
                           DO k=mB(3,1),mB(3,2)
                              n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,2))
                              lambda_min=min(0d0,SpeedsY_%data(SpeedsY_%x(i),j,k,1,1))
                              lambda_max=max(0d0,SpeedsY_%data(SpeedsY_%x(i),j,k,1,n_waves))
                              IF (lLimiter) THEN
                                 IF (limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1) < 1d0) THEN
                                    dqy_%data(dqy_%x(i),j,k,1,:)=dqy_%data(dqy_%x(i),j,k,1,:)*limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)
                                 END IF
                              END IF
                              dw=dqy_%data(dqy_%x(i),j,k,1,:)
                              qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))=w_%data(w_%x(i),j,k,1,oneDy_i(:))-(half+hdtdy*lambda_min)*dw
                              qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))=w_%data(w_%x(i),j,k,1,oneDy_i(:))+(half-hdtdy*lambda_max)*dw
                              DO waves=1, n_waves
                                 IF (SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) < 0d0 .OR. lHllTypeSolver) THEN
                                    deltaq=(lambda_min-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dw*hdtdy
                                 ELSE
                                    CYCLE
                                 END IF
                                 qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))=qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:)) + &
                                      DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,waves,:), deltaq(:))*rightY_%data(rightY_%x(i),j,k,waves,:)
                              END DO
                              DO waves=n_waves, 1,-1
                                 IF (SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) > 0d0 .OR. lHllTypeSolver) THEN
                                    deltaq=(lambda_max-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dw*hdtdy
                                 ELSE
                                    CYCLE
                                 END IF
                                 qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))=qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:)) + &
                                      DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,waves,:), deltaq(:))*rightY_%data(rightY_%x(i),j,k,waves,:)
                              END DO
                           END DO
                        END DO
                     END DO
                     IF (lMHD) THEN
                        qLy_%data(qLy_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBy) = &
                             Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),2)
                        qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBy) = &
                             Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),2)
                     END IF
                  END IF

                  IF (nDim >= 3) THEN
                     IF (istimeshift(index, Info%level, qRz, mB)) THEN
                        CALL shift(index, Info%level, qLz, mC)

                        IF (ANY(mC(1,:) .ne. mB(1,:))) THEN
                           write(*,*) "qRz and qLz are Codependent but there codependent rows are not being updated at the same time"
                           STOP
                        END IF
                        DO i=mB(1,1),mB(1,2)
                           DO j=mB(2,1),mB(2,2)
                              DO k=mB(3,1),mB(3,2)
                                 n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,3))
                                 lambda_min=min(0d0,SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,1))
                                 lambda_max=max(0d0,SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,n_waves))

                                 IF (lLimiter) THEN
                                    IF (limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1) < 1d0) THEN
                                       dqz_%data(dqz_%x(i),j,k,1,:)=dqz_%data(dqz_%x(i),j,k,1,:)*limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)
                                    END IF
                                 END IF
                                 dw=dqz_%data(dqz_%x(i),j,k,1,:)
                                 qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))=w_%data(w_%x(i),j,k,1,oneDz_i(:))-(half+hdtdz*lambda_min)*dw
                                 qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))=w_%data(w_%x(i),j,k,1,oneDz_i(:))+(half-hdtdz*lambda_max)*dw

                                 DO waves=1, n_waves
                                    IF (SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) < 0d0 .OR. lHLLTypeSolver) THEN
                                       deltaq=(lambda_min-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dw*hdtdz
                                    ELSE
                                       CYCLE
                                    END IF
                                    qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))=qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:)) + &
                                         DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,waves,:), deltaq(:))*rightZ_%data(rightZ_%x(i),j,k,waves,:)
                                 END DO
                                 DO waves=n_waves, 1,-1
                                    IF (SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) > 0d0 .OR. lHLLTypeSolver) THEN
                                       deltaq=(lambda_max-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dw*hdtdz
                                    ELSE
                                       CYCLE
                                    END IF
                                    qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))=qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:)) + &
                                         DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,waves,:), deltaq(:))*rightZ_%data(rightZ_%x(i),j,k,waves,:)
                                 END DO

                              END DO
                           END DO
                        END DO
                        IF (lMHD) THEN
                           qLz_%data(qLz_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBz) = &
                                Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),3)
                           qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBz) = &
                                Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),3)
                        END IF
                     END IF
                  END IF
               END IF

            ELSEIF (InterpOrder == 3) THEN
               
               IF(istimeshift(index, Info%level, qex, mB)) THEN
                  DO i=mB(1,1), mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:NrWaves)
                        qex_%data(qex_%x(i),j,k,1,m)=half*(w_%data(w_%x(i-1),j,k,1,oneDx_i(m))+w_%data(w_%x(i),j,k,1,oneDx_i(m))) - &
                             sixth*(dqx_%data(dqx_%x(i),j,k,1,m)-dqx_%data(dqx_%x(i-1),j,k,1,m))
                     END FORALL
                     
                     DO j=mB(2,1),mB(2,2)
                        DO k=mB(3,1),mB(3,2)
                           DO m=1, NrWaves
                              IF ((qex_%data(qex_%x(i),j,k,1,m) - w_%data(w_%x(i-1),j,k,1,oneDx_i(m))) * &
                                   (w_%data(w_%x(i),j,k,1,oneDx_i(m))-qex_%data(qex_%x(i),j,k,1,m)) <= 0) THEN
                                 IF (lUsePPML) THEN !need to use 2nd derivatives limitings as in ColellaSekora2008
                                    !could do this with characteristic variables as well...
                                    d2=3d0*(w_%data(w_%x(i-1),j,k,1,oneDx_i(m))-2d0*qex_%data(qex_%x(i),j,k,1,m)+w_%data(w_%x(i),j,k,1,oneDx_i(m)))
                                    d2L=w_%data(w_%x(i-2),j,k,1,oneDx_i(m))-2d0*w_%data(w_%x(i-1),j,k,1,oneDx_i(m))+w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                    d2R=w_%data(w_%x(i-1),j,k,1,oneDx_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))+w_%data(w_%x(i+1),j,k,1,oneDx_i(m))
                                    IF ((d2 > 0d0 .AND. d2L > 0d0 .AND. d2R > 0d0) .OR. (d2 < 0d0 .AND. d2L < 0d0 .AND. d2R < 0d0)) THEN
                                       d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                    ELSE
                                       d2m=0d0
                                    END IF
                                    qex_%data(qex_%x(i),j,k,1,m) = half*(w_%data(w_%x(i-1),j,k,1,oneDx_i(m))+w_%data(w_%x(i),j,k,1,oneDx_i(m))) - sixth*d2m
                                 ELSE !constrain edge values to lie between adjacent cell averages
                                    qex_%data(qex_%x(i),j,k,1,m) = min(qex_%data(qex_%x(i),j,k,1,m), &
                                         max(w_%data(w_%x(i-1),j,k,1,oneDx_i(m)),w_%data(w_%x(i),j,k,1,oneDx_i(m))))
                                    qex_%data(qex_%x(i),j,k,1,m) = max(qex_%data(qex_%x(i),j,k,1,m), &
                                         min(w_%data(w_%x(i-1),j,k,1,oneDx_i(m)),w_%data(w_%x(i),j,k,1,oneDx_i(m))))
                                 END IF
                              END IF
                           END DO
                        END DO
                     END DO
                  END DO
                  
               END IF

               IF(istimeshift(index, Info%level, qey, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:NrWaves)
                        qey_%data(qey_%x(i),j,k,1,m)=half*(w_%data(w_%x(i),j-1,k,1,oneDy_i(m))+w_%data(w_%x(i),j,k,1,oneDy_i(m))) - &
                             sixth*(dqy_%data(dqy_%x(i),j,k,1,m)-dqy_%data(dqy_%x(i),j-1,k,1,m))
                     END FORALL

                     DO j=mB(2,1),mB(2,2)
                        DO k=mB(3,1),mB(3,2)
                           DO m=1, NrWaves
                              IF ((qey_%data(qey_%x(i),j,k,1,m) - w_%data(w_%x(i),j-1,k,1,oneDy_i(m))) * &
                                   (w_%data(w_%x(i),j,k,1,oneDy_i(m))-qey_%data(qey_%x(i),j,k,1,m)) <= 0) THEN
                                 IF (lUsePPML) THEN 
                                    d2=3d0*(w_%data(w_%x(i),j-1,k,1,oneDy_i(m))-2d0*qey_%data(qey_%x(i),j,k,1,m)+w_%data(w_%x(i),j,k,1,oneDy_i(m)))
                                    d2L=w_%data(w_%x(i),j-2,k,1,oneDy_i(m))-2d0*w_%data(w_%x(i),j-1,k,1,oneDy_i(m))+w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                    d2R=w_%data(w_%x(i),j-1,k,1,oneDy_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))+w_%data(w_%x(i),j+1,k,1,oneDy_i(m))
                                    IF ((d2 > 0d0 .AND. d2L > 0d0 .AND. d2R > 0d0) .OR. (d2 < 0d0 .AND. d2L < 0d0 .AND. d2R < 0d0)) THEN
                                       d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                    ELSE
                                       d2m=0d0
                                    END IF
                                    qey_%data(qey_%x(i),j,k,1,m) = half*(w_%data(w_%x(i),j-1,k,1,oneDy_i(m))+w_%data(w_%x(i),j,k,1,oneDy_i(m))) - sixth*d2m
                                 ELSE
                                    qey_%data(qey_%x(i),j,k,1,m) = min(qey_%data(qey_%x(i),j,k,1,m), &
                                         max(w_%data(w_%x(i),j-1,k,1,oneDy_i(m)),w_%data(w_%x(i),j,k,1,oneDy_i(m))))
                                    qey_%data(qey_%x(i),j,k,1,m) = max(qey_%data(qey_%x(i),j,k,1,m), &
                                         min(w_%data(w_%x(i),j-1,k,1,oneDy_i(m)),w_%data(w_%x(i),j,k,1,oneDy_i(m))))
                                 END IF
                              END IF
                           END DO
                        END DO
                     END DO
                  END DO
               END IF
               
               IF(istimeshift(index, Info%level, qez, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:NrWaves)
                        qez_%data(qez_%x(i),j,k,1,m)=half*(w_%data(w_%x(i),j,k-1,1,oneDz_i(m))+w_%data(w_%x(i),j,k,1,oneDz_i(m))) - &
                             sixth*(dqz_%data(dqz_%x(i),j,k,1,m)-dqz_%data(dqz_%x(i),j,k-1,1,m))
                     END FORALL
                     DO j=mB(2,1),mB(2,2)
                        DO k=mB(3,1),mB(3,2)
                           DO m=1, NrWaves
                              IF ((qez_%data(qez_%x(i),j,k,1,m) - w_%data(w_%x(i),j,k-1,1,oneDz_i(m))) * &
                                   (w_%data(w_%x(i),j,k,1,oneDz_i(m))-qez_%data(qez_%x(i),j,k,1,m)) <= 0) THEN
                                 IF (lUsePPML) THEN 
                                    d2=3d0*(w_%data(w_%x(i),j,k-1,1,oneDz_i(m))-2d0*qez_%data(qez_%x(i),j,k,1,m)+w_%data(w_%x(i),j,k,1,oneDz_i(m)))
                                    d2L=w_%data(w_%x(i),j,k-2,1,oneDz_i(m))-2d0*w_%data(w_%x(i),j,k-1,1,oneDz_i(m))+w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                    d2R=w_%data(w_%x(i),j,k-1,1,oneDz_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))+w_%data(w_%x(i),j,k+1,1,oneDz_i(m))
                                    IF ((d2 > 0d0 .AND. d2L > 0d0 .AND. d2R > 0d0) .OR. (d2 < 0d0 .AND. d2L < 0d0 .AND. d2R < 0d0)) THEN
                                       d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                    ELSE
                                       d2m=0d0
                                    END IF
                                    qez_%data(qez_%x(i),j,k,1,m) = half*(w_%data(w_%x(i),j,k-1,1,oneDz_i(m))+w_%data(w_%x(i),j,k,1,oneDz_i(m))) - sixth*d2m
                                 ELSE
                                    qez_%data(qez_%x(i),j,k,1,m) = min(qez_%data(qez_%x(i),j,k,1,m), &
                                         max(w_%data(w_%x(i),j,k-1,1,oneDz_i(m)),w_%data(w_%x(i),j,k,1,oneDz_i(m))))
                                    qez_%data(qez_%x(i),j,k,1,m) = max(qez_%data(qez_%x(i),j,k,1,m), &
                                         min(w_%data(w_%x(i),j,k-1,1,oneDz_i(m)),w_%data(w_%x(i),j,k,1,oneDz_i(m))))
                                 END IF
                              END IF
                           END DO
                        END DO
                     END DO
                  END DO
               END IF

               !Now we have spatially interpolated interface values we can construct left and right interface states.

               IF (istimeshift(index, Info%level, qRx, mB)) THEN
                  CALL shift(index, Info%level, qLx, mC)
                  IF (ANY(mC(1,:) .ne. mB(1,:)+1)) THEN
                     write(*,*) "qRx and qLx are Codependent but there codependent rows are not being updated at the same time"
                     stop
                  end IF
                  DO i=mB(1,1),mB(1,2)
                     DO j=mB(2,1),mB(2,2)
                        DO k=mB(3,1),mB(3,2)
                           DO m=1,NrWaves
                              IF (.NOT. lUsePPML) THEN
                                 IF ((qex_%data(qex_%x(i+1),j,k,1,m)-w_%data(w_%x(i),j,k,1,oneDx_i(m))) * &
                                      (w_%data(w_%x(i),j,k,1,oneDx_i(m))-qex_%data(qex_%x(i),j,k,1,m)) <= 0) THEN 
                                    ! Reconstructed value is not monotone so flatten reconstruction in this cell
                                    qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                    qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                 ELSE ! Make sure that parabolic reconstruction does not produce local maxima/minima
                                    dq=qex_%data(qex_%x(i+1),j,k,1,m)-qex_%data(qex_%x(i),j,k,1,m)
                                    sdq2=sixth*dq**2
                                    qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDx_i(m))-half*(qex_%data(qex_%x(i),j,k,1,m)+qex_%data(qex_%x(i+1),j,k,1,m)))
                                    IF (qmhsq > sdq2) THEN
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))-2d0*qex_%data(qex_%x(i+1),j,k,1,m)
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i+1),j,k,1,m)
                                    ELSE IF (qmhsq < -sdq2) THEN
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))-2d0*qex_%data(qex_%x(i),j,k,1,m)
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i),j,k,1,m)
                                    ELSE
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i),j,k,1,m)
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i+1),j,k,1,m)
                                    END IF
                                 END IF
                              ELSE
                                 IF ((qex_%data(qex_%x(i+1),j,k,1,m)-w_%data(w_%x(i),j,k,1,oneDx_i(m))) * &
                                      (w_%data(w_%x(i),j,k,1,oneDx_i(m))-qex_%data(qex_%x(i),j,k,1,m)) <= 0d0 .OR. &
                                      (w_%data(w_%x(i+1),j,k,1,oneDx_i(m)) - w_%data(w_%x(i),j,k,1,oneDx_i(m))) * & 
                                      (w_%data(w_%x(i),j,k,1,oneDx_i(m)) - w_%data(w_%x(i-1),j,k,1,oneDx_i(m))) <= 0d0) THEN
                                    ! Edges produce a new local extrema or cell is an existing local extrema

                                    ! Perform 2nd order limiting
                                    ! could do this with characteristic variables as well...
                                    d2= -6d0*(2d0*w_%data(w_%x(i),j,k,1,oneDx_i(m)) - (qex_%data(qex_%x(i),j,k,1,m)+qex_%data(qex_%x(i+1),j,k,1,m)))
                                    d2C=w_%data(w_%x(i-1),j,k,1,oneDx_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))+w_%data(w_%x(i+1),j,k,1,oneDx_i(m))
                                    d2L=w_%data(w_%x(i-2),j,k,1,oneDx_i(m))-2d0*w_%data(w_%x(i-1),j,k,1,oneDx_i(m))+w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                    d2R=w_%data(w_%x(i),j,k,1,oneDx_i(m))-2d0*w_%data(w_%x(i+1),j,k,1,oneDx_i(m))+w_%data(w_%x(i+2),j,k,1,oneDx_i(m))
                                    IF (abs(sum(sign(1d0, (/d2, d2C, d2L, d2R/)))) == 4d0) THEN
                                       d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                    ELSE                                       
                                       d2m=0d0
                                    END IF
                                    IF (d2 == 0d0) THEN
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m))
                                    ELSE
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m)) + d2m/d2*(qex_%data(qex_%x(i),j,k,1,m) - w_%data(w_%x(i),j,k,1,oneDx_i(m)))
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=w_%data(w_%x(i),j,k,1,oneDx_i(m)) + d2m/d2*(qex_%data(qex_%x(i+1),j,k,1,m) - w_%data(w_%x(i),j,k,1,oneDx_i(m)))
                                    END IF
                                 ELSE !use original PPM limiter
                                    dq=qex_%data(qex_%x(i+1),j,k,1,m)-qex_%data(qex_%x(i),j,k,1,m)
                                    sdq2=sixth*dq**2
                                    qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDx_i(m))-half*(qex_%data(qex_%x(i),j,k,1,m)+qex_%data(qex_%x(i+1),j,k,1,m)))
                                    IF (qmhsq > sdq2) THEN
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))-2d0*qex_%data(qex_%x(i+1),j,k,1,m)
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i+1),j,k,1,m)
                                    ELSE IF (qmhsq < -sdq2) THEN
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDx_i(m))-2d0*qex_%data(qex_%x(i),j,k,1,m)
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i),j,k,1,m)
                                    ELSE
                                       qRx_%data(qRx_%x(i),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i),j,k,1,m)
                                       qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(m))=qex_%data(qex_%x(i+1),j,k,1,m)
                                    END IF
                                 END IF
                              END IF
                           END DO
                           

                           n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,1))
                           lambda_min=min(0d0,SpeedsX_%data(SpeedsX_%x(i),j,k,1,1))
                           lambda_max=max(0d0,SpeedsX_%data(SpeedsX_%x(i),j,k,1,n_waves))

                           dw(:)=hdtdx*(qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))-qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:)))
                           dw6(:)=three_dtdx*(w_%data(w_%x(i),j,k,1,oneDx_i(:))-half* &
                                (qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))+qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))))

                           dwmdw6=dw-dw6
                           dwpdw6=dw+dw6
                           dw6=two_thirds_dtdx*dw6 
                           dw_l=hdtdx*dqx_%data(dqx_%x(i),j,k,1,:)
                           
                           qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))=qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))-&
                                lambda_min*(dwpdw6)-dw6*lambda_min**2
                           qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))=qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))-&
                                lambda_max*(dwmdw6)-dw6*lambda_max**2
                           
                           DO waves=1, n_waves
                              IF (SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) < 0d0) THEN
                                 deltaq=(lambda_min-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dwpdw6 + &
                                      (lambda_min**2-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves)**2)*dw6
                              ELSEIF (lHLLTypeSolver .AND. SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) > 0d0) THEN 
                                 deltaq=(lambda_min-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dw_l
                              ELSE            
                                 CYCLE
                              END IF
                              qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:))=qRx_%data(qRx_%x(i),j,k,1,oneDx_i(:)) + &
                                      DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,waves,:), deltaq(:))*rightX_%data(rightX_%x(i),j,k,waves,:)
                           END DO

                           DO waves=n_waves, 1,-1
                              IF (SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) > 0d0) THEN
                                 deltaq=(lambda_max-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dwmdw6 + &
                                      (lambda_max**2-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves)**2)*dw6
                              ELSEIF (lHLLTypeSolver .AND. SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves) < 0d0) THEN 
                                 deltaq=(lambda_max-SpeedsX_%data(SpeedsX_%x(i),j,k,1,waves))*dw_l
                              ELSE
                                 CYCLE
                              END IF
                              qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:))=qLx_%data(qLx_%x(i+1),j,k,1,oneDx_i(:)) + &
                                   DOT_PRODUCT(leftX_%data(leftX_%x(i),j,k,waves,:), deltaq(:))*rightX_%data(rightX_%x(i),j,k,waves,:)
                           END DO
                        END DO
                     END DO
                  END DO

                  IF (lMHD) THEN
                     IF (nDim == 1) THEN
                        qLx_%data(qLx_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2), mC(3,1):mC(3,2),1,iBx) = &
                             Info%q(1,1,1,iBx)
                        qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                             Info%q(1,1,1,iBx)
                     ELSE IF (nDim >= 2) THEN
                        qLx_%data(qLx_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBx) = &
                             Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),1)
                        qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBx) = &
                             Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1)
                     END IF
                  END IF
               END IF
               IF (nDim >= 2) THEN
                  if (istimeshift(index, Info%level, qRy, mB)) THEN
                     CALL shift(index, Info%level, qLy, mC)
                     IF (ANY(mC(1,:) .ne. mB(1,:))) THEN
                        write(*,*) "qRy and qLy are Codependent but there codependent rows are not being updated at the same time"
                        STOP
                     END IF
                     DO i=mB(1,1),mB(1,2)
                        DO j=mB(2,1),mB(2,2)
                           DO k=mB(3,1),mB(3,2)
                              DO m=1,NrWaves
                                 IF (.NOT. lUsePPML) THEN
                                    IF ((qey_%data(qey_%x(i),j+1,k,1,m)-w_%data(w_%x(i),j,k,1,oneDy_i(m))) * &
                                         (w_%data(w_%x(i),j,k,1,oneDy_i(m))-qey_%data(qey_%x(i),j,k,1,m)) <= 0) THEN
                                       qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                       qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                    ELSE
                                       dq=qey_%data(qey_%x(i),j+1,k,1,m)-qey_%data(qey_%x(i),j,k,1,m)
                                       sdq2=sixth*dq**2
                                       qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDy_i(m))-half*(qey_%data(qey_%x(i),j,k,1,m)+qey_%data(qey_%x(i),j+1,k,1,m)))
                                       IF (qmhsq > sdq2) THEN
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))-2d0*qey_%data(qey_%x(i),j+1,k,1,m)
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j+1,k,1,m)
                                       ELSE IF (qmhsq < -sdq2) THEN
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))-2d0*qey_%data(qey_%x(i),j,k,1,m)
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j,k,1,m)
                                       ELSE
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j,k,1,m)
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j+1,k,1,m)
                                       END IF
                                    END IF
                                 ELSE
                                    IF ((qey_%data(qey_%x(i),j+1,k,1,m)-w_%data(w_%x(i),j,k,1,oneDy_i(m))) * &
                                         (w_%data(w_%x(i),j,k,1,oneDy_i(m))-qey_%data(qey_%x(i),j,k,1,m)) <= 0d0 .OR. &
                                         (w_%data(w_%x(i),j+1,k,1,oneDy_i(m)) - w_%data(w_%x(i),j,k,1,oneDy_i(m))) * & 
                                         (w_%data(w_%x(i),j,k,1,oneDy_i(m)) - w_%data(w_%x(i),j-1,k,1,oneDy_i(m))) <= 0d0) THEN
                                       ! Edges produce a new local extrema or cell is an existing local extrema
                                       
                                       ! Perform 2nd order limiting
                                       ! could do this with characteristic variables as well...
                                       d2= -6d0*(2d0*w_%data(w_%x(i),j,k,1,oneDy_i(m)) - (qey_%data(qey_%x(i),j,k,1,m)+qey_%data(qey_%x(i),j+1,k,1,m)))
                                       d2C=w_%data(w_%x(i),j-1,k,1,oneDy_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))+w_%data(w_%x(i),j+1,k,1,oneDy_i(m))
                                       d2L=w_%data(w_%x(i),j-2,k,1,oneDy_i(m))-2d0*w_%data(w_%x(i),j-1,k,1,oneDy_i(m))+w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                       d2R=w_%data(w_%x(i),j,k,1,oneDy_i(m))-2d0*w_%data(w_%x(i),j+1,k,1,oneDy_i(m))+w_%data(w_%x(i),j+2,k,1,oneDy_i(m))
                                       IF (abs(sum(sign(1d0, (/d2, d2C, d2L, d2R/)))) == 4d0) THEN
                                          d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                       ELSE                                       
                                          d2m=0d0
                                       END IF
                                       IF (d2 == 0d0) THEN
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m))
                                       ELSE
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m)) + d2m/d2*(qey_%data(qey_%x(i),j,k,1,m) - w_%data(w_%x(i),j,k,1,oneDy_i(m)))
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=w_%data(w_%x(i),j,k,1,oneDy_i(m)) + d2m/d2*(qey_%data(qey_%x(i),j+1,k,1,m) - w_%data(w_%x(i),j,k,1,oneDy_i(m)))
                                       END IF
                                    ELSE !use original PPM limiter
                                       dq=qey_%data(qey_%x(i),j+1,k,1,m)-qey_%data(qey_%x(i),j,k,1,m)
                                       sdq2=sixth*dq**2
                                       qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDy_i(m))-half*(qey_%data(qey_%x(i),j,k,1,m)+qey_%data(qey_%x(i),j+1,k,1,m)))
                                       IF (qmhsq > sdq2) THEN
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))-2d0*qey_%data(qey_%x(i),j+1,k,1,m)
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j+1,k,1,m)
                                       ELSE IF (qmhsq < -sdq2) THEN
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDy_i(m))-2d0*qey_%data(qey_%x(i),j,k,1,m)
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j,k,1,m)
                                       ELSE
                                          qRy_%data(qRy_%x(i),j,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j,k,1,m)
                                          qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(m))=qey_%data(qey_%x(i),j+1,k,1,m)
                                       END IF
                                    END IF
                                 END IF
                              END DO
                              n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,2))
                              lambda_min=min(0d0,SpeedsY_%data(SpeedsY_%x(i),j,k,1,1))
                              lambda_max=max(0d0,SpeedsY_%data(SpeedsY_%x(i),j,k,1,n_waves))

                              dw(:)=hdtdy*(qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))-qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:)))
                              dw6(:)=three_dtdy*(w_%data(w_%x(i),j,k,1,oneDy_i(:))-half* &
                                   (qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))+qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))))

                              dwmdw6=dw-dw6
                              dwpdw6=dw+dw6
                              dw6=two_thirds_dtdy*dw6
                              dw_l=hdtdy*dqy_%data(dqy_%x(i),j,k,1,:)
                              
                              qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))=qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))-&
                                   lambda_min*(dwpdw6)-dw6*lambda_min**2
                              qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))=qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))-&
                                   lambda_max*(dwmdw6)-dw6*lambda_max**2

                              DO waves=1, n_waves
                                 IF (SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) < 0d0) THEN
                                    deltaq=(lambda_min-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dwpdw6 + &
                                         (lambda_min**2-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves)**2)*dw6
                                 ELSEIF (lHLLTypeSolver .AND. SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) > 0d0) THEN
                                    deltaq=(lambda_min-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dw_l
                                 ELSE
                                    CYCLE
                                 END IF
                                 qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:))=qRy_%data(qRy_%x(i),j,k,1,oneDy_i(:)) + &
                                      DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,waves,:), deltaq(:))*rightY_%data(rightY_%x(i),j,k,waves,:)
                              END DO
                              DO waves=n_waves, 1,-1
                                 IF (SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) > 0d0) THEN
                                    deltaq=(lambda_max-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dwmdw6 + &
                                         (lambda_max**2-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves)**2)*dw6
                                 ELSEIF (lHLLTypeSolver .AND. SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves) < 0d0) THEN 
                                    deltaq=(SpeedsY_%data(SpeedsY_%x(i),j,k,1,n_waves)-SpeedsY_%data(SpeedsY_%x(i),j,k,1,waves))*dw_l
                                 ELSE
                                    CYCLE
                                 END IF
                                 qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:))=qLy_%data(qLy_%x(i),j+1,k,1,oneDy_i(:)) + &
                                      DOT_PRODUCT(leftY_%data(leftY_%x(i),j,k,waves,:), deltaq(:))*rightY_%data(rightY_%x(i),j,k,waves,:)
                              END DO
                           END DO
                        END DO
                     END DO
                     IF (lMHD) THEN
                        qLy_%data(qLy_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBy) = &
                             Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),2)
                        qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBy) = &
                             Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),2)
                     END IF

                  END if
                  IF (nDim >= 3) THEN
                     IF (istimeshift(index, Info%level, qRz, mB)) THEN
                        CALL shift(index, Info%level, qLz, mC)

                        IF (ANY(mC(1,:) .ne. mB(1,:))) THEN
                           write(*,*) "qRz and qLz are Codependent but there codependent rows are not being updated at the same time"
                           STOP
                        END IF
                        DO i=mB(1,1),mB(1,2)
                           DO j=mB(2,1),mB(2,2)
                              DO k=mB(3,1),mB(3,2)
                                 DO m=1,NrWaves
                                    IF (.NOT. lUsePPML) THEN
                                       IF ((qez_%data(qez_%x(i),j,k+1,1,m)-w_%data(w_%x(i),j,k,1,oneDz_i(m))) * &
                                            (w_%data(w_%x(i),j,k,1,oneDz_i(m))-qez_%data(qez_%x(i),j,k,1,m)) <= 0) THEN
                                          qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                          qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                       ELSE
                                          dq=qez_%data(qez_%x(i),j,k+1,1,m)-qez_%data(qez_%x(i),j,k,1,m)
                                          sdq2=sixth*dq**2
                                          qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDz_i(m))-half*(qez_%data(qez_%x(i),j,k,1,m)+qez_%data(qez_%x(i),j,k+1,1,m)))
                                          IF (qmhsq > sdq2) THEN
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))-2d0*qez_%data(qez_%x(i),j,k+1,1,m)
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k+1,1,m)
                                          ELSE IF (qmhsq < -sdq2) THEN
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))-2d0*qez_%data(qez_%x(i),j,k,1,m)
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k,1,m)
                                          ELSE
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k,1,m)
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k+1,1,m)
                                          END IF
                                       END IF
                                    ELSE
                                       IF ((qez_%data(qez_%x(i),j,k+1,1,m)-w_%data(w_%x(i),j,k,1,oneDz_i(m))) * &
                                            (w_%data(w_%x(i),j,k,1,oneDz_i(m))-qez_%data(qez_%x(i),j,k,1,m)) <= 0d0 .OR. &
                                            (w_%data(w_%x(i),j,k+1,1,oneDz_i(m)) - w_%data(w_%x(i),j,k,1,oneDz_i(m))) * & 
                                            (w_%data(w_%x(i),j,k,1,oneDz_i(m)) - w_%data(w_%x(i),j,k-1,1,oneDz_i(m))) <= 0d0) THEN
                                          ! Edges produce a new local extrema or cell is an existing local extrema
                                          
                                          ! Perform 2nd order limiting
                                          ! could do this with characteristic variables as well...
                                          d2= -6d0*(2d0*w_%data(w_%x(i),j,k,1,oneDz_i(m)) - (qez_%data(qez_%x(i),j,k,1,m)+qez_%data(qez_%x(i),j,k+1,1,m)))
                                          d2C=w_%data(w_%x(i),j,k-1,1,oneDz_i(m))-2d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))+w_%data(w_%x(i),j,k+1,1,oneDz_i(m))
                                          d2L=w_%data(w_%x(i),j,k-2,1,oneDz_i(m))-2d0*w_%data(w_%x(i),j,k-1,1,oneDz_i(m))+w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                          d2R=w_%data(w_%x(i),j,k,1,oneDz_i(m))-2d0*w_%data(w_%x(i),j,k+1,1,oneDz_i(m))+w_%data(w_%x(i),j,k+2,1,oneDz_i(m))
                                          IF (abs(sum(sign(1d0, (/d2, d2C, d2L, d2R/)))) == 4d0) THEN
                                             d2m=sign(min(PPMC*abs(d2L), PPMC*abs(d2R), abs(d2)), d2)
                                          ELSE                                       
                                             d2m=0d0
                                          END IF
                                          IF (d2 == 0d0) THEN
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m))
                                          ELSE
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m)) + d2m/d2*(qez_%data(qez_%x(i),j,k,1,m) - w_%data(w_%x(i),j,k,1,oneDz_i(m)))
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=w_%data(w_%x(i),j,k,1,oneDz_i(m)) + d2m/d2*(qez_%data(qez_%x(i),j,k+1,1,m) - w_%data(w_%x(i),j,k,1,oneDz_i(m)))
                                          END IF
                                       ELSE
                                          dq=qez_%data(qez_%x(i),j,k+1,1,m)-qez_%data(qez_%x(i),j,k,1,m)
                                          sdq2=sixth*dq**2
                                          qmhsq=dq*(w_%data(w_%x(i),j,k,1,oneDz_i(m))-half*(qez_%data(qez_%x(i),j,k,1,m)+qez_%data(qez_%x(i),j,k+1,1,m)))
                                          IF (qmhsq > sdq2) THEN
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))-2d0*qez_%data(qez_%x(i),j,k+1,1,m)
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k+1,1,m)
                                          ELSE IF (qmhsq < -sdq2) THEN
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=3d0*w_%data(w_%x(i),j,k,1,oneDz_i(m))-2d0*qez_%data(qez_%x(i),j,k,1,m)
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k,1,m)
                                          ELSE
                                             qRz_%data(qRz_%x(i),j,k,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k,1,m)
                                             qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(m))=qez_%data(qez_%x(i),j,k+1,1,m)
                                          END IF
                                       END IF
                                    END IF
                                 END DO
                                 n_waves=NINT(nWaves_%data(nWaves_%x(i),j,k,1,3))
                                 lambda_min=min(0d0,SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,1))
                                 lambda_max=max(0d0,SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,n_waves))

                                 dw(:)=hdtdz*(qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))-qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:)))
                                 dw6(:)=three_dtdz*(w_%data(w_%x(i),j,k,1,oneDz_i(:))-half* &
                                      (qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))+qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))))

                                 dwmdw6=dw-dw6
                                 dwpdw6=dw+dw6
                                 dw6=two_thirds_dtdz*dw6
                                 dw_l=hdtdz*dqz_%data(dqz_%x(i),j,k,1,:)

                                 qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))=qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))-&
                                      lambda_min*(dwpdw6)-dw6*lambda_min**2
                                 qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))=qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))-&
                                      lambda_max*(dwmdw6)-dw6*lambda_max**2

                                 DO waves=1, n_waves
                                    IF (SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) < 0d0) THEN
                                       deltaq=(lambda_min-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dwpdw6 + &
                                            (lambda_min**2-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves)**2)*dw6
                                    ELSEIF (lHLLTypeSolver .AND. SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) > 0d0) THEN 
                                       deltaq=(lambda_min-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dw_l
                                    ELSE
                                       CYCLE
                                    END IF
                                    qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:))=qRz_%data(qRz_%x(i),j,k,1,oneDz_i(:)) + &
                                         DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,waves,:), deltaq(:))*rightZ_%data(rightZ_%x(i),j,k,waves,:)
                                 END DO
                                 DO waves=n_waves, 1,-1
                                    IF (SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) > 0d0) THEN
                                       deltaq=(lambda_max-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dwmdw6 + &
                                            (lambda_max**2-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves)**2)*dw6
                                    ELSEIF (lHLLTypeSolver .AND. SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves) < 0d0) THEN 
                                       deltaq=(lambda_max-SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,waves))*dw_l
                                    ELSE
                                       CYCLE
                                    END IF
                                    qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:))=qLz_%data(qLz_%x(i),j,k+1,1,oneDz_i(:)) + &
                                         DOT_PRODUCT(leftZ_%data(leftZ_%x(i),j,k,waves,:), deltaq(:))*rightZ_%data(rightZ_%x(i),j,k,waves,:)
                                 END DO
                              END DO
                           END DO
                        END DO
                        IF (lMHD) THEN
                           qLz_%data(qLz_%x(mC(1,1):mC(1,2)),mC(2,1):mC(2,2),mC(3,1):mC(3,2),1,iBz) = &
                                Info%aux(index+mC(1,1):index+mC(1,2),mC(2,1):mC(2,2), mC(3,1):mC(3,2),3)
                           qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,iBz) = &
                                Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),3)
                        END IF
                     END IF
                  END IF
               END IF
            END IF

            IF (lMHD) CALL MHD_Source_Terms(Info, index)
            DEALLOCATE (dleft, dright, dcenter, dw, dw6, dwmdw6, dwpdw6, dw_l, deltaq, &
                 aleft,aright, acenter)
            
         END IF

         IF (lSelfGravity) THEN
            IF (istimeshift(index, Info%level, gradphix, mB)) THEN
               DO i=mB(1,1), mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     gradphix_%data(gradphix_%x(i),j,k,1,1)=(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i-1,j,k,iPhiGas))/dx
                  END FORALL
               END DO
            END IF
            IF (nDim >= 2) THEN
               IF (istimeshift(index, Info%level, gradphiy, mB)) THEN
                  DO i=mB(1,1), mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        gradphiy_%data(gradphiy_%x(i),j,k,1,1)=(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i,j-1,k,iPhiGas))/dx
                     END FORALL
                  END DO
               END IF
               IF (nDim >= 3) THEN
                  IF (istimeshift(index, Info%level, gradphiz, mB)) THEN
                     DO i=mB(1,1), mB(1,2)
                        FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                           gradphiz_%data(gradphiz_%x(i),j,k,1,1)=(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i,j,k-1,iPhiGas))/dx
                        END FORALL
                     END DO
                  END IF
               END IF
            END IF


            IF (istime(index, Info%level, qRx, mB)) THEN
               IF (lSelfGravity) THEN
                  qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) = &
                       qRx_%data(qRx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) - hdt*&
                       gradphix_%data(gradphix_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)
               END IF
            END IF

            IF (istime(index, Info%level, qLx, mB)) THEN
               IF (lSelfGravity) THEN
                  qLx_%data(qLx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) = &
                       qLx_%data(qLx_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) - hdt*&
                       gradphix_%data(gradphix_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)
               END IF
            END IF

            IF (istime(index, Info%level, qRy, mB)) THEN
               IF (lSelfGravity) THEN
                  qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) = &
                       qRy_%data(qRy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) - hdt*&
                       gradphiy_%data(gradphiy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)
               END IF
            END IF

            IF (istime(index, Info%level, qLy, mB)) THEN
               IF (lSelfGravity) THEN
                  qLy_%data(qLy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) = &
                       qLy_%data(qLy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) - hdt*&
                       gradphiy_%data(gradphiy_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)
               END IF
            END IF

            IF (istime(index, Info%level, qRz, mB)) THEN
               IF (lSelfGravity) THEN
                  qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) = &
                       qRz_%data(qRz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) - hdt*&
                       gradphiz_%data(gradphiz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)

               END IF
            END IF

            IF (istime(index, Info%level, qLz, mB)) THEN
               IF (lSelfGravity) THEN
                  qLz_%data(qLz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) = &
                       qLz_%data(qLz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) - hdt*&
                       gradphiz_%data(gradphiz_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1)
               END IF
            END IF

         END IF


         IF (lCautious) THEN
            IF (istime(index, Info%level, qRx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qRx_%data(qRx_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO

            END IF
            IF (istime(index, Info%level, qLx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qLx_%data(qLx_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO

            END IF
            IF (istime(index, Info%level, qRy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qRy_%data(qRy_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO
            END IF
            IF (istime(index, Info%level, qLy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qLy_%data(qLy_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO
            END IF
            IF (istime(index, Info%level, qRz, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qRz_%data(qRz_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO
            END IF
            IF (istime(index, Info%level, qLz, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL protect_all(qLz_%data(qLz_%x(i), mB(2,1):mB(2,2), mB(3,1):mB(3,2), 1,:))
               END DO
            END IF
         END IF


      END SUBROUTINE Reconstruct


      !> Calculate limiters
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates limiter_ppm by calculating 1d limiters
      !! Calculates 1d limiters by calculating pressures
      SUBROUTINE calc_limiters(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mC(3,2)    ! Bounds of slab to update
         REAL(KIND=qPrec) :: dPT1, dPT2
         INTEGER :: m
         REAL(KIND=qPrec), PARAMETER :: CA1=10d0, & !1d0/(.85-.75)
              CA2=.85d0, &
              CZ1=2d0, & !1d0/(.75-.25)
              CZ2=.75d0
         IF (istimeshift(index, Info%level, pT, mB)) THEN
            IF (lIsothermal) THEN
               DO i=mB(1,1), mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     pT_%data(pT_%x(i),j,k,1,1)=w_%data(w_%x(i),j,k,1,1)*Iso_Speed2
                  END FORALL
               END DO
            ELSE
               DO i=mB(1,1), mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     pT_%data(pT_%x(i),j,k,1,1)=w_%data(w_%x(i),j,k,1,iE)
                  END FORALL
               END DO
            END IF
         END IF
         IF (istimeshift(index, Info%level, limiter_x, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     IF(w_%data(w_%x(i+1),j,k,1,ivx) < w_%data(w_%x(i-1),j,k,1,ivx)) THEN
                        dPT2=ABS(pT_%data(pT_%x(i+2),j,k,1,1) - pT_%data(pT_%x(i-2),j,k,1,1))
                        IF (dPT2 > epsilon) THEN
                           dPT1 = ABS(pT_%data(pT_%x(i+1),j,k,1,1) - pT_%data(pT_%x(i-1),j,k,1,1))
                           limiter_x_%data(limiter_x_%x(i),j,k,1,1)=MAX(MIN(1d0,CZ1*(CZ2-dPT1/min(pT_%data(pT_%x(i+1),j,k,1,1),pT_%data(pT_%x(i-1),j,k,1,1)) )), &
                                MAX(0d0, MIN(1d0,CA1*(CA2-dPT1/dPT2))))
                        ELSE
                           limiter_x_%data(limiter_x_%x(i),j,k,1,1)=1d0
                        END IF
                     ELSE
                        limiter_x_%data(limiter_x_%x(i),j,k,1,1)=1d0
                     END IF
                  END DO
               END DO
            END DO
         END IF
         IF (istimeshift(index, Info%level, limiter_y, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     IF(w_%data(w_%x(i),j+1,k,1,ivy) < w_%data(w_%x(i),j-1,k,1,ivy)) THEN
                        dPT2=ABS(pT_%data(pT_%x(i),j+2,k,1,1) - pT_%data(pT_%x(i),j-2,k,1,1))
                        IF (dPT2 > epsilon) THEN
                           dPT1 = ABS(pT_%data(pT_%x(i),j+1,k,1,1) - pT_%data(pT_%x(i),j-1,k,1,1))
                           limiter_y_%data(limiter_y_%x(i),j,k,1,1)=MAX(MIN(1d0,CZ1*(CZ2-dPT1/min(pT_%data(pT_%x(i),j+1,k,1,1),pT_%data(pT_%x(i),j-1,k,1,1)) )), &
                                MAX(0d0, MIN(1d0,CA1*(CA2-dPT1/dPT2))))
                        ELSE
                           limiter_y_%data(limiter_y_%x(i),j,k,1,1)=1d0
                        END IF
                     ELSE
                        limiter_y_%data(limiter_y_%x(i),j,k,1,1)=1d0
                     END IF
                  END DO
               END DO
            END DO
         END IF
         IF (istimeshift(index, Info%level, limiter_z, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     IF(w_%data(w_%x(i),j,k+1,1,ivz) < w_%data(w_%x(i),j,k-1,1,ivz)) THEN
                        dPT2=ABS(pT_%data(pT_%x(i),j,k+2,1,1) - pT_%data(pT_%x(i),j,k-2,1,1))
                        IF (dPT2 > epsilon) THEN
                           dPT1 = ABS(pT_%data(pT_%x(i),j,k+1,1,1) - pT_%data(pT_%x(i),j,k-1,1,1))
                           limiter_z_%data(limiter_z_%x(i),j,k,1,1)=MAX(MIN(1d0,CZ1*(CZ2-dPT1/min(pT_%data(pT_%x(i),j,k+1,1,1),pT_%data(pT_%x(i),j,k-1,1,1)) )), &
                                MAX(0d0, MIN(1d0,CA1*(CA2-dPT1/dPT2))))

                        ELSE
                           limiter_z_%data(limiter_z_%x(i),j,k,1,1)=1d0
                        END IF
                     ELSE
                        limiter_z_%data(limiter_z_%x(i),j,k,1,1)=1d0
                     END IF
                  END DO
               END DO
            END DO
         END IF
         IF (istimeshift(index, Info%level, limiter_ppm, mB)) THEN
            IF (nDim == 1) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)= MINVAL((/limiter_x_%data(limiter_x_%x(i-1),j,k,1,1), &
                          limiter_x_%data(limiter_x_%x(i),j,k,1,1),limiter_x_%data(limiter_x_%x(i+1),j,k,1,1)/))
                  END FORALL
               END DO
            ELSE IF (nDim == 2) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)= MINVAL((/limiter_x_%data(limiter_x_%x(i-1),j-1:j+1,k,1,1), &
                          limiter_x_%data(limiter_x_%x(i),j-1:j+1,k,1,1),limiter_x_%data(limiter_x_%x(i+1),j-1:j+1,k,1,1)/))
                  END FORALL
               END DO
            ELSE IF (nDim == 3) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)= MINVAL((/limiter_x_%data(limiter_x_%x(i-1),j-1:j+1,k-1:k+1,1,1), &
                          limiter_x_%data(limiter_x_%x(i),j-1:j+1,k-1:k+1,1,1),limiter_x_%data(limiter_x_%x(i+1),j-1:j+1,k-1:k+1,1,1)/))
                  END FORALL
               END DO
            END IF
            IF (nDim >= 2) THEN         
               IF (nDim == 2) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)=MIN(limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1), &
                             MINVAL((/limiter_y_%data(limiter_y_%x(i-1),j-1:j+1,k,1,1),limiter_y_%data(limiter_y_%x(i),j-1:j+1,k,1,1),limiter_y_%data(limiter_y_%x(i+1),j-1:j+1,k,1,1)/)))
                     END FORALL
                  END DO
               ELSE
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)=MIN(limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1), &
                             MINVAL((/limiter_y_%data(limiter_y_%x(i-1),j-1:j+1,k-1:k+1,1,1),limiter_y_%data(limiter_y_%x(i),j-1:j+1,k-1:k+1,1,1),limiter_y_%data(limiter_y_%x(i+1),j-1:j+1,k-1:k+1,1,1)/)))
                     END FORALL
                  END DO
               END IF

               IF (nDim >= 3) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1)=MIN(limiter_ppm_%data(limiter_ppm_%x(i),j,k,1,1), &
                             MINVAL((/limiter_z_%data(limiter_z_%x(i-1),j-1:j+1,k-1:k+1,1,1),limiter_z_%data(limiter_z_%x(i),j-1:j+1,k-1:k+1,1,1),limiter_z_%data(limiter_z_%x(i+1),j-1:j+1,k-1:k+1,1,1)/)))
                     END FORALL
                  END DO
               END IF
            END IF
         END IF
      END SUBROUTINE calc_limiters


      !> Calculate eigen system
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates speeds, left and right eigenvectors and number of waves present
      FUNCTION request_eigens(Info, index)!w,lambda,r,l,n)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mC(3,2)    ! Bounds of slab to update
         LOGICAL :: request_eigens
         LOGICAL :: lSpeedsX,lSpeedsY,lSpeedsZ,lnWaves,lLeftX,lRightX,lLeftY,lRightY,lLeftZ,lRightZ
         LOGICAL :: req_dim(3)
         REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:) :: prim
         REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:,:) :: lambda
         REAL(KIND=qPrec), ALLOCATABLE, DIMENSION(:,:,:)  :: r,l
         INTEGER, ALLOCATABLE, DIMENSION(:) :: n
         req_dim(nDim+1:3) = .false.
         req_dim(1:nDim) = .true.
         request_eigens=.true.
         lSpeedsX=istimeshift(index, Info%level, SpeedsX, mC)
         lSpeedsY=istimeshift(index, Info%level, SpeedsY, mC)
         lSpeedsZ=istimeshift(index, Info%level, SpeedsZ, mC)
         lLeftX=istimeshift(index, Info%level, leftX, mC)
         lRightX=istimeshift(index, Info%level, RightX, mC)
         lLeftY=istimeshift(index, Info%level, leftY, mC)
         lRightY=istimeshift(index, Info%level, RightY, mC)
         lLeftZ=istimeshift(index, Info%level, leftZ, mC)
         lRightZ=istimeshift(index, Info%level, RightZ, mC)
         lnWaves=istimeshift(index, Info%level, nWaves, mC)     
         IF (istime(index, Info%level, req_eigens, mB)) THEN
            ALLOCATE(n(nDim),prim(SweepCons))
            ALLOCATE (r(nDim,NrWaves,NrWaves),l(nDim,NrWaves,NrWaves),lambda(nDim,NrWaves))
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     prim=w_%data(w_%x(i),j,k,1,:)
                     CALL calc_eigens(request_eigens, prim, req_dim,lambda, n, l, r,index+i,j,k,Info%level)
                     IF (lSpeedsX) SpeedsX_%data(SpeedsX_%x(i),j,k,1,:)=lambda(1,:)       
                     IF (lSpeedsY) SpeedsY_%data(SpeedsY_%x(i),j,k,1,:)=lambda(2,:)       
                     IF (lSpeedsZ) SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,:)=lambda(3,:)
                     IF (lnWaves) nWaves_%data(nWaves_%x(i),j,k,1,:)=n(:)
                     IF (lleftX) leftX_%data(leftX_%x(i),j,k,:,:)=l(1,:,:)
                     IF (lrightX) rightX_%data(rightX_%x(i),j,k,:,:)=r(1,:,:)
                     IF (lleftY) leftY_%data(leftY_%x(i),j,k,:,:)=l(2,:,:)
                     IF (lrightY) rightY_%data(rightY_%x(i),j,k,:,:)=r(2,:,:)
                     IF (lleftZ) leftZ_%data(leftZ_%x(i),j,k,:,:)=l(3,:,:)
                     IF (lrightZ) rightZ_%data(rightZ_%x(i),j,k,:,:)=r(3,:,:)
                  END DO
               END DO
            END DO
            DEALLOCATE(n,prim, r, l, lambda)
         END IF



      END FUNCTION request_eigens


      !> Calculate MHD source terms
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Applies multi-dimensional source terms to predictor states
      SUBROUTINE MHD_Source_Terms(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (nDim == 2) THEN
            IF (istime(index, Info%level, qLx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qLx_%data(qLx_%x(i),j,k,1,iBy)=qLx_%data(qLx_%x(i),j,k,1,iBy)+w_%data(w_%x(i-1),j,k,1,ivy)*hdtdx * &
                          (Info%aux(index+i,j,k,1)-Info%aux(index+i-1,j,k,1))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qRx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qRx_%data(qRx_%x(i),j,k,1,iBy)=qRx_%data(qRx_%x(i),j,k,1,iBy)+w_%data(w_%x(i),j,k,1,ivy)*hdtdx * &
                          (Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qLy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qLy_%data(qLy_%x(i),j,k,1,iBx)=qLy_%data(qLy_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j-1,k,1,ivx)*hdtdy * &
                          (Info%aux(index+i,j,k,2)-Info%aux(index+i,j-1,k,2))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qRy, mB)) THEN 
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qRy_%data(qRy_%x(i),j,k,1,iBx)=qRy_%data(qRy_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j,k,1,ivx)*hdtdy * &
                          (Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2))
                  END FORALL
               END DO
            END IF
         ELSEIF (nDim == 3) THEN
            IF (istime(index, Info%level, qLx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qLx_%data(qLx_%x(i),j,k,1,iBz)=qLx_%data(qLx_%x(i),j,k,1,iBz)+w_%data(w_%x(i-1),j,k,1,ivz)*minmod(hdtdx * &
                          (Info%aux(index+i,j,k,1)-Info%aux(index+i-1,j,k,1)),hdtdz*(Info%aux(index+i-1,j,k,3) - &
                          Info%aux(index+i-1,j,k+1,3)))
                     qLx_%data(qLx_%x(i),j,k,1,iBy)=qLx_%data(qLx_%x(i),j,k,1,iBy)+w_%data(w_%x(i-1),j,k,1,ivy)*minmod(hdtdx * &
                          (Info%aux(index+i,j,k,1)-Info%aux(index+i-1,j,k,1)), hdtdy*(Info%aux(index+i-1,j,k,2) - &
                          Info%aux(index+i-1,j+1,k,2)))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qRx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qRx_%data(qRx_%x(i),j,k,1,iBz)=qRx_%data(qRx_%x(i),j,k,1,iBz)+w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdx * &
                          (Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1)),hdtdz*(Info%aux(index+i,j,k,3) - &
                          Info%aux(index+i,j,k+1,3)))
                     qRx_%data(qRx_%x(i),j,k,1,iBy)=qRx_%data(qRx_%x(i),j,k,1,iBy)+w_%data(w_%x(i),j,k,1,ivy)*minmod(hdtdx * &
                          (Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1)), hdtdy*(Info%aux(index+i,j,k,2) - &
                          Info%aux(index+i,j+1,k,2)))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qLy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qLy_%data(qLy_%x(i),j,k,1,iBz)=qLy_%data(qLy_%x(i),j,k,1,iBz)+w_%data(w_%x(i),j-1,k,1,ivz)*minmod(hdtdy * &
                          (Info%aux(index+i,j,k,2)-Info%aux(index+i,j-1,k,2)),hdtdz*(Info%aux(index+i,j-1,k,3) - &
                          Info%aux(index+i,j-1,k+1,3)))
                     qLy_%data(qLy_%x(i),j,k,1,iBx)=qLy_%data(qLy_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j-1,k,1,ivx)*minmod(hdtdy * &
                          (Info%aux(index+i,j,k,2)-Info%aux(index+i,j-1,k,2)),hdtdx*(Info%aux(index+i,j-1,k,1) - &
                          Info%aux(index+i+1,j-1,k,1)))                  
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qRy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qRy_%data(qRy_%x(i),j,k,1,iBz)=qRy_%data(qRy_%x(i),j,k,1,iBz)+w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdy * &
                          (Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2)),hdtdz*(Info%aux(index+i,j,k,3) - &
                          Info%aux(index+i,j,k+1,3)))
                     qRy_%data(qRy_%x(i),j,k,1,iBx)=qRy_%data(qRy_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j,k,1,ivx)*minmod(hdtdy * &
                          (Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2)),hdtdx*(Info%aux(index+i,j,k,1) - &
                          Info%aux(index+i+1,j,k,1)))                  
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qLz, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qLz_%data(qLz_%x(i),j,k,1,iBx)=qLz_%data(qLz_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j,k-1,1,ivx)*minmod(hdtdz * &
                          (Info%aux(index+i,j,k,3)-Info%aux(index+i,j,k-1,3)),hdtdx*(Info%aux(index+i,j,k-1,1) - &
                          Info%aux(index+i+1,j,k-1,1)))
                     qLz_%data(qLz_%x(i),j,k,1,iBy)=qLz_%data(qLz_%x(i),j,k,1,iBy)+w_%data(w_%x(i),j,k-1,1,ivy)*minmod(hdtdz * &
                          (Info%aux(index+i,j,k,3)-Info%aux(index+i,j,k-1,3)),hdtdy*(Info%aux(index+i,j,k-1,2) - &
                          Info%aux(index+i,j+1,k-1,2)))
                  END FORALL
               END DO
            END IF
            IF (istime(index, Info%level, qRz, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     qRz_%data(qRz_%x(i),j,k,1,iBx)=qRz_%data(qRz_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j,k,1,ivx)*minmod(hdtdz * &
                          (Info%aux(index+i,j,k+1,3)-Info%aux(index+i,j,k,3)),hdtdx*(Info%aux(index+i,j,k,1) - &
                          Info%aux(index+i+1,j,k,1)))
                     qRz_%data(qRz_%x(i),j,k,1,iBy)=qRz_%data(qRz_%x(i),j,k,1,iBy)+w_%data(w_%x(i),j,k,1,ivy)*minmod(hdtdz * &
                          (Info%aux(index+i,j,k+1,3)-Info%aux(index+i,j,k,3)),hdtdy*(Info%aux(index+i,j,k,2) - &
                          Info%aux(index+i,j+1,k,2)))
                  END FORALL
               END DO
            END IF
         END IF
      END SUBROUTINE MHD_Source_Terms


      !> Calculate predictor fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates fx from qLx, qRx, etc...
      SUBROUTINE calc_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF (istimeshift(index, Info%level, fx, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level),calc_flux_x(qLx_%data(qLx_%x(i),j,k,1,:), qRx_%data(qRx_%x(i),j,k,1,:), fx_%data(fx_%x(i),j,k,1,:)))
                  END DO
               END DO

               DO m=1,nFlux !FORALL(m=1:nFlux)
                  fx_%data(fx_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       fx_%data(fx_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*hdtdx
               END DO !FORALL

            END DO
         END IF

         IF (istimeshift(index, Info%level, fy, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_y(qLy_%data(qLy_%x(i),j,k,1,:), qRy_%data(qRy_%x(i),j,k,1,:), fy_%data(fy_%x(i),j,k,1,:)))
                  END DO
               END DO
               DO m=1,nFlux !FORALL(m=1:nFlux)
                  fy_%data(fy_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       fy_%data(fy_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*hdtdy
               END DO !FORALL
            END DO
         END IF


         IF (istimeshift(index, Info%level, fz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_z(qLz_%data(qLz_%x(i),j,k,1,:), qRz_%data(qRz_%x(i),j,k,1,:), fz_%data(fz_%x(i),j,k,1,:)))
                  END DO
               END DO
               DO m=1,nFlux 
                  fz_%data(fz_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       fz_%data(fz_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*hdtdz
               END DO
            END DO
         END IF
      END SUBROUTINE calc_fluxes


      !> Calculate final fluxes using 1D predictor states
      !! @param Info Info structure
      !! @param index Current sweep position
      !! Calculates f2x from qLx, qRx, etc...

      SUBROUTINE calc_fluxes_noctu(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF (istimeshift(index, Info%level, f2x, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_x(qLx_%data(qLx_%x(i),j,k,1,:), qRx_%data(qRx_%x(i),j,k,1,:), f2x_%data(f2x_%x(i),j,k,1,:)))
                  END DO
               END DO
               if (.false. .and. index+i == Info%mX(1)/4+1) THEN
!                  write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i-1)*dx, ' = ', qLx_%data(qLx_%x(i),1,1,1,1), 1d0+.1d0*sin(2d0*Pi*(real(index+i-1,8)*dx - hdt)),abs(qLx_%data(qLx_%x(i),1,1,1,1)- (1d0+.1d0*sin(2d0*Pi*(real(index+i-1,8)*dx-hdt))))
!                  write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i-1)*dx, ' = ', f2x_%data(f2x_%x(i),1,1,1,1), 1d0+.1d0*sin(2d0*Pi*(real(index+i-1,8)*dx - hdt)),abs(f2x_%data(f2x_%x(i),1,1,1,1)- (1d0+.1d0*sin(2d0*Pi*(real(index+i-1,8)*dx-hdt))))

                  write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i-1)*dx, ' = ', f2x_%data(f2x_%x(i),1,1,1,1), 1d0-.1d0/2d0/pi/dt*(cos(2d0*Pi*(real(index+i-1,8)*dx))-cos(2d0*Pi*(real(index+i-1,8)*dx-dt))),abs(f2x_%data(f2x_%x(i),1,1,1,1)- (1d0-.1d0/2d0/pi/dt*(cos(2d0*Pi*(real(index+i-1,8)*dx))-cos(2d0*Pi*(real(index+i-1,8)*dx-dt)))))

!                  write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i)*dx, ' = ', qLx_%data(qLx_%x(i),1,1,1,1), 1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx - hdt)),abs(qLx_%data(qLx_%x(i),1,1,1,1)- (1d0+.1*sin(2d0*Pi*((real(index+i+1)-half)*dx-hdt))))
!                  write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i)*dx, ' = ', qRx_%data(qRx_%x(i),1,1,1,1), 1d0+.1*sin(2d0*Pi*(real(index+i)-half)*dx - hdt),abs(qRx_%data(qRx_%x(i),1,1,1,1)- (1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx-hdt))))

!
!                        write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i)*dx, ' = ', qRx_%data(qRx_%x(i),1,1,1,1), 1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx - hdt)),abs(qRx_%data(qRx_%x(i),1,1,1,1)- (1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx-hdt))))

!                        write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i)*dx, ' = ', f2x_%data(f2x_%x(i),1,1,1,1), 1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx - hdt)),abs(f2x_%data(f2x_%x(i),1,1,1,1)- (1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx-hdt))))

!                        write(*,'(A,E25.16,A3,3E25.16)') 'qex at ', Info%xBounds(1,1)+real(index+i)*dx, ' = ', qRx_%data(qRx_%x(i),1,1,1,1), 1d0+.1*sin(2d0*Pi*(real(index+i)-half)*dx - hdt),abs(qRx_%data(qRx_%x(i),1,1,1,1)- (1d0+.1*sin(2d0*Pi*((real(index+i)-half)*dx-hdt))))
!                        STOP
!
                  STOP
               END if

               DO m=1,nFlux !FORALL(m=1:nFlux)
                  f2x_%data(f2x_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       f2x_%data(f2x_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*dtdx
               END DO !FORALL

            END DO
         END IF

         IF (istimeshift(index, Info%level, f2y, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_y(qLy_%data(qLy_%x(i),j,k,1,:), qRy_%data(qRy_%x(i),j,k,1,:), f2y_%data(f2y_%x(i),j,k,1,:))) 
                  END DO
               END DO
               DO m=1,nFlux !FORALL(m=1:nFlux)
                  f2y_%data(f2y_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       f2y_%data(f2y_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*dtdy
               END DO !FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, f2z, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1),mB(2,2)
                  DO k=mB(3,1),mB(3,2)
                     maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_z(qLz_%data(qLz_%x(i),j,k,1,:), qRz_%data(qRz_%x(i),j,k,1,:), f2z_%data(f2z_%x(i),j,k,1,:)))
                  END DO
               END DO
               DO m=1,nFlux !FORALL(m=1:nFlux)
                  f2z_%data(f2z_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m)) = &
                       f2z_%data(f2z_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepFluxFields(m))*dtdz
               END DO !FORALL

            END DO
         END IF
      END SUBROUTINE calc_fluxes_noctu


      !> Calculate predictor emf
      !! @param Info Info structure
      !! @param index Current sweep position
      !! calculates predictor emfs from predictor fluxes and cell centered emfs
      SUBROUTINE calc_emf(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (istimeshift(index, Info%level, ex_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ex_bar_%data(ex_bar_%x(i),j,k,1,1) = &
                       w_%data(w_%x(i),j,k,1,ivz)*w_%data(w_%x(i),j,k,1,iBy)-w_%data(w_%x(i),j,k,1,ivy)*w_%data(w_%x(i),j,k,1,iBz)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ey_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ey_bar_%data(ey_bar_%x(i),j,k,1,1) = &
                       w_%data(w_%x(i),j,k,1,ivx)*w_%data(w_%x(i),j,k,1,iBz)-w_%data(w_%x(i),j,k,1,ivz)*w_%data(w_%x(i),j,k,1,iBx)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ez_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ez_bar_%data(ez_bar_%x(i),j,k,1,1) = &
                       w_%data(w_%x(i),j,k,1,ivy)*w_%data(w_%x(i),j,k,1,iBx)-w_%data(w_%x(i),j,k,1,ivx)*w_%data(w_%x(i),j,k,1,iBy)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ex, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ex_%data(ex_%x(i),j,k,1,1)=upwinded_emf(fz_%data(fz_%x(i),j-1:j,k,1,iBy),-fy_%data(fy_%x(i),j,k-1:k,1,iBz),&
                       ex_bar_%data(ex_bar_%x(i),j-1:j,k-1:k,1,1), &
                       fy_%data(fy_%x(i),j,k-1:k,1,1),fz_%data(fz_%x(i),j-1:j,k,1,1))
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ey, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ey_%data(ey_%x(i),j,k,1,1) = &
                       upwinded_emf(fx_%data(fx_%x(i),j,k-1:k,1,iBz),(/-fz_%data(fz_%x(i-1),j,k,1,iBx),-fz_%data(fz_%x(i),j,k,1,iBx)/), &
                       reshape((/ey_bar_%data(ey_bar_%x(i-1),j,k-1:k,1,1),ey_bar_%data(ey_bar_%x(i),j,k-1:k,1,1)/),(/2,2/)), &
                       (/fz_%data(fz_%x(i-1),j,k,1,1),fz_%data(fz_%x(i),j,k,1,1)/),fx_%data(fx_%x(i),j,k-1:k,1,1))
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ez, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  ez_%data(ez_%x(i),j,k,1,1) = &
                       upwinded_emf((/fy_%data(fy_%x(i-1),j,k,1,iBx),fy_%data(fy_%x(i),j,k,1,iBx)/),-fx_%data(fx_%x(i),j-1:j,k,1,iBy), &
                       transpose(reshape((/ez_bar_%data(ez_bar_%x(i-1),j-1:j,k,1,1), ez_bar_%data(ez_bar_%x(i),j-1:j,k,1,1)/),(/2,2/))), &
                       fx_%data(fx_%x(i),j-1:j,k,1,1),(/fy_%data(fy_%x(i-1),j,k,1,1),fy_%data(fy_%x(i),j,k,1,1)/))
               END FORALL
            END DO
         END IF
      END SUBROUTINE calc_emf

      !> Update predictor aux fields
      !! @param Info Info structure
      !! @param index Current sweep position
      !! updates predictor face centered magnetc fields using predictor emfs
      SUBROUTINE updateB(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (istimeshift(index, Info%level, A2x, mB)) THEN
            A2x_%data(A2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                 Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1:1) &
                 - hdtdy*(ez_%data(ez_%x(mB(1,1):mB(1,2)),mB(2,1)+1:mB(2,2)+1, mB(3,1):mB(3,2),1,1:1) &
                 -ez_%data(ez_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1))
            IF (nDim >= 3) THEN
               A2x_%data(A2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    A2x_%data(A2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) &
                    + hdtdz*(ey_%data(ey_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1)+1:mB(3,2)+1,1,1:1) &
                    -ey_%data(ey_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1))
            END IF
         END IF
         IF (istimeshift(index, Info%level, A2y, mB)) THEN
            A2y_%data(A2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                 Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),2:2) &
                 + hdtdx*(ez_%data(ez_%x(mB(1,1)+1:mB(1,2)+1),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1) &
                 -ez_%data(ez_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1))
            IF (nDim >= 3) THEN
               A2y_%data(A2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                    A2y_%data(A2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) &
                    - hdtdz*(ex_%data(ex_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1)+1:mB(3,2)+1,1,1:1) &
                    -ex_%data(ex_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1))
            END IF
         END IF
         IF (istimeshift(index, Info%level, A2z, mB)) THEN
            A2z_%data(A2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:) = &
                 Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),3:3) &
                 + hdtdy*(ex_%data(ex_%x(mB(1,1):mB(1,2)),mB(2,1)+1:mB(2,2)+1, mB(3,1):mB(3,2),1,1:1) &
                 -ex_%data(ex_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1)) &
                 - hdtdx*(ey_%data(ey_%x(mB(1,1)+1:mB(1,2)+1),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1) &
                 -ey_%data(ey_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,1:1))
         END IF

      END SUBROUTINE updateB

      !> Update time centered fluid fields
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE updatew2(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         REAL(KIND=qPREC), DIMENSION(:), POINTER :: q

         IF (istimeshift(index, Info%level, w2, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  w2_%data(w2_%x(i),j,k,1,SweepFluxFields(m))=beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,SweepFluxFields(m))+(fx_%data(fx_%x(i),j,k,1,SweepFluxFields(m)) - &
                       fx_%data(fx_%x(i+1),j,k,1,SweepFluxFields(m)))
               END FORALL
               IF (nDim >= 2) THEN
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     w2_%data(w2_%x(i),j,k,1,SweepFluxFields(m))=w2_%data(w2_%x(i),j,k,1,SweepFluxFields(m))+(fy_%data(fy_%x(i),j,k,1,SweepFluxFields(m)) - &
                          fy_%data(fy_%x(i),j+1,k,1,SweepFluxFields(m)))
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                        w2_%data(w2_%x(i),j,k,1,SweepFluxFields(m))=w2_%data(w2_%x(i),j,k,1,SweepFluxFields(m))+(fz_%data(fz_%x(i),j,k,1,SweepFluxFields(m)) - &
                             fz_%data(fz_%x(i),j,k+1,1,SweepFluxFields(m)))
                     END FORALL
                  END IF
               END IF
            END DO
            IF (lMHD) THEN
               IF (nDim >= 2) THEN
                  w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBx) = &
                       half*(A2x_%data(A2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) + &
                       A2x_%data(A2x_%x(mB(1,1)+1:mB(1,2)+1),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
                  w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBy) = &
                       half*(A2y_%data(A2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) + &
                       A2y_%data(A2y_%x(mB(1,1):mB(1,2)),mB(2,1)+1:mB(2,2)+1,mB(3,1):mB(3,2),1,1))
                  IF (nDim >= 3) THEN
                     w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iBz) = &
                          half*(A2z_%data(A2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) + &
                          A2z_%data(A2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1)+1:mB(3,2)+1,1,1))
                  END IF
               END IF
            END IF
            IF (lSelfGravity) THEN
               w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) = &
                    w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivx) - qdtdx*&
                    Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),irho) * &
                    (Info%q(index+mB(1,1)+1:index+mB(1,2)+1,mB(2,1):mB(2,2), mB(3,1):mB(3,2), iPhiGas) - &
                    Info%q(index+mB(1,1)-1:index+mB(1,2)-1,mB(2,1):mB(2,2), mB(3,1):mB(3,2), iPhiGas))

               w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) = &
                    w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) - qdtdx*&
                    Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),ivx) * &
                    (Info%q(index+mB(1,1)+1:index+mB(1,2)+1,mB(2,1):mB(2,2), mB(3,1):mB(3,2), iPhiGas) - &
                    Info%q(index+mB(1,1)-1:index+mB(1,2)-1,mB(2,1):mB(2,2), mB(3,1):mB(3,2), iPhiGas))

               IF (nDim >= 2) THEN
                  w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) = &
                       w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivy) - qdtdy*&
                       Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),irho) * &
                       (Info%q(index+mB(1,1):index+mB(1,2),mB(2,1)+1:mB(2,2)+1, mB(3,1):mB(3,2), iPhiGas) - &
                       Info%q(index+mB(1,1):index+mB(1,2),mB(2,1)-1:mB(2,2)-1, mB(3,1):mB(3,2), iPhiGas))

                  w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) = &
                       w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) - qdtdy*&
                       Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),ivy) * &
                       (Info%q(index+mB(1,1):index+mB(1,2),mB(2,1)+1:mB(2,2)+1, mB(3,1):mB(3,2), iPhiGas) - &
                       Info%q(index+mB(1,1):index+mB(1,2),mB(2,1)-1:mB(2,2)-1, mB(3,1):mB(3,2), iPhiGas))

                  IF (nDim >= 3) THEN
                     w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) = &
                          w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,ivz) - qdtdz*&
                          Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),irho) * &
                          (Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1)+1:mB(3,2)+1, iPhiGas) - &
                          Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1)-1:mB(3,2)-1, iPhiGas))

                     w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) = &
                          w2_%data(w2_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,iE) - qdtdz*&
                          Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1):mB(3,2),ivz) * &
                          (Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1)+1:mB(3,2)+1, iPhiGas) - &
                          Info%q(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2), mB(3,1)-1:mB(3,2)-1, iPhiGas))

                  END IF
               END IF
            END IF
            DO i=mB(1,1),mB(1,2)
               CALL cons_to_prim_1(w2_%data(w2_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:))
            END DO
         END IF
      END SUBROUTINE updatew2


      !> Update mhd fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      !! updates transverse magnetic flux terms using predictor emf's
      SUBROUTINE update_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (nDim >= 3) THEN
            IF (istimeshift(index, Info%level, ctfx, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     ctfx_%data(ctfx_%x(i),j,k,1,1)=-qdtdx*(ez_%data(ez_%x(i),j,k,1,1)+ez_%data(ez_%x(i),j+1,k,1,1))
                     ctfx_%data(ctfx_%x(i),j,k,1,2)=qdtdx*(ey_%data(ey_%x(i),j,k,1,1)+ey_%data(ey_%x(i),j,k+1,1,1))
                  END FORALL
               END DO
            END IF
            IF (istimeshift(index, Info%level, ctfy, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     ctfy_%data(ctfy_%x(i),j,k,1,1)=-qdtdy*(ex_%data(ex_%x(i),j,k,1,1)+ex_%data(ex_%x(i),j,k+1,1,1))
                     ctfy_%data(ctfy_%x(i),j,k,1,2)=qdtdy*(ez_%data(ez_%x(i),j,k,1,1)+ez_%data(ez_%x(i+1),j,k,1,1))
                  END FORALL
               END DO
            END IF
            IF (istimeshift(index, Info%level, ctfz, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     ctfz_%data(ctfz_%x(i),j,k,1,1)=-qdtdz*(ey_%data(ey_%x(i),j,k,1,1)+ey_%data(ey_%x(i+1),j,k,1,1))
                     ctfz_%data(ctfz_%x(i),j,k,1,2)=qdtdz*(ex_%data(ex_%x(i),j,k,1,1)+ex_%data(ex_%x(i),j+1,k,1,1))
                  END FORALL
               END DO
            END IF
         END IF
      END SUBROUTINE update_fluxes


      !> Update interface states with transverse predictor fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE CTU(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF(istimeshift(index, Info%level, Sx, mB)) THEN

            IF (nDim == 2) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     Sx_%data(Sx_%x(i),j,k,1,m_low:m_high)=hdtdx*w_%data(w_%x(i),j,k,1,iBx:iBz)*(Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1))
                     Sx_%data(Sx_%x(i),j,k,1,iBz)=w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdy*(Info%aux(index+i,j,k,2)-Info%aux(index+i,j+1,k,2)), &
                          hdtdx*(Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1)))
                  END FORALL
                  IF (iE .ne. 0) FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2)) &
                       Sx_%data(Sx_%x(i),j,k,1,iE) = w_%data(w_%x(i),j,k,1,iBz)*Sx_%data(Sx_%x(i),j,k,1,iBz)
               END DO
            ELSE
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     Sx_%data(Sx_%x(i),j,k,1,m_low:m_high)=hdtdx*w_%data(w_%x(i),j,k,1,iBx:iBz)*(Info%aux(index+i+1,j,k,1) - &
                          Info%aux(index+i,j,k,1))
                     Sx_%data(Sx_%x(i),j,k,1,iBy)=w_%data(w_%x(i),j,k,1,ivy)*minmod(hdtdz*(Info%aux(index+i,j,k,3)-Info%aux(index+i,j,k+1,3)), &
                          hdtdx*(Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1)))
                     Sx_%data(Sx_%x(i),j,k,1,iBz)=w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdy*(Info%aux(index+i,j,k,2)-Info%aux(index+i,j+1,k,2)), &
                          hdtdx*(Info%aux(index+i+1,j,k,1)-Info%aux(index+i,j,k,1)))
                  END FORALL
                  IF (iE .ne. 0) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))  
                        Sx_%data(Sx_%x(i),j,k,1,iE)=w_%data(w_%x(i),j,k,1,iBy)*Sx_%data(Sx_%x(i),j,k,1,iBy)+w_%data(w_%x(i),j,k,1,iBz)*Sx_%data(Sx_%x(i),j,k,1,iBz)
                     END FORALL
                  END IF
               END DO
            END IF
         END IF
         IF(istimeshift(index, Info%level, Sy, mB)) THEN
            IF (nDim == 2) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     Sy_%data(Sy_%x(i),j,k,1,m_low:m_high)=hdtdy*w_%data(w_%x(i),j,k,1,iBx:iBz)*(Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2))
                     Sy_%data(Sy_%x(i),j,k,1,iBz)=w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdx*(Info%aux(index+i,j,k,1)-Info%aux(index+i+1,j,k,1)), &
                          hdtdy*(Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2)))
                  END FORALL
                  IF (iE .ne. 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        Sy_%data(Sy_%x(i),j,k,1,iE)=w_%data(w_%x(i),j,k,1,iBz)*Sy_%data(Sy_%x(i),j,k,1,iBz)
                     END FORALL
                  END IF
               END DO
            ELSE
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     Sy_%data(Sy_%x(i),j,k,1,m_low:m_high) = hdtdy*w_%data(w_%x(i),j,k,1,iBx:iBz)*(Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2))
                     Sy_%data(Sy_%x(i),j,k,1,iBz)=w_%data(w_%x(i),j,k,1,ivz)*minmod(hdtdx*(Info%aux(index+i,j,k,1)-Info%aux(index+i+1,j,k,1)), &
                          hdtdy*(Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2)))
                     Sy_%data(Sy_%x(i),j,k,1,iBx)=w_%data(w_%x(i),j,k,1,ivx)*minmod(hdtdz*(Info%aux(index+i,j,k,3)-Info%aux(index+i,j,k+1,3)), &
                          hdtdy*(Info%aux(index+i,j+1,k,2)-Info%aux(index+i,j,k,2)))
                  END FORALL
                  IF (iE .ne. 0) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))              
                        Sy_%data(Sy_%x(i),j,k,1,iE)=w_%data(w_%x(i),j,k,1,iBz)*Sy_%data(Sy_%x(i),j,k,1,iBz)+w_%data(w_%x(i),j,k,1,iBx)*Sy_%data(Sy_%x(i),j,k,1,iBx)
                     END FORALL
                  END IF
               END DO

            END IF
         END IF
         IF(istimeshift(index, Info%level, Sz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                  Sz_%data(Sz_%x(i),j,k,1,m_low:m_high) = hdtdz*w_%data(w_%x(i),j,k,1,iBx:iBz)*(Info%aux(index+i,j,k+1,3)-Info%aux(index+i,j,k,3))
                  Sz_%data(Sz_%x(i),j,k,1,iBx)=w_%data(w_%x(i),j,k,1,ivx)*minmod(hdtdy*(Info%aux(index+i,j,k,2)-Info%aux(index+i,j+1,k,2)), &
                       hdtdz*(Info%aux(index+i,j,k+1,3)-Info%aux(index+i,j,k,3)))
                  Sz_%data(Sz_%x(i),j,k,1,iBy)=w_%data(w_%x(i),j,k,1,ivy)*minmod(hdtdx*(Info%aux(index+i,j,k,1)-Info%aux(index+i+1,j,k,1)), &
                       hdtdz*(Info%aux(index+i,j,k+1,3)-Info%aux(index+i,j,k,3)))
               END FORALL
               IF (iE .ne. 0) THEN
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     Sz_%data(Sz_%x(i),j,k,1,iE)=w_%data(w_%x(i),j,k,1,iBx)*Sz_%data(Sz_%x(i),j,k,1,iBx)+w_%data(w_%x(i),j,k,1,iBy)*Sz_%data(Sz_%x(i),j,k,1,iBy)
                  END FORALL
               END IF
            END DO
         END IF


         IF(istimeshift(index, Info%level, q2Lx, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qLx_%data(qLx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Lx_%data(q2Lx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Lx_%data(q2Lx_%x(i),j,k,1,SweepFluxFields(m))=q2Lx_%data(q2Lx_%x(i),j,k,1,SweepFluxFields(m))+(fy_%data(fy_%x(i-1),j,k,1,SweepFluxFields(m)) - &
                       fy_%data(fy_%x(i-1),j+1,k,1,SweepFluxFields(m)))
               END FORALL
               IF (nDim >= 3) THEN
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     q2Lx_%data(q2Lx_%x(i),j,k,1,SweepFluxFields(m))=q2Lx_%data(q2Lx_%x(i),j,k,1,SweepFluxFields(m))+(fz_%data(fz_%x(i-1),j,k,1,SweepFluxFields(m)) - &
                          fz_%data(fz_%x(i-1),j,k+1,1,SweepFluxFields(m)))
                  END FORALL
               END IF
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Lx_%data(q2Lx_%x(i),j,k,1,m)=q2Lx_%data(q2Lx_%x(i),j,k,1,m)+Sx_%data(Sx_%x(i-1),j,k,1,m)
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     q2Lx_%data(q2Lx_%x(i),j,k,1,iBx) = A2x_%data(A2x_%x(i),j,k,1,1)
                  END FORALL

                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        q2Lx_%data(q2Lx_%x(i),j,k,1,iBy) = q2Lx_%data(q2Lx_%x(i),j,k,1,iBy)+(ctfz_%data(ctfz_%x(i-1),j,k,1,2)-ctfz_%data(ctfz_%x(i-1),j,k+1,1,2))          
                        q2Lx_%data(q2Lx_%x(i),j,k,1,iBz) = q2Lx_%data(q2Lx_%x(i),j,k,1,iBz)+(ctfy_%data(ctfy_%x(i-1),j,k,1,1)-ctfy_%data(ctfy_%x(i-1),j+1,k,1,1))               
                     END FORALL
                  END IF
               END IF
               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                     q2Lx_%data(q2Lx_%x(i),j,k,1,ivy)=q2Lx_%data(q2Lx_%x(i),j,k,1,ivy)-qdt*Info%q(index+i-1,j,k,1)*sum(gradphiy_%data(gradphiy_%x(i-1),j:j+1,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        q2Lx_%data(q2Lx_%x(i),j,k,1,iE)=q2Lx_%data(q2Lx_%x(i),j,k,1,iE)-hdx*(fy_%data(fy_%x(i-1),j,k,1,1)*gradphiy_%data(gradphiy_%x(i-1),j,k,1,1)+fy_%data(fy_%x(i-1),j+1,k,1,1)*gradphiy_%data(gradphiy_%x(i-1),j+1,k,1,1))
                     END FORALL
                  END IF
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                        q2Lx_%data(q2Lx_%x(i),j,k,1,ivz)=q2Lx_%data(q2Lx_%x(i),j,k,1,ivz)-qdt*Info%q(index+i-1,j,k,1)*sum(gradphiz_%data(gradphiz_%x(i-1),j,k:k+1,1,1))
                     END FORALL
                     IF (iE /= 0) THEN
                        FORALL(j=mB(2,1):mB(2,2), k=mB(3,1):mB(3,2))
                           q2Lx_%data(q2Lx_%x(i),j,k,1,iE)=q2Lx_%data(q2Lx_%x(i),j,k,1,iE)-hdx*(fz_%data(fz_%x(i-1),j,k,1,1)*gradphiz_%data(gradphiz_%x(i-1),j,k,1,1)+fz_%data(fz_%x(i-1),j,k+1,1,1)*gradphiz_%data(gradphiz_%x(i-1),j,k+1,1,1))
                        END FORALL
                     END IF
                  END IF
               END IF
               CALL cons_to_prim_1(q2Lx_%data(q2Lx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Lx_%data(q2Lx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO

         END IF
         IF(istimeshift(index, Info%level, q2Rx, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qRx_%data(qRx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Rx_%data(q2Rx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Rx_%data(q2Rx_%x(i),j,k,1,SweepFluxFields(m))=q2Rx_%data(q2Rx_%x(i),j,k,1,SweepFluxFields(m))+(fy_%data(fy_%x(i),j,k,1,SweepFluxFields(m)) - &
                       fy_%data(fy_%x(i),j+1,k,1,SweepFluxFields(m)))
               END FORALL

               IF (nDim >= 3) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     q2Rx_%data(q2Rx_%x(i),j,k,1,SweepFluxFields(m))=q2Rx_%data(q2Rx_%x(i),j,k,1,SweepFluxFields(m))+(fz_%data(fz_%x(i),j,k,1,SweepFluxFields(m)) - &
                          fz_%data(fz_%x(i),j,k+1,1,SweepFluxFields(m)))
                  END FORALL
               END IF
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Rx_%data(q2Rx_%x(i),j,k,1,m)=q2Rx_%data(q2Rx_%x(i),j,k,1,m)+Sx_%data(Sx_%x(i),j,k,1,m)
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Rx_%data(q2Rx_%x(i),j,k,1,iBx)=A2x_%data(A2x_%x(i),j,k,1,1)
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Rx_%data(q2Rx_%x(i),j,k,1,iBz)=q2Rx_%data(q2Rx_%x(i),j,k,1,iBz)+(ctfy_%data(ctfy_%x(i),j,k,1,1)-ctfy_%data(ctfy_%x(i),j+1,k,1,1))
                        q2Rx_%data(q2Rx_%x(i),j,k,1,iBy)=q2Rx_%data(q2Rx_%x(i),j,k,1,iBy)+(ctfz_%data(ctfz_%x(i),j,k,1,2)-ctfz_%data(ctfz_%x(i),j,k+1,1,2))          
                     END FORALL
                  END IF
               END IF

               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Rx_%data(q2Rx_%x(i),j,k,1,ivy)=q2Rx_%data(q2Rx_%x(i),j,k,1,ivy)-qdt*Info%q(index+i,j,k,1)*sum(gradphiy_%data(gradphiy_%x(i),j:j+1,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Rx_%data(q2Rx_%x(i),j,k,1,iE)=q2Rx_%data(q2Rx_%x(i),j,k,1,iE)-hdx*(fy_%data(fy_%x(i),j,k,1,1)*gradphiy_%data(gradphiy_%x(i),j,k,1,1)+fy_%data(fy_%x(i),j+1,k,1,1)*gradphiy_%data(gradphiy_%x(i),j+1,k,1,1))
                     END FORALL
                  END IF
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Rx_%data(q2Rx_%x(i),j,k,1,ivz)=q2Rx_%data(q2Rx_%x(i),j,k,1,ivz)-qdt*Info%q(index+i,j,k,1)*sum(gradphiz_%data(gradphiz_%x(i),j,k:k+1,1,1))
                     END FORALL
                     IF (iE /= 0) THEN
                        FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                           q2Rx_%data(q2Rx_%x(i),j,k,1,iE)=q2Rx_%data(q2Rx_%x(i),j,k,1,iE)-hdx*(fz_%data(fz_%x(i),j,k,1,1)*gradphiz_%data(gradphiz_%x(i),j,k,1,1)+fz_%data(fz_%x(i),j,k+1,1,1)*gradphiz_%data(gradphiz_%x(i),j,k+1,1,1))
                        END FORALL
                     END IF
                  END IF
               END IF

               CALL cons_to_prim_1(q2Rx_%data(q2Rx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Rx_%data(q2Rx_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO
         END IF

         IF(istimeshift(index, Info%level, q2Ly, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qLy_%data(qLy_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Ly_%data(q2Ly_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))

               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Ly_%data(q2Ly_%x(i),j,k,1,SweepFluxFields(m))=q2Ly_%data(q2Ly_%x(i),j,k,1,SweepFluxFields(m))+(fx_%data(fx_%x(i),j-1,k,1,SweepFluxFields(m)) - &
                       fx_%data(fx_%x(i+1),j-1,k,1,SweepFluxFields(m)))
               END FORALL
               IF (nDim >= 3) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     q2Ly_%data(q2Ly_%x(i),j,k,1,SweepFluxFields(m))=q2Ly_%data(q2Ly_%x(i),j,k,1,SweepFluxFields(m))+(fz_%data(fz_%x(i),j-1,k,1,SweepFluxFields(m)) - &
                          fz_%data(fz_%x(i),j-1,k+1,1,SweepFluxFields(m)))
                  END FORALL
               END IF
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Ly_%data(q2Ly_%x(i),j,k,1,m)=q2Ly_%data(q2Ly_%x(i),j,k,1,m)+Sy_%data(Sy_%x(i),j-1,k,1,m)
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Ly_%data(q2Ly_%x(i),j,k,1,iBy)=A2y_%data(A2y_%x(i),j,k,1,1)
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ly_%data(q2Ly_%x(i),j,k,1,iBz)=q2Ly_%data(q2Ly_%x(i),j,k,1,iBz)+(ctfx_%data(ctfx_%x(i),j-1,k,1,2)-ctfx_%data(ctfx_%x(i+1),j-1,k,1,2))
                        q2Ly_%data(q2Ly_%x(i),j,k,1,iBx)=q2Ly_%data(q2Ly_%x(i),j,k,1,iBx)+(ctfz_%data(ctfz_%x(i),j-1,k,1,1)-ctfz_%data(ctfz_%x(i),j-1,k+1,1,1))
                     END FORALL
                  END IF
               END IF


               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Ly_%data(q2Ly_%x(i),j,k,1,ivx)=q2Ly_%data(q2Ly_%x(i),j,k,1,ivx)-qdt*Info%q(index+i,j-1,k,1)*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j-1,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ly_%data(q2Ly_%x(i),j,k,1,iE)=q2Ly_%data(q2Ly_%x(i),j,k,1,iE)-hdx*(fx_%data(fx_%x(i),j-1,k,1,1)*gradphix_%data(gradphix_%x(i),j-1,k,1,1)+fx_%data(fx_%x(i+1),j-1,k,1,1)*gradphix_%data(gradphix_%x(i+1),j-1,k,1,1))
                     END FORALL
                  END IF
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ly_%data(q2Ly_%x(i),j,k,1,ivz)=q2Ly_%data(q2Ly_%x(i),j,k,1,ivz)-qdt*Info%q(index+i,j-1,k,1)*sum(gradphiz_%data(gradphiz_%x(i),j-1,k:k+1,1,1))
                     END FORALL
                     IF (iE /= 0) THEN
                        FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                           q2Ly_%data(q2Ly_%x(i),j,k,1,iE)=q2Ly_%data(q2Ly_%x(i),j,k,1,iE)-hdx*(fz_%data(fz_%x(i),j-1,k,1,1)*gradphiz_%data(gradphiz_%x(i),j-1,k,1,1)+fz_%data(fz_%x(i),j-1,k+1,1,1)*gradphiz_%data(gradphiz_%x(i),j-1,k+1,1,1))
                        END FORALL
                     END IF
                  END IF
               END IF

               CALL cons_to_prim_1(q2Ly_%data(q2Ly_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Ly_%data(q2Ly_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO
         END IF
         IF(istimeshift(index, Info%level, q2Ry, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qRy_%data(qRy_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Ry_%data(q2Ry_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))

               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Ry_%data(q2Ry_%x(i),j,k,1,SweepFluxFields(m))=q2Ry_%data(q2Ry_%x(i),j,k,1,SweepFluxFields(m))+(fx_%data(fx_%x(i),j,k,1,SweepFluxFields(m)) - &
                       fx_%data(fx_%x(i+1),j,k,1,SweepFluxFields(m)))
               END FORALL
               IF (nDim >= 3) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     q2Ry_%data(q2Ry_%x(i),j,k,1,SweepFluxFields(m))=q2Ry_%data(q2Ry_%x(i),j,k,1,SweepFluxFields(m))+(fz_%data(fz_%x(i),j,k,1,SweepFluxFields(m)) - &
                          fz_%data(fz_%x(i),j,k+1,1,SweepFluxFields(m)))
                  END FORALL
               END IF
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Ry_%data(q2Ry_%x(i),j,k,1,m)=q2Ry_%data(q2Ry_%x(i),j,k,1,m)+Sy_%data(Sy_%x(i),j,k,1,m)
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Ry_%data(q2Ry_%x(i),j,k,1,iBy)=A2y_%data(A2y_%x(i),j,k,1,1)
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ry_%data(q2Ry_%x(i),j,k,1,iBz)=q2Ry_%data(q2Ry_%x(i),j,k,1,iBz)+(ctfx_%data(ctfx_%x(i),j,k,1,2)-ctfx_%data(ctfx_%x(i+1),j,k,1,2))
                        q2Ry_%data(q2Ry_%x(i),j,k,1,iBx)=q2Ry_%data(q2Ry_%x(i),j,k,1,iBx)+(ctfz_%data(ctfz_%x(i),j,k,1,1)-ctfz_%data(ctfz_%x(i),j,k+1,1,1))
                     END FORALL
                  END IF
               END IF


               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Ry_%data(q2Ry_%x(i),j,k,1,ivx)=q2Ry_%data(q2Ry_%x(i),j,k,1,ivx)-qdt*Info%q(index+i,j,k,1)*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ry_%data(q2Ry_%x(i),j,k,1,iE)=q2Ry_%data(q2Ry_%x(i),j,k,1,iE)-hdx*(fx_%data(fx_%x(i),j,k,1,1)*gradphix_%data(gradphix_%x(i),j,k,1,1)+fx_%data(fx_%x(i+1),j,k,1,1)*gradphix_%data(gradphix_%x(i+1),j,k,1,1))
                     END FORALL
                  END IF
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Ry_%data(q2Ry_%x(i),j,k,1,ivz)=q2Ry_%data(q2Ry_%x(i),j,k,1,ivz)-qdt*Info%q(index+i,j,k,1)*sum(gradphiz_%data(gradphiz_%x(i),j,k:k+1,1,1))
                     END FORALL
                     IF (iE /= 0) THEN
                        FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                           q2Ry_%data(q2Ry_%x(i),j,k,1,iE)=q2Ry_%data(q2Ry_%x(i),j,k,1,iE)-hdx*(fz_%data(fz_%x(i),j,k,1,1)*gradphiz_%data(gradphiz_%x(i),j,k,1,1)+fz_%data(fz_%x(i),j,k+1,1,1)*gradphiz_%data(gradphiz_%x(i),j,k+1,1,1))
                        END FORALL
                     END IF
                  END IF
               END IF



               CALL cons_to_prim_1(q2Ry_%data(q2Ry_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Ry_%data(q2Ry_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO
         END IF

         IF(istimeshift(index, Info%level, q2Lz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qLz_%data(qLz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Lz_%data(q2Lz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Lz_%data(q2Lz_%x(i),j,k,1,SweepFluxFields(m))=q2Lz_%data(q2Lz_%x(i),j,k,1,SweepFluxFields(m))+(fx_%data(fx_%x(i),j,k-1,1,SweepFluxFields(m)) - &
                       fx_%data(fx_%x(i+1),j,k-1,1,SweepFluxFields(m))) &
                       +(fy_%data(fy_%x(i),j,k-1,1,SweepFluxFields(m))-fy_%data(fy_%x(i),j+1,k-1,1,SweepFluxFields(m)))
               END FORALL
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Lz_%data(q2Lz_%x(i),j,k,1,m)=q2Lz_%data(q2Lz_%x(i),j,k,1,m)+Sz_%data(Sz_%x(i),j,k-1,1,m)
                  END FORALL

                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Lz_%data(q2Lz_%x(i),j,k,1,iBz)=A2z_%data(A2z_%x(i),j,k,1,1)
                     q2Lz_%data(q2Lz_%x(i),j,k,1,iBy)=q2Lz_%data(q2Lz_%x(i),j,k,1,iBy)+(ctfx_%data(ctfx_%x(i),j,k-1,1,1)-ctfx_%data(ctfx_%x(i+1),j,k-1,1,1))
                     q2Lz_%data(q2Lz_%x(i),j,k,1,iBx)=q2Lz_%data(q2Lz_%x(i),j,k,1,iBx)+(ctfy_%data(ctfy_%x(i),j,k-1,1,2)-ctfy_%data(ctfy_%x(i),j+1,k-1,1,2))
                  END FORALL
               END IF

               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Lz_%data(q2Lz_%x(i),j,k,1,ivx)=q2Lz_%data(q2Lz_%x(i),j,k,1,ivx)-qdt*Info%q(index+i,j,k-1,1)*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j,k-1,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Lz_%data(q2Lz_%x(i),j,k,1,iE)=q2Lz_%data(q2Lz_%x(i),j,k,1,iE)-hdx*(fx_%data(fx_%x(i),j,k-1,1,1)*gradphix_%data(gradphix_%x(i),j,k-1,1,1)+fx_%data(fx_%x(i+1),j,k-1,1,1)*gradphix_%data(gradphix_%x(i+1),j,k-1,1,1))
                     END FORALL
                  END IF

                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Lz_%data(q2Lz_%x(i),j,k,1,ivy)=q2Lz_%data(q2Lz_%x(i),j,k,1,ivy)-qdt*Info%q(index+i,j,k-1,1)*sum(gradphiy_%data(gradphiy_%x(i),j:j+1,k-1,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Lz_%data(q2Lz_%x(i),j,k,1,iE)=q2Lz_%data(q2Lz_%x(i),j,k,1,iE)-hdx*(fy_%data(fy_%x(i),j,k-1,1,1)*gradphiy_%data(gradphiy_%x(i),j,k-1,1,1)+fy_%data(fy_%x(i),j+1,k-1,1,1)*gradphiy_%data(gradphiy_%x(i),j+1,k-1,1,1))
                     END FORALL
                  END IF
               END IF

               CALL cons_to_prim_1(q2Lz_%data(q2Lz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Lz_%data(q2Lz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO
         END IF
         IF(istimeshift(index, Info%level, q2Rz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               CALL prim_to_cons_2(qRz_%data(qRz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:), &
                    q2Rz_%data(q2Rz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  q2Rz_%data(q2Rz_%x(i),j,k,1,SweepFluxFields(m))=q2Rz_%data(q2Rz_%x(i),j,k,1,SweepFluxFields(m))+&
                       (fx_%data(fx_%x(i),j,k,1,SweepFluxFields(m))-fx_%data(fx_%x(i+1),j,k,1,SweepFluxFields(m)))+ &
                       (fy_%data(fy_%x(i),j,k,1,SweepFluxFields(m))-fy_%data(fy_%x(i),j+1,k,1,SweepFluxFields(m)))
               END FORALL
               IF (lMHD) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:SweepCons)
                     q2Rz_%data(q2Rz_%x(i),j,k,1,m)=q2Rz_%data(q2Rz_%x(i),j,k,1,m)+Sz_%data(Sz_%x(i),j,k,1,m)
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Rz_%data(q2Rz_%x(i),j,k,1,iBz)=A2z_%data(A2z_%x(i),j,k,1,1)
                     q2Rz_%data(q2Rz_%x(i),j,k,1,iBy)=q2Rz_%data(q2Rz_%x(i),j,k,1,iBy)+(ctfx_%data(ctfx_%x(i),j,k,1,1)-ctfx_%data(ctfx_%x(i+1),j,k,1,1))
                     q2Rz_%data(q2Rz_%x(i),j,k,1,iBx)=q2Rz_%data(q2Rz_%x(i),j,k,1,iBx)+(ctfy_%data(ctfy_%x(i),j,k,1,2)-ctfy_%data(ctfy_%x(i),j+1,k,1,2))
                  END FORALL
               END IF
               IF (lSelfGravity) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Rz_%data(q2Rz_%x(i),j,k,1,ivx)=q2Rz_%data(q2Rz_%x(i),j,k,1,ivx)-qdt*Info%q(index+i,j,k,1)*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Rz_%data(q2Rz_%x(i),j,k,1,iE)=q2Rz_%data(q2Rz_%x(i),j,k,1,iE)-hdx*(fx_%data(fx_%x(i),j,k,1,1)*gradphix_%data(gradphix_%x(i),j,k,1,1)+fx_%data(fx_%x(i+1),j,k,1,1)*gradphix_%data(gradphix_%x(i+1),j,k,1,1))
                     END FORALL
                  END IF

                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     q2Rz_%data(q2Rz_%x(i),j,k,1,ivy)=q2Rz_%data(q2Rz_%x(i),j,k,1,ivy)-qdt*Info%q(index+i,j,k,1)*sum(gradphiy_%data(gradphiy_%x(i),j:j+1,k,1,1))
                  END FORALL
                  IF (iE /= 0) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        q2Rz_%data(q2Rz_%x(i),j,k,1,iE)=q2Rz_%data(q2Rz_%x(i),j,k,1,iE)-hdx*(fy_%data(fy_%x(i),j,k,1,1)*gradphiy_%data(gradphiy_%x(i),j,k,1,1)+fy_%data(fy_%x(i),j+1,k,1,1)*gradphiy_%data(gradphiy_%x(i),j+1,k,1,1))
                     END FORALL
                  END IF
               END IF
               CALL cons_to_prim_1(q2Rz_%data(q2Rz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
               IF (lCautious) CALL protect_all(q2Rz_%data(q2Rz_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,:))
            END DO
         END IF
      END SUBROUTINE CTU

      !> Calculate tracer fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE calc_tracer_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         REAL(Kind=qPrec) :: dq,sdq,dfmin,u_edge
         INTEGER :: m
         IF (istimeshift(index, Info%level, adfx, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLo, nTracerHi
                        dq=half*(beforesweepstep_%data(beforesweepstep_%x(i+1),j,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i-1),j,k,1,m))
                        sdq=sign(1d0,dq)
                        dfmin=MIN(abs(dq), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i-1),j,k,1,m)), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i+1),j,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)))
                        adfx_%data(adfx_%x(i),j,k,1,m-nTracerLo+1)=sdq*MAX(0d0,dfmin)
                     END DO
                  END DO
               END DO
            END DO
         END IF

         IF (istimeshift(index, Info%level, adfy, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLo, nTracerHi
                        dq=half*(beforesweepstep_%data(beforesweepstep_%x(i),j+1,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j-1,k,1,m))
                        sdq=sign(1d0,dq)
                        dfmin=MIN(abs(dq), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j-1,k,1,m)), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i),j+1,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)))
                        adfy_%data(adfy_%x(i),j,k,1,m-nTracerLo+1)=sdq*MAX(0d0,dfmin)
                     END DO
                  END DO
               END DO
            END DO
         END IF

         IF (istimeshift(index, Info%level, adfz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLo, nTracerHi
                        dq=half*(beforesweepstep_%data(beforesweepstep_%x(i),j,k+1,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j,k-1,1,m))
                        sdq=sign(1d0,dq)
                        dfmin=MIN(abs(dq), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j,k-1,1,m)), &
                             sdq*2d0*(beforesweepstep_%data(beforesweepstep_%x(i),j,k+1,1,m)-beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)))
                        adfz_%data(adfz_%x(i),j,k,1,m-nTracerLo+1)=sdq*MAX(0d0,dfmin)
                     END DO
                  END DO
               END DO
            END DO
         END IF


         IF (istime(index, Info%level, f2x, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLO, nTracerHI
                        u_edge=half*(w2_%data(w2_%x(i),j,k,1,ivx)+w2_%data(w2_%x(i-1),j,k,1,ivx))
                        IF (u_edge > 0d0) THEN
                           f2x_%data(f2x_%x(i),j,k,1,m)=w2_%data(w2_%x(i-1),j,k,1,ivx)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i-1),j,k,1,m)+half*adfx_%data(adfx_%x(i-1),j,k,1,m+1-nTracerLo)*(1d0-MIN(1d0, u_edge*dtdx)))
                        ELSEIF (u_edge < 0d0) THEN
                           f2x_%data(f2x_%x(i),j,k,1,m)=w2_%data(w2_%x(i),j,k,1,ivx)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-half*adfx_%data(adfx_%x(i),j,k,1,m+1-nTracerLo)*(1d0-MIN(1d0, -u_edge*dtdx)))
                        ELSE 
                           f2x_%data(f2x_%x(i),j,k,1,m)=0d0
                        END IF
                     END DO
                  END DO
               END DO
            END DO
         END IF

         IF (istime(index, Info%level, f2y, mB)) THEN
            DO i=mB(1,1), mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLO, nTracerHI
                        u_edge=half*(w2_%data(w2_%x(i),j,k,1,ivy)+w2_%data(w2_%x(i),j-1,k,1,ivy))
                        IF (u_edge > 0d0) THEN
                           f2y_%data(f2y_%x(i),j,k,1,m)=w2_%data(w2_%x(i),j-1,k,1,ivy)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i),j-1,k,1,m)+half*adfy_%data(adfy_%x(i),j-1,k,1,m+1-nTracerLo)*(1d0-MIN(1d0, u_edge*dtdy)))
                        ELSEIF (u_edge < 0d0) THEN
                           f2y_%data(f2y_%x(i),j,k,1,m)=w2_%data(w2_%x(i),j,k,1,ivy)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-half*adfy_%data(adfy_%x(i),j,k,1,m+1-nTracerLo)*(1d0-MIN(1d0, -u_edge*dtdy)))
                        ELSE
                           f2y_%data(f2y_%x(i),j,k,1,m)=0d0
                        END IF
                     END DO
                  END DO
               END DO
            END DO
         END IF

         IF (istime(index, Info%level, f2z, mB)) THEN
            DO i=mB(1,1), mB(1,2)
               DO j=mB(2,1), mB(2,2)
                  DO k=mB(3,1), mB(3,2)
                     DO m=nTracerLO, nTracerHI
                        u_edge=half*(w2_%data(w2_%x(i),j,k,1,ivz)+w2_%data(w2_%x(i),j,k-1,1,ivz))
                        IF (u_edge > 0d0) THEN
                           f2z_%data(f2z_%x(i),j,k,1,m)=w2_%data(w2_%x(i),j,k-1,1,ivz)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i),j,k-1,1,m)+half*adfz_%data(adfz_%x(i),j,k-1,1,m+1-nTracerLo)*(1d0-MIN(1d0, u_edge*dtdz)))
                        ELSEIF (u_edge < 0d0) THEN
                           f2z_%data(f2z_%x(i),j,k,1,m)=w2_%data(w2_%x(i),j,k,1,ivz)* &
                                (beforesweepstep_%data(beforesweepstep_%x(i),j,k,1,m)-half*adfz_%data(adfz_%x(i),j,k,1,m+1-nTracerLo)*(1d0-MIN(1d0, -u_edge*dtdz)))
                        ELSE
                           f2z_%data(f2z_%x(i),j,k,1,m)=0d0
                        END IF
                     END DO
                  END DO
               END DO
            END DO
         END IF
      END SUBROUTINE calc_tracer_fluxes

      !> Calculate HViscosity terms
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE HVisc(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         IF (istimeshift(index, Info%level, etax, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  etax_%data(etax_%x(i),j,k,1,1)=half*maxval(abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,:) - &
                       SpeedsX_%data(SpeedsX_%x(i-1),j,k,1,:)))

                  !            etax_%data(etax_%x(i),j,k,1,1)=half*abs(SpeedsX_%data(SpeedsX_%x(i),j,k,1,NrWaves) - &
                  !                 SpeedsX_%data(SpeedsX_%x(i-1),j,k,1,1))
               END FORALL
            END DO
         END IF

         IF (istimeshift(index, Info%level, etay, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  etay_%data(etay_%x(i),j,k,1,1)=half*maxval(abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,:) - &
                       SpeedsY_%data(SpeedsY_%x(i),j-1,k,1,:)))

                  !           etay_%data(etay_%x(i),j,k,1,1)=half*abs(SpeedsY_%data(SpeedsY_%x(i),j,k,1,NrWaves) - &
                  !                 SpeedsY_%data(SpeedsY_%x(i),j-1,k,1,1))
               END FORALL
            END DO
         END IF

         IF (istimeshift(index, Info%level, etaz, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  etaz_%data(etaz_%x(i),j,k,1,1)=half*maxval(abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,:) - &
                       SpeedsZ_%data(SpeedsZ_%x(i),j,k-1,1,:)))

                  !            etaz_%data(etaz_%x(i),j,k,1,1)=half*abs(SpeedsZ_%data(SpeedsZ_%x(i),j,k,1,NrWaves) - &
                  !                 SpeedsZ_%data(SpeedsZ_%x(i),j,k-1,1,1))

               END FORALL
            END DO
         END IF

         IF (ViscCD == 1) THEN !just normal viscosity at boundary
            IF (istimeshift(index, Info%level, eta2x, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2x_%data(eta2x_%x(i),j,k,1,1) = etax_%data(etax_%x(i),j,k,1,1)
                  END FORALL
               END DO
            END IF
            IF (istimeshift(index, Info%level, eta2y, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2y_%data(eta2y_%x(i),j,k,1,1) = etay_%data(etay_%x(i),j,k,1,1)
                  END FORALL
               END DO
            END IF
            IF (istimeshift(index, Info%level, eta2z, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2z_%data(eta2z_%x(i),j,k,1,1) = etaz_%data(etaz_%x(i),j,k,1,1)
                  END FORALL
               END DO
            END IF
         ELSE IF (ViscCD == 2) THEN !H-viscosity
            IF (istimeshift(index, Info%level, eta2x, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2x_%data(eta2x_%x(i),j,k,1,1) = max(etax_%data(etax_%x(i),j,k,1,1), max(maxval(etay_%data(etay_%x(i-1),j:j+1,k,1,1)), &
                          maxval(etay_%data(etay_%x(i),j:j+1,k,1,1))))
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        eta2x_%data(eta2x_%x(i),j,k,1,1) = max(eta2x_%data(eta2x_%x(i),j,k,1,1), max(maxval(etaz_%data(etaz_%x(i-1),j,k:k+1,1,1)), &
                             maxval(etaz_%data(etaz_%x(i),j,k:k+1,1,1))))
                     END FORALL
                  END IF
               END DO
            END IF
            IF (istimeshift(index, Info%level, eta2y, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2y_%data(eta2y_%x(i),j,k,1,1) = max(etay_%data(etay_%x(i),j,k,1,1), max(maxval(etax_%data(etax_%x(i),j-1:j,k,1,1)), &
                          maxval(etax_%data(etax_%x(i+1),j-1:j,k,1,1))))
                  END FORALL
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        eta2y_%data(eta2y_%x(i),j,k,1,1) = max(eta2y_%data(eta2y_%x(i),j,k,1,1), maxval(etaz_%data(etaz_%x(i),j-1:j,k:k+1,1,1)))
                     END FORALL
                  END IF
               END DO

            END IF
            IF (istimeshift(index, Info%level, eta2z, mB)) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2z_%data(eta2z_%x(i),j,k,1,1) = max(etaz_%data(etaz_%x(i),j,k,1,1), maxval(etay_%data(etay_%x(i),j:j+1,k-1:k,1,1)))
                  END FORALL
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     eta2z_%data(eta2z_%x(i),j,k,1,1) = max(eta2z_%data(eta2z_%x(i),j,k,1,1), max(maxval(etax_%data(etax_%x(i),j,k-1:k,1,1)), &
                          maxval(etax_%data(etax_%x(i+1),j,k-1:k,1,1))))
                  END FORALL
               END DO
            END IF
         END IF
      END SUBROUTINE HVisc

      !> Calculate Final fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE calc_final_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF (istimeshift(index, Info%level, f2x, mB)) THEN
            IF (ViscCD > 0) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1), mB(2,2)
                     DO k=mB(3,1), mB(3,2)
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_x(q2Lx_%data(q2Lx_%x(i),j,k,1,1:SweepCons), q2Rx_%data(q2Rx_%x(i),j,k,1,1:SweepCons), &
                             f2x_%data(f2x_%x(i),j,k,1,1:SweepCons), eta2x_%data(eta2x_%x(i),j,k,1,1)))
                     END DO
                  END DO
               END DO
            ELSE
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1), mB(2,2)
                     DO k=mB(3,1), mB(3,2)
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_x(q2Lx_%data(q2Lx_%x(i),j,k,1,1:SweepCons), q2Rx_%data(q2Rx_%x(i),j,k,1,1:SweepCons), &
                             f2x_%data(f2x_%x(i),j,k,1,1:SweepCons))) 
                     END DO
                  END DO
               END DO
            END IF
            DO i=mB(1,1),mB(1,2)
               DO m=1,nFlux
                  f2x_%data(f2x_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m)) = &
                       f2x_%data(f2x_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m))*dtdx
               END DO
            END DO
         END IF
         IF (istimeshift(index, Info%level, f2y, mB)) THEN 
            IF (ViscCD > 0) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)       
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_y(q2Ly_%data(q2Ly_%x(i),j,k,1,1:SweepCons), q2Ry_%data(q2Ry_%x(i),j,k,1,1:SweepCons), &
                             f2y_%data(f2y_%x(i),j,k,1,1:SweepCons), eta2y_%data(eta2y_%x(i),j,k,1,1)))           
                     END DO
                  END DO
               END DO
            ELSE
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)       
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_y(q2Ly_%data(q2Ly_%x(i),j,k,1,1:SweepCons), q2Ry_%data(q2Ry_%x(i),j,k,1,1:SweepCons), &
                             f2y_%data(f2y_%x(i),j,k,1,1:SweepCons)))
                     END DO
                  END DO
               END DO
            END IF
            DO i=mB(1,1),mB(1,2)
               DO m=1,nFlux
                  f2y_%data(f2y_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m)) = &
                       f2y_%data(f2y_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m))*dtdy
               END DO

            END DO
         END IF
         IF (istimeshift(index, Info%level, f2z, mB)) THEN
            IF (ViscCD > 0) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_z(q2Lz_%data(q2Lz_%x(i),j,k,1,1:SweepCons), q2Rz_%data(q2Rz_%x(i),j,k,1,1:SweepCons), &
                             f2z_%data(f2z_%x(i),j,k,1,1:SweepCons), eta2z_%data(eta2z_%x(i),j,k,1,1))) 
                     END DO
                  END DO
               END DO
            ELSE
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        maxsolverspeed(Info%level)=max(maxsolverspeed(Info%level), calc_flux_z(q2Lz_%data(q2Lz_%x(i),j,k,1,1:SweepCons), q2Rz_%data(q2Rz_%x(i),j,k,1,1:SweepCons), &
                             f2z_%data(f2z_%x(i),j,k,1,1:SweepCons)))
                     END DO
                  END DO
               END DO
            END IF

            DO i=mB(1,1),mB(1,2)
               DO m=1,nFlux
                  f2z_%data(f2z_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m)) = &
                       f2z_%data(f2z_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,FluxFields(m))*dtdz
               END DO
            END DO
         END IF
      END SUBROUTINE calc_final_fluxes

      !> Calculate Final EMF's
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE calc_final_emf(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mS(3,2)    ! Bounds of slab for storing
         REAL (kind=qPrec) :: r
         IF (istimeshift(index, Info%level, e2x_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  e2x_bar_%data(e2x_bar_%x(i),j,k,1,1) = &
                       w2_%data(w2_%x(i),j,k,1,ivz)*w2_%data(w2_%x(i),j,k,1,iBy)-w2_%data(w2_%x(i),j,k,1,ivy)*w2_%data(w2_%x(i),j,k,1,iBz)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, e2y_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  e2y_bar_%data(e2y_bar_%x(i),j,k,1,1) = &
                       w2_%data(w2_%x(i),j,k,1,ivx)*w2_%data(w2_%x(i),j,k,1,iBz)-w2_%data(w2_%x(i),j,k,1,ivz)*w2_%data(w2_%x(i),j,k,1,iBx)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, e2z_bar, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  e2z_bar_%data(e2z_bar_%x(i),j,k,1,1) = &
                       w2_%data(w2_%x(i),j,k,1,ivy)*w2_%data(w2_%x(i),j,k,1,iBx)-w2_%data(w2_%x(i),j,k,1,ivx)*w2_%data(w2_%x(i),j,k,1,iBy)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, e2x, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  e2x_%data(e2x_%x(i),j,k,1,1)=  &
                       dtdx*upwinded_emf(f2z_%data(f2z_%x(i), j-1:j,k, 1, iBy),-f2y_%data(f2y_%x(i), j,k-1:k, 1, iBz),&
                       e2x_bar_%data(e2x_bar_%x(i),j-1:j,k-1:k,1,1), &
                       f2y_%data(f2y_%x(i),j,k-1:k,1,1),f2z_%data(f2z_%x(i),j-1:j,k,1,1))
               END FORALL
            END DO
            IF (lStressTest) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL Randomize(e2x_%data(e2x_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1:1))
               END DO
            END IF
            mS(2:3,:)=mB(2:3,:)
            mS(1,:)=index+mB(1,:)
            CALL StoreEmfs(Info, mS, 1, e2x_%data(e2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
         END IF
         IF (istimeshift(index, Info%level, e2y, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  e2y_%data(e2y_%x(i),j,k,1,1)=dtdx*upwinded_emf(f2x_%data(f2x_%x(i), j,k-1:k, 1, iBz), - &
                       (/f2z_%data(f2z_%x(i-1), j,k,1,iBx), f2z_%data(f2z_%x(i), j,k,1,iBx)/), &
                       reshape((/e2y_bar_%data(e2y_bar_%x(i-1),j,k-1:k,1,1),e2y_bar_%data(e2y_bar_%x(i),j,k-1:k,1,1)/),(/2,2/)), &
                       (/f2z_%data(f2z_%x(i-1),j,k,1,1),f2z_%data(f2z_%x(i),j,k,1,1)/),f2x_%data(f2x_%x(i),j,k-1:k,1,1))
               END FORALL
            END DO
            IF (lStressTest) THEN
               DO i=mB(1,1), mB(1,2)
                  CALL Randomize(e2y_%data(e2y_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1:1))
               END DO
            END IF
            mS(2:3,:)=mB(2:3,:)
            mS(1,:)=index+mB(1,:)
            CALL StoreEmfs(Info, mS, 2, e2y_%data(e2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2), 1, 1))
         END IF
         IF (istimeshift(index, Info%level, e2z, mB)) THEN
            IF (iCylindrical == NoCyl) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     e2z_%data(e2z_%x(i), j, k, 1,1)=&
                          dtdx*upwinded_emf((/f2y_%data(f2y_%x(i-1), j,k, 1, iBx),f2y_%data(f2y_%x(i), j,k, 1, iBx)/), - &
                          f2x_%data(f2x_%x(i), j-1:j,k, 1, iBy), &
                          transpose(reshape((/e2z_bar_%data(e2z_bar_%x(i-1),j-1:j,k,1,1), e2z_bar_%data(e2z_bar_%x(i),j-1:j,k,1,1)/),(/2,2/))), &
                          f2x_%data(f2x_%x(i),j-1:j,k,1,1),(/f2y_%data(f2y_%x(i-1),j,k,1,1),f2y_%data(f2y_%x(i),j,k,1,1)/))
                  END FORALL
               END DO
            ELSE
               r=Info%xBounds(1,1)+(index+i-1)*dx
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     e2z_%data(e2z_%x(i),j,k,1,1)=r*&
                          dtdx*upwinded_emf((/f2y_%data(f2y_%x(i-1), j,k, 1, iBx),f2y_%data(f2y_%x(i), j,k, 1, iBx)/), - &
                          f2x_%data(f2x_%x(i), j-1:j,k, 1, iBy), &
                          transpose(reshape((/e2z_bar_%data(e2z_bar_%x(i-1),j-1:j,k,1,1), e2z_bar_%data(e2z_bar_%x(i),j-1:j,k,1,1)/),(/2,2/))), &
                          f2x_%data(f2x_%x(i),j-1:j,k,1,1),(/f2y_%data(f2y_%x(i-1),j,k,1,1),f2y_%data(f2y_%x(i),j,k,1,1)/))
                  END FORALL
               END DO
            END IF
            IF (lStressTest) THEN
               DO i=mB(1,1),mB(1,2)
                  CALL Randomize(e2z_%data(e2z_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1:1))
               END DO
            END IF
            mS(2:3,:)=mB(2:3,:)
            mS(1,:)=index+mB(1,:)
            CALL StoreEmfs(Info, mS, 3, e2z_%data(e2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
         END IF
      END SUBROUTINE calc_final_emf

      !> Update Final fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE update_final_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         REAL(KIND=qPREC) :: rho, rho_min, rho_max
         LOGICAL :: mask(4)
         LOGICAL :: lCheck
         INTEGER :: l
         ! The emf components of the fluxes (ie f2x) in MHD don't get multiplied by dtdx - but instead are stored in ctf2x
         IF (lApplyDiffusion) THEN

            IF (istimeshift(index, Info%level, cornerdiv, mB)) THEN

               IF (nDim == 1) THEN
                  DO i=mB(1,1), mB(1,2)
                     DO j=mB(2,1), mB(2,2)
                        DO k=mB(3,1), mB(3,2)

                           cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)= (w_%data(w_%x(i),j,k,1,imom(1)) - w_%data(w_%x(i-1),j,k,1,imom(1)))
                        END DO
                     END DO
                  END DO

               ELSEIF (nDim == 2) THEN
                  DO i=mB(1,1), mB(1,2)
                     DO j=mB(2,1), mB(2,2)
                        DO k=mB(3,1), mB(3,2)
                           cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)= .5d0*( &
                                ((w_%data(w_%x(i),j-1,k,1,imom(1)) - w_%data(w_%x(i-1),j-1,k,1,imom(1))) + &
                                (w_%data(w_%x(i),j,k,1,imom(1)) - w_%data(w_%x(i-1),j,k,1,imom(1)))) + &
                                ((w_%data(w_%x(i-1),j,k,1,imom(2)) - w_%data(w_%x(i-1),j-1,k,1,imom(2))) + &
                                (w_%data(w_%x(i),j,k,1,imom(2)) - w_%data(w_%x(i),j-1,k,1,imom(2)))))
                        END DO
                     END DO
                  END DO
               ELSE
                  DO i=mB(1,1), mB(1,2)
                     DO j=mB(2,1), mB(2,2)
                        DO k=mB(3,1), mB(3,2)
                           cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)= .25d0*( &
                                (((w_%data(w_%x(i),j-1,k-1,1,imom(1)) - w_%data(w_%x(i-1),j-1,k-1,1,imom(1))) + &
                                (w_%data(w_%x(i),j,k-1,1,imom(1)) - w_%data(w_%x(i-1),j,k-1,1,imom(1)))) + &
                                ((w_%data(w_%x(i),j-1,k,1,imom(1)) - w_%data(w_%x(i-1),j-1,k,1,imom(1))) + &
                                (w_%data(w_%x(i),j,k,1,imom(1)) - w_%data(w_%x(i-1),j,k,1,imom(1))))) + &
                                (((w_%data(w_%x(i-1),j,k-1,1,imom(2)) - w_%data(w_%x(i-1),j-1,k-1,1,imom(2))) + &
                                (w_%data(w_%x(i),j,k-1,1,imom(2)) - w_%data(w_%x(i),j-1,k-1,1,imom(2)))) + &
                                ((w_%data(w_%x(i-1),j,k,1,imom(2)) - w_%data(w_%x(i-1),j-1,k,1,imom(2))) + &
                                (w_%data(w_%x(i),j,k,1,imom(2)) - w_%data(w_%x(i),j-1,k,1,imom(2))))) + &
                                (((w_%data(w_%x(i-1),j-1,k,1,imom(3)) - w_%data(w_%x(i-1),j-1,k-1,1,imom(3))) + &
                                (w_%data(w_%x(i),j-1,k,1,imom(3)) - w_%data(w_%x(i),j-1,k-1,1,imom(3)))) + &
                                ((w_%data(w_%x(i-1),j,k,1,imom(3)) - w_%data(w_%x(i-1),j,k-1,1,imom(3))) + &
                                (w_%data(w_%x(i),j,k,1,imom(3)) - w_%data(w_%x(i),j,k-1,1,imom(3))))))
                        END DO
                     END DO
                  END DO
               END IF
            END IF
         END IF

         IF (istime(index, Info%level, f2x, mB)) THEN
            IF (lStressTest) THEN
               DO i=mB(1,1), mB(1,2)
                  CALL Randomize(f2x_%data(f2x_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:))
               END DO
            END IF
            IF (lApplyDiffusion) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        IF (nDim == 1) THEN
                           f2x_%data(f2x_%x(i),j,k,1,FluxFields) = f2x_%data(f2x_%x(i),j,k,1,FluxFields) + &
                                dtdx*DIFF_ALPHA*max(0d0,-(cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)))*( &
                                Info%q(index+i-1,j,k,FluxFields) - &
                                Info%q(index+i,j,k,FluxFields))

                        ELSEIF (nDim == 2) THEN

                           f2x_%data(f2x_%x(i),j,k,1,FluxFields) = f2x_%data(f2x_%x(i),j,k,1,FluxFields) + &
                                dtdx*DIFF_ALPHA*max(0d0,-.5d0*(cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)+cornerdiv_%data(cornerdiv_%x(i),j+1,k,1,1)))*( &
                                Info%q(index+i-1,j,k,FluxFields) - &
                                Info%q(index+i,j,k,FluxFields))
                        ELSE
                           f2x_%data(f2x_%x(i),j,k,1,FluxFields) = f2x_%data(f2x_%x(i),j,k,1,FluxFields) + &
                                dtdx*DIFF_ALPHA*max(0d0,-.25d0*(sum(cornerdiv_%data(cornerdiv_%x(i),j,k:k+1,1,1))+sum(cornerdiv_%data(cornerdiv_%x(i),j+1,k:k+1,1,1))))*( &
                                Info%q(index+i-1,j,k,FluxFields) - &
                                Info%q(index+i,j,k,FluxFields))

                        END IF
                     END DO
                  END DO
               END DO
            END IF

            IF (lApplyLOF) THEN
               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        lCheck = (limiter_y_%data(limiter_y_%x(i),j,k,1,1) == 1 .OR. limiter_y_%data(limiter_y_%x(i-1),j,k,1,1) == 1)
                        IF (.NOT. lCheck .AND. nDim == 3) lCheck = (limiter_z_%data(limiter_z_%x(i),j,k,1,1) == 1 .OR. limiter_z_%data(limiter_z_%x(i-1),j,k,1,1) == 1)
                        IF (lCheck) THEN
                           rho=half*(Info%q(index+i,j,k,1)+Info%q(index+i-1,j,k,1))
                           rho_max=rho*(1d0+1d-6)
                           rho_min=rho*(1d0-1d-6)
                           DO l=1,4
                              IF (Info%q(index+i-3+l,j,k,1) > rho_max) THEN
                                 mask(l)=.true.
                              ELSEIF (Info%q(index+i-3+l,j,k,1) < rho_min) THEN
                                 mask(l)=.false.
                              ELSE
                                 mask=.false.
                                 exit
                              END IF
                           END DO
                           IF (ALL(mask .eqv. (/.true.,.false.,.true.,.false./)) .OR. ALL(mask .eqv. (/.false.,.true.,.false.,.true./))) THEN
                              f2x_%data(f2x_%x(i),j,k,1,FluxFields) = f2x_%data(f2x_%x(i),j,k,1,FluxFields) - LOF_ALPHA*( &
                                   Info%q(index+i,j,k,FluxFields) - &
                                   Info%q(index+i-1,j,k,FluxFields))
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
            END IF

         END IF

         IF (istime(index, Info%level, f2y, mB)) THEN 
            IF (lStressTest) THEN
               DO i=mB(1,1), mB(1,2)
                  CALL Randomize(f2y_%data(f2y_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:))
               END DO
            END IF

            IF (lApplyDiffusion) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        IF (nDim == 2) THEN
                           f2y_%data(f2y_%x(i),j,k,1,FluxFields) = f2y_%data(f2y_%x(i),j,k,1,FluxFields) + &
                                dtdy*DIFF_ALPHA*max(0d0,-.5d0*(cornerdiv_%data(cornerdiv_%x(i),j,k,1,1)+cornerdiv_%data(cornerdiv_%x(i+1),j,k,1,1)))*( &
                                Info%q(index+i,j-1,k,FluxFields) - &
                                Info%q(index+i,j,k,FluxFields))
                        ELSE
                           f2y_%data(f2y_%x(i),j,k,1,FluxFields) = f2y_%data(f2y_%x(i),j,k,1,FluxFields) + &
                                dtdy*DIFF_ALPHA*max(0d0,-.25d0*(sum(cornerdiv_%data(cornerdiv_%x(i),j,k:k+1,1,1))+sum(cornerdiv_%data(cornerdiv_%x(i+1),j,k:k+1,1,1))))*( &
                                Info%q(index+i,j-1,k,FluxFields) - &
                                Info%q(index+i,j,k,FluxFields))

                        END IF
                     END DO
                  END DO
               END DO
            END IF

            IF (LApplyLOF) THEN
               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        lCheck = (limiter_x_%data(limiter_x_%x(i),j,k,1,1) == 1 .OR. limiter_x_%data(limiter_x_%x(i),j-1,k,1,1) == 1)
                        IF (.NOT. lCheck .AND. nDim == 3) lCheck = (limiter_z_%data(limiter_z_%x(i),j,k,1,1) == 1 .OR. limiter_z_%data(limiter_z_%x(i),j-1,k,1,1) == 1)
                        IF (lCheck) THEN
                           rho=half*(Info%q(index+i,j,k,1)+Info%q(index+i,j-1,k,1))
                           rho_max=rho*(1d0+1d-6)
                           rho_min=rho*(1d0-1d-6)
                           DO l=1,4
                              IF (Info%q(index+i,j-3+l,k,1) > rho_max) THEN
                                 mask(l)=.true.
                              ELSEIF (Info%q(index+i,j-3+l,k,1) < rho_min) THEN
                                 mask(l)=.false.
                              ELSE
                                 mask=.false.
                                 exit
                              END IF
                           END DO
                           IF (ALL(mask .eqv. (/.true.,.false.,.true.,.false./)) .OR. ALL(mask .eqv. (/.false.,.true.,.false.,.true./))) THEN
                              f2y_%data(f2y_%x(i),j,k,1,FluxFields) = f2y_%data(f2y_%x(i),j,k,1,FluxFields) - LOF_ALPHA*( &
                                   Info%q(index+i,j,k,FluxFields) - &
                                   Info%q(index+i,j-1,k,FluxFields))
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
            END IF
         END IF
         IF (istime(index, Info%level, f2z, mB)) THEN
            !      f2z_%data(f2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepCons+1:)=f2z_%data(f2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,SweepCons+1:)*dtdz
            IF (lStressTest) THEN
               DO i=mB(1,1), mB(1,2)
                  CALL Randomize(f2z_%data(f2z_%x(i),mB(2,1):mB(2,2), mB(3,1):mB(3,2),1,:))
               END DO
            END IF

            IF (lApplyDiffusion) THEN
               DO i=mB(1,1), mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        f2z_%data(f2z_%x(i),j,k,1,FluxFields) = f2z_%data(f2z_%x(i),j,k,1,FluxFields) + &
                             dtdz*DIFF_ALPHA*max(0d0,-.25d0*(sum(cornerdiv_%data(cornerdiv_%x(i),j:j+1,k,1,1))+sum(cornerdiv_%data(cornerdiv_%x(i+1),j:j+1,k,1,1))))*( &
                             Info%q(index+i,j,k-1,FluxFields) - &
                             Info%q(index+i,j,k,FluxFields))

                     END DO
                  END DO
               END DO
            END IF

            IF (lApplyLOF) THEN
               DO i=mB(1,1),mB(1,2)
                  DO j=mB(2,1),mB(2,2)
                     DO k=mB(3,1),mB(3,2)
                        lCheck = (limiter_x_%data(limiter_x_%x(i),j,k,1,1) == 1 .OR. limiter_x_%data(limiter_x_%x(i-1),j,k,1,1) == 1)
                        IF (.NOT. lCheck) lCheck = (limiter_y_%data(limiter_y_%x(i),j,k,1,1) == 1 .OR. limiter_y_%data(limiter_y_%x(i),j-1,k,1,1) == 1)
                        IF (lCheck) THEN
                           rho=half*(Info%q(index+i,j,k,1)+Info%q(index+i,j,k-1,1))
                           rho_max=rho*(1d0+1d-6)
                           rho_min=rho*(1d0-1d-6)
                           DO l=1,4
                              IF (Info%q(index+i,j,k-3+l,1) > rho_max) THEN
                                 mask(l)=.true.
                              ELSEIF (Info%q(index+i,j,k-3+l,1) < rho_min) THEN
                                 mask(l)=.false.
                              ELSE
                                 mask=.false.
                                 exit
                              END IF
                           END DO
                           IF (ALL(mask .eqv. (/.true.,.false.,.true.,.false./)) .OR. ALL(mask .eqv. (/.false.,.true.,.false.,.true./))) THEN
                              f2z_%data(f2z_%x(i),j,k,1,FluxFields) = f2z_%data(f2z_%x(i),j,k,1,FluxFields) - LOF_ALPHA*( &
                                   Info%q(index+i,j,k,FluxFields) - &
                                   Info%q(index+i,j,k-1,FluxFields))
                           END IF
                        END IF
                     END DO
                  END DO
               END DO
            END IF

         END IF

         IF (istimeshift(index, Info%level, ctf2x, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  ctf2x_%data(ctf2x_%x(i),j,k,1,1)=dtdx*f2x_%data(f2x_%x(i),j,k,1,iBy)
                  ctf2x_%data(ctf2x_%x(i),j,k,1,2)=dtdx*f2x_%data(f2x_%x(i),j,k,1,iBz)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ctf2y, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  ctf2y_%data(ctf2y_%x(i),j,k,1,1)=dtdy*f2y_%data(f2y_%x(i),j,k,1,iBz)
                  ctf2y_%data(ctf2y_%x(i),j,k,1,2)=dtdy*f2y_%data(f2y_%x(i),j,k,1,iBx)
               END FORALL
            END DO
         END IF
         IF (istimeshift(index, Info%level, ctf2z, mB)) THEN
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                  ctf2z_%data(ctf2z_%x(i),j,k,1,1)=dtdz*f2z_%data(f2z_%x(i),j,k,1,iBx)
                  ctf2z_%data(ctf2z_%x(i),j,k,1,2)=dtdz*f2z_%data(f2z_%x(i),j,k,1,iBy)
               END FORALL
            END DO
         END IF


         IF (lSelfGravity) THEN
            IF (nDim == 1) THEN
               IF (istime(index, Info%level,  f2x, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2x_%data(f2x_%x(i),j,k,1,ivx) = f2x_%data(f2x_%x(i),j,k,1,ivx) + &
                             dtdx*(.125d0/Pi/ScaleGrav*gradphix_%data(gradphix_%x(i),j,k,1,1)**2+half*mean_density*(Info%q(index+i,j,k,iPhiGas)+ Info%q(index+i-1,j,k,iPhiGas)))
                     END FORALL
                  END DO
               END IF
            ELSEIF (nDim == 2) THEN
               IF (istime(index, Info%level,  f2x, mB)) THEN

                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2x_%data(f2x_%x(i),j,k,1,ivx) = f2x_%data(f2x_%x(i),j,k,1,ivx) + dtdx*(.125d0/Pi/ScaleGrav*&
                             (gradphix_%data(gradphix_%x(i),j,k,1,1)**2-(.25d0*sum(gradphiy_%data((/gradphiy_%x(i-1:i)/),j:j+1,k,1,1)))**2)+&
                             half*mean_density*SUM(Info%q(index+i-1:index+i,j,k,iPhiGas)))
                        f2x_%data(f2x_%x(i),j,k,1,ivy) = f2x_%data(f2x_%x(i),j,k,1,ivy) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphix_%data(gradphix_%x(i),j,k,1,1)*.25d0*sum(gradphiy_%data((/gradphiy_%x(i-1:i)/),j:j+1,k,1,1))))
                     END FORALL
                  END DO
               END IF
               IF (istime(index, Info%level,  f2y, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2y_%data(f2y_%x(i),j,k,1,ivy) = f2y_%data(f2y_%x(i),j,k,1,ivy) + dtdx*(.125d0/Pi/ScaleGrav*&
                             (gradphiy_%data(gradphiy_%x(i),j,k,1,1)**2-(.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j-1:j,k,1,1)))**2)+&
                             half*mean_density*SUM(Info%q(index+i,j-1:j,k,iPhiGas)))
                        f2y_%data(f2y_%x(i),j,k,1,ivx) = f2y_%data(f2y_%x(i),j,k,1,ivx) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphiy_%data(gradphiy_%x(i),j,k,1,1)*.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j-1:j,k,1,1))))
                     END FORALL
                  END DO
               END IF
            ELSE! nDim == 3
               IF (istime(index, Info%level,  f2x, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2x_%data(f2x_%x(i),j,k,1,ivx) = f2x_%data(f2x_%x(i),j,k,1,ivx) + dtdx*(.125d0/Pi/ScaleGrav*&
                             (gradphix_%data(gradphix_%x(i),j,k,1,1)**2-(.25d0*sum(gradphiy_%data((/gradphiy_%x(i-1:i)/),j:j+1,k,1,1)))**2-(.25d0*sum(gradphiz_%data((/gradphiz_%x(i-1:i)/),j,k:k+1,1,1)))**2)+&
                             half*mean_density*SUM(Info%q(index+i-1:index+i,j,k,iPhiGas)))
                        f2x_%data(f2x_%x(i),j,k,1,ivy) = f2x_%data(f2x_%x(i),j,k,1,ivy) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphix_%data(gradphix_%x(i),j,k,1,1)*.25d0*sum(gradphiy_%data((/gradphiy_%x(i-1:i)/),j:j+1,k,1,1))))
                        f2x_%data(f2x_%x(i),j,k,1,ivz) = f2x_%data(f2x_%x(i),j,k,1,ivz) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphix_%data(gradphix_%x(i),j,k,1,1)*.25d0*sum(gradphiz_%data((/gradphiz_%x(i-1:i)/),j,k:k+1,1,1))))
                     END FORALL
                  END DO
               END IF
               IF (istime(index, Info%level,  f2y, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2y_%data(f2y_%x(i),j,k,1,ivy) = f2y_%data(f2y_%x(i),j,k,1,ivy) + dtdx*(.125d0/Pi/ScaleGrav*&
                             (gradphiy_%data(gradphiy_%x(i),j,k,1,1)**2-(.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j-1:j,k,1,1)))**2-(.25d0*sum(gradphiz_%data(gradphiz_%x(i),j-1:j,k:k+1,1,1)))**2)+&
                             half*mean_density*SUM(Info%q(index+i,j-1:j,k,iPhiGas)))
                        f2y_%data(f2y_%x(i),j,k,1,ivx) = f2y_%data(f2y_%x(i),j,k,1,ivx) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphiy_%data(gradphiy_%x(i),j,k,1,1)*.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j-1:j,k,1,1))))
                        f2y_%data(f2y_%x(i),j,k,1,ivz) = f2y_%data(f2y_%x(i),j,k,1,ivz) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphiy_%data(gradphiy_%x(i),j,k,1,1)*.25d0*sum(gradphiz_%data(gradphiz_%x(i),j-1:j,k:k+1,1,1))))
                     END FORALL
                  END DO
               END IF
               IF (istime(index, Info%level,  f2z, mB)) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        f2z_%data(f2z_%x(i),j,k,1,ivz) = f2z_%data(f2z_%x(i),j,k,1,ivz) + dtdx*(.125d0/Pi/ScaleGrav*&
                             (gradphiz_%data(gradphiz_%x(i),j,k,1,1)**2-(.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j,k-1:k,1,1)))**2-(.25d0*sum(gradphiy_%data(gradphiy_%x(i),j:j+1,k-1:k,1,1)))**2)+&
                             half*mean_density*SUM(Info%q(index+i,j,k-1:k,iPhiGas)))
                        f2z_%data(f2z_%x(i),j,k,1,ivx) = f2z_%data(f2z_%x(i),j,k,1,ivx) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphiz_%data(gradphiz_%x(i),j,k,1,1)*.25d0*sum(gradphix_%data((/gradphix_%x(i:i+1)/),j,k-1:k,1,1))))
                        f2z_%data(f2z_%x(i),j,k,1,ivy) = f2z_%data(f2z_%x(i),j,k,1,ivy) + dtdx*(.25d0/Pi/ScaleGrav*&
                             (gradphiz_%data(gradphiz_%x(i),j,k,1,1)*.25d0*sum(gradphiy_%data(gradphiy_%x(i),j:j+1,k-1:k,1,1))))
                     END FORALL
                  END DO
               END IF
            END IF
         END IF
      END SUBROUTINE update_final_fluxes

      !> Store fixup fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE store_fixup_fluxes(Info, index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: mS(3,2)    ! Bounds of slab to store fluxes

         IF (istime(index, Info%level, f2x, mB)) THEN      
            ms(1,:)=index+mB(1,:)
            mS(2:3,:)=mB(2:3,:)
            CALL storefixupfluxes(Info,mS,1,f2x_%data(f2x_%x(mB(1,1):mB(1,2)),:,:,1,:))
         END IF
         IF (istime(index, Info%level, f2y, mB)) THEN
            ms(1,:)=index+mB(1,:)
            mS(2:3,:)=mB(2:3,:)
            CALL storefixupfluxes(Info,mS,2,f2y_%data(f2y_%x(mB(1,1):mB(1,2)),:,:,1,:))
         END IF
         IF (istime(index, Info%level, f2z, mB)) THEN
            ms(1,:)=index+mB(1,:)
            mS(2:3,:)=mB(2:3,:)
            CALL storefixupfluxes(Info,mS,3,f2z_%data(f2z_%x(mB(1,1):mB(1,2)),:,:,1,:))
         END IF
      END SUBROUTINE store_fixup_fluxes


      !> Update aux fields
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE updateB_final(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2), mS(3,2)    ! Bounds of slab to update
         REAL (kind=qPrec) :: ri, rl,rh

         IF (istime(index, Info%level, A3x, mB)) THEN
            Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1)=&
                 Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1) &
                 - (e2z_%data(e2z_%x(mB(1,1):mB(1,2)),mB(2,1)+1:mB(2,2)+1,mB(3,1):mB(3,2),1,1) &
                 - e2z_%data(e2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
            IF (nDim >= 3) THEN
               Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1)=&
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1) &
                    +(e2y_%data(e2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1)+1:mB(3,2)+1,1,1) &
                    - e2y_%data(e2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
            END IF
         END IF
         IF (istime(index, Info%level, A3y, mB)) THEN
            IF (iCylindrical==NoCyl) THEN
               Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),2)=&
                    Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),2) &
                    +(e2z_%data(e2z_%x(mB(1,1)+1:mB(1,2)+1),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) &
                    - e2z_%data(e2z_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
               IF (nDim >= 3) THEN
                  Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),2)=&
                       Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),2) &
                       -(e2x_%data(e2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1)+1:mB(3,2)+1,1,1) &
                       - e2x_%data(e2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
               END IF
            ELSE
               DO i=mB(1,1), mB(1,2)
                  ri=1.d0/(Info%xBounds(1,1)+(REAL(index+i)-half)*dx)
                  rl=(Info%xBounds(1,1)+(index+i-1)*dx)
                  rh=(Info%xBounds(1,1)+(index+i)*dx)
                  Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2),2)=&
                       Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2),2) &
                       + ri*(rh*e2z_%data(e2z_%x(i+1),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) &
                       - rl*e2z_%data(e2z_%x(i),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
               END DO
            END IF
         END IF
         IF (istime(index, Info%level, A3z, mB)) THEN
            Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),3) = &
                 Info%aux(index+mB(1,1):index+mB(1,2),mB(2,1):mB(2,2),mB(3,1):mB(3,2),3) &
                 +(e2x_%data(e2x_%x(mB(1,1):mB(1,2)),mB(2,1)+1:mB(2,2)+1,mB(3,1):mB(3,2),1,1) &
                 - e2x_%data(e2x_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1)) &
                 -(e2y_%data(e2y_%x(mB(1,1)+1:mB(1,2)+1),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1) &
                 - e2y_%data(e2y_%x(mB(1,1):mB(1,2)),mB(2,1):mB(2,2),mB(3,1):mB(3,2),1,1))
         END IF

      END SUBROUTINE updateB_final


      !> Update cell centered fields
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE update_final(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2), mS(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF (istime(index, Info%level, w3, mB)) THEN     
            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2), m=1:nFlux)
                  Info%q(index+i,j,k,FluxFields(m))=Info%q(index+i,j,k,FluxFields(m))+&
                       (f2x_%data(f2x_%x(i),j,k,1,FluxFields(m))-f2x_%data(f2x_%x(i+1),j,k,1,FluxFields(m)))
               END FORALL
               IF (MaintainAuxArrays) Info%q(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2),iBx) = &
                    half*(Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2), 1) + &
                    Info%aux(index+i+1,mB(2,1):mB(2,2),mB(3,1):mB(3,2), 1))
               IF (nDim >= 2) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2), m=1:nFlux)
                     Info%q(index+i,j,k,FluxFields(m))=Info%q(index+i,j,k,FluxFields(m))+&
                          (f2y_%data(f2y_%x(i),j,k,1,FluxFields(m))-f2y_%data(f2y_%x(i),j+1,k,1,FluxFields(m)))
                  END FORALL
                  IF (MaintainAuxArrays) Info%q(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2),iBy) = &
                       half*(Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2), 2) + &
                       Info%aux(index+i,mB(2,1)+1:mB(2,2)+1,mB(3,1):mB(3,2), 2))
                  IF (nDim >= 3) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2), m=1:nFlux)
                        Info%q(index+i,j,k,FluxFields(m))=Info%q(index+i,j,k,FluxFields(m))+&
                             (f2z_%data(f2z_%x(i),j,k,1,FluxFields(m))-f2z_%data(f2z_%x(i),j,k+1,1,FluxFields(m)))
                     END FORALL
                     IF (MaintainAuxArrays) Info%q(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2),iBz) = &
                          half*(Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1):mB(3,2), 3) + &
                          Info%aux(index+i,mB(2,1):mB(2,2),mB(3,1)+1:mB(3,2)+1, 3))
                  END IF
               END IF
               IF (lSelfGravity .AND. iE /= 0) THEN
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                     Info%q(index+i,j,k,iE)=Info%q(index+i,j,k,iE)-half*( &
                          f2x_%data(f2x_%x(i),j,k,1,1)*(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i-1,j,k,iPhiGas)) + &
                          f2x_%data(f2x_%x(i+1),j,k,1,1)*(Info%q(index+i+1,j,k,iPhiGas)-Info%q(index+i,j,k,iPhiGas)))
                  END FORALL
                  IF (nDim >= 2) THEN
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                        Info%q(index+i,j,k,iE)=Info%q(index+i,j,k,iE)-half*( &
                             f2y_%data(f2y_%x(i),j,k,1,1)*(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i,j-1,k,iPhiGas)) + &
                             f2y_%data(f2y_%x(i),j+1,k,1,1)*(Info%q(index+i,j+1,k,iPhiGas)-Info%q(index+i,j,k,iPhiGas)))
                     END FORALL
                     IF (nDim == 3) THEN
                        FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2))
                           Info%q(index+i,j,k,iE)=Info%q(index+i,j,k,iE)-half*( &
                                f2z_%data(f2z_%x(i),j,k,1,1)*(Info%q(index+i,j,k,iPhiGas)-Info%q(index+i,j,k-1,iPhiGas)) + &
                                f2z_%data(f2z_%x(i),j,k+1,1,1)*(Info%q(index+i,j,k+1,iPhiGas)-Info%q(index+i,j,k,iPhiGas)))
                        END FORALL
                     END IF
                  END IF
               END IF
            END DO
         END IF
      END SUBROUTINE update_final

      !> Update cell centered fields using predictor fluxes
      !! @param Info Info structure
      !! @param index Current sweep position
      SUBROUTINE update_final_noctu(Info,index)
         TYPE(InfoDef) :: Info ! Info structure currently updating
         INTEGER :: index      ! Current row in q being updated
         INTEGER :: i,j,k      ! Loop counters
         INTEGER :: mB(3,2)    ! Bounds of slab to update
         INTEGER :: m
         IF (istime(index, Info%level, w3, mB)) THEN

            DO i=mB(1,1),mB(1,2)
               FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                  Info%q(index+i,j,k,SweepFluxFields(m))=Info%q(index+i,j,k,SweepFluxFields(m))+2d0*(fx_%data(fx_%x(i),j,k,1,SweepFluxFields(m)) - &
                       fx_%data(fx_%x(i+1),j,k,1,SweepFluxFields(m)))
               END FORALL
            END DO
            !          write(*,*) 'updating Info%q', index+i,mB(2:3,1:2), SweepFluxFields, nSweepFlux
            !          write(*,*) fx_%data(fx_%x(i),1,1,1,1)
            IF (nDim >= 2) THEN
               DO i=mB(1,1),mB(1,2)
                  FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                     Info%q(index+i,j,k,SweepFluxFields(m))=Info%q(index+i,j,k,SweepFluxFields(m))+2d0*(fy_%data(fy_%x(i),j,k,1,SweepFluxFields(m)) - &
                          fy_%data(fy_%x(i),j+1,k,1,SweepFluxFields(m)))
                  END FORALL
               END DO
               IF (nDim >= 3) THEN
                  DO i=mB(1,1),mB(1,2)
                     FORALL(j=mB(2,1):mB(2,2),k=mB(3,1):mB(3,2),m=1:nSweepFlux)
                        Info%q(index+i,j,k,SweepFluxFields(m))=Info%q(index+i,j,k,SweepFluxFields(m))+2d0*(fz_%data(fz_%x(i),j,k,1,SweepFluxFields(m)) - &
                             fz_%data(fz_%x(i),j,k+1,1,SweepFluxFields(m)))
                     END FORALL
                  END DO
               END IF
            END IF
         END IF
      END SUBROUTINE update_final_noctu


      !> Calculate upwinded_emf at edge centers using surrounding emf's at face centers and cell centers and transverse velocities at cell centers
      !! @param ez_fy z component of emf at y face centers located at +-x
      !! @param ez_fx z component of emf at x face centers located at +-y
      !! @param ez_c  z component of emf at +-x +-y cell centers
      !! @param vx    x component of velocity at x face centers located at +-y
      !! @param vy    y component of velocity at y face centers located at +-x
      real(Kind=qPrec) pure FUNCTION upwinded_emf(ez_fy, ez_fx, ez_c, vx, vy)
         !ez^c method of Gardiner and Stone
         !ez_fy is the z component of the emf at the y cell faces
         !ez_fx is the z component of the emf at the x cell faces
         !ez_c is a two by two array of the emf at the cell centers
         !vx and vy are the mass flux across the x and y boundaries.
         !REAL(KIND=qPrec) :: upwinded_emf
         REAL(KIND=qPrec), DIMENSION(2), INTENT(IN) :: ez_fy, ez_fx, vx, vy
         REAL(KIND=qPrec), DIMENSION(2,2), INTENT(IN):: ez_c
         REAL(KIND=qPrec), DIMENSION(2) :: dezdx,dezdy

         IF (vy(1) > 0) THEN
            dezdx(1) = ez_fx(1)-ez_c(1,1)
         ELSE IF (vy(1) < 0) THEN
            dezdx(1) = ez_fx(2)-ez_c(1,2)
         ELSE
            dezdx(1) = half*(sum(ez_fx(:))-sum(ez_c(1,:)))
         END IF

         IF (vy(2) > 0) THEN
            dezdx(2) = ez_c(2,1)-ez_fx(1)
         ELSE IF (vy(2) < 0) THEN
            dezdx(2) = ez_c(2,2)-ez_fx(2)
         ELSE
            dezdx(2) = half*(sum(ez_c(2,:))-sum(ez_fx(:)))
         END IF

         IF (vx(1) > 0) THEN
            dezdy(1) = ez_fy(1)-ez_c(1,1)
         ELSE IF (vx(1) < 0) THEN
            dezdy(1) = ez_fy(2)-ez_c(2,1)
         ELSE
            dezdy(1) = half*(sum(ez_fy(:))-sum(ez_c(:,1)))
         END IF

         IF (vx(2) > 0) THEN
            dezdy(2) = ez_c(1,2)-ez_fy(1)
         ELSE IF (vx(2) < 0) THEN
            dezdy(2) = ez_c(2,2)-ez_fy(2)
         ELSE
            dezdy(2) = half*(sum(ez_c(:,2))-sum(ez_fy(:)))
         END IF

         upwinded_emf=fourth*(SUM(ez_fx(1:2))+SUM(ez_fy(1:2)) + (dezdx(1)-dezdx(2))+(dezdy(1)-dezdy(2)))
         !        upwinded_emf=fourth*(SUM(ez_c))
         !      write(*,'(A,10E36.28)') 'dezdx=', dezdx
         !      write(*,'(A,10E36.28)') 'dezdy=', dezdy
         !      write(*,'(A,10E36.28)') 'ez_fx=', ez_fx
         !      write(*,'(A,10E36.28)') 'ez_fy=', ez_fy
         !      write(*,'(A,10E36.28)') 'upwinded_emf=', upwinded_emf
         !      write(*,'(A,10E36.28)') 'ez_c', ez_c
      END FUNCTION upwinded_emf




      SUBROUTINE LoadLevelStencilBuffers(n)
         INTEGER :: n
         CurrentLevelLoaded=n
         IF (w /= 0) w_ => LevelBuffers(n)%StencilBuffer(w)
         IF (qLx /= 0) qLx_ => LevelBuffers(n)%StencilBuffer(qLx)
         IF (qRx /= 0) qRx_ => LevelBuffers(n)%StencilBuffer(qRx)
         IF (qLy /= 0) qLy_ => LevelBuffers(n)%StencilBuffer(qLy)
         IF (qRy /= 0) qRy_ => LevelBuffers(n)%StencilBuffer(qRy)
         IF (qLz /= 0) qLz_ => LevelBuffers(n)%StencilBuffer(qLz)
         IF (qRz /= 0) qRz_ => LevelBuffers(n)%StencilBuffer(qRz)
         IF (fx /= 0) fx_ => LevelBuffers(n)%StencilBuffer(fx)
         IF (fy /= 0) fy_ => LevelBuffers(n)%StencilBuffer(fy)
         IF (fz /= 0) fz_ => LevelBuffers(n)%StencilBuffer(fz)
         IF (q2Lx /= 0) q2Lx_ => LevelBuffers(n)%StencilBuffer(q2Lx)
         IF (q2Rx /= 0) q2Rx_ => LevelBuffers(n)%StencilBuffer(q2Rx)
         IF (q2Ly /= 0) q2Ly_ => LevelBuffers(n)%StencilBuffer(q2Ly)
         IF (q2Ry /= 0) q2Ry_ => LevelBuffers(n)%StencilBuffer(q2Ry)
         IF (q2Lz /= 0) q2Lz_ => LevelBuffers(n)%StencilBuffer(q2Lz)
         IF (q2Rz /= 0) q2Rz_ => LevelBuffers(n)%StencilBuffer(q2Rz)
         IF (f2x /= 0) f2x_ => LevelBuffers(n)%StencilBuffer(f2x)
         IF (f2y /= 0) f2y_ => LevelBuffers(n)%StencilBuffer(f2y)
         IF (f2z /= 0) f2z_ => LevelBuffers(n)%StencilBuffer(f2z)
         IF (limiter_x /= 0) limiter_x_ => LevelBuffers(n)%StencilBuffer(limiter_x)
         IF (limiter_y /= 0) limiter_y_ => LevelBuffers(n)%StencilBuffer(limiter_y)
         IF (limiter_z /= 0) limiter_z_ => LevelBuffers(n)%StencilBuffer(limiter_z)
         IF (limiter_ppm /= 0) limiter_ppm_ => LevelBuffers(n)%StencilBuffer(limiter_ppm)
         IF (pT /= 0) pT_ => LevelBuffers(n)%StencilBuffer(pT)
         IF (qex /= 0) qex_ => LevelBuffers(n)%StencilBuffer(qex)
         IF (qey /= 0) qey_ => LevelBuffers(n)%StencilBuffer(qey)
         IF (qez /= 0) qez_ => LevelBuffers(n)%StencilBuffer(qez)
         IF (dqx /= 0) dqx_ => LevelBuffers(n)%StencilBuffer(dqx)
         IF (dqy /= 0) dqy_ => LevelBuffers(n)%StencilBuffer(dqy)
         IF (dqz /= 0) dqz_ => LevelBuffers(n)%StencilBuffer(dqz)
         IF (ex /= 0) ex_ => LevelBuffers(n)%StencilBuffer(ex)
         IF (ey /= 0) ey_ => LevelBuffers(n)%StencilBuffer(ey)
         IF (ez /= 0) ez_ => LevelBuffers(n)%StencilBuffer(ez)
         IF (ex_bar /= 0) ex_bar_ => LevelBuffers(n)%StencilBuffer(ex_bar)
         IF (ey_bar /= 0) ey_bar_ => LevelBuffers(n)%StencilBuffer(ey_bar)
         IF (ez_bar /= 0) ez_bar_ => LevelBuffers(n)%StencilBuffer(ez_bar)
         IF (e2x /= 0) e2x_ => LevelBuffers(n)%StencilBuffer(e2x)
         IF (e2y /= 0) e2y_ => LevelBuffers(n)%StencilBuffer(e2y)
         IF (e2z /= 0) e2z_ => LevelBuffers(n)%StencilBuffer(e2z)
         IF (e2x_bar /= 0) e2x_bar_ => LevelBuffers(n)%StencilBuffer(e2x_bar)
         IF (e2y_bar /= 0) e2y_bar_ => LevelBuffers(n)%StencilBuffer(e2y_bar)
         IF (e2z_bar /= 0) e2z_bar_ => LevelBuffers(n)%StencilBuffer(e2z_bar)
         IF (ctfy /= 0) ctfy_ => LevelBuffers(n)%StencilBuffer(ctfy)
         IF (ctfz /= 0) ctfz_ => LevelBuffers(n)%StencilBuffer(ctfz)
         IF (ctfx /= 0) ctfx_ => LevelBuffers(n)%StencilBuffer(ctfx)
         IF (ctf2x /= 0) ctf2x_ => LevelBuffers(n)%StencilBuffer(ctf2x)
         IF (ctf2y /= 0) ctf2y_ => LevelBuffers(n)%StencilBuffer(ctf2y)
         IF (ctf2z /= 0) ctf2z_ => LevelBuffers(n)%StencilBuffer(ctf2z)
         IF (w2 /= 0) w2_ => LevelBuffers(n)%StencilBuffer(w2)
         IF (A2x /= 0) A2x_ => LevelBuffers(n)%StencilBuffer(A2x)
         IF (A2y /= 0) A2y_ => LevelBuffers(n)%StencilBuffer(A2y)
         IF (A2z /= 0) A2z_ => LevelBuffers(n)%StencilBuffer(A2z)
         IF (Sx /= 0) Sx_ => LevelBuffers(n)%StencilBuffer(Sx)
         IF (Sy /= 0) Sy_ => LevelBuffers(n)%StencilBuffer(Sy)
         IF (Sz /= 0) Sz_ => LevelBuffers(n)%StencilBuffer(Sz)
         IF (w3 /= 0) w3_ => LevelBuffers(n)%StencilBuffer(w3)
         IF (A3x /= 0) A3x_ => LevelBuffers(n)%StencilBuffer(A3x)
         IF (A3y /= 0) A3y_ => LevelBuffers(n)%StencilBuffer(A3y)
         IF (A3z /= 0) A3z_ => LevelBuffers(n)%StencilBuffer(A3z)
         IF (SpeedsX /= 0) SpeedsX_ => LevelBuffers(n)%StencilBuffer(SpeedsX)
         IF (SpeedsY /= 0) SpeedsY_ => LevelBuffers(n)%StencilBuffer(SpeedsY)
         IF (SpeedsZ /= 0) SpeedsZ_ => LevelBuffers(n)%StencilBuffer(SpeedsZ)
         IF (leftX /= 0) leftX_ => LevelBuffers(n)%StencilBuffer(leftX)
         IF (leftY /= 0) leftY_ => LevelBuffers(n)%StencilBuffer(leftY)
         IF (leftZ /= 0) leftZ_ => LevelBuffers(n)%StencilBuffer(leftZ)
         IF (rightX /= 0) rightX_ => LevelBuffers(n)%StencilBuffer(rightX)
         IF (rightY /= 0) rightY_ => LevelBuffers(n)%StencilBuffer(rightY)
         IF (rightZ /= 0) rightZ_ => LevelBuffers(n)%StencilBuffer(rightZ)
         IF (nWaves /= 0) nWaves_ => LevelBuffers(n)%StencilBuffer(nWaves)
         IF (req_eigens /= 0) req_eigens_ => LevelBuffers(n)%StencilBuffer(req_eigens)
         IF (adfx /= 0) adfx_ => LevelBuffers(n)%StencilBuffer(adfx)
         IF (adfy /= 0) adfy_ => LevelBuffers(n)%StencilBuffer(adfy)
         IF (adfz /= 0) adfz_ => LevelBuffers(n)%StencilBuffer(adfz)
         IF (eta2x /= 0) eta2x_ => LevelBuffers(n)%StencilBuffer(eta2x)
         IF (eta2y /= 0) eta2y_ => LevelBuffers(n)%StencilBuffer(eta2y)
         IF (eta2z /= 0) eta2z_ => LevelBuffers(n)%StencilBuffer(eta2z)
         IF (etax /= 0) etax_ => LevelBuffers(n)%StencilBuffer(etax)
         IF (etay /= 0) etay_ => LevelBuffers(n)%StencilBuffer(etay)
         IF (etaz /= 0) etaz_ => LevelBuffers(n)%StencilBuffer(etaz)
         IF (q /= 0) q_ => LevelBuffers(n)%StencilBuffer(q)
         IF (aux /= 0) aux_ => LevelBuffers(n)%StencilBuffer(aux)
         IF (recon /= 0) recon_ => LevelBuffers(n)%StencilBuffer(recon)
         IF (beforesweepstep /= 0) beforesweepstep_ => LevelBuffers(n)%StencilBuffer(beforesweepstep)
         IF (aftersweepstep /= 0) aftersweepstep_ => LevelBuffers(n)%StencilBuffer(aftersweepstep)
         IF (source /= 0) source_ => LevelBuffers(n)%StencilBuffer(source)
         IF (source2 /= 0) source2_ => LevelBuffers(n)%StencilBuffer(source2)
         IF (cornerdiv /= 0) cornerdiv_ => LevelBuffers(n)%StencilBuffer(cornerdiv)
         IF (gradphix /= 0) gradphix_ => LevelBuffers(n)%StencilBuffer(gradphix)
         IF (gradphiy /= 0) gradphiy_ => LevelBuffers(n)%StencilBuffer(gradphiy)
         IF (gradphiz /= 0) gradphiz_ => LevelBuffers(n)%StencilBuffer(gradphiz)
         !    write(*,*) 'loaded level stencil buffers for level', n
      END SUBROUTINE LoadLevelStencilBuffers

   END SUBROUTINE sweepAdvance



  !> Transform an array of cells from conservative to primitive form
  !! @param q array of cells
   SUBROUTINE cons_to_prim_1(q)
      REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(INOUT) :: q
      INTEGER :: j,k
      REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: v
      IF (lMHD) THEN
         ALLOCATE(v(3))
         DO j=lbound(q,1),ubound(q,1)
            DO k=lbound(q,2),ubound(q,2)
               v(1:3)=q(j,k,ivx:ivz)/q(j,k,1)
               IF (iE .ne. 0) q(j,k,iE)=gamma1*(q(j,k,iE)-half*(DOT_PRODUCT(q(j,k,ivx:ivz),v(1:3)) + &
                    DOT_PRODUCT(q(j,k,iBx:iBz),q(j,k,iBx:iBz))))
               q(j,k,ivx:ivz)=v(1:3)
            END DO
         END DO
      ELSE
         ALLOCATE(v(1:m_high-m_low+1))
         DO j=lbound(q,1),ubound(q,1)
            DO k=lbound(q,2),ubound(q,2)
               v=q(j,k,m_low:m_high)/q(j,k,1)
               IF (iE .ne. 0) q(j,k,iE)=gamma1*(q(j,k,iE)-half*(DOT_PRODUCT(q(j,k,m_low:m_high),v)))
               q(j,k,m_low:m_high)=v
            END DO
         END DO
      END IF
      DEALLOCATE(v)
   END SUBROUTINE cons_to_prim_1


  !> Transform an array of cells from conservative to primitive form
  !! @param q array of cells
  SUBROUTINE cons_to_prim_2(q,w)
     REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(IN) :: q
     REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(OUT) :: w
     INTEGER :: j,k
     w=q
     IF (lMHD) THEN
        DO j=lbound(q,1),ubound(q,1)
           DO k=lbound(q,2),ubound(q,2)
              w(j,k,m_low:ivz)=q(j,k,m_low:ivz)/q(j,k,1)
              IF (iE .ne. 0) w(j,k,iE)=gamma1*(q(j,k,iE)-half*(DOT_PRODUCT(q(j,k,m_low:ivz),w(j,k,m_low:ivz)) + &
                   DOT_PRODUCT(q(j,k,iBx:iBz),q(j,k,iBx:iBz))))
           END DO
        END DO
     ELSE
        DO j=lbound(q,1),ubound(q,1)
           DO k=lbound(q,2),ubound(q,2)
              w(j,k,m_low:m_high)=q(j,k,m_low:m_high)/q(j,k,1)
              IF (iE .ne. 0) w(j,k,iE)=gamma1*(q(j,k,iE)-half*(DOT_PRODUCT(q(j,k,m_low:m_high),w(j,k,m_low:m_high))))
           END DO
        END DO
     END IF
  END SUBROUTINE cons_to_prim_2


  !> Transform an array of cells from conservative to primitive form
  !! @param q array of cells
  SUBROUTINE prim_to_cons_1(q)
     REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(INOUT) :: q
     INTEGER :: j,k
     REAL(KIND=qPrec), DIMENSION(:), ALLOCATABLE :: v
     IF (lMHD) THEN
        ALLOCATE(v(3))
        DO j=lbound(q,1),ubound(q,1)
           DO k=lbound(q,2),ubound(q,2)
              v(1:3)=q(j,k,ivx:ivz)*q(j,k,1)
              IF (iE .ne. 0) q(j,k,iE)=gamma7*q(j,k,iE)+half*(DOT_PRODUCT(q(j,k,ivx:ivz),v(1:3)) + &
                   DOT_PRODUCT(q(j,k,iBx:iBz),q(j,k,iBx:iBz)))
              q(j,k,ivx:ivz)=v(1:3)
           END DO
        END DO
     ELSE
        ALLOCATE(v(1:m_high-m_low+1))
        DO j=lbound(q,1),ubound(q,1)
           DO k=lbound(q,2),ubound(q,2)
              v=q(j,k,m_low:m_high)*q(j,k,1)
              IF (iE .ne. 0) q(j,k,iE)=gamma7*q(j,k,iE)+half*(DOT_PRODUCT(q(j,k,m_low:m_high),v))
              q(j,k,m_low:m_high)=v
           END DO
        END DO
     END IF
     DEALLOCATE(v)
  END SUBROUTINE prim_to_cons_1


  !> Transform an array of cells from primitive to conservative form
  !! @param w array of cells in primitive form
  !! @param q output array of cells in conservative form
  SUBROUTINE prim_to_cons_2(w,q)
    REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(IN) :: w
    REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(OUT) :: q
    INTEGER :: j,k
    q=w
    IF (lMHD) THEN
       DO j=lbound(q,1),ubound(q,1)
          DO k=lbound(q,2),ubound(q,2)
             q(j,k,m_low:ivz)=w(j,k,m_low:ivz)*w(j,k,1)
             IF (iE .ne. 0) q(j,k,iE)=gamma7*w(j,k,iE)+half*(DOT_PRODUCT(q(j,k,m_low:ivz),w(j,k,m_low:ivz)) + &
                  DOT_PRODUCT(w(j,k,iBx:iBz),w(j,k,iBx:iBz)))
          END DO
       END DO
    ELSE
       DO j=lbound(q,1),ubound(q,1)
          DO k=lbound(q,2),ubound(q,2)
             q(j,k,m_low:m_high)=w(j,k,m_low:m_high)*w(j,k,1)
             IF (iE .ne. 0) q(j,k,iE)=gamma7*w(j,k,iE)+half*(DOT_PRODUCT(q(j,k,m_low:m_high),w(j,k,m_low:m_high)))
          END DO
       END DO
    END IF
  END SUBROUTINE prim_to_cons_2

   !> Calculate fluxes in the x-direction
   !! @param left left interface state
   !! @param right right inteface state
   !! @param flux resulting flux
   !! @param lambda_max optional parameter for H-viscosity
   FUNCTION calc_flux_x(left,right,flux, lambda_max)
      REAL(KIND=qPREC) :: calc_flux_x
      REAL(KIND=qPREC), DIMENSION(:) :: left, right
      REAL(KIND=qPREC), DIMENSION(:), INTENT(INOUT) :: flux
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: tempflux
      REAL(KIND=qPREC), OPTIONAL :: lambda_max
      ALLOCATE(tempflux(size(flux,1)))
      IF (present(lambda_max)) THEN
         calc_flux_x=calc_flux(left(wDx_i),right(wDx_i),tempflux, lambda_max)
      ELSE
         calc_flux_x=calc_flux(left(wDx_i),right(wDx_i),tempflux)
      END IF
      flux(:)=tempflux(fDx_i)
      DEALLOCATE(tempflux)
   END FUNCTION calc_flux_x

   !> Calculate fluxes in the y-direction
   !! @param left left interface state
   !! @param right right inteface state
   !! @param flux resulting flux
   !! @param lambda_max optional parameter for H-viscosity
   FUNCTION calc_flux_y(left,right,flux,lambda_max)
      REAL(KIND=qPREC) :: calc_flux_y
      REAL(KIND=qPREC), DIMENSION(:), INTENT(IN) :: left, right
      REAL(KIND=qPREC), DIMENSION(:), INTENT(OUT) :: flux
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: tempflux
      REAL(KIND=qPREC), OPTIONAL :: lambda_max
      ALLOCATE(tempflux(size(flux,1)))
      IF (present(lambda_max)) THEN
         calc_flux_y= calc_flux(left(wDy_i),right(wDy_i),tempflux,lambda_max)
      ELSE
         !print*,'18mar2011';stop
         calc_flux_y= calc_flux(left(wDy_i),right(wDy_i),tempflux)
      END IF
      flux(:)=tempflux(fDy_i)
      DEALLOCATE(tempflux)
   END FUNCTION calc_flux_y


   !> Calculate fluxes in the z-direction
   !! @param left left interface state
   !! @param right right inteface state
   !! @param flux resulting flux
   !! @param lambda_max optional parameter for H-viscosity
   FUNCTION calc_flux_z(left,right,flux,lambda_max)
      REAL(KIND=qPREC) :: calc_flux_z
      REAL(KIND=qPREC), DIMENSION(:), INTENT(IN) :: left, right
      REAL(KIND=qPREC), DIMENSION(:), INTENT(OUT) :: flux
      REAL(KIND=qPREC), DIMENSION(:), ALLOCATABLE :: tempflux
      REAL(KIND=qPREC), OPTIONAL :: lambda_max
      ALLOCATE(tempflux(size(flux,1)))
      IF (present(lambda_max)) THEN
         calc_flux_z= calc_flux(left(wDz_i),right(wDz_i),tempflux,lambda_max)
      ELSE
         calc_flux_z= calc_flux(left(wDz_i),right(wDz_i),tempflux)
      END IF
      flux(:)=tempflux(fDz_i)
      DEALLOCATE(tempflux)
   END FUNCTION calc_flux_z


   !> Minmod function
   !! @param x slope 1
   !! @param y slope 2
   pure REAL(KIND=qPREC) function minmod(x,y)
      REAL(KIND=qPREC), INTENT(IN) :: x,y
      if ((x <= 0 .AND. y >= 0) .OR. (x >= 0 .AND. y <= 0)) THEN
         minmod=0
      else
         minmod=sign(min(abs(x),abs(y)),x)
      end if
   end function minmod

   !> Sweeps version of protection of primitive variables
   !! @param w primitive variables (iE is pressure)
   SUBROUTINE protect(w)
     REAL(KIND=qPrec), DIMENSION(:), INTENT(INOUT) :: w
     REAL(KIND=qPrec) :: my_TolDens, BE
     IF (lMHD) THEN
        BE=half*(DOT_PRODUCT(w(iBx:iBz),w(iBx:iBz)))
     ELSE
        BE=0d0
     END IF
     my_TolDens=MinDensity
     IF (w(1) < my_TolDens) THEN
        w(1)=my_TolDens
        IF (.NOT. lIsothermal) w(iE)=Iso_Speed2*w(1)
        w(m_low:m_high)=0d0
     ELSE IF (.NOT. lIsothermal) THEN
        w(iE)=MAX(w(iE),Iso_Speed2*w(1))     
     END IF
   END SUBROUTINE protect

   !> Protects a slab of cells
   !! @param w slab of cells in primitive form
   SUBROUTINE protect_all(w)
      REAL(KIND=qPrec), DIMENSION(:,:,:), INTENT(INOUT) :: w
      REAL(KIND=qPrec) :: BE, my_TolDens
      INTEGER :: j,k
      DO j=lbound(w,1), ubound(w,1)
         DO k=lbound(w,2), ubound(w,2)
            !          IF (lMHD) THEN
            !             BE=half*(DOT_PRODUCT(w(j,k,iBx:iBz),w(j,k,iBx:iBz)))
            !          ELSE           
            !             BE=0d0
            !          END IF
            my_TolDens=MinDensity
            IF (w(j,k,1) < my_TolDens) THEN
               w(j,k,1)=my_TolDens
               w(j,k,m_low:m_high)=0d0
               IF (iE .ne. 0) w(j,k,iE)=Iso_Speed2*w(j,k,1)
            ELSE IF (.NOT. lIsothermal) THEN
               w(j,k,iE)=MAX(w(j,k,iE), Iso_Speed2*w(j,k,1))
            END IF
         END DO
      END DO
   END SUBROUTINE protect_all


   !> Returns a random array
   !! @param q 3D array to fill with random values
   SUBROUTINE Randomize(q)
      REAL(KIND=qPREC), DIMENSION(:,:,:) :: q
      INTEGER :: i,j,k
      REAL(KIND=qPREC) :: rand
      DO i=1,size(q,1)
         DO j=1,size(q,2)
            DO k=1,size(q,3)
               CALL Random_number(rand)
               q(i,j,k)=.001*real(i) !q(i,j,k)*(1d0+(.001)*(REAL(i)))!rand*1e-2)
               !               q(i,j,1)=q(i,j,1)*(1d0+.01*REAL(level)/(levels(level)%dt/levels(level)%dx)) !(.001)*(REAL(i+j+k)))!rand*1e-2)
            END DO
         END DO
      END DO
   END SUBROUTINE Randomize


   !  INCLUDE 'sweep_scheme_.f90'
   !  INCLUDE 'i_dependencies.f90'

END MODULE SweepScheme

