!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    chombo_declarations.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file chombo_declarations.f90
!! @brief Main file for module ChomboDeclarations

!> @defgroup ChomboDeclarations Chombo Declarations
!! @ingroup IOChomboGroup
!! @brief Declarations file for the Chombo output format.

!> Declarations file for the Chombo output format.
!! @author Brandon D. Shroyer
!! @date 10-12-2010
!! @ingroup ChomboDeclarations
MODULE ChomboDeclarations

    USE HDF5
    USE GlobalDeclarations
    USE HDF5Declarations
    USE IODeclarations
    IMPLICIT NONE
    SAVE
    PUBLIC

    LOGICAL, PARAMETER :: CHOMBO_HANDLE_READ = .FALSE.
    LOGICAL, PARAMETER :: CHOMBO_HANDLE_WRITE = .TRUE.

    INTEGER, PARAMETER :: COOLING_SOURCE_WRITE = 0
    INTEGER, PARAMETER :: CYLINDRICAL_SOURCE_WRITE = 1

    INTEGER, PARAMETER :: COOLING_SOURCE_READ = 2
    INTEGER, PARAMETER :: CYLINDRICAL_SOURCE_READ = 3

    INTEGER, PARAMETER :: CHOMBO_ROOT_LEVEL = -2
    INTEGER, PARAMETER :: CHOMBO_DOMAIN_LEVEL = -1

    !> An object containing information used to manage an hdf5 chombo file
    !! @details This structure contains pointers used to pass data between subroutines
    !! in a chombo restart so care should be taken to deallocate it properly
    TYPE ChomboHandle

        INTEGER(HID_T) :: file_handle
        INTEGER(HID_T) :: root_group_id
        INTEGER(HID_T) :: particle_group_id
        INTEGER(HID_T) :: source_group_id
        INTEGER(HID_T) :: level_id
        INTEGER(HID_T) :: child_level_id

        INTEGER :: finest_level
        INTEGER, POINTER, DIMENSION(:) :: next_level_cost

        LOGICAL :: lWrite

        INTEGER(HSIZE_T) :: box_offset
        INTEGER(HSIZE_T) :: childbox_offset
        INTEGER(HSIZE_T) :: childbox_count_offset
        INTEGER(HSIZE_T) :: domainbox_offset
        INTEGER(HSIZE_T) :: q_offset
        INTEGER(HSIZE_T) :: aux_offset
        INTEGER(HSIZE_T) :: costmap_offset
        INTEGER(HSIZE_T) :: particle_offset
        INTEGER(HSIZE_T) :: source_offset

        TYPE(NodeBoxList), POINTER :: box_list
        TYPE(NodeBoxList), POINTER :: last_box
        TYPE(NodeBoxList), POINTER :: current_box

        TYPE(NodeBoxList), POINTER :: child_box_list
        TYPE(NodeBoxList), POINTER :: last_child_box
        TYPE(NodeBoxList), POINTER :: current_child_box

    END TYPE ChomboHandle

    TYPE(ChomboHandle), POINTER :: curr_handle
CONTAINS

    !> Allocates a new Chombo handle object and opens a file with it.
    !! @param filename The name of the file being opened.
    !! @param chandle A pointer to a chombo handle object.
    !! @param lWrite A logical flag that when true opens a file for write access
    SUBROUTINE CreateChomboHandle(filename, chandle, lWrite)

        CHARACTER(LEN=*) :: filename
        TYPE(ChomboHandle), POINTER :: chandle
        LOGICAL :: lWrite

        INTEGER :: i_err
        INTEGER(HID_T) :: hid_property_id


        ALLOCATE(chandle)

        chandle%box_offset = 0
        chandle%childbox_offset = 0
        chandle%childbox_count_offset = 0
        chandle%q_offset = 0
        chandle%aux_offset = 0
        chandle%costmap_offset = 0
        chandle%particle_offset = 0
        chandle%source_offset = 0
        chandle%lWrite = lWrite

        NULLIFY(chandle%box_list, chandle%last_box, chandle%current_box, chandle%next_level_cost)
        NULLIFY(chandle%child_box_list, chandle%last_child_box, chandle%current_child_box)

        ! Initializes the HDF5 library.  This function is safe to call
        ! even if it has already been called.
        CALL h5open_f(i_err)
        CALL CatchHDF5Error("h5open_f", i_err)

        IF (lWrite) THEN

          ! Create property list
          CALL h5pcreate_f(H5P_FILE_ACCESS_F, hid_property_id, i_err)
          CALL CatchHDF5Error("h5pcreate_f", i_err)

          ! Create chombo file.
          CALL h5fcreate_f(filename, H5F_ACC_TRUNC_F, chandle%file_handle, i_err, access_prp = hid_property_id)
          IF (i_err /= 0) THEN
             PRINT*, ''
             PRINT*, ''
             PRINT*, '******************************************************************************'
             PRINT*, '******************************************************************************'
             PRINT*, '**************       Unable to open file for writing.        *****************'
             PRINT*, '**************  Perhaps you need to create an out directory? *****************'
             PRINT*, '******************************************************************************'
             PRINT*, '******************************************************************************'
             
             STOP
          END IF

          CALL CatchHDF5Error("h5fcreate_f", i_err)

          ! Close property list, now that it's purpose (file creation) has been fulfilled.
          CALL h5pclose_f(hid_property_id, i_err)
          CALL CatchHDF5Error("h5pclose_f", i_err)

          ! Allocate space for the next level cost buffers.
          ALLOCATE(chandle%next_level_cost(0:MPI_np-1))
          chandle%next_level_cost = 0

        ELSE

            ! Create read-only chombo file.
            CALL h5fopen_f(TRIM(filename), H5F_ACC_RDONLY_F, chandle%file_handle, i_err)
            CALL CatchHDF5Error("h5fopen_f", i_err)

       END IF

       ! Open the root group and save the handle.
       CALL h5gopen_f(chandle%file_handle, "/", chandle%root_group_id, i_err)
       CALL CatchHDF5Error("h5gopen_f('/')", i_err)

       ! Retrieve the finest level written to the file.
       IF (.NOT. lWrite)  chandle%finest_level = Read_HDF5_Attribute_Int("max_level", chandle%root_group_id)

    END SUBROUTINE CreateChomboHandle

    !> Resets all offsets in the given handle to 0.
    !! @param chandle An active chombo handle.
    SUBROUTINE ClearChomboHandleOffsets(chandle)

        TYPE(ChomboHandle), POINTER :: chandle

        chandle%box_offset = 0
        chandle%childbox_offset = 0
        chandle%childbox_count_offset = 0
        chandle%q_offset = 0
        chandle%aux_offset = 0
        chandle%costmap_offset = 0

    END SUBROUTINE ClearChomboHandleOffsets

    !> Deallocates and nullifies a Chombo handle.  This process includes closing the root group and file.
    !! @param chandle A pointer to a chombo handle object that will be destroyed.
    SUBROUTINE CloseChomboHandle(chandle)

        TYPE(ChomboHandle), POINTER :: chandle

        INTEGER :: i_err


        IF (ASSOCIATED(chandle)) THEN

            CALL h5gclose_f(chandle%root_group_id, i_err)
            CALL CatchHDF5Error("h5gclose_f('/')", i_err)
           
            CALL h5fclose_f(chandle%file_handle, i_err)
            CALL CatchHDF5Error("h5fclose_f", i_err)

            CALL IO_ClearAllNodeBoxes(chandle)
            CALL IO_ClearAllChildBoxes(chandle)

            IF (ASSOCIATED(chandle%next_level_cost)) THEN
                DEALLOCATE(chandle%next_level_cost)
                NULLIFY(chandle%next_level_cost)
            END IF

            DEALLOCATE(chandle)
            NULLIFY(chandle)
        ELSE
            PRINT *, "CloseChomboHandleError:  handle not associated."
            STOP
        END IF

    END SUBROUTINE CloseChomboHandle

    !> Takes a level and returns a Chombo-style level_group name
    !! @param level An non-negative integer representing a level.
    !! @param level_name A character string to hold the level name.
    SUBROUTINE IO_GetLevelName(level, level_name)

        INTEGER :: level
        CHARACTER(LEN=*) :: level_name
        CHARACTER(8) :: digit_string


        ! Convert the level integer into a character string.  This assumes that our code 
        ! will be using less than 100 additional levels of refinement, but that seems
        ! FAIRLY reasonable.
        IF (level >= 0 .AND. level < 100) THEN
           WRITE(level_name, '(A6,I1)') 'level_', level
        ELSE
           IF (level < 1) THEN
               WRITE(level_name, '(A12)') 'domain_level'
           ELSE
               WRITE(level_name, '(A6,I2)') 'level_', level
           END IF
        END IF

    END SUBROUTINE IO_GetLevelName

    !> Opens a level group handle and associates it with the level ID in the Chombo handle.
    !! @param chandle A ChomboHandle pointer.
    !! @param level The level of the group to be opened.
    SUBROUTINE IO_OpenCurrentLevel(chandle, level)

        TYPE(ChomboHandle), POINTER :: chandle
        INTEGER :: level

        CHARACTER(I_MAX_LEVELNAME_LENGTH) :: level_name
        INTEGER :: i_err


        ! Should fail on an invalid Chombo level.
        IF (level >= -1 .AND. level <= MaxLevel) THEN

           ! Obtain the name of the level group.
           CALL IO_GetLevelName(level, level_name)

           ! If this is a read option or a non-root level write option, then the current level
           ! has already been created, and you just open it.  Otherwise, you create a new group.
           IF (.NOT. chandle%lWrite) THEN
           ! Open the level group and save the handle.
              CALL h5gopen_f(chandle%root_group_id, TRIM(level_name), chandle%level_id, i_err)       
              CALL CatchHDF5Error("IO_OpenCurrentLevel::h5gopen_f('level_*')", i_err)    
           ELSE
               ! Create level group.
               CALL h5gcreate_f(chandle%root_group_id, TRIM(level_name), chandle%level_id, i_err)
               CALL CatchHDF5Error("IO_OpenCurrentLevel::h5gcreate_f('level_*')", i_err)

           END IF

           CALL ClearChomboHandleOffsets(chandle)

        ELSE
           PRINT "('IO_OpenCurrentLevel() error: invalid level ', i8, '.')", level
           STOP
        END IF

    END SUBROUTINE IO_OpenCurrentLevel

    !> Closes the level group associated with a chombo file.
    !! @param chandle A ChomboHandle pointer.
    SUBROUTINE IO_CloseCurrentLevel(chandle)

        TYPE(ChomboHandle), POINTER :: chandle
        INTEGER :: i_err

        ! Close the level group handle associated with this chombo handle.
        CALL h5gclose_f(chandle%level_id, i_err)       
        CALL CatchHDF5Error("IO_CloseCurrentLevel::h5gclose_f()", i_err)    

    END SUBROUTINE IO_CloseCurrentLevel

   !> Creates a new NodeBox object within the chombo handle's box list array.
   !! @param chandle An active chombo handle object.
   !! @param box_array A 3x2 grid coordinate array.
   !! @param proc_id An MPI processor rank.
   SUBROUTINE IO_CacheNodeBox(chandle, box_array, proc_id)
       
       TYPE(ChomboHandle), POINTER :: chandle
       INTEGER, DIMENSION(3,2) :: box_array
       INTEGER :: proc_id

       CALL AddNodeBoxToList(chandle%last_box, chandle%box_list, chandle%current_box)

!       CALL CreateNodeBox(box_array, chandle%last_box%self, proc_id)
       chandle%last_box%self%MPI_id = proc_id
       chandle%last_box%self%mGlobal = box_array

   END SUBROUTINE IO_CacheNodeBox

   !> Creates a new NodeBox object within the chombo handle's child box list array.
   !! @param chandle An active chombo handle object.
   !! @param box_array A 3x2 grid coordinate array.
   !! @param proc_id An MPI processor rank.
   SUBROUTINE IO_CacheChildNodeBox(chandle, box_array, proc_id)
       
       TYPE(ChomboHandle), POINTER :: chandle
       INTEGER, DIMENSION(3,2) :: box_array
       INTEGER :: proc_id

       CALL AddNodeBoxToList(chandle%last_child_box, chandle%child_box_list, chandle%current_child_box)

!       CALL CreateNodeBox(box_array, chandle%last_child_box%self, proc_id)
       chandle%last_child_box%self%MPI_id = proc_id
       chandle%last_child_box%self%mGlobal = box_array

   END SUBROUTINE IO_CacheChildNodeBox

   !> Deletes all node boxes associated with a chombo handle.
   !! @param chandle An active chombo handle.
   SUBROUTINE IO_ClearAllNodeBoxes(chandle)

      TYPE(ChomboHandle), POINTER :: chandle


      NULLIFY(chandle%last_box)
      NULLIFY(chandle%current_box)
      IF (ASSOCIATED(chandle%box_list))  CALL ClearNodeBoxList(chandle%box_list)
      NULLIFY(chandle%box_list)
!      chandle%box_offset = 0

   END SUBROUTINE IO_ClearAllNodeBoxes

   !> Deletes all child node boxes associated with a chombo handle.
   !! @param chandle An active chombo handle.
   SUBROUTINE IO_ClearAllChildBoxes(chandle)

      TYPE(ChomboHandle), POINTER :: chandle


      NULLIFY(chandle%last_child_box)
      NULLIFY(chandle%current_child_box)
      IF (ASSOCIATED(chandle%child_box_list))  CALL ClearNodeBoxList(chandle%child_box_list)
      NULLIFY(chandle%child_box_list)
!      chandle%childbox_offset = 0

   END SUBROUTINE IO_ClearAllChildBoxes

!> Extract a grid box from an active chombo file.
!! @param chandle An active chombo handle.
!! @param box_set_name String describing box set
!! @param box A 3x2 integer array to be filled with a grid box.
SUBROUTINE IO_GetBoxFromChomboFile(chandle, box_set_name, box)

    TYPE(ChomboHandle), POINTER :: chandle
    CHARACTER(LEN=*) :: box_set_name
    INTEGER, DIMENSION(3,2) :: box

    INTEGER, DIMENSION(6) :: box_data
    INTEGER(HID_T) :: source_group_id


    ! The only major box dataset that is not associated with some kind of level ID
    ! (including the ones in domain_level) is "root_child_boxes", which holds the children
    ! of the level -2 node.
    IF (box_set_name /= "root_child_boxes") THEN
        source_group_id = chandle%level_id
    ELSE
        source_group_id = chandle%root_group_id
    END IF

    ! Read box data in from chombo file and increment the appropriate offset.
    IF (box_set_name == "boxes") THEN
        CALL Read_Slab_From_Dataset_Box(box_set_name, source_group_id, box_data, chandle%box_offset)
        chandle%box_offset = chandle%box_offset + 1
    ELSE
        CALL Read_Slab_From_Dataset_Box(box_set_name, source_group_id, box_data, chandle%childbox_offset)
        chandle%childbox_offset = chandle%childbox_offset + 1
    END IF

    ! Reorder box data to match output array.
    box(1:3,1) = box_data(1:3) + 1
    box(1:3,2) = box_data(4:6) + 1

END SUBROUTINE IO_GetBoxFromChomboFile

!> Extract the cell-centered data (the q data) for a grid from a chombo file.
!! @param chandle An active chombo handle.
!! @param mGlobal The boundaries of the grid whose data we are retrieving.
!! @param qdata The 4D data array we will initialize.
SUBROUTINE IO_GetQDataFromChomboFile(chandle, mGlobal, qdata)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER, DIMENSION(3,2) :: mGlobal
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: qdata

    REAL(KIND=qPrec), DIMENSION(:), POINTER :: data_array
    INTEGER :: i,j,k,nvar
    INTEGER :: mx,my,mz
    INTEGER :: i_index

    
    ! Initialize array bounds.
    mx = mGlobal(1,2) - mGlobal(1,1) + 1
    my = mGlobal(2,2) - mGlobal(2,1) + 1
    mz = mGlobal(3,2) - mGlobal(3,1) + 1

    i_index = 0
    ! Allocate a 1D data array for chombo.
    ALLOCATE(data_array(mx * my * mz * (NrInputVars)), qdata(mx, my, mz, NrVars))
!    ALLOCATE(data_array(mx * my * mz * (NrVars+NrDiagnosticVars)), qdata(mx, my, mz, NrVars))

    ! Reads cell-centered data in from chombo file.
    CALL Read_Slab_From_Dataset_Double("data:datatype=0", chandle%level_id, data_array, chandle%q_offset)

    ! Increments chombo handle's cell-centered data offset.
    chandle%q_offset = chandle%q_offset + SIZE(data_array)

    ! Move data from data array into grid.
    DO nvar = 1, NrVars
       DO k = 1, mz
          DO j = 1, my
             DO i = 1, mx
                i_index = i_index + 1
                qdata(i, j, k, nvar) = data_array(i_index)
             END DO
          END DO
       END DO
    END DO

    DEALLOCATE(data_array)
    NULLIFY(data_array)

END SUBROUTINE IO_GetQDataFromChomboFile

!> Extract the face-centered data (the aux data) for a grid from a chombo file.
!! @param chandle An active chombo handle.
!! @param mGlobal The boundaries of the grid whose data we are retrieving.
!! @param auxdata The 4D data array we will initialize.
SUBROUTINE IO_GetAuxDataFromChomboFile(chandle, mGlobal, auxdata)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER, DIMENSION(3,2) :: mGlobal
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: auxdata

    REAL(KIND=qPrec), DIMENSION(:), POINTER :: data_array
    INTEGER :: i,j,k,n
    INTEGER :: mx,my,mz,nvar
    INTEGER, DIMENSION(3) :: extension
    INTEGER :: i_index

    
    ! Initialize array bounds.
    mx = mGlobal(1,2) - mGlobal(1,1) + 1
    my = mGlobal(2,2) - mGlobal(2,1) + 1
    mz = mGlobal(3,2) - mGlobal(3,1) + 1

    i_index = 0

    ! Allocate a 1D data array for chombo.
    IF (nDim == 3) THEN
       ALLOCATE(data_array((mx + 1) * my * mz + mx * (my + 1) * mz + mx * my * (mz + 1)))
       ALLOCATE(auxdata(mx+1, my+1, mz+1, 3))

    ELSE
       ALLOCATE(data_array((mx + 1) * my + mx * (my + 1)))
       ALLOCATE(auxdata(mx+1, my+1, mz, 2))

    END IF

    ! Reads cell-centered data in from chombo file.
    CALL Read_Slab_From_Dataset_Double("MHD_data", chandle%level_id, data_array, chandle%aux_offset)

    ! Increments chombo handle's cell-centered data offset.
    chandle%aux_offset = chandle%aux_offset + SIZE(data_array)

    ! Zero out the aux array.  This is important, because while the whole aux array isn't used
    ! in calculations, not zeroing out the unused sections can introduce divergence.
    auxdata = 0.d0

    DO n = 1, nDim

        extension = (/ mx, my, mz /)
        extension(n) = extension(n) + 1

       ! Move data from data array into grid.
       DO k = 1, extension(3)
          DO j = 1, extension(2)
             DO i = 1, extension(1)
                i_index = i_index + 1
                auxdata(i, j, k, n) = data_array(i_index)
             END DO
          END DO
       END DO

    END DO

    DEALLOCATE(data_array)
    NULLIFY(data_array)

END SUBROUTINE IO_GetAuxDataFromChomboFile

!> Extract cost map for a grid from a chombo file.
!! @param chandle An active chombo handle.
!! @param mGlobal The boundaries of the grid whose data we are retrieving.
!! @param level The level of the data we are retrieving.
!! @param costmap The 4D data array we will initialize.
SUBROUTINE IO_GetCostMapFromChomboFile(chandle, mGlobal, level, costmap)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER, DIMENSION(3,2) :: mGlobal
    INTEGER :: level
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: costmap

   REAL(KIND=qPrec), DIMENSION(:), POINTER :: data_array
   INTEGER :: mx, my, mz


    ! Initialize array bounds.
    mx = mGlobal(1,2) - mGlobal(1,1) + 1
    my = mGlobal(2,2) - mGlobal(2,1) + 1
    mz = mGlobal(3,2) - mGlobal(3,1) + 1

   ALLOCATE(data_array(mx * my * mz * 2), costmap(mx, my, mz, 2))

   ! Extract costmap from the chombo file.  The root level does not have a level
   ! group, so it just uses the root group ID.  The domain level doesn't have an official
   ! level group, but the domain_level handle is managed by OpenCurrentLevel() and is
   ! associated with the level ID, so it's all good.
   IF (level >= -1) THEN
       CALL Read_Slab_From_Dataset_Double("costmap", chandle%level_id, data_array, &
                                                chandle%costmap_offset)
   ELSE
       CALL Read_Slab_From_Dataset_Double("root_costmap", chandle%root_group_id, &
                                                data_array, chandle%costmap_offset)
   END IF

   ! Reshape the 1D array into a 4D array and store it in costmap.
   costmap(1:mx,1:my,1:mz,1:2) = RESHAPE(data_array, (/ mx, my, mz, 2 /))

   ! Increment the offset by the amount of data we have extracted from the file.
   chandle%costmap_offset = chandle%costmap_offset + mx*my*mz*2

   ! Deallocate the chombo buffer once it has served its purpose.
   DEALLOCATE(data_array)
   NULLIFY(data_array)

END SUBROUTINE IO_GetCostMapFromChomboFile

!> Extract a single child-count (integer element) for a grid from a chombo file.
!! @param chandle An active chombo handle.
!! @param child_count An integer to hold the child count.
SUBROUTINE IO_GetChildBoxCountFromChomboFile(chandle, child_count)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: child_count

    INTEGER, DIMENSION(1):: childcount_array


    ! Reads cell-centered data in from chombo file.
    CALL Read_Slab_From_Dataset_Int("childbox_count", chandle%level_id, childcount_array, &
                                    chandle%childbox_count_offset)

    child_count = childcount_array(1)

    ! Increments chombo handle's cell-centered data offset.
    chandle%childbox_count_offset = chandle%childbox_count_offset + 1

END SUBROUTINE IO_GetChildBoxCountFromChomboFile

!> Assessing child_proclists and proctimes and mpi_id's given the 
SUBROUTINE ChomboReCreateChildren(node, level, nChildren, chombo_childboxes)

    USE DataDeclarations

    TYPE(Nodedef), POINTER :: node
    INTEGER :: level
    INTEGER :: nChildren
    TYPE(NodeBoxList), POINTER :: chombo_childboxes

    TYPE(Nodedef), POINTER :: child
    TYPE(NodedefList), POINTER :: childlist
    TYPE(NodeBoxList), POINTER :: nodebox_list
    TYPE(NodeBox), POINTER :: box
    INTEGER, POINTER, DIMENSION(:,:,:) :: childgrids
    TYPE(NodeBox), POINTER :: child_box
    INTEGER :: iErr,j,i,k
    INTEGER :: nprocs = 0
    INTEGER :: n
    REAL, DIMENSION(:,:), POINTER :: childproctimes
    INTEGER, DIMENSION(:), POINTER :: childprocs


    IF (nChildren > 0) THEN

        ALLOCATE(childgrids(3,2,nChildren))
        nodebox_list => chombo_childboxes
        DO n=1,nChildren
           childgrids(1:3,1:2,n)= nodebox_list%self%mGlobal 
           nodebox_list => nodebox_list%next
        END DO
        ! Zeroes out EMF and fixup arrays.
        IF (MPI_NP == 1 .AND. level >= 0) CALL AllocChildFixups(node%info,childgrids)
        ! Deallocate childgrids once they are no longer needed.
        DEALLOCATE(childgrids)
        NULLIFY(childgrids)


        ! Repoint the nodebox iterator to the beginning of the handle's child list.
        nodebox_list => chombo_childboxes

        DO j=1, nChildren
           box => nodebox_list%self
           NULLIFY(child)
           CALL CreateNodeBox(box%mGlobal, child_box, MPI_ID) !node%proclist(childprocs(j)))
           IF (MPI_NP == 1) THEN
              CALL AddNode(level+1,child_box,child)
              CALL AddParent(child, node)
           ELSE
              ALLOCATE(child)
              CALL NullifyNodeFields(child)
              child%box=child_box
           END IF
           CALL AddChild(node, child)
           CALL DestroyNodeBox(child_box)
           nodebox_list => nodebox_list%next
        END DO

    END IF

  END SUBROUTINE ChomboReCreateChildren

!> Opens a new file group using the source group handle within a ChomboHandle object.
!! @param chandle An active Chombo handle.
!! @param source_name The name of the source group to be opened.
!! @details This subroutine is called from the SourceControl module.  It returns the number of source objects.
INTEGER FUNCTION Chombo_OpenSourceGroup(chandle, source_name)

    TYPE(ChomboHandle), POINTER :: chandle
    CHARACTER(LEN=*) :: source_name
    INTEGER :: i_err


    Chombo_OpenSourceGroup = 0

    ! Open the 'cooling_objects' group and save the handle.
    CALL h5gopen_f(chandle%root_group_id, source_name, chandle%source_group_id, i_err)
    CALL CatchHDF5Error("Chombo_OpenSourceGroup::h5gopen_f", i_err)

    ! Read in the number of source objects in this group and return the value.
    Chombo_OpenSourceGroup = Read_HDF5_Attribute_Int("num_objects", chandle%source_group_id)

END FUNCTION Chombo_OpenSourceGroup

!> Closes the source group handle within a ChomboHandle object.
!! @param chandle An active Chombo handle.
!! @details This subroutine is called from the SourceControl module.
SUBROUTINE Chombo_CloseSourceGroup(chandle)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: i_err

    IF (HandleIsValid(chandle%source_group_id)) THEN
        CALL h5gclose_f(chandle%source_group_id, i_err)
        CALL CatchHDF5Error("Chombo_CloseSourceGroup::h5gclose_f", i_err)
        chandle%source_offset = 0
    END IF

END SUBROUTINE Chombo_CloseSourceGroup

!> Opens a new file group using the particle group handle within a ChomboHandle object.
!! @param chandle An active Chombo handle.
!! @details This subroutine is called from the ParticleControl module.  On a read operation it returns the number of source objects.
INTEGER FUNCTION Chombo_OpenParticleGroup(chandle)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: i_err


    Chombo_OpenParticleGroup = 0

    IF (chandle%lWrite) THEN
        ! Create a new particles group within the root level
        CALL h5gcreate_f(chandle%root_group_id, "particles", chandle%particle_group_id, i_err)
        CALL CatchHDF5Error("Particle_WriteData::h5gcreate_f", i_err)		      
    ELSE
        ! Open the 'cooling_objects' group and save the handle.
        CALL h5gopen_f(chandle%root_group_id, "particles", chandle%particle_group_id, i_err)
        CALL CatchHDF5Error("Chombo_OpenParticleGroup::h5gopen_f", i_err)

        ! Read in the number of source objects in this group and return the value.
        Chombo_OpenParticleGroup = Read_HDF5_Attribute_Int("num_particles", chandle%particle_group_id)
    END IF

END FUNCTION Chombo_OpenParticleGroup

!> Closes the particle group handle within a ChomboHandle object.
!! @param chandle An active Chombo handle.
!! @details This subroutine is called from the ParticleControl module.
SUBROUTINE Chombo_CloseParticleGroup(chandle)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: i_err

    IF (HandleIsValid(chandle%particle_group_id)) THEN
        CALL h5gclose_f(chandle%particle_group_id, i_err)
        CALL CatchHDF5Error("Chombo_CloseParticleGroup::h5gclose_f", i_err)
        chandle%particle_offset = 0
    END IF

END SUBROUTINE Chombo_CloseParticleGroup

!> Returns the number of children on the next level.
!! @param chandle An active Chombo handle.
!! @param level The level whose CHILDREN we are querying.
INTEGER FUNCTION Chombo_LevelChildCount(chandle, level)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: level


    Chombo_LevelChildCount = 0

    SELECT CASE(level)
        CASE(CHOMBO_ROOT_LEVEL)
            Chombo_LevelChildCount = IO_GetDatasetElementCount(chandle%root_group_id, "root_child_boxes")
        CASE DEFAULT
            Chombo_LevelChildCount = IO_GetDatasetElementCount(chandle%level_id, "child_boxes")
    END SELECT

END FUNCTION Chombo_LevelChildCount

END MODULE ChomboDeclarations
!!@}
