!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    io_chombo.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @defgroup IOChomboGroup IO Chombo Group
!! @ingroup IO
!! @brief Group containing modules used for outputting chombo formatted files

!> @file io_chombo.f90
!! @brief Main file for module IO Chombo

!> @defgroup IOChombo IO Chombo
!! @brief Provides a formaIOCommst-agnostic interface for AstroBEAR to read and write data files.
!! @ingroup IOChomboGroup

!> Provides a formaIOCommst-agnostic interface for AstroBEAR to read and write data files.
!! @ingroup IOChombo
!! @author Brandon D. Shroyer
!! @date 7-9-2010
MODULE IOChombo

  USE HDF5
  USE HDF5Declarations
  USE ChomboDeclarations
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE ProcessingDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE MessageDeclarations
  USE MpiTransmission
  USE EllipticDeclarations
  USE CommunicationDeclarations
  USE MpiPacking
  USE IOComms
  USE IOParsing
  USE DistributionControl
  USE TimeStep, ONLY : olddt, overall_maxspeed,maxcfl
  IMPLICIT NONE
  PRIVATE

  PUBLIC MakeChomboFile, ChomboRestartInit, ChomboRestartFinalize, ChomboReloadLevel !ChomboRestart

CONTAINS


  !> Writes the current data frame to an HDF file using a modified Chombo specification.
  !! @param nframe the number of the sequence frame being written.
  !! @return time spent generating the chombo file.
  SUBROUTINE MakeChomboFile(nframe)

    USE ParticleControl, ONLY: lParticles, Particle_WriteData
    USE CoolingSrc, ONLY: CoolingCheck

    ! input declarations
    INTEGER, INTENT(IN) :: nframe
    INTEGER :: FinestLevel

    ! Variable declarations
    CHARACTER(LEN=23) :: s_filename

    INTEGER(HID_T) :: hid_property_id
    INTEGER(HID_T) :: hid_file_id

    INTEGER(HID_T) :: hid_group_id
    INTEGER(HID_T) :: hid_chombo_global_group_id
    INTEGER(HID_T) :: hid_level_group_id

    INTEGER :: i_err

    INTEGER(HID_T) :: hid_boxes_property_id
    INTEGER(HID_T) :: hid_boxes_dataspace_id
    INTEGER(HID_T) :: hid_boxes_dataset_id

    INTEGER(HID_T) :: hid_data_attributes_group_id

    INTEGER(HSIZE_T), DIMENSION(1) :: ia_dataset_dims     ! Holds the size of the dataset.
    INTEGER, PARAMETER :: MaxNodesPerLevel=200
    INTEGER (HSIZE_T):: i_index        ! Used for any incidental loops along the way.
    INTEGER (HSIZE_T):: i_mhd_index    ! Used when looping through aux data.

    ! Stores the cumulative offset of the chombo dataset hyperslab that we are writing to.
    INTEGER(HSIZE_T) :: i_qvar_offset

    ! Stores the cumulative offset of the chombo dataset hyperslab that we are writing to.
    INTEGER(HSIZE_T) :: i_aux_offset    

    INTEGER :: i, j, k        ! indices to use when looping along spatial dimensions.

    REAL(KIND = qPrec), ALLOCATABLE, DIMENSION(:) :: dbla_chombo_qvar

    INTEGER :: i_level        ! Tracks the level in several loops.
    INTEGER :: i_dimension    ! Used in the loop that assembles the boxes data.
    INTEGER :: i_variable    ! Used in the loop that assembles chombo data.


    CHARACTER(LEN = I_MAX_CNAME_LENGTH), DIMENSION(0:I_MAX_COMPONENTS) :: sa_component_names
    CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_tracer
    CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_component_tag

    CHARACTER(LEN = I_MAX_CNAME_LENGTH) :: s_level_name

    INTEGER, DIMENSION(I_CHOMBO_BOX_SIZE) :: ia_box_global        ! array for holding box data.

    INTEGER :: i_level_node_index

    INTEGER :: NrCons
    INTEGER :: mx, my, mz

    REAL(KIND = qPrec) :: dbl_domain_offset

    INTEGER :: i_grid_offset
    INTEGER, DIMENSION(I_CHOMBO_BOX_SIZE) :: box_array
    INTEGER, ALLOCATABLE, DIMENSION(:,:) :: ia_box_data

    INTEGER(HSIZE_T) :: i_box_offset
    INTEGER :: i_grid_data_size

    ! Used for iterating over the nodes on a level.    
    TYPE(NodedefList), POINTER :: nodelist
    TYPE(InfoDef), POINTER :: Info

    ! MHD (auxiliary) variable handling.
    INTEGER :: i_aux_data_size
    INTEGER, DIMENSION(3) :: ia_aux_extension
    REAL(KIND = qPrec), ALLOCATABLE, DIMENSION(:) :: dbla_chombo_aux

    ! Additional communication-related variables.
    TYPE(PackedMessage), POINTER :: message
    TYPE(PackedMessage), POINTER :: next_message
    INTEGER, DIMENSION(4) :: data_sizes

    INTEGER :: i_box_data_size
    INTEGER :: i_counter
    INTEGER :: remote_grid_count
    INTEGER, DIMENSION(3) :: local_mX
    INTEGER :: proc_id
    TYPE(Nodedef), POINTER :: node

    TYPE(ChomboHandle), POINTER :: chandle
    TYPE(NodedefList), POINTER :: child_list
    INTEGER :: child_count
    INTEGER :: i_childbox_data_size, i_acc_childbox_data_size
    LOGICAL :: lChildDatasetsInitialized
    INTEGER :: DummyInt, iErr
    TYPE(DiagnosticListDef), POINTER :: Diagnostic

    NULLIFY(Info)

    WRITE(s_filename, '(A10,I5.5,A4)') 'out/chombo', nframe, '.hdf'

    CALL CreateChomboHandle(s_filename, chandle, CHOMBO_HANDLE_WRITE)

    ! Start out by assuming that the finest level is the maximum allowable level.
    ! If it turns out there is no data on the max level, then FinestLevel could change.
    FinestLevel = MaxLevel

!!!!! Add required attributes for Chombo HDF specification.  New attributes are
!!!!! added to the root group usingAdd_ChomboHDF_Attribute() subroutine, which
!!!!! is described below.

    ! time attribute (uses Info's current time).
    CALL Add_HDF5_Attribute_Double("time", chandle%root_group_id, levels(0)%tnow)
    CALL Add_HDF5_Attribute_Double("olddt", chandle%root_group_id, olddt)
    CALL Add_HDF5_Attribute_Double("overall_maxspeed", chandle%root_group_id, overall_maxspeed)
    CALL Add_HDF5_Attribute_Double("maxcfl", chandle%root_group_id, maxcfl)

    ! iteration attribute (uses Info's current frame number).
    CALL Add_HDF5_Attribute_Int("iteration", chandle%root_group_id, nFrame)

    ! num_components attribute (use Info's number of output variables).
    CALL Add_HDF5_Attribute_Int("num_components", chandle%root_group_id, NrVars+NrDiagnosticVars)

    ! node centering (fixes contour maps in visit)
    CALL Add_HDF5_Attribute_Int("data_centering", chandle%root_group_id, 0) ! cell centered


 !!!This pro_lo attribute causes pseudocolor slices to not work with multiple root patches in visit
    !CALL Add_HDF5_Attribute_FloatVector("prob_lo", chandle%root_group_id, GxBounds(:,1))




!!!!!    The scaling factors below are not part of the Chombo HDF specification, but they're
!!!!!    useful enough that we're including them anyway.

    ! Number density.
    CALL Add_HDF5_Attribute_Double("n_density_scale", chandle%root_group_id, nScale)

    ! Mass density.
    CALL Add_HDF5_Attribute_Double("rho_scale", chandle%root_group_id, rScale)

    CALL Add_HDF5_Attribute_Double("length_scale", chandle%root_group_id, lScale)

    CALL Add_HDF5_Attribute_Double("velocity_scale", chandle%root_group_id, velScale)

    CALL Add_HDF5_Attribute_Double("pressure_scale", chandle%root_group_id, pScale)

    CALL Add_HDF5_Attribute_Double("temperature_scale", chandle%root_group_id, TempScale)

    !Uses var_index to keep track of what variables are in what slots
    NrCons = NrHydroVars - NrTracerVars

    ! Create tags for conservative variables using indices initialized in 
    ! PhysicsControl::SetVariableIndices().
    DO i=1, NrCons
       sa_component_names(i-1)= trim(FieldName(i))
    END DO

    ! Creates tags for tracer variables.
    DO i = 1, NrTracerVars !NrCons+1,NrHydroVars
       sa_component_names(nTracerLo-1+i-1) = trim(TracerName(i))
    END DO

    ! Creates tags for Elliptic variables.
    DO i = 1, NrEllipticVars !NrHydroVars+1,NrHydroVars+NrEllipticVars
       sa_component_names(nEllipticLo-1+i-1) = trim(EllipticName(i))       
!       IF (i == iPhi) sa_component_names(i-1)="Phi"
!       IF (i == iPhiSinks) sa_component_names(i-1)="PhiSinks"
!       IF (i == iPhiGas) sa_component_names(i-1)="PhiGas"
!       IF (i == iPhiDot) sa_component_names(i-1)="PhiDot"
    END DO

    Diagnostic=>FirstDiagnostic
    i=1
    DO WHILE (ASSOCIATED(Diagnostic))
       sa_component_names(NrHydroVars+NrEllipticVars+i-1) = trim(Diagnostic%Field%Name)
       i=i+1
       Diagnostic=>Diagnostic%next
    END DO

    ! If MaintainAuxArrays is true, then the current problem uses MHD.
    IF (MaintainAuxArrays) THEN    

       ! All MHD problems are three-dimensional, so you have to include all three
       ! directional momentum and magnetic field components.
       ! Add derived scaling factors.
       CALL Add_HDF5_Attribute_Double("px_scale", chandle%root_group_id, rScale * velScale)
       CALL Add_HDF5_Attribute_Double("py_scale", chandle%root_group_id, rScale * velScale)
       CALL Add_HDF5_Attribute_Double("pz_scale", chandle%root_group_id, rScale * velScale)

       IF (iE .ne. 0) CALL Add_HDF5_Attribute_Double("E_scale", chandle%root_group_id, pScale)

       CALL Add_HDF5_Attribute_Double("Bx_scale", chandle%root_group_id, pScale)
       CALL Add_HDF5_Attribute_Double("By_scale", chandle%root_group_id, pScale)
       CALL Add_HDF5_Attribute_Double("Bz_scale", chandle%root_group_id, pScale)

    ELSE

       ! If the project is three-dimensional, then the pz component must be added;
       ! otherwise, begin adding tracer components.
       IF (nDim == 2) THEN
!          ! Add derived scaling factors.
          CALL Add_HDF5_Attribute_Double("px_scale", chandle%root_group_id, rScale * velScale)
          CALL Add_HDF5_Attribute_Double("py_scale", chandle%root_group_id, rScale * velScale)
          IF (iE .ne. 0) CALL Add_HDF5_Attribute_Double("E_scale", chandle%root_group_id, pScale)
       ELSE
          ! Add derived scaling factors.
          CALL Add_HDF5_Attribute_Double("px_scale", chandle%root_group_id, rScale * velScale)
          CALL Add_HDF5_Attribute_Double("py_scale", chandle%root_group_id, rScale * velScale)
          CALL Add_HDF5_Attribute_Double("pz_scale", chandle%root_group_id, rScale * velScale)    
          IF (iE .ne. 0) CALL Add_HDF5_Attribute_Double("E_scale", chandle%root_group_id, pScale)
       END IF
    END IF


    ! This part is separated out so that changing the assembly of the components
    ! is easy, but the addition of attributes may eventually have to be done
    ! inline in order to reduce overhead.
    !
    ! Construct new component_n attributes for each sa_component_name(n).
    DO i_index = 0, NrVars + NrDiagnosticVars - 1

       ! Create component tag (attribute name).
       IF (i_index <= 9) THEN
          WRITE(s_component_tag,'(A10,I1.1)') 'component_', i_index
       ELSE
          WRITE(s_component_tag,'(A10,I2.2)') 'component_', i_index
       END IF

       CALL Add_HDF5_Attribute_String(s_component_tag, &
            chandle%root_group_id, TRIM(sa_component_names(i_index)))
    END DO

!!! End Assembling Component Array.

!!! Create Group Chombo_global And Its Attributes

    ! Create Chombo_global group.
    CALL h5gcreate_f(chandle%file_handle, "Chombo_global", hid_chombo_global_group_id, i_err)
    CALL CatchHDF5Error("h5gcreate_f ('Chombo_global')", i_err)

    ! Add testReal attribute to Chombo_global group (value is a literal).
    CALL Add_HDF5_Attribute_Double("testReal", hid_chombo_global_group_id, &
         DBL_TEST_REAL_DAT) 

    ! Add SpaceDim attribute to Chombo_global group.  We can just use Info%nDim
    ! because we don't generally do 4D problems.
    IF (nDim == 1) THEN
          CALL Add_HDF5_Attribute_Int("SpaceDim", hid_chombo_global_group_id, 2)
       ELSE
          CALL Add_HDF5_Attribute_Int("SpaceDim", hid_chombo_global_group_id, nDim)
       END IF
    ! Close the Chombo_global group.
    CALL h5gclose_f(hid_chombo_global_group_id, i_err)
    CALL CatchHDF5Error("h5gclose_f ('Chombo_global')", i_err)

!!! End Create Chombo_global group.

!!! Storing the domain boundaries in the Chombo file.
    CALL Add_HDF5_Attribute_FloatVector("lower_bound", chandle%root_group_id, GxBounds(:,1))
    CALL Add_HDF5_Attribute_FloatVector("upper_bound", chandle%root_group_id, GxBounds(:,2))


    i_acc_childbox_data_size = 1 ! The initial value for i_childbox_data_size; after level -2
    ! it will have more meaningful values, but we want to make
    ! sure that the loop below functions properly.

    ! Clear the next_level_cost array before using it.
    chandle%next_level_cost = 0

!!!    Assembling the group_level(n) groups.

    DO i_level = -2, FinestLevel

!       PRINT "('Level ', i2, ' begun.')", i_level

       ! The size of the current level's boxes dataset is equal to the number of child boxes on the
       ! previous level.  Therefore, rather than recalculate it, we just move over the accumulated
       ! childbox count.
       i_box_data_size = i_acc_childbox_data_size

       ! Clear level data counters.
       i_grid_data_size = 0
       i_acc_childbox_data_size = 0
       IF (MaintainAuxArrays)  i_aux_data_size = 0
       lChildDatasetsInitialized = .FALSE.

       IF (i_level >= 0) THEN

          ! [BDS] Select the youngest node on level i_level.
          nodelist => Nodes(i_level)%p

          DO proc_id = 0, MPI_np - 1

             i_counter = 0
             
             IF (proc_id > Master) THEN

                ! Tell worker to send data
                CALL MPI_SEND(DummyInt, 1, MPI_INTEGER, proc_id, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, iErr)
                ! Create a message to receive data from this process.
                CALL CreatePackedMessage(i_level, &
                                         proc_id, &
                                         TRANSMIT_FRAME_DATA, &
                                         STAGE_RECV, &
                                         message, &
                                         chandle%next_level_cost(proc_id))

                ! Retrieve the level statistics from the remote processor.
                CALL IO_UnparseLevelStatistics(message, data_sizes)

                chandle%next_level_cost(message%remote_proc) = TERMINATION_BOX_BYTES + &
                                                               IO_LEVEL_STAT_BYTES + &
                                                               data_sizes(IO_NEXTLEVELCOST)

                !Stores the number of child grids on this level.
                i_childbox_data_size = data_sizes(IO_CHILDCOUNT)

                IF (i_level >= 0) THEN
                   ! Initializes the size of the data:datatype=0 array.
                   i_grid_data_size = data_sizes(IO_Q_SIZE)


                   ! Initializes the size of the MHD_data array.  If there is no MHD in this problem
                   ! (i.e., if lMHD is false), then this will be 0.
                   i_aux_data_size = data_sizes(IO_AUX_SIZE)
                END IF

             ELSE
                ! Obtain level statistics for the local processor.
                CALL IO_GetDatasetSizes(i_level, FinestLevel, data_sizes)
                

                !Stores the number of child grids on this level.
                i_childbox_data_size = data_sizes(IO_CHILDCOUNT)

                IF (i_level >= 0) THEN

                   ! Initializes the size of the data:datatype=0 array.
                   i_grid_data_size = data_sizes(IO_Q_SIZE)

                   ! Initializes the size of the MHD_data array.  If there is no MHD in this problem
                   ! (i.e., if lMHD is false), then this will be 0.
                   i_aux_data_size = data_sizes(IO_AUX_SIZE)

                   ! [BDS] Select the youngest node on level i_level.
                   nodelist => Nodes(i_level)%p

                END IF

             END IF

             i_acc_childbox_data_size = i_acc_childbox_data_size + i_childbox_data_size

             DO   ! Loop over nodes on i_level.

                IF (proc_id > Master) THEN

                   NULLIFY(node, Info)

                   ! Extract the next node from the message.  This will return null on
                   ! a termination box.
                   CALL IO_UnparseRemoteNode(message, node)
                   
                   IF (ASSOCIATED(node)) THEN
                      ! If the level is less than the finest level, then unparse any
                      ! child nodes that have been sent along with the current node.
                      IF (i_level < FinestLevel)  CALL IO_UnparseNodeChildren(message, child_count, node)

                      ! Unparse the grid data associated with the current node.
                      IF (i_level > -1) CALL IO_UnparseRemoteGrid(message, Info, FinestLevel)
                   ELSE
                      EXIT
                   END IF

                ELSE

                   ! If node_ptr is now null, then there are no nodes
                   ! left on level i_level and we can exit the loop.
                   IF (.NOT. ASSOCIATED(nodelist)) EXIT
                   IF (.NOT. ASSOCIATED(nodelist%self))  EXIT

                   ! Retrieve the InfoDef structure of node_ptr.
                   node => nodelist%self
                   child_count = NodeCount(node%children)
                   Info => nodelist%self%Info

                   ! Die if unable to obtain node_ptr's Info structure.
                   IF (.NOT. ASSOCIATED(Info)) THEN
                      PRINT *, "MakeChomboFile error ", i_err, &
                           ": unable to obtain node info structure."
                      STOP
                   END IF
                   
                END IF

                ! If there is cell-centered data to be stored in the dataset and the offset
                ! is empty, then assume the level has not been created and create it.  If 
                ! there is no grid data, then do nothing.  If there is data to store and the 
                ! offset is not 0, then the dataset extension code will fix the problem when
                ! this routine goes to write the data to the dataset.

                ! q data is never sent without costmap data, so an empty q offset will be accompanied
                ! by an empty costmap offset as well.
                
                IF (i_grid_data_size > 0) THEN
                   IF (.NOT. HandleIsValid(chandle%level_id)) THEN

                      ! Open a chombo handle for this level.
                      CALL IO_OpenCurrentLevel(chandle, i_level)

                      ! Create the dx attribute
                      CALL Add_HDF5_Attribute_Double("dx", chandle%level_id, levels(i_level)%dx)
                      ! Create the anisotropic attribute (optional chomobo attrib for anisotropic spacing)
                      ! [BDS][20100712]:  The design does not currently support anisotropic systems.
                      CALL Add_HDF5_Attribute_FloatVector("anisotropic", chandle%level_id, (/1d0,1d0,1d0/))!Info%dX/Info%dX(1))         

                      ! Create the ref_ratio attribute.
                      CALL Add_HDF5_Attribute_Int("ref_ratio", chandle%level_id, levels(i_level)%CoarsenRatio)

                      ! loop over the three spatial dimensions to construct the box object..
                      DO i_index = 1,3

                         ! Set the lower-bounds of the box object.  This will be the 
                         ! lower-bound value in mGlobal, minus one as an offset for 
                         ! Chombo's field-space.

                         ia_box_global(i_index) = 0

                         ! ia_box_global = #cells along each dimension, adjusted by the 
                         !           refinement level.  This can be found by taking
                         !          the difference between the max and min values
                         !          for each dimension and dividing by dX(dim).

                         IF (levels(i_level)%dX /= zero) THEN
                            ia_box_global(i_index+3)=Gmx(i_index) * product(levels(0:i_level-1)%coarsenratio)-1 !(GxBounds(i_index,2) - GxBounds(i_index,1)) / levels(i_level)%dX
                         END IF

                         ! If there is a fractional part to the domain offset, then 
                         ! add another cell to the global box to accomodate it.
!                         ia_box_global(i_index + 3) = MAX(dbl_domain_offset - 1, 0)

                      END DO

                      ! Create prob_domain attribute using a box object.
                      CALL Add_HDF5_Attribute_Box("prob_domain", hid_box_id, chandle%level_id, ia_box_global)

                   END IF



                   IF (HandleIsValid(chandle%level_id)) THEN

                      ! ** box data **
!                      IF (i_box_data_size > 0) THEN

                          ! If the handle's offset is 0, then no data has been added to the dataset,
                          ! meaning that it hasn't been initialized yet.  This conditional will correct that.
                          IF (chandle%box_offset == 0) THEN
                              ! The dataset that holds the grid dimensions.
                              CALL Initialize_HDF5_Dataset_Box("boxes", chandle%level_id, i_box_data_size)
                          END IF

                          CALL IO_WriteBoxToChomboFile(chandle, "boxes", node%box%mGlobal)
!
 !                     END IF

                      ! ** childbox data **
                      ! On levels below FinestLevel, store the child boxes of each node.  These boxes
                      ! are stored in breadth-first order relative to their parents.
                      IF ((i_level < FinestLevel)) THEN

                         ! If the handle's offset is 0, then no data has been added to the dataset,
                         ! meaning that it hasn't been initialized yet.  This conditional will correct that.
                         IF ((.NOT. lChildDatasetsInitialized) .AND. (chandle%childbox_offset == 0)) THEN
                             ! The size of the childbox count dataset is always known--it is the number of boxes
                             ! on the current level, since each element has a 1-1 correspondence with the current
                             ! level.

                             CALL Initialize_HDF5_Dataset_Int("childbox_count", chandle%level_id, i_box_data_size)
                             CALL Initialize_HDF5_Dataset_Box("child_boxes", chandle%level_id, i_childbox_data_size)
                             lChildDatasetsInitialized = .TRUE.
                         END IF

                         ! Write the number of children to the child_boxes array.
                         CALL IO_WriteChildCountToChomboFile(chandle, child_count)

                         child_list => node%children

                         ! Loop over the child box list and add each child to the child boxes dataset.
                         DO WHILE (ASSOCIATED(child_list))
                            CALL IO_WriteBoxToChomboFile(chandle, "child_boxes", child_list%self%box%mGlobal)
                            child_list => child_list%next
                         END DO

                         NULLIFY(child_list)

                      END IF

                      ! ** grid data **
                      ! Initialize the dataset that holds cell-centered data.  Populating these
                      ! datasets is more involved than the box datasets, so we'll do that later.
                      IF ((i_grid_data_size > 0) .AND. (chandle%q_offset == 0)) THEN

                         CALL Initialize_HDF5_Dataset_Double("data:datatype=0", &
                                                             chandle%level_id, &
                                                             i_grid_data_size)

                         ! If this is not the finest level, then initialize a costmap dataset
                         ! for this level.
!                         IF (i_level < FinestLevel) THEN
!                               CALL Initialize_HDF5_Dataset_Double("costmap", &
!                                                                   chandle%level_id, &
!                                                                   i_grid_data_size * 2 / NrVars)
!                         END IF

                      END IF


                      ! Same as for the cell-centered data, except now we're creating a face-centered dataset.
                      IF (MaintainAuxArrays .AND. (i_aux_data_size > 0) .AND. (chandle%aux_offset == 0)) THEN
                         CALL Initialize_HDF5_Dataset_Double("MHD_data", chandle%level_id, i_aux_data_size)
                      END IF

 
                      ! Obtain the grid dimensions from the coordinate box.
                      mx = node%box%mGlobal(1,2) - node%box%mGlobal(1,1) + 1
                      my = node%box%mGlobal(2,2) - node%box%mGlobal(2,1) + 1
                      mz = node%box%mGlobal(3,2) - node%box%mGlobal(3,1) + 1

                      ! Allocate memory for the current node's grid.
                      ALLOCATE(dbla_chombo_qvar(mx * my * mz * (NrVars+NrDiagnosticVars)), stat=i_err)

                      ! Stops execution on a memory allocation error.
                      IF (i_err /= 0) THEN
                         PRINT *,"MakeChomboFile error: unable to allocate memory for q-var array."
                         STOP
                      END IF

                      ! Clear index; it will be used to reference chombo data array in the loop below.
                      i_index = 0

                      ! Loop over each variable tracked by the program.    
                      DO i_variable = 1, NrVars

                         ! The nested loops below loop over each spatial dimension and reference the
                         ! q value for each variable within a specific 3D coordinate; this value is
                         ! then stored in the chombo data array.  In effect, what this does is flatten
                         ! the multi-dimensional data in q down to a 1D array for HDF5 storage.

                         DO k = 1, mz
                            DO j = 1, my
                               DO i = 1, mx

                                  i_index = i_index + 1
                                  dbla_chombo_qvar(i_index) = Info%q(i, j, k, i_variable)

                               END DO ! End i
                            END DO ! End j
                         END DO ! End k

                      END DO ! End do i_variable =  to Info%NrOutVars


                      ! Loop over each variable tracked by the program.    
                      DO i_variable = 1, NrDiagnosticVars

                         ! The nested loops below loop over each spatial dimension and reference the
                         ! q value for each variable within a specific 3D coordinate; this value is
                         ! then stored in the chombo data array.  In effect, what this does is flatten
                         ! the multi-dimensional data in q down to a 1D array for HDF5 storage.

                         DO k = 1, mz
                            DO j = 1, my
                               DO i = 1, mx

                                  i_index = i_index + 1
                                  dbla_chombo_qvar(i_index) = Info%diagnostics(i, j, k, i_variable)

                               END DO ! End i
                            END DO ! End j
                         END DO ! End k

                      END DO ! End do i_variable =  to Info%NrOutVars
                      
                      !Deallocate the diagnostic array always once it is outputted.
                      IF (NrDiagnosticVars > 0) THEN
                         IF (.NOT.(ASSOCIATED(Info%diagnostics))) THEN
                            write(*,*) 'info diagnostics not associated'
                            STOP
                         END IF
                         DEALLOCATE(Info%diagnostics)
                         NULLIFY(Info%diagnostics)
                      END IF
                      
                      ! Write the data array to the Chombo file using the supplied offset.
                      CALL Write_Slab_To_Dataset_Double("data:datatype=0", chandle%level_id, dbla_chombo_qvar, chandle%q_offset)
                      ! Adjust the offset by the size of the grid just written.

                      chandle%q_offset = chandle%q_offset + i_index

!                      IF( i_level < FinestLevel) THEN
!                         CALL Write_Slab_To_Dataset_Double("costmap", chandle%level_id, &
!                              RESHAPE(Info%costmap(1:mx,1:my,1:mz,1:2), (/ mx*my*mz*2 /)), chandle%costmap_offset)

                         ! Adjust the costmap offset.
!                         chandle%costmap_offset = chandle%costmap_offset + mx*my*mz*2
!                      END IF

                      ! Release array memory so that it can be used by the next grid.
                      DEALLOCATE(dbla_chombo_qvar)

                      ! If MHD is running, then do the same thing for the Info%aux array.
                      IF (MaintainAuxArrays) THEN 

                         ! Allocate memory for the current node's aux grid.
                         IF (nDim == 3) THEN
                            ALLOCATE(dbla_chombo_aux(((mx + 1) * my * mz) + &
                                 (mx * (my + 1) * mz) + &
                                 (mx * my * (mz + 1))), stat=i_err)
                         ELSE
                            ALLOCATE(dbla_chombo_aux(((mx + 1) * my) + &
                                 (mx * (my + 1))), stat=i_err)
                         END IF

                         ! Stops execution on a memory allocation error.
                         IF (i_err /= 0) THEN
                            PRINT *,"MakeChomboFile error: unable to allocate memory for aux array."
                            STOP
                         END IF

                         ! Clear index; it will be used to reference chombo data array in the loop below.
                         i_mhd_index = 0

                         ! There will be one Bx variable for each dimension of the problem; we
                         ! don't need to keep track of any other aux variables.
                         DO i_variable = 1, nDim

                            ! Set the dimension of the current Bx variable to be extended by 1.
                            ia_aux_extension = 0
                            ia_aux_extension(i_variable) = 1

                            ! The nested loops below loop over each spatial dimension and reference the
                            ! aux value for each variable within a specific 3D coordinate; this value is
                            ! then stored in the chombo data array.  In effect, what this does is flatten
                            ! the multi-dimensional data in aux down to a 1D array for HDF5 storage.

                            DO k = 1, mz + ia_aux_extension(3)
                               DO j = 1, my + ia_aux_extension(2)
                                  DO i = 1, mx + ia_aux_extension(1)

                                     i_mhd_index = i_mhd_index + 1
                                     dbla_chombo_aux(i_mhd_index) = Info%aux(i, j, k, i_variable)

                                  END DO    ! End i
                               END DO        ! End j
                            END DO            ! End k

                         END DO        ! End do i_variable =  to Info%maux

                         ! Write the data array to the Chombo file using the supplied offset.
                         CALL Write_Slab_To_Dataset_Double("MHD_data", chandle%level_id, &
                              dbla_chombo_aux, chandle%aux_offset)

                         ! Adjust the offset by the size of the grid just written.
                         chandle%aux_offset = chandle%aux_offset + i_mhd_index

                         ! Release array memory so that it can be used by the next grid.
                         DEALLOCATE(dbla_chombo_aux)

                      END IF         ! (if MaintainAuxArrays).

                   END IF
                END IF
                ! Destroy the structures created from remote data.
                IF (proc_id > Master) THEN

                   ! Deallocate the cell-centered data array.
                   DEALLOCATE(Info%q)
                   NULLIFY(Info%q)

                   ! Deallocate the aux array imported from the remote processor
                   IF (MaintainAuxArrays) THEN
                      DEALLOCATE(Info%aux)
                      NULLIFY(Info%aux)
                   END IF

                   DEALLOCATE(Info)
                   NULLIFY(Info)

                   ! Deallocate the node's variable-length lists.
!                   DEALLOCATE(node%proclist, node%proctime)
!                   NULLIFY(node%proclist, node%proctime)

                   IF (ASSOCIATED(node%children))  CALL DestroyNodeList(node%children)

                   DEALLOCATE(node)
                   NULLIFY(node)

                ELSE
                   ! [BDS] Advance to the next node on the level.
                   NULLIFY(child_list)
                   nodelist => nodelist%next

                END IF

             END DO  ! Loop over nodes on i_level.

             ! Destroy this processor's message to make room for a new one.
             IF (proc_id > Master)  CALL DestroyPackedMessage(message)

          END DO  ! Loop over processors.

!          PRINT *, "Level ", i_level, " datasets finished."

!!!!!!!! END DATAZERO LEVEL TRAVERSAL

          ! If there are boxes on this level, then add the remaining level data and close
          ! the level handle.  Otherwise, set the FinestLevel to be the previous level and exit.
          IF (HandleIsValid(chandle%level_id)) THEN

             ! Create the data attributes group.
             CALL h5gcreate_f(chandle%level_id, "data_attributes", hid_data_attributes_group_id, i_err)
             CALL CatchHDF5Error("h5gcreate_f('data_attributes')", i_err)

             ! Adds the output_ghost attribute using the literal intvector [0, 0, 0].
             CALL Add_HDF5_Attribute_IntVector("outputGhost", hid_intvect_id, &
                  hid_data_attributes_group_id, (/0,0,0/))

             ! Adds the comps attribute (the same as num_components)
             CALL Add_HDF5_Attribute_Int("comps", hid_data_attributes_group_id, &
                  NrVars+NrDiagnosticVars)

             ! Adds the objectType attribute.
             CALL Add_HDF5_Attribute_String("objectType", hid_data_attributes_group_id, "FArrayBox")

             ! Close data attributes group.
             CALL h5gclose_f(hid_data_attributes_group_id, i_err)
             CALL CatchHDF5Error("h5gclose_f('data_attributes')", i_err)

             ! Close level_n group.
             CALL IO_CloseCurrentLevel(chandle)

          ELSE
             ! If there were no boxes on this level, then some previous level was the finest level.
             ! 
             IF (FinestLevel >= i_level)  FinestLevel = i_level - 1
          END IF


       ELSE IF (i_level == CHOMBO_DOMAIN_LEVEL) THEN

          IF (nDomains > 1) THEN 
             PRINT*, 'not yet implemented for IO'
             STOP
          END IF
          ! Clear all level offsets.
          CALL ClearChomboHandleOffsets(chandle)
          chandle%next_level_cost = 0

          ! Obtain level statistics for the local processor.
          CALL IO_GetDatasetSizes(i_level, FinestLevel, data_sizes)

          !Stores the number of child grids on this level.
          i_childbox_data_size = data_sizes(IO_CHILDCOUNT)
          i_acc_childbox_data_size=i_acc_childbox_data_size+i_childbox_data_size

          ! Open a chombo handle for this level.
          CALL IO_OpenCurrentLevel(chandle, i_level)

          ! The -1 level stores its own boxes, child boxes (plus their count), and its costmap.
          ! It does not, however, have any cell- or face-centered data to store. 
!          CALL Initialize_HDF5_Dataset_Double("costmap", chandle%level_id, PRODUCT(GmX)*2, &
!               L_NON_EXTENSIBLE)

          CALL Initialize_HDF5_Dataset_Box("boxes", chandle%level_id, i_box_data_size, L_NON_EXTENSIBLE)
          CALL Initialize_HDF5_Dataset_Int("childbox_count", chandle%level_id, i_childbox_data_size)
          CALL Initialize_HDF5_Dataset_Box("child_boxes", chandle%level_id, i_childbox_data_size)

          CALL IO_WriteDomainData(chandle, data_sizes(IO_CHILDCOUNT))

          ! Close level_n group.
          CALL IO_CloseCurrentLevel(chandle)

       ELSE IF (i_level == CHOMBO_ROOT_LEVEL) THEN

          ! Clear all level offsets.
          CALL ClearChomboHandleOffsets(chandle)
          chandle%next_level_cost = 0

          ! Obtain level statistics for the local processor.
          CALL IO_GetDatasetSizes(i_level, FinestLevel, data_sizes)

          !Stores the number of child grids on this level.
          i_childbox_data_size = data_sizes(IO_CHILDCOUNT)
          i_acc_childbox_data_size=i_acc_childbox_data_size+i_childbox_data_size

          ! The -2 level only has one grid, and it's on the master.  Since that grid is always generated
          ! by DomainInit(), the -2 level only needs a costmap, a single integer for its child count, and
          ! its child boxes.
!          CALL Initialize_HDF5_Dataset_Double("root_costmap", chandle%root_group_id, &
!               PRODUCT(GmX)*2, L_NON_EXTENSIBLE)

          CALL Add_HDF5_Attribute_Int("root_child_count", chandle%root_group_id, i_childbox_data_size)
          CALL Initialize_HDF5_Dataset_Box("root_child_boxes", chandle%root_group_id, i_childbox_data_size)

       END IF

       PRINT *, 'Level ', i_level, ' data written.'
!       PRINT *

    END DO         !i_level = -2 to FinestLevel.


    ! num_levels attribute (Chombo format requires that this be 1 greater the maximum
    ! level--essentially so that it acts as a level-index.
    CALL Add_HDF5_Attribute_Int("num_levels", chandle%root_group_id, FinestLevel + 1)

    ! max_level attribute (use Info's current FinestLevel value).
    CALL Add_HDF5_Attribute_Int("max_level", chandle%root_group_id, FinestLevel)

!!! END OF LEVEL_N groups.

    ! [BDS][20081126]:  Add sink particles to chombo if required components are active.
    ! [BDS][20100712]:  Sink particles are not currently implemented in scrambler, so this
    !                   feature is deactivated.
    ! [BDS][20101130]:  Sink particles are now part of scrambler.  The routine below will stores them
	!                   in the format described by the chombo DDF.
    CALL Particle_WriteData(chandle)

    CALL WriteCoolingObjects(chandle)

    CALL WritePointGravityObjects(chandle)

    ! Close the chombo file.
    CALL CloseChomboHandle(chandle)

  END SUBROUTINE MakeChomboFile

  !> Writes a box to a named dataset within the active chombo file.
  !! @param chandle The handle for the active chombo file.
  !! @param box_set_name The name of the dataset being written to.
  !! @param box The box array to be written.
  SUBROUTINE IO_WriteBoxToChomboFile(chandle, box_set_name, box)

    TYPE(ChomboHandle), POINTER :: chandle
    CHARACTER(*) :: box_set_name
    INTEGER, DIMENSION(3,2) :: box

    INTEGER, DIMENSION(6) :: box_data
    INTEGER(HSIZE_T) :: box_offset
    INTEGER(HID_T) :: source_group_id


    box_data(1:3) = box(1:3,1) - 1
    box_data(4:6) = box(1:3,2) - 1

    ! Four possible options:  "boxes", "child_boxes", "root_child_boxes"
    IF (box_set_name == "child_boxes") THEN
       box_offset = chandle%childbox_offset
       source_group_id = chandle%level_id

    ELSE IF (box_set_name == "boxes") THEN
       box_offset = chandle%box_offset
       source_group_id = chandle%level_id

    ELSE IF (box_set_name == "root_child_boxes") THEN
       box_offset = chandle%childbox_offset
       source_group_id = chandle%root_group_id

    END IF

    CALL Write_Slab_To_Dataset_Box(box_set_name, &
         source_group_id, &
         box_data, &
         box_offset)

    IF (box_set_name == "child_boxes") THEN
       chandle%childbox_offset = chandle%childbox_offset + 1
    ELSE IF (box_set_name == "boxes") THEN
       chandle%box_offset = chandle%box_offset + 1
    ELSE
       chandle%domainbox_offset = chandle%domainbox_offset + 1
    END IF

  END SUBROUTINE IO_WriteBoxToChomboFile

  !> Writes a single integer to the childbox_count dataset within the active chombo file.
  !! @param chandle The handle for the active chombo file.
  !! @param int_data The integer data to be written.
  SUBROUTINE IO_WriteChildCountToChomboFile(chandle, int_data)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: int_data


    CALL Write_Slab_To_Dataset_Int("childbox_count", &
         chandle%level_id, &
         (/ int_data /), &
         chandle%childbox_count_offset)

    chandle%childbox_count_offset = chandle%childbox_count_offset + 1

  END SUBROUTINE IO_WriteChildCountToChomboFile

  !> Collects box and cost map data for level -1 and writes it to the chombo file.
  !! @param chandle An active chombo handle.
  !! @param master_childcount Number of master children
  SUBROUTINE IO_WriteDomainData(chandle, master_childcount)

    TYPE(ChomboHandle), POINTER :: chandle

    TYPE(PackedMessage), POINTER :: message
    TYPE(NodedefList), POINTER :: nodelist, child_list
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: cost_map
    INTEGER :: proc_id
    INTEGER :: master_childcount

    INTEGER :: mx, my, mz
    INTEGER, DIMENSION(3,2) :: mGlobal
    TYPE(Nodedef), POINTER :: node
    INTEGER :: child_count
    INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT) :: data_sizes
    INTEGER :: i_childbox_data_size
    INTEGER :: DummyInt, iErr

    chandle%costmap_offset = 0
    child_count = 0

    ! Points to the node list for the -1 (domain) level.
    nodelist => Nodes(CHOMBO_DOMAIN_LEVEL)%p
    
    ! Loop over processors and retrieve any cost map data.
    DO proc_id = 0, MPI_np - 1

       ! If proc-id is a worker, then retrieve its message.
       IF (proc_id > Master) THEN
          ! Create packed message object to recieve data.
          CALL CreatePackedMessage(CHOMBO_DOMAIN_LEVEL, &
                                   proc_id, &
                                   TRANSMIT_DOMAIN_DATA, &
                                   STAGE_RECV, &
                                   message)

          ! Retrieve the level statistics from the remote processor.  On the domain
          ! level, the q and aux data have no significance, but the child nodes do.
          CALL IO_UnparseLevelStatistics(message, data_sizes)
          
          ! Store the cost of the next level's messages from this processor.
          chandle%next_level_cost(message%remote_proc) = TERMINATION_BOX_BYTES + &
                                                         IO_LEVEL_STAT_BYTES + &
                                                         data_sizes(IO_NEXTLEVELCOST)

       ELSE
          ! Initialize the child count on the master processor.
          i_childbox_data_size = master_childcount
       END IF

       DO   ! Loop over domain nodes.

          NULLIFY(cost_map)

          IF (proc_id > Master) THEN

             NULLIFY(node)

             ! Extract a node that will go into the boxes array.
             CALL IO_UnparseRemoteNode(message, node)

             IF (ASSOCIATED(node)) THEN
                ! Extract the child nodes from the message.
                CALL IO_UnparseNodeChildren(message, child_count, node)
                mGlobal=node%box%mGlobal
                ! Increment the size counter for the childbox dataset.
                i_childbox_data_size = i_childbox_data_size + child_count

                ! Unparse a costmap from the packed message and break this loop if
                ! none can be found.
!                CALL IO_UnparseLowLevelCostMap(message, mGlobal, cost_map)
             ELSE
                EXIT
             END IF

             ! Obtain the grid dimensions from the box coordinates.
             mx = mGlobal(1,2) - mGlobal(1,1) + 1
             my = mGlobal(2,2) - mGlobal(2,1) + 1
             mz = mGlobal(3,2) - mGlobal(3,1) + 1

          ELSE

             ! Retrieve the costmap from the next InfoDef structure in the list.
             IF (ASSOCIATED(nodelist)) THEN
                ! Retrieve the InfoDef structure of node_ptr.
                node => nodelist%self
!                cost_map => node%Info%costmap
                child_count = NodeCount(node%children)
             ELSE
                EXIT
             END IF

             mx = node%box%mGlobal(1,2)-node%box%mGlobal(1,1) !Info%mX(1)
             my = node%box%mGlobal(2,2)-node%box%mGlobal(2,1) !Info%mX(2)
             mz = node%box%mGlobal(3,2)-node%box%mGlobal(3,1) !Info%mX(3)

             mGlobal = node%box%mGlobal

          END IF

          ! Write box to domain box list.
          CALL IO_WriteBoxToChomboFile(chandle, "boxes", mGlobal)

          ! Write box to root-level child box list (which we can do because there is no
          ! level group for level -2; we're just writing to a dataset in the root group,
          ! which stays open until the CloseChomboHandle() subroutine is called.
          CALL IO_WriteBoxToChomboFile(chandle, "root_child_boxes", mGlobal)

          ! Write the data array to the Chombo file using the supplied offset.
!          CALL Write_Slab_To_Dataset_Double("costmap", chandle%level_id, &
!               RESHAPE(cost_map(1:mx,1:my,1:mz,1:2), (/ mx*my*mz*2 /)), chandle%costmap_offset)

          ! Increment the offset by the size of the cost map.
!          chandle%costmap_offset = chandle%costmap_offset + mx*my*mz*2
          
          CALL IO_WriteChildCountToChomboFile(chandle, child_count)

          child_list => node%children

          ! The childbox attribute is being used twice on this level, so we clear it
          ! again here.  It's klugey, but so's adding another offset that's only used once.
          chandle%childbox_offset = 0

          ! 
          DO WHILE (ASSOCIATED(child_list))
             CALL IO_WriteBoxToChomboFile(chandle, "child_boxes", child_list%self%box%mGlobal)

             child_list => child_list%next
          END DO

          ! If the cost map came from a worker processor, then deallocated it.  Otherwise,
          ! just advance to the next node.
          IF (proc_id > Master) THEN
!             DEALLOCATE(cost_map)
!             NULLIFY(cost_map)
             IF (ASSOCIATED(node%children))  CALL DestroyNodeList(node%children)
          ELSE
             nodelist => nodelist%next
          END IF

       END DO ! node list loop

       ! Destroy worker message so that a new one can be created.
       IF (proc_id > MASTER)  CALL DestroyPackedMessage(message)

    END DO ! processor loop

  END SUBROUTINE IO_WriteDomainData

  !> Reads in the boxes and childboxes data for the level.  On the root level, it also pulls costmap.
  !! @param chandle An active Chombo handle.
  !! @param level The level of the data being pulled.
  SUBROUTINE IO_ReadInLevelData(chandle, level)

    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: level

    TYPE(NodedefList), POINTER :: nodelist, childlist
    TYPE(Nodedef), POINTER :: node, child
    TYPE(InfoDef), POINTER :: Info
    TYPE(NodeBox) :: box
    INTEGER :: box_count, child_count
    INTEGER :: i,j
    INTEGER, DIMENSION(3,2) :: mGlobal
    TYPE(NodeBoxList), POINTER :: masterbox_list, last_masterbox
    TYPE(NodeBoxList), POINTER :: new_child_list, last_new_child
    INTEGER :: effective_finest_level


    nodelist => Nodes(level)%p
    child_count = 0
    NULLIFY(new_child_list, last_new_child)

    effective_finest_level = MIN(chandle%finest_level, MaxLevel)

    IF (level >= -1) THEN

       ! Obtain the number of boxes on this level that are stored in the chombo file.
       box_count = IO_GetDatasetElementCount(chandle%level_id, "boxes")

       ! Temporarily cache the child boxes in a list stored in the chombo handle.
       DO i = 1, box_count
          ! Obtain the next box and store it in the chombo handle structure.
          CALL IO_GetBoxFromChomboFile(chandle, "boxes", mGlobal)
          CALL IO_CacheNodeBox(chandle, mGlobal, Master)

          IF (level < effective_finest_level) THEN

             ! Attempt to find the node on this level that matches this box.
             box%mGlobal = mGlobal
             box%MPI_id = 0
             CALL FindNode(level, box, node)

             CALL IO_GetChildBoxCountFromChomboFile(chandle, child_count)

             ! Retrieve each of this box's child boxes.
             DO j = 1, child_count 
                ! Temporarily cache the child boxes in a list stored in the chombo handle.
                CALL IO_GetBoxFromChomboFile(chandle, "child_boxes", mGlobal)
                CALL IO_CacheChildNodeBox(chandle, mGlobal, Master)

                ! If the current box is on the master processor, then cache these child boxes
                ! separately so that they can be redistributed.
                IF (ASSOCIATED(node)) THEN
                   CALL AddNodeBoxToList(last_new_child, new_child_list)
                   last_new_child%self%mGlobal = mGlobal
                   last_new_child%self%MPI_id = Master
                END IF

             END DO

             ! If the current box is on this processor and has children, then
             ! redistribute the children.
             IF (ASSOCIATED(node) .AND. (child_count > 0)) THEN
                CALL ChomboReCreateChildren(node, level, child_count, new_child_list)

                ! Clear the cached child list so that it can be used for the next
                ! round of local child boxes.
                CALL ClearNodeBoxList(new_child_list)
                NULLIFY(last_new_child)
             END IF

          END IF

       END DO

       chandle%childbox_offset = 0
       chandle%childbox_count_offset = 0

    ELSE
       ! There is only one node on level -2, and it's on the master.
       node => nodelist%self
!       Info => node%Info
!       CALL IO_GetCostMapFromChomboFile(chandle, node%box%mGlobal, -2, Info%costmap)

       child_count = Read_HDF5_Attribute_Int("root_child_count", chandle%root_group_id)

       ! Temporarily cache the child boxes in a list stored in the chombo handle.
       DO i = 1, child_count 
          CALL IO_GetBoxFromChomboFile(chandle, "root_child_boxes", mGlobal)
          CALL IO_CacheChildNodeBox(chandle, mGlobal, Master)
       END DO

       ! Redistribute children on level n.
       IF (child_count > 0) THEN
          CALL ChomboReCreateChildren(node, level, child_count, chandle%current_child_box)
          chandle%current_child_box => chandle%last_child_box
       END IF

    END IF

    CALL ClearChomboHandleOffsets(chandle)

  END SUBROUTINE IO_ReadInLevelData

  !> A recursive subroutine that reads in chombo restart data for the given level.
  !! @param chandle An active chombo handle.
  !! @param level The level being read in.
  SUBROUTINE ChomboReloadLevel(level)

!    USE TreeLevelComms
!    USE DataLevelOps


    TYPE(ChomboHandle), POINTER :: chandle
    INTEGER :: level

    TYPE(StageMessageGroup), POINTER :: smg_recv_grids, smg_send_data
    TYPE(PackedMessage), POINTER :: pm_send_grids, pm_recv_data, message
    INTEGER :: n, i, message_size, status(MPI_STATUS_SIZE)
    INTEGER :: effective_finest_level
    INTEGER :: DummyInt, iErr
    
    chandle=>curr_handle

!    IF (MPI_id == 0)  PRINT *, "Starting level ", level, "..."

    ! Determines the highest level to which Chombo refinements will be pursued.
    ! Essentially, this is the highest level of data that we will attempt to pull 
    ! from the Chombo file.
!    effective_finest_level = MIN(chandle%finest_level, MaxLevel)

    NULLIFY(smg_recv_grids, smg_send_data, pm_send_grids, pm_recv_data)

    IF (level >= -1) THEN
       CALL IO_OpenCurrentLevel(chandle, level)
    END IF

    IF (MPI_ID == MASTER) THEN

       ! Open level handles for level.
!       IF (level >= -1)  CALL IO_OpenCurrentLevel(chandle, level)

       ! **** 
       !      IF YOU ARE TRYING TO FOLLOW THE THREAD OF THE RESTART PROCESS, IT STARTS
       !      HERE, AT LEVEL -2.
       ! ****
       ! Obtain grid and child grid data for this level.  This grid information is
       ! attached to the ChomboHandle structure for the time being.  At this point, 
       ! the master also distributes the child grids associated with its own nodes
       ! on this level.
       CALL IO_ReadInLevelData(chandle, level)

       IF (level >= -1) THEN
          ! Each worker now sends its grids for this level to the master.  This
          ! information is used to assign MPI_ids for each grid that the master
          ! has read in from the Chombo file.
          DO i=1,MPI_NP-1
             CALL MPI_RECV(message_size, 1, MPI_INTEGER, i, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, status, iErr)
             CALL CreatePackedMessage(level, i, TRANSMIT_IO_WORKER_GRIDS, STAGE_RECV, message, message_size)
             CALL MPI_SEND(message_size, 1, MPI_INTEGER, i, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, iErr)          
             CALL ReceivePackedMessage(message)
             CALL IO_UnparseGridsFromWorkers(chandle, message)
             CALL DestroyPackedMessage(message)
          END DO
       END IF
!             CALL IO_PostRecvGridsFromWorkers(smg_recv_grids, level)
 !            CALL IO_CompRecvGridsFromWorkers(chandle, smg_recv_grids)
!          END DO
!       END IF

       ! Read in one box at a time from chombo, figure out where it lives, and send it.
       ! Now that they have the appropriate MPI_ids associated, the boxes read
       ! in from the Chombo file can be used to retrieve the data for this level.
       ! These routines pull q, aux, and costmap data from the file and send it to
       ! the appropriate processors.
       !
       ! Child grids for workers are also packed and sent by these processors; the
       ! master processor has already distributed its child grids at this point, so
       ! it just copies over the InfoDef data from the Chombo files and ignores the
       ! child grids.
       CALL IO_MpiSendDataToWorkers(chandle, level)

       ! Clear all the cached boxes from the ChomboHandle structure.
       CALL IO_ClearAllNodeBoxes(chandle)
       CALL IO_ClearAllChildBoxes(chandle)

       ! Close level handles, since no further level data will be needed from the file.
!       IF (level >= -1)  CALL IO_CloseCurrentLevel(chandle)

    ELSE
       ! Send grids to master to let it know what blocks of data to send.  This information
       ! will be used on the master to assign MPI_ids to the boxes it reads in from the file.
       IF (level >= -1) THEN
          CALL IO_PostSendGridsToMaster(pm_send_grids, level)
          CALL IO_CompSendGridsToMaster(pm_send_grids)
       END IF

       ! Receive data from master.  This includes child boxes that will be transmitted
       ! to the children of this processor's nodes.
       ! CompRecvDataFromMaster() also copies the data to the InfoDef structures on this level.
!       CALL IO_PostRecvDataFromMaster(pm_recv_data, level, chandle)
       CALL IO_CompRecvDataFromMaster(level, chandle)
    END IF

    ! Close level handles, since no further level data will be needed from the file.
    IF (level >= -1)  CALL IO_CloseCurrentLevel(chandle)

 END SUBROUTINE ChomboReloadLevel


  !> Begins the process of restarting from the nframe-th chombo file.
  !! @param nframe The frame number to restart from.
  SUBROUTINE ChomboRestartInit(nframe)

    INTEGER :: nframe
    INTEGER :: n

    CHARACTER(LEN=I_FILENAME_LENGTH) :: s_filename
!    TYPE(ChomboHandle), POINTER :: chandle
    ! Retrieve the nframe-th chombo file name.
    WRITE(s_filename, '(A10,I5.5,A4)') 'out/chombo', nframe, '.hdf'

    ! Create a chombo handle to manage the object.
    CALL CreateChomboHandle(s_filename, curr_handle, CHOMBO_HANDLE_READ)

    current_time = Read_HDF5_Attribute_Double("time", curr_handle%root_group_id)
    olddt = Read_HDF5_Attribute_Double("olddt", curr_handle%root_group_id)
    overall_maxspeed = Read_HDF5_Attribute_Double("overall_maxspeed", curr_handle%root_group_id)
    maxcfl=Read_HDF5_Attribute_Double("maxcfl", curr_handle%root_group_id)
    levels(:)%tnow = current_time
    levels(:)%dt = 0.d0
    levels(:)%step = 1
    RestartLevel=min(MaxLevel,curr_handle%finest_level)
    NrInputVars = READ_HDF5_Attribute_Int("num_components", curr_handle%root_group_id)
!    IF (MPI_ID == 0) write(*,*) 'found ', NrInputVars, ' variables', NrVars
    IF (NrInputVars < NrVars) THEN
       IF (MPI_ID == 0) PRINT*, 'Error - not enough variables in chombo file fro restart'
       STOP
    END IF
          
 END SUBROUTINE ChomboRestartInit

 SUBROUTINE ChomboRestartFinalize(nframe)
    INTEGER :: nframe
    ! close the chombo handle.
    CALL CloseChomboHandle(curr_handle)
 END SUBROUTINE ChomboRestartFinalize



!> Creates a group for particle data within the Chombo file and populates it with the contents of the SinkParticles list.
!! @param chandle A chombo file handle.
SUBROUTINE WritePointGravityObjects(chandle)

    USE PointGravitySrc, ONLY: FirstPointGravityObj, PointGravityDef, PointGravity_CountObjects, &
                               PointGravity_InitChomboDatasets, PointGravity_WriteObjectToChombo

    TYPE(ChomboHandle), POINTER :: chandle

	TYPE(PointGravityDef), POINTER :: gravity_object
	INTEGER :: i_err
    INTEGER :: obj_count
    INTEGER :: iFixed


    ! Open a new particles group within the root level
    CALL h5gcreate_f(chandle%root_group_id, "point_gravity_objects", chandle%source_group_id, i_err)
    CALL CatchHDF5Error("WritePointGravityObjects::h5gcreate_f", i_err)

    chandle%source_offset = 0

    ! Retrieve the number of point gravity objects from the object list.
    obj_count = PointGravity_CountObjects()

    ! Store the number of objects.
    CALL Add_HDF5_Attribute_Int("num_objects", chandle%source_group_id, obj_count)

    ! If there are any gravity objects, then create datasets to store their data.  Otherwise,
	 ! don't create the datasets (since there's no need to take up file space).
    IF (obj_count > 0) THEN

        CALL PointGravity_InitChomboDatasets(chandle, obj_count)

        gravity_object => FirstPointGravityObj
	 
        ! Loop over the list of point gravity objects and write their data to the new component datasets.
        DO WHILE (ASSOCIATED(gravity_object))

            CALL PointGravity_WriteObjectToChombo(chandle, gravity_object)

            gravity_object => gravity_object%next

        END DO

    END IF

    ! Close the source term group.
    CALL h5gclose_f(chandle%source_group_id, i_err)
    CALL CatchHDF5Error("WritePointGravityObjects::h5gclose_f", i_err)

END SUBROUTINE WritePointGravityObjects

!> Creates a group for cooling data within the Chombo file and populates it with the contents of the cooling objects list.
!! @param chandle A chombo file handle.
SUBROUTINE WriteCoolingObjects(chandle)

    USE CoolingSrc, ONLY: CoolingDef, Cooling_CountObjects, firstcoolingobj, &
                          Cooling_InitChomboDatasets, Cooling_WriteObjectToChombo

    TYPE(ChomboHandle), POINTER :: chandle

	TYPE(CoolingDef), POINTER :: cooling_object
	INTEGER :: i_err
    INTEGER :: obj_count
    INTEGER :: iFixed


    ! Open a new particles group within the root level
    CALL h5gcreate_f(chandle%root_group_id, "cooling_objects", chandle%source_group_id, i_err)
    CALL CatchHDF5Error("WriteCoolingObjects::h5gcreate_f", i_err)

    chandle%source_offset = 0

    ! Retrieve the number of cooling objects from the object list.
    obj_count = Cooling_CountObjects()

    ! Store the number of objects.  The INT() is required to recast obj_count from an HSIZE_T
    ! type for the function definition.
    CALL Add_HDF5_Attribute_Int("num_objects", chandle%source_group_id, obj_count)

    ! If there are any cooling objects, then create datasets to store their data.  Otherwise,
	 ! don't create the datasets (since there's no need to take up file space).
    IF (obj_count > 0) THEN

        ! If there are cooling objects to store, then create datasets for them.
        CALL Cooling_InitChomboDatasets(chandle, obj_count)

        cooling_object => firstcoolingobj
	 
        ! Loop over the list of sink particles and write their data to the new component datasets.
        DO WHILE (ASSOCIATED(cooling_object))

            ! Writes the current object to the data file.  Also increments the source offset.
            CALL Cooling_WriteObjectToChombo(chandle, cooling_object)

            cooling_object => cooling_object%next
        END DO

    END IF

    ! Close the source term group.
    CALL h5gclose_f(chandle%source_group_id, i_err)
    CALL CatchHDF5Error("WriteCoolingObjects::h5gclose_f", i_err)

END SUBROUTINE WriteCoolingObjects

END MODULE IOChombo

