!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    io_comms.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file io_comms.f90
!! @brief Main file for module IOComms

!> @defgroup IOComms IO Communications
!! @brief Provides communication routines for getting data to the master processor for I/O.
!! @ingroup IO

!> Provides communication routines for getting data to the master processor for I/O.
!! @ingroup IOComms
!! @author Brandon D. Shroyer
!! @date 8-29-2010
MODULE IOComms

   USE TreeDeclarations
   USE DataDeclarations
   USE GlobalDeclarations
   USE CommunicationDeclarations
   USE TreeLevelComms
   USE ChomboDeclarations
   USE MessageDeclarations
   USE MpiPacking
   USE TreeNodeOps
   USE DataInfoOps
   USE IOParsing
   USE ProcessingDeclarations

   IMPLICIT NONE

   !    INCLUDE 'mpif.h'
   PRIVATE

   PUBLIC IO_ScheduleSendFrameData, IO_GetDatasetSizes, IO_ScheduleSendDomainData

   PUBLIC IO_PostSendGridsToMaster, IO_CompSendGridsToMaster
   PUBLIC IO_PostRecvGridsFromWorkers, IO_UnparseGridsFromWorkers
   PUBLIC IO_MpiSendDataToWorkers
   PUBLIC IO_CompRecvDataFromMaster
   PUBLIC GetFinestLevel, IO_CalcMessageCost
   PUBLIC IO_UnparseLevelStatistics

   !    INTEGER, PUBLIC, PARAMETER :: TRANSMIT_ROOT_DATA=13000
   !    INTEGER, PUBLIC, PARAMETER :: TRANSMIT_DOMAIN_DATA=14000
   !    INTEGER, PUBLIC, PARAMETER :: TRANSMIT_FRAME_DATA=15000

   !    INTEGER, PUBLIC, PARAMETER :: TRANSMIT_IO_WORKER_GRIDS = 16000
   !    INTEGER, PUBLIC, PARAMETER :: TRANSMIT_IO_WORKER_DATA = 17000

   !    INTEGER, PUBLIC, PARAMETER :: IO_NODECOUNT = 1
   INTEGER, PUBLIC, PARAMETER :: IO_Q_SIZE = 1
   INTEGER, PUBLIC, PARAMETER :: IO_AUX_SIZE = 2
   INTEGER, PUBLIC, PARAMETER :: IO_CHILDCOUNT = 3
   INTEGER, PUBLIC, PARAMETER :: IO_NEXTLEVELCOST = 4
   INTEGER, PUBLIC, PARAMETER :: IO_LEVEL_STAT_COUNT = 4

   INTEGER, PUBLIC, PARAMETER :: IO_LEVEL_STAT_BYTES = 16    ! 4 integers, at 4 bytes apiece.

CONTAINS




   !> Sends the node and grid data for a given level to the master processor.
   !! @param level The level of the data to be sent.
   !! @param buffer_totals A 3-element integer array holding the level's total q cells, aux cells,
   !! and child nodes.
   !! @param finest_level Finest Level

   SUBROUTINE IO_ScheduleSendFrameData(level, finest_level, buffer_size, buffer_totals)

      INTEGER :: level
      INTEGER :: finest_level
      INTEGER :: buffer_size
      INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT) :: buffer_totals

      TYPE(NodedefList), POINTER :: node_list, child_list
      TYPE(PackedMessage), POINTER :: message
      TYPE(Nodedef), POINTER :: node
      TYPE(InfoDef), POINTER :: Info
      INTEGER :: status(MPI_STATUS_SIZE)
      INTEGER :: iErr
      INTEGER :: DummyInt

      node_list => Nodes(level)%p
      NULLIFY(message, node, Info)

      ! Create two packed messages to send to the master processor--one for the tree data, 
      ! one for the grid data.
      CALL CreatePackedMessage(level, &
           Master, &
           TRANSMIT_FRAME_DATA, &
           STAGE_SEND, &
           message, &
           buffer_size)

      ! Pack the size of the given level's cell-centered and face-centered data for this processor,
      ! as well as the number of child nodes on this processor.  This will be used on the master
      ! processor to extend the data set.
      CALL PackData(message, buffer_totals)

      ! Pack the pertinent node data from each node into the buffers.
      DO WHILE (ASSOCIATED(node_list))
         node => node_list%self
         Info => node%Info
         CALL IO_ParseRemoteNode(message, node)

         IF (level < finest_level)  CALL IO_ParseNodeChildren(message, node)

         CALL IO_ParseRemoteGrid(message, Info, finest_level)
         node_list => node_list%next
      END DO

      ! Pack a termination box, signaling to the receiving unparsing routine
      ! that there are no more nodes or grids in this message. 
      CALL PackTerminationBox(message)

      ! The call to destroy will trigger the message to be sent.  However we don't want every processor to send a message at once since on some systems this will cause the eager messaging to overload the Master processor...  So first wait for a signal indicating the Master proc is ready...                            
      CALL MPI_RECV(DummyInt, 1, MPI_INTEGER, MASTER, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, status, iErr)

      CALL DestroyPackedMessage(message)

   END SUBROUTINE IO_ScheduleSendFrameData

   !> Packs and sends boxes and cost maps from level -1 to master.
   !! @param buffer_totals A 3-element integer array holding the level's total q cells, aux cells,
   !!                      and child nodes.
   SUBROUTINE IO_ScheduleSendDomainData(buffer_totals)

      TYPE(NodedefList), POINTER :: node_list, child_list
      TYPE(PackedMessage), POINTER :: message
      TYPE(Nodedef), POINTER :: node
      TYPE(InfoDef), POINTER :: Info
      INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT) :: buffer_totals

      node_list => Nodes(CHOMBO_DOMAIN_LEVEL)%p
      NULLIFY(message)

      ! Create a packed message to send cost maps from level -1 to the master processor.
      CALL CreatePackedMessage(CHOMBO_DOMAIN_LEVEL, &
           Master, &
           TRANSMIT_DOMAIN_DATA, &
           STAGE_SEND, &
           message)

      ! Pack the cell-centered and face-centered data counts for this level, as well as
      ! the number of children associated with this level.
      CALL PackData(message, buffer_totals)

      ! Pack the pertinent node data from each node into the buffers:
      !    --the domain node
      !    --the domain node's children
      !    --the domain's associated costmap.
      DO WHILE (ASSOCIATED(node_list))
         CALL IO_ParseRemoteNode(message, node_list%self)
         CALL IO_ParseNodeChildren(message, node_list%self)
         !       CALL IO_ParseLowLevelCostMap(message, node_list%self)
         node_list => node_list%next
      END DO

      ! Pack a termination box, signaling to the receiving unparsing routine
      ! that there are no more cost maps in this message. 
      CALL PackTerminationBox(message)

      ! We don't really need to keep these messages around once we're done with them, so 
      ! we can just destroy the nodes instead of closing them.
      CALL DestroyPackedMessage(message)

   END SUBROUTINE IO_ScheduleSendDomainData

   !> Use collective MPI communication routines to aggregate dataset information on Master.
   !! @param level The level of the data being collected.
   !! @param dataset_sizes An optional 1D integer array that will hold the output variables.
   !! @param finest_level Finest Level
   SUBROUTINE IO_GetDatasetSizes(level, finest_level, dataset_sizes)

      INTEGER :: level
      INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT), INTENT(OUT) :: dataset_sizes
      INTEGER :: finest_level

      TYPE(InfoDef), POINTER :: Info
      INTEGER :: iErr
      INTEGER :: total_nodes, node_buf
      INTEGER :: child_nodes, child_buf
      INTEGER :: next_level_cost
      !    INTEGER :: this_level_cost
      INTEGER :: q_size, q_buf
      INTEGER :: aux_size, aux_buf
      TYPE(NodedefList), POINTER :: node_list
      TYPE(NodeDefList), POINTER :: next_list


      node_list => Nodes(level)%p
      NULLIFY(next_list)

      total_nodes = 0
      child_nodes = 0
      q_size = 0
      aux_size = 0
      dataset_sizes = 0

      node_buf = 0
      child_buf = 0
      q_buf = 0
      aux_buf = 0

      !    this_level_cost = 0
      next_level_cost = 0

      ! Loop over the level and accumulate the following statistics:
      !    - child node count (this can be reused in the next pass of the Chombo write algorithm
      !      as the level count).
      !    - size of the q array (essentially the amount of hydrodynamic, tracer, and elliptic data).
      !    - size of the aux fields that will be stored (in a non-MHD problem, this will be 0).
      DO WHILE (ASSOCIATED(node_list))

         Info => node_list%self%Info

         ! Add this node's children to the number of child nodes, and add the childrens' cost to
         ! the cost of the next level.
         IF (level < finest_level)  child_nodes = child_nodes + NodeCount(node_list%self%children)

         IF (level >= 0) THEN
            q_size = q_size + PRODUCT(Info%mX) * (NrVars+NrDiagnosticVars)

            ! We don't actually want to send the full aux array, since there's that one
            ! corner that doesn't get used anyway.
            IF (MaintainAuxArrays) THEN
               aux_size = aux_size + (Info%mX(1) + 1) * Info%mX(2) * Info%mX(3) + &
                    Info%mX(1) * (Info%mX(2) + 1) * Info%mX(3)

               IF (nDim == 3)  aux_size = aux_size + Info%mX(1) * Info%mX(2) * (Info%mX(3) + 1)
            END IF

         END IF

         !       this_level_cost = this_level_cost + IO_CalcMessageCost(node_list%self, level, finest_level)

         node_list => node_list%next
      END DO

      IF(level < finest_level) THEN
         next_list => Nodes(level+1)%p

         DO WHILE (ASSOCIATED(next_list))
            next_level_cost = next_level_cost + IO_CalcMessageCost(next_list%self, level+1, finest_level)
            next_list => next_list%next
         END DO
      END IF
      ! MPI_reduce calls are blocking calls, so there should be no need for barriers.
      ! [BDS][20100928]:  This is tremendously inefficient; I hope to figure out something better some day.
      !        IF (level >= 0) THEN
      !            CALL MPI_REDUCE(q_size, q_buf, 1, MPI_INTEGER, MPI_SUM, Master, MPI_COMM_WORLD, ierr)
      !            CALL MPI_REDUCE(aux_size, aux_buf, 1, MPI_INTEGER, MPI_SUM, Master, MPI_COMM_WORLD, ierr)
      !        END IF

      ! Only reduce child node counts on the non-finest levels, since the finest level has no children.
      !        IF (level < finest_level) &
      !            CALL MPI_REDUCE(child_nodes, child_buf, 1, MPI_INTEGER, MPI_SUM, Master, MPI_COMM_WORLD, iErr)

      ! Return the level statistics.  Note that this parameter should only be present in calls on the
      ! master processor.
      !        IF (PRESENT(dataset_sizes))  dataset_sizes = (/ q_buf, aux_buf, child_buf /)
      dataset_sizes = (/ q_size, aux_size, child_nodes, next_level_cost /)

   END SUBROUTINE IO_GetDatasetSizes

   !> Posts receives for grids passed from workers as part of the Chombo restart process.
   !! @param sm_group A StageMessageGroup pointer to be initialized and used.
   !! @param level The level of the information being transferred.
   SUBROUTINE IO_PostRecvGridsFromWorkers(sm_group, level)

      TYPE(StageMessageGroup), POINTER :: sm_group
      INTEGER :: level

      TYPE(Nodedef), POINTER :: node
      TYPE(PackedMessage), POINTER :: message
      INTEGER :: n

!      CALL CreatePackedMessage(level, proc_id, TRANSMIT_IO_WORKER_GRIDS, STAGE_RECV, message)

!      IF (MPI_np > 1)  CALL CreateMessageGroup(sm_group, TRANSMIT_IO_WORKER_GRIDS, STAGE_RECV, level)

      ! Create a message for each worker processor.
      
      DO n = 1, MPI_np - 1
         CALL ExtractMessageFromGroup(sm_group, n, message)
      END DO

   END SUBROUTINE IO_PostRecvGridsFromWorkers

   !> Receives and processes grids passed from workers as part of the Chombo restart process.
   !! @param chandle An active Chombo handle.
   !! @param sm_group A StageMessageGroup pointer to be initialized and used.
   SUBROUTINE IO_UnparseGridsFromWorkers(chandle, message)

      TYPE(ChomboHandle), POINTER :: chandle
      TYPE(StageMessageGroup), POINTER :: sm_group

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(3,2) :: box_array
      TYPE(NodeBox), POINTER :: box


      ! Extract box array objects from packed message and replace their Chombo counterparts'
      ! MPI rank.  Stop extraction upon receiving a termination box.
      DO WHILE (GetNextBox(message, box_array))
         CALL CreateNodeBox(box_array, box, message%remote_proc)
         CALL MatchNodeBoxMpiRank(chandle, box)
         CALL DestroyNodeBox(box)
      END DO

   END SUBROUTINE IO_UnparseGridsFromWorkers

   !> Attempts to locate the counterpart of mod_box within the Chombo handle and replace its MPI_id.
   !! @param chandle An active Chombo handle.
   !! @param mod_box A NodeBox object whose MPI rank will replace the one in its Chombo counterpart.
   SUBROUTINE MatchNodeBoxMpiRank(chandle, mod_box)

      TYPE(ChomboHandle), POINTER :: chandle
      TYPE(NodeBox), POINTER :: mod_box

      TYPE(NodeBoxList), POINTER :: box_list
      TYPE(NodeBox), POINTER :: box


      box_list => chandle%box_list

      DO WHILE (ASSOCIATED(box_list))
         box => box_list%self

         ! If box is the counterpart of mod_box, then replace its MPI_id and exit.
         IF (ALL(mod_box%mGlobal == box%mGlobal)) THEN 
            box%MPI_id = mod_box%MPI_id
            EXIT
         END IF

         box_list => box_list%next
      END DO

   END SUBROUTINE MatchNodeBoxMpiRank

   !> Returns true if the dimensions of the first box lie entirely within the dimensions of the outside box.
   !! @param inside_array The box that (we hope) is inside the outside box.
   !! @param outside_array The box that (we hope) contains the inside box.
   !! @param inside_level The level of the inside box.
   !! @param outside_level The level of the outside box.
   !! @details This is kind of a crummy design, in that the boxes may be no more than one level apart.  I hope to improve it.
   LOGICAL FUNCTION IsContainedWithin(inside_array, inside_level, outside_array, outside_level)

      INTEGER, DIMENSION(MAX_DIMS, 2) :: inside_array
      INTEGER :: inside_level
      INTEGER, DIMENSION(MAX_DIMS, 2) :: outside_array
      INTEGER :: outside_level

      INTEGER, DIMENSION(MAX_DIMS, 2) :: work_array



      IF (inside_level <= outside_level) THEN
         PRINT *, "IsContainedWithin() error::inside level ", inside_level, " must be less than outside level ", &
              outside_level, "."
         STOP
      END IF

      work_array = 1

      work_array(1:nDim,1) = (inside_array(1:nDim,1)-1)/levels(outside_level)%CoarsenRatio + 1
      work_array(1:nDim,2) = (inside_array(1:nDim,2)  )/levels(outside_level)%CoarsenRatio

      IsContainedWithin = ALL(work_array(:,1) >= outside_array(:,1)) .AND. (ALL(work_array(:,2) <= outside_array(:,2)))

   END FUNCTION IsContainedWithin


   !> Packs and posts sends with grids for the master processor as part of the Chombo restart process.
   !! @param message A PackedMessageObject that will be initialized.
   !! @param level The level of the information being transferred.
   SUBROUTINE IO_PostSendGridsToMaster(message, level)

      TYPE(PackedMessage), POINTER :: message
      INTEGER :: level, accumulator
      INTEGER :: iErr
      INTEGER :: status(MPI_STATUS_SIZE)
      TYPE(NodedefList), POINTER :: nodelist


      NULLIFY(message)

      accumulator=0
      nodelist=>Nodes(level)%p
      DO WHILE (ASSOCIATED(nodelist))
         nodelist=>nodelist%next
         accumulator=accumulator+6*PACK_INTEGER_SIZE
      END DO
      accumulator=accumulator+6*PACK_INTEGER_SIZE

      CALL MPI_SEND(accumulator, 1, MPI_INTEGER, 0, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, iErr)
      CALL MPI_RECV(accumulator, 1, MPI_INTEGER, 0, TRANSMIT_IO_WORKER_GRIDS, MPI_COMM_WORLD, status, iErr)

      ! Create a new packed message object.  There is only one destination here (the master
      ! processor), so there's no need to create a full message group.
      CALL CreatePackedMessage(level, MASTER, TRANSMIT_IO_WORKER_GRIDS, STAGE_SEND, message, accumulator)

      nodelist=>Nodes(level)%p 

      IF (ASSOCIATED(message)) THEN

         ! Loop over the nodes on this level and pack their box dimensions into the outgoing message.
         DO WHILE (ASSOCIATED(nodelist))

            CALL PackData(message, nodelist%self%box%mGlobal)
            nodelist => nodelist%next
         END DO

         ! Terminate the box list by packing a termination box.
         CALL PackTerminationBox(message)

         ! Close the PackedMessage object to post the MPI send.
         CALL ClosePackedMessage(message)

      END IF

   END SUBROUTINE IO_PostSendGridsToMaster

   !> Receives and processes grids passed from workers as part of the Chombo restart process.
   !! @param message A PackedMessage object with uncompleted sends.
   SUBROUTINE IO_CompSendGridsToMaster(message)

      TYPE(PackedMessage), POINTER :: message
      INTEGER :: DummyInt, iErr

      ! Destroying the packed message will force the associated MPI send(s) to complete.
      IF (ASSOCIATED(message))  CALL DestroyPackedMessage(message)

   END SUBROUTINE IO_CompSendGridsToMaster




   !> Packs and posts sends with data for workers as part of the Chombo restart process.  Data for the master processor is simply copied to the appropriate InfoDef structure.
   !! @param chandle An active chombo handle.
   !! @param sm_group A StageMessageGroup object that will be initialized here.
   !! @param level The level of the information being transferred.  Should always be >= -1.
   SUBROUTINE IO_MpiSendDataToWorkers(chandle, level)

      TYPE(ChomboHandle), POINTER :: chandle
      INTEGER :: level

      TYPE(Nodedef), POINTER :: node
      TYPE(InfoDef), POINTER :: Info
      TYPE(PackedMessage), POINTER :: message
      TYPE(NodeBoxList), POINTER :: box_list, child_list
      TYPE(NodeBox), POINTER :: box, child_box
      REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: qdata, auxdata, costmap
      INTEGER :: mx, my, mz
      INTEGER :: i,j,k
      INTEGER :: effective_finest_level
      INTEGER :: proc_id
      INTEGER :: child_count

      INTEGER, TARGET, DIMENSION(0:MPI_np-1) :: bufsize_array
      INTEGER, POINTER, DIMENSION(:) :: proc_buffer_sizes
      INTEGER :: iErr

      effective_finest_level = MIN(chandle%finest_level, MaxLevel)

      proc_buffer_sizes => bufsize_array
      proc_buffer_sizes = 0
      child_count = 0

      ! If this is a multi-processor run, then create a message group and a message
      ! for each non-master processor.
      IF (MPI_np > 1) THEN

         proc_buffer_sizes => bufsize_array
         proc_buffer_sizes = 0

         ! Do a pass through the child lists to calculate the size required for each message's buffer.  The results
         ! will be stored in the array pointed to by proc_buffer_sizes
         !       CALL IO_SendDataToWorkers_LevelPrecalc(chandle, level, proc_buffer_sizes)

      END IF

      !   DO proc_id = 0, MPI_np - 1

      box_list => chandle%box_list
      child_list => chandle%child_box_list

      ! Clear all of the offsets in the chombo handle.
      CALL ClearChomboHandleOffsets(chandle)

      ! Create a packed message to send proc_id's data.
      !      IF (proc_id > Master)  CALL CreatePackedMessage(level, &
      !                                                      proc_id, &
      !                                                      TRANSMIT_IO_WORKER_DATA, &
      !                                                      STAGE_SEND, &
      !                                                      message, &
      !                                                      proc_buffer_sizes(proc_id))

      ! Go through the level's boxes, retrieving the associated data from the Chombo file
      ! and packing it into the appropriate processor's message.
      DO WHILE (ASSOCIATED(box_list))

         box => box_list%self

         mx = box%mGlobal(1,2) - box%mGlobal(1,1) + 1
         my = box%mGlobal(2,2) - box%mGlobal(2,1) + 1
         mz = box%mGlobal(3,2) - box%mGlobal(3,1) + 1

         IF (level < effective_finest_level) THEN
            CALL IO_GetChildBoxCountFromChomboFile(chandle, child_count)
         END IF
         ! This routine is only called on the master processor, so if this is a single-processor
         ! job then this conditional will never be true.
         IF ((box%MPI_id > Master)) THEN

            ! Pack this box's data into the message being sent to proc_id.
            CALL IO_SendDataToWorkers(chandle, level, box, child_list, child_count)

         ELSE ! Do local data transfers for master processor.

            ! Retrieve the node that matches the given box.  At this point, the node
            ! should exist already, so it's okay to throw a fit if it's not found.
            CALL StrictFindNode(level, box, node, "IO_MpiSendDataToWorkers(Master)")

            ! This is done both to advance the childbox offset and to obtain the size of the advance
            ! for the childbox_offset.
            IF (level < effective_finest_level) THEN

               ! Increment the childbox offsets and advance the child list so that worker processors do not get confused.
               DO i = 1, child_count
                  child_list => child_list%next
               END DO
            END IF

            IF (level >= 0) THEN
               Info => node%Info             

               ! Retrieve the cell-centered data from the chombo file.
               CALL IO_GetQDataFromChomboFile(chandle, box%mGlobal, qdata)
               Info%q(1:mx,1:my,1:mz,1:NrVars) = qdata(1:mx,1:my,1:mz,1:NrVars)
               DEALLOCATE(qdata)
               NULLIFY(qdata)

               ! For MHD problems, retrieve the face-centered data.    
               IF (MaintainAuxArrays) THEN
                  CALL IO_GetAuxDataFromChomboFile(chandle, box%mGlobal, auxdata)
                  Info%aux(1:mx+1,1:my,1:mz,1) =  auxdata(1:mx+1, 1:my, 1:mz, 1)
                  Info%aux(1:mx,1:my+1,1:mz,2) =  auxdata(1:mx, 1:my+1, 1:mz, 2)
                  IF (nDim == 3)  Info%aux(1:mx,1:my,1:mz+1,3) =  auxdata(1:mx, 1:my, 1:mz+1, 3)

                  CALL UpdateAux(Info, RESHAPE((/ 1, 1, 1, mx, my, mz /), (/3, 2/)))

                  DEALLOCATE(auxdata)
                  NULLIFY(auxdata)
               END IF

            END IF

         END IF   ! If (box%MPI_id > Master)

         box_list => box_list%next
      END DO

      DO i=1,MPI_NP-1
         CALL MPI_Send(TERMINATIONBOX, 6, MPI_INTEGER, i, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
      END DO

      ! Clear all of the offsets in the chombo handle.
      CALL ClearChomboHandleOffsets(chandle)

      NULLIFY(proc_buffer_sizes)

   END SUBROUTINE IO_MpiSendDataToWorkers



   !> Receives data from master during Chombo restart and stores it in the appropriate Info structures.
   !! @param message A PackedMessage object.  The level of this message should always be >= -1.
   !! @param chandle An active Chombo file handle.
   SUBROUTINE IO_CompRecvDataFromMaster(level, chandle) 

      TYPE(PackedMessage), POINTER :: message
      TYPE(ChomboHandle), POINTER :: chandle

      INTEGER, DIMENSION(3,2) :: mGlobal
      TYPE(NodeBox), POINTER :: remote_box, child_box
      TYPE(Nodedef), POINTER :: node
      TYPE(InfoDef), POINTER :: Info
      INTEGER :: mx, my, mz
      INTEGER :: child_count
      INTEGER :: n,level
      TYPE(NodeBoxList), POINTER :: box_list, last_box
      CHARACTER(LEN=28) :: fname
      INTEGER :: iErr
      INTEGER :: status(MPI_STATUS_SIZE)
      DO
         CALL MPI_RECV(mGlobal, 6, MPI_INTEGER, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr)
         IF (ALL(mGlobal==TERMINATIONBOX)) EXIT
         CALL IO_RecvDataFromMaster(level, mGlobal)
      END DO

   END SUBROUTINE IO_CompRecvDataFromMaster

   !> Calculates the cost in bytes of sending a node's contribution to a Chombo file.
   !! @param node The node whose cost is being sent.
   !! @param level The level of the node in question.
   !! @param finest_level The finest level of data this Chombo file will be recording.
   INTEGER FUNCTION IO_CalcMessageCost(node, level, finest_level)

      TYPE(NodeDef), POINTER :: node
      INTEGER :: level
      INTEGER :: finest_level

      INTEGER, DIMENSION(3) :: mX
      INTEGER :: accumulator


      ! The base cost is the cost of the node itself (as determined by IO_ParseNode()).
      accumulator = PACK_INTEGER_SIZE + PACK_BOX_SIZE



      IF (level >= 0) THEN
         ! Add the cost of node%box%mGlobal.  This is the first element sent by all grids.
         accumulator = accumulator + PACK_BOX_SIZE
         mX(:) = node%box%mGlobal(:,2) - node%box%mGlobal(:,1) + 1

         ! Calculate the cost of the cell-centered  data.
         accumulator = accumulator + PRODUCT(mX) * (NrVars+NrDiagnosticVars) * PACK_DOUBLE_SIZE

         ! If a value is given for face-centered variables, then calculate the cost of aux data.
         IF (MaintainAuxArrays) THEN
            IF (nDim == 3) THEN
               accumulator = accumulator + &
                    ((mX(1)+1)*mx(2)*mX(3) + mX(1)*(mx(2)+1)*mX(3) + mX(1)*mx(2)*(mX(3)+1)) * PACK_DOUBLE_SIZE
            ELSE
               accumulator = accumulator + ((mX(1)+1)*mx(2) + mX(1)*(mx(2)+1)) * PACK_DOUBLE_SIZE
            END IF
         END IF
      END IF

      ! On nodes that might have children, calculate the cost in bytes of sending the node's children.
      ! The extra PACK_INTEGER_SIZE accomodates the size of each node's child_count variable.
      IF (level < finest_level)  accumulator = accumulator + &
           PACK_INTEGER_SIZE + &                     ! child count
           NodeCount(node%children) * (PACK_BOX_SIZE+PACK_INTEGER_SIZE)  ! child boxes

      IO_CalcMessageCost = accumulator

   END FUNCTION IO_CalcMessageCost




   !> Retrieves a 1D integer array of level statistics from a packed message.
   !! @param message The message from which the data comes.
   !! @param level_stats The 1D integer array in which the data goes.
   SUBROUTINE IO_UnparseLevelStatistics(message, level_stats)

      TYPE(PackedMessage), POINTER :: message
      INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT) :: level_stats

      ! If the message is valid (associated), then extract the level statistics.
      ! Otherwise, print an error and die.
      IF (ASSOCIATED(message)) THEN
         CALL UnpackData(message, level_stats)
      ELSE
         PRINT *, "IO_UnparseLevelStatistics() error:  message not associated."
         STOP
      END IF

   END SUBROUTINE IO_UnparseLevelStatistics


END MODULE IOComms
