!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    io_control.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir io
!! @brief Contains modules for managing input and output

!> @defgroup IO Input/Output
!! @brief Contains modules for managing input and output

!> @file io_control.f90
!! @brief Main file for module IOControl

!> @defgroup IOControl IO Control
!! @brief Provides a format-agnostic interface for AstroBEAR to read and write data files.
!! @ingroup IO

!> Provides a format-agnostic interface for AstroBEAR to read and write data files.
!! @ingroup IOControl
!! @author Brandon D. Shroyer
!! @date 7-13-2010
MODULE IOControl

    USE GlobalDeclarations
    USE CommunicationDeclarations
    USE IODeclarations
    USE HDF5Declarations
    USE IOChombo
    USE IOComms
    USE IOOkc
    USE DataDeclarations
    USE Timing
    IMPLICIT NONE
    PRIVATE

!    INCLUDE 'mpif.h'

    PUBLIC IOInit, WriteDataFrame, IORestartInit, IORestartFinalize, IOReloadLevel, IOClose, PostIO
    
CONTAINS
    

   SUBROUTINE PostIO(frame)
      INTEGER :: frame
      CHARACTER(21) :: syscmd
      LOGICAL :: ex
      IF (MPI_ID == 0) THEN
         Inquire(file="postprocess.s", exist=ex)
         IF (ex) THEN
            write(syscmd,'(A16,I5.5)') './postprocess.s ', frame
            CALL System(syscmd)
         END IF
      END IF
   END SUBROUTINE PostIO
   
    !> Reads in any generic IO data contained in the io.data file.
    SUBROUTINE IOInit

        INTEGER :: iErr

        ! Read in scheme value.
!        READ(GLOBAL_DATA_HANDLE,NML=IOData,IOStat=iErr)
        
!        IF(iErr/=0) THEN
!           PRINT *, "IOInit() error:  unable to read ", GLOBAL_DATA_FILE, "."
!           STOP
!        END IF

        iDataFileType=IO_CHOMBO

        SELECT CASE (iDataFileType)
            CASE (IO_CHOMBO)
                CALL HDF5Init()
                t_io_cumulative = 0.d0

            CASE (IO_BOV)
                PRINT *, "WriteDataFrame() error:  BOV output format not yet implemented."
                STOP
                
            CASE DEFAULT
                PRINT "('WriteDataFrame() error: invalid data file output ', i2, '.')", iDataFileType
                STOP

        END SELECT
        
    END SUBROUTINE IOInit

    !> Closes any remaining files and releases any remaining I/O resources.
    SUBROUTINE IOClose

        SELECT CASE (iDataFileType)
            CASE (IO_CHOMBO)
                CALL HDF5Finalize()

            CASE (IO_BOV)
                PRINT *, "IOClose() error:  BOV output format not yet implemented."
                STOP
                
            CASE DEFAULT
                PRINT "('IOClose() error: invalid data file output ', i2, '.')", iDataFileType
                STOP

        END SELECT

    END SUBROUTINE IOClose
    
    !> Writes a data frame
    !! @param nframe the sequence number of the frame to be written.
    SUBROUTINE WriteDataFrame(nframe)

       USE MessageDeclarations, ONLY: TERMINATION_BOX_BYTES
       USE ChomboDeclarations, ONLY: CHOMBO_DOMAIN_LEVEL

       INTEGER :: nframe

       INTEGER :: level, n
       INTEGER :: finest_level
       INTEGER, DIMENSION(IO_LEVEL_STAT_COUNT) :: data_sizes
       INTEGER :: buffer_size

       CALL StartTimer(iWriteData, -2)

       IF (MPI_ID == 0) t_io_start = MPI_Wtime()

       ! The iDataFileType variable is read in from the GlobalData namelist.
       SELECT CASE (iDataFileType)

       CASE (IO_CHOMBO)

          !                finest_level = GetFinestLevel()
          finest_level = MaxLevel

          IF (MPI_id == Master) THEN

             ! Write the chombo file (timing the process).

             CALL MakeChomboFile(nframe)
             CALL MakeOKCFile(nframe)
             !                    t_end = MPI_Wtime()

             !                    t_io_cumulative = t_io_cumulative + t_end - t_start

          ELSE
             ! Non-master processors send their data to the master level-by-level; 
             ! first the tree data, then the grid data.

             data_sizes = 0

             DO level = -2, finest_level

                ! Add the overhead for a Chombo level transmission to the cost of the next level's data.
                IF (level > CHOMBO_DOMAIN_LEVEL)  buffer_size = data_sizes(IO_NEXTLEVELCOST) + &
                     TERMINATION_BOX_BYTES + &
                     IO_LEVEL_STAT_BYTES

                CALL IO_GetDatasetSizes(level, finest_level, data_sizes)

                IF (level > CHOMBO_DOMAIN_LEVEL) THEN
                   CALL IO_ScheduleSendFrameData(level, finest_level, buffer_size, data_sizes)
                ELSE IF (level == CHOMBO_DOMAIN_LEVEL) THEN
                   CALL IO_ScheduleSendDomainData(data_sizes)
                END IF
             END DO
          END IF

       CASE (IO_BOV)
          PRINT *, "WriteDataFrame() error:  BOV output format not yet implemented."
          STOP

       CASE DEFAULT
          PRINT "('WriteDataFrame() error: invalid data file output ', i2, '.')", iDataFileType
          STOP

       END SELECT

       ! print statistics on how much time is spent writing output.
       IF (MPI_id == Master) THEN
          t_io_end=MPI_WTime()           
          t_io_cumulative=t_io_end-t_io_start
          PRINT "('Time to make output file ', i4, ': ', f8.4, ' seconds.')", nframe,  t_io_end - t_io_start
!          PRINT "('Cumulative output file time: ', f11.4, ' seconds.')", t_io_cumulative
       END IF

       CALL StopTimer(iWriteData, -2)
    END SUBROUTINE WriteDataFrame

    !> Uses the contents of the frame restart_frame to reconstruct a simulation.
    !! @param nframe The frame being used to restart the problem.
    SUBROUTINE IORestartInit(nframe)

        INTEGER :: nframe



        IF (MPI_id == Master) t_io_start=MPI_WTIME()

        ! The iDataFileType variable is read in from the GlobalData namelist.
        SELECT CASE (iDataFileType)
            CASE (IO_CHOMBO)
                CALL ChomboRestartInit(nframe)

            CASE (IO_BOV)
                PRINT *, "IORestartInit() error:  BOV output format not yet implemented."
                STOP
                
            CASE DEFAULT
                PRINT "('IORestartInit() error: invalid data file output ', i2, '.')", iDataFileType
                STOP
                
        END SELECT

     END SUBROUTINE IORestartInit



    !> Uses the contents of the frame restart_frame to reconstruct a simulation.
    !! @param nframe The frame being used to restart the problem.
     SUBROUTINE IORestartFinalize(nframe)

        INTEGER :: nframe

        ! The iDataFileType variable is read in from the GlobalData namelist.
        SELECT CASE (iDataFileType)
            CASE (IO_CHOMBO)

                CALL ChomboRestartFinalize(nframe)

            CASE (IO_BOV)
                PRINT *, "IORestartFinalize() error:  BOV output format not yet implemented."
                STOP
                
            CASE DEFAULT
                PRINT "('IORestartFinalize() error: invalid data file output ', i2, '.')", iDataFileType
                STOP
                
        END SELECT

      ! print statistics on how much time is spent restarting.
      IF (MPI_id == 0) THEN
          t_io_end=MPI_WTime()           
          t_io_cumulative=t_io_end-t_io_start
          PRINT "('Time to restart from output file ', i4, ': ', f8.4, ' seconds.')", restart_frame,  t_io_end - t_io_start
!          PRINT "('Cumulative output file time: ', f8.4, ' seconds.')", t_io_cumulative          
       END IF

     END SUBROUTINE IORestartFinalize

    !> Uses the contents of the frame restart_frame to reconstruct a simulation.
    !! @param nframe The frame being used to restart the problem.
     SUBROUTINE IOReloadLevel(n)

        INTEGER :: n

        REAL(KIND=qPrec) :: t_start, t_end


        ! The iDataFileType variable is read in from the GlobalData namelist.
        SELECT CASE (iDataFileType)
            CASE (IO_CHOMBO)

                CALL ChomboReloadLevel(n)

            CASE (IO_BOV)
                PRINT *, "ReLoadLevel() error:  BOV output format not yet implemented."
                STOP
                
            CASE DEFAULT
                PRINT "('ReLoadLevel() error: invalid data file output ', i2, '.')", iDataFileType
                STOP
                
        END SELECT


     END SUBROUTINE IOReloadLevel





END MODULE IOControl

