!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    io_parsing.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @file io_parsing.f90
!! @brief Main file for module IOParsing

!> @defgroup IOParsing IO Parsing
!! @brief Provides parsing layer functionality for Chombo I/O system.
!! @ingroup IO

!> Provides parsing and unparsing routines for Chombo I/O.  These subroutines interface with the packing and scheduling layers.
!! @ingroup IOParsing
!! @author Brandon D. Shroyer
!! @date 2-15-2011
MODULE IOParsing
  
  USE ChomboDeclarations
  USE MessageDeclarations
  USE MPIPacking
  USE GlobalDeclarations
  USE TreeDeclarations
  USE DataDeclarations
  USE DataInfoOps
  USE CommunicationDeclarations
  USE ProcessingDeclarations
  IMPLICIT NONE

  PRIVATE

  !Restart parsing
  PUBLIC IO_SendDataToWorkers, IO_RecvDataFromMaster

  !Output frame parsing
  PUBLIC IO_ParseRemoteNode, IO_UnparseRemoteNode, IO_ParseRemoteGrid, IO_UnParseRemoteGrid, IO_ParseNodeChildren, IO_UnParseNodeChildren
CONTAINS

   !> Parses the data elements of a node relevant to the I/O routines.
   !! @param message A packed message object to store the node's data.
   !! @param node the Nodedef object to be parsed.
   SUBROUTINE IO_ParseRemoteNode(message, node)

      TYPE(PackedMessage), POINTER :: message
      TYPE(Nodedef), POINTER :: node

      CALL PackData(message, node%box%mGlobal)
      CALL PackData(message, node%box%MPI_ID)
      !    CALL PackList(message, node%proclist)
      !    CALL PackList(message, node%proctime)

   END SUBROUTINE IO_ParseRemoteNode

   !> Unparse the data for a single node from the given message.
   !! @param message The PackedMessage object from which we will extract data.
   !! @param node the Nodedef object to be unparsed.
   SUBROUTINE IO_UnparseRemoteNode(message, node)

      TYPE(PackedMessage), POINTER :: message
      TYPE(Nodedef), POINTER :: node

      INTEGER :: proc_id
      INTEGER, DIMENSION(3,2) :: mGlobal
      TYPE(NodeBox), POINTER :: box
      INTEGER, POINTER, DIMENSION(:) :: proclist
      REAL, POINTER, DIMENSION(:) :: proctime
      INTEGER :: iErr


      NULLIFY(node)

      !PRINT *
      !PRINT *, "*** IO_UnparseRemoteNode::starting ***"
      !PRINT *

      ! Get the coordinate box from the message.  This will fail on a non-coordinate,
      ! non-termination box (i.e., any box with a lower or upper bound of 0.
      IF (StrictGetNextBox(message, mGlobal, "IO_UnparseRemoteNode")) THEN

         !PRINT "('IO_UnparseRemoteNode::mGlobal = [', 6i4, '].')", mGlobal

         ! Retrieve the node's process ID.
         CALL UnpackData(message, proc_id)

         !PRINT "('IO_UnparseRemoteNode::proc_id = ', i1, '.')", proc_id

         ! Use the coordinates and MPI_id to construct a NodeBox.
         CALL CreateNodeBox(mGlobal, box, proc_id)

         ! If there are any proclist or proctime elements, these PackList calls
         ! will retrieve them.
         !       CALL UnpackList(message, proclist)
         !PRINT *, "IO_UnparseRemoteNode::proclist = ", proclist

         !       CALL UnpackList(message, proctime)

         !PRINT *, "IO_UnparseRemoteNode::proctime = ", proctime

         ! Allocate a new node object (we don't want to use AddNode here, since 
         ! this is a temporary structure to hold worker data and will not be taking
         ! up permanent residence here).
         ALLOCATE(node, STAT=iErr)

         IF (iErr /= 0) THEN
            PRINT *, "IO_UnparseRemoteNode() error:  failure to allocate node."
            STOP
         END IF

         ! Clear all the node pointers.
         CALL NullifyNodeFields(node)

         ! Assign the appropriate attributes to the node.
         node%box = box
         !       node%proclist => proclist
         !       node%proctime => proctime

         ! Nullify the pointers used to construct this node.
         ! DO NOT DEALLOCATE THEM, AS THIS WOULD DESTROY DATA.
         CALL DestroyNodeBox(box)
         NULLIFY(box, proclist, proctime)

      END IF
      !PRINT *, "IO_UnparseRemoteNode::done."
      !PRINT *

   END SUBROUTINE IO_UnparseRemoteNode

   !> Parses the data elements of a grid relevant to the I/O routines.
   !! @param message A packed message object to store the grid's data.
   !! @param Info the InfoDef object to be parsed.
   !! @param finest_level Finest Level
   SUBROUTINE IO_ParseRemoteGrid(message, Info, finest_level)

      TYPE(PackedMessage), POINTER :: message
      TYPE(InfoDef), POINTER :: Info
      INTEGER :: finest_level

      INTEGER :: mx, my, mz


      !PRINT *
      !PRINT "('*** IO_ParseRemoteGrid(', i2, ', ', i2, ')::starting ***')", MPI_id, Info%level
      !PRINT "('*** IO_ParseRemoteGrid(', i2, ', ', i2, ')::buffer size = ', i8, ' ***')", MPI_id, Info%level, message%buffer_size
      !PRINT *

      mx = Info%mX(1)
      my = Info%mX(2)
      mz = Info%mX(3)

      !PRINT "('IO_ParseRemoteGrid::Info%mGlobal = [', 6i4, '].')", Info%mGlobal
      !PRINT "('IO_ParseRemoteGrid::total doubles = ', i6, '.')", PRODUCT(Info%mX) * NrVars
      ! Pack the grid structure's global coordinates.
      CALL PackData(message, Info%mGlobal)

      ! Pack the cell-centered data.
      CALL PackData(message, Info%q(1:mx, 1:my, 1:mz, 1:NrVars))
      IF (NrDiagnosticVars > 0) THEN
         CALL PackData(message, Info%diagnostics(1:mx, 1:my, 1:mz, 1:NrDiagnosticVars))
         DEALLOCATE(Info%diagnostics)
         NULLIFY(Info%diagnostics)
      END IF


      !IF (ALL(Info%mGlobal == RESHAPE((/19, 19, 33, 26, 26, 38 /), (/3, 2/)))) THEN
      !    PRINT *, "Killing ParseRemoteGrid..."
      !    STOP
      !END IF

      ! Only gets executed if the problem is MHD.
      IF (MaintainAuxArrays) THEN
         ! face-centered data does not take up the entire aux cube, so we pack
         ! the data in slices along a single dimension.
         CALL PackData(message, Info%aux(1:mx+1, 1:my, 1:mz, 1))
         CALL PackData(message, Info%aux(1:mx, 1:my+1, 1:mz, 2))
         IF (nDim == 3)  CALL PackData(message, Info%aux(1:mx, 1:my, 1:mz+1, 3))
      END IF

      ! Pack costmap data if the level is not the finest level (otherwise there's 
      ! no need for costmap data).
      !    IF (message%level < finest_level)  CALL PackData(message, Info%costmap(1:mx, 1:my, 1:mz, 1:2))i
      !PRINT "('*** IO_ParseRemoteGrid(', i2, ', ', i2, ')::done ***')", MPI_id, Info%level
      !PRINT *, "done."

   END SUBROUTINE IO_ParseRemoteGrid

   !> Unparse the data for a single grid from the given message.
   !! @param message The PackedMessage object from which we will extract data.
   !! @param Info the InfoDef object to be unparsed.
   !! @param finest_level Finest Level
   SUBROUTINE IO_UnparseRemoteGrid(message, Info, finest_level)

      TYPE(PackedMessage), POINTER :: message
      TYPE(InfoDef), POINTER :: Info
      INTEGER :: finest_level

      INTEGER, DIMENSION(3,2) :: mGlobal
      INTEGER :: iErr
      INTEGER :: mx, my, mz
      INTEGER :: i,j,k


      !PRINT *
      !PRINT "('*** IO_UnparseRemoteGrid(', i2, ', ', i2, ') starting ***')", message%remote_proc, message%level
      !PRINT "('*** IO_UnparseRemoteGrid(', i2, ', ', i2, ')::buffer size = ', i8, ' ***')", message%remote_proc, message%level, message%buffer_size
      ! Unpack global coordinates of grid.
      CALL UnpackData(message, mGlobal)

      !PRINT "('IO_UnparseRemoteGrid::mGlobal = [', 6i4, '].')", mGlobal

      ! Calculate local dimensions from global coordinates.
      mx = mGlobal(1,2) - mGlobal(1,1) + 1
      my = mGlobal(2,2) - mGlobal(2,1) + 1
      mz = mGlobal(3,2) - mGlobal(3,1) + 1

      !PRINT "('IO_UnparseRemoteGrid::[mx, my, mz] = [', 3i4, ']; product = ', i6, '.')", mx, my, mz, mx*my*mz*NrVars

      ! Construct a node info structure from the coordinates we have obtained.
      !    CALL InitInfo(Info, message%level, mGlobal)
      ALLOCATE(Info)
      ALLOCATE(Info%q(1:mx, 1:my, 1:mz, 1:NrVars))
      ! Unparse the cell-centered data into the new InfoDef structure.
      CALL UnpackData(message, Info%q)
      IF (NrDiagnosticVars > 0) THEN
         ALLOCATE(Info%diagnostics(1:mx,1:my,1:mz,1:NrDiagnosticVars))
         CALL UnpackData(message, Info%diagnostics)
      END IF

      ! Only runs on MHD jobs.
      IF (MaintainAuxArrays) THEN
         ! face-centered data does not take up the entire aux cube, so we unpack
         ! the data in slices along a single dimension, with a different MHD flux variable
         ! associated with each slice.
         IF (nDim == 2) THEN
            ALLOCATE(Info%aux(1:mx+1,1:my+1,1,2))
         ELSE
            ALLOCATE(Info%aux(1:mx+1,1:my+1,1:mz+1,3))
         END IF
         CALL UnpackData(message, Info%aux(1:mx+1, 1:my, 1:mz, 1))
         CALL UnpackData(message, Info%aux(1:mx, 1:my+1, 1:mz, 2))
         IF (nDim == 3)  CALL UnpackData(message, Info%aux(1:mx, 1:my, 1:mz+1, 3))
      END IF

      ! Unpack costmap data to Info structure if we are not on the finest level.
      !    IF (message%level < finest_level)  CALL UnpackData(message, Info%costmap(1:mx, 1:my, 1:mz, 1:2))
      !PRINT "('*** IO_UnparseRemoteGrid(', i2, ', ', i2, ') done ***')", message%remote_proc, message%level
      !PRINT *

   END SUBROUTINE IO_UnparseRemoteGrid


   !> Parses the children of the given node.
   !! @param message A packed message object to store the node's child data.
   !! @param node the Nodedef object whose children will be parsed.
   SUBROUTINE IO_ParseNodeChildren(message, node)

      TYPE(PackedMessage), POINTER :: message
      TYPE(Nodedef), POINTER :: node

      TYPE(NodedefList), POINTER :: child_list



      ! Pack the number of children this node has.
      CALL PackData(message, NodeCount(node%children))

      child_list => node%children

      ! Loop over the list and pack each of the node's children.
      DO WHILE (ASSOCIATED(child_list))
         CALL IO_ParseRemoteNode(message, child_list%self)
         child_list => child_list%next
      END DO


   END SUBROUTINE IO_ParseNodeChildren

   !> Retrieves child data from a PackedMessage object and stores it as temporary child
   !!        nodes within the passed-in node strucure.
   !! @param message A PackedMessage object from which the routine reads.
   !! @param child_count An integer that will be set to the number of children and returned.
   !! @param node The node that will carry references to the retrieved children.
   SUBROUTINE IO_UnparseNodeChildren(message, child_count, node)

      TYPE(PackedMessage), POINTER :: message
      INTEGER :: child_count
      TYPE(Nodedef), POINTER :: node

      INTEGER :: nchildren
      INTEGER :: n
      TYPE(Nodedef), POINTER :: staging_node


      ! Retrieve the number of children packed into the message.
      CALL UnpackData(message, child_count)

      NULLIFY(node%children)
      NULLIFY(node%lastchild)
      DO n = 1, child_count
         NULLIFY(staging_node)
         ! Unpack the node data to a temporary node structure.
         CALL IO_UnparseRemoteNode(message, staging_node)

         ! Add the new temporary child node to the given node's child list.
         CALL AddNodeToList(staging_node, node%lastchild, node%children) 
      END DO

   END SUBROUTINE IO_UnparseNodeChildren






  !> Packs Chombo HDF5 data for a single grid into a message to be sent to a worker processor.
  !! @param chandle An active chombo handle.
  !! @param sm_group The PackedMessage object that will be handling this grid's data.
  !! @param box The coordinates of the grid whose data will be packed.
  !! @param child_list A pointer to the Chombo file's list of child boxes for this level.  This pointer need not start at the beginning of the list.
  !! @param child_count The number of children associated with the node corresponding to this box.
  SUBROUTINE IO_SendDataToWorkers(chandle, level, box, child_list, child_count)

    TYPE(ChomboHandle), POINTER :: chandle
    TYPE(NodeBoxList), POINTER :: child_list
    INTEGER :: child_count
    INTEGER :: n
    TYPE(Nodedef), POINTER :: node
    TYPE(InfoDef), POINTER :: Info
    TYPE(NodeBox), POINTER :: box, child_box
    REAL(KIND=qPrec), DIMENSION(:,:,:,:), POINTER :: qdata, auxdata
    INTEGER :: mx, my, mz
    INTEGER :: i,j,k
    INTEGER :: effective_finest_level
    INTEGER :: iErr
    INTEGER :: level

    effective_finest_level = MIN(chandle%finest_level, MaxLevel)

    NULLIFY(qdata, auxdata)
    mx = box%mGlobal(1,2) - box%mGlobal(1,1) + 1
    my = box%mGlobal(2,2) - box%mGlobal(2,1) + 1
    mz = box%mGlobal(3,2) - box%mGlobal(3,1) + 1

    ! Pack the bounds of the arrays to follow
    CALL MPI_Send(box%mGlobal, 6, MPI_INTEGER, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
    
    CALL MPI_Send(child_count, 1, MPI_INTEGER, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
    ! Loop through the child box list and pack the appropriate number of children.
    DO i = 1, child_count
        child_box => child_list%self
        child_box%MPI_id = box%MPI_id
        CALL MPI_Send(child_box%mGlobal, 6, MPI_INTEGER, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
        child_list => child_list%next
    END DO

    IF (level >= 0) THEN
               ! Retrieve and pack the cell-centered data.
        CALL IO_GetQDataFromChomboFile(chandle, box%mGlobal, qdata)
        CALL MPI_Send(qdata, size(qdata), MPI_DOUBLE_PRECISION, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
        DEALLOCATE(qdata)
        NULLIFY(qdata)

        ! For MHD problems, retrieve and pack the face-centered data.    
        IF (MaintainAuxArrays) THEN
            CALL IO_GetAuxDataFromChomboFile(chandle, box%mGlobal, auxdata)
            IF (nDim == 2) THEN
               CALL MPI_Send(auxdata, (mx+1)*(my+1)*2, MPI_DOUBLE_PRECISION, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
            ELSE
               CALL MPI_Send(auxdata, (mx+1)*(my+1)*(mz+1)*3, MPI_DOUBLE_PRECISION, box%MPI_ID, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, iErr)
            END IF
            DEALLOCATE(auxdata)
            NULLIFY(auxdata)
        END IF
    END IF

  END SUBROUTINE IO_SendDataToWorkers

  !> Receives data for a single grid from master during Chombo restart and stores it in the appropriate Info structure.
  !! @param message A PackedMessage object.  The level of this message should always be >= -1.
  !! @param mGlobal The global coordinate array being received.
  SUBROUTINE IO_RecvDataFromMaster(level, mGlobal) 

     TYPE(PackedMessage), POINTER :: message
     INTEGER, DIMENSION(3,2) :: mGlobal
     INTEGER, DIMENSION(3,2) :: mChild
     TYPE(NodeBox), POINTER :: remote_box, child_box
     TYPE(Nodedef), POINTER :: node
     TYPE(InfoDef), POINTER :: Info
     INTEGER :: mx, my, mz
     INTEGER :: child_count
     INTEGER :: n, level
     TYPE(NodeBoxList), POINTER :: box_list, last_box
     CHARACTER(LEN=28) :: fname
     REAL(KIND=qPREC), DIMENSION(:,:,:,:), ALLOCATABLE :: recv_buffer
     INTEGER :: iErr
     INTEGER :: status(MPI_STATUS_SIZE)

     CALL CreateNodeBox(mGlobal, remote_box, MPI_id)
     CALL StrictFindNode(level, remote_box, node, fname)

     CALL MPI_RECV(child_count, 1, MPI_INTEGER, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr)

     NULLIFY(last_box)
     NULLIFY(box_list)

     ! Create a box list containing this node's children.  This consists of retrieving
     ! the child count and then unpacking child_count node objects from the message.
     DO n = 1, child_count
        NULLIFY(child_box)
        CALL MPI_RECV(mChild, 6, MPI_INTEGER, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr) 
        CALL AddNodeBoxToList(last_box, box_list)
        last_box%self%mGlobal=mChild
        last_box%self%MPI_ID=MPI_ID
    END DO

     ! If this is a data level, then there will be face-centered and edge-centered data.
     IF (level >= 0) THEN
        Info => node%Info

        mx = Info%mX(1)
        my = Info%mX(2)
        mz = Info%mX(3)

        ! Unpack the cell-centered data.
        ALLOCATE(recv_buffer(mx,my,mz,NrVars))
        CALL MPI_RECV(recv_buffer, size(recv_buffer), MPI_DOUBLE_PRECISION, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr)
        Info%q(1:mx,1:my,1:mz,1:NrVars)=recv_buffer
        DEALLOCATE(recv_buffer)

        ! If using face-centered arrays, unpack the face-centered data.
        IF (MaintainAuxArrays) THEN
           IF (nDIm == 2) THEN
              ALLOCATE(recv_buffer(mx+1,my+1,1,2))
              CALL MPI_RECV(recv_buffer, (mx+1)*(my+1)*2, MPI_DOUBLE_PRECISION, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr)
              Info%aux(1:mx+1,1:my+1,1:1,1:2)=recv_buffer
           ELSEIF (nDim == 3) THEN
              ALLOCATE(recv_buffer(mx+1,my+1,mz+1,3))
              CALL MPI_RECV(recv_buffer, (mx+1)*(my+1)*(mz+1)*3, MPI_DOUBLE_PRECISION, 0, TRANSMIT_IO_WORKER_DATA, MPI_COMM_WORLD, status, iErr)
              Info%aux(1:mx+1,1:my+1,1:mz+1,1:3)=recv_buffer
           END IF
           DEALLOCATE(recv_buffer)

           CALL UpdateAux(Info, RESHAPE((/ 1, 1, 1, mx, my, mz /), (/3, 2/)))
        END IF

     END IF

     ! Redistribute child nodes.
     CALL ChomboReCreateChildren(node, level, child_count, box_list)

     CALL ClearNodeBoxList(box_list)      
     CALL DestroyNodeBox(remote_box)

  END SUBROUTINE IO_RecvDataFromMaster

END MODULE IOParsing
