!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module 2DWaves is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir MultiClumps
!! @brief Contains files necessary for the Shape Tester problem

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup MultiClumps Shape Tester Module
!! @brief Module for setting up orbiting particles
!! @ingroup Modules

!> MultiClump Module
!! @ingroup MultiClumps
MODULE Problem
  USE DataDeclarations
  USE SplitRegions
  USE Shapes
  USE EOS
  USE RiemannSolvers
  USE Ambients
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  REAL(KIND=qPREC) :: amplitude, wave_number,qout(10)
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()      
      TYPE(AmbientDef), POINTER :: Ambient
      REAL(KIND=qPREC) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      NAMELIST /ProblemData/ amplitude, wave_number
      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      IF (MPI_NP /= 1 .OR. MaxLevel > 0) THEN
         PRINT*, 'error - this is only designed to run on a single processor with maxlevel=0'
         PRINT*, 'Stopping'
         STOP
      END IF
      CALL CreateAmbient(Ambient)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)
      qout(1)=rhoOut
      IF (ivx /= 0) qOut(ivx)=rhoOut*vxOut
      IF (ivy /= 0) qout(ivy)=rhoOut*vyOut
      IF (ivz /= 0) qout(ivz)=rhoOut*vzOut
      IF (iE /= 0) qout(iE)=gamma7*pOut+half*sum(qout(m_low:m_high)**2)/qout(1)
      CLOSE(PROBLEM_DATA_HANDLE)
   END SUBROUTINE ProblemModuleInit

   !> Applies initial conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      TYPE(InfoDef) :: Info
      INTEGER :: i,j
      REAL(KIND=qPREC) :: x,y
       CALL ConvertTotalToInternalEnergy(Info%q(:,:,:,:))          
       DO i=1, Info%mX(1)
          DO j=1, Info%mX(2)
             x=Info%xbounds(1,1)+(real(i)-half)*levels(Info%level)%dx
             y=Info%xbounds(2,1)+(real(j)-half)*levels(Info%level)%dx            
             Info%q(i,j,:,1:m_high)=Info%q(i,j,:,1:m_high)*(1d0+amplitude*sin(4d0*Pi*wave_number*(x+y)/(GxBounds(1,2)-GxBounds(1,1) + Gxbounds(2,2)-GxBounds(2,1))))
          END DO
       END DO
       CALL ConvertInternalToTotalEnergy(Info%q(:,:,:,:))          
    END SUBROUTINE ProblemGridInit

    !> Applies Boundary conditions
    !! @param Info Info object
    SUBROUTINE ProblemBeforeStep(Info)
       TYPE(InfoDef) :: Info
    END SUBROUTINE ProblemBeforeStep

    !> Could be used to update grids pre-output
    !! @param Info Info Object
    SUBROUTINE ProblemAfterStep(Info)
       TYPE(InfoDef) :: Info
       INTEGER :: i,j
       REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:,:,:) :: qExact
       REAL(KIND=qPREC) :: x,y
       REAL(KIND=qPREC), ALLOCATABLE, DIMENSION(:) :: wmiddle
       IF (levels(info%level)%tnow+levels(info%level)%dt == final_time) THEN
          OPEN(UNIT=11, FILE='data.curve', status='unknown')
          write(11,*) ' #rho'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,1)
          END DO
          write(11,*) ' #vx'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, Info%q(i,1,1,ivx)/Info%q(i,1,1,1)
          END DO
          write(11,*) ' #P'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, (Press(Info%q(i,1,1,:)))
          END DO
          ALLOCATE(qExact(Info%mX(1), Info%mX(2),NrHydroVars))
          !         ALLOCATE(wmiddle(1:NrHydroVars))
          DO i=1, Info%mX(1)
             DO j=1, Info%mX(2)
                x=Info%xbounds(1,1)+(real(i)-half)*levels(Info%level)%dx
                y=Info%xbounds(2,1)+(real(j)-half)*levels(Info%level)%dx            
                qexact(i,j,1:m_high)=qout(1:m_high)*(1d0+amplitude*sin(4d0*Pi*wave_number*(x+y)/(GxBounds(1,2)-GxBounds(1,1) + Gxbounds(2,2)-GxBounds(2,1))))
                IF (iE /= 0) qExact(i,j,iE)=qout(iE)
             END DO
          END DO
          write(11,*) ' #rho_Exact'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, qExact(i,i,1)
          END DO
          write(11,*) ' #vx_Exact'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, sqrt(sum(qExact(i,i,ivx:ivy)**2))
          END DO
          write(11,*) ' #P_Exact'
          DO i=1, Info%mX(1)
             write(11,*) Info%xbounds(1,1)+(REAL(i)-half)*levels(Info%level)%dx, qExact(i,i,iE)
          END DO
          CLOSE(11)
          write(*,*) 'dx, L2 Norm=', levels(Info%level)%dx, sum(abs(qExact(:,:,1)-Info%q(1:Info%mX(1),1:Info%mX(2),1,1)))/product(Info%mX(1:2))
       END IF

    END SUBROUTINE ProblemAfterStep

   !> Could be used to set force refinement
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

