!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module BasicDisk is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!> @dir BasicDisk
!! @brief Contains files necessary for the BasicDisk Calculation

!> @file problem.f90
!! @brief Main file for module Problem

!> @defgroup BasicDisk Basic Disk Module
!! @brief Module for calculating collapse of a uniform cloud
!! @ingroup Modules

!> Basic Dixk Module
!! @ingroup BasicDisk
MODULE Problem
  USE Disks
  USE Winds
  USE DataDeclarations
  USE Ambients
  USE PointGravitySrc
  USE ParticleDeclarations !n
  USE Refinements
  IMPLICIT NONE
  SAVE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
  TYPE(DiskDef), POINTER :: mydisk
  TYPE(WindDef), POINTER :: wind
  INTEGER :: nWinds
  TYPE(AmbientDef), POINTER :: Ambient
  REAL(KIND=qPREC) :: iAccretion=KRUMHOLZ_ACCRETION !n
  REAL(KIND=qPREC) :: Height=1   !Disk peak density
  REAL(KIND=qPREC) :: soft_radius=1d0, radius=10d0
  REAL(KIND=qPREC), DIMENSION(3) :: xloc=(/0,0,0/)  !Disk location      
  TYPE(RefinementDef), POINTER :: Refinement
CONTAINS

  !> Initializes module variables
   SUBROUTINE ProblemModuleInit()
      INTEGER :: i,edge,j, soft_function
      TYPE(ParticleDef), POINTER :: Particle !n
      REAL(KIND=qPREC) :: thickness=0 !thickness of disk smoothing region
      REAL(KIND=qPREC) :: ddensity=10d0, adensity=1d0   !Disk peak density
      REAL(KIND=qPREC) :: dtemp=10d0, atemp=10d0
      REAL(KIND=qpREC) :: mass=1d0
      REAL(KIND=qPREC), DIMENSION(3) :: velocity= (/0,0,0/)  !Disk velocity (in direction of disk axis)
      REAL(KIND=qPREC) :: theta=0     !Angle between X-axis and disk axis (towards y-axis)
      REAL(KIND=qPREC) :: phi=0       !Angle around X-axis to disk axis
      LOGICAL :: lSinkParticle=.true.
      !     REAL(KIND=qPREC) :: mu=0        !Defines ratio of maximum magnetic pressure to ambient pressure
      !     REAL(KIND=qPREC) :: eta=0       !Parameter that determines ratio of maximum poloidal pressure to maximum toroidal pressure
      !     REAL(KIND=qPREC) :: B_theta=0   !Angle from disk axis to define disk orientation
      !     REAL(KIND=qPREC) :: B_phi=0     !Rotation around disk axis to define disk orientation(velocity)
!      REAL(KIND=qPREC) :: B_tor=0     !Maximum Bfield for toroidal configuration
!      REAL(KIND=qPREC) :: B_pol=0     !Maximum Bfield for poloidal configuration
!      REAL(KIND=qPREC) :: Omega=0     !Solid body angular rotation
!      REAL(KIND=qPREC) :: m2A         !Azimuthal density perturbation
!      INTEGER :: iTracer                                !Disk Tracer
      
!      INTEGER :: nwaves=0, nMHDwaves=0
!      REAL(KIND=qPREC), DIMENSION(3) :: wavevector
!      REAL(KIND=qPREC) :: phase, amplitude, amplitudes(3)

!      REAL(KIND=qPREC) :: alpha        !< True love alpha parameter (ratio of thermal to gravitational energy). \f$ \alpha = \frac{5}{2} \left ( \frac{3}{4 \pi \rho_o M^2} \right )^(1/3) \frac{c_s^2}{G} \f$
!      REAL(KIND=qPREC) :: beta_rot   !< True love beta rotational parameter (ratio of rotational to gravitational energy). \f$ \beta_{\Omega} = \frac{1}{4 \pi} \frac{\Omega^2}{G \rho_o} \f$

      TYPE(PointGravityDef), POINTER :: PointGravityObj

      NAMELIST /ProblemData/ ddensity, adensity, velocity, xloc, radius, thickness, dtemp, atemp, theta, phi, mass, soft_radius, soft_function, height, lSinkParticle

      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data', STATUS="OLD")
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CALL CreateAmbient(Ambient)
      Ambient%density=adensity/rScale !cu
      Ambient%pressure= adensity*atemp/rScale/TempScale !cu

      NULLIFY(PointGravityObj)
      IF (lSinkParticle) THEN
         NULLIFY(Particle) !n
         CALL CreateParticle(Particle) !n
         Particle%q(1)=mass*mSolar/mScale !n
         Particle%xloc=xloc !n
         Particle%iAccrete=iAccretion !n
         CALL CreatePointGravityObject(Particle%PointGravityObj)
         Particle%lFixed=.true. !n
         CALL AddSinkParticle(Particle)
         PointGravityObj=>Particle%PointGravityObj
      ELSE
         CALL CreatePointGravityObject(PointGravityObj)
      END IF
   
         !        CALL CreatePointGravityObject(PointGravityObj)    
      PointGravityObj%mass=mass*mSolar/mScale
      PointGravityObj%x0=xloc
      PointGravityObj%v0=velocity
      PointGravityObj%t0=levels(0)%tnow
      PointGravityObj%soft_length=soft_radius*radius/lScale
      PointGravityObj%soft_function=soft_function

      CALL CreateDisk(mydisk)

      mydisk%soft_length=soft_radius*radius/lScale
      mydisk%soft_function=soft_function
      mydisk%pressure=ddensity*dtemp/rScale/TempScale !cu
      mydisk%density=ddensity/rScale
      !mydisk%Type=HYDROSTATIC
      ! Calculate solid body rotational velocity by beta_rot
!      mydisk%omega = sqrt(4d0*pi*ScaleGrav*density*beta_rot)
      mydisk%velocity=velocity
      mydisk%theta=theta
      mydisk%phi=phi
!      mydisk%B_tor=B_tor
!      mydisk%B_pol=B_pol
      mydisk%position=xloc
      mydisk%thickness=4d0*sink_dx
      mydisk%radius=radius/lScale
      mydisk%height=height/lScale
      mydisk%central_mass=mass*mSolar/mScale
      !mydisk%Background_density=ambient%density
      !mydisk%SubSample=0
!      mydisk%m2A=m2A
      CALL UpdateDisk(mydisk)
      
      CLOSE(PROBLEM_DATA_HANDLE)

      nWinds=0
      DO i=1,nDim
         DO edge=1,2
            IF (Gmthbc(i,edge) == 1) THEN 
               nWinds=nWinds+1
               CALL CreateWind(Wind)
               Wind%Type=OUTFLOW_ONLY
               Wind%dir=i
               Wind%edge=edge
            END IF
         END DO
      END DO


      CALL CreateRefinement(Refinement)
      CALL CreateShape(Refinement%Shape)
      CALL SetShapeType(Refinement%Shape, CYLINDER, (/myDisk%radius,myDisk%radius, myDisk%Height/))
      Refinement%BufferCells=4

   END SUBROUTINE ProblemModuleInit

  !> Applies initial conditions
  !! @param Info Info object
  SUBROUTINE ProblemGridInit(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemGridInit

  !> Applies Boundary conditions
  !! @param Info Info object
  SUBROUTINE ProblemBeforeStep(Info)
    TYPE(InfoDef) :: Info
    INTEGER :: i
!    DO i=1,nWinds
!       CALL BeforeStepWind(Info,Wind(i))
!    END DO
  END SUBROUTINE ProblemBeforeStep

  !> Could be used to update grids pre-output
  !! @param Info Info Object
  SUBROUTINE ProblemAfterStep(Info)
    TYPE(InfoDef) :: Info
  END SUBROUTINE ProblemAfterStep

  !> Could be used to set force refinement
  !! @param Info Info object
  SUBROUTINE ProblemSetErrFlag(Info)
    TYPE(InfoDef) :: Info
      INTEGER :: i,j,k,mx,my,mz,rmbc,zrmbc,level
      REAL(KIND=xprec) :: x,y,z,xl,yl,zl,dx,dy,dz


      level=Info%level
      rmbc=levels(level)%gmbc(levels(level)%step)
      zrmbc=rmbc
		mx = Info%mX(1)
		my = Info%mX(2)
		mz = Info%mX(3)
      dx=levels(level)%dX
      dy=dx;dz=dx
      xl=Info%xBounds(1,1)
      yl=Info%xBounds(2,1)
      zl=Info%xBounds(3,1)

      DO k=1, mz ; z = (zl+(REAL(k,xPrec)-half)*dz)
		   if (abs(z-xloc(3)).gt.height/lScale+2d0*dz) cycle 
	      DO i=1, mx ; x = (xl+(REAL(i,xPrec)-half)*dx)
            DO j=1, my ; y = (yl+(REAL(j,xPrec)-half)*dy)
		         if (sqrt((x-xloc(1))**2+(y-xloc(2))**2).le.radius/lScale+2d0*dx) Info%ErrFlag(i,j,k)=1
      END DO; END DO; END DO
  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem

