!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module Bondi is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!Bondi Module

MODULE Problem

   USE DataDeclarations
   USE GlobalDeclarations
   USE PhysicsDeclarations
   USE SourceDeclarations
   USE ParticleDeclarations
   USE CommonFunctions
   USE Bondi
   IMPLICIT NONE
   PRIVATE 

   PUBLIC ProblemModuleInit, ProblemGridInit, &
        ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
   TYPE(PointGravityDef), POINTER :: PointGravityObj
   REAL(KIND=qprec) ::   namb, tamb, ibs, obs, mcent, r_bh
CONTAINS

   SUBROUTINE ProblemModuleInit
      INTEGER :: iErr
      TYPE(ParticleDef), POINTER :: Particle
      NAMELIST /ProblemData/ namb, tamb, ibs, obs, mcent


      OPEN(UNIT=PROBLEM_DATA_HANDLE, FILE='problem.data')
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CLOSE(PROBLEM_DATA_HANDLE, IOSTAT=iErr)

      IF (.not. lRestart) THEN
         NULLIFY(Particle)
         CALL CreateParticle(Particle)
         Particle%q(1)=mcent*MSolar/rScale/lScale**3
         Particle%xloc=0
         Particle%iAccrete=0 !KRUMHOLZ_ACCRETION
         Particle%lFixed=.true.
         particle%buffer(0:MaxLevel)=ceiling(ibs/levels(0:MaxLevel)%dx)
         CALL AddSinkParticle(Particle)
         CALL CreatePointGravityObject(PointGravityObj)
         PointGravityObj%soft_length=1d0*levels(MaxLevel)%dx
         PointGravityObj%soft_function=SPLINESOFT
         PointGravityObj%Mass=mcent*MSolar/rScale/lScale**3 !Central object mass in computational units

      END IF

      !Calculate Bondi radius
      r_BH=ScaleGrav*Particle%q(1)/(gamma*tamb/TempScale)
      write(*,*) 'Bondi radius = ', r_BH
   END SUBROUTINE ProblemModuleInit

   !> Initial Conditions
   !! @param Info Info object
   SUBROUTINE ProblemGridInit(Info)
      !! @brief Initializes the grid data according to the requirements of the problem.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
      INTEGER :: i,j,k
      INTEGER :: rmbc,zrmbc,level
      INTEGER :: mx, my, mz
      INTEGER :: iErr
      REAL (KIND=qPrec), POINTER, DIMENSION (:,:,:,:) :: q
      REAL(KIND=xprec) :: x,y,z,xl,yl,zl,dx,dy,dz,r
      REAL(KIND=qPREC) :: x_, y_, z_, rho, v, temp
      level=Info%level
      q=>Info%q
      ! Calculating the number of ghost cells on each side of the grid.
      rmbc=levels(level)%gmbc(levels(level)%step)
      dx=levels(level)%dX
      dy=dx
      SELECT CASE(nDim)
      CASE(2)
         zrmbc=0
         dz=0d0
      CASE(3)
         zrmbc=rmbc
         dz=dx
      END SELECT
      mx = Info%mX(1)
      my = Info%mX(2)
      mz = Info%mX(3)
      xl=Info%xBounds(1,1)
      yl=Info%xBounds(2,1)
      zl=Info%xBounds(3,1)

      !All of the values below are in computational units

      DO i=1-rmbc, mx+rmbc  
         x = (xl+(REAL(i,xPrec)-half)*dx) 
         DO j=1-rmbc, my+rmbc  
            y = (yl+(REAL(j,xPrec)-half)*dy)
            DO k=1-zrmbc, mz+zrmbc 
               z = (zl+(REAL(k,xPrec)-half)*dz)
               r = sqrt(x**2 + y**2+z**2)

               ! Calculate non-dimensional radius(x_), density(z_), and velocity(y_)
               x_=max(r, ibs-5d0*levels(0)%dx)/r_BH
               z_=BH_alpha(x_)
               y_=Bondi_lambda/(z_*x_**(myDim-1))

               ! Then calculate physical values
               rho=z_*namb/nScale
               v=-y_*sqrt(gamma*tamb/TempScale)
               temp=tamb*(z_**(gamma1))/TempScale

               q(i,j,k,1)=rho
               q(i,j,k,ivx)=rho*v*x/r
               q(i,j,k,ivy)=rho*v*y/r
               q(i,j,k,iE)=half*rho*v**2+gamma7*rho*temp

            END DO
         END DO
      END DO
   END SUBROUTINE ProblemGridInit

   !! @param Info Info object
   SUBROUTINE ProblemBeforeStep(Info)
      !! @brief Performs any tasks required before the advance step.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
      INTEGER :: i,j,k
      INTEGER :: rmbc,zrmbc,level
      INTEGER :: mx, my, mz
      REAL (KIND=qPrec), POINTER, DIMENSION (:,:,:,:) :: q
      REAL(KIND=xprec) :: x,y,z,xl,yl,zl,dx,dy,dz,r
      REAL(KIND=qPREC) :: x_, y_, z_, rho, v, temp

      level=Info%level
      q=>Info%q
      ! Calculating the number of ghost cells on each side of the grid.
      rmbc=levels(level)%gmbc(levels(level)%step)
      dx=levels(level)%dX
      dy=dx
      SELECT CASE(nDim)
      CASE(2)
         zrmbc=0
         dz=0d0
      CASE(3)
         zrmbc=rmbc
         dz=dx
      END SELECT
      mx = Info%mX(1)
      my = Info%mX(2)
      mz = Info%mX(3)
      xl=Info%xBounds(1,1)
      yl=Info%xBounds(2,1)
      zl=Info%xBounds(3,1)

      !All of the values below are in computational units

      DO i = 1-rmbc, mx+rmbc
         x = (xl+(REAL(i,xPrec)-half)*dx) 
         DO j = 1-rmbc, my+rmbc  
            y = (yl+(REAL(j,xPrec)-half)*dy)
            DO k = 1-zrmbc,mz+zrmbc 
               z = (zl+(REAL(k,xPrec)-half)*dz)

               r = sqrt(x**2 + y**2)

               IF (r < ibs .OR. r > obs) THEN !Set cells to analytic solution

                  x_=max(ibs-5d0*levels(0)%dx, r)/r_BH !Adjust values deep inside inner region to avoid extremely high velocities etc...
                  z_=BH_alpha(x_)
                  y_=Bondi_lambda/(z_*x_**(myDim-1))
                  rho=z_*namb/nScale
                  v=-y_*sqrt(gamma*tamb/TempScale)
                  temp=tamb*(z_**(gamma1))/TempScale

                  q(i,j,k,1)=rho
                  q(i,j,k,ivx)=rho*v*x/r
                  q(i,j,k,ivy)=rho*v*y/r
                  q(i,j,k,iE)=half*rho*v**2+gamma7*rho*temp

               END IF



            END DO
         END DO
      END DO

   END SUBROUTINE ProblemBeforeStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemAfterStep(Info)
      !! @brief Performs any post-step corrections that are required.
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
      CALL ProblemBeforeStep(Info)
   END SUBROUTINE ProblemAfterStep

   !> Does nothing
   !! @param Info Info object
   SUBROUTINE ProblemSetErrFlag(Info)
      !! @brief Sets error flags according to problem-specific conditions..
      !! @param Info A grid structure.	
      TYPE (InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep

END MODULE Problem


