!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module BonnorEbertSphere is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!============================================================================
! This problem places a bonner ebert sphere in the center of the grid. 
!============================================================================


MODULE PROBLEM
  USE DataDeclarations
  USE GlobalDeclarations
  USE PhysicsDeclarations
  USE BE_MODULE
  USE WINDS
  USE CLUMPS
  USE Ambients
  USE Totals

  IMPLICIT NONE    ! It's safer to require explicit declarations
  SAVE             ! Save module information
  PRIVATE

  PUBLIC ProblemModuleInit, ProblemGridInit, &
       ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep

  TYPE(WindDef), POINTER :: Wind
  TYPE(ClumpDef), POINTER :: MyClump
  TYPE(AmbientDef), POINTER :: Ambient
  TYPE(TotalDef), POINTER :: Total
  INTEGER :: nWinds
  REAL(KIND=qPrec), DIMENSION(3) :: xloc=(/0,0,0/) !Clump Location -- Erica 1/18/2012
  REAL(KIND=qPrec) :: central_rho, clump_rad, rho_out, m_star, at, rho_c, gravo_thermal, time_f, iso_t, rho_weight! ,p_weight, tff, omega
  REAL :: xi
  REAL(KIND=qPrec) :: rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut, crit_rad, p_crit
  REAL(KIND=qPrec) :: Omega, tff
  
  NAMELIST /ProblemData/ xloc, central_rho, xi, clump_rad, rho_weight, omega, CellsPerJeansLength !p_weight -- var not used anymore. Also, remember when define omega in prob.data to * with timescale
  NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut


CONTAINS

  SUBROUTINE ProblemModuleInit 

  INTEGER :: i, edge

    ! Read in problem params and vars

    OPEN(UNIT = PROBLEM_DATA_HANDLE, FILE = 'problem.data')
    READ(PROBLEM_DATA_HANDLE, NML=ProblemData)
    READ(PROBLEM_DATA_HANDLE, NML=AmbientData)
    CLOSE(PROBLEM_DATA_HANDLE)
    central_rho = central_rho*hmass*Xmu ! now is g/cc
    CALL CalcAmbientParams(central_rho, xi, clump_rad, M_star, at, rho_out, iso_t)
    CALL CreateAmbient(Ambient)
    !gravo_thermal = Sqrt((4*pi*G*central_rho)/(at**2))    ! Gravo_thermal radius -- constant that scales r to xi
    pOut=(rho_out/rscale)*(iso_t/tempscale)!((10*(aT**8))/((G**3)*(M_star**2)))/pscale
    rhoOut=(rho_out/rscale)/rho_weight
    Ambient%density=rhoOut
    Ambient%pressure=pOut
    Ambient%B(:)=(/BxOut,ByOut,BzOut/)
    Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)
    crit_rad = (0.41*G*M_star)/at**2
    p_crit = ((1.4*(aT**8))/((G**3)*(M_star**2)))/pscale
    CALL CreateClump(MyClump) 
    MyClump%position = xloc
    MyClump%density = central_rho/rscale
    MyClump%density_profile = BE_PROFILE
    MyClump%temperature = iso_t/tempScale
    MyClump%omega = omega*72046714255133.6 !* timeScale to put omega in cu
    MyClump%radius = clump_rad/lscale
    CALL UpdateClump(MyClump)
      
      CALL CreateTotal(Total)
      Total%Field%Component=GASCOMP
      Total%Field%id=ivx
      Total%Field%name='Gas Px'
      
      CALL CreateTotal(Total)
      Total%Field%Component=PARTICLECOMP
      Total%Field%id=ivx
      Total%Field%name='Particle Px'
      
      CALL CreateTotal(Total)
      Total%Field%Component=BOTHCOMP
      Total%Field%id=ivx
      Total%Field%name='Combined Px'

      CALL CreateTotal(Total)
      Total%Field%Component=GASCOMP
      Total%Field%id=1
      Total%Field%name='Gas rho'
      
      CALL CreateTotal(Total)
      Total%Field%Component=PARTICLECOMP
      Total%Field%id=1
      Total%Field%name='Particle mass'
      
      CALL CreateTotal(Total)
      Total%Field%Component=BOTHCOMP
      Total%Field%id=1
      Total%Field%name='Combined mass'   


    tff= (0.5427)/(Sqrt(G*central_rho))

    IF (MPI_ID == 0) THEN 
       write(*,*) ambient%density, central_rho*BE_RHO(xi), rho_out, xi
       WRITE (*,*) 'The mass (in solar masses) of your star is:' , M_star/msolar 
       WRITE (*,*) 'The isothermal sound speed (cm/s) of your star and surrounding medium is:' , aT
       WRITE (*,*) 'The isothermal temperature (K) of your star and surrounding medium is:', iso_T

       IF ((central_rho/rho_out) < 14.1) THEN  
          WRITE (*,*) 'Your BE sphere IS stable. Rho_c/Rho_o = ', central_rho/rho_out, "(less than 14.1)"
       ELSE 
          WRITE(*,*) '!!Your BE sphere is NOT stable!! Rho_c/Rho_o =', central_rho/rho_out, "(greater than 14.1)"
       END IF
       time_f = clump_rad/at !sound crossing time in cgs
       time_f = time_f/timeScale !sound crossing time in cu
       WRITE(*,*) 'The sound crossing time for your clump in computational units is:', time_f
       WRITE(*,*) 'The central density of your clump (in CGS) is:', central_rho
       WRITE(*,*) 'The outter density of your clump (in CGS) is:', rho_out
       WRITE(*,*) 'The free fall time in seconds is', tff
       WRITE(*,*) 'G', g, 'Scalegrav', scalegrav
       WRITE(*,*) 'Omega in cgs', (0.1)/tff
    END IF

      nWinds=0
      DO i=1,nDim
         DO edge=1,2
            IF (Gmthbc(i,edge) == 1) THEN 
               nWinds=nWinds+1
               CALL CreateWind(Wind)
               Wind%dir=i
               Wind%density=Ambient%density
               Wind%temperature=Ambient%pressure/Ambient%density
               Wind%edge=edge
            END IF
         END DO
      END DO


  END SUBROUTINE ProblemModuleInit

  SUBROUTINE ProblemGridInit(Info) !Specific for each info structure
    Type (InfoDef) :: Info
    INTEGER        :: i, j, k, mx, my, mz
    INTEGER        :: mbc, rmbc, zrmbc
    INTEGER        :: nx, ny, nz
    INTEGER        :: ndim
    REAL(KIND=xPrec):: x, y, z, xlower, ylower, zlower,dx, dy, dz, r


!To initialize collapse, uncomment the below line to 
!lower the internal energy everywhere in the grid.
!Since objects are initialized before the grid is, placing 
!this line here effectively moves the clump out of equilibrium


!Info%q(:,:,:,1) = (1.1)*Info%q(:,:,:,1) !UNCOMMENT TO INDUCE COLLAPSE
!Info%q(:,:,:,iE) = (1.1)*Info%q(:,:,:,iE) !UNCOMMENT TO INDUCE COLLAPSE




RETURN  
  
    ! Assigning shorthand notation for variables already defined in Info  
    rmbc=levels(Info%level)%gmbc(levels(Info%level)%step)! Sets up ghost cells and boundaries
    mx=Info%mx(1);  xlower=Info%Xbounds(1,1) 
    my=Info%mx(2);  ylower=Info%Xbounds(2,1)
    mz=Info%mx(3);  zlower=Info%Xbounds(3,1)

    dx=levels(Info%level)%dx

    ! These seem to be "nested" variable declarations a=b=c...?


    SELECT CASE(nDim)
    CASE(2)
       zrmbc=0
       mz=1
       zlower=0
       dz=0  
    CASE(3)
       zrmbc=rmbc
    END SELECT

    ! Initializing the environment
    !Info%q(:,:,:,:)=0.d0                                  ! Sanity check -- setting grid to zero
    !gravo_thermal = Sqrt((4*pi*G*central_rho)/(at**2))    ! Gravo_thermal radius -- constant that scales r to xi
    ! Must I allocate an array anywhere for Astrobear 2.0?
    ! i,j,k are indices of cells, and x,y,z are spatial cartesian coords
    
    Do k=1-zrmbc, mz+zrmbc
       Do j=1-rmbc, my+rmbc
          Do i=1-rmbc, mx+rmbc
             x=(xlower + (REAL(i, xPrec)-half)*dx)
             y=(ylower + (REAL(j, xPrec)-half)*dx)
             z=(zlower + (REAL(k, xPrec)-half)*dx) 
             r = Sqrt(x**2+y**2+z**2) 
             r = (r*lscale)*gravo_thermal                 ! takes r to cgs to xi 
             IF (r <= xi) THEN                            ! if r<pc
                Info%q(i,j,k,1) = (central_rho*BE_RHO(REAL(r)))/rscale 
             ELSE 
                Info%q(i,j,k,1) = rho_out/rscale                                              
             END IF
             ! M_star = 8*PI*central_rho*clump_rad**3 *((1 + alpha)/(6 + xi**2) - (6*alpha)/(6 + xi**2)**2 - alpha/(3**(1/alpha)*12*e + xi**2) + D*(2**(-alpha)-1)/ (1+2**(-alpha) * D* xi**2)*(1+D*xi**2))
             Info%q(i,j,k,iE) = (((Info%q(i,j,k,1)*rscale*(aT**2))/gamma)*gamma7)/pScale
          END DO
       END DO
    END DO


  END SUBROUTINE ProblemGridInit

  ! Applies Boundary Conditions
  SUBROUTINE ProblemBeforeStep(INFO)

    TYPE(INFODEF) :: INFO
    INTEGER :: i

  END SUBROUTINE ProblemBeforeStep

  SUBROUTINE ProblemAfterStep(INFO)
    TYPE(INFODEF) :: INFO
    ! No special after step instructions needed; this is a stub.

  END SUBROUTINE ProblemAfterStep

  SUBROUTINE ProblemSetErrFlag(INFO)
    TYPE(INFODEF) :: INFO
    ! No special instructions needed; this is a stub.

  END SUBROUTINE ProblemSetErrFlag

  SUBROUTINE ProblemBeforeGlobalStep(n)
     INTEGER :: n
  END SUBROUTINE ProblemBeforeGlobalStep


END MODULE PROBLEM
