!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    fldgen.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!=======================================================================
!
! SUBROUTINE: fldgen.f90
!
! PURPOSE   : generates random field of perturbations, currently
!             with Gaussian distribution. 
!             Generates a cube of (dots)^3 pixels with plane waves in
!             x-, y- and z-direction with power spectrum P(k) ~ k^-en.
!             The amplitudes are Gaussian distributed with variance
!             P(k) and the phases are evenly distributed in [0,2*pi].
!
! INPUT     : nx,ny,nz: array dimensions
!             power   : exponent for powerlaw
!             kmin    : minimum wavenumber with power 
!             kmax    : maximum wavenumber with power
!             iseed   : for random number generator
!
! OUTPUT    : matrix  : 3D-array with perturbations
!
!=======================================================================
!

SUBROUTINE fldgen(nx     , nxtot  , ixbeg  , ixend  ,  &
                  ny     , nytot  , iybeg  , iyend  ,  &
                  nz     , nztot  , izbeg  , izend  ,  &
                  power  , kmin   , kmax   , iseed,    &
                  velo) 


  IMPLICIT NONE

  integer    nx     , ny     , nz    , kx    , ky    , kz   , &    
             l      , i      , j, k, iseed , dots  ,                &
             kmin  , kmax    ,nwork , ntable,        &
             nxtot , nytot, nztot, ixbeg, iybeg, izbeg,       &
             ixend, iyend, izend , ndim, isign

   double precision     pk     , abs_k  , akmax , akmin , akdel , aksum, &
                        xk     , yk     , zk    , lo    , hi    , off  ,&
                        p1     , p2     , p3    , kf1   , kf2   , kf3  ,tmp_k, &
                        minmat , maxmat , kx2   , ky2   , kz2   , pi   , power

   integer    dimlen(3)

   double precision velo(1:nxtot,1:nytot,1:nztot)
   double precision, allocatable :: cvel(:)

!      complex    x     (0:nx-1,0:ny-1,0:nz-1)! input/output matrix 
!      real       table (1:ntable)
!      real       work  (1:nwork)             ! matrix of fourier - components

!   external   nr_fourn, genak

!**************************************************************
!
!
!**************************************************************


  dots      = nx*ny*nz
  dimlen(1) = nx
  dimlen(2) = ny
  dimlen(3) = nz
  kf1       = 1
  kf2       = 1
  kf3       = 1

  isign     = -1
 
  pi        = 4d0*atan(1d0)

  if (nz .eq. 1) then 
    ndim = 2 
  else 
    ndim = 3
  endif

  allocate(cvel(2*dots))
    
!
!       since the frequencies are real there is the symmetry
!       h(x) --> H(K) with H(-K)=H(K)* !

  do kz = 0, nz/2
    zk  = real(kz)
    kz2 = (zk/kf3)**2

    do ky = 0, ny-1 

      yk = real(ky)
      if (ky .gt. ny/2) yk = real(ny) - yk
      ky2 = (yk/kf2)**2

      do kx = 0, nx-1 

        xk = real(kx)
        if (kx .gt. nx/2) xk = real(nx) - xk
        kx2 = (xk/kf1)**2
!       Upper half 
!       The domain is not cubic any more. Thus the wave numbers
!       can be different for each direction. The factors kf1,
!       kf2 and kf3 have to be determined by the calling routine.
        tmp_k = sqrt(kx2 + ky2 + kz2)
        if ((tmp_k .lt. kmin)  .or. (tmp_k .gt. kmax)) then
          pk = 0.0
        else
          pk = tmp_k**power 
        endif
!       correlation <Ak,Ak'> = Pk * delta(k,k')
        call genak (nx, ny, nz, kx, ky, kz, pk, iseed, pi, cvel)
!       generates Ak(kx,ky,kx) and Ak(-kx,-ky,-kz),
!       which is just Ak*; it means it computes
!       the upper half as well as the lower one      
      end do
    end do
  end do

!     Some Ak's need special treatment:

!  x(0   ,0   ,0   ) = (real(x(0   ,0   ,0   )),0.0)
!  x(0   ,0   ,nz/2) = (real(x(0   ,0   ,nz/2)),0.0)
!  x(0   ,ny/2,0   ) = (real(x(0   ,ny/2,0   )),0.0)
!  x(0   ,ny/2,nz/2) = (real(x(0   ,ny/2,nz/2)),0.0)
!  x(nx/2,0   ,0   ) = (real(x(nx/2,0   ,0   )),0.0)
!  x(nx/2,0   ,nz/2) = (real(x(nx/2,0   ,nz/2)),0.0)
!  x(nx/2,ny/2,0   ) = (real(x(nx/2,ny/2,0   )),0.0)
!  x(nx/2,ny/2,nz/2) = (real(x(nx/2,ny/2,nz/2)),0.0)

  cvel(2*(   0*ny*nz +    0*nz +    0)+2) = 0.0            
  cvel(2*(   0*ny*nz +    0*nz + nz/2)+2) = 0.0            
  cvel(2*(   0*ny*nz + ny/2*nz +    0)+2) = 0.0      
  cvel(2*(   0*ny*nz + ny/2*nz + nz/2)+2) = 0.0
  cvel(2*(nx/2*ny*nz +    0*nz +    0)+2) = 0.0            
  cvel(2*(nx/2*ny*nz +    0*nz + nz/2)+2) = 0.0            
  cvel(2*(nx/2*ny*nz + ny/2*nz +    0)+2) = 0.0      
  cvel(2*(nx/2*ny*nz + ny/2*nz + nz/2)+2) = 0.0

  call nr_fourn (dimlen,ndim,isign,cvel)

  do k=1,nz
    do j=1,ny
      do i=1,nx
        velo(ixbeg+i-1,iybeg+j-1,izbeg+k-1) = cvel(2*((i-1)*ny*nz+(j-1)*nz+(k-1))+1)/dble(dots)
      end do
    end do
  end do

  deallocate(cvel)

!=======================================================================
!  CONTAINS
!    INCLUDE 'genak.f90'
!    INCLUDE 'nr_fourn.f90'


  return
END SUBROUTINE fldgen

