!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    nr_fourn.f90 is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!=======================================================================
!
! SUBROUTINE: nr_fourn.f90
!
! PURPOSE   : n-dimensional fast fourier transformation
! SOURCE    : Numerical Recipes for C, Press et al. 1992
!
! INPUT     : ndim : number of dimensions
!             nn   : integer array of length ndim containing length of each
!                    dimension. Must be power of 2.
!             isign: +1: x is replaced by forward transform
!                    -1: x is replaced by inverse transform multiplied by
!                        lengths of dimensions
!             x    : array to be transformed. Real array of length twice
!                    the product of all dimension lengths. Real and 
!                    imaginary part are stored consecutively, i.e.
!                    real(x(k)) = x(2k-1), imag(x(k)) = x(2k), k = 1...n.
!
! OUTPUT    : x    : array with transform. 
!
!=======================================================================
!

subroutine nr_fourn (nn,ndim,isign,x)

  implicit none

  integer isign  , ndim   , &
          i1     , i2     , i2rev  , i3     , i3rev  , ibit   , &
          idim   , ifp1   , ifp2   , ip1    , ip2    , ip3    , &
          k1     , k2     , n      , nprev  , nrem   , ntot

  integer nn(1:ndim)

  double precision tempi  , tempr  , theta  , wi     , wpi    , wpr    , &
                   wr     , wtemp

  double precision x(*)

!----------------------------------------------------------

      ntot=1
      do idim=1,ndim
        ntot=ntot*nn(idim)
      end do
      nprev=1
      do idim=1,ndim
        n     = nn(idim)
        nrem  = ntot/(n*nprev)
        ip1   = 2*nprev
        ip2   = ip1*n
        ip3   = ip2*nrem
        i2rev = 1
        do i2=1,ip2,ip1
          if(i2.lt.i2rev)then
            do i1=i2,i2+ip1-2,2
              do i3=i1,ip3,ip2
                i3rev      = i2rev+i3-i2
                tempr      = x(i3)
                tempi      = x(i3+1)
                x(i3)      = x(i3rev)
                x(i3+1)    = x(i3rev+1)
                x(i3rev)   = tempr
                x(i3rev+1) = tempi
              end do
            end do
          end if
          ibit = ip2/2
          do while ((ibit.ge.ip1).and.(i2rev.gt.ibit))
            i2rev = i2rev-ibit
            ibit  = ibit/2
          end do
          i2rev = i2rev+ibit
        end do
        ifp1 = ip1
        do while (ifp1.lt.ip2)
          ifp2=2*ifp1
          theta=isign*6.28318530717959d0/(ifp2/ip1)
          wpr=-2.d0*sin(0.5d0*theta)**2
          wpi=sin(theta)
          wr=1.d0
          wi=0.d0
          do i3=1,ifp1,ip1
            do i1=i3,i3+ip1-2,2
              do i2=i1,ip3,ifp2
                k1      = i2
                k2      = k1+ifp1
                tempr   = (wr)*x(k2)-(wi)*x(k2+1)
                tempi   = (wr)*x(k2+1)+(wi)*x(k2)
                x(k2)   = x(k1)-tempr
                x(k2+1) = x(k1+1)-tempi
                x(k1)   = x(k1)+tempr
                x(k1+1) = x(k1+1)+tempi
              end do
            end do
            wtemp = wr
            wr    = wr*wpr-wi*wpi+wr
            wi    = wi*wpr+wtemp*wpi+wi
          end do
          ifp1=ifp2
        end do
        nprev=n*nprev 
      end do

   return
end subroutine nr_fourn
