!#########################################################################
!		
!    Copyright (C) 2003-2012 Department of Physics and Astronomy,
!                            University of Rochester,
!                            Rochester, NY
!
!    problem.f90 of module Christina_Original is part of AstroBEAR.
!
!    AstroBEAR is free software: you can redistribute it and/or modify	  
!    it under the terms of the GNU General Public License as published by 
!    the Free Software Foundation, either version 3 of the License, or    
!    (at your option) any later version.
!
!    AstroBEAR is distributed in the hope that it will be useful, 
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with AstroBEAR.  If not, see <http://www.gnu.org/licenses/>.
!
!#########################################################################
!============================================================================
!                  BEARCLAW Astrophysical Applications
!============================================================================
! Christina Haig, Fabian Heitsch
! Department of Physics and Astronomy
! University of North Carolina, Chapel Hill
! Chapel Hill, NC
!============================================================================
MODULE Problem
   ! User definitions of data structures associated with each node
   USE DataDeclarations
   USE CoolingSrc
   USE PhysicsDeclarations
   ! Global declarations for the BEARCLAW Astrophysical Applications
   USE GlobalDeclarations
   USE EOS
   USE Winds
   USE Ambients
   IMPLICIT NONE    ! It's safer to require explicit declarations
   SAVE             ! Save module information
   PRIVATE          ! Everything is implicitly private, i.e. accessible only
   ! to this module.

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   !                                                                           !
   ! Declarations Local to the Module                                          ! 
   !                                                                           !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   PUBLIC ProblemModuleInit, ProblemGridInit, CloudsBC, &
        ProblemBeforeStep, ProblemAfterStep, ProblemSetErrFlag, ProblemBeforeGlobalStep
   TYPE(CoolingDef),POINTER :: coolingobj
   INTEGER :: iCooling
   INTEGER :: iTracer1=0
   INTEGER :: iTracer2=0
   !PUBLIC :: scaleClouds,InitializeClouds,qinitClouds,b4stepClouds,afterstepClouds,CloudsBC!,CloudsErrFlag

   !Cloud parameters
   REAL(KIND=qPrec) :: power,nCloud,TCloud,vx0,vy0,vz0,kx0,ky0,kz0,bmin,amaj,wght,lamda,sig0,pert,radius,slope
   REAL(KIND=xPrec) :: xlen,ylen,zlen,xmin,ymin,zmin
   INTEGER :: iseed, kmin, kmax
   REAL(KIND = qPrec),ALLOCATABLE,DIMENSION(:,:,:) :: amplitrand
   INTEGER :: mxtot,mytot,mztot 
 
   REAL(KIND=qPrec) :: alpha,beta  ! analytic cooling parameters

   NAMELIST/ProblemData/iCooling,iseed,power,kmin,kmax,nCloud,TCloud,vx0,vy0,vz0,kx0,ky0,kz0,bmin,amaj,sig0,pert,radius,slope


CONTAINS

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
   !                                                                           !
   ! Public Module Prodecures                                                  !
   !                                                                           !
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


   SUBROUTINE ProblemModuleInit
      !REAL(KIND = qPrec),ALLOCATABLE,DIMENSION(:,:,:) :: amplitrand
      REAL(KIND=xPrec) :: dX(3), rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut
      REAL(KIND = qPrec) :: mina, maxa
      INTEGER :: iErr
      TYPE(pWindDef), DIMENSION(:), ALLOCATABLE :: MyWinds
      TYPE(AmbientDef), POINTER :: Ambient
      NAMELIST /AmbientData/ rhoOut, pOut, vxOut, vyOut, vzOut, BxOut, ByOut, BzOut

      OPEN(UNIT=PROBLEM_DATA_HANDLE,IOSTAT=iErr,FILE='problem.data')
      READ(PROBLEM_DATA_HANDLE,NML=ProblemData)
      CALL CreateAmbient(Ambient)
      READ(PROBLEM_DATA_HANDLE,NML=AmbientData)
      Ambient%density=rhoOut
      Ambient%pressure=pOut
      Ambient%B(:)=(/BxOut, ByOut, BzOut/)
      Ambient%velocity(:)=(/vxOut, vyOut, vzOut/)

      CLOSE(PROBLEM_DATA_HANDLE)

      CALL AddTracer(iTracer1, 'FlowTracer_1')
      CALL AddTracer(iTracer2, 'FlowTracer_2')

      mxtot = levels(MAXLEVEL)%mX(1);mytot=levels(MaxLEVEL)%mX(2);mztot=LEVELS(MaxLEVEL)%mX(3)
      xlen = GxBounds(1,2)-GxBounds(1,1)
      ylen = GxBounds(2,2)-GxBounds(2,1)
      zlen = GxBounds(3,2)-GxBounds(3,1)
      xmin = GxBounds(1,1)

      !   This part sets the random interface perturbations
      IF (iSeed.NE.0) THEN

         ALLOCATE(amplitrand(1:mytot,1:mztot,1))
         CALL fldgen(mytot, mytot, 1, mztot,  &
              mztot, mztot, 1, mztot,  &
              1 , 1 , 1, 1 ,  &
              power , kmin , kmax ,  &
              iseed , amplitrand)
    

         mina = minval(amplitrand(1:mytot,1:mztot,1)) 
         maxa = maxval(amplitrand(1:mytot,1:mztot,1))
         amplitrand(:,:,:) = (pert*3.08568025d18/lscale)*((amplitrand(:,:,:)-mina)/(maxa-mina)-0.5d0)
         mina = minval(amplitrand(1:mytot,1:mztot,1)) 
         maxa = maxval(amplitrand(1:mytot,1:mztot,1))  

      END IF


    ! [BDS][20110106]: Do not create a new cooling object on a restart.  The cooling object will be
    !                  read in from the Chombo files.
    IF(iCooling>0) THEN
       IF (.NOT. lRestart) THEN
           ! see sources/cooling.f90::CreateCoolingObject for
           ! default values of a cooling source term
           CALL CreateCoolingObject(coolingobj)
           write(*,*) 'created cooling object'
       ELSE
           coolingobj => firstcoolingobj
       END IF
    END IF

    coolingobj%iCooling=iCooling
    SELECT CASE(iCooling) ! cases defined in sources/cooling.f90
    CASE(NoCool)
    CASE(AnalyticCool)
       coolingobj%alpha=alpha
       coolingobj%beta=beta
    CASE(DMCool)
    CASE(IICool)
    CASE DEFAULT
    END SELECT

    coolingobj%floortemp=0.001d0
    coolingobj%mintemp=0.001


   END SUBROUTINE ProblemModuleInit

   SUBROUTINE ProblemGridInit(Info) 

      TYPE(InfoDef) :: Info                       ! Data associated with this grid

      REAL (KIND = qPrec), POINTER, DIMENSION (:,:,:,:) :: q

      ! The q-variable fields are as follows:
      !
      ! q(:,:,:,:,1) = density
      ! q(:,:,:,:,2) = px
      ! q(:,:,:,:,3) = py
      ! q(:,:,:,:,4) = E
      ! q(:,:,:,:,5) = phi
      
      INTEGER            :: i, j, k, mx, my, mz,ix,iy,iz
      INTEGER            :: mbc, rmbc, zrmbc
      REAL(KIND = xPrec) :: x, y, z, xl, yl, zl, dx, dy, dz, r,grdx,grdy,grdz,dx_maxlevel
      REAL(KIND = qPrec) :: mina, maxa, cosa, arlo, arhi,r2
      LOGICAL, SAVE      :: lfirst = .TRUE.


      q => Info%q
      mbc = 0!Info%mbc                      !number of ghost cells
      rmbc=0

      !  IF (Info%ErrorEstimation) THEN
      !     rmbc = (Info%AMRSteps-Info%AMRStep+1)*mbc
      !  ELSE
      !     rmbc = Info%r*mbc    
      !  END IF
      !Erase this if not needed: mxtot = levels(MAXLEVEL)%mX(1)

      mx = Info%mX(1);my = Info%mX(2);mz = Info%mX(3)
      dx = levels(Info%level)%dx;dy=dx;dz=dx     !Info%dX(1);dy = Info%dX(2);dz = Info%dX(3)
      dx_maxlevel = levels(maxlevel)%dx
      xl = Info%XBounds(1,1);yl = Info%XBounds(2,1);zl = Info%XBounds(3,1)
      xmin = GxBounds(1,1);ymin = GxBounds(2,1);zmin = GxBounds(3,1)

        q(:,:,:,:) = 0.d0

      cosa = 1.d0


      SELECT CASE(nDim)
      CASE(2)
         zrmbc = 0; mz = 1; zl = 0; dz = 0

         DO i = 1-rmbc, mx + rmbc 
            x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y, z coordinates at cell centers
            grdx = (xl+REAL(i,xPrec)*dx)           ! set x, y, z coordinates at cell walls to the right of cell centers
            ix = Info%mGlobal(1,1)+i-1
            DO j = 1, my
               y = (yl+(REAL(j,xPrec)-half)*dy)
               grdy = (yl+REAL(j,xPrec)*dy)
               iy = Info%mGlobal(2,1)+j-1

               IF (iseed.EQ.0) THEN    ! geometrical collision interface
                  arlo =   (xmin+ slope*y + 0.5d0*xlen - (grdx-dx))*dy &  ! subtract everything below cell
                       + (ylen*xlen*dabs(pert)/(2d0*pi*ky0)) &
                       * (dsin(2d0*pi*ky0*grdy/ylen)-dsin(2d0*pi*ky0*(grdy-dy)/ylen))

                  IF (kz0 .GT. 0d0) THEN ! this is for overlaying a perturbation at different scale.
                     ! Scale is given by ky0/kz0. For ky0>kz0, the additional perturbation will have
                     ! a larger scale at higher amplitude. Only for code stability tests.
                     arlo = arlo + (ylen*xlen*dabs(pert*(ky0/kz0)**(5d0/3d0)) &
                          /(2d0*pi*kz0))*(dsin(2d0*pi*kz0*grdy/ylen)-dsin(2d0*pi*kz0*(grdy-dy)/ylen))
                  END IF

                  arlo = arlo / (dx*dy)
                  arhi = 1d0 - arlo
                  ! Now we have the "lower" and "higher" area in each cell, i.e. the area (=volume...)
                  ! of the current cell located left of perturbation (arlo) and right of perturbation (arhi).

               ELSE  ! Random interface
                  
                  arlo = amplitrand(iy,1,1) - (grdx-dx) + slope*y
                  arhi = grdx - amplitrand(iy,1,1) - slope*y

               END IF

               IF (arlo .LT. 0d0) THEN ! perturbation interface does not intersect cell. Cell to the right.
                  arlo = 0d0
                  arhi = 1d0
               END IF

               IF (arhi .LT. 0d0) THEN ! perturbation interface does not intersect cell. Cell to the left.
                  arhi = 0d0
                  arlo = 1d0
               END IF

               q(i,j,1,1) = nCloud
               q(i,j,1,2) = q(i,j,1,1)*(vx0*arlo - vx0*arhi)*cosa
               q(i,j,1,3) = q(i,j,1,1)*(vy0*arlo - vy0*arhi)

               IF (radius.NE.0.d0) THEN ! we use a limited inflow region 
                  IF (dabs(y).LE.radius) THEN
                     wght=1.d0
                  ELSE
                     wght=0.d0
                  END IF
               ELSE
                  wght=1.d0

               END IF

               q(i,j,1,2) = q(i,j,1,2) * wght


               IF (iTracer1 .GT.0) THEN
                  IF (q(i,j,1,2).GT.1e-2) THEN
                     q(i,j,1,iTracer1) = 1.d0
                  ELSE
                     q(i,j,1,iTracer1) = 1.d-10
                  ENDIF
               END IF

               IF (iTracer2.GT.0) THEN
                  IF (q(i,j,1,2).LT.1e-2) THEN
                     q(i,j,1,iTracer2) = 1.d0
                  ELSE
                     q(i,j,1,iTracer2) = 1.d-10
                  ENDIF
               END IF

               CALL CalcComp(q(i,j,1,:),TCloud)
            END DO
         END DO


      CASE(3)
         zrmbc = rmbc

         DO i = 1-rmbc, mx + rmbc 
            x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y, z coordinates at cell centers
            grdx = (xl+REAL(i,xPrec)*dx)           ! set x, y, z coordinates at cell walls to the right of cell centers
            !ix = Info%mGlobal(1,1)+i-1
             ix = NInt(half+(x-xmin)/dx_maxlevel)
            DO j = 1, my
               y = (yl+(REAL(j,xPrec)-half)*dy)
               grdy = (yl+REAL(j,xPrec)*dy)
               !iy = Info%mGlobal(2,1)+j-1
                iy = NInt(half+(y-ymin)/dx_maxlevel)
               DO k = 1, mz
                  z = (zl+(REAL(k,xPrec)-half)*dz)
                  grdz = (zl+REAL(k,xPrec)*dz)
                  !iz = Info%mGlobal(3,1)+k-1
                   iz = NInt(half+(z-zmin/dx_maxlevel))


                  IF (iseed.EQ.0) THEN    ! geometrical collision interface
                     arlo = (xmin+0.5d0*xlen-(grdx-dx)+slope*y)*dy*dz + (zlen*ylen*xlen*dabs(pert)/(4d0*pi**2*ky0*kz0))*&
                          (dsin(2d0*pi*ky0*grdy/ylen)-dsin(2d0*pi*ky0*(grdy-dy)/ylen)) &
                          * (dsin(2d0*pi*kz0*grdz/zlen)-dsin(2d0*pi*kz0*(grdz-dz)/zlen))
                     arlo = arlo / (dx*dy*dz)
                     arhi = 1d0 - arlo  
                  ELSE    ! random collision interface. Needs to be centered on x = 0d0.


                     arlo = amplitrand(iy,iz,1) - (grdx-dx) + slope*y
                     arhi = grdx - amplitrand(iy,iz,1) - slope*y

                  END IF

                  !IF (radius .GT. 0d0) THEN ! mimicking taper due to expanding blast wave
                  !   cosa = dcos(datan(dsqrt(y**2 + z**2)/radius))
                  !END IF

                  IF (arlo .LT. 0d0) THEN ! perturbation interface does not intersect cell. Cell to the right.
                     arlo = 0d0
                     arhi = 1d0
                  END IF

                  IF (arhi .LT. 0d0) THEN ! perturbation interface does not intersect cell. Cell to the left.
                     arhi = 0d0
                     arlo = 1d0
                  END IF

                  q(i,j,k,1) = nCloud
                  q(i,j,k,2) = nCloud*(vx0*arlo - vx0*arhi)*cosa
                  q(i,j,k,3) = nCloud*(vy0*arlo - vy0*arhi)
                  q(i,j,k,4) = nCloud*(vz0*arlo - vz0*arhi)

                  wght=1.d0

                  IF ((bmin .NE. 0d0) .AND. (amaj .NE. 0d0)) THEN ! we use a limited inflow region 
                     r2 = ((y/amaj)**2+(z/bmin)**2) ! normalized elliptical "radius"

                     IF (sig0 .EQ. 0d0) THEN ! box profile
                        IF (r2 .GT. 1d0) THEN ! ellipse condition
                           wght = 0d0 ! outside inflow region
                        ELSE 
                           wght = 1d0 ! inside inflow region
                        END IF
                     ELSE IF (sig0 .LT. 0d0) THEN ! Gaussian profile
                        wght = dexp(-r2/(sig0/xlen)**2) ! needs to be normalized
                     ELSE IF (sig0 .GT. 0d0) THEN ! double tanh profile as a taper for viscous profile (BGK!)
                        r2 = dsqrt(r2)
                        wght = 0.5d0*(1d0-(dexp((r2-1d0)*xlen/sig0)-dexp(-(r2-1d0)*xlen/sig0))&
                             /(dexp((r2-1d0)*xlen/sig0)+dexp(-(r2-1d0)*xlen/sig0)))
                     END IF

                     q(i,j,k,2) = q(i,j,k,2) * wght
                  END IF


                  IF (iTracer1.GT.0) THEN
                     IF ((q(i,j,k,2).GT.0).AND.(wght.GT.1e-2)) THEN
                        q(i,j,k,iTracer1) = 1.d0
                     ELSE
                        q(i,j,k,iTracer1) = 1.d-10
                     ENDIF
                  END IF

                  IF (iTracer2.GT.0) THEN
                     IF ((q(i,j,k,2).LT.0).AND.(wght.GT.1e-2)) THEN
                        q(i,j,k,iTracer2) = 1.d0
                     ELSE
                        q(i,j,k,iTracer2) = 1.d-10
                     ENDIF
                  END IF

                  CALL CalcComp(q(i,j,k,:),TCloud)


               END DO
            END DO
         END DO

      CASE DEFAULT
         PRINT *, "Error in qinitCloud, invalid problem dimensions"
         STOP
      END SELECT



   END SUBROUTINE ProblemGridInit

   SUBROUTINE ProblemBeforeSTep(Info)
      TYPE(InfoDef) :: Info
      CALL CloudsBC(Info)
   END SUBROUTINE ProblemBeforeSTep

   SUBROUTINE ProblemSetErrFlag(Info)
      TYPE(InfoDef) :: Info
   END SUBROUTINE ProblemSetErrFlag

   SUBROUTINE ProblemAfterStep(Info)
      TYPE(InfoDef) :: Info
      REAL (KIND = qPrec), POINTER, DIMENSION (:,:,:,:) :: q

      INTEGER            :: i, j, k, mx, my, mz
      INTEGER            :: mbc, rmbc, zrmbc, ibc     ! See assignment statement
      REAL(KIND = xPrec) :: x, y, z, xl, yl, zl, dx, dy, dz
      REAL(KIND = qPrec) :: Temp
      REAL (KIND=qPrec), PARAMETER :: F = 2d-26
      LOGICAL, SAVE      :: lfirst = .TRUE.

      q => Info%q
      mbc = 0!Info%mbc                   ! number of ghost cells
      rmbc=0
      
      !      IF (Info%ErrorEstimation) THEN
      !         rmbc = (Info%AMRSteps-Info%AMRStep+1)*mbc
      !      ELSE
      !         rmbc = Info%r*mbc    
      !      END IF


      mx = Info%mX(1);my = Info%mX(2);mz = Info%mX(3)
      dx = levels(Info%level)%dx;dy=dx;dz=dx!;Info%dX(1);dy = Info%dX(2);dz = Info%dX(3)
      xl = Info%XBounds(1,1);yl = Info%XBounds(2,1);zl = Info%xBounds(3,1)

      SELECT CASE(nDim)
      CASE(2)
         zrmbc = 0; mz = 1; zl = 0; dz = 0

         DO i = 1-rmbc, mx + rmbc 
            x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
            DO j = 1-rmbc, my + rmbc
               y = (yl+(REAL(j,xPrec)-half)*dy)

               ! Floor on both metallicity tracers
               IF (iTracer1.GT.0) THEN
                  IF (q(i,j,1,iTracer1).LT.1.d-10) q(i,j,1,iTracer1)=1.d-10
               END IF

               IF (iTracer2.GT.0) THEN
                  IF (q(i,j,1,iTracer2).LT.1.d-10) q(i,j,1,iTracer2)=1.d-10      
               END IF


               ! Temp = Tempscale*Press(q(i,j,1,:))/q(i,j,1,1)
               ! q(i,j,1,iTracer1) = q(i,j,1,1)*nscale*F*(1-q(i,j,1,1)*nscale*IICoolingRate(Temp))
               !IF (i==1.AND.j==1) PRINT*,IIColingRate(Temp)
            END DO
         END DO




      CASE(3)
         zrmbc = rmbc
         DO i = 1-rmbc, mx + rmbc 
            x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y, z coordinates at cell centers
            DO j = 1-rmbc, my + rmbc
               y = (yl+(REAL(j,xPrec)-half)*dy)
               DO k = 1-rmbc, mz + rmbc
                  z = (zl+(REAL(k,xPrec)-half)*dz)

                  ! Floor on both metallicity tracers
                  IF (iTracer1.GT.0) THEN
                     IF (q(i,j,k,iTracer1).LT.1.d-10) q(i,j,k,iTracer1)=1.d-10
                  END IF

                  IF (iTracer2.GT.0) THEN
                     IF (q(i,j,k,iTracer2).LT.1.d-10) q(i,j,k,iTracer2)=1.d-10      
                  END IF


                  ! Temp = Tempscale*Press(q(i,j,1,:))/q(i,j,1,1)
                  ! q(i,j,1,iTracer1) = q(i,j,1,1)*nscale*F*(1-q(i,j,1,1)*nscale*IICoolingRate(Temp))
                  !IF (i==1.AND.j==1) PRINT*,IICoolingRate(Temp)
               END DO
            END DO
         END DO


      CASE DEFAULT
         PRINT *, "Error in b4stepCloud, invalid problem dimensions"
         STOP
      END SELECT


   END SUBROUTINE ProblemAfterStep



   SUBROUTINE CloudsBC(Info)

      USE EOS
      TYPE(InfoDef) :: Info                       

      REAL (KIND = qPrec), POINTER, DIMENSION (:,:,:,:) :: q

      INTEGER            :: i, j, k, mx, my, mz,ibc
      INTEGER            :: mbc, rmbc, zrmbc     ! See assignment statement
      REAL(KIND = xPrec) :: x, y, z, xl, yl, zl, dx, dy, dz, r2
      REAL(KIND = qPrec) :: Temp
      REAL (KIND=qPrec), PARAMETER :: F = 2d-26
      LOGICAL, SAVE      :: lfirst = .TRUE.

      q => Info%q
      mbc = levels(Info%level)%ambc(levels(Info%level)%step)                   ! number of ghost cells
      rmbc = mbc
      mx = Info%mX(1);my = Info%mX(2);mz = Info%mX(3)
      dx = levels(Info%level)%dx     !Info%dX(1);dy = Info%dX(2);dz = Info%dX(3)
      dy=dx
      dz=dx
      xl = Info%XBounds(1,1);yl = Info%XBounds(2,1);zl = Info%XBounds(3,1)


      SELECT CASE(nDim)
      CASE(2)
         zrmbc = 0; mz = 1; zl = 0; dz = 0

         IF (Info%xBounds(1,1)==GxBounds(1,1) .AND. Gmthbc(1,1)==10) THEN    ! Left Side

            DO i = 1-rmbc, 0 
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  y = (yl+(REAL(j,xPrec)-half)*dy)

                  ! First set extrapolating conditions, just in case
                  Info%q(i,j,1,:)=Info%q(1,j,1,:)

                  IF (radius.NE.0.d0) THEN ! we use a limited inflow region

                     IF (dabs(y).LE.radius) THEN  ! Inside 'circle'
                        q(i,j,1,1) = nCloud
                        q(i,j,1,2) = q(i,j,1,1)*vx0
                        q(i,j,1,3) = q(i,j,1,1)*vy0
                        CALL CalcComp(q(i,j,1,:),TCloud)

                        IF (iTracer1.GT.0) THEN
                           q(i,j,1,iTracer1) = 1.d0
                        END IF

                        IF (iTracer2.GT.0) THEN
                           q(i,j,1,iTracer2) = 1.d-10
                        END IF

                     ELSE   ! Outside 'circle'                  
                        IF (q(i,j,1,2).GT.0.d0) THEN               
                           Temp   = TempScale*Press(q(i,j,1,:))/q(i,j,1,1)
                           q(i,j,1,2)=0.d0
                           CALL CalcComp(q(i,j,1,:),Temp)      
                        END IF

                     END IF

                  ELSE      ! Inflow region is not limited (entire left side)
                     q(i,j,1,1) = nCloud
                     q(i,j,1,2) = q(i,j,1,1)*vx0
                     q(i,j,1,3) = q(i,j,1,1)*vy0
                     CALL CalcComp(q(i,j,1,:),TCloud)

                     IF (iTracer1.GT.0) THEN
                        q(i,j,1,iTracer1) = 1.d0
                     END IF

                     IF (iTracer2.GT.0) THEN
                        q(i,j,1,iTracer2) = 1.d-10
                     END IF

                  END IF

               END DO
            END DO

         END IF

         IF (Info%xBounds(1,2)==GxBounds(1,2) .AND. Gmthbc(1,2)==10) THEN    ! Right Side

            DO i = mx+1, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  y = (yl+(REAL(j,xPrec)-half)*dy)

                  ! First set boundary conditions to extrapolating, just in case
                  Info%q(i,j,1,:)=Info%q(mx,j,1,:)

                  IF (radius.NE.0.d0) THEN ! we use a limited inflow region 

                     IF (dabs(y).LE.radius) THEN   ! Inside 'circle'
                        q(i,j,1,1) = nCloud
                        q(i,j,1,2) = -q(i,j,1,1)*vx0
                        q(i,j,1,3) = q(i,j,1,1)*vy0
                        CALL CalcComp(q(i,j,1,:),TCloud)

                        IF (iTracer1.GT.0) THEN
                           q(i,j,1,iTracer1) = 1.d-10
                        END IF
                        IF (iTracer2.GT.0) THEN
                           q(i,j,1,iTracer2) = 1.d0
                        END IF

                     ELSE     ! Outside 'circle'
                        IF (q(i,j,1,2).LT.0.d0) THEN               
                           Temp   = TempScale*Press(q(i,j,1,:))/q(i,j,1,1)
                           q(i,j,1,2)=0.d0
                           CALL CalcComp(q(i,j,1,:),Temp)      
                        END IF
                     END IF

                  ELSE         ! Inflow region is not limited (entire right side)
                     q(i,j,1,1) = nCloud
                     q(i,j,1,2) = -q(i,j,1,1)*vx0
                     q(i,j,1,3) = q(i,j,1,1)*vy0
                     CALL CalcComp(q(i,j,1,:),TCloud)

                     IF (iTracer1.GT.0) THEN
                        q(i,j,1,iTracer1) = 1.d-10
                     END IF
                     IF (iTracer2.GT.0) THEN
                        q(i,j,1,iTracer2) = 1.d0
                     END IF

                  END IF


               END DO
            END DO
         END IF


         IF (Info%xBounds(2,1)==GxBounds(2,1) .AND. Gmthbc(2,1)==10) THEN    ! Bottom

            DO ibc=1,rmbc
               Info%q(:,1-ibc,1,:) = Info%q(:,1,1,:)
            END DO

            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, 0
                  !y = (yl+(REAL(j,xPrec)-half)*dy)

                  ! Extrapolating
                  Info%q(i,j,1,:) = Info%q(i,1,1,:)

                  IF (q(i,j,1,3).GT.0.d0) THEN               
                     Temp   = TempScale*Press(q(i,j,1,:))/q(i,j,1,1)
                     q(i,j,1,3)=0.d0
                     CALL CalcComp(q(i,j,1,:),Temp)      
                  END IF


               END DO
            END DO

         END IF

         IF (Info%xBounds(2,2)==GxBounds(2,2) .AND. Gmthbc(2,2)==10) THEN    ! Top


            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = my+1, my+rmbc
                  !y = (yl+(REAL(j,xPrec)-half)*dy)

                  ! Extrapolating
                  Info%q(i,j,1,:) = Info%q(i,my,1,:)

                  IF (q(i,j,1,3).LT.0.d0) THEN

                     Temp   = TempScale*Press(q(i,j,1,:))/q(i,j,1,1)
                     q(i,j,1,3)=0.d0
                     CALL CalcComp(q(i,j,1,:),Temp) 
                  END IF
               END DO
            END DO

         END IF

      CASE(3)
         zrmbc = rmbc


         IF (Info%xBounds(1,1)==GxBounds(1,1) .AND. Gmthbc(1,1)==10) THEN    ! Left Side

            DO i = 1-rmbc, 0 
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = 1-rmbc, mz+rmbc
                     z = (zl+(REAL(k,xPrec)-half)*dz)

                     ! First set extrapolating conditions, just in case
                     Info%q(i,j,k,:)=Info%q(1,j,k,:)

                     IF ((bmin .NE. 0d0) .AND. (amaj .NE. 0d0)) THEN ! we use a limited inflow region 
                        r2 = ((y/amaj)**2+(z/bmin)**2) ! normalized elliptical "radius"

                        IF (sig0 .EQ. 0d0) THEN ! box profile
                           IF (r2 .GT. 1d0) THEN ! ellipse condition
                              wght = 0d0 ! outside inflow region
                           ELSE 
                              wght = 1d0 ! inside inflow region
                           END IF

                        ELSE IF (sig0 .LT. 0d0) THEN ! Gaussian profile
                           wght = dexp(-r2/(sig0/xlen)**2) ! needs to be normalized

                        ELSE IF (sig0 .GT. 0d0) THEN ! double tanh profile as a taper for viscous profile (BGK!)
                           r2 = dsqrt(r2)
                           wght = 0.5d0*(1d0-(dexp((r2-1d0)*xlen/sig0)-dexp(-(r2-1d0)*xlen/sig0))&
                                /(dexp((r2-1d0)*xlen/sig0)+dexp(-(r2-1d0)*xlen/sig0)))
                        END IF

                        IF (wght.gt.1e-2) THEN  ! Consider yourself in inflow region
                           q(i,j,k,1) = nCloud
                           q(i,j,k,2) = q(i,j,k,1)*vx0*wght
                           q(i,j,k,3) = q(i,j,k,1)*vy0*wght
                           CALL CalcComp(q(i,j,k,:),TCloud)
                           IF (iTracer1.GT.0) q(i,j,k,iTracer1) = 1.d0
                           IF (iTracer2.GT.0) q(i,j,k,ITracer1+1) = 1.d-10

                        ELSE                    ! Outflow conditions only outside of inflow region
                           IF (q(i,j,k,2).GT.0.d0) THEN               
                              Temp   = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                              q(i,j,k,2)=0.d0
                              CALL CalcComp(q(i,j,k,:),Temp) 
                           END IF

                        END IF

                     ELSE   ! Inflow is not limited - entire left side

                        q(i,j,k,1) = nCloud
                        q(i,j,k,2) = q(i,j,k,1)*vx0
                        q(i,j,k,3) = q(i,j,k,1)*vy0
                        CALL CalcComp(q(i,j,k,:),TCloud)

                        IF (iTracer1.GT.0) q(i,j,k,iTracer1) = 1.d0
                        IF (iTracer2.GT.0) q(i,j,k,iTracer2) = 1.d-10

                     END IF

                  END DO
               END DO
            END DO

         END IF
         IF (Info%xBounds(1,2)==GxBounds(1,2) .AND. Gmthbc(1,2)==10) THEN    ! Right Side

            DO i = mx+1, mx+rmbc 
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = 1-rmbc, mz+rmbc
                     z = (zl+(REAL(k,xPrec)-half)*dz)

                     ! First set extrapolating conditions, just in case
                     Info%q(i,j,k,:)=Info%q(mx,j,k,:)

                     IF ((bmin .NE. 0d0) .AND. (amaj .NE. 0d0)) THEN ! we use a limited inflow region 
                        r2 = ((y/amaj)**2+(z/bmin)**2) ! normalized elliptical "radius"

                        IF (sig0 .EQ. 0d0) THEN ! box profile
                           IF (r2 .GT. 1d0) THEN ! ellipse condition
                              wght = 0d0 ! outside inflow region
                           ELSE 
                              wght = 1d0 ! inside inflow region
                           END IF

                        ELSE IF (sig0 .LT. 0d0) THEN ! Gaussian profile
                           wght = dexp(-r2/(sig0/xlen)**2) ! needs to be normalized

                        ELSE IF (sig0 .GT. 0d0) THEN ! double tanh profile as a taper for viscous profile (BGK!)
                           r2 = dsqrt(r2)
                           wght = 0.5d0*(1d0-(dexp((r2-1d0)*xlen/sig0)-dexp(-(r2-1d0)*xlen/sig0))&
                                /(dexp((r2-1d0)*xlen/sig0)+dexp(-(r2-1d0)*xlen/sig0)))
                        END IF

                        IF (wght.gt.1e-2) THEN
                           q(i,j,k,1) = nCloud
                           q(i,j,k,2) = -q(i,j,k,1)*vx0*wght
                           q(i,j,k,3) = -q(i,j,k,1)*vy0*wght
                           CALL CalcComp(q(i,j,k,:),TCloud)
                           IF (iTracer1.GT.0) q(i,j,k,iTracer1) = 1.d-10
                           IF (iTracer2.GT.0) q(i,j,k,ITracer1+1) = 1.d0

                        ELSE
                           IF (q(i,j,k,2).LT.0.d0) THEN               
                              Temp   = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                              q(i,j,k,2)=0.d0
                              CALL CalcComp(q(i,j,k,:),Temp) 
                           END IF

                        END IF

                     ELSE   ! Inflow is not limited - entire right side

                        q(i,j,k,1) = nCloud
                        q(i,j,k,2) = -q(i,j,k,1)*vx0
                        q(i,j,k,3) = q(i,j,k,1)*vy0
                        CALL CalcComp(q(i,j,k,:),TCloud)

                        IF (iTracer1.GT.0) q(i,j,k,iTracer1) = 1.d-10
                        IF (iTracer2.GT.0) q(i,j,k,iTracer2) = 1.d0

                     END IF

                  END DO
               END DO
            END DO

         END IF
         IF (Info%xBounds(2,1)==GxBounds(2,1) .AND. Gmthbc(2,1)==10) THEN    ! bottom

            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, 0
                  !y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = 1-rmbc, mz+rmbc

                     ! Extrapolating
                     Info%q(i,j,k,:) = Info%q(i,k,1,:)

                     IF (q(i,j,k,3).GT.0.d0) THEN               
                        Temp   = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                        q(i,j,k,3)=0.d0
                        CALL CalcComp(q(i,j,k,:),Temp)      
                     END IF


                  END DO
               END DO
            END DO

         END IF

         IF (Info%xBounds(2,2)==GxBounds(2,2) .AND. Gmthbc(2,2)==10) THEN    ! Top Side

            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = my+1, my+rmbc
                  !y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = 1-rmbc, mz+rmbc

                     ! Extrapolating
                     Info%q(i,j,k,:) = Info%q(i,my,k,:)

                     IF (q(i,j,k,3).LT.0.d0) THEN                    
                        Temp   = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                        q(i,j,k,3)=0.d0
                        CALL CalcComp(q(i,j,k,:),Temp) 
                     END IF

                  END DO
               END DO
            END DO

         END IF

         IF (Info%xBounds(3,1)==GxBounds(3,1) .AND. Gmthbc(3,1)==10) THEN    ! front Side

            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  !y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = 1-rmbc, 0

                     ! Extrapolating
                     Info%q(i,j,k,:) = Info%q(i,j,1,:)

                     IF (q(i,j,k,4).GT.0.d0) THEN               
                        Temp   = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                        q(i,j,k,4)=0.d0
                        CALL CalcComp(q(i,j,k,:),Temp)      
                     END IF

                  END DO
               END DO
            END DO

         END IF

         IF (Info%xBounds(3,2)==GxBounds(3,2) .AND. Gmthbc(3,2)==10) THEN    ! back Side


            DO i = 1-rmbc, mx+rmbc
               !x = (xl+(REAL(i,xPrec)-half)*dx)       ! set x, y coordinates at cell centers
               DO j = 1-rmbc, my+rmbc
                  !y = (yl+(REAL(j,xPrec)-half)*dy)
                  DO k = mz+1, mz+rmbc

                     ! Extrapolating
                     Info%q(i,j,k,:) = Info%q(i,j,mz,:)

                     IF (q(i,j,k,4).LT.0.d0) THEN

                        Temp  = TempScale*Press(q(i,j,k,:))/q(i,j,k,1)
                        q(i,j,k,4)=0.d0
                        CALL CalcComp(q(i,j,k,:),Temp) 
                     END IF
                  END DO
               END DO
            END DO

         END IF


      CASE DEFAULT
         PRINT *, "Error in CloudsBC, invalid problem dimensions"
         STOP
      END SELECT




   END SUBROUTINE CloudsBC


   SUBROUTINE CalcComp(q,T,P)
      REAL(KIND=xprec), DIMENSION(nRvars) :: q
      REAL(KIND=xprec),INTENT(IN) :: T
      REAL(KIND=xprec),INTENT(OUT),OPTIONAL :: P
      REAL(KIND=xprec) :: gam,mu,Tin,Pin
      !      REAL (KIND=qPrec), DIMENSION(0:nSpeciesHI) :: nvec

      mu=Xmu
      gam=gamma

      !      IF(icooling==2) THEN
      ! for chemical flow, get the molecular weight and 
      ! gamma for computing energy
      !         CALL EOS_vars(q,nvec=nvec(iHaux:nSpeciesHI),mu=mu,gamma=gam)
      !      ELSE IF(iEOS==1) THEN
      !         gam=gammac
      !      END IF

      ! T is scaled by TempScale from Physics.data, so input in real units       
      q(iE)= q(1)*one/(gam-one)*T/(TempScale*mu)

      Pin=(gam-one)*q(iE)
      IF(PRESENT(P)) P=Pin

      q(iE) = q(iE)+half*DOT_PRODUCT(q(2:m_high),q(2:m_high))/q(1)

      !      IF(iEntropy/=0) THEN
      !         q(iEntropy) = Pin*q(1)**(one-gam)
      !      END IF
   END SUBROUTINE CalcComp
   ! End CalcComp

   INCLUDE 'fldgen.f90'
   INCLUDE 'genak.f90'
   INCLUDE 'nr_fourn.f90'
   INCLUDE 'nr_randnum.f90'

   SUBROUTINE ProblemBeforeGlobalStep(n)
      INTEGER :: n
   END SUBROUTINE ProblemBeforeGlobalStep


END MODULE Problem
